/***********************************************************************
* as - Assembler shim. This program acts as the "as" command on UNIX.
*      The source file has the tabs expanded and is then sent to the
*      assembler (via "cc" command).
***********************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#define NORMAL 0
#define ABORT  12

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#define AS_CMD "/bin/cc"
#define TEMPSPEC "asXXXXXX"

/***********************************************************************
* tabpos - return TRUE if col is a tab stop
***********************************************************************/

static int
tabpos (int col,
	int tabs[]) 
{
   if (col < BUFSIZ)
      return (tabs[col]);

    return (TRUE); 

} /* tabpos */

/***********************************************************************
* detabit - convert tabs to equivalent number of blanks
***********************************************************************/

static void
detabit (int tabs[],
	 FILE *ifd,
         FILE *ofd)
         
{
   int c, col;
 
   col = 0;
   while ((c = fgetc (ifd)) != EOF)
   {

      switch (c)
      {

         case '\t' :
            while (TRUE)
	    { 
               fputc (' ', ofd);
               if (tabpos (++col, tabs) == TRUE) 
                  break ;
            } 
            break;

         case '\n' :
            fputc (c, ofd);
            col = 0;
            break;

         default: 
            fputc (c, ofd);
            col++;

      } /* switch */

   } /* while */

} /* detab */

/***********************************************************************
* alldig - check if all digits
***********************************************************************/

static int
alldig (char *digs)
{
   while (*digs)
   {
      if (!isdigit (*digs))
      {
         return (FALSE);
      }
      digs++;
   }

   return (TRUE);

} /* alldig */

/***********************************************************************
* settab - set initial tab stops
***********************************************************************/

static void
settab (int tabs[],
	int argc,
	char *argv[])
{
   int m, p, i, j, l;
   char *bp;
  
   p = 0;

   for (i = 0; i < BUFSIZ; i++)
       tabs[i] = FALSE;

   for (j = 1; j < argc; j++)
   {

      bp = argv[j];

      if (*bp == '+')
         bp++;

      if (alldig (bp))
      {

         l = atoi (bp) ;

         if (l < 0 || l >= BUFSIZ)
             continue;

         if (*argv[j] != '+')
         { 
            p = l;
            tabs[p] = TRUE;
         } 

         else
         { 
            if (p == 0) 
               p = l;
            for (m = p; m < BUFSIZ; m += l) 
            {
               tabs[m] = TRUE;
            }
         }

      }

   } 

   if (p == 0)
   {
      for (i = 8; i < BUFSIZ; i += 8) 
      {
         tabs[i] = TRUE;
      }
   }

} /* settab */

/***********************************************************************
* detab source
***********************************************************************/

static int
detab (char *infile,
       char *outfile) 
{
   FILE *ifd;
   FILE *ofd;
   int tabs[BUFSIZ];
   char *bp;
   char *tabstops[] = { "as", "9", "16", "+8", "\0" };
  
#ifdef DEBUG
   fprintf (stderr,
	    "detab: Entered: infile = %s, outfile = %s\n",
	    infile,
	    outfile);
#endif

   /* set initial tab stops */

   settab (tabs, 4, tabstops);

   ofd = fopen (outfile, "w");
  
   if ((ifd = fopen (infile, "r")) == NULL)
   {
      fprintf (stderr, "%s : ", infile);
      perror ("Can't open input ");
      return (-1);
   }

   detabit (tabs, ifd, ofd);
   fclose (ifd);
   fclose (ofd);

   return (0);

}
/***********************************************************************
* Main procedure
***********************************************************************/

int
main (int argc, char *argv[])
{
   char *infile;
   char *outfile;
   char *bp;
   int i;
   int status;
   char mkname[32];
   char tname[64];
   char command[256];
  
#ifdef DEBUG
   fprintf (stderr,
	    "as: Entered\n");
#endif
   infile = NULL;
   outfile = NULL;

   for (i = 1; i < argc; i++)
   {
      bp = argv[i];

      if (*bp == '-')
      {
         bp++;
         switch (*bp)
         {
         case 'o':
            i++;
            outfile = argv[i];
            break;

         default:
      USAGE:
	    printf ("usage:  as -o file.o file.s\n");
	    exit (ABORT);
         }
      }

      else
      {
         if (infile) goto USAGE;
         infile = argv[i];
      }

   }

   if (!infile && !outfile) goto USAGE;

   strcpy (mkname,
	   TEMPSPEC);
   if ((bp = mktemp (mkname)) == NULL)
   {
      perror ("Can't mktemp");
      exit (ABORT);
   }

   sprintf (tname,
	    "/tmp/%s.s",
	    bp);

   status  = detab (infile,
		    tname);

   if (status == 0)
   {

      sprintf (command,
	       "%s -c -o %s %s 2>/dev/null",
	       AS_CMD,
	       outfile,
	       tname);

#ifdef DEBUG
      fprintf (stderr,
	       "as: command = %s\n",
	       command);
#endif

      status = system (command);
#ifdef DEBUG
      fprintf (stderr,
	       "system: command: status = %d\n",
	       status);
#endif
      if (status == 0)
      {
         struct stat stbuf;

	 stat (outfile,
	       &stbuf);
#ifdef DEBUG
         fprintf (stderr,
		  "stat: st_mode = %o\n",
		  stbuf.st_mode);
#endif
         stbuf.st_mode &= ~S_IXUSR;
         stbuf.st_mode &= ~S_IXGRP;
         stbuf.st_mode &= ~S_IXOTH;
#ifdef DEBUG
         fprintf (stderr,
		  "chmod: st_mode = %o\n",
		  stbuf.st_mode);
#endif
         chmod (outfile,
		stbuf.st_mode);
      }
   }

   unlink (tname);

   strcat (mkname,
	   ".o");
   unlink (mkname);

   exit (status);
}

