/***************************************************************************
*
* E2A - EBCDIC to ASCII conversion utility.
*
* Usage:
*
*    e2a [-t] [-] [filename]
*
* Where:
*	-t	 = Specifies that trailing blanks are trimmed.
*       - 	 = Specifies to read stdin. This is to allow options.
*	filename = Specifies the file to be processed.
*
* If no arguments are given; the default is to read from stdin and
* write to stdout without trimming.
*
***************************************************************************/

#include <stdio.h>
#include <ctype.h>
#include <string.h>

#define LINE_MAX 80

/*
** EBCDIC to ASCII conversion table. 
*/

static unsigned char ebcasc[256] =
{
 /*00  NU    SH    SX    EX    PF    HT    LC    DL */
      0x00, 0x01, 0x02, 0x03, 0x3F, 0x09, 0x3F, 0x7F,
 /*08              SM    VT    FF    CR    SO    SI */
      0x3F, 0x3F, 0x3F, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
 /*10  DE    D1    D2    TM    RS    NL    BS    IL */
      0x10, 0x11, 0x12, 0x13, 0x14, 0x0A, 0x08, 0x3F,
 /*18  CN    EM    CC    C1    FS    GS    RS    US */
      0x18, 0x19, 0x3F, 0x3F, 0x1C, 0x1D, 0x1E, 0x1F,
 /*20  DS    SS    FS          BP    LF    EB    EC */
      0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x0A, 0x17, 0x1B,
 /*28              SM    C2    EQ    AK    BL       */
      0x3F, 0x3F, 0x3F, 0x3F, 0x05, 0x06, 0x07, 0x3F,
 /*30              SY          PN    RS    UC    ET */
      0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x04,
 /*38                    C3    D4    NK          SU */
      0x3F, 0x3F, 0x3F, 0x3F, 0x14, 0x15, 0x3F, 0x1A,
 /*40  SP                                           */
      0x20, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
 /*48             CENT    .     <     (     +     | */
      0x3F, 0x3F, 0x9B, 0x2E, 0x3C, 0x28, 0x2B, 0x7C,
 /*50   &                                           */
      0x26, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
 /*58               !     $     *     )     ;     ^ */
      0x3F, 0x3F, 0x21, 0x24, 0x2A, 0x29, 0x3B, 0x5E,
 /*60   -     /                                     */
      0x2D, 0x2F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
 /*68               |     ,     %     _     >     ? */
      0x3F, 0x3F, 0x7C, 0x2C, 0x25, 0x5F, 0x3E, 0x3F,
 /*70                                               */
      0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
 /*78         `     :     #     @     '     =     " */
      0x3F, 0x60, 0x3A, 0x23, 0x40, 0x27, 0x3D, 0x22,
 /*80         a     b     c     d     e     f     g */
      0x3F, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
 /*88   h     i           {                         */
      0x68, 0x69, 0x3F, 0x7B, 0x3F, 0x3F, 0x3F, 0x3F,
 /*90         j     k     l     m     n     o     p */
      0x3F, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70,
 /*98   q     r           }                         */
      0x71, 0x72, 0x3F, 0x7D, 0x3F, 0x3F, 0x3F, 0x3F,
 /*A0         ~     s     t     u     v     w     x */
      0x3F, 0x7E, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
 /*A8   y     z                       [             */
      0x79, 0x7A, 0x3F, 0x3F, 0x3F, 0x5B, 0x3F, 0x3F,
 /*B0                                               */
      0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
 /*B8                                 ]             */
      0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x5D, 0x3F, 0x3F,
 /*C0   {     A     B     C     D     E     F     G */
      0x7B, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
 /*C8   H     I                                     */
      0x48, 0x49, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
 /*D0   }     J     K     L     M     N     O     P */
      0x7D, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50,
 /*D8   Q     R                                     */
      0x51, 0x52, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
 /*E0   \           S     T     U     V     W     X */
      0x5C, 0x3F, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
 /*E8   Y     Z                                     */
      0x59, 0x5A, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
 /*F0   0     1     2     3     4     5     6     7 */
      0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
 /*F8   8     9                                     */
      0x38, 0x39, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0xFF
};

static int
convert (FILE *ifd, int trim)
{
   int col = 0;
   int c;
   char buf[LINE_MAX + 2];

   while ((c = fgetc (ifd)) >= 0)
   {
      c = ebcasc[c &0xFF];
      if (trim)
      {
	 if (col == LINE_MAX)
	 {
	    while (buf[--col] == ' ') ;
	    buf[++col] = '\n';
	    buf[++col] = '\0';
	    fputs (buf, stdout);
	    col = 0;
	 }
         buf[col++] = c;
      }
      else
	 fputc (c, stdout);
   }
   if (trim && col)
   {
      while (buf[--col] == ' ') ;
      buf[++col] = '\n';
      buf[++col] = '\0';
      fputs (buf, stdout);
   }
}

main (int argc, char *argv[])
{
   int trim = 0;
   int i;
   FILE	*ifd;

   if (argc == 1)
   {
      convert (stdin, trim);
   }
   else
   {
      for (i = 1; i < argc; i++)
      {
         if (*argv[i] == '-')
	 {
	    argv[i]++;
	    switch (*argv[i])
	    {
	       case '\0':
	          convert (stdin, trim);
		  break;

	       case 't':
		  trim = 1;
	          break;

	       default:
	          fprintf (stderr, "Unknown option: %c\n", *argv[i]);
		  exit (1);
	    }
	 }
	 else
	 {
	    if ((ifd = fopen (argv[i], "r")) != NULL)
	    {
	       convert (ifd, trim);
	       fclose (ifd);

	    }
	    else
	    {
	       fprintf (stderr, "Can't open input file: %s\n", argv[i]);
	       exit (1);
	    }
	 }
      }
   }
}
