/************************************************************************
*
* An SLR(1) parser constructor - CHAT
*
* Changes:
*   version 1.0 (05/11/77) TI-990
*   version 2.0 (06/01/81) VAX
*   version 3.0 (03/17/82) FTN
*   version 4.0 (03/30/87) Turbo - generate VAX-MACRO
*   version 5.0 (04/10/89) Rewrite in C
*
* Abstract:
*     Accepts a Backus-Naur-Form (BNF) description of a language and
*     outputs tables used by an SLR(1) parser for syntactic recognition
*     of the language described by the input grammar.
*
* Interface:
*     Chat is written in 'C'. The following files are used:
*
*     source  - File pathname containing a BNF description
*               of the language
*     list    - File pathname for listing of input and for outputing
*               all productions, the vocabulary, and the lr0 state
*               configurations
*     ptables - File pathname for outputing the parsing
*               action and goto tables
*     semant  - File pathname for outputting semantic actions
*     errors  - File pathname for outputting parse error messages
*     tokens  - File pathname for outputting tokens
*
* References:
*     The implementation of chat is patterned after
*     'YACC - Yet Another Compiler-Compiler' by Stephen C. Johnson
*     operating on the unix operating system (Bell Laboratories).
*
*     1) "LR Parsing", Aho and Johnson; Computing Surveys
*          Vol 6, #2 June 1974.
*     2) "Principles of Compiler Design", Aho and Ullman,
*          Addison Wesley 1976.
*
************************************************************************/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#if defined(IBMMVS) || defined(IBMVM)
#include <stdefs.h>
#include <stdlib.h>
#else
#include <stddef.h>
#endif
#if defined(gccsun)
typedef long time_t;
#endif

#if defined(VAX) && defined(VMS)
#define VAXVMS
#include <stsdef.h>
#include <ssdef.h>
#define NORMAL (SS$_NORMAL|STS$M_INHIB_MSG)
#define ABORT  (SS$_ABORT|STS$M_INHIB_MSG)
#else
#define NORMAL 0
#define ABORT  12
#endif

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif
#ifndef NULL
#define NULL (void *)0
#endif

#define MAXSTATES    1000 /* 400 */
#define MAXSYMBOLS   200
#define MAXSYMLEN    32
#define MAXPRODLEN   500 /* 100 */
#define MAXMARKER    (MAXPRODLEN + 1)
#define MAXPRODS     1000 /* 500 */
#define MAXITEMS     2500
#define MAXGOTOS     1500
#define MAXSQUEEZE   200 /* 20 */
#define NONTERMINAL  1
#define TERMINAL     2
#define TOKEN        3
#define MAXLINES     55

typedef int nameindex;    /*  1..MAXSYMLEN */
typedef int symbolindex;  /*  1..MAXSYMBOLS */
typedef int symbolarg;    /* -1..MAXSYMBOLS */
typedef int stateindex;   /*  1..MAXSTATES */
typedef int statearg;     /* -1..MAXSTATES */
typedef int gtlength;     /*  0..MAXGOTOS */
typedef int prodindex;    /*  0..MAXPRODS */
typedef int vlength;      /*  0..MAXSYMBOLS */
typedef int glength;      /*  0..MAXPRODS */
typedef int itemindex;    /*  1..MAXITEMS */
typedef int gotoindex;    /*  1..MAXGOTOS */
typedef int rplength;     /*  0..MAXPRODLEN - right part length */
typedef int markerindex;  /*  0..MAXMARKER - valid marker positions */
typedef int trmtype;      /* (NONTERMINAL, TERMINAL, TOKEN) */
typedef int boolean;

typedef struct 
{ 
   char    name[MAXSYMLEN];
   trmtype ttype;
   boolean closed;
   boolean effclosed;
   int     glisthead;
} symbol;

typedef struct 
{
   symbolindex lpart;
   rplength    length;
   symbolindex rpart[MAXPRODLEN];
} prod;

typedef struct 
{
   int         cnt;
   prodindex   prd;
   symbolindex syms[100];
} reduce_tbl;

typedef struct 
{
   int        cnt;
   stateindex nxtsta;
   stateindex oldsta[100];
} goto_tbl;


static char ch;
static char pathname[80], filename[80], tempstring[80];
static int  lineno, symlen, errcnt, i;
static boolean listflag, checkflag, bad, eofflg, verbose;
static int pagenum, linenum;
static time_t loctim;

static struct 
{
   struct 
   {
      stateindex  oldstate;
      symbolindex symbol;
      stateindex  gnextstate;
   } graph[MAXGOTOS];
   gtlength gotolen;
} gotograph;

static struct 
{
   vlength length;
   symbol  *symbols[MAXSYMBOLS];
   boolean laset1[MAXSYMBOLS];
   boolean laset2[MAXSYMBOLS];
} vocabulary;

static struct 
{
   glength length;
   prod    *prods[MAXPRODS];
} grammar;

static struct 
{
   itemindex  state[MAXSTATES];
   struct 
   {
      prodindex   production;
      markerindex imarker;
   } item[MAXITEMS];
   stateindex opensta;
   stateindex markstate;
   itemindex  openitem;
} lr0state;

static reduce_tbl reduce_lst[MAXSQUEEZE];
static goto_tbl   goto_lst[MAXSQUEEZE];
static int        ptr_lst[MAXSQUEEZE];

static FILE *source;
static FILE *list;
static FILE *semant;
static FILE *ptables;
static FILE *errors;
static FILE *tokens;

static void eff (boolean laset[], prodindex p, markerindex l);
static itemindex firstitem (stateindex statenum);
static void follow (boolean laset[], symbolindex n);
static symbolindex getterm (prodindex p, rplength t);
static markerindex marker (itemindex itemnum);
static symbolindex insertsym (char *s, trmtype t);
static prodindex itemprod (itemindex itemnum);
static rplength itemlen (itemindex itemnum);
static void ivocab (void);
static itemindex lastitem (stateindex statenum);
static symbolindex lhsnonterminal (prodindex p);
static void page (void);
static void printi (prodindex p, markerindex i);
static rplength prodlen (prodindex p);
static void psymbol(FILE *t, symbolindex sym, boolean f);
static void sclose (void);
static trmtype termtype (symbolindex s);
static void unclose (void);
static void uneffclose (void);

static char
TOUPPER (char ch)
{
   if (islower (ch))
      ch = toupper (ch);
   return (ch);
}

static char
TOLOWER (char ch)
{
   if (isupper (ch))
      ch = tolower (ch);
   return (ch);
}

/************************************************************************
* Add item
************************************************************************/

static void
additem (prodindex lprod,
	 markerindex lmark)
{
   lr0state.item[lr0state.openitem].imarker = lmark;
   lr0state.item[lr0state.openitem].production = lprod;
   lr0state.openitem++;
}

/************************************************************************
* Add a goto arc
************************************************************************/

static void
addgotoarc (stateindex old,
	    stateindex next,
	    symbolindex sym)
{
   if (listflag)
   {
      fprintf (list, "         %4d   ", old);
      psymbol (list, sym, FALSE);
      fprintf (list, "    %6d\n", next);
      if (++linenum > MAXLINES) page();
   }
   gotograph.gotolen++;
   gotograph.graph[gotograph.gotolen].oldstate = old;
   gotograph.graph[gotograph.gotolen].symbol = sym;
   gotograph.graph[gotograph.gotolen].gnextstate= next;
}

/************************************************************************
* Add item to the open state
************************************************************************/

static void
addopenstate (void)
{
   lr0state.opensta++;
   lr0state.state[lr0state.opensta] = lr0state.openitem;
}

/************************************************************************
* Extend the production being built with another term.
* Exceptions: startprod-not-called: not handled
*     production-overflow: 'length' > MAXPRODLEN
************************************************************************/

static void
addterm (char *s,
	 trmtype t)
{
   grammar.prods[grammar.length]->length++;
   if (grammar.prods[grammar.length]->length > MAXPRODLEN) 
   {
      strcpy (tempstring, "**Production to long**\n");
      fputs (tempstring, stdout);
      fputs (tempstring, list);
      if (++linenum > MAXLINES) page();
   }
   else grammar.prods[grammar.length]->
         rpart[grammar.prods[grammar.length]->length] = insertsym (s, t);
}

/************************************************************************
* Mark a specified symbol as 'closed'.
************************************************************************/

static void
closesymbol (symbolindex s)
{
   vocabulary.symbols[s]->closed = TRUE;
}

/************************************************************************
* Look for and print parsing shift-reduce and reduce-reduce conflicts in
* the input grammar. Assume that the lookahead set of the item "i" of
* interest has already been computed.
************************************************************************/

static void
conflicts (stateindex s,
	   itemindex i)
{
   itemindex   j;
   symbolindex sym;

   for (j = firstitem(s); j <= lastitem(s); j++)
      if (j != i) 
      {
         for (sym = 1; sym <= vocabulary.length; sym++)
            vocabulary.laset2[sym] = FALSE;
         unclose();
         uneffclose();
         if (marker(j) == itemlen(j) + 1) 
         {
            follow (vocabulary.laset2, lhsnonterminal(itemprod(j)));
            for (sym = 1; sym <= vocabulary.length; sym++)
               if (vocabulary.laset1[sym] && vocabulary.laset2[sym]) 
               {
                  errcnt++;
                  fprintf (list, "Reduce-Reduce conflict in state: %4d\n",
                          s);
                  if (++linenum > MAXLINES) page();
                  fprintf (list, "%20.20son symbol: ", "");
                  psymbol (list, sym, FALSE);
                  fputc ('\n', list);
                  if (++linenum > MAXLINES) page();
                  fprintf (list, "%16.16sbetween items:\n", "");
                  if (++linenum > MAXLINES) page();
                  fprintf (list, "%15.15s", "");
                  printi (itemprod(i), marker(i));
                  fprintf (list, "%15.15s", "");
                  printi (itemprod(j), marker(j));
                  fputc ('\n', list);
                  if (++linenum > MAXLINES) page();
               }
         }
         else 
         {
            eff (vocabulary.laset2, itemprod(j), marker(j));
            for (sym = 1; sym <= vocabulary.length; sym++)
               if (vocabulary.laset1[sym] && vocabulary.laset2[sym]) 
               {
                  errcnt++;
                  fprintf (list, "Shift-Reduce conflict in state: %4d\n",
                          s);
                  if (++linenum > MAXLINES) page();
                  fprintf (list, "%20.20son symbol: ", "");
                  psymbol (list, sym, FALSE);
                  fputc ('\n', list);
                  if (++linenum > MAXLINES) page();
                  fprintf (list, "%16.16sbetween items:\n", "");
                  if (++linenum > MAXLINES) page();
                  fprintf (list, "%15.15s", "");
                  printi (itemprod(i), marker(i));
                  fprintf (list, "%15.15s", "");
                  printi (itemprod(j), marker(j));
                  fputc ('\n', list);
                  if (++linenum > MAXLINES) page();
              }
         }
      }
}

/************************************************************************
* Return current goto state
************************************************************************/

static stateindex
currentstate (gotoindex g)
{
   return (gotograph.graph[g].oldstate);
}

/************************************************************************
* Delete an open state
************************************************************************/

static void
deleteopenstate (void)
{
   lr0state.openitem = lr0state.state[lr0state.opensta];
}

/************************************************************************
* Assume that the item is not a reduce item, ie, that the dot position
* is not greater than the number of right hand side symbols of the base
* production of the item.
************************************************************************/

static symbolindex
dotsymbol (itemindex itemnum)
{
   return (getterm (lr0state.item[itemnum].production,
                      lr0state.item[itemnum].imarker));
}

/************************************************************************
*
************************************************************************/

static void
eff (boolean laset[],
     prodindex p,
     markerindex l)
{
   symbolindex sym;
   prodindex   prd;
   boolean     lambdap;

   sym = grammar.prods[p]->rpart[l];
   if ((termtype(sym) == TOKEN) || (termtype(sym) == TERMINAL))
      laset[sym]= TRUE;
   else if (!(vocabulary.symbols[sym]->effclosed)) 
   {
      vocabulary.symbols[sym]->effclosed = TRUE;
      lambdap = FALSE;
      for (prd = 1; prd <= grammar.length; prd++)
         if (lhsnonterminal(prd) == sym)
            if (grammar.prods[prd]->length == 0)
               lambdap = TRUE;
            else eff (laset, prd, 1);
      if (lambdap)
         if (l >= grammar.prods[p]->length)
            follow (laset, grammar.prods[p]->lpart);
         else eff (laset, prd, l+1);
   }
}

/************************************************************************
* end the definition of a new production
* exceptions: startprod-not-called: not handled
************************************************************************/

static void
endprod (void)
{
}

/***********************************************************************
* Return state length.
***********************************************************************/

static int
statelen (stateindex statenum)
{
   return (lastitem(statenum) - firstitem(statenum) + 1);
}

/***********************************************************************
* Check if items are equal
***********************************************************************/

static boolean
eqitem (itemindex item1, itemindex item2)
{
   return ((marker(item1) == marker(item2)) &&
            (itemprod(item1) == itemprod(item2)));
}

/************************************************************************
* Check for equal states
************************************************************************/

static boolean
eqstate (stateindex state1,
	 stateindex state2)
{
   int i;
   int n;

   i = 0;
   n = statelen(state1);

   if (n == statelen(state2))
      while (((i < n) && eqitem (firstitem(state1) + i,
               firstitem (state2) + i)))  i++;

   if (i == n) return (TRUE);
   return (FALSE);
}

/************************************************************************
* Returns the index into the vocabulary table if the symbol is defined.
* execeptions: symbol-not-found: symbol is not defined (returns -1)
************************************************************************/

static vlength
findsym (symbol *s)
{
   vlength i;

   for (i = 1; i <= vocabulary.length; i++)
      if ((vocabulary.symbols[i]->ttype == s->ttype) &&
            !strcmp (vocabulary.symbols[i]->name, s->name))
         return (i);
   return (0);
}

/************************************************************************
* Get the first item
************************************************************************/

static itemindex
firstitem (stateindex statenum)
{
   return (lr0state.state[statenum]);
}

/************************************************************************
* Follow a parse tree
************************************************************************/

static void
follow (boolean laset[],
	symbolindex n)
{
   prodindex   p;
   markerindex l;

   vocabulary.symbols[n]->closed = TRUE;
   for (p = 1; p <= grammar.length; p++) 
   {
      l = 1;
      while (l < grammar.prods[p]->length) 
      {
         if (grammar.prods[p]->rpart[l] == n) 
         {
            uneffclose();
            eff (laset, p, l+1);
         }
         l++;
      }
      if ((grammar.prods[p]->rpart[l] == n) &&
            !(vocabulary.symbols[lhsnonterminal(p)]->closed))
         follow (laset, lhsnonterminal(p));
   }
}

/************************************************************************
* Determine the symbol index of a particular term of a given production.
************************************************************************/

static symbolindex
getterm (prodindex p,
	 rplength t)
{
   return (grammar.prods[p]->rpart[t]);
}

/************************************************************************
* Check if goto exists.
************************************************************************/

static boolean
gotoexists (gotoindex g,
	    symbolindex u)
{
   return (gotograph.graph[g].symbol == u);
}

/************************************************************************
* Get goto length.
************************************************************************/

static gtlength
gotolength (void)
{
   return (gotograph.gotolen);
}

/************************************************************************
* Set goto state.
************************************************************************/

static stateindex
gotostate (stateindex opnstate,
	   stateindex markstate,
                    symbolindex focussymbol)
{
   stateindex s;

   s = 1;
   while ((s < opnstate) && (!eqstate(s,opnstate)))
      s++;
   addgotoarc (markstate, s, focussymbol);
   return (s);
}

/************************************************************************
* Determine the number of productions in the grammar
************************************************************************/

static glength
gramlen (void)
{
   return (grammar.length);
}

/************************************************************************
* Initialize gotograph
************************************************************************/

static void
igotograph (void)
{
   gotograph.gotolen = 0;
}

/************************************************************************
* grammar initialization
************************************************************************/

static void
igrammar (void)
{
   ivocab();                 /* Initialize the vocabulary tables */
   grammar.length = 0;       /* Initial number of entries */
}

/************************************************************************
* Initialize lr0 state data area.
************************************************************************/

static void
ilr0state (void)
{
   itemindex  i;
   stateindex s;

   for (s = 1; s <= MAXSTATES; s++) lr0state.state[s] = 1;
   lr0state.item[1].imarker = 1;
   lr0state.item[1].production = 1;
   for (i = 2; i <= MAXITEMS ; i++) 
   {
      lr0state.item[i].imarker = 0;
      lr0state.item[i].production = 0;
   }
   lr0state.opensta = 1;
   lr0state.openitem = 2;
   sclose();
   addopenstate();
}

/***********************************************************************
* Defines the symbol s in the vocabulary table.
***********************************************************************/

static void
addsym (symbol *sym)
{
   /* insert a new symbol into the vocabulary */
   vocabulary.length++;
   if (vocabulary.length > MAXSYMBOLS) 
   {
      strcpy (tempstring, "**Vocabulary table overflow**\n");
      fputs (tempstring, stdout);
      fputs (tempstring, list);
      exit (ABORT);
   }
   else 
   {
      vocabulary.symbols[vocabulary.length] = 
           (symbol *) malloc (sizeof (symbol));
      memcpy (vocabulary.symbols[vocabulary.length], sym, sizeof (symbol));
   }
}

/************************************************************************
* Return the vocabulary index of this symbol, adding the symbol to the 
* vocabulary if necessary.
************************************************************************/

static symbolindex
insertsym (char *s,
	   trmtype t)
{
   symbol  sym;
   vlength i;
   char    c;

   /* build a symbol record */
   strcpy (sym.name, s);
   sym.ttype = t;
   sym.closed = FALSE;      /* initial value */
   sym.glisthead = 0;       /* initial value */

   /* determine if the symbol is defined */
   i = findsym (&sym);

   /* if the symbol is not defined add it to the vocabulary */
   if (i == 0) 
   {
      addsym (&sym);
      i = vocabulary.length;
   }
   return (i);
}

/************************************************************************
* Determine if a symbol is 'closed' ?
************************************************************************/

static boolean
isclosed (symbolindex s)
{
   return (vocabulary.symbols[s]->closed);
}

/************************************************************************
* Test if a symbol is in the current lookahead set assume that the
* lookahead set of interest has already been generated.
************************************************************************/

static boolean
isinla (symbolindex sym)
{
   return (vocabulary.laset1[sym]);
}

/************************************************************************
* Get the item length.
************************************************************************/

static rplength
itemlen (itemindex itemnum)
{
   return (prodlen (lr0state.item[itemnum].production));
}

/************************************************************************
* Get the item production.
************************************************************************/

static prodindex
itemprod (itemindex itemnum)
{
#ifdef DEBUG
   printf (" item = %d, open = %d, mark = %d, prod = %d\n", 
      itemnum, lr0state.openitem, lr0state.item[itemnum].imarker, 
      lr0state.item[itemnum].production);
#endif
   if (itemnum > lr0state.openitem)
      return (0);
   return (lr0state.item[itemnum].production);
}

/************************************************************************
* Initialize vocabulary.
************************************************************************/

static void
ivocab (void)
{
   vocabulary.length = 0;
}

/************************************************************************
* Get the last item.
************************************************************************/

static itemindex
lastitem (stateindex statenum)
{
   if (statenum == lr0state.opensta) return (lr0state.openitem - 1);
   else return (lr0state.state[statenum+1] - 1);
}

/************************************************************************
* Find the symbol index of the left hand part of a production.
* exceptions: p > grammar.length : not handled
************************************************************************/

static symbolindex
lhsnonterminal (prodindex p)
{
   return (grammar.prods[p]->lpart);
}

/************************************************************************
* Set up look ahead.
************************************************************************/

static void
lookaheadset (itemindex i)
{
   symbolindex s;

   for (s = 1; s <= vocabulary.length; s++)
      vocabulary.laset1[s] = FALSE;
   unclose();
   uneffclose();
   if (marker(i) == itemlen(i) + 1)
      follow (vocabulary.laset1, lhsnonterminal(itemprod(i)));
   else
      eff (vocabulary.laset1, itemprod(i), marker(i));
}

/************************************************************************
* Get the marker.
************************************************************************/

static markerindex
marker (itemindex itemnum)
{
   return (lr0state.item[itemnum].imarker);
}

/************************************************************************
* Enter a new state into goto graph.
************************************************************************/

static stateindex
newstate (symbolindex u,
	  stateindex s)
{
   gtlength g;

   g = 0;
   while (TRUE) 
   {
      g++;
      if ((gotograph.graph[g].oldstate == s) && 
           (gotograph.graph[g].symbol == u))
         return (gotograph.graph[g].gnextstate);
   }
}

/************************************************************************
* Return next goto state.
************************************************************************/

static stateindex
nextstate (gotoindex g)
{
   return (gotograph.graph[g].gnextstate);
}

/************************************************************************
* Get the current number of states not including the open state.
************************************************************************/

static stateindex
numstates (void)
{
   return (lr0state.opensta - 1);
}

/************************************************************************
* Get the current open state number.
************************************************************************/

static stateindex
openstate (void)
{
   return (lr0state.opensta);
}

/************************************************************************
* Print a symbol excluding blanks.
************************************************************************/

static void
p2symbol (FILE *t,
	  symbolindex sym)
{
   int i;

   for (i = 0; i < strlen (vocabulary.symbols[sym]->name); i++)
      fputc (TOUPPER (vocabulary.symbols[sym]->name[i]), t);
}

/***********************************************************************
* Generate a new page in the listing
***********************************************************************/

static void
page (void)
{
   char *bp;

   bp = ctime (&loctim);
   *strchr (bp, '\n') = '\0';
   fprintf (list, "\f\n\nCHAT 5.0  %s  %-30.30s  Page: %d\n\n",
      bp, pathname, ++pagenum);
   linenum = 0;
}

/************************************************************************
* Print a production on the output device.
* Exceptions: production-index-out-of-bounds: not handled
************************************************************************/

static void
printi (prodindex p,
	markerindex i)
{
   int         j,l;
   markerindex k;
   char        c;

   /* print the production number and left hand part */
   fprintf (list, "   %3d)   <", p);
   fputs (vocabulary.symbols[grammar.prods[p]->lpart]->name, list);
   fputc ('>', list);
   for (j = symlen - 
         strlen (vocabulary.symbols[grammar.prods[p]->lpart]->name);
            j >= 0; j--) fputc (' ', list);
   fputs (" = ", list);

   /* print out each term of the right hand part */
   for (k = 1; k <= grammar.prods[p]->length ; k++) 
   {
      fputc (' ', list);
      if (k == i) fputs ("_ ", list);
      switch (vocabulary.symbols[grammar.prods[p]->rpart[k]]->ttype) 
      {
         case NONTERMINAL : c = '<'; break;
         case TERMINAL :    c = ' '; break;
         case TOKEN :       c = '"'; break;
      }
      fputc (c, list);
      p2symbol (list, grammar.prods[p]->rpart[k]);
      if (c == '<') c = '>';
      fputc (c, list);
   }
   if (i == grammar.prods[p]->length + 1) fputs (" _", list);
   fputs (" ;\n", list);
   if (++linenum > MAXLINES) page();

}

/************************************************************************
* Determine the length of the right hand side of a given producton.
************************************************************************/

static rplength
prodlen (prodindex p)
{
   return (grammar.prods[p]->length);
}

/************************************************************************
* Print a symbol.
************************************************************************/

static void
psymbol (FILE *t,
	 symbolindex sym,
	 boolean f)
{
   int       i;
   nameindex j;

   i = strlen (vocabulary.symbols[sym]->name);
   for (j = 0; j < i; j++)
      fputc (TOUPPER (vocabulary.symbols[sym]->name[j]), t);
   for (; i <= symlen; i++) fputc (' ', t);
}

/************************************************************************
* Close out current state.
************************************************************************/

static void
sclose (void)
{
   itemindex i;
   prodindex p;

   unclose();
   i = firstitem (openstate());
   while (i <= lastitem(openstate())) 
   {
      if (marker(i) <= itemlen(i))
         if ((termtype(dotsymbol(i)) == NONTERMINAL) &&
            !isclosed(dotsymbol(i))) 
         {
            closesymbol (dotsymbol(i));
            for (p = 1; p <= gramlen(); p++)
               if (lhsnonterminal(p) == dotsymbol(i))
                  additem (p, 1);
         }
      i++;
   }
}

/************************************************************************
* Start the definition of a new production in the grammar.
* Exceptions: grammar-overflow: 'length' > MAXPRODS
************************************************************************/

static void
startprod (char *lp)
{
   grammar.length++;
   if (grammar.length > MAXPRODS)
   {
      strcpy (tempstring, "**Grammar table overflow**\n");
      fputs (tempstring, stdout);
      fputs (tempstring, list);
      exit (ABORT);
   }
   else 
   {
      grammar.prods[grammar.length] = 
          (prod *) malloc (sizeof (prod));
      grammar.prods[grammar.length]->lpart = insertsym (lp, NONTERMINAL);
      grammar.prods[grammar.length]->length = 0;
   }

}

/************************************************************************
* Determine the term type of a symbol.
************************************************************************/

static trmtype
termtype (symbolindex s)
{
   return (vocabulary.symbols[s]->ttype);
}

/************************************************************************
* Marks all grammar symbols as 'not closed'.
************************************************************************/

static void
unclose (void)
{
   symbolindex s;

   for (s = 1; s <= vocabulary.length; s++)
      vocabulary.symbols[s]->closed = FALSE;
}

/************************************************************************
* Marks all grammar symbols as 'not effclosed'.
************************************************************************/

static void
uneffclose (void)
{
   symbolindex s;

   for (s = 1; s <= vocabulary.length; s++)
      vocabulary.symbols[s]->effclosed = FALSE;
}

/************************************************************************
* Define the number of symbols in the vocabulary.
************************************************************************/

static vlength
vocablen (void)
{
   return (vocabulary.length);
}

/***********************************************************************
* Get input character and produce source listing.
***********************************************************************/

static char
getch (boolean fl)
{
   int ec;

   if (eofflg) return (' ');
   if ((ec = fgetc (source)) == EOF) 
   {
      eofflg = TRUE;
      return (' ');
   }

   if (ec == '\n') 
   {
      fputc (ec, list);
      if (++linenum > MAXLINES) page();
      fprintf (list, "%6d    ", ++lineno);
      if (fl) ec = ' ';
   }
   else if (ec == '\r') 
   {
      ec = ' ';
   }
   else 
   {
      fputc (ec, list);
      if (isspace (ec) && fl) ec = ' ';
   }
   return ((char)ec);
}

/***********************************************************************
* Skip blanks in the input stream 
***********************************************************************/

static char
skipblanks (void)
{
   char c;

   c = ' ';
   while (!eofflg && c == ' ') c = getch (TRUE);
   return (c);
}

/***********************************************************************
* Build a string of characters from the input stream. the string is
* delimited by 'endchar'. 'endchar' cannot be part of the string. The
* string must contain at least one character. Characters upto and
* including 'endchar' are removed from the input stream. The string
* constructed is built in 's' 
***********************************************************************/

static void
buildsymbol (char *s,
	     char stchar,
	     char endchar)
{
   char c;
   symbolindex i;

   c = stchar;
   i = 0;
   do 
   {
      if ((i <= MAXSYMLEN)) 
      {
         s[i] = c;
         i++;
         s[i] = '\0';
      }
      c = getch (TRUE);
   } while ((c != endchar) && (c != ';'));
   i--;
   if (i > symlen) symlen = i;
}

/***********************************************************************
* Semantic routine or comment handler.
***********************************************************************/

static void
semcom (boolean a)
{
   char    c;
   boolean bol;

   c = getch (TRUE);

   /* Comment */
   if (c == '*') 
   {
      do 
      {
         do 
         {
            c = getch (FALSE);
         } while (c != '*');
         c = getch (TRUE);
      } while (c != ')');
   }

   /* Semantic routine */
   else if (c == '#') 
   {
      bol = TRUE;
      if (a) fprintf (semant, "case %d:\n", gramlen());
      fputs ("   ", semant);
      c = ' ';
      do 
      {
         do 
         {
            if (!bol) fputc (c, semant);
            if (c == '\n') 
            {
               bol = TRUE;
               fputs ("   ", semant);
            }
            c = getch (FALSE);
            if (c != ' ') bol = FALSE;
         } while (c != '#');
         c = getch (TRUE);
         if (c != ')') fputc ('#', semant);
      } while (c != ')');
      fputs ("\n   break;\n", semant);
   }

   /* Error on input */
   else 
   {
      fputs ("**Unexpected character**\n", stdout);
      errcnt++;
      bad = TRUE;
      fprintf (list, " \\%c\\ ", c);
      while (c != '\n') c = getch (FALSE);
   }

}

/************************************************************************
*
* SLR parser constructor BNF input scanner.
*     The scanner is called to convert the BNF description of the slr
*     grammar into the vocabulary and grammar tables used by the slr
*     constructor routine.
*
*     The BNF description of the grammar also contains semantic routines
*     and actions. These may be written in any language as they are simply
*     copied to a seperate output file 'semant'. The only additional 
*     information added by this scanner is the production number
*     associated with each semantic action.
*
* The valid syntax of the input BNF is defined below in BNF (in itself):
*
*     <grammar> = <production>
*               ! <grammar> <production> ;
*     <production> = <comment or semantic routine>
*               ! <left part> "=" <production> ";" ;
*     <comment or semantic routine> = <comment>
*               ! <semantic routine> ;
*     <comment> = "(*" <ascii> "*" ")" ;
*
*     <semantic routine> = "(#" <ascii> "#)" ;
*
*     <left part> = <non terminal> ;
*
*     <production> = <right part> <comment or semantic routine>
*               ! <production> "!" <production> ;
*     <right part> = <nil>
*               ! <right part> <symbol>
*     <symbol>  = <non terminal>
*               ! <terminal> ;
*     <non terminal> = "<" <ascii> ">" ;
*
*     <terminal> =  <ascii>
*               ! <token> ;
*     <token>   = " <ascii_character> " ;
*
*  The result of 'scanner' is the generation of two tables:
*    'vocabulary', and 'grammar' (see slrdat for details) which
*    are used by the parser constructor.
*
* Exceptions: the input is written to the list device as it is read in. 
*    Error conditions may occur if the input is not in the form as 
*    described above. In such cases, an error message is printed onto
*    the console.
*
************************************************************************/

static void
scanner (void)
{
   char c;
   int  j;
   char lp[MAXSYMLEN], sym[MAXSYMLEN];

   /* Initialize the vocabulary and grammar data area */
   igrammar();

   page();
   lineno = 1;

   fprintf (list, "%6d  ", lineno);

   /* find the first non blank character */
   c = skipblanks();

   while (!eofflg) 
   {
      switch (c) 
      {
         case '(' :   /* comment or semantic routine*/
            semcom (FALSE);
            break;
         case '<' : /* start of a producton */
            c = getch (TRUE);
            buildsymbol (lp, c, '>');
            c = skipblanks();
            if (c != ':') goto bad_char;
            c = getch (TRUE);
            if (c != '=') goto bad_char;
            /* Start a production */
            startprod (lp);
            do 
            {
               c = skipblanks();
               switch (c) 
               {
                  case '<' : /* non-terminal symbol */
                     c = getch (TRUE);
                     buildsymbol (sym, c, '>');
                     addterm (sym, NONTERMINAL);
                     break;
                  case '!' : /* start new production */
                     endprod();
                     startprod (lp);
                     break;
                  case '"' : /* token symbol */
                     c = getch (TRUE);
                     buildsymbol (sym, c, '"');
                     addterm (sym, TOKEN);
                     if (strlen (sym) > 1) 
                     {
                        printf ("**Token too long '%s'**\n", sym);
                        errcnt++;
                        bad = TRUE;
                     }
                     break;
                  case '(' : /* possible semantic routine */
                     semcom (TRUE); 
                     break;
                  case ';' : /* end production */
                     endprod();
                     break;
                  default:  /* terminal symbol */
                     if (isalpha(c)) 
                     {
                        buildsymbol (sym, c, ' ');
                        addterm (sym, TERMINAL);
                     }
                     else goto bad_char;
               } /* of switch */
            } while (c != ';');
            break;

         default:
bad_char:   fputs ("**Unexpected character**\n", stdout);
            bad = TRUE;
            fprintf (list, " \\%c\\ ", c);
            while (c != ';') c = getch (FALSE);

      } /* of switch */

      /* Get the next valid character */
      c = skipblanks();
   } /* of while */
   fputc ('\n', list);
}

/************************************************************************
* Print the grammar.
************************************************************************/

static void
pgrammar (void)
{
   prodindex p;

   page();
   fprintf (list, "%10.10sGrammar Productions\n\n", "");
   linenum += 2;
   for (p = 1; p < gramlen(); p++) 
   {
      printi (p, 0);
   }
}

/***********************************************************************
* Print the terms
***********************************************************************/

static void
printterms (trmtype t)
{
   vlength   i;
   nameindex j;

   for (i = 1; i <= vocabulary.length; i++)
      if (vocabulary.symbols[i]->ttype == t) 
      {
         fputs ("     ", list);

         if (vocabulary.symbols[i]->ttype == TOKEN) 
         {
            fputc ('"', list);
            p2symbol (list, i);
            fputs ("\"  ", list);
         }

         else
            psymbol (list, i, FALSE);

         if (vocabulary.symbols[i]->ttype != NONTERMINAL)
            fprintf (list, "      %3d", i);
         fputc ('\n', list);

         if (++linenum > MAXLINES) page();

      }
}

/************************************************************************
* Print the vocabulary
************************************************************************/

static void
pvocab (void)
{
   int i;

   page();
   fprintf (list, "%10.10sVocabulary\n\n", "");
   linenum += 2;

   /* write out the nonterminals */
   fprintf (list, "\n    Nonterminals\n");
   linenum++;
   printterms (NONTERMINAL);

   /* write out the tokens */
   fprintf (list, "\n    Terminals\n");
   fprintf (list, "      Name");
   for (i = 0; i < symlen; i++) fputc (' ', list);
   fprintf (list, "Type #\n");
   linenum += 3;
   printterms (TERMINAL);

   /* write out the tokens */
   fprintf (list, "\n    Tokens\n");
   fprintf (list, "      Token     Type #\n");
   linenum += 3;
   printterms (TOKEN);

}

/***********************************************************************
* Find kernal item.
***********************************************************************/

static void
kernel (symbolindex focussymbol,
	stateindex markstate)
{
   itemindex i;

   for (i = firstitem(markstate); i <= lastitem(markstate); i++)
      if (marker(i) <= itemlen(i))
         if (focussymbol == dotsymbol(i))
            additem (itemprod(i), marker(i) + 1);
}

/************************************************************************
* Construct LR(0) grammar.
************************************************************************/

static void
lr0con (void)
{
   symbolindex focussymbol;
   stateindex  markstate;
   itemindex   i;

   /* Initialization */
   ilr0state();
   igotograph();

   if (listflag) 
   {
      page();
      fprintf (list, "%10.10sGoto Graph\n\n", "");
      fprintf (list, "     Oldstate   Symbol    Newstate\n");
      linenum += 3;
   }

   markstate = 1;
   while (markstate <= numstates()) 
   {
      for (focussymbol = 1; focussymbol <= vocablen(); focussymbol++) 
      {
         kernel (focussymbol, markstate);
         sclose();
         if (lastitem(openstate()) >= firstitem(openstate()))
            if ((gotostate(openstate(),markstate,focussymbol)
                       == openstate())) addopenstate();
            else deleteopenstate();
      }
      markstate++;
   }
}

/************************************************************************
* Print LR(0) state configurations.
************************************************************************/

static void
pconfig (void)
{
   stateindex s;
   itemindex  i;

   page();
   fprintf (list, "%10.10sLR(0) State Configurations\n", "");
   linenum++;

   for (s = 1; s <= numstates(); s++) 
   {
      if (linenum+2 > MAXLINES) page();
      else
      {
         fprintf (list, "\n");
         linenum++;
      }
      fprintf (list, 
         "State %4d       First Item: %4d       Last Item: %4d\n",
         s, firstitem(s), lastitem(s));
      linenum++;
      for (i = firstitem(s); i <= lastitem(s); i++) 
      {
         printi (itemprod(i), marker(i));
      }
   }

}

/***********************************************************************
* Emit parse table header.
***********************************************************************/

static void
begintables (void)
{
   fprintf (ptables, 
   "/*****************************************************************\n");
   fprintf (ptables, "* Parser tables, Generated at %s",
        ctime (&loctim));
   fprintf (ptables, 
   "*****************************************************************/\n");
}

/***********************************************************************
* Print the token type equates in the language parse tables.
***********************************************************************/

static void
pequates (void)
{
   vlength   i;
   nameindex j;

   for (i = 1; i <= vocabulary.length; i++)
      if (vocabulary.symbols[i]->ttype == TERMINAL) 
      {
         fprintf (ptables, "#define ");
         psymbol (ptables, i, TRUE);
         fprintf (ptables, "   %d\n", i);
         fprintf (tokens, "#define ");
         psymbol (tokens, i, TRUE);
         fprintf (tokens, "   %d\n", i);
      }
      /**********
      **else if (vocabulary.symbols[i]->ttype == TOKEN) 
      **{
      **   fprintf (ptables, "T%d = ^A'", i);
      **   p2symbol (ptables, i);
      **   fputs ("'\n", ptables);
      **}
      **********/
}

/***********************************************************************
* Emit action table header.
***********************************************************************/

static void
actionheader (void)
{
   fprintf (ptables, "/*\n");
   fprintf (ptables, "** Parser action macros\n");
   fprintf (ptables, "*/\n");
   fprintf (ptables, "#define SHIFT(co,ar)  (ar<<8|co)\n");
   fprintf (ptables, "#define REDUCE(co)    (co<<8|255)\n");
   fprintf (ptables, "#define ERROR         -1\n");
   fprintf (ptables, "#define GOTO(c,n)     (n<<8|(c&255))\n");

   fprintf (ptables, "\n/*\n");
   fprintf (ptables, "** Parser token type equates\n");
   fprintf (ptables, "*/\n");
   pequates();

   fprintf (ptables, "\n/*\n");
   fprintf (ptables, "** SLR(1) parser action tables\n");
   fprintf (ptables, "*/\n");
}

/***********************************************************************
* Emit parse header.
***********************************************************************/

static void
parseheader (void)
{
   int        i;
   stateindex s;

   fprintf (ptables, "};\n\n");
   fprintf (ptables, "static short int const *parsetable[] = {\n");
   i = 0;
   fprintf (ptables, "       ");
   for (i = 0, s = 1; s < numstates(); i++, s++) 
   {
      fprintf (ptables, "P%d, ", s);
      if (i == 10) 
      {
         fprintf (ptables, "\n       ");
         i = 0;
      }
   }
   fprintf (ptables, "P%d\n};\n\n", numstates());
   fprintf (ptables, "/*\n");
   fprintf (ptables, "** SLR(1) parser goto tables\n");
   fprintf (ptables, "*/\n");
}

/***********************************************************************
* Emit state header.
***********************************************************************/

static void
state (stateindex s)
{
   if (s > 1)
      fprintf (ptables, "};\n");
   fprintf (ptables, "static short int P%d[] = {\n", s);

   fprintf (errors, "case %d:\n", s);
   fprintf (errors, "   fprintf (stderr, \"Missing ");
}

/***********************************************************************
* Emit "SHIFT" entry.
***********************************************************************/

static void
shift (symbolindex u,
       stateindex s)
{
   trmtype t;

   t = vocabulary.symbols[u]->ttype;
   fprintf (ptables, "       SHIFT(");
   if (t == TOKEN) 
   {
      fputc ('\'', ptables);
      p2symbol (ptables, u);
      fputc ('\'', ptables);
      fputc ('\'', errors);
      p2symbol (errors, u);
      fputc ('\'', errors);
   }
   else
   {
      p2symbol (ptables, u);
      p2symbol (errors, u);
   }
   fprintf (ptables, ",%d),\n", s);

   fprintf (errors, ", ");
}

/***********************************************************************
* Emit "REDUCE" entry.
***********************************************************************/

static void
reduce (symbolarg u,
        prodindex p)
{
   trmtype t;

   fprintf (ptables, "       REDUCE(");
   if (u != -1) 
   {
      t = vocabulary.symbols[u]->ttype;
      if (t == TOKEN) fprintf (ptables, "%d", u);
      else p2symbol (ptables, u);
      fprintf (ptables, ",");
   }
   fprintf (ptables, "%d)\n", p);

   fprintf (errors, "\");\n");
   fprintf (errors, "   break;\n");
}

/***********************************************************************
* Emit "ERROR" entry.
***********************************************************************/

static void
error (void)
{
   fprintf (ptables, "       ERROR\n");

   fprintf (errors, "\");\n");
   fprintf (errors, "   break;\n");
}

/***********************************************************************
* Emit goto table header.
***********************************************************************/

static void
gotoheader (void)
{
   prodindex p;

   fprintf (ptables, "\nstatic struct {\n");
   fprintf (ptables, "   short int *go;\n");
   fprintf (ptables, "   int  handle;\n");
   fprintf (ptables, "} const gototable[] = {\n");
   for (p = 1; p < gramlen(); p++) 
   {
      fprintf (ptables, "   /* G%d */ ", p);
      p2symbol (ptables, lhsnonterminal(p));
      fprintf (ptables, ",%d,\n", prodlen(p));
   }
   fprintf (ptables, "   /* G%d */ ", gramlen());
   p2symbol (ptables, lhsnonterminal(gramlen()));
   fprintf (ptables, ",%d\n", prodlen(gramlen()));
   fprintf (ptables, "};\n");
}

/***********************************************************************
* Emit left hand side.
***********************************************************************/

static void
lhs (prodindex p)
{
   fprintf (ptables, "static short int ");
   p2symbol (ptables, lhsnonterminal(p));
   fprintf (ptables, "[] = {\n");
}

/***********************************************************************
* Emit goto states.
***********************************************************************/

static void
pgoto (statearg c,
       statearg n)
{
   if (c < 0)
      fprintf (ptables, "       GOTO(%d,%d)\n};\n", c, n);
   else
      fprintf (ptables, "       GOTO(%d,%d),\n", c, n);
}

/***********************************************************************
* Emit ".end"
***********************************************************************/

static void
endtables (void)
{
}

/************************************************************************
*
* Meta code:
*
* 1. For each A(K) set of LR(0) items do:
*
*     For the null string (LAMBDA) and then for each U an element of
*     the set of vocabulary terminals do the following:
*     1.1  For each item I an element of A(K) that is of the form
*          "A := TERM _ TERMINAL"  (shift item) and is such that
*          "TERMINAL" equals the symbol U; add the parse action
*          entry: "SHIFT U,GOTO(A(K),U)"
*     1.2  For each item I an element of A(K) that is of the form
*          "A := TERM _ ;" (reduce item) and is such that U is
*          contained in the set "LOOKAHEAD1(A)"; do the following:
*          1.2.1  Check for shift-reduce and reduce-reduce
*               conflicts between this item and the previously
*               visited items. Elicit messages indicating any
*               conflicts.
*          1.2.2   Print the parse action entry:
*               "REDUCE U,PROD#("A := TERM _ ;")"
*     1.3  Print the default parse action entry "ERROR" to end the
*          action entries for the state set A(K).
*
* 2.   Print the reduce-goto table from the goto graph.
*
************************************************************************
*
* The action table is as follows:
*
*            15         8  7  6      0
*  Format:   +-----------+---+--------+
*            !           !   !        !
*            +-----------+---+--------+
*                    ^     ^     ^
*                    |     |     |
*                    |     |     +---- TERMINAL TYPE OR TOKEN
*                    |     +---------- ACTION 0=SHIFT/1=REDUCE
*                    +---------------- NEXT STATE/PRODUCTION
*
************************************************************************
*
* Assembly language macros :
*
* ; SLR(1) PARSER ACTION TABLES
*     .address S1
*     .address S2
*     .address S<N>
*     ...
*
* S1:
*     SHIFT     <TOKEN>,<NEXT STATE>
*     ERROR
* S2:
*     SHIFT     <TOKEN>,<NEXT STATE>
*     REDUCE     <TOKEN>,<PROD NUMBER>
*     ...
*
* ; SLR(1) PARSER GOTO TABLE
*
*     .long     <LEFT HAND SIDE>,<RIGHT HAND SIDE LENGTH>
*     .long     <LEFT HAND SIDE>,<RIGHT HAND SIDE LENGTH>
*     ...
*
* <LEFT HAND SIDE>:
*     .byte     <CURRENT STATE>,<NEXT STATE>
*     ...
*
************************************************************************/

static void
slr1con (void)
{
   stateindex s;
   prodindex p;
   itemindex i;
   vlength u;
   gotoindex g;
   int j, k, tmp, ndx;
   boolean red_flg, laempty;

   begintables();

   /* Action table */
   actionheader();

   page();
   fprintf (list, "%10.10sSLR(1) Parsing Conflicts\n\n", "");
   linenum += 2;

   /* for each A(K) set of LR(0) items: */
   for (s = 1; s <= numstates(); s++) 
   {
      state (s);
      /* scan for shift actions */
      unclose();
      for (i = firstitem(s); i <= lastitem(s); i++)
         if (marker(i) != itemlen(i) + 1) 
         {
            u = dotsymbol(i);
            if ((termtype(u) != NONTERMINAL) && !isclosed(u)) 
            {
               shift (u, newstate(u,s));
               closesymbol(u);
            }
         }
      /* scan for reduce actions */
      red_flg = FALSE;
      for (ndx = 1; ndx <= MAXSQUEEZE; ndx++) 
      {
         reduce_lst[ndx].cnt = 0;
         ptr_lst[ndx] = ndx;
      }
      ndx = 0;
      for (i = firstitem(s); i <= lastitem(s); i++)
         if (marker(i) == itemlen(i) + 1) 
         {
            red_flg = TRUE;
            lookaheadset (i);
            conflicts (s, i);
            laempty = TRUE;
            p = itemprod(i);
            for (u = 1; u <= vocablen(); u++)
               if (termtype(u) != NONTERMINAL)
                  if (isinla(u)) 
                  {
                     if (ndx != 0) 
                     {
                        k = 1;
                        while (reduce_lst[k].prd != p && k <= ndx) k++;
                        if (k > ndx) ndx = k;
                     }
                     else 
                     {
                        k = 1;
                        ndx = 1;
                     }
                     reduce_lst[k].cnt++;
                     reduce_lst[k].prd = p;
                     reduce_lst[k].syms[reduce_lst[k].cnt] = u;
                     laempty = FALSE;
                  }
                  if (laempty) reduce (-1, p);
                  else if (ndx == 1)
                     reduce (-1, reduce_lst[ndx].prd);
                  else 
                  {
                     for (j = 1; j <= ndx; j++)
                        for (k = 1; k <= ndx-1; k++)
                           if (reduce_lst[ptr_lst[k]].cnt >
                              reduce_lst[ptr_lst[k+1]].cnt) 
                           {
                              tmp = ptr_lst[k];
                              ptr_lst[k] = ptr_lst[k+1];
                              ptr_lst[k+1] = tmp;
                           }
                           for (k = 1; k <= ndx; k++)
                              if (k != ndx)
                                 for (tmp = 1;
                                       tmp <= reduce_lst[ptr_lst[k]].cnt;
                                       tmp++)
                                    reduce (reduce_lst[ptr_lst[k]].syms[tmp],
                                             reduce_lst[ptr_lst[k]].prd);
                              else reduce (-1, reduce_lst[ptr_lst[k]].prd);
                  }
         }

         /* default parse action entry */
         if (!red_flg) error();
   }

   /* Emit the parser table header */
   parseheader();

   /* Generate the parser goto tables */
   lhs(1);
   pgoto (-1, 0);
   u = 0;
   for (p = 2; p <= gramlen(); p++)
      if (u != lhsnonterminal(p)) 
      {
         u = lhsnonterminal(p);
         lhs(p);
         for (ndx = 1; ndx <= MAXSQUEEZE; ndx++) 
         {
            goto_lst[ndx].cnt = 0;
            ptr_lst[ndx] = ndx;
         }
         ndx = 0;
         for (g = 1; g <= gotolength(); g++)
            if (gotoexists(g,u)) 
            {
               s = nextstate(g);
               if (ndx != 0) 
               {
                  k = 1;
                  while (goto_lst[k].nxtsta != s && k <= ndx) k++;
                  if (k > ndx) ndx = k;
               }
               else 
               {
                  k = 1;
                  ndx = 1;
               }
               goto_lst[k].cnt++;
               goto_lst[k].nxtsta = s;
               goto_lst[k].oldsta[goto_lst[k].cnt] = currentstate(g);
            }
            if (ndx == 1) pgoto (-1, goto_lst[ndx].nxtsta);
            else 
            {
               for (j = 1; j <= ndx; j++)
                  for (k = 1; k <= ndx-1; k++)
                     if (goto_lst[ptr_lst[k]].cnt >
                        goto_lst[ptr_lst[k+1]].cnt) 
                     {
                        tmp = ptr_lst[k];
                        ptr_lst[k] = ptr_lst[k+1];
                        ptr_lst[k+1] = tmp;
                     }
               for (k = 1; k <= ndx; k++)
                  if (k != ndx)
                     for (tmp = 1; tmp <= goto_lst[ptr_lst[k]].cnt; tmp++)
                         pgoto (goto_lst[ptr_lst[k]].oldsta[tmp],
                                 goto_lst[ptr_lst[k]].nxtsta);
                  else pgoto (-1, goto_lst[ptr_lst[k]].nxtsta);
            }
      }

   /* Emit goto tables header */
   gotoheader();

   /* Emit the trailers */
   endtables();
}

/************************************************************************
* chat main procedure
************************************************************************/

int
main (int argc,
      char *argv[])
{
   char *arg;

   pagenum = 0;
   time (&loctim);

   listflag = FALSE;
   checkflag = FALSE;
   eofflg = FALSE;
   verbose = FALSE;

   errcnt = 0;
   filename[0] = '\0';

   /* process parameters */
   for (i = 1; i < argc; i++) 
   {
      arg = argv[i];
      if (*arg == '-') 
      {
         arg++;
         switch (TOLOWER (*arg)) 
         {
            case 'c':
               checkflag = TRUE;
               break;
            case 'l':
               listflag = TRUE;
               break;
            case 'v':
               verbose = TRUE;
               break;
            default:
usage:         fprintf (stderr, "usage: chat bnf_file [-c][-l][-v]\n");
               return (ABORT);
         }
      }
      else if (isalpha (*arg))
         strcpy (filename, arg);
      else
         goto usage;
   }

#if defined(IBMMVS)

   source = stdin;

#else

   if (filename[0] == '\0') 
   {
      puts ("Enter BNF filename: ");
      fgets (filename, sizeof (filename), stdin);
   }

   /* Initialize I/O files */
   
   if ((arg = strchr (filename, '.')) == NULL) 
   {
      sprintf (pathname, "%s.bnf", filename);
   }
   else 
   {
      strcpy (pathname, filename);
      *arg = '\0';
   }

   source = fopen (pathname, "r");
      
#endif

   if (source != NULL) 
   {

#if defined(IBMMVS)

      list    = fopen ("DD:LISTING", "w");
      semant  = fopen ("DD:SEMANT",  "w");
      ptables = fopen ("DD:PTABLES", "w");
      errors  = fopen ("DD:ERRORS",  "w");
      tokens  = fopen ("DD:TOKENS",  "w");

#else

      sprintf (tempstring, "%s.lst", filename);
      list = fopen (tempstring, "w");
      sprintf (tempstring, "%s.sem", filename);
      semant = fopen (tempstring, "w");
      sprintf (tempstring, "%s.ptb", filename);
      ptables = fopen (tempstring, "w");
      sprintf (tempstring, "%s.err", filename);
      errors  = fopen (tempstring, "w");
      sprintf (tempstring, "%s.tok", filename);
      tokens  = fopen (tempstring, "w");

#endif

      bad = FALSE;
      symlen = 0;
      if (verbose) 
      {
         printf ("CHAT - Parser table generator - version 5.0\n");
         printf ("Read BNF language description\n");
      }
      scanner();

      if (!bad) 
      {

         if (verbose) printf ("Print the grammar\n");
         pgrammar();

         if (verbose) printf ("Print the vocabulary\n");
         pvocab();

         if (!checkflag) 
         {

            if (verbose) printf ("Construct LR(0) item sets\n");
            lr0con();

            if (listflag) 
            {
               if (verbose) printf ("Print LR(0) state configurations\n");
               pconfig();
            }

            if (verbose) printf ("Construct SLR(1) tables\n");
            slr1con();

         }
      }

      fclose (ptables);
      fclose (semant);
      fclose (errors);
      fclose (tokens);
      fprintf (list, "\n %d errors\n", errcnt);
      fclose (list);
      if (verbose) printf ("%d errors\n", errcnt);
      /** fclose (source); **/
   }

   else 
   {
      perror (tempstring);
      return (ABORT);
   }

   return (NORMAL);

}
