/***********************************************************************
*
* lnkdef.h - Linkage editor header for the TI 990 computer.
*
* Changes:
*   06/05/03   DGP   Original.
*   06/25/03   DGP   Added Module type and associated formats.
*   05/25/04   DGP   Added Long entry/extern tags.
*   04/27/05   DGP   Put IDT in record sequence column 72.
*   12/30/05   DGP   Changed SymNode to use flags.
*   01/21/07   DGP   Added scansymbols() for non-listmode.
*   01/23/07   DGP   Added library support.
*   12/02/10   DGP   Added EXTNDX support.
*   12/07/10   DGP   Added ABSDEF flag.
*   12/14/10   DGP   Boost MAXFILES and MAXMODULES to 1000.
*   01/07/14   DGP   Changed for lnked990.
*   01/28/15   DGP   Added NOTGLOBAL, ALLGLOBAL, GLOBAL support.
*   02/04/15   DGP   Fixed partial link to correct EXTNDX tag processing.
*   10/09/15   DGP   Added Long symbol common tags.
*
***********************************************************************/

#include <stdio.h>

/*
** Definitions
*/

#define NORMAL 0
#define ABORT  12

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#define VERSION "1.3.5"

#define EOFSYM ':'

#define MAXLINE 256
#define LINESPAGE 60
#define MAXFILES 1000
#define IDTSIZE 8

#define MAXCSEGS 50
#define MAXPROCS 2
#define MAXPHASES 50
#define MAXSYMBOLS 2000

#define MAXSYMLEN 32
#define MAXSHORTSYMLEN 6

#define MEMSIZE 65768

#define MAXPATHNAMESIZE 256

/*
** Object output formats
*/

#define OBJFORMAT "%c%04X"
#define IDTFORMAT "%-8.8s"
#define REFFORMAT "%-6.6s"
#define LREFFORMAT "%-32.32s"
#define WORDFORMAT "%04X"
#define SEQFORMAT "%4.4d\n"

/*
** Listing formats
*/

#define H1FORMAT "LNKED990 %-8.8s %-24.24s %s %s Page %04d\n\n"

#define WMODFORMAT "%-8.8s %3d   %04X    %04X   %s  %s  %s  %-60.60s\n"
#define MODFORMAT  "%-8.8s %3d   %04X    %04X   %s  %s  %s  %-19.19s\n"
#define CMNFORMAT "%-8.8s %3d   %04X    %04X\n"
#define SYMFORMAT "%%-%d.%ds %%4.4X%%c%%c %%3d"
#define UNDEFFORMAT "%%-%d.%ds %%3d  "

/*
** Data type definitions
*/

#define int8            char
#define int16           short
#define int32           int
typedef int             t_stat;                         /* status */
typedef int             t_bool;                         /* boolean */
typedef unsigned int8   uint8;
typedef unsigned int16  uint16;
typedef unsigned int32  uint32, t_addr;                 /* address */

#if defined (WIN32)                                     /* Windows */
#define t_int64 __int64
#elif defined (__ALPHA) && defined (VMS)                /* Alpha VMS */
#define t_int64 __int64
#elif defined (__ALPHA) && defined (__unix__)           /* Alpha UNIX */
#define t_int64 long
#else                                                   /* default GCC */
#define t_int64 long long
#endif                                                  /* end OS's */
typedef unsigned t_int64        t_uint64, t_value;      /* value */
typedef t_int64                 t_svalue;               /* signed value */

#define MSB 0
#define LSB 1

/*
** Memory access macros
*/

#define GETMEM(m) \
(((memory[(m)+MSB] & 0xFF) << 8)|(memory[(m)+LSB] & 0xFF))

#define PUTMEM(m,v) \
{ memory[(m)+MSB] = (v) >> 8 & 0xFF; memory[(m)+LSB] = (v) & 0xFF; }

#define GETDSEGMEM(m) \
(((dsegmem[(m)+MSB] & 0xFF) << 8)|(dsegmem[(m)+LSB] & 0xFF))

#define PUTDSEGMEM(m,v) \
{ dsegmem[(m)+MSB] = (v) >> 8 & 0xFF; dsegmem[(m)+LSB] = (v) & 0xFF; }

#define GETCSEGMEM(m) \
(((csegmem[(m)+MSB] & 0xFF) << 8)|(csegmem[(m)+LSB] & 0xFF))

#define PUTCSEGMEM(m,v) \
{ csegmem[(m)+MSB] = (v) >> 8 & 0xFF; csegmem[(m)+LSB] = (v) & 0xFF; }

/*
** Module table
*/

typedef struct ModuleNode_t
{
   struct ModuleNode_t *next;
   char name[MAXSYMLEN];
   char date[12];
   char time[12];
   char creator[12];
   char objfile[MAXPATHNAMESIZE+2];
   char fullfile[MAXPATHNAMESIZE+2];
   int dsegmodule;
   int libmodule;
   int origin;
   int length;
   int number;
   int adjust;
   int symcount;
   off_t offset;
   struct SymNode_t *symbols[MAXSYMBOLS];
} Module;

/*
** CSEG table
*/

typedef struct
{
   char name[MAXSYMLEN];
   int length;
   int origin;
   int offset;
} CSEGNode;

typedef struct CSEGUserNode_t
{
   struct CSEGUserNode_t *next;
   char name[MAXSYMLEN];
   int baseorigin;
   int origin;
   int length;
   int number;
} CSEGUserNode;

/*
** File node
*/

typedef struct 
{
   char name[MAXPATHNAMESIZE+2];
   int library;
   off_t offset;
} FILENode;

/*
** Phase node
*/

typedef struct PhaseNode_t
{
   struct PhaseNode_t *next;
   Module *module_head;
   Module *module_tail;
   CSEGUserNode *cseg_head;
   CSEGUserNode *cseg_tail;
   char name[MAXSYMLEN];
   int dummy;
   int number;
   int length;
   int origin;
   int end;
   int adjust;
   int dseguserorg;
   int dsegorg;
   int dseglen;
   int cseguserorg;
   int csegorg;
   int cseglen;
   int cmncount;
   int symcount;
   int refsymcount;
   CSEGNode cmnnodes[MAXCSEGS];
   struct SymNode_t *symbols[MAXSYMBOLS];
   struct SymNode_t *refsymbols[MAXSYMBOLS];
} PHASENode;

typedef struct
{
   PHASENode *phase;
   int pindex;
   int origin;
   int length;
} PHASEControl;

/*
** Symbol table
*/

typedef struct EXTndx_t
{
    struct EXTndx_t *next;
    int location;
} EXTndxlist;

typedef struct SymNode_t
{
   PHASENode *phase;
   EXTndxlist *extlist;
   char symbol[MAXSYMLEN+2];
   int flags;
   int value;
   int modnum;
   int extndx;
   int cmnndx;
} SymNode;

#define RELOCATABLE 0x000000001
#define EXTERNAL    0x000000002
#define GLOBAL      0x000000004
#define LONGSYM     0x000000008
#define MULDEF      0x000000010
#define UNDEF       0x000000020
#define DSEG        0x000000040
#define CSEG        0x000000080
#define ABSDEF      0x000000100
#define SREF        0x000000200
#define LOAD        0x000000400
#define SYMTAG      0x000001000
#define DEFGLOBAL   0x000010000

typedef struct
{
   SymNode *sym;
   int extndx;
} EXTList;

typedef struct
{
   SymNode *sym;
   char tag;
   int	value;
} MemCTL;

/*
** Control file command definitions.
*/

enum ctlops
{
   CMD_ILLEGAL = 0, CMD_ADJUST,      CMD_ALLGLOBAL,   CMD_ALLOCATE,
   CMD_COMMON,      CMD_DATA,        CMD_DUMMY,       CMD_END,
   CMD_ERROR,       CMD_FIND,        CMD_FORMAT,      CMD_GLOBAL,
   CMD_INCLUDE,     CMD_LIBRARY,     CMD_LOAD,        CMD_MAP,
   CMD_NOAUTO,      CMD_NOERROR,     CMD_NOLOAD,      CMD_NOMAP,
   CMD_NOPAGE,      CMD_NOSYMT,      CMD_NOTGLOBAL,   CMD_PAGE,
   CMD_PARTIAL,     CMD_PHASE,       CMD_PROCEDURE,   CMD_PROGRAM,
   CMD_SEARCH,      CMD_SHARE,       CMD_SYMT,        CMD_TASK
};

typedef struct
{
   char *command;
   int cmdvalue;
} Command;

/*
** Object tags
*/

#define BINIDT_TAG	01
#define IDT_TAG		'0'
#define ABSENTRY_TAG	'1'
#define RELENTRY_TAG	'2'
#define RELEXTRN_TAG	'3'
#define ABSEXTRN_TAG	'4'
#define RELGLOBAL_TAG	'5'
#define ABSGLOBAL_TAG	'6'
#define CKSUM_TAG	'7'
#define NOCKSUM_TAG	'8'
#define ABSORG_TAG	'9'
#define RELORG_TAG	'A'
#define ABSDATA_TAG	'B'
#define RELDATA_TAG	'C'
#define LOADBIAS_TAG	'D'
#define EXTNDX_TAG	'E'
#define EOR_TAG		'F'

#define RELSYMBOL_TAG	'G'
#define ABSSYMBOL_TAG	'H'
#define PGMIDT_TAG	'I'
#define CMNSYMBOL_TAG	'J'
#define COMMON_TAG	'M'
#define CMNDATA_TAG	'N'
#define CMNORG_TAG	'P'
#define CBLSEG_TAG	'Q'
#define DSEGORG_TAG	'S'
#define DSEGDATA_TAG	'T'
#define LOAD_TAG	'U'
#define RELSREF_TAG	'V'
#define CMNGLOBAL_TAG	'W'
#define CMNEXTRN_TAG	'X'
#define ABSSREF_TAG	'Y'
#define CMNSREF_TAG	'Z'

#define LRELSYMBOL_TAG	'i'
#define LABSSYMBOL_TAG	'j'
#define LRELEXTRN_TAG	'l'
#define LABSEXTRN_TAG	'm'
#define LRELGLOBAL_TAG	'n'
#define LABSGLOBAL_TAG	'o'
#define LLOAD_TAG	'p'
#define LRELSREF_TAG	'q'
#define LABSSREF_TAG	'r'
#define LCMNGLOBAL_TAG	'w'
#define LCMNEXTRN_TAG	'x'
#define LCMNSREF_TAG	'z'

#define CHARSPERREC	67  /* Chars per object record */
#define CHARSPERCASREC	73  /* Chars per cassette object record */
#define WORDTAGLEN	5   /* Word + Object Tag length */
#define BINWORDTAGLEN	3   /* Word + Binary Object Tag length */
#define EXTRNLEN	11  /* Tag + SYMBOL + addr */
#define GLOBALLEN	11  /* Tag + SYMBOL + addr */
#define IDTLEN		13  /* Tag + IDT + length */
#define BINIDTLEN	11  /* Tag + IDT + binary length */
#define IDTCOLUMN	72  /* Where to put the IDT */
#define SEQUENCENUM	76  /* Where to put the sequence */

#define TIMEOFFSET 17
#define DATEOFFSET 27
#define CREATOROFFSET 38


/*
** External functions.
*/

extern int lnkloader (PHASENode *, FILE *, int, char *, int);
extern uint16 getnumfield (unsigned char *, int);
extern void processdef (PHASENode *, Module *, char *, int, int, uint16, int, int);

extern int lnklibrary (PHASENode *, FILE *, int, char *, char *);

extern int lnkpunch (PHASENode *, int);

extern int cmdlookup (char *);
extern char *gettoken (char *, char *, char *);

extern SymNode *symlookup (PHASENode *, Module *, char *, int);
extern SymNode *reflookup (PHASENode *, Module *, char *, int);
extern SymNode *modsymlookup (PHASENode *, Module *, char *, int);
extern CSEGUserNode *cseglookup (PHASENode *, char *, int, int, int, int, int *);
