/************************************************************************
*
* lnked990 - Links objects from asm990, uses SDSLNK control file.
*
* Changes:
*   01/07/14   DGP   Original lnked990 version. Hacked from lnk990
*   01/29/14   DGP   Revamp symbols and EXTTAG references.
*   05/02/14   DGP   Added NOSEQ to FORMAT ASCII.
*   10/15/14   DGP   Return -1 on non-fatal loader error.
*   01/28/15   DGP   Fixed segment update to not overflow value.
*   01/28/15   DGP   Added NOTGLOBAL, ALLGLOBAL, GLOBAL support.
*   02/04/15   DGP   Fixed partial link to correct EXTNDX tag processing.
*   02/10/15   DGP   Move reftoextndx() here and call at end of phase.
*   12/10/15   DGP   Added a.out format.
*   12/13/16   DGP   Added program too big detection.
*   12/30/16   DGP   Fixed symbol print.
*   01/24/17   DGP   Fixed defered Common relocations.
*	
************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>

#include "lnkdef.h"

FILE *ctlfd = NULL;
FILE *outfd = NULL;
FILE *mapfd = NULL;

int pc; /* the linker pc */
int absolute = FALSE;
int partiallink = FALSE;
int errcount = 0;
int warncount = 0;
int pgmlength = 0;
int absentry = -1;
int relentry = -1;
int phasenum = -1;
int linecnt = MAXLINE;
int modcount = 0;
int modnumber = 0;
int undefs = FALSE;
int muldefs = FALSE;
int cassette = FALSE;
int compressed = FALSE;
int aoutformat = FALSE;
int curmod = 0;
int extndx = 0;
int inpass = 0;
int libcnt = 0;
int phasecnt = 0;
int proccnt = 0;

char idtname[IDTSIZE+2];

struct tm *timeblk;

FILENode libfiles[MAXFILES];

PHASENode phases[MAXPHASES];
PHASENode procs[MAXPROCS];
PHASEControl pcontrol[MAXPHASES];

uint8 memory[MEMSIZE];
MemCTL memctl[MEMSIZE];
uint8 dsegmem[MEMSIZE];
MemCTL dsegctl[MEMSIZE];
uint8 csegmem[MEMSIZE];
MemCTL csegctl[MEMSIZE];

static int nomap = FALSE;
static int nopage = FALSE;
static int nosymt = FALSE;
static int noauto = FALSE;
static int widelist = FALSE;
static int noerror = TRUE;
static int pagenum = 0;
static int undefcount = 0;
static int muldefcount = 0;
static int maxlength;
static char datebuf[48];
static char errorline[120];
static char spaces[41];

/***********************************************************************
* printheader - Print header on listing.
***********************************************************************/

static void
printheader (void)
{
   if (linecnt >= LINESPAGE)
   {
      linecnt = 0;
      if (pagenum) fputc ('\f', mapfd);
      fprintf (mapfd, H1FORMAT, VERSION, idtname,
	       widelist ? spaces : "", datebuf, ++pagenum);
   }
}

/***********************************************************************
* printsymbols - Print the phase symbol table.
***********************************************************************/

static void
printsymbols (PHASENode *phase)
{
   SymNode *sym;
   int i, j;
   int maxsymlen;
   int symperline;
   int hdrout = FALSE;
   char type;
   char type1;
   char format[40];

   maxsymlen = 6;
   for (i = 0; i < phase->symcount; i++)
   {
      sym = phase->symbols[i];
      if (sym->flags & LONGSYM)
      {
	 if ((j = strlen (sym->symbol)) > maxsymlen)
	    maxsymlen = j;
      }
   }
   maxsymlen += 1;
   symperline = (widelist ? 132 : 80) / (12 + maxsymlen);
   sprintf (format, SYMFORMAT, maxsymlen, maxsymlen);

   j = 0;
   for (i = 0; i < phase->symcount; i++)
   {
      sym = phase->symbols[i];
      if (sym->phase != phase) continue;

      if (!hdrout)
      {
	 printheader();
	 fputs ("\n                        D E F I N I T I O N S \n\n", mapfd);
	 linecnt += 3;
	 hdrout = TRUE;
      }

      type = ' ';
      if (partiallink)
      {
	 if (sym->flags & GLOBAL) type = 'D';
	 else if (sym->flags & EXTERNAL) type = 'R';
      }
      else if (sym->flags & RELOCATABLE) type = '\'';

      type1 = ' ';
      if (sym->flags & MULDEF)
      {
	 muldefs = TRUE;
	 type1 = 'M';
	 warncount++;
	 muldefcount++;
      }
      else if (sym->flags & UNDEF)
      {
	 if (sym->value == 0) continue; /* NOT USED */
	 if (!partiallink && !(sym->flags & SREF))
	 {
	    continue; /* process it in printundefs */
	 }
      }

      fprintf (mapfd, format,
	       sym->symbol, sym->value, type, type1, sym->modnum);
      if (++j == symperline)
      {
	 fputc ('\n', mapfd);
	 linecnt++;
	 printheader();
	 j = 0;
      }
      else fputs ("  ", mapfd);
   }
   if (hdrout && j)
   {
      fputc ('\n', mapfd);
      linecnt++;
   }
}

/***********************************************************************
* printerror - Print error message.
***********************************************************************/

static int
printerror (char *msg, int error)
{
   printheader();
   fprintf (mapfd, "**%s: %s\n", error ? "ERROR" : "WARNING", msg);
   linecnt++;
   if (error)
      errcount++;
   else
      warncount++;
   if (noerror)
   {
      fprintf (stderr, "**%s: %s\n", error ? "ERROR" : "WARNING", msg);
   }
   return (-1);
}

/***********************************************************************
* printundefs - Print the udefined symbols in this phase.
***********************************************************************/

static void
printundefs (PHASENode *phase)
{
   SymNode *sym;
   int i, j;
   int hdrout = FALSE;
   int maxsymlen;
   int symperline;
   char format[40];

   maxsymlen = 6;
   for (i = 0; i < phase->symcount; i++)
   {
      sym = phase->symbols[i];
      if (sym->flags & LONGSYM)
      {
	 if ((j = strlen (sym->symbol)) > maxsymlen)
	    maxsymlen = j;
      }
   }
   maxsymlen += 1;
   symperline = (widelist ? 120 : 80) / (5 + maxsymlen);
   sprintf (format, UNDEFFORMAT, maxsymlen, maxsymlen);

   j = 0;
   for (i = 0; i < phase->symcount; i++)
   {
      sym = phase->symbols[i];
      if (sym->flags & UNDEF && (partiallink ? 1 : (sym->value != 0)))
      {
	 if (!(sym->flags & SREF))
	 {
	    if (!hdrout)
	    {
	       printheader();
	       fputs (
	   "\n                   U N R E S O L V E D   R E F E R E N C E S\n\n",
		       mapfd);
	       linecnt += 3;
	       hdrout = TRUE;
	    }
	    fprintf (mapfd, format, sym->symbol, sym->modnum);
	    undefs = TRUE;
	    if (!partiallink) warncount++;
	    undefcount++;
	    if (++j == symperline)
	    {
	       fputc ('\n', mapfd);
	       linecnt++;
	       printheader();
	       j = 0;
	    }
	 }
      }
   }
   if (hdrout && j)
   {
      fputc ('\n', mapfd);
      linecnt++;
   }
}

/***********************************************************************
* printphasemap - Process linkage map for this phase.
***********************************************************************/

static void
printphasemap (PHASENode *phase, char *mode)
{
   Module *module;
   char format[60];

   if (widelist)
      strcpy (format, WMODFORMAT);
   else
      strcpy (format, MODFORMAT);

   if (!nopage)
      linecnt = MAXLINE;

   /*
   ** Print the phase header.
   */

   printheader ();
   fprintf (mapfd, "%s%s %d, %-8.8s ORIGIN = %04X  LENGTH = %04X",
	    nopage ? "\n\n" : "",
	    mode, phase->number, phase->name, phase->origin, phase->length);
   if (phase->dummy)
      fputs (", DUMMY", mapfd);
   if (phase->number == 0 && relentry >= 0)
      fprintf (mapfd, "  ENTRY = %04X", relentry);
   fputc ('\n', mapfd);
   linecnt += nopage ? 3 : 1;

   if (!nomap)
   {
      /*
      ** Print the modules.
      */

      printheader ();
      fputs (
       "\nMODULE    NO  ORIGIN  LENGTH    DATE      TIME    CREATOR   FILE\n\n",
	     mapfd);
      linecnt += 2;

      for (module = phase->module_head; module; module = module->next)
      {
	 int modorigin;

	 if (partiallink && module->dsegmodule)
	    modorigin = 0;
	 else
	    modorigin = module->origin;
	 printheader ();
	 if ((strlen (module->objfile) > 18) && 
	     (module->objfile[0] == '/'))
	 {
	    char *ep = strrchr (module->objfile, '/');
	    if (ep)
	       strcpy (module->objfile, ep+1);
	 }
	 printheader();
	 fprintf (mapfd, format, module->name, module->number,
		  modorigin, module->length, module->date,
		  module->time, module->creator, module->objfile);
	 linecnt ++;
      }

      /*
      ** Print the COMMON areas.
      */

      if (phase->cseg_head)
      {
	 CSEGUserNode *node;

	 printheader();
	 fputs ("\nCOMMON    NO  ORIGIN  LENGTH\n\n", mapfd);
	 linecnt += 3;
	 for (node = phase->cseg_head; node; node = node->next)
	 {
	    printheader();
	    fprintf (mapfd, CMNFORMAT, node->name, node->number,
		     partiallink ? 0 : node->origin, node->length);
	    linecnt ++;
	 }
      }

      /*
      ** Print the symbols defined in this phase.
      */

      printsymbols (phase);
   }

   /*
   ** Print the undefined symbols in this phase.
   */

   printundefs (phase);
}

/***********************************************************************
* printlinkmap - Process linkage map.
***********************************************************************/

static void
printlinkmap (void)
{
   PHASENode *phase;
   int pndx;

   for (pndx = 0; pndx < proccnt; pndx++)
   {
      phase = &procs[pndx];
      strcpy (idtname, phase->name);
      printphasemap (phase, "PROCEDURE");
   }

   for (pndx = 0; pndx < phasecnt; pndx++)
   {
      phase = &phases[pndx];
      strcpy (idtname, phase->name);
      printphasemap (phase, "PHASE");
   }

   printheader ();
   fprintf (mapfd, "\n");
   linecnt ++;

   if (undefs)
   {
      printheader();
      fprintf (mapfd, "** %d Undefined symbols\n", undefcount);
      linecnt ++;
   }
   if (muldefs)
   {
      printheader();
      fprintf (mapfd, "** %d Multiply defined symbols\n", muldefcount);
      linecnt ++;
   }

   printheader();
   fprintf (mapfd, "%d Errors, %d Warnings\n", errcount, warncount);
}

/***********************************************************************
* updatesegs - Scan and update the C/DSEG symbol definitions.
***********************************************************************/

static void
updatesegs (PHASENode *phase, int segorigin, int seg)
{
   SymNode *sym;
   CSEGUserNode *node;
   int i;

#ifdef DEBUGSEGS
   printf ("updatesegs: seg = %s, symcount = %d, segorigin = %04X\n",
   	    seg==CSEG ? "CSEG" : "DSEG", phase->symcount, segorigin);
#endif

   if (seg == CSEG)
   {
      int neworigin = segorigin;

#ifdef DEBUGCOMMON
      printf ("updatesegs: CSEG: cmncount = %d, segorigin = %04X\n",
	       phase->cmncount, segorigin);
#endif
      for (node = phase->cseg_head; node; node = node->next)
      {
	 if (node->baseorigin < 0)
	 {
	    node->origin = neworigin;
	    neworigin += node->length;
#ifdef DEBUGCOMMON
	    printf ("common = %s, origin = %04X, length = >%04X\n",
		    node->name, node->origin, node->length);
#endif
	 }
      }

      for (i = 1; i < (phase->cmncount+1); i++)
      {
	 int j;

	 node = cseglookup (phase, phase->cmnnodes[i].name, 0, 0, 0, FALSE, &j);
	 if (node)
	 {
	    phase->cmnnodes[i].origin = node->origin;
	    phase->cmnnodes[i].length = node->length;
#ifdef DEBUGCOMMON
	    printf ("cmnodes[%d] = %s, origin = %04X, length = >%04X\n",
		     i, node->name, node->origin, node->length);
#endif
	 }
      }
   }

   for (i = 0; i < phase->symcount; i++)
   {
      sym = phase->symbols[i];

#ifdef DEBUGSEGS
      printf ("   flags = %04X, value = %04X, sym = %s, cmnndx = %d\n",
         sym->flags, sym->value, sym->symbol, sym->cmnndx);
#endif
      if ((sym->flags & seg) && (sym->flags & RELOCATABLE))
      {
	 if (sym->cmnndx == 0)
	    sym->value += segorigin;
	 else
	    sym->value += phase->cmnnodes[sym->cmnndx].origin;
	 sym->value &= 0xFFFF;
#ifdef DEBUGSEGS
	 printf ("      update: value = %04X\n", sym->value);
#endif
      }
   }

}

/***********************************************************************
* getnum - Get a number from the command line.
***********************************************************************/

static int
getnum (char *bp)
{
   char *cp;
   long i = 0;

   while (*bp && isspace (*bp)) bp++;
   cp = bp;
   if (*bp == '>' || *bp == '0')
      i = strtol (++bp, &cp, 16);
   else 
      i = strtol (bp, &cp, 0);
   return ((int)i & 0xFFFF);
}

/***********************************************************************
* openlibrary - Search Libraries.
***********************************************************************/

static FILE *
openlibrary (char *member, char *fullfile, char *templib)
{
   FILE *infd;
   int i;

#ifdef DEBUGLIBRARY
   printf ("openlibrary: member = (%s), templib = %s\n",
	   member, templib ? templib : "NULL");
#endif

   if (templib)
   {
      sprintf (fullfile, "%s/%s", templib, member);
      if ((infd = fopen (fullfile, "rb")) == NULL)
      {
	 sprintf (fullfile, "%s/%s.obj", templib, member);
	 infd = fopen (fullfile, "rb");
      }
   }
   else for (i = 0, infd = NULL; (i < libcnt) && !infd; i++)
   {
      sprintf (fullfile, "%s/%s", libfiles[i].name, member);
      if ((infd = fopen (fullfile, "rb")) == NULL)
      {
	 sprintf (fullfile, "%s/%s.obj", libfiles[i].name, member);
	 infd = fopen (fullfile, "rb");
      }
   }
#ifdef DEBUGLIBRARY
   if (infd)
      printf ("   fullfile = %s\n", fullfile);
#endif
   return (infd);
}

/***********************************************************************
* searchlibraries - Search Libraries.
***********************************************************************/

static void
searchlibraries (PHASENode *phase, char *templib)
{
   SymNode *sym;
   FILE *infd;
   int more;
   int i;

#ifdef DEBUGLIBRARY
   printf ("searchlibraries: pass = %d, libcnt = %d, templib = %s\n",
	 inpass, libcnt, templib ? templib : "NULL");
#endif

   if (!templib && !libcnt) return;
   if (inpass == 1)
   {
      char fullfile[MAXPATHNAMESIZE+2];

      do {
	 more = FALSE;
	 for (i = 0; i < phase->refsymcount; i++)
	 {
	    sym = phase->refsymbols[i];
	    if (sym->phase != phase) continue;
#ifdef DEBUGLIBRARY
	    printf ("   Name = %s, flags = %X\n", sym->symbol, sym->flags);
#endif
	    if (sym->flags & UNDEF && !(sym->flags & SREF))
	    {
	       if ((infd = openlibrary (sym->symbol, fullfile, templib))
		     != NULL)
	       {
		  char *bp;
		  char libname[MAXLINE+2];

		  if ((bp = strrchr (fullfile, '/')) != NULL)
		     strcpy (libname, bp+1);
		  else
		     strcpy (libname, fullfile);
		  if ((bp = strchr (libname, '.')) != NULL)
		     *bp = 0;

#ifdef DEBUGLIBRARY
		  printf ("   LOAD: origin = %04X, pc = %04X\n",
			  phase->origin, pc);
		  printf ("      fullfile = %s, libname = %s\n",
			  fullfile, libname);
#endif
		  pc = lnkloader (phase, infd, pc, libname, TRUE);
		  if (pc < 0)
		  {
		     fprintf (stderr, "Error: file = %s\n", fullfile);
		     exit (ABORT);
		  }
		  phase->module_tail->libmodule = TRUE;
		  phase->module_tail->offset = 0;
		  strcpy (phase->module_tail->fullfile, fullfile);
		  fclose (infd);
		  more = TRUE;
	       }
	    }
	 }
      } while (more);
   }

   else /* pass 2 */
   {
      Module *module;

      for (module = phase->module_head; module; module = module->next)
      {
	 if (module->libmodule)
	 {
	    int loadit;

	    loadit = TRUE;
	    if (templib)
	    {
	       char *sp, *ep;

	       sp = module->fullfile;
	       ep = strrchr (sp, '/');
	       if (ep && strncmp (sp, templib, (ep - 1) - sp))
		  loadit = FALSE;
	    }
	    if (loadit)
	    {
	       if ((infd = fopen (module->fullfile, "rb")) != NULL)
	       {
#ifdef DEBUGLIBRARY
		  printf ("   LOAD: Name = %s, origin = %04X, pc = %04X\n",
			  module->name, phase->origin, pc);
		  printf ("      fullfile = %s\n", module->fullfile);
#endif
		  pc = lnkloader (phase, infd, pc, module->objfile, TRUE);
		  if (pc < 0)
		  {
		     fprintf (stderr, "Error: file = %s\n", module->objfile);
		     exit (ABORT);
		  }
		  fclose (infd);
	       }
	    }
	 }
      }
   }
}

/************************************************************************
* reftoextndx - Convert REF backchain to EXTNDX for partial link.
************************************************************************/

static void
reftoextndx (PHASENode *phase)
{
   SymNode *s;
   int i, j;

   if (!partiallink || (inpass == 1)) return;

#if defined(DEBUGEXTNDX)
   printf ("reftoextndx: entered: phase.name = %s\n", phase->name);
#endif

   /*
   ** For partial link, convert back chain to EXTNDX.
   */

   j = 0;
   for (i = 0; i < phase->symcount; i++)
   {
      int ref;

      s = phase->symbols[i];
      if (s->flags & UNDEF)
      {
#if defined(DEBUGEXTNDX)
	 printf ("   fill ext symbol = %s, value = %04X\n",
		  s->symbol, s->value);
#endif
	 s->extndx = j;
	 ref = s->value;
	 while (ref)
	 {
	    int k;
	    k = GETMEM(ref);
	    PUTMEM (ref, j);
	    memctl[ref].tag = EXTNDX_TAG;
	    memctl[ref].value = 0;
	    memctl[ref].sym = s;
	    ref = k;
#if defined(DEBUGEXTNDX)
	    printf ("       ref = %04X\n", ref);
#endif
	 }
	 j++;
      }
   }
}

/***********************************************************************
* processend - Process end of a phase or procedure.
***********************************************************************/

static int
processend (PHASENode *phase)
{
   CSEGUserNode *node;
   int i;
   int status = 0;

   /* Length of CSEGs defined in this phase */

   phase->cseglen = 0;
   for (node = phase->cseg_head; node; node = node->next)
   {
      if (node->baseorigin < 0)
	 phase->cseglen += node->length;
   }

   if (inpass == 1)
   {
      pc = (pc + 1) & 0xFFFE;

      if (maxlength < pc)
      {
	 maxlength = pc + phase->dseglen + phase->cseglen;
      }
      phase->end = pc;
#ifdef DEBUGEND
      printf ("processend-1: pc = %04X, origin = %04X, end = %04X, "
              "length = %04X(%04X), adjust = %04X\n",
	      pc, phase->origin, phase->end, phase->length,
	      phase->end - phase->origin,
	      phase->adjust);
      printf ("   maxlength = %04X:%04X\n",
	       maxlength, maxlength - phases[0].origin);
#endif

      if (partiallink)
	 phase->dsegorg = 0;
      else
	 phase->dsegorg = phase->dseguserorg > 0 ? phase->dseguserorg : pc;
      if (phase->dseguserorg < 0)
      {
	 updatesegs (phase, phase->dsegorg, DSEG);
	 if (!partiallink)
	    pc += phase->dseglen;
      }
#ifdef DEBUGEND
      printf ("processend-1a: pc = %04X, dsegorg = %04X, dseglen = %04X\n",
	      pc, phase->dsegorg, phase->dseglen);
#endif
      pc = (pc + 1) & 0xFFFE;

      /*
      ** Set up common origins to requested base.
      */

      if ((node = phase->cseg_head) != NULL)
      {
	 while (node)
	 {
	    int baseorigin, origin;

	    origin = baseorigin = node->baseorigin;
	    if (baseorigin >= 0)
	    {
	       while (node && (node->baseorigin == baseorigin))
	       {
		  node->origin = origin;
		  origin += node->length;
#ifdef DEBUGCOMMON
		  printf ("processend-1b: COMMON: name = %s, "
			  "baseorigin = %04X, origin = %04X, length = %04X\n",
			  node->name, node->baseorigin,
			  node->origin, node->length);
#endif
		  for (i = 1; i < (phase->cmncount+1); i++)
		  {
		     if (!strcmp (phase->cmnnodes[i].name, node->name))
		     {
			phase->cmnnodes[i].origin = node->origin;
			break;
		     }
		  }
		  node = node->next;
	       }
	    }
	    else
	    {
	       node = node->next;
	    }
	 }
      }

      if (partiallink)
	 phase->csegorg = 0;
      else
	 phase->csegorg = phase->cseguserorg > 0 ? phase->cseguserorg : pc;

      if (phase->cseguserorg < 0)
      {
	 updatesegs (phase, phase->csegorg, CSEG);
	 if (!partiallink)
	    pc += phase->cseglen;
      }
      pc = (pc + 1) & 0xFFFE;
#ifdef DEBUGCOMMON
      printf ("processend-1c: csegorg = %04X, cseglen = %04X\n",
	      phase->csegorg, phase->cseglen);
#endif
      phase->length += phase->dseglen + phase->cseglen;
      phase->length = (phase->length + 1) & 0xFFFE;
      phase->dseglen = 0;
      phase->cseglen = 0;
      pc = (pc + phase->adjust) & ~phase->adjust;
   }

   else /* pass 2 */
   {
      Module *module;

      pc = (pc + 1) & 0xFFFE;

      if (maxlength < pc)
      {
	 maxlength = pc + phase->dseglen + phase->cseglen;
      }

      phase->dsegorg = partiallink ? 0 : pc;
      phase->dseglen = 0;
      for (module = phase->module_head; module; module = module->next)
      {
	 if (module->dsegmodule)
	 {
	    if (phase->dseguserorg < 0)
	       module->origin += phase->dsegorg;
	    phase->dseglen += module->length;
	 }
      }

      phase->csegorg = partiallink ? 0 : (pc + phase->dseglen);
      pc = (pc + 1) & 0xFFFE;
#ifdef DEBUGEND
      printf ("processend-2: pc = %04X, dsegorg = %04X, dseglen = %04X\n",
	      pc, phase->dsegorg, phase->dseglen);
      printf ("   csegorg = %04X, cseglen = %04X\n",
	      phase->csegorg, phase->cseglen);
      printf ("   maxlength = %04X:%04X\n",
	       maxlength, maxlength - phases[0].origin);
#endif
      if (phase->dseglen == 0)
         phase->dsegorg = 0;

      /*
      ** Punch out the object.
      */

      reftoextndx (phase);
      if (phase->length < 0xFFFF)
	 lnkpunch (phase, !nosymt);
      else
         status = printerror ("Phase exceeds memory", TRUE);
   }
   return (status);
}

/***********************************************************************
* processcmds - Process control file commands and perform actions.
***********************************************************************/

static int
processcmds (void)
{
   PHASENode *phase;
   FILE *infd = NULL;
   char *bp;
   char *mode;
   int i;
   int j;
   int num;
   int status;
   int done;
   int adjust;
   int endstmt;
   int p2find = FALSE;
   char cmdline[MAXLINE];
   char token[MAXLINE];
   char fullfile[MAXPATHNAMESIZE+2];
   char term;

#ifdef DEBUGCOMMANDS
   printf ("processcmds: inpass = %d, proccnt = %d, phasecnt = %d\n",
	   inpass, proccnt, phasecnt);
#endif

   if (fseek (ctlfd, 0L, SEEK_SET) < 0)
   {
      perror ("lnked990: Can't rewind control file");
      return (-1);
   }

   phase = NULL;
   phasenum = -1;
   maxlength = -1;
   adjust = 0;
   status = 0;
   endstmt = FALSE;
   while (fgets (cmdline, MAXLINE, ctlfd))
   {
      if ((bp = strchr (cmdline, '\r')) != NULL)
         *bp = 0;
      if ((bp = strchr (cmdline, '\n')) != NULL)
         *bp = 0;

      if (inpass == 1)
      {
         printheader();
	 fprintf (mapfd, "%s\n", cmdline);
	 linecnt++;
      }

      /* Ignore comments */
      if (cmdline[0] == ';' || cmdline[0] == '#')
         continue;

      bp = gettoken (cmdline, token, &term);
#ifdef DEBUGCOMMANDS
      printf ("COMMAND = %s, ARGS = %s\n", token, bp);
#endif
      switch (cmdlookup (token))
      {
      case CMD_ILLEGAL:
	 sprintf (errorline, "Illegal command: %s", token);
         status = printerror (errorline, TRUE);
	 break;

      case CMD_ADJUST:
         bp = gettoken (bp, token, &term);
         if (!token[0])
	 {
	    sprintf (errorline, "ADJUST value required.");
	    status = printerror (errorline, TRUE);
	    break;
	 }
	 num = getnum (token);
	 adjust = 0;
	 for (i = 0; i < num; i++)
	 {
	    adjust = (adjust << 1) | 1;
	 }
         break;

      case CMD_ALLGLOBAL:
	 if (!partiallink)
	 {
	 PARTIAL_ERROR:
	    sprintf (errorline,
		     "Command %s allowed only during a PARTIAL link.", 
		     token);
	    status = printerror (errorline, TRUE);
	    break;
	 }
	 if (!phase) goto SEQ_ERROR;

	 for (i = 0; i < phase->symcount; i++)
	 {
	    SymNode *sym;
	    
	    sym = phase->symbols[i];
	    if (sym->phase != phase) continue;
	    if (sym->flags & DEFGLOBAL)
	       sym->flags |= GLOBAL;
	 }
         break;

      case CMD_COMMON:
	 if (!phase)
	 {
	 SEQ_ERROR:
	    sprintf (errorline, "Command sequence error.");
	    status = printerror (errorline, TRUE);
	    break;
	 }
         bp = gettoken (bp, token, &term);
         if (!token[0])
	 {
	    sprintf (errorline, "COMMON address required.");
	    status = printerror (errorline, TRUE);
	    break;
	 }
	 if (isalpha (token[0]))
	 {
	    CSEGUserNode *node;

	    if ((node = cseglookup (phase, token, 0, 0, 0, FALSE, &j)) == NULL)
	    {
	       sprintf (errorline, "Base COMMON %s undefined.", token);
	       status = printerror (errorline, TRUE);
	       break;
	    }
	    num = node->baseorigin;
	 }
	 else
	 {
	    num = getnum (token);
	 }
#ifdef DEBUGCOMMON
         printf ("   COMMON: baseorigin = %04X\n", num);
#endif
	 if (term == ',')
	 {
	    done = FALSE;
	    while (!done)
	    {
	       CSEGUserNode *node;

	       bp = gettoken (bp, token, &term);
	       node = cseglookup (phase, token, num, 0, -1, inpass == 1, &j);
#ifdef DEBUGCOMMON
	       if (node)
		  printf ("   name = %s, origin = %04X, len = %04X\n",
			  token, node->origin, node->length);
#endif
	       if (term != ',')
	          done = TRUE;
	    }
	 }
	 else
	 {
	    phase->cseguserorg = num;
	 }
         break;

      case CMD_DATA:
	 if (!phase) goto SEQ_ERROR;
         bp = gettoken (bp, token, &term);
         if (!token[0])
	 {
	    sprintf (errorline, "DATA address required.");
	    status = printerror (errorline, TRUE);
	    break;
	 }
	 phase->dseguserorg = getnum (token);
         break;

      case CMD_DUMMY:
	 if (!phase) goto SEQ_ERROR;
         phase->dummy = TRUE;
	 break;

      case CMD_END:
	 endstmt = TRUE;
	 if (!phase)
	 {
	    return (-1);
	 }
	 if (!noauto)
	 {
	    searchlibraries (phase, NULL);
	 }
	 processend (phase);

         if (inpass == 1)
	 {
	    phases[0].length = (maxlength - phases[0].origin) +
	    		phases[0].dseglen + phases[0].cseglen;
	 }
	 if (phase->length > pcontrol[phasenum].length)
	 {
	    pcontrol[phasenum].length = phase->length + phase->dseglen +
		     phase->cseglen;
	 }
#ifdef DEBUGPHASE
         for (i = 0; i < (phasenum+1); i++)
	 {
	    printf (
	     "phase %d: name = %s, origin = %04X, length = %04X, pindex = %d\n",
	            i, pcontrol[i].phase->name,
		    pcontrol[i].origin, pcontrol[i].length, pcontrol[i].pindex);
	 }
#endif
         return (status);

      case CMD_ERROR:
	 noerror = FALSE;
         break;

      case CMD_FIND:
	 if (!phase) goto SEQ_ERROR;
	 bp = gettoken (bp, token, &term);
	 if (!token[0])
	 {
	    sprintf (errorline, "FIND filename required.");
	    status = printerror (errorline, TRUE);
	    break;
	 }
	 if (inpass == 1)
	 {
	    if ((infd = fopen (token, "rb")) == NULL)
	    {
	       sprintf (errorline, "Can't open file: %s: %s",
			token, strerror(errno));
	       status = printerror (errorline, TRUE);
	    }
	    else
	    {
	       char libname[MAXLINE+2];

	       if ((bp = strrchr (token, '/')) != NULL)
		  strcpy (libname, bp+1);
	       else
		  strcpy (libname, token);
	       if ((bp = strchr (libname, '.')) != NULL)
		  *bp = 0;

	       status = lnklibrary (phase, infd, pc, libname, token);
	       fclose (infd);
	    }
	 }

	 else if (!p2find) /* pass 2 */
	 {
	    Module *module;

	    p2find = TRUE;
	    for (module = phase->module_head; module; module = module->next)
	    {
	       if (module->libmodule)
	       {
		  if ((infd = fopen (module->fullfile, "rb")) != NULL)
		  {
		     if (fseek (infd, module->offset, SEEK_SET) >= 0)
		     {
			pc = lnkloader (phase, infd, pc, module->objfile, TRUE);
			if (pc < 0)
			{
			   fprintf (stderr, "Error: file = %s\n",
				    module->objfile);
			   exit (ABORT);
			}
		     }
		     fclose (infd);
		  }
	       }
	    }
	 }
         break;

      case CMD_FORMAT:
	 if (inpass == 1)
	 {
	    bp = gettoken (bp, token, &term);
	    if (!strcmp (token, "ASCII"))
	    {
	       compressed = FALSE;
	       if (term == ',')
	       {
	          bp = gettoken (bp, token, &term);
		  if (!strcmp (token, "NOSEQ"))
		     cassette = TRUE;
	       }
	    }
	    else if (!strcmp (token, "COMPRESSED"))
	    {
	       compressed = TRUE;
	    }
	    else if (!strcmp (token, "IMAGE"))
	    {
	       sprintf (errorline, "FORMAT %s not supported.", token);
	       status = printerror (errorline, FALSE);
	    }
	    else if (!strcmp (token, "AOUT"))
	    {
	       aoutformat = TRUE;
	    }
	    else
	    {
	       sprintf (errorline, "Illegal FORMAT: %s", token);
	       status = printerror (errorline, TRUE);
	    }
	 }
	 break;

      case CMD_GLOBAL:
	 if (!partiallink) goto PARTIAL_ERROR;
	 if (!phase) goto SEQ_ERROR;

	 done = FALSE;
	 while (!done)
	 {
	    SymNode *sym;

	    bp = gettoken (bp, token, &term);
	    if (!token[0])
	    {
	       sprintf (errorline, "GLOBAL name required.");
	       status = printerror (errorline, TRUE);
	       break;
	    }
	    if ((sym = symlookup (phase, NULL, token, FALSE)) != NULL)
	    {
	       if (sym->flags & DEFGLOBAL)
	          sym->flags |= GLOBAL;
	    }
	    else
	    {
	       sprintf (errorline, "GLOBAL symbol %s not found.", token);
	       status = printerror (errorline, TRUE);
	    }
	    if (term != ',')
	       done = TRUE;
	 }
         break;

      case CMD_INCLUDE:
	 if (!phase) goto SEQ_ERROR;
	 done = FALSE;
         while (!done)
	 {
	    char *sp, *ep;

	    bp = gettoken (bp, token, &term);
	    sp = token;
	    if (!*sp)
	    {
	       sprintf (errorline, "INCLUDE filename required.");
	       status = printerror (errorline, TRUE);
	       break;
	    }
	    if (*sp == '(')
	    {
	       *sp++ = ' ';
	       if ((ep = strchr (sp, ')')) != NULL)
	          *ep = 0;
	       infd = openlibrary (sp, fullfile, NULL);
	    }
	    else
	    {
	       strcpy (fullfile, sp);
	       infd = fopen (sp, "rb");
	    }

	    if (infd)
	    {
#ifdef DEBUGINCLUDE
               printf ("   origin = %04X, pc = %04X\n", phase->origin, pc);
#endif
	       pc = lnkloader (phase, infd, pc, sp, FALSE);
	       if (pc < 0)
	       {
		  fprintf (stderr, "Error: file = %s\n", sp);
		  exit (ABORT);
	       }
	       if (inpass == 1)
		  strcpy (phase->module_tail->fullfile, fullfile);
	       pc = (pc + adjust) & ~adjust;
	       adjust = 0;
	       fclose (infd);
	    }
	    else
	    {
	       sprintf (errorline, "Can't open file: %s: %s",
			sp, strerror(errno));
	       status = printerror (errorline, TRUE);
	    }
	    if (term != ',')
	       done = TRUE;
	 }
	 break;

      case CMD_LIBRARY:
	 if (inpass == 1)
	 {
	    done = FALSE;
	    while (!done)
	    {
	       struct stat stbuf;

	       bp = gettoken (bp, token, &term);
	       if (!token[0])
	       {
		  sprintf (errorline, "LIBRARY filename required.");
		  status = printerror (errorline, TRUE);
		  break;
	       }
	       if (stat (token, &stbuf) < 0)
	       {
		  sprintf (errorline, "Can't stat library: %s: %s",
			   token, strerror(errno));
		  status = printerror (errorline, TRUE);
	       }
	       else if (S_ISDIR(stbuf.st_mode))
	       {
		  if (libcnt > MAXFILES)
		  {
		     sprintf (errorline, "Too many libraries.");
		     status = printerror (errorline, TRUE);
		     break;
		  }
		  strcpy (libfiles[libcnt].name, token);
		  libfiles[libcnt].library = TRUE;
		  libfiles[libcnt].offset = 0;
		  libcnt++;
	       }
	       else
	       {
		  sprintf (errorline, "Library: %s: %s",
			   token, strerror(ENOTDIR));
		  status = printerror (errorline, TRUE);
	       }
	       if (term != ',')
		  done = TRUE;
	    }
	 }
	 break;

      case CMD_NOAUTO:
	 noauto = TRUE;
         break;

      case CMD_NOERROR:
	 noerror = TRUE;
         break;

      case CMD_NOMAP:
	 nomap = TRUE;
         /* Fall through */

      case CMD_NOPAGE:
	 nopage = TRUE;
         break;

      case CMD_NOSYMT:
	 nosymt = TRUE;
         break;

      case CMD_NOTGLOBAL:
	 if (!partiallink) goto PARTIAL_ERROR;
	 if (!phase) goto SEQ_ERROR;

	 for (i = 0; i < phase->symcount; i++)
	 {
	    SymNode *sym;
	    
	    sym = phase->symbols[i];
	    if (sym->phase != phase) continue;
	    if (sym->flags & DEFGLOBAL)
	       sym->flags &= ~GLOBAL;
	 }
         break;

      case CMD_PAGE:
	 nopage = FALSE;
         break;

      case CMD_PARTIAL:
	 partiallink = TRUE;
         break;

      case CMD_PHASE:
         bp = gettoken (bp, token, &term);
         if (!token[0])
	 {
	    sprintf (errorline, "PHASE number required.");
	    status = printerror (errorline, TRUE);
	    break;
	 }
	 num = getnum (token);
         if (aoutformat && num)
	 {
	    sprintf (errorline, "Overlays not supported in a.out format.");
	    status = printerror (errorline, TRUE);
	    break;
	 }
	 mode = "PHASE";
	 goto PHASE_NAME;

      case CMD_PROCEDURE:
	 if (phase)
	 {
	    processend (phase);
	 }
	 mode = "PROCEDURE";
	 p2find = FALSE;
	 bp = gettoken (bp, token, &term);
         if (!token[0])
	 {
	    sprintf (errorline, "%s name required.", mode);
	    status = printerror (errorline, TRUE);
	    break;
	 }
         if (aoutformat)
	 {
	    sprintf (errorline, "Procedures not supported in a.out format.");
	    status = printerror (errorline, TRUE);
	    break;
	 }

	 done = TRUE;
	 phase = &procs[0];
	 for (i = 0; i < proccnt; i++)
	 {
	    phase = &procs[i];
	    if (!strcmp (phase->name, token))
	    {
	       if (inpass == 1)
	       {
		  sprintf (errorline, "Duplicate %s name: %s", mode, token);
		  status = printerror (errorline, TRUE);
	       }
	       done = FALSE;
	       break;
	    }
	 }
	 if (done)
	 {
	    if (i > MAXPROCS)
	    {
	       sprintf (errorline, "Too many %s definitions", mode);
	       status = printerror (errorline, TRUE);
	    }
	    else
	    {
	       phase = &procs[i];
	       memset (phase, 0, sizeof(PHASENode));
	       strcpy (phase->name, token);
	       phase->number = proccnt + 1;
	       phase->origin = pc;
	       phase->adjust = 0x1F;
	       phase->dseguserorg = -1;
	       phase->cseguserorg = -1;
	       proccnt++;
	    }
	 }
	 pc = phase->origin;
	 break;

      case CMD_PROGRAM:
	 if (!phase) goto SEQ_ERROR;
         bp = gettoken (bp, token, &term);
         if (!token[0])
	 {
	    sprintf (errorline, "PROGRAM address required.");
	    status = printerror (errorline, TRUE);
	    break;
	 }
	 absolute = TRUE;
	 phase->origin = pc = getnum (token);
         break;

      case CMD_SEARCH:
	 if (!phase) goto SEQ_ERROR;
         bp = gettoken (bp, token, &term);
	 if (token[0])
	 {
	    done = FALSE;
	    while (!done)
	    {
	       struct stat stbuf;

	       if (stat (token, &stbuf) < 0)
	       {
		  sprintf (errorline, "Can't stat library: %s: %s",
			   token, strerror(errno));
		  status = printerror (errorline, TRUE);
	       }
	       else if (S_ISDIR(stbuf.st_mode))
	       {
		  searchlibraries (phase, token);
	       }
	       else
	       {
		  sprintf (errorline, "Library: %s: %s",
			   token, strerror(ENOTDIR));
		  status = printerror (errorline, TRUE);
	       }
	       if (term != ',')
		  done = TRUE;
	       else
		  bp = gettoken (bp, token, &term);
	    }
	 }
	 else
	 {
	    searchlibraries (phase, NULL);
	 }
         break;

      case CMD_SYMT:
	 nosymt = FALSE;
         break;

      case CMD_TASK:
	 mode = "TASK";
	 num = 0;

      PHASE_NAME:
	 if (phase)
	 {
	    processend (phase);
	    if ((phasenum >= 0) && (phase->length > pcontrol[phasenum].length))
	    {
	       pcontrol[phasenum].length = phase->length;
	    }
	    if ((inpass == 1) && (num > 0) && (num > phasenum))
	    {
	       phase->length = (phase->length + phase->adjust) & ~phase->adjust;
	    }
#ifdef DEBUGPHASE
	    for (i = 0; i < (phasenum+1); i++)
	    {
	       printf (
	     "phase %d: name = %s, origin = %04X, length = %04X, pindex = %d\n",
	               i, pcontrol[i].phase->name, pcontrol[i].origin,
		       pcontrol[i].length, pcontrol[i].pindex);
	    }
#endif
	 }
	 p2find = FALSE;
	 bp = gettoken (bp, token, &term);
         if (!token[0])
	 {
	    sprintf (errorline, "%s name required.", mode);
	    status = printerror (errorline, TRUE);
	    break;
	 }

	 done = TRUE;
	 phase = &phases[0];
	 for (i = 0; i < phasecnt; i++)
	 {
	    phase = &phases[i];
	    if (!strcmp (phase->name, token) && (num == phase->number))
	    {
	       if (inpass == 1)
	       {
		  sprintf (errorline, "Duplicate %s name: %s", mode, token);
		  status = printerror (errorline, TRUE);
	       }
	       done = FALSE;
	       break;
	    }
	 }
	 if (done)
	 {
	    if (i > MAXPHASES)
	    {
	       sprintf (errorline, "Too many %s definitions", mode);
	       status = printerror (errorline, TRUE);
	    }
	    else
	    {
	       phase = &phases[i];
	       memset (phase, 0, sizeof(PHASENode));
	       strcpy (phase->name, token);
	       phase->number = num;
	       phase->adjust = adjust;
	       phase->dseguserorg = -1;
	       phase->cseguserorg = -1;
	       phasecnt++;
	       if (num > phasenum)
	       {
		  pcontrol[num].origin = pc;
		  pcontrol[num].pindex = phasenum;
		  phase->origin = pc;
	       }
	       else
	       {
		  phase->origin = pcontrol[num].origin;
	       }
	    }
	 }

	 if (term == ',')
	 {
	    bp = gettoken (bp, token, &term);
	    if ((i = cmdlookup (token)) == CMD_PROGRAM)
	    {
	       bp = gettoken (bp, token, &term);
	       if (!token[0])
	       {
		  sprintf (errorline, "PROGRAM address required.");
		  status = printerror (errorline, TRUE);
		  break;
	       }
	       phase->origin = pc = getnum (token);
	    }
	 }
	 pcontrol[num].phase = phase;
	 phasenum = num;
	 pc = phase->origin;
	 adjust = 0;
	 break;

      default: 
	 sprintf (errorline, "Command %s not supported.", token);
	 status = printerror (errorline, FALSE);
      }
   }

   if (errcount && !noerror)
   {
      status = 0;
   }
   if (!endstmt)
   {
      status = printerror ("No END command", TRUE);
   }
   return (status);
}

/***********************************************************************
* Main procedure
***********************************************************************/

int
main (int argc, char **argv)
{
   char *ctlfile = NULL;
   char *outfile = NULL;
   char *mapfile = NULL;
   char *bp;
   int i;
   int status = 0;
   time_t curtime;
  
#ifdef DEBUGARGS
   printf ("lnked990: Entered:\n");
   printf ("args:\n");
   for (i = 1; i < argc; i++)
      printf ("   arg[%2d] = %s\n", i, argv[i]);
#endif

   /*
   ** Clear out structures.
   */

   memset (&libfiles, 0, sizeof (libfiles));
   libcnt = 0;

   memset (&procs, 0, sizeof(procs));
   memset (&phases, 0, sizeof(phases));
   memset (&pcontrol, 0, sizeof(pcontrol));
   proccnt = 0;
   phasecnt = 0;

   i = sizeof(spaces);
   memset (spaces, ' ', i);
   spaces[i] = 0;

   /*
   ** Scan off the the args.
   */

   for (i = 1; i < argc; i++)
   {
      bp = argv[i];

      if (*bp == '-')
      {
	 bp++;
         while (*bp) switch (*bp)
         {
	 case 'c':
	    cassette = TRUE;
	    bp++;
	    break;

	 case 'w':
	    widelist = TRUE;
	    bp++;
	    break;

         default:
      USAGE:
	    printf ("lnked990 - version %s\n", VERSION);
	    printf ("usage: lnked990 [-options] link.ctl link.map link.obj\n");
            printf (" options:\n");
	    printf ("    -c           - Cassette mode\n");
	    printf ("    -w           - Wide map listing\n");
	    return (ABORT);
         }
      }

      else
      {
	 if (!ctlfile)
	 {
	    ctlfile = argv[i];
	 }
	 else if (!mapfile)
	 {
	    mapfile = argv[i];
	 }
	 else if (!outfile)
	 {
	    outfile = argv[i];
	 }
	 else
	 {
	    fprintf (stderr, "Too many files sepecified\n");
	    exit (ABORT);
	 }
      }
   }

   if (!(ctlfile && mapfile && outfile)) goto USAGE;
   if (!strcmp (ctlfile, mapfile))
   {
      fprintf (stderr, "lnked990: Identical control and map file names\n");
      exit (ABORT);
   }
   if (!strcmp (ctlfile, outfile))
   {
      fprintf (stderr, "lnked990: Identical control and object file names\n");
      exit (ABORT);
   }
   if (!strcmp (outfile, mapfile))
   {
      fprintf (stderr, "lnked990: Identical object and map file names\n");
      exit (ABORT);
   }

#ifdef DEBUGARGS
   printf (" ctlfile = %s\n", ctlfile);
   printf (" outfile = %s\n", outfile);
   printf (" mapfile = %s\n", mapfile);
#endif

   /*
   ** Open the files.
   */

   if ((ctlfd = fopen (ctlfile, "r")) == NULL)
   {
      perror ("lnked990: Can't open control file");
      exit (ABORT);
   }
   if ((outfd = fopen (outfile, "wb")) == NULL)
   {
      perror ("lnked990: Can't open output file");
      exit (ABORT);
   }
   if ((mapfd = fopen (mapfile, "w")) == NULL)
   {
      perror ("lnked990: Can't open map file");
      exit (ABORT);
   }

   /*
   ** Get current date/time.
   */

   curtime = time(NULL);
   timeblk = localtime (&curtime);
   strcpy (datebuf, ctime(&curtime));
   *strchr (datebuf, '\n') = '\0';

   /*
   ** Clear memory.
   */

   memset (&memory, 0, sizeof(memory));
   memset (&dsegmem, 0, sizeof(dsegmem));
   memset (&csegmem, 0, sizeof(csegmem));
   memset (&memctl, 0, sizeof (memctl));
   memset (&dsegctl, 0, sizeof (dsegctl));
   memset (&csegctl, 0, sizeof (csegctl));

   /*
   ** Process pass 1.
   */

   idtname[0] = '\0';
   inpass = 1;
   pc = 0;
   if ((status = processcmds()) == 0)
   {

      /*
      ** If a.out mode add _etext, _edata, _end symbols.
      */

      if (aoutformat)
      {
	 PHASENode *phase = &phases[0];
	 Module *module = phase->module_head;
	 int pgmlen = phase->length + phase->origin;

	 processdef (phase, module, "_etext", 1, LONGSYM, phase->dsegorg, 0, 0);
	 processdef (phase, module, "_edata", 1, LONGSYM, phase->dsegorg, 0, 0);
	 processdef (phase, module, "_end",   1, LONGSYM, pgmlen, 0, 0);
      }

      /*
      ** Process pass 2.
      */

      memset (&memory, 0, sizeof(memory));
      memset (&dsegmem, 0, sizeof(dsegmem));
      memset (&csegmem, 0, sizeof(csegmem));
      memset (&memctl, 0, sizeof (memctl));
      memset (&dsegctl, 0, sizeof (dsegctl));
      memset (&csegctl, 0, sizeof (csegctl));

      pc = 0;
      curmod = 0;
      inpass = 2;

      if ((status = processcmds()) == 0)
      {

	 /*
	 ** Process link map.
	 */

	 linecnt = MAXLINE;
	 printlinkmap();
      }
   }

   /*
   ** Close the files
   */

   fclose (ctlfd);
   fclose (outfd);
   fclose (mapfd);

   if (errcount || warncount)
      fprintf (stderr, "lnked990: %d errors, %d warnings\n",
               errcount, warncount);

   return (status == 0 ? NORMAL : ABORT);
}
