/************************************************************************
*
* lnklibrary - Loads library objects from asm990 for linking.
*
* Changes:
*   01/22/07   DGP   Original, Hacked from lnkloader.
*   01/24/07   DGP   Added Common REF/DEF and DSEG processing.
*   12/02/10   DGP   Corrected EXTNDX tag support.
*   12/07/10   DGP   Added ABSDEF flag.
*   12/14/10   DGP   Correct module number in undefined reference.
*   01/26/11   DGP   Reread library if we have more to do...
*   02/08/11   DGP   Added PGMIDT and LOAD tag support.
*   02/10/11   DGP   Corrected EXTNDX in dseg and DSEGORG with offset.
*   11/03/11   DGP   Fixed pc usage to always even between modules.
*                    Changed the way uninitalized memory is used.
*   08/21/13   DGP   Revamp library support.
*                    Default D/C SEGS at the end like SDSLNK.
*   01/11/16   DGP   Added Long COMMON symbol support.
*
************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>
#include <errno.h>

#include "lnkdef.h"

extern int pc;
extern int inpass;

/************************************************************************
* checkdef - Check if DEF is needed.
************************************************************************/

static int
checkdef (PHASENode *phase, char *item, int relo, int flags)
{
   SymNode *s;
   char *bp;

   bp = item;
   for (; *bp; bp++) if (isspace(*bp)) *bp = '\0';

#ifdef DEBUGLIBRARY
   printf ("checkdef: item = %s", item);
#endif
#if defined(DEBUGLIBRARYLOAD) || defined(DEBUGLIBRARY)
   printf (", %c%s = %s", relo ? 'R' : 'A', (flags & LONGSYM) ? "LDEF" : "DEF",
	   item);
#endif

   if ((s = reflookup (phase, NULL, item, FALSE)) != NULL)
   {
      if (s->flags & UNDEF && !(s->flags & SREF))
      {
#if defined(DEBUGLIBRARYLOAD) || defined(DEBUGLIBRARY)
	 printf ("\n      flags = %04X, value = %04X", s->flags, s->value);
#endif
#ifdef DEBUGLIBRARY
	 putchar('\n');
#endif
         return (1);
      }
   }

#ifdef DEBUGLIBRARY
   putchar('\n');
#endif
   return (0);
}

/************************************************************************
* lnkcheck - Check if file is referenced.
************************************************************************/

static int
lnkcheck (PHASENode *phase, FILE *fd)
{
   int status;
   int binarymode = FALSE;
   int reclen;
   int relo;
   int done;
   int i;
   int wordlen = WORDTAGLEN-1;
   unsigned char inbuf[82];

#ifdef DEBUGLIBRARY
   printf ("lnkcheck: \n");
#endif

   status = 0;
   i = fgetc (fd);
   ungetc (i, fd);
   if (i == BINIDT_TAG)
   {
      binarymode = TRUE;
   }

   done = FALSE;
   while (!done)
   {
      unsigned char *op = inbuf;

      if (binarymode)
      {
	 reclen = fread (inbuf, 1, 81, fd);
	 if (feof(fd))
	 {
	    done = TRUE;
	    break;
	 }
      }
      else
      {
	 if (fgets ((char *)inbuf, 82, fd) == NULL)
	 {
	    done = TRUE;
	    break;
	 }
	 reclen = strlen ((char *)inbuf);
      }

#ifdef DEBUGLIBRARYLOAD
      printf ("reclen = %d\n", reclen);
#endif
      if (*op == EOFSYM)
      {
	 done = TRUE;
	 break;
      }

      for (i = 0; i < reclen; i++)
      {
	 char otag;
	 char item[80];
	 uint16 wdata;

	 otag = *op++;

	 wdata = getnumfield (op, binarymode);

#ifdef DEBUGLIBRARYLOAD
	 printf ("   otag = %c, data = %04X", isprint(otag) ? otag : '0',
		 wdata & 0xFFFF);
#endif

	 relo = FALSE;
	 switch (otag)
	 {
	 case BINIDT_TAG: /* Binary IDT */
#ifdef DEBUGLIBRARYLOAD
	    printf (", Binary mode");
#endif
	    wordlen = BINWORDTAGLEN-1;
	 case IDT_TAG:
	 case PGMIDT_TAG:
	    op += wordlen;
	    strncpy (item, (char *)op, IDTSIZE);
	    item[IDTSIZE] = '\0';
#ifdef DEBUGLIBRARY
	    printf ("   IDT = %s\n", item);
#endif
#ifdef DEBUGLIBRARYLOAD
	    printf (", IDT = %s", item);
#endif
	    op += IDTSIZE;
	    break;

	 case RELEXTRN_TAG:
	 case ABSEXTRN_TAG:
	 case RELSREF_TAG:
	 case LOAD_TAG:
	 case RELSYMBOL_TAG:
	 case ABSSYMBOL_TAG:
	    op += wordlen;
	    op += MAXSHORTSYMLEN;
	    break;

	 case RELGLOBAL_TAG:
	    relo = TRUE;
	 case ABSGLOBAL_TAG:
	    op += wordlen;
	    strncpy (item, (char *)op, MAXSHORTSYMLEN);
	    item[MAXSHORTSYMLEN] = '\0';
	    status += checkdef (phase, item, relo, 0);
	    op += MAXSHORTSYMLEN;
	    break;

	 case COMMON_TAG:
	 case CMNEXTRN_TAG:
	    op += wordlen;
            op += MAXSHORTSYMLEN;
            op += wordlen;
            break;
	    
         case CMNORG_TAG:
         case CMNDATA_TAG:
         case EXTNDX_TAG:
            op += wordlen;
            op += wordlen;
            break;

	 case CMNGLOBAL_TAG:
            op += wordlen;
	    strncpy (item, (char *)op, MAXSHORTSYMLEN);
	    item[MAXSHORTSYMLEN] = '\0';
	    status += checkdef (phase, item, TRUE, 0);
            op += MAXSHORTSYMLEN;
            op += wordlen;
            break;

	 case LRELSREF_TAG:
	 case LLOAD_TAG:
	 case LRELSYMBOL_TAG:
	 case LABSSYMBOL_TAG:
	 case LRELEXTRN_TAG:
	 case LABSEXTRN_TAG:
	    op += wordlen;
	    op += MAXSYMLEN;
	    break;

	 case LRELGLOBAL_TAG:
	    relo = TRUE;
	 case LABSGLOBAL_TAG:
	    op += wordlen;
	    strncpy (item, (char *)op, MAXSYMLEN);
	    item[MAXSYMLEN] = '\0';
	    status += checkdef (phase, item, relo, LONGSYM);
	    op += MAXSYMLEN;
	    break;

	 case LCMNGLOBAL_TAG:
	    relo = TRUE;
	    op += wordlen;
	    strncpy (item, (char *)op, MAXSYMLEN);
	    item[MAXSYMLEN] = '\0';
	    status += checkdef (phase, item, relo, LONGSYM);
	    op += MAXSYMLEN;
	    op += wordlen;
	    break;

	 case LCMNEXTRN_TAG:
	 case LCMNSREF_TAG:
	    op += wordlen;
	    op += MAXSYMLEN;
	    op += wordlen;
	    break;
	 case EOR_TAG:
	    i = 81;
	    break;

	 case NOCKSUM_TAG:
	 case CKSUM_TAG:
	    op += wordlen;
	    break; 

	 default:
	    op += wordlen;
	 }
#ifdef DEBUGLIBRARYLOAD
	 printf ("\n");
#endif
      }
   }

#ifdef DEBUGLIBRARY
   if (status)
      printf ("   LOAD module\n");
#endif
   return (status);

}

/************************************************************************
* lnklibrary - Object library loader for lnked990.
************************************************************************/

int
lnklibrary (PHASENode *phase, FILE *fd, int loadpt, char *file, char *fullfile)
{
   off_t modstart;
   int more;
   int pcstart;
   int curraddr;

   curraddr = loadpt;

#ifdef DEBUGLIBRARY
   printf ("lnklibrary: loadpt = %04X, pass = %d, file = %s\n",
           loadpt, inpass, fullfile);
#endif

   /*
   ** Process the entire library
   */

   do
   {
      fseek (fd, 0, SEEK_SET);
      clearerr(fd);
      more = FALSE;

      while (!feof(fd))
      {
	 if ((modstart = ftell(fd)) < 0)
	 {
	    char errbuf[256];
	    sprintf (errbuf, "lnked990: Can't ftell input file %s", fullfile);
	    perror (errbuf);
	    exit (ABORT);
	 }
	 pcstart = curraddr;

	 /*
	 ** Check if current module has DEFs that are needed for an
	 ** unresolved reference.
	 */

	 if (lnkcheck (phase, fd))
	 {
	    /*
	    ** Yes, load module.
	    */

	    more = TRUE;
	    pcstart = (pcstart + 1) & 0xFFFE;
	    if (fseek (fd, modstart, SEEK_SET) < 0)
	    {
	       char errbuf[256];
	       sprintf (errbuf, "lnked990: Can't fseek input file %s",
			fullfile);
	       perror (errbuf);
	       exit (ABORT);
	    }
	    curraddr = lnkloader (phase, fd, pcstart, file, TRUE);
	    phase->module_tail->libmodule = TRUE;
	    phase->module_tail->offset = modstart;
	    strcpy (phase->module_tail->fullfile, fullfile);
	 }
      }
   } while (more);
   pc = curraddr;
   return (0);
}
