/************************************************************************
*
* lnkloader - Loads objects from asm990 for linking.
*
* Changes:
*   06/05/03   DGP   Original.
*   06/25/03   DGP   Changed to print listing like TXLINK.
*                    Changed to link undefined external chains.
*   06/27/03   DGP   Added compressed binary support.
*   07/10/03   DGP   Changed to use first global definition and ignore 
*                    absolute externals with a zero value.
*   04/12/04   DGP   Changed to allow non-80 byte records.
*   05/25/04   DGP   Added long ref/def support.
*   05/05/04   DGP   Fixed binary mode reads. Added getnumfield().
*   12/30/05   DGP   Changed SymNode to use flags.
*   01/18/07   DGP   Functionalize REF/DEF processing.
*   01/24/07   DGP   Added Common REF/DEF and DSEG processing.
*   03/17/09   DGP   Allow for back relorg (odd pc bss).
*   06/11/09   DGP   Added CSEG support.
*   12/02/10   DGP   Corrected EXTNDX tag support.
*   12/07/10   DGP   Added ABSDEF flag.
*   12/14/10   DGP   Correct module number in undefined reference.
*   01/07/11   DGP   Allow for EXTNDX to be filled in later. We link the
*                    references when undefined.
*   01/25/11   DGP   Correct dseglen usage.
*   02/08/11   DGP   Added PGMIDT and LOAD tag support.
*   02/10/11   DGP   Corrected EXTNDX in dseg and DSEGORG with offset.
*   03/04/11   DGP   Fixed processing of concatenated object modules.
*   11/03/11   DGP   Don't let LOAD references bump the extndx.
*                    Fixed "back" RORG handling.
*                    Fixed module length return (don't return curraddr).
*   08/20/13   DGP   Many fixes to CSEG/DSEG processing.
*                    Fixed concatenated modules processing.
*   08/21/13   DGP   Revamp library support.
*                    Default D/C SEGS at the end like SDSLNK.
*   01/16/14   DGP   Fixed backward CSEG/DSEG orgs.
*   01/29/14   DGP   Revamp symbols and EXTTAG references.
*   10/15/14   DGP   Return -1 on non-fatal loader error.
*   01/28/15   DGP   Added NOTGLOBAL, ALLGLOBAL, GLOBAL support.
*   02/04/15   DGP   Fixed partial link to correct EXTNDX tag processing.
*   10/09/15   DGP   Added Long symbol common tags.
*   12/12/15   DGP   Fixed DSEG def.
*   01/24/17   DGP   Defer Common def. relocation.
*
************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>
#include <errno.h>
#include <memory.h>

#include "lnkdef.h"

extern int absentry;
extern int relentry;
extern int modcount;
extern int modnumber;
extern int curmod;
extern int extndx;
extern int inpass;
extern int phasenum;
extern int partiallink;

extern PHASEControl pcontrol[MAXPHASES];

extern uint8 memory[MEMSIZE];
extern MemCTL memctl[MEMSIZE];
extern uint8 dsegmem[MEMSIZE];
extern MemCTL dsegctl[MEMSIZE];
extern uint8 csegmem[MEMSIZE];
extern MemCTL csegctl[MEMSIZE];

static EXTList extlist[MAXSYMBOLS];

/************************************************************************
* getnumfield - Gets the next numeric field.
************************************************************************/

uint16
getnumfield (unsigned char *op, int binarymode)
{
   uint16 wdata;
   char item[80];

   if (binarymode)
   {
      wdata = (*op << 8) | *(op+1);
   }
   else
   {
      int wd;

      strncpy (item, (char *)op, 4);
      item[4] = '\0';
      sscanf (item, WORDFORMAT, &wd);
      wdata = wd & 0xFFFF;
   }
   return (wdata);
}

/************************************************************************
* processdef - Process DEF.
************************************************************************/

void
processdef (PHASENode *phase, Module *module, char *item, int relo,
	    int flags, uint16 wdata, int cmnndx, int libload)
{
   SymNode *s;
   char *bp;

   if (inpass == 1)
   {
      bp = item;
      for (; *bp; bp++) if (isspace(*bp)) *bp = '\0';
#ifdef DEBUGLOADER
      printf (", %c%s = %s, libload = %s, cmnndx = %d", relo ? 'R' : 'A',
	      (flags & LONGSYM) ? "LDEF" : (flags & CSEG) ? "CMNDEF" : 
		  (flags & DSEG) ? "DSEGDEF" : "DEF",
	      item, libload ? "TRUE" : "FALSE", cmnndx);
#endif
      if ((s = symlookup (phase, module, item, FALSE)) == NULL)
      {
	 if ((s = symlookup (phase, module, item, TRUE)) == NULL)
	 {
	    fprintf (stderr, "lnked990: Internal error\n");
	    exit (ABORT);
	 }
	 s->flags = relo ? RELOCATABLE : ABSDEF;
	 s->flags |= DEFGLOBAL | GLOBAL | flags;
	 s->value = wdata;
	 s->modnum = modnumber;
	 s->cmnndx = cmnndx;
#ifdef DEBUGLOADER
	 printf ("\n      flags = %04X, value = %04X", s->flags, s->value);
#endif

	 /* Go through the ref tables and add for each referencing phase */

	 while ((s = reflookup (phase, module, item, FALSE)) != NULL)
	 {
	    s->flags = relo ? RELOCATABLE : ABSDEF;
	    s->flags |= GLOBAL | flags;
	    if (s->phase != phase)
	    {
	       int oldnum = phasenum;

	       phasenum = s->phase->number;
	       if ((s = symlookup (s->phase, module, item, TRUE)) == NULL)
	       {
		  fprintf (stderr, "lnked990: Internal error\n");
		  exit (ABORT);
	       }
	       s->flags = relo ? RELOCATABLE : ABSDEF;
	       s->flags |= GLOBAL | flags;
	       s->value = wdata;
	       s->modnum = modnumber;
	       s->phase = phase;
	       phasenum = oldnum;
	    }
	 }
      }
      else
      {
	 if (libload)
	 {
	    s->modnum = modnumber;
	    if (s->flags & UNDEF)
	    {
	       int refaddr;

#ifdef DEBUGLOADER
	       printf ("\n      fill: flags = %04X, value = %04X",
		       s->flags, s->value);
#endif
	       refaddr = s->value;
	       while (refaddr)
	       {
		  int k;
		  k = GETMEM (refaddr);
		  PUTMEM (refaddr, wdata);
		  memctl[refaddr].tag = relo ? RELDATA_TAG : ABSDATA_TAG;
		  refaddr = k;
#ifdef DEBUGLOADER
		  printf ("\n          refaddr = %04X", refaddr);
#endif
	       }
	       s->flags = relo ? RELOCATABLE : ABSDEF;
	       s->flags |= GLOBAL | flags;
	       s->value = wdata;
	    }
	    else
	       s->flags |= MULDEF;
	 }
	 else
	    s->flags |= MULDEF;
      }
   }
}

/************************************************************************
* processref - Process REF.
************************************************************************/

static void
processref (PHASENode *phase, Module *module, char *item, int relo,
	    int flags, uint16 wdata, int libload)
{
   SymNode *s;
   char *bp;

   bp = item;
   for (; *bp; bp++) if (isspace(*bp)) *bp = '\0';
#ifdef DEBUGLOADER
   printf (", %c%s = %s", relo ? 'R' : 'A',
	   (flags & LONGSYM) ? "LREF" : (flags & SREF) ? "SREF" :
	       (flags & LOAD) ? "LOAD" : "REF", item);
#endif

   if (inpass == 1)
   {
      if ((s = symlookup (phase, module, item, FALSE)) == NULL)
      {
	 if ((s = reflookup (phase, module, item, FALSE)) == NULL)
	 {
	    if ((s = reflookup (phase, module, item, TRUE)) == NULL)
	    {
	       fprintf (stderr, "lnked990: Internal error\n");
	       exit (ABORT);
	    }
	    s->flags = relo ? RELOCATABLE : 0;
	    s->flags |= EXTERNAL | UNDEF | flags;
	 }
	 else
	 {
	    if ((flags & LOAD) && (s->flags & SREF))
	    {
	       s->flags &= ~(LOAD|SREF);
	    }
	 }
      }
   }

   else /* pass 2 */
   {
      if ((s = symlookup (phase, module, item, FALSE)) == NULL)
      {
	 if ((s = symlookup (phase, module, item, TRUE)) == NULL)
	 {
	    fprintf (stderr, "lnked990: Internal error\n");
	    exit (ABORT);
	 }
#ifdef DEBUGLOADER
         printf (", module = %d", curmod);
#endif
	 s->flags = relo ? RELOCATABLE : 0;
	 s->flags |= EXTERNAL | UNDEF | flags;
	 s->value = wdata;
	 if (!(flags & LOAD))
	 {
#if defined(DEBUGEXTNDX)
            printf ("processref-1: %4s symbol = %-6s, extndx = %d\n", 
		    (flags & LONGSYM) ? "LREF" : (flags & SREF) ? "SREF" :
			(flags & LOAD) ? "LOAD" : "REF",
	            s->symbol, extndx);
#endif
	    extlist[extndx].sym = s;
	    extlist[extndx].extndx = extndx;
	    extndx++;
	 }
	 s->modnum = curmod;
      }
      else
      {
	 int refaddr;

         if (!(flags & LOAD))
	 {
#if defined(DEBUGEXTNDX)
            printf ("processref-2: %4s symbol = %-6s, extndx = %d\n", 
		    (flags & LONGSYM) ? "LREF" : (flags & SREF) ? "SREF" :
			(flags & LOAD) ? "LOAD" : "REF",
	            s->symbol, extndx);
#endif
	    extlist[extndx].sym = s;
	    extlist[extndx].extndx = extndx;
	    extndx++;
	 }
	 if (relo && !(s->flags & ABSDEF))
	    s->flags |= RELOCATABLE;
	 refaddr = wdata;
	 if (flags & LOAD)
	 {
	    if (s->flags & SREF)
	       s->flags &= ~(LOAD|SREF);
	    return;
	 }
#ifdef DEBUGLOADER
	 printf ("\n      flags = %04X, value = %04X", s->flags, s->value);
	 printf ("\n          refaddr = %04X", refaddr);
#endif
	 /*
	 ** If defined, fill in back chain
	 */

	 if (!(s->flags & UNDEF))
	 {
	    char otag;

	    if (partiallink && (s->flags & DSEG))
	       otag = DSEGDATA_TAG;
	    else if (partiallink && (s->flags & CSEG))
	       otag = CMNDATA_TAG;
	    else 
	       otag = (s->flags & RELOCATABLE) ? RELDATA_TAG : ABSDATA_TAG;

	    while (refaddr)
	    {
	       int k;

	       if (flags & DSEG)
	       {
		  k = GETDSEGMEM (refaddr);
		  PUTDSEGMEM (refaddr, s->value);
		  dsegctl[refaddr].tag = otag;
	       }
	       else if (flags & CSEG)
	       {
		  k = GETCSEGMEM (refaddr);
		  PUTCSEGMEM (refaddr, s->value);
		  csegctl[refaddr].tag = otag;
	       }
	       else
	       {
		  k = GETMEM (refaddr);
		  PUTMEM (refaddr, s->value);
		  memctl[refaddr].tag = otag;
		  if (otag == CMNDATA_TAG)
		     memctl[refaddr].value = s->cmnndx;
	       }
	       refaddr = k;
#ifdef DEBUGLOADER
	       printf ("\n          refaddr = %04X", refaddr);
#endif
	    }
	 }

	 /*
	 ** If undefined, connect back chain
	 */

	 else
	 {
	    while (refaddr)
	    {
	       int k;

	       if (flags & DSEG)
	       {
		  k = GETDSEGMEM (refaddr);
		  if (k == 0)
		  {
		     dsegctl[refaddr].tag = RELDATA_TAG;
		     dsegctl[refaddr].sym = s;
		     PUTDSEGMEM(refaddr, s->value);
		     s->value = wdata;
		  }
	       }
	       else if (flags & CSEG)
	       {
		  k = GETCSEGMEM (refaddr);
		  if (k == 0)
		  {
		     csegctl[refaddr].tag = RELDATA_TAG;
		     csegctl[refaddr].sym = s;
		     PUTCSEGMEM(refaddr, s->value);
		     s->value = wdata;
		  }
	       }
	       else
	       {
		  k = GETMEM (refaddr);
		  if (k == 0)
		  {
		     memctl[refaddr].tag = RELDATA_TAG;
		     memctl[refaddr].sym = s;
		     PUTMEM(refaddr, s->value);
		     s->value = wdata;
		  }
	       }
	       refaddr = k;
	    }
	 }
      }
   }
}

/************************************************************************
* fillinextndx - Process EXTNDX back fills.
************************************************************************/

static void
fillinextndx (PHASENode *phase)
{
   SymNode *s;
   int i;

   if (inpass == 1) return;

#if defined(DEBUGEXTNDX)
   printf ("fillinextndx: entered: phase.name = %s, extndx = %d\n",
	   phase->name,  extndx);
#endif

   /*
   ** Go through EXTNDX table and update values.
   */

   for (i = 0; i < extndx; i++)
   {
      EXTndxlist *ref;
      int curraddr;

      s = extlist[i].sym;
#if defined(DEBUGEXTNDX)
      if (s->extlist)
	 printf ("   ext symbol = %s\n", s->symbol);
#endif
      for (ref = s->extlist; ref; ref = ref->next)
      {
	 curraddr = ref->location;
	 if (s->flags & UNDEF)
	 {
	    memctl[curraddr].sym = s;
	    if (!partiallink)
	    {
	       memctl[curraddr].tag = ABSDATA_TAG;
	       PUTMEM (curraddr, 0);
	    }
	    else
	    {
	       PUTMEM (curraddr, s->extndx);
	    }
#if defined(DEBUGEXTNDX)
	    printf ("   sym = %s-U, flags = %04X, addr = %04X, value = %04X",
		    s->symbol, s->flags, curraddr, GETMEM(curraddr));
	    printf (", extval = (%04X, %04X)\n",
		    memctl[curraddr].value, s->value);
#endif
	 }
	 else
	 {
	    memctl[curraddr].tag =
		   (s->flags & RELOCATABLE) ? RELDATA_TAG : ABSDATA_TAG;
	    PUTMEM (curraddr, memctl[curraddr].value + s->value);
#if defined(DEBUGEXTNDX)
	    printf (
         "   sym = %s, flags = %04X, addr = %04X, value = %04X (%04X + %04X)\n",
		    s->symbol, s->flags, curraddr, GETMEM(curraddr),
		    memctl[curraddr].value, s->value);
#endif
	 }
      }
   }
}

/************************************************************************
* processextndx - Process EXTNDX tag.
************************************************************************/

static void
processextndx (PHASENode *phase, int curraddr, int seg)
{
   SymNode *s;
   uint16 wdata;

   if (inpass == 1) return;

   if (seg == DSEG)
      wdata = GETDSEGMEM (curraddr);
   else if (seg == CSEG)
      wdata = GETCSEGMEM (curraddr);
   else
      wdata = GETMEM (curraddr);

#if defined(DEBUGEXTNDX)
   printf ("processextndx: curraddr = %04X, seg = %d, wdata = %04X\n",
	   curraddr, seg, wdata);
   printf ("   symbol = %s, extndx = %d\n",
	   extlist[wdata].sym->symbol, extlist[wdata].extndx);
#endif

   s = extlist[wdata].sym;
   if (wdata == extlist[wdata].extndx)
   {
      if (s->flags & UNDEF)
      {
	 EXTndxlist *new;

	 if ((new = malloc (sizeof (EXTndxlist))) == NULL)
	 {
	    fprintf (stderr, "lnked990: Unable to allocate memory\n");
	    exit (ABORT);
	 }
#if defined(DEBUGEXTNDX)
	 printf ("       sym = %s-U, flags = %04X, value = %04X, addr = %04X\n",
	         s->symbol, s->flags, GETMEM(curraddr), curraddr);
#endif
	 new->next = s->extlist;
	 new->location = curraddr;
	 s->extlist = new;
      }
      else
      {
	 if (seg == DSEG)
	 {
	    dsegctl[curraddr].tag =
		   (s->flags & RELOCATABLE) ? RELDATA_TAG : ABSDATA_TAG;
	    PUTDSEGMEM (curraddr, dsegctl[curraddr].value + s->value);
#if defined(DEBUGEXTNDX)
	    printf (
	        "       sym-D = %s, flags = %04X, value = %04X (%04X + %04X)\n",
		    s->symbol, s->flags, GETDSEGMEM(curraddr),
		    dsegctl[curraddr].value, s->value);
#endif
	 }
	 else if (seg == CSEG)
	 {
	    csegctl[curraddr].tag =
		   (s->flags & RELOCATABLE) ? RELDATA_TAG : ABSDATA_TAG;
	    PUTCSEGMEM (curraddr, csegctl[curraddr].value + s->value);
#if defined(DEBUGEXTNDX)
	    printf (
	        "       sym-C = %s, flags = %04X, value = %04X (%04X + %04X)\n",
		    s->symbol, s->flags, GETCSEGMEM(curraddr),
		    csegctl[curraddr].value, s->value);
#endif
	 }
	 else
	 {
	    memctl[curraddr].tag =
		   (s->flags & RELOCATABLE) ? RELDATA_TAG : ABSDATA_TAG;
	    PUTMEM (curraddr, memctl[curraddr].value + s->value);
#if defined(DEBUGEXTNDX)
	    printf (
	        "       sym-P = %s, flags = %04X, value = %04X (%04X + %04X)\n",
		    s->symbol, s->flags, GETMEM(curraddr),
		    memctl[curraddr].value, s->value);
#endif
	 }
      }
   }
}

/************************************************************************
* resetextndx - Reset EXTNDX tag.
************************************************************************/

static void
resetextndx (PHASENode *phase)
{
   int i;

   if (inpass == 1) return;

#if defined(DEBUGEXTNDX)
   printf ("resetextndx: extndx = %d\n", extndx);
#endif

   for (i = 0; i < extndx; i++)
   {
      EXTndxlist *ref, *next;

      ref = extlist[i].sym->extlist; 
      while (ref)
      {
         next = ref->next;
	 free (ref);
	 ref = next;
      }
      extlist[i].sym->extlist = NULL; 
   }
   extndx = 0;
   memset (extlist, 0, sizeof(extlist));
   for (i = 0; i < MAXSYMBOLS; i++)
   {
      extlist[i].extndx = -1;
   }
}

/************************************************************************
* lnkloader - Object loader for lnked990.
************************************************************************/

int
lnkloader (PHASENode *phase, FILE *fd, int loadpt, char *file, int libload)
{
   Module *newmodule;
   Module *dsegmodule;
   SymNode *sym;
   int binarymode = FALSE;
   int reclen;
   int curraddr;
   int modendaddr;
   int relo;
   int done;
   int i, j;
   int dsegpc;
   int csegpc;
   int indseg;
   int incseg;
   int inphase;
   int wordlen = WORDTAGLEN-1;
   int cdseglen;
   int ccseglen;
   int lastaddr;
   uint16 pseglen = 0;
   unsigned char inbuf[82];

#if defined(DEBUGLOADER) || defined(DEBUGEXTNDX)
   printf ("lnkloader: loadpt = %04X, pass = %d, libload = %s, file = %s\n",
	   loadpt, inpass, libload ? "TRUE" : "FALSE", file);
#endif

   indseg = FALSE;
   incseg = FALSE;
   inphase = FALSE;
   curraddr = loadpt;
   modendaddr = loadpt;
   
   dsegpc = phase->dseguserorg > 0 ? phase->dseguserorg : phase->dsegorg;
   csegpc = phase->cseguserorg > 0 ? phase->cseguserorg : phase->csegorg;
   cdseglen = 0;
   ccseglen = 0;

   resetextndx (phase);

   i = fgetc (fd);
   if (i == BINIDT_TAG)
      binarymode = TRUE;
   ungetc (i, fd);

   done = FALSE;
   pseglen = 0;
   while (!done)
   {
      unsigned char *op = inbuf;

      if (binarymode)
      {
	 reclen = fread (inbuf, 1, 81, fd);
	 if (feof(fd))
	 {
	    done = TRUE;
	    break;
	 }
      }
      else
      {
	 if (fgets ((char *)inbuf, 82, fd) == NULL)
	 {
	    done = TRUE;
	    break;
	 }
	 reclen = strlen ((char *)inbuf);
      }

#ifdef DEBUGLOADER
      printf ("reclen = %d : %s\n", reclen, &inbuf[71]);
#endif
      if (*op == EOFSYM)
      {
	 if (inpass == 1)
	 {
	    if (reclen > 60)
	    {
	       strncpy (newmodule->date, (char *)&inbuf[TIMEOFFSET], 8);
	       newmodule->date[9] = '\0';
	       strncpy (newmodule->time, (char *)&inbuf[DATEOFFSET], 8);
	       newmodule->time[9] = '\0';
	       strncpy (newmodule->creator,
		        (char *)&inbuf[CREATOROFFSET], 8);
	       newmodule->creator[9] = '\0';
	    }
	    else
	    {
	       strncpy (newmodule->date, "        ", 8);
	       strncpy (newmodule->time, "        ", 8);
	       strncpy (newmodule->creator, "        ", 8);
	    }
	 }

	 if (cdseglen)
	 {
	    phase->dsegorg += cdseglen;
	    phase->dseglen += cdseglen;
	    if (phase->dseguserorg > 0)
	       phase->dseguserorg += cdseglen;
	 }
	 cdseglen = 0;
	 dsegpc = phase->dseguserorg > 0 ? phase->dseguserorg : phase->dsegorg;
	 if (ccseglen)
	 {
	    phase->csegorg += ccseglen;
	    phase->cseglen += ccseglen;
	 }
	 ccseglen = 0;
	 csegpc = phase->cseguserorg > 0 ? phase->cseguserorg : phase->csegorg;

	 incseg = FALSE;
	 indseg = FALSE;
	 binarymode = FALSE;
	 loadpt += pseglen;
	 loadpt = (loadpt + 1) & 0xFFFE;
	 fillinextndx (phase);
	 if (libload) break;
	 resetextndx (phase);
	 continue;
      }

      for (i = 0; i < reclen; i++)
      {
	 char *bp;
	 uint16 wdata;
	 uint16 cmnndx;
	 char otag, ltag;
	 char item[80];

	 relo = FALSE;
	 otag = *op++;

	 wdata = getnumfield (op, binarymode);

#ifdef DEBUGLOADER
	 printf ("   otag = %c, loadpt = %04X, curraddr = %04X",
		 isprint(otag) ? otag : '0', loadpt, curraddr);
	 printf (", dsegpc = %04X, csegpc = %04X, data = %04X",
	         dsegpc, csegpc, wdata & 0xFFFF);
#endif

         switch (otag)
	 {
	 case BINIDT_TAG: /* Binary IDT */
	    binarymode = TRUE;
#ifdef DEBUGLOADER
	    printf (", Binary mode");
#endif
	    wdata = (*op << 8) | *(op+1);
	    wordlen = BINWORDTAGLEN-1;
	 case IDT_TAG:
	    op += wordlen;
	    strncpy (item, (char *)op, IDTSIZE);
	    item[IDTSIZE] = '\0';
#ifdef DEBUGLOADER
	    printf (", IDT = %s", item);
#endif
	    bp = item;
	    for (; *bp; bp++) if (isspace(*bp)) *bp = '\0';
	    pseglen = ((wdata + 1) & 0xFFFE);
	    modendaddr += ((wdata + 1) & 0xFFFE);
	    if (inpass == 1)
	    {
	       if ((newmodule = malloc (sizeof(Module))) == NULL)
	       {
	          fprintf (stderr, "lnked990: Unable to allocate memory\n");
		  exit (ABORT);
	       }
	       memset (newmodule, 0, sizeof(Module));
	       modnumber++;
	       strcpy (newmodule->objfile, file);
	       strcpy (newmodule->name, item);
	       newmodule->length = wdata;
	       newmodule->origin = loadpt;
	       newmodule->number = modnumber;
	       newmodule->creator[0] = '\0';
	       newmodule->date[0] = '\0';
	       newmodule->time[0] = '\0';
	       phase->length += pseglen;
	       if (!phase->module_head)
	       {
		  phase->module_head = newmodule;
	       }
	       else
	       {
		  phase->module_tail->next = newmodule;
	       }
	       phase->module_tail = newmodule;
	       curmod = modcount;
	       modcount++;
	    }

	    else /* pass 2 */
	    {
	       curmod++;
	       for (newmodule = phase->module_head; newmodule;
		    newmodule = newmodule->next)
	       {
	          if (!strcmp (newmodule->name, item))
		     break;
	       }
	    }
#ifdef DEBUGLOADER
            printf (", module = %d", curmod);
#endif
	    op += IDTSIZE;
	    memset (&memory[curraddr], 0, pseglen);
	    for (j = curraddr; j < (curraddr+pseglen); j++)
	    {
	       memset (&memctl[j], 0, sizeof(MemCTL));
	    }
	    break;

	 case PGMIDT_TAG:
	    op += wordlen;
	    strncpy (item, (char *)op, IDTSIZE);
	    item[IDTSIZE] = '\0';
#ifdef DEBUGLOADER
	    printf (", IDT = %s", item);
#endif
	    op += IDTSIZE;
	    break;

         case RELORG_TAG:
	    wdata += loadpt;
         case ABSORG_TAG:
	    if (!indseg && !incseg && pseglen)
	    {
	       if (wdata >= curraddr)
	       {
#ifdef DEBUGLOADER
		  printf (", ORGTAG = %04X, memtag = %c", wdata,
			  memctl[curraddr].tag ? memctl[curraddr].tag : '0');
#endif
		  lastaddr = curraddr;
		  if (!memctl[curraddr].tag || (memctl[curraddr].tag == otag))
		  {
		     PUTMEM (curraddr, wdata);
		     memctl[curraddr].tag = otag;
		  }
	       }
	       else
	       {
#ifdef DEBUGLOADER
		  printf (", BACK ORGTAG = %04X, ltag = %c:%c, lastaddr = %04X",
			  wdata, ltag, memctl[lastaddr].tag, lastaddr);
#endif
		  if ((ltag == otag) && (memctl[lastaddr].tag == otag))
		  {
		     PUTMEM (lastaddr, wdata);
		  }
	       }
	    }
	    curraddr = wdata & 0xFFFF;
	    indseg = FALSE;
	    incseg = FALSE;
	    op += wordlen;
	    break;

	 case RELDATA_TAG:
	    wdata += loadpt;
	    relo = TRUE;
	 case ABSDATA_TAG:
	    if (indseg)
	    {
#ifdef DEBUGLOADER
	       printf (", DSEGDATA");
#endif
	       PUTDSEGMEM (dsegpc, wdata);
	       dsegctl[dsegpc].tag = otag;
	       dsegpc += 2;
	    }
	    else if (incseg)
	    {
#ifdef DEBUGLOADER
	       printf (", CSEGDATA");
#endif
	       if (inphase)
	       {
		  PUTCSEGMEM (csegpc, wdata);
		  csegctl[csegpc].tag = otag;
		  csegpc += 2;
	       }
	    }
	    else
	    {
	       PUTMEM (curraddr, wdata);
	       memctl[curraddr].tag = otag;
	       curraddr += 2;
	    }
	    op += wordlen;
	    break;

	 case RELENTRY_TAG:
	    if (relentry <= 0)
	       relentry = (wdata + loadpt) & 0xFFFF;
	    op += wordlen;
	    break;

	 case ABSENTRY_TAG:
	    if (absentry <= 0)
	       absentry = wdata & 0xFFFF;
	    op += wordlen;
	    break;

	 case RELEXTRN_TAG:
	    wdata += loadpt;
	    relo = TRUE;
	 case ABSEXTRN_TAG:
	    op += wordlen;
	    strncpy (item, (char *)op, MAXSHORTSYMLEN);
	    item[MAXSHORTSYMLEN] = '\0';
	    processref (phase, newmodule, item, relo, 0, wdata, libload);
	    op += MAXSHORTSYMLEN;
	    break;

	 case RELGLOBAL_TAG:
	    wdata += loadpt;
	    relo = TRUE;
	 case ABSGLOBAL_TAG:
	    op += wordlen;
	    strncpy (item, (char *)op, MAXSHORTSYMLEN);
	    item[MAXSHORTSYMLEN] = '\0';
	    processdef (phase, newmodule, item, relo, 0, wdata, 0, libload);
	    op += MAXSHORTSYMLEN;
	    break;

	 case RELSREF_TAG:
	    if (wdata)
	    {
	       wdata += loadpt;
	       relo = TRUE;
	    }
	    op += wordlen;
	    strncpy (item, (char *)op, MAXSHORTSYMLEN);
	    item[MAXSHORTSYMLEN] = '\0';
	    processref (phase, newmodule, item, TRUE, SREF, wdata, libload);
	    op += MAXSHORTSYMLEN;
	    break;

	 case COMMON_TAG:
	    op += wordlen;
	    strncpy (item, (char *)op, MAXSHORTSYMLEN);
	    item[MAXSHORTSYMLEN] = '\0';
            op += MAXSHORTSYMLEN;
            cmnndx = getnumfield (op, binarymode);
            op += wordlen;
#ifdef DEBUGLOADER
            printf (", CMN = %s, NDX = %04X", item, cmnndx);
#endif

            if (!strncmp (item, "$DATA", 5))
	    {
	       /* $DATA sections are always index 0 */
	       strcpy (phase->cmnnodes[cmnndx].name, item);
	       phase->cmnnodes[cmnndx].length = wdata;
	       phase->cmnnodes[cmnndx].offset = 0;
	       cdseglen = wdata;
	       phase->cmnnodes[cmnndx].origin = dsegpc;
#ifdef DEBUGLOADER
	       printf (", DSEG length = %04X, origin = %04X",
		       phase->cmnnodes[cmnndx].length,
		       phase->cmnnodes[cmnndx].origin);
#endif
	       if (inpass == 1 && wdata)
	       {
		  /*
		  ** A dseg gets a dummy module entry.
		  */

		  if ((dsegmodule = malloc (sizeof(Module))) == NULL)
		  {
		     fprintf (stderr, "lnked990: Unable to allocate memory\n");
		     exit (ABORT);
		  }
		  memset (dsegmodule, 0, sizeof(Module));
		  strcpy (dsegmodule->objfile, "");
		  strcpy (dsegmodule->name, item);
		  dsegmodule->length = phase->cmnnodes[cmnndx].length;
		  dsegmodule->origin = phase->cmnnodes[cmnndx].origin;
		  dsegmodule->number = modnumber;
		  dsegmodule->dsegmodule = TRUE;
		  strncpy (dsegmodule->date, "        ", 8);
		  strncpy (dsegmodule->time, "        ", 8);
		  strncpy (dsegmodule->creator, "        ", 8);
		  if (!phase->module_head)
		  {
		     phase->module_head = dsegmodule;
		  }
		  else
		  {
		     phase->module_tail->next = dsegmodule;
		  }
		  phase->module_tail = dsegmodule;
		  modcount++;
	       }
	    }
	    else
	    {
	       CSEGUserNode *node;
	       int j;
	       int found;

	       if ((node = cseglookup (phase, item, 0, 0, 0,
				       FALSE, &inphase)) == NULL)
	       {
		  if ((node = cseglookup (phase, item, phase->cseguserorg,
					  wdata, modnumber,
					  TRUE, &inphase)) != NULL)
		  {
		     phase->csegorg = node->origin;
		     csegpc = phase->csegorg;
		  }
	       }
	       else
	       {
		  phase->csegorg = phase->cseguserorg > 0 ?
			      phase->cseguserorg : csegpc;
	       }
	       phase->cseglen = 0;
	       if (node && node->number == -1)
	       {
		  node->length = wdata;
		  node->number = modnumber;
	       }

#ifdef DEBUGCOMMON
               printf ("COMMON: item = '%s', cmnndx = %d, pass = %d\n",
		       item, cmnndx, inpass);
#endif
	       found = FALSE;
	       for (j = 1; j < (phase->cmncount+1); j++)
	       {
		  phase->csegorg = phase->cmnnodes[j].origin;
		  phase->cseglen = phase->cmnnodes[j].length;
#ifdef DEBUGCOMMON
		  printf (
		      "   j = %d, name = '%s', origin = %04X, length = %04X\n",
			  j, phase->cmnnodes[j].name,
			  phase->cmnnodes[j].origin,
			  phase->cmnnodes[j].length);
#endif
	          if (!strcmp (phase->cmnnodes[j].name, item))
		  {
#ifdef DEBUGCOMMON
		     printf ("   FOUND\n");
#endif
		     found = TRUE;
		     csegpc = phase->cmnnodes[j].origin;
		     phase->cmnnodes[cmnndx].offset = j;
		     cmnndx = j;
		     break;
		  }
	       }
	       if (!found)
	       {
#ifdef DEBUGCOMMON
		  printf ("   NOT FOUND\n");
#endif
		  phase->cmncount++;
		  if (phase->cmncount >= MAXCSEGS)
		  {
		     fprintf (stderr,
		              "lnked990: Too many Common segments: %d\n",
			      MAXCSEGS);
		     return (-1);
		  }
	          phase->cmnnodes[cmnndx].offset = phase->cmncount;
		  if (inpass == 1)
		  {
		     strcpy (phase->cmnnodes[phase->cmncount].name, item);
		     phase->cmnnodes[phase->cmncount].length = wdata;
		     phase->cmnnodes[phase->cmncount].origin =
			phase->csegorg + phase->cseglen;
		  }
		  cmnndx = phase->cmncount;
	       }

	       phase->csegorg =
		  phase->cmnnodes[phase->cmnnodes[cmnndx].offset].origin;
#ifdef DEBUGLOADER
	       printf (", CSEG length = %04X, origin = %04X",
		       phase->cmnnodes[cmnndx].length,
		       phase->cmnnodes[cmnndx].origin);
#endif
	    }
            break;
	    
         case EXTNDX_TAG:
            op += wordlen;
            cmnndx = getnumfield (op, binarymode);
            op += wordlen;
#ifdef DEBUGEXTNDX
            printf ("EXTNDX_TAG: wdata = %04X, OFFSET = %04X\n", wdata, cmnndx);
#endif
#ifdef DEBUGLOADER
            printf (", wdata = %04X, OFFSET = %04X", wdata, cmnndx);
#endif      
	    if (indseg)
	    {
	       PUTDSEGMEM (dsegpc, wdata);
	       dsegctl[dsegpc].tag = otag;
	       dsegctl[dsegpc].value = cmnndx;
	       processextndx (phase, dsegpc, DSEG);
	       dsegpc += 2;
	    }
	    else if (incseg && inphase)
	    {
	       PUTCSEGMEM (csegpc, wdata);
	       csegctl[csegpc].tag = otag;
	       csegctl[csegpc].value = cmnndx;
	       processextndx (phase, csegpc, CSEG);
	       csegpc += 2;
	    }
	    else
	    {
	       PUTMEM (curraddr, wdata);
	       memctl[curraddr].tag = otag;
	       memctl[curraddr].value = cmnndx;
	       processextndx (phase, curraddr, 0);
	       curraddr += 2;
	    }
            break;

         case DSEGORG_TAG:
            op += wordlen; 
	    phase->dsegorg = phase->cmnnodes[0].origin;

	    if (wdata >= dsegpc)
	    {
#ifdef DEBUGLOADER
	       printf (", DSEGORG = %04X, dsegpc = %04X, dsegorg = %04X",
		       wdata, dsegpc, phase->dsegorg);
#endif      
	       PUTDSEGMEM (dsegpc, phase->dsegorg+wdata);
	       dsegctl[dsegpc].tag = RELORG_TAG;
	    }
	    else
	    {
#ifdef DEBUGLOADER
	       printf (", BACK DSEGORG = %04X, dsegpc = %04X, dsegorg = %04X",
		       wdata, dsegpc, phase->dsegorg);
#endif      
	    }
            dsegpc = phase->dsegorg + wdata;
            indseg = TRUE;
	    incseg = FALSE;
            break;
         
         case DSEGDATA_TAG:
            op += wordlen;
	    wdata += phase->cmnnodes[0].origin;
#ifdef DEBUGLOADER
	    printf (", DSEGREF = %04X", wdata);
#endif
	    if (indseg)
	    {
	       PUTDSEGMEM (dsegpc, wdata);
	       if (inpass == 2)
		  dsegctl[dsegpc].tag = RELDATA_TAG;
	       dsegpc += 2;
	    }
	    else
	    {
#ifdef DEBUGLOADER
	       printf (", DSEGORG = %04X", phase->dsegorg);
#endif
	       PUTMEM (curraddr, wdata);
	       if (inpass == 2)
		  memctl[curraddr].tag = RELDATA_TAG;
	       curraddr += 2;
	    }
            break;
         
         case CMNORG_TAG:
            op += wordlen; 
            cmnndx = getnumfield (op, binarymode);
            op += wordlen;
	    j = phase->cmnnodes[cmnndx].offset;
	    phase->csegorg = phase->cmnnodes[j].origin;
	    if ((csegpc + wdata) > csegpc)
	    {
#ifdef DEBUGLOADER
	       printf (", CMNORG = %04X, NDX = %d:%d", wdata, cmnndx, j);
	       printf (", csegpc = %04X, csegorg = %04X", csegpc,
		       phase->csegorg);
#endif
	       PUTCSEGMEM (csegpc, phase->csegorg+wdata);
	       csegctl[csegpc].tag = partiallink ? otag : RELORG_TAG;
	    }
	    else
	    {
#ifdef DEBUGLOADER
	       printf (", BACK CSEGORG = %04X, NDX = %d:%d", wdata, cmnndx, j);
	       printf (", csegpc = %04X, csegorg = %04X",
	               csegpc, phase->csegorg);
#endif
	    }
            csegpc = phase->csegorg + wdata;
            incseg = TRUE;
	    indseg = FALSE;
            break;
         
         case CMNDATA_TAG:
            op += wordlen;
            cmnndx = getnumfield (op, binarymode);
            op += wordlen;
	    j = phase->cmnnodes[cmnndx].offset;
	    wdata += phase->cmnnodes[j].origin;
#ifdef DEBUGLOADER
	    printf (", CMNDATA = %04X, NDX = %d:%d", wdata, cmnndx, j);
#endif
	    if (incseg && inphase)
	    {
	       PUTCSEGMEM (csegpc, wdata);
	       if (inpass == 2)
		  csegctl[csegpc].tag = RELDATA_TAG;
	       csegpc += 2;
	    }
	    else
	    {
#ifdef DEBUGLOADER
               printf (", CSEGORG = %04X", phase->csegorg);
#endif
	       PUTMEM (curraddr, wdata);
	       if (inpass == 2)
		  memctl[curraddr].tag = RELDATA_TAG;
	       curraddr += 2;
	    }
            break;

	 case CMNEXTRN_TAG:
            op += wordlen;
	    strncpy (item, (char *)op, MAXSHORTSYMLEN);
	    item[MAXSHORTSYMLEN] = '\0';
            op += MAXSHORTSYMLEN;
            cmnndx = getnumfield (op, binarymode);
            op += wordlen;
	    j = phase->cmnnodes[cmnndx].offset;
	    wdata += phase->cmnnodes[j].origin;
#ifdef DEBUGLOADER
            printf (", NDX = %d:%d", cmnndx, j);
#endif
	    processref (phase, newmodule, item, TRUE, cmnndx ? CSEG : DSEG,
			wdata, libload);
            break;

	 case CMNGLOBAL_TAG:
            op += wordlen;
	    strncpy (item, (char *)op, MAXSHORTSYMLEN);
	    item[MAXSHORTSYMLEN] = '\0';
            op += MAXSHORTSYMLEN;
            cmnndx = getnumfield (op, binarymode);
            op += wordlen;
	    j = phase->cmnnodes[cmnndx].offset;
	    if (cmnndx == 0)
	       wdata += phase->cmnnodes[j].origin;
#ifdef DEBUGLOADER
            printf (", NDX = %d:%d", cmnndx, j);
#endif
	    processdef (phase, newmodule, item, TRUE, (cmnndx ? CSEG : DSEG),
	    		wdata, j, libload);
            break;

	 case LCMNEXTRN_TAG:
            op += wordlen;
	    strncpy (item, (char *)op, MAXSYMLEN);
	    item[MAXSYMLEN] = '\0';
            op += MAXSYMLEN;
            cmnndx = getnumfield (op, binarymode);
            op += wordlen;
	    j = phase->cmnnodes[cmnndx].offset;
	    wdata += phase->cmnnodes[j].origin;
#ifdef DEBUGLOADER
            printf (", NDX = %d:%d", cmnndx, j);
#endif
	    processref (phase, newmodule, item, TRUE,
			LONGSYM | cmnndx ? CSEG : DSEG, wdata, libload);
            break;

	 case LCMNGLOBAL_TAG:
            op += wordlen;
	    strncpy (item, (char *)op, MAXSYMLEN);
	    item[MAXSYMLEN] = '\0';
            op += MAXSYMLEN;
            cmnndx = getnumfield (op, binarymode);
            op += wordlen;
	    j = phase->cmnnodes[cmnndx].offset;
	    if (cmnndx == 0)
	       wdata += phase->cmnnodes[j].origin;
#ifdef DEBUGLOADER
            printf (", NDX = %d:%d", cmnndx, j);
#endif
	    processdef (phase, newmodule, item, TRUE,
			LONGSYM | (cmnndx ? CSEG : DSEG), wdata, j, libload);
            break;

	 case LRELEXTRN_TAG:
	    wdata += loadpt;
	    relo = TRUE;
	 case LABSEXTRN_TAG:
	    op += wordlen;
	    strncpy (item, (char *)op, MAXSYMLEN);
	    item[MAXSYMLEN] = '\0';
	    processref (phase, newmodule, item, relo, LONGSYM, wdata, libload);
	    op += MAXSYMLEN;
	    break;

	 case LRELGLOBAL_TAG:
	    wdata += loadpt;
	    relo = TRUE;
	 case LABSGLOBAL_TAG:
	    op += wordlen;
	    strncpy (item, (char *)op, MAXSYMLEN);
	    item[MAXSYMLEN] = '\0';
	    processdef (phase, newmodule, item, relo, LONGSYM, wdata,
			0, libload);
	    op += MAXSYMLEN;
	    break;

	 case LOAD_TAG:
	    relo = TRUE;
	    op += wordlen;
	    strncpy (item, (char *)op, MAXSHORTSYMLEN);
	    item[MAXSHORTSYMLEN] = '\0';
	    processref (phase, newmodule, item, relo, LOAD, wdata, libload);
	    op += MAXSHORTSYMLEN;
	    break;

	 case RELSYMBOL_TAG:
	    relo = TRUE;
	    wdata += loadpt;
	 case ABSSYMBOL_TAG:
	    op += wordlen;
	    strncpy (item, (char *)op, MAXSHORTSYMLEN);
	    item[MAXSHORTSYMLEN] = '\0';
	    op += MAXSHORTSYMLEN;
	    if ((sym = modsymlookup (phase, newmodule, item, FALSE)) == NULL)
	    {
	       if ((sym = modsymlookup (phase, newmodule, item, TRUE)) == NULL)
	       {
		  fprintf (stderr, "lnked990: Internal error\n");
		  exit (ABORT);
	       }
	       sym->flags = relo ? RELOCATABLE : ABSDEF;
	       sym->flags |= SYMTAG;
	       sym->value = wdata;
	       sym->modnum = modnumber;
	    }
	    break;

	 case LRELSYMBOL_TAG:
	    relo = TRUE;
	    wdata += loadpt;
	 case LABSSYMBOL_TAG:
	    op += wordlen;
	    strncpy (item, (char *)op, MAXSYMLEN);
	    item[MAXSYMLEN] = '\0';
	    op += MAXSYMLEN;
	    if ((sym = modsymlookup (phase, newmodule, item, FALSE)) == NULL)
	    {
	       if ((sym = modsymlookup (phase, newmodule, item, TRUE)) == NULL)
	       {
		  fprintf (stderr, "lnked990: Internal error\n");
		  exit (ABORT);
	       }
	       sym->flags = relo ? RELOCATABLE : ABSDEF;
	       sym->flags |= SYMTAG | LONGSYM;
	       sym->value = wdata;
	       sym->modnum = modnumber;
	    }
	    break;

	 case EOR_TAG:
	    i = 81;
	    break;

	 case NOCKSUM_TAG:
	 case CKSUM_TAG:
	    op += wordlen;
	    break; 

	 default:
	    fprintf (stderr, "lnkloader: UNSUPPORTED TAG, tag = %c(%02X), %s",
		     otag, otag, &inbuf[71]);
	    return (-1);
	 }
#ifdef DEBUGLOADER
	 printf ("\n");
#endif
         ltag = otag;
      }
   }

#ifdef DEBUGLOADER
   printf ("   end modendaddr = %04X\n", modendaddr);
#endif
   return (modendaddr);
}
