/************************************************************************
*
* lnkpunch - Punches out phase memory.
*
* Changes:
*   06/05/03   DGP   Original.
*   06/23/03   DGP   Put date/time and creator on EOF record.
*   06/25/03   DGP   Added punchsymbols for partial link.
*   04/08/04   DGP   Added "Cassette" mode.
*   04/09/04   DGP   Added object checksum.
*   05/25/04   DGP   Added long ref/def support.
*   04/27/05   DGP   Put IDT in record sequence column 72.
*   12/30/05   DGP   Changed SymNode to use flags.
*   06/11/09   DGP   Added CSEG support.
*   11/03/11   DGP   Changed the way uninitalized memory is used/skipped.
*   08/20/13   DGP   Many fixes to CSEG/DSEG processing.
*                    Fixed concatenated modules processing.
*   02/03/14   DGP   Added compressed format.
*   02/04/15   DGP   Fixed partial link to correct EXTNDX tag processing.
*   12/10/15   DGP   Added a.out format.
*
************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>
#include <errno.h>
#include <time.h>

#include "lnkdef.h"

extern FILE *outfd;
extern int cassette;
extern int partiallink;
extern int compressed;
extern int aoutformat;
extern int relentry;
extern int absentry;
extern int absolute;

extern struct tm *timeblk;

extern uint8 memory[MEMSIZE];
extern MemCTL memctl[MEMSIZE];
extern uint8 dsegmem[MEMSIZE];
extern MemCTL dsegctl[MEMSIZE];
extern uint8 csegmem[MEMSIZE];
extern MemCTL csegctl[MEMSIZE];

static int objcnt = 0;
static int objrecnum = 0;
static int lastorg = -1;
static char objbuf[80];
static char objrec[82];

/* a.out definition */

#define	A_FMAGIC	0407		/* normal */

/***********************************************************************
* punchword - format a word in the object record.
***********************************************************************/

static int 
punchword (char *buf, int word)
{
   int wordlen;

   if (compressed)
   {
      wordlen = 2;
      *buf++ = (word >> 8) & 0xFF;
      *buf = word & 0xFF;
   }
   else
   {
      wordlen = 4;
      sprintf (buf, WORDFORMAT, word & 0xFFFF);
   }
   return (wordlen);
}

/***********************************************************************
* punchfinish - Punch a record with sequence numbers.
***********************************************************************/

static void
punchfinish (PHASENode *phase)
{
   if (objcnt)
   {
      if (!compressed)
      {
	 int i;
	 short int cksum;
	 char temp[10];

	 cksum = 0;
	 for (i = 0; i < objcnt; i++) cksum += objrec[i];
	 cksum += CKSUM_TAG;
	 cksum = -cksum;
	 sprintf (temp, OBJFORMAT, CKSUM_TAG, cksum & 0xFFFF);
	 strncpy (&objrec[objcnt], temp, 5);
	 objcnt += 5;
      }
      objrec[objcnt++] = EOR_TAG;

      if (cassette)
      {
	 strcpy (&objrec[objcnt], "\n");
      }
      else
      {
	 if (phase->name[0])
	    strncpy (&objrec[IDTCOLUMN], phase->name, strlen(phase->name));
	 sprintf (&objrec[SEQUENCENUM], SEQFORMAT, ++objrecnum);
      }
      if (compressed)
         fwrite (objrec, 1, 81, outfd);
      else
	 fputs (objrec, outfd);
      memset (objrec, ' ', sizeof(objrec));
      objcnt = 0;
   }
}

/***********************************************************************
* punchrecord - Punch an object value into record.
***********************************************************************/

static void 
punchrecord (PHASENode *phase, int len)
{
   if (objcnt+len >= (cassette ? CHARSPERCASREC :
		     CHARSPERREC + (compressed ? 5 : 0)))
   {
      punchfinish (phase);
   }
   memcpy (&objrec[objcnt], objbuf, len);
   objcnt += len;
   objbuf[0] = '\0';
}

/***********************************************************************
* punchformat - format an object record.
***********************************************************************/

static int 
punchformat (char tag, int value1, int value2, char *symbol)
{
   int len;

   len = 1;
   objbuf[0] = tag;

   switch (tag)
   {
   case BINIDT_TAG:
   case IDT_TAG:
   case PGMIDT_TAG:
      len += punchword (&objbuf[1], value1);
      sprintf (&objbuf[len], IDTFORMAT, symbol);
      len += IDTSIZE;
      break;

   case ABSENTRY_TAG:
   case RELENTRY_TAG:
   case ABSDATA_TAG:
   case RELDATA_TAG:
   case DSEGDATA_TAG:
      len += punchword (&objbuf[1], value1);
      break;

   case ABSORG_TAG:
   case RELORG_TAG:
   case DSEGORG_TAG:
      if (value1 != lastorg)
      {
	 len += punchword (&objbuf[1], value1);
	 lastorg = value1;
      }
      else
      {
         len = 0;
      }
      break;

   case RELEXTRN_TAG:
   case ABSEXTRN_TAG:
   case RELGLOBAL_TAG:
   case ABSGLOBAL_TAG:
   case RELSREF_TAG:
   case LOAD_TAG:
   case RELSYMBOL_TAG:
   case ABSSYMBOL_TAG:
      len += punchword (&objbuf[1], value1);
      sprintf (&objbuf[len], REFFORMAT, symbol);
      len += MAXSHORTSYMLEN;
      break;

   case COMMON_TAG:
   case CMNEXTRN_TAG:
   case CMNGLOBAL_TAG:
      len += punchword (&objbuf[1], value1);
      sprintf (&objbuf[len], REFFORMAT, symbol);
      len += MAXSHORTSYMLEN;
      len += punchword (&objbuf[len], value2);
      break;
      
   case CMNORG_TAG:
   case CMNDATA_TAG:
   case EXTNDX_TAG:
      len += punchword (&objbuf[1], value1);
      len += punchword (&objbuf[len], value2);
      break;

   case LRELSYMBOL_TAG:
   case LABSSYMBOL_TAG:
   case LRELEXTRN_TAG:
   case LABSEXTRN_TAG:
   case LRELGLOBAL_TAG:
   case LABSGLOBAL_TAG:
      len += punchword (&objbuf[1], value1);
      sprintf (&objbuf[len], LREFFORMAT, symbol);
      len += MAXSYMLEN;
      break;

   default:
      fprintf (stderr, "lnkpunch: UNSUPPORTED TAG, tag = %c(%02X)\n", tag, tag);
      exit (ABORT);
   }
   return (len);
}

/***********************************************************************
* puncheof - Punch EOF mark.
***********************************************************************/

static void
puncheof (PHASENode *phase)
{
   char temp[80];

   punchfinish (phase);
   objrec[0] = EOFSYM;
   if (cassette)
   {
      strcpy (&objrec[1], "\n");
   }
   else
   {
      sprintf (temp, "%-8.8s  %02d/%02d/%02d  %02d:%02d:%02d    LNKEDT %s",
	       phase->name,
	       timeblk->tm_mon+1, timeblk->tm_mday, timeblk->tm_year - 100,
	       timeblk->tm_hour, timeblk->tm_min, timeblk->tm_sec,
	       VERSION);
      strncpy (&objrec[7], temp, strlen(temp));
      if (phase->name[0])
	 strncpy (&objrec[IDTCOLUMN], phase->name, strlen(phase->name));
      sprintf (&objrec[SEQUENCENUM], SEQFORMAT, ++objrecnum);
   }
   fputs (objrec, outfd);
}

/***********************************************************************
* punchsymbols - Punch partial link REF and DEF symbols.
***********************************************************************/

static void
punchsymbols (PHASENode *phase)
{
   SymNode *sym;
   int len;
   int i;

   for (i = 0; i < phase->symcount; i++)
   {
      sym = phase->symbols[i];

      if (sym->flags & EXTERNAL)
      {
	 if (sym->flags & LONGSYM)
	 {
	    if (sym->flags & SREF)
	       len = punchformat (LRELSREF_TAG, 0, 0, sym->symbol);
	    else
	       len = punchformat (LABSEXTRN_TAG, 0, 0, sym->symbol);
	 }
	 else
	 {
	    if (sym->flags & SREF)
	       len = punchformat (RELSREF_TAG, 0, 0, sym->symbol);
	    else
	       len = punchformat (ABSEXTRN_TAG, 0, 0, sym->symbol);
	 }
         punchrecord (phase, len);
      }
      else if (sym->flags & GLOBAL)
      {
	 if (sym->flags & LONGSYM)
	 {
	    len = punchformat ((sym->flags & RELOCATABLE) ?
				    LRELGLOBAL_TAG : LABSGLOBAL_TAG,
			       sym->value & 0xFFFF, 0, sym->symbol);
	 }
	 else
	 {
	    len = punchformat ((sym->flags & RELOCATABLE) ?
				    RELGLOBAL_TAG : ABSGLOBAL_TAG,
			       sym->value & 0xFFFF, 0, sym->symbol);
	 }
         punchrecord (phase, len);
      }
   }
}

/***********************************************************************
* punchsegment - Punch DSEG/CSEG segment memory.
***********************************************************************/

static int
punchsegment (PHASENode *phase, int origin, int endaddress, int segtype,
	      int segindex)
{
   MemCTL *mptr;
   int indseg;
   int skip;
   int i, j;
   int emitorg;
   char orgtag, sorgtag;

#ifdef DEBUGPUNCH
   printf ("punchsegment: origin = %04X, endaddress = %04X, segtype = %s\n",
           origin, endaddress, segtype == DSEG ? "DSEG" : "CSEG");
#endif

   if (segtype == DSEG)
   {
      indseg = TRUE;
      sorgtag = DSEGORG_TAG;
   }
   else
   {
      indseg = FALSE;
      sorgtag = CMNORG_TAG;
   }
   if (absolute)
      orgtag = ABSORG_TAG;
   else
      orgtag = RELORG_TAG;

   skip = FALSE;
   emitorg = FALSE;
   for (i = origin; i < endaddress; )
   {
      int len;
      int segdata;

      mptr = indseg ? &dsegctl[i] : &csegctl[i];
      if (mptr->tag)
      {
	 if (skip)
	 {
#ifdef DEBUGPUNCH
	    printf ("   skip end: new pc = >%04X\n", i);
#endif
	    if ((mptr->tag == sorgtag) || (mptr->tag == RELORG_TAG))
	    {
	       j = i;
	       i = indseg ? GETDSEGMEM(i) : GETCSEGMEM(i);
	       if (i == j) i = (i + 2) & 0xFFFE;
	    }
	    if (partiallink)
	       len = punchformat (sorgtag, i - origin, segindex, NULL);
	    else
	       len = punchformat (orgtag, i, 0, NULL);
	    punchrecord (phase, len);
	    emitorg = TRUE;
	    skip = FALSE;
	 }
         mptr = indseg ? &dsegctl[i] : &csegctl[i];
	 segdata = indseg ? GETDSEGMEM(i) : GETCSEGMEM(i);
	 if (absolute && mptr->tag == RELDATA_TAG)
	 {
	    mptr->tag = ABSDATA_TAG;
	 }
	 if (!emitorg && (mptr->tag == ABSDATA_TAG || mptr->tag == RELDATA_TAG))
	 {
	    if (partiallink)
	       len = punchformat (sorgtag, i - origin, segindex, NULL);
	    else
	       len = punchformat (orgtag, i, 0, NULL);
	    punchrecord (phase, len);
	    emitorg = TRUE;
	 }
#ifdef DEBUGPUNCH
	 printf ("    addr = %04X, tag = %c, data = %04X\n", i, mptr->tag,
	          segdata);
#endif
	 if ((mptr->tag == sorgtag) || (mptr->tag == RELORG_TAG))
	 {
	    int j;

	    if (segdata < endaddress)
	    {
	       if (mptr->tag == sorgtag)
		  len = punchformat (sorgtag, i - origin, segindex, NULL);
	       else
		  len = punchformat (mptr->tag, segdata, mptr->value, NULL);
	       punchrecord (phase, len);
	    }
	    j = i;
	    i = indseg ? GETDSEGMEM(i) : GETCSEGMEM(i);
#ifdef DEBUGPUNCH
	    printf ("   RELORG: pc = >%04X, new pc = >%04X\n", j, i);
#endif
	    if (i == j) i = (i + 2) & 0xFFFE;
	    emitorg = TRUE;
	 }
	 else
	 {
	    len = punchformat (mptr->tag, segdata, mptr->value, NULL);
	    punchrecord (phase, len);
	    i = (i + 2) & 0xFFFE;
	 }
      }
      else
      {
#ifdef DEBUGPUNCH
	 if (!skip)
	    printf ("   skip start: pc = >%04X\n", i);
#endif
	 i = (i + 2) & 0xFFFE;
	 skip = TRUE;
      }
   }

#ifdef DEBUGPUNCH
   printf ("   skip = %s, retpc = >%04X\n", skip ? "TRUE" : "FALSE", i);
#endif
   return (i);
}

/***********************************************************************
* punchaword - Punch a.out word.
***********************************************************************/

static void
punchaword (int j)
{
   fputc ((j >> 8) & 0xFF, outfd);
   fputc (j & 0xFF, outfd);
}

/***********************************************************************
* punchaout - Punch a.out format.
***********************************************************************/

static void
punchaout (PHASENode *phase)
{
   Module *module;
   int i, j;

   punchaword (A_FMAGIC);
   punchaword (phase->length - (phase->dseglen + phase->cseglen));
   punchaword (0);
   punchaword (phase->dseglen + phase->cseglen);
   punchaword (0);
   punchaword ((relentry >= 0) ? relentry : 0);
   punchaword (0);
   punchaword (1);

   for (module = phase->module_head; module; module = module->next)
   {
      int lpc = module->origin;
      int len = module->length;
      for (i = lpc; i < (lpc+len); )
      {
	 if (memctl[i].tag)
	 {
	    j = GETMEM(i);
	    if (memctl[i].tag == RELORG_TAG)
	    {
	       for (; i < j; i += 2)
	       {
		  punchaword (0);
	       }
	    }
	    else
	    {
	       punchaword (j);
	       i += 2;
	    }
	 }
	 else 
	 {
	    i += 2;
	 }
      }
   }

}

/***********************************************************************
* lnkpunch - Punch out memory for the specified phase.
***********************************************************************/

int
lnkpunch (PHASENode *phase, int punchsyms)
{
   Module *module;
   CSEGUserNode *node;
   int i, j;
   int skip;
   int lpc;
   int len;
   int plen;
   int emit;
   int lastaddr;
   char lasttag;

#if defined(DEBUGPUNCH) || defined(DEBUGEXTNDX)
   printf ("lnkpunch: %sorigin = %04X, len, = %04X, "
           "dsegorg = %04X, dseglen = %04X\n",
	    phase->dummy ? "DUMMY: " : "",
	    phase->origin, phase->length,
	    phase->dsegorg, phase->dseglen);
   printf ("   csegorg = %04X, cseglen = %04X\n",
	    phase->csegorg, phase->cseglen);
#endif
   if (phase->dummy) return (0);

   if (aoutformat)
   {
      punchaout (phase);
      return (0);
   }

   /*
   ** Punch IDT
   */

   objrecnum = 0;
   lastorg = -1;
   memset (objrec, ' ', sizeof(objrec));
   if (partiallink)
      plen = punchformat (compressed ? BINIDT_TAG : IDT_TAG,
			  phase->length - phase->dseglen - phase->cseglen,
			  0, phase->name);
   else
      plen = punchformat (compressed ? BINIDT_TAG : IDT_TAG,
			  phase->length, 0, phase->name);
   punchrecord (phase, plen);

   /*
   ** For partial links, punch DSEG and COMMON headers.
   */

   if (partiallink)
   {
      plen = punchformat (COMMON_TAG, phase->dseglen, 0, "$DATA");
      punchrecord (phase, plen);
      punchfinish (phase);

      j = 1;
      for (node = phase->cseg_head; node; node = node->next)
      {
	 plen = punchformat (COMMON_TAG, node->length, j++, node->name);
	 punchrecord (phase, plen);
      }
   }

   /*
   ** Punch entry
   */

   if (phase->number == 0)
   {
      if (relentry >= 0)
      {
	 plen = punchformat (absolute ? ABSENTRY_TAG : RELENTRY_TAG,
			     relentry, 0, NULL);
	 punchrecord (phase, plen);
      }
      else if (absentry >= 0)
      {
	 plen = punchformat (ABSENTRY_TAG, absentry, 0, NULL);
	 punchrecord (phase, plen);
      }
   }

   /*
   ** For partial links, punch external symbols.
   */

   if (partiallink)
   {
      punchsymbols (phase);
   }

   /*
   ** Punch out the PSEG area.
   */

   skip = FALSE;
   lasttag = RELORG_TAG;
   lastaddr = -1;
   for (module = phase->module_head; module; module = module->next)
   {
      /* Ignore $DATA (DSEG) modules here. */
      if (module->dsegmodule) continue;

      lpc = module->origin;
      len = module->length;
#ifdef DEBUGPUNCH
      printf ("  Module %s: lastaddr = %04X, lpc = %04X, len = %04X\n",
              module->name, lastaddr, lpc, len);
#endif

      if (lastaddr != lpc)
      {
	 plen = punchformat (absolute ? ABSORG_TAG : RELORG_TAG, lpc, 0, NULL);
	 punchrecord (phase, plen);
      }

      for (i = lpc; i < (lpc+len); )
      {
	 emit = TRUE;
#ifdef DEBUGPUNCH
         printf ("    addr = %04X, tag = %c, data = %04X\n",
		  i, memctl[i].tag, GETMEM(i));
#endif
	 if (memctl[i].tag)
	 {
	    char ptag;

	    if (skip)
	    {
#ifdef DEBUGPUNCH
	       printf ("   skip end: new pc = >%04X\n", i);
#endif
	       if (memctl[i].tag == RELORG_TAG)
	       {
		  j = i;
		  i = GETMEM(i);
		  if (i == j) i = (i + 2) & 0xFFFE;
	       }
	       plen = punchformat (absolute ? ABSORG_TAG : RELORG_TAG, i,
				   0, NULL);
	       punchrecord (phase, plen);
	       skip = FALSE;
	       lasttag = RELORG_TAG;
	    }

	    ptag = memctl[i].tag;
	    if (absolute)
	    {
	       if (ptag == RELDATA_TAG)
		  ptag = ABSDATA_TAG;
	       else if (ptag == RELORG_TAG)
		  ptag = ABSORG_TAG;
	    }

	    if (ptag == CMNDATA_TAG)
	    {
	       int origin;

	       j = memctl[i].value;
	       origin = (partiallink ? phase->cmnnodes[j].origin : 0);

	       plen = punchformat (ptag, GETMEM(i) - origin, j, NULL);
	    }
	    else if (ptag == EXTNDX_TAG)
	    {
#ifdef DEBUGEXTNDX
	       printf ("                 extndx = %04X, memctl.value = %04X\n",
		       memctl[i].sym->extndx, memctl[i].value);
#endif

	       plen = punchformat (ptag, memctl[i].sym->extndx,
				   memctl[i].value, NULL);
	    }
	    else
	    {
	       plen = punchformat (ptag, GETMEM(i), memctl[i].value, NULL);
	    }
	    if (memctl[i].tag == RELORG_TAG)
	    {
	       j = i;
	       i = GETMEM(i);
#ifdef DEBUGPUNCH
	       printf ("   RELORG: pc = >%04X, new pc = >%04X\n", j, i);
#endif
	       if (i == j)
	       {
		  i = (i + 2) & 0xFFFE;
		  emit = FALSE;
	       }
	       if (lasttag == memctl[i].tag)
		  emit = FALSE;
	    }
	    else
	    {
	       i = (i + 2) & 0xFFFE;
	    }
	    if (emit)
	    {
	       punchrecord (phase, plen);
	    }
	 }
	 else
	 {
#ifdef DEBUGPUNCH
	    if (!skip)
	       printf ("   skip start: pc = >%04X\n", i);
#endif
	    i = (i + 2) & 0xFFFE;
	    skip = TRUE;
	 }
	 lasttag = memctl[i].tag;
      }
      lastaddr = i;
   }

   /*
   ** Punch out the DSEG area(s).
   */

   for (module = phase->module_head; module; module = module->next)
   {
      if (!module->dsegmodule) continue;

      lpc = module->origin;
      len = module->length;
#ifdef DEBUGPUNCH
      printf ("  DSEG: lastaddr = %04X, lpc = %04X, len = %04X\n",
	      lastaddr, lpc, len);
#endif

      if (!partiallink && (lastaddr != lpc))
      {
	 plen = punchformat (absolute ? ABSORG_TAG : RELORG_TAG, lpc, 0, NULL);
	 punchrecord (phase, plen);
      }
      lpc = punchsegment (phase, lpc, lpc+len, DSEG, 0);
      lastaddr = lpc;
   }

   /*
   ** Punch out the CSEG area(s).
   */

   j = 1;
   for (node = phase->cseg_head; node; node = node->next)
   {
      lpc = node->origin;
      len = node->length;
#ifdef DEBUGPUNCH
      printf ("  CSEG: lastaddr = %04X, lpc = %04X, len = %04X\n",
	       lastaddr, lpc, len);
#endif
      if (!partiallink && (lastaddr != lpc))
      {
	 plen = punchformat (absolute ? ABSORG_TAG : RELORG_TAG, lpc, 0, NULL);
      }
      lpc = punchsegment (phase, lpc, lpc+len, CSEG, j++);
      lastaddr = lpc;
   }

   if (punchsyms)
   {
      for (module = phase->module_head; module; module = module->next)
      {
	 SymNode *sym;

	 if (module->dsegmodule) continue;

	 plen = punchformat (PGMIDT_TAG, module->origin, 0, module->name);
	 punchrecord (phase, plen);

	 for (i = 0; i < module->symcount; i++)
	 {
	    sym = module->symbols[i];
	    if (sym->flags & LONGSYM)
	    {
	       plen = punchformat ((sym->flags & RELOCATABLE) ?
				       LRELSYMBOL_TAG : LABSSYMBOL_TAG,
				   sym->value & 0xFFFF, 0, sym->symbol);
	    }
	    else
	    {
	       plen = punchformat ((sym->flags & RELOCATABLE) ?
				       RELSYMBOL_TAG : ABSSYMBOL_TAG,
				   sym->value & 0xFFFF, 0, sym->symbol);
	    }
	    punchrecord (phase, plen);
	 }
      }
   }

   puncheof (phase);

   return (0);
}
