/***********************************************************************
*
* lnksupt.c - Support routines for the TI 990 linker.
*
* Changes:
*   06/05/03   DGP   Original.
*   10/27/04   DGP   Set longsym to FALSE.
*   12/30/05   DGP   Changed SymNode to use flags.
*   03/31/10   DGP   Added module numbering.
*   08/21/13   DGP   Revamp library support.
*                    Default D/C SEGS at the end like SDSLNK.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <time.h>

#include "lnkdef.h"

extern int pc;
extern int absolute;
extern int modnumber;
extern int phasenum;
extern int phasecnt;
extern int proccnt;

extern PHASENode phases[MAXPHASES];
extern PHASENode procs[MAXPROCS];
extern PHASEControl pcontrol[MAXPHASES];

static Command commands[] = 
{
   { "ADJUST",    CMD_ADJUST },
   { "ALLGLOBAL", CMD_ALLGLOBAL },
   { "ALLOCATE",  CMD_ALLOCATE },
   { "COMMON",    CMD_COMMON },
   { "DATA",      CMD_DATA },
   { "DUMMY",     CMD_DUMMY },
   { "END",       CMD_END },
   { "ERROR",     CMD_ERROR },
   { "FIND",      CMD_FIND },
   { "FORMAT",    CMD_FORMAT },
   { "GLOBAL",    CMD_GLOBAL },
   { "INCLUDE",   CMD_INCLUDE },
   { "LIBRARY",   CMD_LIBRARY },
   { "LOAD",      CMD_LOAD },
   { "MAP",       CMD_MAP },
   { "NOAUTO",    CMD_NOAUTO },
   { "NOERROR",   CMD_NOERROR },
   { "NOLOAD",    CMD_NOLOAD },
   { "NOMAP",     CMD_NOMAP },
   { "NOPAGE",    CMD_NOPAGE },
   { "NOSYMT",    CMD_NOSYMT },
   { "NOTGLOBAL", CMD_NOTGLOBAL },
   { "PAGE",      CMD_PAGE },
   { "PARTIAL",   CMD_PARTIAL },
   { "PHASE",     CMD_PHASE },
   { "PROCEDURE", CMD_PROCEDURE },
   { "PROGRAM",   CMD_PROGRAM },
   { "SEARCH",    CMD_SEARCH },
   { "SHARE",     CMD_SHARE },
   { "SYMT",      CMD_SYMT },
   { "TASK",      CMD_TASK },
   { NULL,        CMD_ILLEGAL }
};

/***********************************************************************
* cmdlookup - Lookup a command in the command table.
***********************************************************************/

int
cmdlookup (char *cmd)
{
   int i;

   for (i = 0; commands[i].command; i++)
   {
      if (!strcmp (commands[i].command, cmd) ||
          !strncmp (commands[i].command, cmd, 4))
         return (commands[i].cmdvalue);
   }
   return (CMD_ILLEGAL);
}

/***********************************************************************
* lookupsym - Lookup a symbol in the symbol table.
***********************************************************************/

static SymNode *
lookupsym (PHASENode *phase, Module *module, SymNode *symtbl[], int *symcount,
	   int maxcount, char *sym, int add)
{
   SymNode *ret = NULL;
   int done = FALSE;
   int count;
   int mid;
   int last = 0;
   int lo;
   int up;
   int r;

   count = *symcount;
#ifdef DEBUGSYM
   printf ("   lookupsym: Entered: count = %d, sym = %s, add = %s\n",
	   count, sym, add ? "TRUE" : "FALSE");
#endif

   /*
   ** Empty symbol table.
   */

   if (count == 0)
   {
      if (!add) return (NULL);

#ifdef DEBUGSYM
      printf ("      add symbol at top: value = >%04X\n", pc);
#endif
      if ((symtbl[0] = (SymNode *)malloc (sizeof (SymNode))) == NULL)
      {
         fprintf (stderr, "lnked990: Unable to allocate memory\n");
	 exit (ABORT);
      }
      memset (symtbl[0], '\0', sizeof (SymNode));
      strcpy (symtbl[0]->symbol, sym);
      symtbl[0]->value = pc;
      symtbl[0]->modnum = modnumber;
      symtbl[0]->phase = phase;
      if (!absolute)
	 symtbl[0]->flags = RELOCATABLE;
      count++;
      *symcount = count;
      return (symtbl[0]);
   }

   /*
   ** Locate using binary search
   */

   lo = 0;
   up = count;
   last = -1;
   
   while (!done)
   {
      mid = (up - lo) / 2 + lo;
#ifdef DEBUGSYM
      printf ("       mid = %d, last = %d\n", mid, last);
#endif
      if (last == mid) break;
      r = strcmp (symtbl[mid]->symbol, sym);
      if (r == 0)
      {
	 if (add) return (NULL); /* must be a duplicate */
         return (symtbl[mid]);
      }
      else if (r < 0)
      {
         lo = mid;
      }
      else 
      {
         up = mid;
      }
      last = mid;
   }

   /*
   ** Not found, check to add
   */

   if (add)
   {
      SymNode *new;

#ifdef DEBUGSYM
      printf ("      add new symbol: value = >%04X\n", pc);
#endif
      if (count+1 > maxcount)
      {
         fprintf (stderr, "lnked990: Symbol table exceeded\n");
	 exit (ABORT);
      }

      if ((new = (SymNode *)malloc (sizeof (SymNode))) == NULL)
      {
         fprintf (stderr, "lnked990: Unable to allocate memory\n");
	 exit (ABORT);
      }

      memset (new, '\0', sizeof (SymNode));
      strcpy (new->symbol, sym);
      new->value = pc;
      new->modnum = modnumber;
      new->phase = phase;
      if (!absolute)
	 new->flags = RELOCATABLE;

      /*
      ** Insert pointer in sort order.
      */

      for (lo = 0; lo < count; lo++)
      {
         if (strcmp (symtbl[lo]->symbol, sym) > 0)
	 {
	    for (up = count + 1; up > lo; up--)
	    {
	       symtbl[up] = symtbl[up-1];
	    }
	    symtbl[lo] = new;
	    count++;
	    *symcount = count;
	    return (symtbl[lo]);
	 }
      }
      symtbl[count] = new;
      ret = symtbl[count];
      count++;
      *symcount = count;
   }
   return (ret);
}

/***********************************************************************
* symlookup - Lookup a symbol in the symbol table(s).
***********************************************************************/

SymNode *
symlookup (PHASENode *phase, Module *module, char *symbol, int add)
{
   SymNode *sym;
   int i;

#ifdef DEBUGSYM
   printf ("symlookup: Entered: symbol = %s, phasenum = %d\n",
	   symbol, phasenum);
#endif

   /* Attempt to lookup symbol up the phase tree */

   sym = NULL;
   for (i = phasenum; i >= 0; i--)
   {
#ifdef DEBUGSYM
      printf ("   phase = %s, count = %d\n",
	      pcontrol[i].phase->name, pcontrol[i].phase->symcount);
#endif
      if ((sym = lookupsym (phase, module, pcontrol[i].phase->symbols,
			    &(pcontrol[i].phase->symcount), MAXSYMBOLS,
			    symbol, FALSE)) != NULL)
         break;
   }

   /* Look in procs */

   if (!sym)
   {
      for (i = proccnt-1; i >= 0; i--)
      {
#ifdef DEBUGSYM
	 printf ("   proc = %s, count = %d\n",
		 procs[i].name, procs[i].symcount);
#endif
	 if ((sym = lookupsym (&procs[i], module, procs[i].symbols,
			       &(procs[i].symcount), MAXSYMBOLS,
			       symbol, FALSE)) != NULL)
	    break;
      }
   }

   /* Not found, add it in the current phase */

   if (!sym && add)
   {
      sym = lookupsym (phase, module, phase->symbols,
		       &(phase->symcount), MAXSYMBOLS,
		       symbol, add);
   }
   return (sym);
}

/***********************************************************************
* reflookup - Lookup a symbol in the symbol table(s).
***********************************************************************/

SymNode *
reflookup (PHASENode *phase, Module *module, char *symbol, int add)
{
   SymNode *sym;
   int i;

#ifdef DEBUGSYM
   printf ("reflookup: Entered: symbol = %s, phasenum = %d\n",
	   symbol, phasenum);
#endif

   /* Attempt to lookup symbol up the phase tree */

   sym = NULL;
   for (i = phasenum; i >= 0; i--)
   {
#ifdef DEBUGSYM
      printf ("   phase = %s, count = %d\n",
	      pcontrol[i].phase->name, pcontrol[i].phase->symcount);
#endif
      if ((sym = lookupsym (phase, module, pcontrol[i].phase->refsymbols,
			    &(pcontrol[i].phase->refsymcount), MAXSYMBOLS,
			    symbol, FALSE)) != NULL)
      {
	 if (sym->flags & UNDEF)
	    break;
         sym = NULL;
      }
   }

   /* Not found, add it in the current phase */

   if (!sym && add)
   {
      sym = lookupsym (phase, module, phase->refsymbols,
		       &(phase->refsymcount), MAXSYMBOLS,
		       symbol, add);
   }
   return (sym);
}

/***********************************************************************
* modsymlookup - Lookup a symbol in the modules symbol table.
***********************************************************************/

SymNode *
modsymlookup (PHASENode *phase, Module *module, char *sym, int add)
{
#ifdef DEBUGSYM
   printf ("modsymlookup: Entered: sym = %s\n", sym);
#endif
   return (lookupsym (phase, module, module->symbols, &module->symcount,
	   MAXSYMBOLS, sym, add));
}

/***********************************************************************
* gettoken - Get the next token.
***********************************************************************/

char *
gettoken (char *bp, char *token, char *term)
{
   while (*bp && isspace (*bp)) bp++;
   *term = 0;
   while (*bp)
   {
      if (isspace(*bp) || *bp == ',')
      {
         *term = *bp++;
	 break;
      }
      *token++ = *bp++;
   }
   *token = 0;
   return (bp);
}

/***********************************************************************
* csegnamelookup - Lookup a CSEG in the current phase.
***********************************************************************/

static CSEGUserNode *
csegnamelookup (PHASENode *phase, char *token, int *origin, int *len)
{
   CSEGUserNode *node;

   *origin = 0;
   *len = 0;
   for (node = phase->cseg_head; node; node = node->next)
   {
      *origin = node->origin;
      *len = node->length;
      if (!strcmp (node->name, token))
      {
	 break;
      }
   }
   return (node);
}

/***********************************************************************
* cseglookup - Lookup a CSEG.
***********************************************************************/

CSEGUserNode *
cseglookup (PHASENode *phase, char *cmnname, int cmnorigin, int cmnlength, 
            int cmnnumber, int addit, int *thisphase)
{
   CSEGUserNode *node;
   int origin;
   int len;
   int i;

#ifdef DEBUGCOMMON
   printf ("cseglookup: cmnname = %s, cmnorigin = %04X, cmnlength = %04X, "
           " cmnnumber = %d, addit = %s\n",
           cmnname, cmnorigin & 0xFFFF, cmnlength, cmnnumber,
	   addit ? "TRUE" : "FALSE");
#endif

   /* Attempt to lookup common in the phase tree */

   node = NULL;
   *thisphase = TRUE;
   for (i = phasenum; i >= 0; i--)
   {
#ifdef DEBUGCOMMON
      printf ("   phase = %s, count = %d\n",
	      pcontrol[i].phase->name, pcontrol[i].phase->cmncount);
#endif
      if ((node = csegnamelookup (pcontrol[i].phase, cmnname, 
				  &origin, &len)) != NULL)
         break;
      *thisphase = FALSE;
   }

   /* Look in procs */

   if (!node)
   {
      for (i = proccnt-1; i >= 0; i--)
      {
#ifdef DEBUGCOMMON
	 printf ("   proc = %s, count = %d\n",
		 procs[i].name, procs[i].cmncount);
#endif
	 if ((node = csegnamelookup (&procs[i], cmnname, 
				     &origin, &len)) != NULL)
	    break;
      }
   }

   if (addit && !node)
   {
      if ((node = malloc (sizeof(CSEGUserNode))) == NULL)
      {
	  fprintf (stderr, "lnked990: Unable to allocate memory\n");
	  exit (ABORT);
      }
      *thisphase = TRUE;
      memset (node, 0, sizeof(CSEGUserNode));
      strcpy (node->name, cmnname);
      node->baseorigin = cmnorigin;
      node->origin = origin + len;
      node->length = cmnlength;
      node->number = cmnnumber;

      if (!phase->cseg_head)
      {
	 phase->cseg_head = node;
      }
      else
      {
	 phase->cseg_tail->next = node;
      }
      phase->cseg_tail = node;
   }

#ifdef DEBUGCOMMON
   printf ("   node = %p, thisphase = %s\n", node, *thisphase ? "YES" : "NO");
#endif
   return (node);
}
