/***********************************************************************
*
* decimal.c - Decimal operations for the TI-990/12 computer.
*
* Changes:
*   01/09/07   DGP   Original. Hacked from Hercules decimal.c
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>

#include "simdef.h"
#include "decimal.h"

#define MAX_DECIMAL_LENGTH      32

#if defined(WIN32) && !defined(MINGW)
#define MSBMASK 0x8000000000000000I64
#define MONE -1I64
#else
#define MSBMASK 0x8000000000000000ULL
#define MONE -1LL
#endif
#define LMASK 0xFFFFFFFF

#define ASCII_BLANK 0x20
#define ASCII_PLUS  0x2B
#define ASCII_MINUS 0x2D
#define ASCII_ZERO  0x30
#define ASCII_NINE  0x39

/***********************************************************************
* CIA128 - Complement and increment 128-bit value.
***********************************************************************/

static void
cia128 (t_uint64 *acc)
{
   uint32 a0, a1, a2, a3;
   uint32 c0, c1, c2;

#ifdef DEBUG_DECIMAL
   fprintf (stderr, "CIA: acc = %16.16llx %16.16llx\n", acc[0], acc[1]);
#endif
   a0 = ~(acc[1] & LMASK);
   a1 = ~((acc[1] >> 32) & LMASK);
   a2 = ~(acc[0] & LMASK);
   a3 = ~((acc[0] >> 32) & LMASK);

   a0 = (a0 + 1) & LMASK;
   c0 = (a0 < 1);
   a1 = (a1 + c0) & LMASK;
   c1 = (c0 && (a1 == 0));
   a2 = (a2 + c1) & LMASK;
   c2 = (c1 && (a2 == 0));
   a3 = (a3 + c2) & LMASK;

   acc[1] = ((t_uint64)a1 << 32) | (t_uint64)a0;
   acc[0] = ((t_uint64)a3 << 32) | (t_uint64)a2;

#ifdef DEBUG_DECIMAL
   fprintf (stderr, "     acc = %16.16llx %16.16llx\n", acc[0], acc[1]);
#endif

}

/***********************************************************************
* decimal_to_binary - Convert decimal number to binary
*
* It performs the conversion of a 32-byte decimal ASCII number into a
* 128-bit signed binary result.
*
* Input:
*      dec     A 32 byte area containing a copy of the decimal ASCII
*              storage operand.
*      len     Length (in bytes) of the decimal input
* Output:
*      result  Points to an t_uint64[2] field which will receive the
*              result as a 128-bit signed binary number.
*      ovf     Points to an int field which will be set to 1 if
*              the result overflows 63 bits plus sign, else 0.
*              If overflow occurs, the result field will contain
*              the rightmost 64 bits of the result.
*      dxf     Points to an int field which will be set to 1 if
*              invalid digits or sign were detected, else 0.
*              The result field is not set if the dxf is set to 1.
***********************************************************************/

void
decimal_to_binary (uint8 *dec, int len, t_uint64 *result, int *ovf, int *dxf)
{
   t_uint64 dreg[2];
   t_uint64 oreg[2] = { 0, 0 };
   int      i;
   int      sign=1;
   uint8    ch;

#ifdef DEBUG_DECIMAL
   fprintf (stderr, "decimal_to_binary: dec = %s, len = %d\n", dec, len);
#endif

    /* Initialize result flags */

    *ovf = 0;
    *dxf = 0;

    /* Initialize 128-bit result accumulator */

    dreg[0] = dreg[1] = 0;

    /* Convert decimal digits to binary */

    for (i = 0; i < len; i++)
    {
	ch = dec[i];

	/* Check for blank */

	if (ch == ASCII_BLANK) ;

	/* Check for sign */

	else if (ch == ASCII_PLUS)
	{
	   sign = 1;
	}
	else if (ch == ASCII_MINUS)
	{
	   sign = -1;
	}

	else if (ch >= ASCII_ZERO && ch <= ASCII_NINE)
	{
	   int index = (i < 16) ? 1 : 0;

	   /* Accumulate digit into result */

	   dreg[index] *= 10;
	   dreg[index] += ch - ASCII_ZERO;

	   /* Set overflow indicator if an overflow has occurred */

	   if (dreg[index] < oreg[index])
	       *ovf = 1;

	   /* Save current value */

	   oreg[index] = dreg[index];
	}

        /* Data exception if digit is invalid */

        else 
        {
            *dxf = 1;
        }


    }

    /* Result is negative if sign was '-' */

    if (sign == -1)
    {
        if( (t_int64)dreg[0] == MONE)
            *ovf = 1;
        cia128 (dreg);
    }

    /* Set result field and return */

#ifdef DEBUG_DECIMAL
   fprintf (stderr, "   dreg = %lld (%llx %llx)\n", dreg[1], dreg[0], dreg[1]);
   fprintf (stderr, "   ovf = %d, dxf = %d\n", *ovf, *dxf);
#endif

    result[0] = dreg[0];
    result[1] = dreg[1];

}

/************************************************************************
* binary_to_decimal - Convert binary number to decimal
*
* Input:
*      bin     Binary number (127 bits plus sign)
* Output:
*      result  Points to a 32-byte field which will receive the
*              result as an ASCII decimal number (31 digits + sign)
*
************************************************************************/

void
binary_to_decimal (t_uint64 *bin, uint8 *result)
{
   int     i;
   int     d;
   uint8   sign;

#ifdef DEBUG_DECIMAL
   fprintf (stderr, "binary_to_decimal: bin = %lld (%llx %llx)\n",
	    bin[1], bin[0], bin[1]);
#endif

   /* Load absolute value and generate sign */

   if ((t_uint64)bin[0] < MSBMASK)
   {
      /* Value is positive */
      sign = ASCII_PLUS;
   }
   else
   {
      /* Value is negative */
      cia128 (bin);
      sign = ASCII_MINUS;
   }

   /* Store sign and decimal digits from right to left */

   memset (result, ASCII_ZERO, MAX_DECIMAL_LENGTH);
   result[MAX_DECIMAL_LENGTH-1] = sign;
   d = 1;
   for (i = MAX_DECIMAL_LENGTH - 2; d != 0 || bin[0] != 0 || bin[1] != 0; i--)
   {
      if (i < 16)
      {
	 d = bin[0] % 10;
	 bin[0] /= 10;
      }
      else
      {
	 d = bin[1] % 10;
	 bin[1] /= 10;
      }
      result[i] = d + ASCII_ZERO;
   }

   /* Convert leading zeros to blanks */

   for (i = 0; i < MAX_DECIMAL_LENGTH-2; i++)
   {
      if (result[i] != ASCII_ZERO)
	 break;
      result[i] = ASCII_BLANK;
   }

#ifdef DEBUG_DECIMAL
   fprintf (stderr, "   result = %32.32s\n", result);
#endif

}

