/***********************************************************************
*
* screen.c - Screen control routines.
*
* Changes:
*   10/20/03   DGP   Original.
*   12/28/11   DGP   Added Windows screen support.
*   04/04/14   DGP   Added general telnet support for serial devices.
*   06/02/15   DGP   Removed USS (z/OS) support.
*   
***********************************************************************/

#include <stdio.h>
#include <string.h>

#include "simdef.h"

#if defined(WIN32)

#include <windows.h>
#include <wincon.h>

static void
set_screen_pos (short rowY1, short colX1)
{
   CONSOLE_SCREEN_BUFFER_INFO csbi;
   HANDLE hStdErr;
   COORD ptConsole;
   int cons_fd;

   if (!_isatty (cons_fd = fileno (stderr)))
      return;

   if ((hStdErr = (HANDLE) _get_osfhandle (cons_fd)) == NULL)
      return;

   GetConsoleScreenBufferInfo (hStdErr, &csbi);

   ptConsole.X = colX1 - 1;
   ptConsole.Y = rowY1 - 1;

   SetConsoleCursorPosition (hStdErr, ptConsole);
}

static void
clear_screen (int whole)
{
   CONSOLE_SCREEN_BUFFER_INFO csbi;
   HANDLE hStdErr;
   DWORD dwNumCells, dwCellsWritten;
   COORD ptConsole = { 0, 0 };
   int cons_fd;

   if (!_isatty (cons_fd = fileno (stderr)))
      return;

   if ((hStdErr = (HANDLE) _get_osfhandle (cons_fd)) == NULL)
      return;

   GetConsoleScreenBufferInfo (hStdErr, &csbi);
   if (!whole)
   {
      ptConsole = csbi.dwCursorPosition;
   }
   dwNumCells = (csbi.dwSize.X - ptConsole.X) * (csbi.dwSize.Y - ptConsole.Y);
   FillConsoleOutputCharacter (hStdErr, ' ', dwNumCells, ptConsole,
			       &dwCellsWritten);
   GetConsoleScreenBufferInfo (hStdErr, &csbi);
   FillConsoleOutputAttribute (hStdErr, csbi.wAttributes, dwNumCells,
			       ptConsole, &dwCellsWritten);
   SetConsoleCursorPosition (hStdErr, ptConsole);
}

static void
erase_to_eol (int start)
{
   CONSOLE_SCREEN_BUFFER_INFO csbi;
   HANDLE hStdErr;
   DWORD dwCellsWritten;
   COORD ptConsole;
   int cons_fd;

   if (!_isatty (cons_fd = fileno (stderr)))
      return;

   if ((hStdErr = (HANDLE) _get_osfhandle (cons_fd)) == NULL)
      return;

   GetConsoleScreenBufferInfo (hStdErr, &csbi);
   ptConsole = csbi.dwCursorPosition;
   if (start)
      ptConsole.X = 0;
   FillConsoleOutputAttribute (hStdErr, csbi.wAttributes,
			       csbi.dwSize.X - ptConsole.X, ptConsole,
			       &dwCellsWritten);
   FillConsoleOutputCharacter (hStdErr, ' ', csbi.dwSize.X - ptConsole.X,
			       ptConsole, &dwCellsWritten);
}

static void
set_attribute (int Attr)
{
   CONSOLE_SCREEN_BUFFER_INFO csbi;
   HANDLE hStdErr;
   WORD attr;
   int cons_fd;

   if (!_isatty (cons_fd = fileno (stderr)))
      return;

   if ((hStdErr = (HANDLE) _get_osfhandle (cons_fd)) == NULL)
      return;

   GetConsoleScreenBufferInfo (hStdErr, &csbi);
   attr = csbi.wAttributes;

   if (Attr & VDT_ATTR_BOLD) /* High Intensity */
      attr |= FOREGROUND_INTENSITY;
   else
      attr &= ~FOREGROUND_INTENSITY;

   SetConsoleTextAttribute (hStdErr, attr);
}

#endif

/***********************************************************************
* clearscreen - clear the screen
***********************************************************************/
 
void
clearscreen (int display)
{
   if (!display) return;
#ifdef ANSICRT
   fprintf (stdout, "%c[2J", ESCAPE);
   fprintf (stdout, "%c[%d;%dH", ESCAPE, 1, 1);
#endif

#if defined(WIN32)
   clear_screen (1);
#endif
}
 
/***********************************************************************
* cleartoendscreen - clear to end of the screen
***********************************************************************/
 
void
cleartoendscreen (int display)
{
   if (!display) return;
#ifdef ANSICRT
   fprintf (stdout, "%c[J", ESCAPE);
#endif

#if defined(WIN32)
   clear_screen (0);
#endif
}
 
/***********************************************************************
* clearline - clear the line
***********************************************************************/
 
void
clearline (int display)
{
   if (!display) return;
#ifdef ANSICRT
   fprintf (stdout, "%c[2K", ESCAPE);
#endif

#if defined(WIN32)
   erase_to_eol (1);
#endif
}
 
/***********************************************************************
* cleartoendline - clear toend of the line
***********************************************************************/
 
void
cleartoendline (int display)
{
   if (!display) return;
#ifdef ANSICRT
   fprintf (stdout, "%c[K", ESCAPE);
#endif

#if defined(WIN32)
   erase_to_eol (0);
#endif
}
 
/***********************************************************************
* screenposition - position on the screen
***********************************************************************/
 
void
screenposition (int display, int row, int col)
{
   if (!display)
   {
      putc ('\n', stdout);
      return;
   }
#ifdef ANSICRT
   fprintf (stdout, "%c[%d;%dH", ESCAPE, row, col);
#endif

#if defined(WIN32)
   set_screen_pos ((short)row, (short)col);
#endif
}

/***********************************************************************
* screenattribute - Set screen attribute.
***********************************************************************/
 
void
screenattribute (int display, int Attr)
{
#ifdef ANSICRT
   uint8 setattr[] = { ESCAPE, '[', '0', 'm' };
#endif

   if (!display) return;
#ifdef ANSICRT
   if (Attr & VDT_ATTR_REVERSE) /* Reverse Image */
      setattr[2] = '7';
   else if (Attr & VDT_ATTR_BLINK) /* Blinking */
      setattr[2] = '5';
   else if (Attr & VDT_ATTR_UNDERLINE) /* Underline */
      setattr[2] = '4';
   else if (Attr & VDT_ATTR_BOLD) /* High Intensity */
      setattr[2] = '1';
   fprintf (stdout, "%s", setattr);
#endif

#if defined(WIN32)
   set_attribute (Attr);
#endif
}
 
