/***********************************************************************
*
* sim401.c - CI401 port processing for the TI 990 Simulator.
*
* Changes:
*   03/28/14   HSW   Original. To support DNOS.
*   04/11/14   DGP   Put into new serial terminal support.
*   07/21/14   DGP   Fixed LDCR into OutputWord. 
*   06/02/15   DGP   Removed USS (z/OS) support.
*
* References:
*   bitsavers /pdf/ti/990/datacomm/0945409-9701_Communication_Sys_May79.pdf
*
***********************************************************************/

/*
**
** References:
**
** CI401
**   bitsavers.com:pdf/ti/990/datacomm/0945409-9701_Communication_Sys_May79.pdf
**
** Switched vs Unswitched Lines
**    DNOS System Design Document
**    bitsavers.com:pdf/ti/990/dnos/2270512B_DNOS_SysDesignDoc_Nov83.pdf
**    page 10-25
**/

/*
**
** TODO:
**   Unify CI40 1nput/Output words 2 & 3, so copy is not needed.
**
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>

#include "simdef.h"

extern int enablepanel;
extern uint16 statreg;          /* The program status register */
extern int run;

extern uint8 memory[SYSMEMSIZE];

extern Device devices[MAXDEVICES];
extern int devcnt;

/* bit macros */
#define SET_BIT( r, b ) r |=  ( 1 << b )
#define CLR_BIT( r, b ) r &= ~( 1 << b )
#define TST_BIT( r, b ) r &   ( 1 << b )

/* set hardware line */
#define SET_DSR       SET_BIT( devinfo->InputWord[ 1 ], 0 )
#define SET_CTS       SET_BIT( devinfo->InputWord[ 1 ], 1 )
#define SET_SDCD      SET_BIT( devinfo->InputWord[ 1 ], 2 )
#define SET_RING      SET_BIT( devinfo->InputWord[ 1 ], 4 )
#define SET_DCD       SET_BIT( devinfo->InputWord[ 1 ], 5 )

/* clear hardware line */
#define CLR_DSR       CLR_BIT( devinfo->InputWord[ 1 ], 0 )
#define CLR_CTS       CLR_BIT( devinfo->InputWord[ 1 ], 1 )
#define CLR_SDCD      CLR_BIT( devinfo->InputWord[ 1 ], 2 )
#define CLR_RING      CLR_BIT( devinfo->InputWord[ 1 ], 4 )
#define CLR_DCD       CLR_BIT( devinfo->InputWord[ 1 ], 5 )

/* set flag */
#define SET_RRQ       SET_BIT( devinfo->InputWord[ 6 ], 0 )
#define SET_NSF       SET_BIT( devinfo->InputWord[ 6 ], 4 )
#define SET_TMR       SET_BIT( devinfo->InputWord[ 6 ], 5 )
#define SET_INT       SET_BIT( devinfo->InputWord[ 6 ], 6 )
#define SET_WRQ       SET_BIT( devinfo->InputWord[ 6 ], 7 )

/* clear flag */
#define CLR_RRQ       CLR_BIT( devinfo->InputWord[ 6 ], 0 )
#define CLR_NSF       CLR_BIT( devinfo->InputWord[ 6 ], 4 )
#define CLR_TMR       CLR_BIT( devinfo->InputWord[ 6 ], 5 )
#define CLR_INT       CLR_BIT( devinfo->InputWord[ 6 ], 6 )
#define CLR_WRQ       CLR_BIT( devinfo->InputWord[ 6 ], 7 )

/* Is the flag set ? */
#define TST_RRQ       TST_BIT( devinfo->InputWord[ 6 ], 0 )
#define TST_NSF       TST_BIT( devinfo->InputWord[ 6 ], 4 )
#define TST_TMR       TST_BIT( devinfo->InputWord[ 6 ], 5 )
#define TST_INT       TST_BIT( devinfo->InputWord[ 6 ], 6 )
#define TST_WRQ       TST_BIT( devinfo->InputWord[ 6 ], 7 )

/* Is the flag enanbled ? */
#define ENA_RRQ       TST_BIT( devinfo->OutputWord[ 2 ], 2 )
#define ENA_NSF       TST_BIT( devinfo->OutputWord[ 5 ], 4 )
#define ENA_TMR       TST_BIT( devinfo->OutputWord[ 5 ], 5 )
#define ENA_INT       TST_BIT( devinfo->OutputWord[ 5 ], 6 )
#define ENA_WRQ       TST_BIT( devinfo->OutputWord[ 5 ], 7 )

#define TST_DTR       TST_BIT( devinfo->OutputWord[ 2 ], 0 )
#define TST_DCD       TST_BIT( devinfo->InputWord[ 1 ], 5 )
#define TST_DSR       TST_BIT( devinfo->InputWord[ 1 ], 0 )

static uint16 CNT_Mask[16] = {
   0xFFFF, 0x0001, 0x0003, 0x0007, 0x000F, 0x001F, 0x003F, 0x007F,
   0x00FF, 0x01FF, 0x03FF, 0x07FF, 0x0FFF, 0x1FFF, 0x3FFF, 0x7FFF
};

#ifdef DEBUGCI401
static char *OutputBits[6][8] = {
   {
    /* OUTPUT WORD 0 LSB */
    "XMIT DATA 0",              /* 0 */
    "XMIT DATA 1",              /* 1 */
    "XMIT DATA 2",              /* 2 */
    "XMIT DATA 3",              /* 3 */
    "XMIT DATA 4",              /* 4 */
    "XMIT DATA 5",              /* 5 */
    "XMIT DATA 6",              /* 6 */
    "XMIT DATA 7"               /* 7 */
    },
   {
    /* OUTPUT WORD 1 LSB */
    "SYNC/DLE CHAR 0",          /* 0 */
    "SYNC/DLE CHAR 1",          /* 1 */
    "SYNC/DLE CHAR 2",          /* 2 */
    "SYNC/DLE CHAR 3",          /* 3 */
    "SYNC/DLE CHAR 4",          /* 4 */
    "SYNC/DLE CHAR 5",          /* 5 */
    "SYNC/DLE CHAR 6",          /* 6 */
    "SYNC/DLE CHAR 7"           /* 7 */
    },
   {
    /* OUTPUT WORD 2 LSB */
    "DTR",                      /* 0 */
    "RTS",                      /* 1 */
    "EN RRQ",                   /* 2 */
    "EN PARITY",                /* 3 */
    "ECHO",                     /* 4 */
    "SBS",                      /* 5 */
    "BREAK",                    /* 6 */
    "SELF TEST"                 /* 7 */
    },
   {
    /* OUTPUT WORD 3 LSB */
    "CLOCK SEL A0",             /* 0 */
    "CLOCK SEL A1",             /* 1 */
    "CLOCK SEL A2",             /* 2 */
    "ALT CLK",                  /* 3 */
    "ODD PARITY",               /* 4 */
    "SYNC MODE",                /* 5 */
    "CHAR LENGTH 0",            /* 6 */
    "CHAR LENGTH 1"             /* 7 */
    },
   {
    /* OUTPUT WORD 4 LSB */
    "CLOCK SEL B0",             /* 0 */
    "CLOCK SEL B1",             /* 1 */
    "SRTS",                     /* 2 */
    "RES MODEM LD OUT",         /* 3 */
    "PULSED LODEM LD",          /* 4 */
    "MASTER RESET",             /* 5 */
    "HALF-DUPLEX",              /* 6 */
    "AN L-B REM TEST"           /* 7 */
    },
   {
    /* OUTPUT WORD COMMON MSB */
    "ADDR 0",                   /*  8 */
    "ADDR 1",                   /*  9 */
    "ADDR 2",                   /* 10 */
    "STROBE",                   /* 11 */
    "CLR-EN NSF",               /* 12 */
    "CLR SET EN TIMER",         /* 13 */
    "EN INTERRUPT",             /* 14 */
    "EN WRQ"                    /* 15 */
    }
};

static char *InputBits[7][8] = {
   {
    /* INPUT WORD 0 LSB */
    "RCV DATA 0",               /* 0 */
    "RCV DATA 1",               /* 1 */
    "RCV DATA 2",               /* 2 */
    "RCV DATA 3",               /* 3 */
    "RCV DATA 4",               /* 4 */
    "RCV DATA 5",               /* 5 */
    "RCV DATA 6",               /* 6 */
    "RCV DATA 7"                /* 7 */
    },
   {
    /* INPUT WORD 1 LSB */
    "DSR",                      /* 0 */
    "CTS",                      /* 1 */
    "SDCD",                     /* 2 */
    "RES MODEM LD OUT",         /* 3 */
    "RING",                     /* 4 */
    "DCD",                      /* 5 */
    "UNUSED 6",                 /* 6 */
    "UNUSED 7"                  /* 7 */
    },
   {
    /* INPUT WORD 2 LSB */
    "DTR",                      /* 0 */
    "RTS",                      /* 1 */
    "EN RRQ",                   /* 2 */
    "EN PARITY",                /* 3 */
    "ECHO/DLES TRIP",           /* 4 */
    "SBS",                      /* 5 */
    "BREAK",                    /* 6 */
    "SELF-TEST"                 /* 7 */
    },
   {
    /* INPUT WORD 3 LSB */
    "CLOCK SEL A0",             /* 0 */
    "CLOCK SEL A1",             /* 1 */
    "CLOCK SEL A2",             /* 2 */
    "ALT-CLOCK",                /* 3 */
    "ODD PARITY",               /* 4 */
    "SYNC MODE",                /* 5 */
    "CHAR LEN 0",               /* 6 */
    "CHAR LEN 1"                /* 7 */
    },
   {
    /* INPUT WORD 4 LSB */
    "MODEM ID 0",               /* 0 */
    "MODEM ID 1",               /* 1 */
    "MODEM ID 2",               /* 2 */
    "MODEM ID 3",               /* 3 */
    "MODEM ID 4",               /* 4 */
    "MODEM ID 5",               /* 5 */
    "MODEM ID 6",               /* 6 */
    "GND"                       /* 7 */
    },
   {
    /* INPUT WORD 0 MSB */
    "UNUSED 8",                 /*  8 */
    "SYNC MODE",                /*  9 */
    "UNUSED 10",                /* 10 */
    "R/W BUSY",                 /* 11 */
    "PE/DLE DETECT",            /* 12 */
    "FE/SYNC DETECT",           /* 13 */
    "OVER-RUN",                 /* 14 */
    "RCV ERROR SUM"             /* 15 */
    },
   {
    /* INPUT WORD 1 MSB */
    "RRQ",                      /*  8 */
    "SYNC MODE",                /*  9 */
    "UNDER-RUN",                /* 10 */
    "R-W BUSY",                 /* 11 */
    "NSF",                      /* 12 */
    "TIMER EXP",                /* 13 */
    "INTR SUM",                 /* 14 */
    "WRQ"                       /* 15 */
    }
};
#endif

/***********************************************************************
* ci401checkint - Set interrupt flags & generate interrupt.
***********************************************************************/

static void
ci401checkint (Device * dev, int GenerateInterrupt)
{
   CI401Info *devinfo;

   devinfo = &dev->info.terminfo.controller.ci401info;

   if ((ENA_RRQ && TST_RRQ) || (ENA_NSF && TST_NSF) || (ENA_TMR && TST_TMR)
       || (ENA_WRQ && TST_WRQ))
   {
      SET_INT;
      if (ENA_INT)
      {
         if (GenerateInterrupt)
         {
            gendevinterrupt (dev);
         }
      }
   }
   else
   {
      CLR_INT;
   }
}

/***********************************************************************
* ci401timer - Startup sequence & timer.
***********************************************************************/

void
ci401timer (Device *dev)
{
   CI401Info *devinfo;
   VDTInfo *vdtinfo;

   devinfo = &dev->info.terminfo.controller.ci401info;
   vdtinfo = &dev->info.terminfo.terminal.vdtinfo;

   /* 5 mS timer */
   devinfo->ClockTicks++;
   if (devinfo->ClockTicks > 50)
   {
      /* 250 mS */
      devinfo->ClockTicks = 0;
      if (devinfo->StartupSequence)
      {
	 /* ------------------- STARTUP SEQUENCE ------------------- */
	 devinfo->StartupStep++;
#ifdef DEBUGCI401
	 fprintf (stderr,
		  "ci401timer:  StartupStep = %d\n", devinfo->StartupStep);
#endif
	 switch (devinfo->StartupStep)
	 {
	 case 1:
	    SET_RING;
	    SET_NSF;
	    ci401checkint (dev, TRUE);
	    break;

	 case 2:
	    CLR_RING;
	    ci401checkint (dev, FALSE);
	    break;

	 case 3:
	    SET_RING;
	    SET_NSF;
	    ci401checkint (dev, TRUE);
	    break;

	 case 4:
	    CLR_RING;
	    ci401checkint (dev, FALSE);
	    break;

	 case 5:
	    SET_RING;
	    SET_NSF;
	    ci401checkint (dev, TRUE);
	    break;

	 case 6:
	    CLR_RING;
	    ci401checkint (dev, FALSE);
	    break;

	 case 7:
	    SET_DCD;
	    SET_NSF;
	    ci401checkint (dev, TRUE);
	    devinfo->StartupSequence = FALSE;
	    if (dev->switches & SWSTATUS)
	    {
	       vdtinfo->CommDCD = TRUE;
	       vdtinfo->CommDSR = TRUE;
	       vdtinfo->CommState = VDT_COMMSTATE_RDY;
	    }
	    break;
	 }
	 return;
      }
      /* ------------------- 250mS TIMER ------------------- */
      SET_TMR;
   }
   if (!termdringempty (dev, &dev->info.terminfo.inring))
   {
      /* we have characters to send to the Host */
      SET_RRQ;
   }
   ci401checkint (dev, TRUE);
}

/***********************************************************************
* ci401seto - Set CRU bit to one.
***********************************************************************/

void
ci401seto (Device * dev, uint16 devaddr, uint8 disp)
{
   CI401Info *devinfo;
   VDTInfo *vdtinfo;

   devinfo = &dev->info.terminfo.controller.ci401info;
   vdtinfo = &dev->info.terminfo.terminal.vdtinfo;

   if (dev->infd == DEV_NULL)
      return;
   if (disp > 7)
   {
      devinfo->OutputIndex = 5;
      disp -= 8;
   }
   else
   {
      devinfo->OutputIndex = devinfo->WordIndex;
   }
#ifdef DEBUGCI401
   fprintf (stderr, "CRU %04X SBO %d ( %s )", devaddr, disp,
            OutputBits[devinfo->OutputIndex][disp]);
   fprintf (stderr, ", OutputIndex = %d", devinfo->OutputIndex);
#endif
   /*
    ** data bits to Output Register
    */
   SET_BIT (devinfo->OutputWord[devinfo->OutputIndex], disp);
   switch (devinfo->OutputIndex)
   {
   case 0:
      break;

   case 1:
      break;

   case 2:
      devinfo->InputWord[devinfo->OutputIndex] =
	    devinfo->OutputWord[devinfo->OutputIndex];
      switch (disp)
      {
      case 0: /* DTR */
         SET_DSR;
         SET_DCD;               /* NEW */
         SET_NSF;
	 if (dev->switches & SWSTATUS)
	 {
	    vdtinfo->CommDTR = TRUE;
	    vdtinfo->CommDSR = TRUE;
	    vdtinfo->CommDCD = TRUE;
	 }
#ifdef DEBUGCI401
         fprintf (stderr, ", DTR ON");
#endif
         break;

      case 1: /* RTS */
         SET_CTS;
         SET_NSF;
         SET_TMR;
         break;

      case 2: /* RRQ */
         CLR_RRQ;
         break;

      }
      break;

   case 3:
      devinfo->InputWord[devinfo->OutputIndex] =
	    devinfo->OutputWord[devinfo->OutputIndex];
      break;

   case 4:
      switch (disp)
      {
      case 2: /* SRTS */
         SET_SDCD;
         SET_NSF;
         break;

      case 5: /* MASTER RESET */
         if (TRUE)
         {
            int i;

            for (i = 0; i < 6; i++)
            {
               devinfo->OutputWord[i] = 0;
            }
            for (i = 0; i < 7; i++)
            {
               devinfo->InputWord[i] = 0;
            }
         }
         break;
      }
      break;

   case 5: /* COMMON MSB */
      switch (disp)
      {
      case 0:
      case 1:
      case 2:
         /* Address Register */
         devinfo->AddressRegister =
            devinfo->OutputWord[devinfo->OutputIndex] & 0x07;
         switch (devinfo->AddressRegister)
         {
         case 0:
            devinfo->WordIndex = 0;
            break;

         case 1:
            devinfo->WordIndex = 2;
            break;

         case 2:
            devinfo->WordIndex = 1;
            break;

         case 3:
            break;

         case 4:
            devinfo->WordIndex = 3;
            break;

         case 5:
            devinfo->WordIndex = 4;
            break;

         case 6:
            devinfo->WordIndex = 4;
            break;

         case 7:
            break;
         }
         break;

      case 3: /* STROBE */
         devinfo->OutputWord[devinfo->OutputIndex] &= 0xF0;
         devinfo->WordIndex = 0;
         break;

      case 4: /* CLR-EN NSF */
         CLR_NSF;
         break;

      case 5: /* CLR SET EN TIMER */
         CLR_TMR;
         break;

      case 6: /* EN INTERRUPT */
         dev->intenabled = TRUE;
         /* Startup */
	 devinfo->StartupSequence = TRUE;
	 devinfo->StartupStep = 0;
         break;

      case 7: /* EN WRQ */
         SET_WRQ;
#ifdef DEBUGCI401
         fprintf (stderr, ", WRQ ON");
#endif
         break;
      }
      break;
   }
#ifdef DEBUGCI401
   fprintf (stderr, "\n");
#endif

   ci401checkint (dev, FALSE);
}

/***********************************************************************
* ci401setz - Set CRU bit to zero.
***********************************************************************/

void
ci401setz (Device * dev, uint16 devaddr, uint8 disp)
{
   CI401Info *devinfo;
   VDTInfo *vdtinfo;

   devinfo = &dev->info.terminfo.controller.ci401info;
   vdtinfo = &dev->info.terminfo.terminal.vdtinfo;

   if (dev->infd == DEV_NULL)
      return;
   if (disp > 7)
   {
      devinfo->OutputIndex = 5;
      disp -= 8;
   }
   else
   {
      devinfo->OutputIndex = devinfo->WordIndex;
   }
#ifdef DEBUGCI401
   fprintf (stderr, "CRU %04X SBZ %d ( %s )", devaddr, disp,
            OutputBits[devinfo->OutputIndex][disp]);
   fprintf (stderr, ", OutputIndex = %d", devinfo->OutputIndex);
#endif
   /* data bits to Output Register */
   CLR_BIT (devinfo->OutputWord[devinfo->OutputIndex], disp);
   switch (devinfo->OutputIndex)
   {
   case 0:
      break;

   case 1:
      break;

   case 2:
      devinfo->InputWord[devinfo->OutputIndex] =
	    devinfo->OutputWord[devinfo->OutputIndex];
      switch (disp)
      {
      case 0: /* DTR */
         CLR_DSR;
         CLR_DCD;               /* NEW */
         SET_NSF;
	 if (dev->switches & SWSTATUS)
	 {
	    vdtinfo->CommDTR = FALSE;
	    vdtinfo->CommDSR = FALSE;
	    vdtinfo->CommDCD = FALSE;
	 }
#ifdef DEBUGCI401
         fprintf (stderr, ", DTR OFF");
#endif
         break;

      case 1: /* RTS */
         CLR_CTS;
         SET_NSF;
         break;

      case 2: /* RRQ */
         CLR_RRQ;
         break;
      }
      break;

   case 3:
      devinfo->InputWord[devinfo->OutputIndex] =
	    devinfo->OutputWord[devinfo->OutputIndex];
      break;

   case 4:
      switch (disp)
      {
      case 2: /* SRTS */
         CLR_SDCD;
         SET_NSF;

         break;
      case 5: /* MASTER RESET */
         break;
      }
      break;

   case 5: /* common */
      switch (disp)
      {
      case 0:
      case 1:
      case 2:
         /* Address Register */
         devinfo->AddressRegister =
            devinfo->OutputWord[devinfo->OutputIndex] & 0x07;
         switch (devinfo->AddressRegister)
         {
         case 0:
            devinfo->WordIndex = 0;
            break;

         case 1:
            devinfo->WordIndex = 2;
            break;

         case 2:
            devinfo->WordIndex = 1;
            break;

         case 3:
            break;

         case 4:
            devinfo->WordIndex = 3;
            break;

         case 5:
            devinfo->WordIndex = 4;
            break;

         case 6:
            devinfo->WordIndex = 4;
            break;

         case 7:
            break;
         }
         break;

      case 3: /* STROBE */
         devinfo->OutputWord[devinfo->OutputIndex] &= 0xF0;
         devinfo->WordIndex = 1;
         break;

      case 4: /* CLR-EN NSF */
         CLR_NSF;
         break;

      case 5: /* CLR SET EN TIMER */
         CLR_TMR;
         break;

      case 6: /* EN INTERRUPT */
         dev->intenabled = FALSE;
         break;

      case 7: /* EN WRQ */
         CLR_WRQ;
#ifdef DEBUGCI401
         fprintf (stderr, ", WRQ OFF");
#endif
         break;
      }
      break;
   }
#ifdef DEBUGCI401
   fprintf (stderr, "\n");
#endif

   ci401checkint (dev, FALSE);
}

/***********************************************************************
* ci401tb - Test CRU bit.
***********************************************************************/

void
ci401tb (Device * dev, uint16 devaddr, uint8 disp)
{
   CI401Info *devinfo;

   devinfo = &dev->info.terminfo.controller.ci401info;

   if (dev->infd == DEV_NULL)
      return;
   CLR_EQ;

   if (disp > 7)
   {
      if (devinfo->WordIndex == 0)
      {
         devinfo->InputIndex = 5;
      }
      else
      {
         devinfo->InputIndex = 6;
      }
      disp -= 8;
   }
   else
   {
      devinfo->InputIndex = devinfo->WordIndex;
   }
#ifdef DEBUGCI401
   fprintf (stderr, "CRU %04X TB %d ( %s )", devaddr, disp,
            InputBits[devinfo->InputIndex][disp]);
   fprintf (stderr, ", InputIndex = %d", devinfo->InputIndex);
#endif
   /* data bits from Input Register */
   if (TST_BIT (devinfo->InputWord[devinfo->InputIndex], disp))
   {
      SET_EQ;
   }

   ci401checkint (dev, FALSE);

#ifdef DEBUGCI401
   fprintf (stderr, " %s\n", IS_EQ ? "TRUE" : "FALSE");
#endif
}

/***********************************************************************
* ci401ldcr - Load CRU.
***********************************************************************/

void
ci401ldcr (Device * dev, uint16 devaddr, uint16 cnt, uint16 dat)
{
   CI401Info *devinfo;

   devinfo = &dev->info.terminfo.controller.ci401info;

   if (dev->outfd == DEV_NULL)
      return;
   devinfo->OutputIndex = devinfo->WordIndex;

#ifdef DEBUGCI401
   if (dat >= 0x20 && dat <= 0x7E)
      fprintf (stderr, "CRU %04X LDCR CNT %d DAT %02X( %c )",
               devaddr, cnt, dat, dat);
   else
      fprintf (stderr, "CRU %04X LDCR CNT %d DAT %04X", devaddr, cnt, dat);
   fprintf (stderr, ", OutputIndex = %d", devinfo->OutputIndex);
#endif
#ifdef DEBUGCI401BITS
   fprintf (stderr, ", old value = %02X",
            devinfo->OutputWord[devinfo->OutputIndex]);
#endif
   /* data bits to Output Register */
   dat &= CNT_Mask[cnt];
#ifdef DEBUGCI401BITS
   fprintf (stderr, ", set bits = %02X", dat);
#endif
   devinfo->OutputWord[devinfo->OutputIndex] &= ~CNT_Mask[cnt];
   devinfo->OutputWord[devinfo->OutputIndex] |= dat;
#ifdef DEBUGCI401BITS
   fprintf (stderr, ", new value = %02X",
            devinfo->OutputWord[devinfo->OutputIndex]);
#endif
   if ((cnt == 0 || cnt > 8) && (devinfo->OutputIndex == 0))
   {
      devinfo->OutputWord[5] &= ~(CNT_Mask[cnt] >> 8);
      devinfo->OutputWord[5] = (dat >> 8);
   }

   switch (devinfo->OutputIndex)
   {
   case 0: /* OUTPUT WORD 0 */
   case 1: /* OUTPUT WORD 1 */
      /* HACK -- there is something wrong with the Register Addressing
      **-- this should only occur on OUTPUT WORD 0 */
#ifdef DEBUGCI401BITS
      if (dat >= 0x20 && dat <= 0x7E)
      {
         fprintf (stderr, ", output DAT %02X == ( %c )", dat, dat);
      }
      else
      {
         fprintf (stderr, ", output DAT %02X -> ( %c )", dat, dat & 0x7F);
      }
#endif
      serialoutput (dev, devinfo->OutputWord[devinfo->OutputIndex]);
      SET_WRQ;
      break;

   case 2: /* OUTPUT WORD 2 */
      devinfo->InputWord[devinfo->OutputIndex] =
	    devinfo->OutputWord[devinfo->OutputIndex];
      break;

   case 3: /* OUTPUT WORD 3 */
      devinfo->InputWord[devinfo->OutputIndex] =
	    devinfo->OutputWord[devinfo->OutputIndex];
      break;

   case 4: /* OUTPUT WORD 4 */
      break;
   }

#ifdef DEBUGCI401
   fprintf (stderr, "\n");
#endif
   ci401checkint (dev, TRUE);
}

/***********************************************************************
* ci401stcr - Store CRU.
***********************************************************************/

uint16
ci401stcr (Device * dev, uint16 devaddr, uint16 cnt)
{
   CI401Info *devinfo;

   uint16 dat = 0;

   devinfo = &dev->info.terminfo.controller.ci401info;
   if (dev->infd == DEV_NULL)
      return 0;
#ifdef DEBUGCI401
   fprintf (stderr, "CRU %04X STCR CNT %d", devaddr, cnt);
#endif
   devinfo->InputIndex =
      devinfo->WordIndex;
#ifdef DEBUGCI401
   fprintf (stderr, ", InputIndex = %d", devinfo->InputIndex);
#endif
   /* data bits from Input Register */
   switch (devinfo->InputIndex)
   {
   case 0: /* INPUT WORD 0 */
      if (!termdringempty (dev, &dev->info.terminfo.inring))
      {
         devinfo->InputWord[0] = dev->inchar =
	    termdgetring (dev, &dev->info.terminfo.inring);
	 dev->select = FALSE;
#ifdef DEBUGCI401BITS
         dat = devinfo->InputWord[0];
         if (dat >= 0x20 && dat <= 0x7E)
         {
            fprintf (stderr, ", input DAT %02X( %c )", dat, dat);
         }
         else if ((dat & 0x7F) >= 0x20 && (dat & 0x7F) <= 0x7E)
         {
            fprintf (stderr, ", input DAT %02X == ( %c )", dat, dat & 0x7F);
         }
         else
         {
            fprintf (stderr, "Input DAT %02X", dat);
         }
#endif
         if (!termdringempty (dev, &dev->info.terminfo.inring))
         {
            SET_RRQ;
         }
         else
         {
            CLR_RRQ;
         }
      }
      else
      {
         devinfo->InputWord[0] = 0;
#ifdef DEBUGCI401
         fprintf (stderr, ", Read Overrun");
#endif
         CLR_RRQ;
      }
      break;

   case 1: /* INPUT WORD 1 */
      break;

   case 2: /* INPUT WORD 2 */
      break;

   case 3: /* INPUT WORD 3 */
      break;

   case 4: /* INPUT WORD 4 */
      break;
   }

   dat = devinfo->InputWord[devinfo->InputIndex];
   if (cnt == 0 || cnt > 8)
   {
      if (devinfo->InputIndex == 0)
      {
         /* INPUT WORD 0 MSB */
         dat |= (devinfo->InputWord[5] << 8);
      }
      else
      {
         /* INPUT WORD 1 MSB */
         dat |= (devinfo->InputWord[6] << 8);
      }
   }
   dat &= CNT_Mask[cnt];
#ifdef DEBUGCI401
   fprintf (stderr, ", dat = %04X\n", dat);
#endif
   ci401checkint (dev, FALSE);
   return dat;
}
