/***********************************************************************
*
* sim403.c - CI403/CI404 TILINE IO processing for the TI 990 Simulator.
*
* Changes:
*   04/14/14   DGP   Original.
*   06/02/15   DGP   Removed USS (z/OS) support.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>

#if defined (WIN32)
#define __TTYROUTINES 0
#include <conio.h>
#include <windows.h>
#include <signal.h>
#include <process.h>
#include <winsock.h>
#endif

#if defined(UNIX)
#include <unistd.h>
#include <sys/time.h>
#include <netdb.h>
#include <sys/types.h>
#include <pthread.h>
#include <semaphore.h>
#endif

#include "simdef.h"

extern int devcnt;

extern uint8 memory[SYSMEMSIZE];
extern Device devices[MAXDEVICES];
extern char view[MAXVIEW][MAXVIEWLEN+1];
extern char *emulations[];

/*
** Command words access macros
*/

#define GETW1(d)   GETMEM0((d)->devaddr+TPCSSTART+2)
#define PUTW1(d,v) PUTMEM0((d)->devaddr+TPCSSTART+2,v)
#define GETW2(d)   GETMEM0((d)->devaddr+TPCSSTART+4)
#define PUTW2(d,v) PUTMEM0((d)->devaddr+TPCSSTART+4,v)
#define GETW3(d)   GETMEM0((d)->devaddr+TPCSSTART+6)
#define PUTW3(d,v) PUTMEM0((d)->devaddr+TPCSSTART+6,v)

#define GETSTAT(d) (((d) >> 12) & 0x7)
#define GETCHAN(d) (((d) >> 8) & 0x7)
#define GETDATA(d) ((d) & 0xFF)

#define SETSTAT(v) (((v) & 07) << 12)
#define SETCHAN(v) (((v) & 07) << 8)
#define SETDATA(v) ((v) & 0xFF)

/*
** Command word control bits
*/

#define IPBIT	0x8000		/* Interrupt Pending */
#define IEBIT	0x4000		/* Interrupt Enable */
#define ISBIT	0x2000		/* Interrupt Select */
#define IRBIT	0x1000		/* Input Ready */
#define MRBIT	0x0800		/* Master Reset */
#define TEBIT	0x0400		/* Timer Enable */
#define FABIT	0x0100		/* Failed channel test */
#define TSTBIT	0x0080		/* Test mode */
#define C403BIT 0x0003		/* CI 403 ID */
#define C404BIT 0x0004		/* CI 404 ID */

#define RWBIT	0x8000		/* R/W bit */

#define DAIBIT	0x8000		/* Data invalid */
#define CHGBIT  0x0800		/* Channel change */

#define TICKS	10		/* Timer ticks */

/***********************************************************************
* ci403ringempty - Is ring empty.
***********************************************************************/

static int
ci403ringempty (Device *dev)
{
   int status;
   CI403Info *devinfo;

   devinfo = &dev->info.terminfo.controller.ci403info;
   status = FALSE;

   MUTEX_LOCK (devinfo->W3ringmutex);

   if (devinfo->W3head == devinfo->W3tail)
      status = TRUE;

   MUTEX_UNLOCK (devinfo->W3ringmutex);

#ifdef DEBUGCI403RING1
   fprintf (stderr, "ci403ringempty: empty = %s\n", status ? "YES" : "NO");
#endif
   return (status);
}

/***********************************************************************
* ci403putring - Put data in ring buffer.
***********************************************************************/

static void
ci403putring (Device *dev, uint16 data)
{
   CI403Info *devinfo;
   uint16 W0;

   devinfo = &dev->info.terminfo.controller.ci403info;

   MUTEX_LOCK (devinfo->W3ringmutex);

#ifdef DEBUGCI403RING
   fprintf (stderr, "ci403putring: head = %d, tail = %d, data = >%04X\n",
            devinfo->W3head, devinfo->W3tail, data);
#endif

   devinfo->W3ring[devinfo->W3head++] = data;
   if (devinfo->W3head == CI4RINGLEN)
      devinfo->W3head = 0;

   W0 = devinfo->oldW0 & ~IPBIT;
   W0 |= (dev->intenabled ? IPBIT : 0) | IRBIT | C403BIT;
   devinfo->oldW0 = W0;
   PUTW0 (dev, W0);
#ifdef DEBUGCI403RING
   fprintf (stderr, "   W0 = >%04X\n", W0);
#endif
   if (dev->intenabled)
      gendevinterrupt (dev);

   MUTEX_UNLOCK (devinfo->W3ringmutex);
}

/***********************************************************************
* ci403getring - Get data from ring buffer.
***********************************************************************/

static uint16
ci403getring (Device *dev)
{
   CI403Info *devinfo;
   uint16 data;

   devinfo = &dev->info.terminfo.controller.ci403info;
   data = 0xFF00;

   MUTEX_LOCK (devinfo->W3ringmutex);

   if (devinfo->W3head != devinfo->W3tail)
   {
#ifdef DEBUGCI403RING
      fprintf (stderr, "ci403getring: head = %d, tail = %d",
	       devinfo->W3head, devinfo->W3tail);
#endif
      data = devinfo->W3ring[devinfo->W3tail++];
      if (devinfo->W3tail == CI4RINGLEN)
	 devinfo->W3tail = 0;
#ifdef DEBUGCI403RING
      fprintf (stderr, ", data = >%04X\n", data);
#endif
   }

   MUTEX_UNLOCK (devinfo->W3ringmutex);

   return (data);
}

/***********************************************************************
* ci403findunit - Find the unit.
***********************************************************************/

static Device *
ci403findunit (Device *dev, int unit)
{
   Device *udev;
   int i;

   udev = NULL;
   for (i = 0; i < devcnt; i++)
   {
      if (devices[i].devaddr == dev->devaddr &&
          devices[i].info.terminfo.controller.ci403info.unit == unit)
      {
	 udev = &devices[i];
	 break;
      }
   }
   return (udev);
}

/***********************************************************************
* ci403timer - Check ci403timer. Called from the clock handler.
***********************************************************************/

void
ci403timer (Device *dev)
{
   CI403Info *devinfo;

   devinfo = &dev->info.terminfo.controller.ci403info;

   if ((devinfo->unit == 0) && devinfo->timerenb && (devinfo->timecnt > 0))
   {
      if (--devinfo->timecnt == 0)
      {
	 devinfo->timecnt = TICKS;
	 devinfo->timerint++;
      }
   }
}

/***********************************************************************
* ci403task - Process input task.
***********************************************************************/

static void
ci403task (void *vdev)
{
   Device *dev;
   CI403Info *devinfo;
   int i;
   int delaycnt;
   uint16 W3;

   dev = (Device *)vdev;
   devinfo = &dev->info.terminfo.controller.ci403info;

#ifdef DEBUGCI403
   fprintf (stderr, "ci403task: ENTRY: devaddr = >%04X, unit = %d\n",
	    dev->devaddr, devinfo->unit);
#endif

   delaycnt = 0;
   while (TRUE)
   {
      int processed;

      processed = FALSE;
      if (devinfo->unit == 0)
      {
	 Device *udev;

	 /*
	 ** Check the rings for input.
	 */

	 for (i = 0; i < CI4CHANS; i++)
	 {
	    if ((udev = ci403findunit (dev, i)) != NULL)
	    {
	       int c;

	       if (!termdringempty (udev, &udev->info.terminfo.inring))
	       {
		  c = termdgetring (udev, &udev->info.terminfo.inring) & 0xFF;
		  W3 = CHGBIT | SETSTAT (0) | SETCHAN (i) | SETDATA (c);
#ifdef DEBUGCI403
		  fprintf (stderr,
			   "  chr W3 = >%04X, chan = %d, c = >%02X(%c)\n",
			   W3, i, c, isprint(c) ? c : '.');
#endif
		  ci403putring (dev, W3);
		  devinfo->timerint = 0;
		  processed = TRUE;
	       }
	    }
	 }

	 /*
	 ** Check for timer event.
	 */

	 if (devinfo->timerint)
	 {
	    W3 = CHGBIT | SETSTAT (4) | SETCHAN (7) | SETDATA (0);
	    ci403putring (dev, W3);
	    devinfo->timerint = 0;
	 }
      }

      if (!processed)
      {
	 if (++delaycnt > 100)
	 {
	    smallsleep (100);
	    delaycnt = 101;
	 }
      }
      else
      {
         delaycnt = 0;
      }
   }
}

/***********************************************************************
* ci403getcmd - Do 403 get commands.
***********************************************************************/

uint16
ci403getcmd (Device *dev, uint32 addr)
{
   Device *udev;
   CI403Info *udevinfo;
   uint16 data, W0;

   /*
   ** Only unit 0 handles the channel commands.
   */

   udev = ci403findunit (dev, 0);
   udevinfo = &udev->info.terminfo.controller.ci403info;

#ifdef DEBUGCI403CMD
   fprintf (stderr, "ci403getcmd: ENTRY: addr = >%06X, unit = %d",
	    addr, dev->info.terminfo.controller.ci403info.unit);
#endif

   switch (addr & 0x0F)
   {
   case 0: /* W0 */
   case 1:
      data = W0 = GETW0 (udev);
      W0 &= 0xFF00;
      if (udevinfo->mreset)
      {
         if (--udevinfo->mreset == 0)
	 {
	    W0 &= ~MRBIT;
	    udevinfo->oldW0 = W0;
	 }
      }
      else
      {
	 if (!ci403ringempty (udev))
	    W0 |= IRBIT;
      }
      PUTW0 (udev, W0 | C403BIT);
      break;

   case 2: /* W1 */
   case 3:
      data = GETW1 (udev);
      break;

   case 4: /* W2 */
   case 5:
      data = GETW2 (udev);
      PUTW2 (udev, 0xFF00);
      break;

   case 6: /* W3 */
   case 7:
      if (!ci403ringempty (udev))
      {
	 W0 = udevinfo->oldW0 & ~IPBIT;
	 W0 |= (udev->intenabled ? IPBIT : 0) | IRBIT | C403BIT;
	 data = ci403getring (udev);
	 PUTW3 (udev, data);
	 PUTW0 (udev, W0);
	 if (udev->intenabled)
	    gendevinterrupt (udev);
      }
      else
      {
	 data = 0xFF00;
	 W0 = GETW0 (udev);
	 if (W0 & IRBIT) W0 &= ~IRBIT;
	 if (W0 & IPBIT) W0 &= ~IPBIT;
	 PUTW3 (udev, data);
	 PUTW0 (udev, W0 | C403BIT);
      }
      break;

   default: ;
   }

#ifdef DEBUGCI403CMD
      fprintf (stderr, ", data = >%04X\n", data);
#endif
   return (data);
}

/***********************************************************************
* ci403putcmd - Do 403 commands.
***********************************************************************/

void
ci403putcmd (Device *pdev, uint32 addr, uint16 pdata)
{
   Device *dev;
   Device *udev;
   CI403Info *devinfo;
   CI403Info *udevinfo;
   int i;
   int chan;
   uint16 W0, W1, W2, W3;
   uint16 data;

   /*
   ** Only unit 0 handles the channel commands.
   */

   dev = ci403findunit (pdev, 0);
   devinfo = &dev->info.terminfo.controller.ci403info;
   PUTMEM0 (addr, pdata);

#ifdef DEBUGCI403CMD
   fprintf (stderr,
	    "ci403putcmd: ENTRY: addr = >%06X, unit = %d, data = >%04X\n",
	    addr, devinfo->unit, pdata);
#endif

   switch (addr & 15)
   {
   case 0: /* W0 */
   case 1:
      W0 = GETW0 (dev);
#ifdef DEBUGCI403
      fprintf (stderr, "  W0 = >%04X", W0);
#endif
      if (W0 & MRBIT) /* MASTER RESET */
      {
	 dev->intenabled = FALSE;
	 for (i = 0; i < CI4CHANS; i++)
	 {
	    if ((udev = ci403findunit (dev, i)) != NULL)
	    {
	       udevinfo = &udev->info.terminfo.controller.ci403info;
	       udevinfo->regs[0] = 0x00;
	       udevinfo->regs[1] = 0x0D;
	       udevinfo->regs[2] = 0x01;
	       udevinfo->regs[3] = 0x1A;
	       udevinfo->regs[4] = 0x10;
	       udevinfo->regs[5] = 0x00;
	       udevinfo->regs[6] = 0xB9;
	       udevinfo->regs[7] = 0x00;
	       udevinfo->oldW0 = 0;
	       udevinfo->W3head = udevinfo->W3tail = 0;
	       udev->info.terminfo.inring.head = 0;
	       udev->info.terminfo.inring.tail = 0;
	       udev->info.terminfo.outring.head = 0;
	       udev->info.terminfo.outring.tail = 0;
	       udev->info.terminfo.prcring.head = 0;
	       udev->info.terminfo.prcring.tail = 0;
	    }
	 }
	 devinfo->mreset = 20;
      }

      else
      {
	 if (W0 & IEBIT) /* Interrupt Enable */
	 {
	    dev->intenabled = TRUE;
	 }
	 else
	 {
	    dev->intenabled = FALSE;
	 }

	 if (W0 & TEBIT) /* Timer Enable */
	 {
	    devinfo->timerenb = TRUE;
	    devinfo->timecnt = TICKS;
	 }
	 else
	 {
	    devinfo->timerenb = FALSE;
	 }

	 if (W0 & IRBIT) /* Input Ready */
	    W0 &= ~IRBIT;
	 if (!ci403ringempty (dev)) W0 |= IRBIT;
      }

      W0 |= C403BIT;

#ifdef DEBUGCI403
      fprintf (stderr, ", new W0 = >%04X\n", W0);
#endif
      devinfo->oldW0 = W0;
      PUTW0 (dev, W0);
      break;

   case 2: /* W1 */
   case 3:
      W1 = GETW1 (dev);
      chan = GETCHAN(W1);
      if ((udev = ci403findunit (dev, chan)) != NULL)
      {
	 int reg;

	 udevinfo = &udev->info.terminfo.controller.ci403info;
	 reg = GETSTAT(W1);
	 data = GETDATA(W1);
#if defined(DEBUGCI403) || defined(DEBUGCI403CTL)
	 fprintf (stderr,
		  "  W1 = >%04X, op = %s, reg = %d, chan = %d, data = >%02X",
		  W1, W1 & RWBIT ? " READ" : "WRITE", reg, chan, data);
#endif
	 if (W1 & RWBIT) /* Read */
	 {
	    W1 = SETSTAT (reg) | SETCHAN(chan) | SETDATA (udevinfo->regs[reg]);
	    PUTW1 (dev, W1);
	    if (reg == 6)
	    {
	       udevinfo->regs[reg] &= 0xF0;
	    }
#if defined(DEBUGCI403) || defined(DEBUGCI403CTL)
	    fprintf (stderr, ", new W1 = >%04X\n", W1);
#endif
	 }
	 else /* Write */
	 {
	    int sendit;

	    udevinfo->regs[reg] = data;
	    sendit = FALSE;
	    if ((reg == 4) && (data == 0x01)) /* Fake a status change */
	    {
	       sendit = TRUE;
	       udevinfo->regs[6] &= 0xF0;
	       if (udev->switches & SWSTATUS)
	       {
		  VDTInfo *vdtinfo = &udev->info.terminfo.terminal.vdtinfo;
		  vdtinfo->CommDTR = TRUE;
		  vdtinfo->CommDSR = TRUE;
		  vdtinfo->CommDCD = TRUE;
		  vdtinfo->CommState = VDT_COMMSTATE_RDY;
	       }
	       W3 = CHGBIT | SETSTAT (6) | SETCHAN (chan) | SETDATA (0xB9);
	    }

	    if ((reg == 7) && (data & 0x80)) /* Xmit FIFO empty, tell host OK */
	    {
	       sendit = TRUE;
	       W3 = CHGBIT | SETSTAT (1) | SETCHAN (chan) | SETDATA (0);
	    }

	    if (sendit)
	    {
#if defined(DEBUGCI403) || defined(DEBUGCI403CTL)
	       fprintf (stderr, ", W3 = >%04X\n", W3);
#endif
	       ci403putring (dev, W3);
	       devinfo->timerint = 0;
	    }
#if defined(DEBUGCI403) || defined(DEBUGCI403CTL)
	    else
	       fputc ('\n', stderr);
#endif
	 }
      }
      break;

   case 4: /* W2 */
   case 5:
      W2 = GETW2 (dev);
      chan = GETCHAN(W2);
      data = GETDATA(W2);
      if ((udev = ci403findunit (dev, chan)) != NULL)
      {
	 udevinfo = &udev->info.terminfo.controller.ci403info;
#ifdef DEBUGCI403
	 fprintf (stderr, "  W2 = >%04X, chan = %d, data = >%02X(%c)\n",
		  W2, chan, data, isprint(data) ? data : '.' );
#endif
	 if (udev->type == PRINTCI403)
	 {
	    printputchar (udev, data);
	 }
	 else
	 {
	    serialoutput (udev, data);
	 }
	 W2 = 0xFF00;
	 PUTW2 (dev, W2);
      }
      break;

   default: ;
   }
}

/***********************************************************************
* ci403start - Start CI403.
***********************************************************************/

int
ci403start (Device *dev)
{
   CI403Info *devinfo;
   THREAD_HANDLE (thread_handle);

   devinfo = &dev->info.terminfo.controller.ci403info;
#ifdef DEBUGCI403
   fprintf (stderr, "ci403start: ENTERED\n");
   fprintf (stderr, "   fd = %d, devaddr = >%04X, unit = %d, file = %s\n",
	    dev->infd == DEV_TELNET ? 3 :
	    dev->infd == DEV_LOOP ? 4 : fileno(dev->infd),
	    dev->devaddr, devinfo->unit, dev->file);
#endif

   /* Only unit zero needs a task */

   if (devinfo->unit != 0)
      return (0);

   if (dev->infd == DEV_NULL)
   {
      PUTW0 (dev, 0);
      PUTW1 (dev, 0);
      PUTW2 (dev, 0);
      PUTW3 (dev, 0);
      return (0);
   }
   PUTW0 (dev, C403BIT);
   PUTW1 (dev, 0xFF00);
   PUTW2 (dev, 0xFF00);
   PUTW3 (dev, 0xFF00);

   dev->intenabled = FALSE;
   devinfo->oldW0 = 0;
   devinfo->W3head = devinfo->W3tail = 0;

   if (MUTEX_INIT (devinfo->W3ringmutex))
   {
      sprintf (view[0], "Device %s: mutex init failed: %s",
	       dev->name, strerror (ERRNO));
      return (-1);
   }

   if (THREAD_CREATE (thread_handle, ci403task, dev))
   {
      sprintf (view[0], "Device %s: thread create failed: %s",
	       dev->name, strerror (ERRNO));
#ifdef DEBUGCI403
      fprintf (stderr, "%s\n", view[0]);
#endif
      return (-1);
   }
   return (0);
}
