/***********************************************************************
*
* sim911.c - 911 VDT processing for the TI 990 Simulator.
*
* Changes:
*   11/06/03   DGP   Original.
*   11/17/03   DGP   Added escape sequence processing.
*   04/13/05   DGP   Added Backspace, PgUp and PgDn processing.
*                    Added Upcase keys switch.
*   04/14/05   DGP   Eat bare ESCAPE.
*   04/28/05   DGP   Changed to use switches.
*   09/21/05   DGP   Added SWLOGOUT switch.
*   05/21/07   DGP   Activate screen on task start instead of connect.
*   07/18/07   DGP   Added Hi/Lo intensity support.
*   10/26/07   DGP   Improve screen display.
*   11/13/08   DGP   Improve CRU bit addressing.
*   02/25/14   DGP   Position cursor for each display line.
*   06/02/15   DGP   Removed USS (z/OS) support.
*   06/10/15   DGP   Refactor to cleanup.
*   02/18/16   DGP   Added SO_LINGER to clear socket on disconnect.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>

#if defined (WIN32)
#define __TTYROUTINES 0
#include <conio.h>
#include <windows.h>
#include <signal.h>
#include <process.h>
#include <winsock.h>
#endif

#if defined(UNIX)
#include <unistd.h>
#include <sys/time.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <arpa/telnet.h>
#include <pthread.h>
#endif

#ifndef IAC
#define IAC 255
#define DONT 254
#define WILL 251
#define TELOPT_ECHO 1
#define TELOPT_SGA 3
#endif

#include "simdef.h"

#define DEPTH 1

/*
** 911 Function button definitions
*/

#ifdef DELETE
#undef DELETE
#endif

#define EFIELD 0x80
#define EINPUT 0x81
#define HOME   0x82
#define HTAB   0x83
#define DELETE 0x84
#define SKIP   0x85
#define INSERT 0x86
#define LEFTF  0x87
#define LEFT   0x88
#define UP     0x89
#define RIGHT  0x8A
#define DOWN   0x8B
#define RIGHTF 0x8C
#define XF1    0x8D
#define XF2    0x8E
#define XF3    0x8F
#define F1     0x92
#define F2     0x93
#define F3     0x94
#define F4     0x95
#define F5     0x96
#define F6     0x97
#define F7     0x98
#define F8     0x99
#define PRINT  0x9A
#define CMD    0x9B
#define GOLD   0x9C

extern uint16 statreg;	/* The program status register */
extern char view[MAXVIEW][MAXVIEWLEN+1];

static struct sockaddr_in sinme;

static uint16 graphics911[32] = 
{
/* 00   01   02   03   04   05   06   07  */
   ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ',
/* 08   09   0A   0B   0C   0D   0E   0F  */
   ' ', '|', '+', '+', '+', '+', ' ', ' ',
/* 10   11   12   13   14   15   16   17  */
   ' ', ' ', ' ', ' ', ' ', ' ', '-', '+',
/* 18   19   1A   1B   1C   1D   1E   1F  */
   '+', '_', '+', '+', '+', '+', ' ', ' '
};

static uint8 will[] = {IAC,WILL,'%','c',0};
static uint8 sethi[] = {ESCAPE,'[','1','m'};
static uint8 setlo[] = {ESCAPE,'[','0','m'};
static uint8 clear[] = {ESCAPE,'[','2','J',ESCAPE,'[','1',';','1','H'};

/***********************************************************************
* v911getchar - VDT 911 get a character.
***********************************************************************/

static int
v911getchar (int fd, uint8 *buf)
{
   return (Nread (fd, buf, 1));
}

/***********************************************************************
* v911eatchar - VDT 911 eat characters to tilde.
***********************************************************************/

static void
v911eatchar (int fd, uint8 *buf)
{
   *buf = 0;
   while (*buf != '~')
   {
      if (Nread (fd, buf, 1) != 1) return;
#ifdef DEBUG911ESCAPE
      fprintf (stderr, "   cx = %02X(%c)\n", buf[0], buf[0]);
#endif
   }
}

/***********************************************************************
* v911doiac - VDT 911 IAC processor. We just eat them.
***********************************************************************/

static void
v911doiac (int fd)
{
   uint8 buf[4];

   if (v911getchar (fd, &buf[1]) == 1)
   {
      if (v911getchar (fd, &buf[2]) == 1)
      {
#ifdef DEBUG911TASK
	 fprintf (stderr, "v911doiac: c1 = %02X, c2 = %02X\n", buf[1], buf[2]);
#endif
	 if (buf[1] == WILL)
	 {
	    buf[0] = IAC;
	    buf[1] = DONT;
	    Nwrite (fd, buf, 3);
	 }
      }
   }
}

/***********************************************************************
* v911escape - VDT 911 escape processor.
***********************************************************************/

static uint8
v911escape (int fd)
{
   uint8 retchar = 0;
   uint8 buf[2];

#ifdef DEBUG911ESCAPE
   fprintf (stderr, "v911escape: entered\n");
#endif
   v911getchar (fd, buf);

#ifdef DEBUG911ESCAPE
   fprintf (stderr, "   c1 = %02X(%c)\n", buf[0], buf[0]);
#endif
   switch (buf[0])
   {
   case 'O':
      v911getchar (fd, buf);
#ifdef DEBUG911ESCAPE
      fprintf (stderr, "   c2 = %02X(%c)\n", buf[0], buf[0]);
#endif
      switch (buf[0])
      {
      case 'P': /* F1 0x92 */
	 retchar = F1;
	 break;
      case 'Q': /* F2 0x93 */
	 retchar = F2;
	 break;
      case 'R': /* F3 0x94 */
	 retchar = F3;
	 break;
      case 'S': /* F4 0x95 */
	 retchar = F4;
	 break;
      default: ;
      }
      break;

   case '[':
      v911getchar (fd, buf);
#ifdef DEBUG911ESCAPE
      fprintf (stderr, "   c2 = %02X(%c)\n", buf[0], buf[0]);
#endif
      switch (buf[0])
      {
      case 'A': /* Up arrow 0x89 */
	 retchar = UP;
	 break;
      case 'B': /* Down arrow 0x8B */
	 retchar = DOWN;
	 break;
      case 'C': /* Right arrow 0x8A */
	 retchar = RIGHT;
	 break;
      case 'D': /* Left arrow 0x88 */
	 retchar = LEFT;
	 break;

      case '1':
	 v911getchar (fd, buf);
#ifdef DEBUG911ESCAPE
	 fprintf (stderr, "   c3 = %02X(%c)\n", buf[0], buf[0]);
#endif
	 switch (buf[0])
	 {
	 case '5': /* F5 0x96 */
	    v911eatchar (fd, buf);
	    retchar = F5;
	    break;
	 case '7': /* F6 0x97 */
	    v911eatchar (fd, buf);
	    retchar = F6;
	    break;
	 case '8': /* F7 0x98 */
	    v911eatchar (fd, buf);
	    retchar = F7;
	    break;
	 case '9': /* F8 0x99 */
	    v911eatchar (fd, buf);
	    retchar = F8;
	    break;
	 case '~': /* HOME 0x82 */
	    retchar = HOME;
	    break;
	 default: 
	    v911eatchar (fd, buf);
	 }
	 break;

      case '2':
	 v911getchar (fd, buf);
#ifdef DEBUG911ESCAPE
	 fprintf (stderr, "   c3 = %02X(%c)\n", buf[0], buf[0]);
#endif
	 switch (buf[0])
	 {
	 case '0': /* F9  - CMD 0x9B */
	    v911eatchar (fd, buf);
	    retchar = CMD;
	    break;
	 case '1': /* F10 - GOLD 0x9C */
	    v911eatchar (fd, buf);
	    retchar = GOLD;
	    break;
	 case '3': /* F11 - ERASE FIELD 0x80 */
	    v911eatchar (fd, buf);
	    retchar = EFIELD;
	    break;
	 case '4': /* F12 - ERASE INPUT 0x81 */
	    v911eatchar (fd, buf);
	    retchar = EINPUT;
	    break;
	 case '~': /* INSERT 0x86 */
	    retchar = INSERT;
	    break;
	 default:
	    v911eatchar (fd, buf);
	 }
	 break;

      case '3':
	 v911getchar (fd, buf);
#ifdef DEBUG911ESCAPE
	 fprintf (stderr, "   c3 = %02X(%c)\n", buf[0], buf[0]);
#endif
	 switch (buf[0])
	 {
	 case '~': /* DELETE 0x84 */
	    retchar = DELETE;
	    break;
	 default: 
	    v911eatchar (fd, buf);
	 }
	 break;

      case '5': /* PgUp - F2 */
	 v911eatchar (fd, buf);
	 retchar = F2;
	 break;

      case '6': /* PgDn - F1 */
	 v911eatchar (fd, buf);
	 retchar = F1;
	 break;

      default: 
	 v911eatchar (fd, buf);
      }
      break;
   default: 
      return (0);
   }

#ifdef DEBUG911ESCAPE
   fprintf (stderr, "   retchar = %02X\n", retchar);
#endif
   return (retchar);
}

/***********************************************************************
* v911task - VDT 911 connection task
***********************************************************************/

static void
v911task (void *vdev)
{
   Device *dev;
   VDT911Info *devinfo;
   uint8 *screen;
   int sfd;
   int fd;
   int maxfd;
   int i;
   int fromlen;
   int row, col;
   struct timeval tv;
   struct sockaddr_in peer;
   struct sockaddr_in frominet;
   struct linger linger;
   fd_set readfds;
   uint8 msg[82];
   uint8 buf[2];

   dev = (Device *)vdev;
   devinfo = &dev->info.vdt911info;

   sfd = dev->sfd;

   if ((screen = (uint8 *) malloc (V911SCREENSIZE * 5)) == NULL)
   {
      fprintf (stderr, "v911task: can't allocate screen buffer\n");
      return;
   }

SERVER_LOOP:

   devinfo->cursorpos = 1840;
   devinfo->active = TRUE;

#ifdef DEBUG911TASK
   fprintf (stderr, "v911task: starting: dev = %s, sfd = %d\n", dev->name, sfd);
#endif

   sprintf (dev->file, "%d:LISTEN", dev->unit);

   /*
   ** Accept user connection
   */

   fromlen = sizeof(struct sockaddr_in);
   if ((fd = accept (sfd, (struct sockaddr *)&frominet, (void *)&fromlen)) < 0)
   {
      if (ERRNO == EAGAIN || ERRNO == EINTR)
      {
	 goto SERVER_LOOP;
      }
#if defined(WIN32)
      if (fd == INVALID_SOCKET) return;
#endif
      PERROR ("v911task: accept");
      return;
   }
#ifdef DEBUG911TASK
   fprintf (stderr, "v911task: accepting: fd = %d\n", fd);
#endif

   linger.l_onoff = 1;
   linger.l_linger = 0;
   if (setsockopt (fd, SOL_SOCKET, SO_LINGER, &linger, sizeof(linger)) < 0)
   {
      PERROR ("v911task: setsockopt");
      CLOSE (fd);
      goto SERVER_LOOP;
   }

   i = sizeof(peer);

   if (getpeername (fd, (struct sockaddr *)&peer, (void *)&i) < 0)
   {
      PERROR ("v911task: getpeername");
      CLOSE (fd);
      goto SERVER_LOOP;
   }
   sprintf (dev->file, "%d:%s", dev->unit, inet_ntoa (peer.sin_addr));

#ifdef DEBUG911TASK
   fprintf (stderr, "v911task: from %s\n", dev->file);
#endif
   
   /*
   ** Tell the telnet client we're doing the work
   */

   sprintf ((char *)msg, (char *)will, TELOPT_SGA);
   Nwrite (fd, msg, 3);
   sprintf ((char *)msg, (char *)will, TELOPT_ECHO);
   Nwrite (fd, msg, 3);

   /*
   ** Display the introduction banner
   */

   Nwrite (fd, clear, sizeof (clear));
   sprintf ((char *)msg, "sim990 %s VDT911 %s(V911) on %s",
	    VERSION, dev->name, dev->file);
   Nwrite (fd, msg, strlen((char *)msg));

   /*
   ** Process connection
   */

   while (TRUE)
   {
      /*
      ** Display screen, if updated
      */

      if (dev->cbreak)
      {
	 uint8 *sp;
         int ishi = FALSE;

	 dev->cbreak = FALSE;
#ifdef DEBUG911TASKaa
         fprintf (stderr, "Display screen\n");
#endif
	 sp = screen;
	 memcpy (sp, clear, sizeof (clear));
	 sp += sizeof (clear);
	 memcpy (sp, setlo, sizeof (setlo));
	 sp += sizeof (setlo);
	 for (i = 0; i < 24; i++)
	 {
	    int jj;
	    int row = i*80;

	    sprintf ((char *)msg, "%c[%d;%dH", ESCAPE, i + 1, 1);
	    memcpy (sp, msg, strlen((char *)msg));
	    sp += strlen((char *)msg);

	    for (jj = 0; jj < 80; jj++)
	    {
	       if (!devinfo->intensity[row+jj])
               {
                  if (!ishi)
                  {
                      memcpy (sp, sethi, sizeof (sethi));
                      sp += sizeof (sethi);
                      ishi = TRUE;
                  }
               }
	       else
               {
                  if (ishi)
                  {
                      memcpy (sp, setlo, sizeof (setlo));
                      sp += sizeof (setlo);
                      ishi = FALSE;
                  }
               }
	       *sp++ = devinfo->screen[row+jj];
	    }
	 }
	 if (Nwrite (fd, screen, sp-screen) < 0) break;
#ifdef DEBUG911DUMP
         fprintf (stderr, "Screen:\n");
	 HEXDUMP (stderr, screen, sp-screen, 0);
#endif
      }

      /*
      ** Position cursor on screen
      */

      row = (devinfo->cursorpos / 80) + 1;
      col = (devinfo->cursorpos % 80) + 1;
      sprintf ((char *)msg, "%c[0m%c[%d;%dH", ESCAPE, ESCAPE, row, col);
      if (Nwrite (fd, msg, strlen((char *)msg)) < 0) break;

      /*
      ** Await user input or timeout
      */

      FD_ZERO (&readfds);
      FD_SET (fd, &readfds);

      tv.tv_sec = 0;
      tv.tv_usec = 10000;
      if ((maxfd = select (fd+1, &readfds, NULL, NULL, &tv)) < 0)
      {
	 if (ERRNO != EINTR && ERRNO != EAGAIN)
	 {
	    PERROR ("v911task: select failed");
	    break;
	 }
      }
#ifdef DEBUG911TASKaa
      fprintf (stderr, "Select: fd = %d, maxfd = %d\n", fd, maxfd);
#endif
      /*
      ** If input data, process it
      */

      if (maxfd > 0)
      {
         if (v911getchar (fd, buf) != 1) break;
	 if (buf[0] == IAC)
	 {
	    v911doiac (fd);
	 }
	 else if (buf[0] == ESCAPE)
	 {
	    uint8 c;

	    if ((c = v911escape (fd)) != 0)
	    {
	       dev->inchar = c;
	       devinfo->kbchar = TRUE;
	       if (dev->intenabled)
		  devinfo->kbint = TRUE;
#ifdef DEBUG911TASK
	       fprintf (stderr, "Complete select: eschar = %02X\n", c);
#endif
	    }
	 }
	 else if (buf[0] > 0 && buf[0] <= 0x7F)
	 {
	    devinfo->kbchar = TRUE;
	    if ((dev->switches & SWUPCASE) && islower(buf[0]))
	       buf[0] = toupper(buf[0]);
	    if (buf[0] == BACKSPACE || buf[0] == DBACKSPACE)
	       buf[0] = LEFT;
	    dev->inchar = buf[0];
#ifdef DEBUG911TASK
	    fprintf (stderr, "Complete select: inchar = %02X(%c)\n",
		     buf[0] & 0xFF, isprint(buf[0]) ? buf[0] : '.');
#endif
	    if (dev->intenabled)
	       devinfo->kbint = TRUE;
	 }
      }
   }

   /*
   ** If SWLOGOUT specified, Send Logout sequence
   */

   if (dev->switches & SWLOGOUT)
   {
      devinfo->kbchar = TRUE;
      dev->inchar = 'Q';
      if (dev->intenabled)
	 devinfo->kbint = TRUE;

      FD_ZERO (&readfds);
      FD_SET (fd, &readfds);
      tv.tv_sec = 0;
      tv.tv_usec = 100;
      maxfd = select (fd+1, &readfds, NULL, NULL, &tv);
      devinfo->kbchar = TRUE;
      dev->inchar = '\r';
      if (dev->intenabled)
	 devinfo->kbint = TRUE;
      memset (devinfo->screen, ' ', V911SCREENSIZE);
   }

   CLOSE (fd);
   devinfo->active = FALSE;
   dev->cbreak = TRUE;
   goto SERVER_LOOP;

}

/***********************************************************************
* v911start - Start connection task
***********************************************************************/

int
v911start (Device *dev)
{
   VDT911Info *devinfo;
   THREAD_HANDLE (thread_handle);

   devinfo = &dev->info.vdt911info;
   memset (devinfo->screen, ' ', V911SCREENSIZE);
   devinfo->kbchar = FALSE;

   if (THREAD_CREATE (thread_handle, v911task, dev))
   {
      sprintf (view[0], "Device %s: thread create failed: %s",
	       dev->name, strerror (ERRNO));
#ifdef DEBUG911TASK
      fprintf (stderr, "%s\n", view[0]);
#endif
      return (-1);
   }
   return (0);
}

/***********************************************************************
* v911open - Open a connection for the 911
***********************************************************************/

FILE *
v911open (char *fdev, char *bp, char *mode, int *pfd)
{
   int sfd;
   short port;
#if defined(WIN32)
   WSADATA nt_data;
#endif /* WIN32 */

#ifdef DEBUG911TASK
   fprintf (stderr, "v911open: bp = %s\n", bp);
#endif

   memset ((char *)&sinme, '\0', sizeof(sinme));

#if defined(WIN32)
   if (WSAStartup (MAKEWORD (1, 1), &nt_data) != 0)
   {
      sprintf (view[0], "Device %s port %s: WSAStartup failed: %s",
	       fdev, bp, strerror (ERRNO));
#ifdef DEBUG911TASK
      fprintf (stderr, "%s\n", view[0]);
#endif
      return (NULL);
   }
#endif /* WIN32 */

   port = atoi(bp);
   sinme.sin_port = htons (port);
   sinme.sin_family = AF_INET;

   if ((sfd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
   {
      sprintf (view[0], "Device %s port %d: socket failed: %s",
	       fdev, port, strerror (ERRNO));
#ifdef DEBUG911TASK
      fprintf (stderr, "%s\n", view[0]);
#endif
      return (NULL);
   }

   if (bind (sfd, (struct sockaddr *)&sinme, sizeof(sinme)) < 0)
   {
      sprintf (view[0], "Device %s port %d: bind failed: %s",
	       fdev, port, strerror (ERRNO));
#ifdef DEBUG911TASK
      fprintf (stderr, "%s\n", view[0]);
#endif
      return (NULL);
   }

   if (listen (sfd, DEPTH) < 0)
   {
      sprintf (view[0], "Device %s port %d: listen failed: %s",
	       fdev, port, strerror (ERRNO));
#ifdef DEBUG911TASK
      fprintf (stderr, "%s\n", view[0]);
#endif
      return (NULL);
   }

   *pfd = sfd;
   return (DEV_VDT);
}

/***********************************************************************
* v911seto - Set CRU bit to one.
***********************************************************************/

void
v911seto (Device *dev, uint8 disp)
{
   VDT911Info *devinfo;

   devinfo = &dev->info.vdt911info;

#if defined(DEBUG911CRU) || defined(CHECKCRU)
   fprintf (stderr, "CRU %04X SBO %d\n", dev->devaddr, disp);
#endif

   if (devinfo->wrdsel == 0)
   {
      switch (disp)
      {
      case 7: /* Hi/Lo intensity select */
#ifdef DEBUG911INTEN
         fprintf (stderr, "HiLo SBO: row = %02d, col = %02d\n",
		  (devinfo->cursorpos / 80) + 1,
		  (devinfo->cursorpos % 80) + 1);
#endif
         devinfo->intensity[devinfo->cursorpos] = 0;
         break;

      case 8: /* Write data strobe */
         devinfo->screen[devinfo->cursorpos] = dev->outchar;
         devinfo->intensity[devinfo->cursorpos] = devinfo->intenenable;
	 dev->cbreak = TRUE;
         break;

      case 9: /* Test mode */
         break;

      case 10: /* Cursor Move */
         devinfo->cursorpos--;
	 if (devinfo->cursorpos < 0)
	    devinfo->cursorpos = V911SCREENSIZE;
	 break;

      case 11: /* Blinking cursor enable */
         break;

      case 12: /* Keyboard interrupt enable */
	 dev->intenabled = TRUE;
	 if (devinfo->kbchar)
	    gendevinterrupt (dev);
	 break;

      case 13: /* Dual intensity enable */
#ifdef DEBUG911INTEN
         fprintf (stderr, "Dual SBO: row = %02d, col = %02d\n",
		  (devinfo->cursorpos / 80) + 1,
		  (devinfo->cursorpos % 80) + 1);
#endif
         devinfo->dualenable = 1;
	 break;

      case 14: /* Display Enable */
	 devinfo->dispenable = 1;
	 break;

      case 15: /* Word select */
	 devinfo->oldsel = devinfo->wrdsel;
	 devinfo->wrdsel = 1;
	 break;

      default: ;
      }
   }
   else
   {
      switch (disp)
      {
      case 12: /* Display cursor */
         break;

      case 13: /* Keyboard ack */
         break;

      case 14: /* Beep enable strobe */
         break;

      case 15: /* Word select */
	 devinfo->oldsel = devinfo->wrdsel;
	 devinfo->wrdsel = 1;
	 break;

      default: ;
      }
   }
}

/***********************************************************************
* v911setz - Set CRU bit to zero.
***********************************************************************/

void
v911setz (Device *dev, uint8 disp)
{
   VDT911Info *devinfo;

   devinfo = &dev->info.vdt911info;

#if defined(DEBUG911CRU) || defined(CHECKCRU)
   fprintf (stderr, "CRU %04X SBZ %d\n", dev->devaddr, disp);
#endif

   if (devinfo->wrdsel == 0)
   {
      switch (disp)
      {
      case 7: /* Hi/Lo intensity select */
#ifdef DEBUG911INTEN
         fprintf (stderr, "HiLo SBZ: row = %02d, col = %02d\n",
		  (devinfo->cursorpos / 80) + 1,
		  (devinfo->cursorpos % 80) + 1);
#endif
         devinfo->intensity[devinfo->cursorpos] = 1;
         break;

      case 8: /* Write data strobe */
         devinfo->screen[devinfo->cursorpos] = dev->outchar;
         devinfo->intensity[devinfo->cursorpos] = devinfo->intenenable;
	 dev->cbreak = TRUE;
	 break;

      case 9: /* Test mode */
         break;

      case 10: /* Cursor Move */
         devinfo->cursorpos++;
	 if (devinfo->cursorpos > V911SCREENSIZE)
	    devinfo->cursorpos = 0;
	 break;

      case 11: /* Blinking cursor enable */
         break;

      case 12: /* Keyboard interrupt enable */
	 dev->intenabled = FALSE;
	 break;

      case 13: /* Dual intensity enable */
#ifdef DEBUG911INTEN
         fprintf (stderr, "Dual SBZ: row = %02d, col = %02d\n",
		  (devinfo->cursorpos / 80) + 1,
		  (devinfo->cursorpos % 80) + 1);
#endif
         devinfo->dualenable = 0;
	 break;

      case 14: /* Display Enable */
	 devinfo->dispenable = 0;
	 break;

      case 15: /* Word select */
	 devinfo->oldsel = devinfo->wrdsel;
	 devinfo->wrdsel = 0;
	 break;

      default: ;
      }
   }
   else
   {
      switch (disp)
      {
      case 12: /* Display cursor */
         break;

      case 13: /* Keyboard ack */
         break;

      case 14: /* Beep enable strobe */
         break;

      case 15: /* Word select  */
	 devinfo->oldsel = devinfo->wrdsel;
	 devinfo->wrdsel = 0;
	 break;

      default: ;
      }
   }
}

/***********************************************************************
* v911tb - Test CRU bit.
***********************************************************************/

void
v911tb (Device *dev, uint8 disp)
{
   VDT911Info *devinfo;

   devinfo = &dev->info.vdt911info;

#if defined(DEBUG911CRU) || defined(CHECKCRU)
   fprintf (stderr, "CRU %04X TB %d", dev->devaddr, disp);
#endif

   CLR_EQ;

   if (devinfo->wrdsel == 0)
   {
      switch (disp)
      {
      case 7: /* Low intensity */
#ifdef DEBUG911INTEN
         fprintf (stderr, "HiLo TB : row = %02d, col = %02d\n",
		  (devinfo->cursorpos / 80) + 1,
		  (devinfo->cursorpos % 80) + 1);
#endif
         if (devinfo->intensity[devinfo->cursorpos])
	    SET_EQ;
         break;

      case 15: /* Keyboard ready */
	 if (devinfo->active && devinfo->kbchar)
	 {
	    devinfo->kbchar = FALSE;
	    SET_EQ;
#if defined(DEBUG911CRU) || defined(CHECKCRU)
	    fprintf (stderr, " TRUE");
#endif
	    if (dev->intenabled)
	       gendevinterrupt (dev);
	 }
	 break;

      default: ;
      }
   }
   else
   {
      switch (disp)
      {
      case 10: /* Cursor MSB */
         if (devinfo->cursorpos & 0x400) SET_EQ;
	 break;

      case 11: /* Keyboard MSB */
         if (dev->inchar & 0x80)
	 {
	    SET_EQ;
#if defined(DEBUG911CRU) || defined(CHECKCRU)
	    fprintf (stderr, " TRUE");
#endif
	 }
	 break;

      case 12: /* Terminal ready */
	 if (!devinfo->active)
         {
	    SET_EQ;
#if defined(DEBUG911CRU) || defined(CHECKCRU)
	    fprintf (stderr, " TRUE");
#endif
         }
	 break;

      case 13: /* Previous state */
	 if (devinfo->oldsel == 1)
	 {
	    SET_EQ;
#if defined(DEBUG911CRU) || defined(CHECKCRU)
	    fprintf (stderr, " TRUE");
#endif
	 }
         break;

      case 14: /* Keyboard parity error */
	 break;

      case 15: /* Keyboard ready */
	 if (devinfo->active && devinfo->kbchar)
	 {
	    devinfo->kbchar = FALSE;
	    SET_EQ;
#if defined(DEBUG911CRU) || defined(CHECKCRU)
	    fprintf (stderr, " TRUE");
#endif
	 }
	 break;

      default: ;
      }
   }
#if defined(DEBUG911CRU) || defined(CHECKCRU)
   fprintf (stderr, "\n");
#endif
}

/***********************************************************************
* v911ldcr - Load CRU.
***********************************************************************/

void
v911ldcr (Device *dev, uint16 cnt, uint16 dat)
{
   VDT911Info *devinfo;

   devinfo = &dev->info.vdt911info;

#ifdef DEBUG911CRU
   fprintf (stderr, "CRU %04X LDCR CNT %d DAT %02X", R12, cnt, dat);
#endif

   if (devinfo->wrdsel == 0)
   {
      uint16 intensity;

      intensity = dat & 0x80;
      dat &= 0x7F;
      if (dat < BLANK)
      {
#ifdef DEBUG911GRAPHICS
	 fprintf (stderr, "MAP %02X -> %02X(%c)\n",
		  dat, graphics911[dat], graphics911[dat]);
#endif
	 dat = graphics911[dat];
      }
      else if (dat > TILDE) dat = BLANK;
#ifdef DEBUG911CRU
      if (isprint (dat))
	 fprintf (stderr, "(%c)", dat);
#endif
      dev->outchar = dat;
      if (devinfo->dualenable)
	 devinfo->intenenable = intensity;
      else
	 devinfo->intenenable = 0;
   }
   else
   {
      if (dat >= V911SCREENSIZE) dat = V911SCREENSIZE - 1;
      devinfo->cursorpos = dat;
   }
#ifdef DEBUG911CRU
   fprintf (stderr, "\n");
#endif
}

/***********************************************************************
* v911stcr - Store CRU.
***********************************************************************/

uint16
v911stcr (Device *dev, uint16 cnt)
{
   VDT911Info *devinfo;
   uint16 dat = 0;

   devinfo = &dev->info.vdt911info;

   if (devinfo->wrdsel == 0)
   {
      int mask;

      if (cnt == 7) mask = 0x7F;
      else mask = 0xFF;

      if (R12 == dev->devaddr)
      {
	 if (cnt > 8)
	    dat = (dev->inchar << 8) | devinfo->screen[devinfo->cursorpos];
	 else
	    dat = devinfo->screen[devinfo->cursorpos];
      }
      else
	 dat = dev->inchar & mask;

#ifdef DEBUG911CRU
      fprintf (stderr, "CRU %04X STCR CNT %d DAT %02X", R12, cnt, dat);
      if (isprint (dat))
	 fprintf (stderr, "(%c)", dat);
      fprintf (stderr, "\n");
#endif
   }
   else
   {
      dat = devinfo->cursorpos;
#ifdef DEBUG911CRU
      fprintf (stderr, "CRU %04X STCR CNT %d DAT %02X\n", R12, cnt, dat);
#endif
   }

   return (dat);
}
