/***********************************************************************
*
* sim931.c - 931/940 VDT support for the TI 990 Simulator.
*
* Changes:
*   03/28/14   HSW   Original. To support DNOS.
*   04/04/14   DGP   Added general telnet support for serial devices.
*   05/08/14   DGP   Added DEBUG functions and functionalized ring input.
*   05/28/15   DGP   Enabled to run on "console" TTY port.
*
***********************************************************************/

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>
#include <time.h>

#include "simdef.h"
#include "screen.h"

extern int run;

static char *v931_CommMessage[] = 
{
   "Conn", "Disc", "Rdy", "Busy", "Rng", "Lpbk"
};

#if defined(DEBUG931) || defined (DEBUG931DS) || defined(DEBUG931ATTR)
extern char *emulations[];

/***********************************************************************
* v931debugprint - VDT 931 DEBUG printing.
***********************************************************************/

static void
v931debugprint (Device *dev, const char *format, ...)
{
   va_list ap;

   fprintf (stderr, "DEBUG-%s: devaddr = >%04X: ", 
	    emulations[dev->emutype], dev->devaddr);
   va_start (ap, format);
   vfprintf (stderr, format, ap);
   va_end (ap);
}
#endif

/***********************************************************************
* v931DumpSequence - Dump unknown escape sequences.
***********************************************************************/

#ifdef DEBUG931DS
static void
v931DumpSequence (Device *dev, char *Reason, int FirstIndex, int LastIndex)
{
   TermRing *prcring;
   int ThisIndex;
   uint8 Char;

   prcring = &dev->info.terminfo.prcring;
   v931debugprint (dev, "%s:\n", Reason);

   for (ThisIndex = FirstIndex; ThisIndex != LastIndex; )
   {
      Char = prcring->data[ThisIndex++];
      fprintf (stderr, "    %02X", Char);
      if (Char & 0x80)
      {
         Char &= 0x7F;
         fprintf (stderr, " -->");
      }
      if (Char > 0x1F && Char < 0x7F)
      {
         fprintf (stderr, "( %c )\n", Char);
      }
      fprintf (stderr, "\n");
      if (ThisIndex == TERMRINGLEN)
         ThisIndex = 0;
   }

   fprintf (stderr, "\n");
}
#endif

/***********************************************************************
* v931init - VDT 931 Initialization.
***********************************************************************/

void
v931init (Device *dev)
{
   VDTInfo *vdtinfo;
   int r;
   int c;

   vdtinfo = &dev->info.terminfo.terminal.vdtinfo;

   if ((vdtinfo->screen =
          (uint8 *) malloc (VDT_NUM_ROWS * VDT_NUM_COLS * 5)) == NULL)
   {
      fprintf (stderr, "v931init: can't allocate screen buffer\n");
      return;
   }

   vdtinfo->Fill = ' ';
   vdtinfo->Attr = VDT_ATTR_DEFAULT;

   memset (vdtinfo->ABM_Buffer, vdtinfo->Fill, 32);
   vdtinfo->ABM_Count = 0;

   for (r = 0; r < VDT_NUM_ROWS; r++)
   {
      for (c = 0; c < VDT_NUM_COLS; c++)
      {
	 vdtinfo->Buffer[r][c] = vdtinfo->Fill;
	 vdtinfo->Shadow[r][c] = vdtinfo->Attr;
      }
   }
   vdtinfo->Row = VDT_MIN_ROW;
   vdtinfo->Col = VDT_MIN_COL;

   for (r = 0; r < 2; r++)
   {
      for (c = 0; c < VDT_NUM_COLS; c++)
      {
	 vdtinfo->StatusLine[r][c] = ' ';
      }
   }
   vdtinfo->StatusActive = VDT_STATUS_PRIMARY;
   vdtinfo->VideoOn = 0x02;
   vdtinfo->CursorOn = 0x01;
   vdtinfo->AlternateCharacterSet = 0;
   vdtinfo->LockOn = 0x00;
   vdtinfo->GraphicCharSet = FALSE;
   vdtinfo->CommState = VDT_COMMSTATE_DISC;  /* "Comm:Disc " */
}

/***********************************************************************
* v931input - VDT 931 input processor.
***********************************************************************/

void
v931input (Device *dev, int fd)
{
   TermRing *inring;
   uint8 buf[2];

   inring = &dev->info.terminfo.inring;

#ifdef DEBUG931
   fprintf (stderr, "v931input-%s: entered: devaddr = >%04X\n",
	    emulations[dev->emutype], dev->devaddr);
#endif

   termdgetchar (dev, fd, buf);

#ifdef DEBUG931
   fprintf (stderr, "   c1 = %02X(%c)\n", buf[0], buf[0]);
#endif

   switch (buf[0])
   {
   case 'O':
      termdgetchar (dev, fd, buf);
#ifdef DEBUG931
      fprintf (stderr, "   c2 = %02X(%c)\n", buf[0], buf[0]);
#endif
      switch (buf[0])
      {
      case 'P': /* F1 0x92 */
	 termdputring (dev, inring, ESCAPE);
	 termdputring (dev, inring, 'i');
	 termdputring (dev, inring, '1');
	 break;
      case 'Q': /* F2 0x93 */
	 termdputring (dev, inring, ESCAPE);
	 termdputring (dev, inring, 'i');
	 termdputring (dev, inring, '2');
	 break;
      case 'R': /* F3 0x94 */
	 termdputring (dev, inring, ESCAPE);
	 termdputring (dev, inring, 'i');
	 termdputring (dev, inring, '3');
	 break;
      case 'S': /* F4 0x95 */
	 termdputring (dev, inring, ESCAPE);
	 termdputring (dev, inring, 'i');
	 termdputring (dev, inring, '4');
	 break;
      default: ;
      }
      break;

   case '[':
      termdgetchar (dev, fd, buf);
#ifdef DEBUG931
      fprintf (stderr, "   c2 = %02X(%c)\n", buf[0], buf[0]);
#endif
      switch (buf[0])
      {
      case 'A': /* Up arrow 0x89 */
	 termdputring (dev, inring, ESCAPE);
	 termdputring (dev, inring, 'A');
	 break;
      case 'B': /* Down arrow 0x8B */
	 termdputring (dev, inring, ESCAPE);
	 termdputring (dev, inring, 'B');
	 break;
      case 'C': /* Right arrow 0x8A */
	 termdputring (dev, inring, ESCAPE);
	 termdputring (dev, inring, 'C');
	 break;
      case 'D': /* Left arrow 0x88 */
	 termdputring (dev, inring, ESCAPE);
	 termdputring (dev, inring, 'D');
	 break;
      case '1':
	 termdgetchar (dev, fd, buf);
#ifdef DEBUG931
	 fprintf (stderr, "   c3 = %02X(%c)\n", buf[0], buf[0]);
#endif
	 switch (buf[0])
	 {
	 case '5': /* F5 0x96 */
	    termdeatchar (dev, fd, buf);
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'i');
	    termdputring (dev, inring, '5');
	    break;
	 case '7': /* F6 0x97 */
	    termdeatchar (dev, fd, buf);
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'i');
	    termdputring (dev, inring, '6');
	    break;
	 case '8': /* F7 0x98 */
	    termdeatchar (dev, fd, buf);
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'i');
	    termdputring (dev, inring, '7');
	    break;
	 case '9': /* F8 0x99 */
	    termdeatchar (dev, fd, buf);
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'i');
	    termdputring (dev, inring, '8');
	    break;
	 case '~': /* HOME 0x82 */
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'H');
	    break;
	 default: 
	    termdeatchar (dev, fd, buf);
	 }
	 break;
      case '2':
	 termdgetchar (dev, fd, buf);
#ifdef DEBUG931
	 fprintf (stderr, "   c3 = %02X(%c)\n", buf[0], buf[0]);
#endif
	 switch (buf[0])
	 {
	 case '0': /* F9  - CMD 0x9B */
	    termdeatchar (dev, fd, buf);
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'h');
	    break;
	 case '1': /* F10 - GOLD 0x9C */
	    termdeatchar (dev, fd, buf);
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'g');
	    break;
	 case '3': /* F11 - ERASE FIELD 0x80 */
	    termdeatchar (dev, fd, buf);
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, '=');
	    break;
	 case '4': /* F12 - ERASE INPUT 0x81 */
	    termdeatchar (dev, fd, buf);
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'K');
	    break;
	 case '~': /* INSERT 0x86 */
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'P');
	    break;
	 default: 
	    termdeatchar (dev, fd, buf);
	 }
	 break;
      case '3':
	 termdgetchar (dev, fd, buf);
#ifdef DEBUG931
	 fprintf (stderr, "   c3 = %02X(%c)\n", buf[0], buf[0]);
#endif
	 switch (buf[0])
	 {
	 case '~': /* DELETE 0x84 */
	    termdeatchar (dev, fd, buf);
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'Q');
	    break;
	 default: 
	    termdeatchar (dev, fd, buf);
	 }
	 break;
      case '5': /* PgUp - F2 */
	 termdeatchar (dev, fd, buf);
	 termdputring (dev, inring, ESCAPE);
	 termdputring (dev, inring, 'i');
	 termdputring (dev, inring, '2');
	 break;

      case '6': /* PgDn - F1 */
	 termdeatchar (dev, fd, buf);
	 termdputring (dev, inring, ESCAPE);
	 termdputring (dev, inring, 'i');
	 termdputring (dev, inring, '1');
	 break;

      default: 
	 termdeatchar (dev, fd, buf);
      }
      break;

   default: ;
   }
}

/***********************************************************************
* v931getdata - v931 Get data field.
*
* Format:
*    ESC ( ... data ... ESC )
*
***********************************************************************/

static int
v931getdata (Device *dev, TermRing *ring, char *data, int first, int max)
{
   char *bp;
   int length;
   int ch;

   length = 0;
   bp = data;

   while (TRUE)
   {
      if ((ch = termdgetcond (dev, ring, first)) < 0) return (-1);
      if ((ch == ESCAPE) || (length >= max)) break;
      if (length < max)
         *bp++ = ch;
      length++;
   }
   *bp = 0;

   if ((ch = termdgetcond (dev, ring, first)) < 0) return (-1);
   if (ch != ')')
      return (0); /* error */

#ifdef DEBUG931
   fprintf (stderr, "v931getdata-%s: first = %d, max = %d, length = %d\n",
	    emulations[dev->emutype], first, max, length);
   HEXDUMP (stderr, data, length, 0);
#endif
   return (length);
}

/***********************************************************************
* v931process - v931 process output emulation.
***********************************************************************/

void
v931process (Device *dev)
{
   TermInfo *terminfo;
   TermRing *prcring;
   TermRing *inring;
   VDTInfo *vdtinfo;
   int r;
   int c;
   int DataLength;
   int Char;
   int Attr;
   int Row;
   int Col;
   int Nrows;
   int Ncols;
   int Count;
   int Variant;
   int FirstIndex;
   uint8 Data;
   char databuf[VDT_BUF_LEN];

   terminfo = &dev->info.terminfo;
   prcring = &terminfo->prcring;
   inring = &terminfo->inring;
   vdtinfo = &dev->info.terminfo.terminal.vdtinfo;

   while (!termdringempty (dev, prcring))
   {
      FirstIndex = prcring->tail;
      Char = termdgetring (dev, prcring);
      switch (Char)
      {
      case 0x00:
	 break;

      case 0x05: /* ENQ - Enquiry */
	 /*
	 ** RESPONSE:  Transmit contents of ABM as set by "Define ABM"
	 */
#ifdef DEBUG931
	 v931debugprint (dev, "Enquiry\n");
#endif
	 if (vdtinfo->ABM_Count > 0)
	 {
	    int i;

	    for (i = 0; i < vdtinfo->ABM_Count; i++)
	    {
	       termdputring (dev, inring, vdtinfo->ABM_Buffer[i]);
	    }
	 }
	 break;

      case 0x07: /* BEL - Bell */
#ifdef DEBUG931
	 v931debugprint (dev, "Bell\n");
#endif

	 break;
      case 0x08: /* BS - Back Space */
#ifdef DEBUG931
	 v931debugprint (dev, "Back Space\n");
#endif
	 vdtinfo->Col--;
	 VDT_AdjustCursor (dev);
	 break;

      case 0x0A: /* LF - Line Feed */
#ifdef DEBUG931
	 v931debugprint (dev, "Line Feed\n");
#endif
	 vdtinfo->Row++;
	 VDT_AdjustCursor (dev);
	 break;

      case 0x0D: /* CR - Carriage Return */
#ifdef DEBUG931
	 v931debugprint (dev, "Carriage Return\n");
#endif
	 vdtinfo->Col = VDT_MIN_COL;
	 VDT_AdjustCursor (dev);
	 break;

      case 0x0E: /* SO - Select Alternate Character Set */
#ifdef DEBUG931
	 v931debugprint (dev, "Select Alternate Character Set\n");
#endif
	 vdtinfo->GraphicCharSet = TRUE;
	 break;

      case 0x0F: /* SI - Select Primary Character Set */
#ifdef DEBUG931
	 v931debugprint (dev, "Select Primary Character Set\n");
#endif
	 vdtinfo->GraphicCharSet = FALSE;
	 vdtinfo->AlternateCharacterSet = 0x00;
	 break;

      case 0x11: /* DC1 - Device Control 1 - Resume transmission to Host */
#ifdef DEBUG931
	 v931debugprint (dev, "Device Control 1\n");
#endif
	 vdtinfo->CommState = VDT_COMMSTATE_RDY;  /* "Comm:Rdy " */
	 break;

      case 0x13: /* DC3 - Device Control 3 - Pause transmmission to Host */
#ifdef DEBUG931
	 v931debugprint (dev, "Device Control 3\n");
#endif
	 vdtinfo->CommState = VDT_COMMSTATE_BUSY;  /* "Comm:Busy" */
	 break;

      case ESCAPE:
	 if ((Char = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	 while (Char == ESCAPE)
	 {
	    /* spurious extra ESC */
	    if ((Char = termdgetcond (dev, prcring, FirstIndex)) < 0) return;
	 }

	 switch (Char)
	 {
	 case '!': /* ESC ! Char Attr - Select Fill Character and Attribute */
	    if ((Char = termdgetcond (dev, prcring, FirstIndex)) < 0) return;
	    if ((Attr = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

#if defined(DEBUG931) || defined(DEBUG931ATTR)
	    v931debugprint (dev,
			"Select Fill Character and Attribute ( >%02X,>%02X )\n",
			   Char, Attr);
#endif

	    vdtinfo->Fill = Char;
	    vdtinfo->Attr = Attr;
	    break;

	 case '"': /* ESC " Row Col - Select Char Editing Extent Boundary */
	    if ((Row = termdgetcond (dev, prcring, FirstIndex)) < 0) return;
	    if ((Col = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    Row -= 0x20;        /* 0..24 */
	    if (Row < VDT_MIN_ROW || Row > VDT_MAX_ROW)
	       Row = VDT_MIN_ROW;

	    Col -= 0x20;        /* 0..79 */
	    if (Col < VDT_MIN_COL || Col > VDT_MAX_COL)
	       Col = VDT_MIN_COL;
#ifdef DEBUG931
	    v931debugprint (dev, 
			"Select Character Editing Extent Boundary ( %d,%d )\n",
			   Row, Col);
#endif

	    vdtinfo->CharExtentRow = Row;    /* Insert/Delete Char */
	    vdtinfo->CharExtentCol = Col;    /* Insert/Delete Char */
	    break;

	 case '#': /* ESC # Row - Select Line Editing Extent Boundary */
	    if ((Row = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    Row -= 0x20;        /* 0..24 */
	    if (Row < VDT_MIN_ROW || Row > VDT_MAX_ROW)
	       Row = VDT_MIN_ROW;

#ifdef DEBUG931
	    v931debugprint (dev, "Select Line Editing Extent Boundary ( %d )\n",
			    Row);
#endif
	    vdtinfo->LineExtentRow = Row;    /* Insert/Delete Line */
	    break;

	 case '$': /*  ESC $ ... */

	    if ((Char = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    switch (Char)
	    {
	    case '0': /* ESC $ 0 - Disable Character Editing Extent Boundary */
#ifdef DEBUG931
	       v931debugprint (dev,
			       "Disable Character Editing Extent Boundary\n");
#endif
	       vdtinfo->CharExtentEnabled = FALSE;   /* Insert/Delete Char */
	       break;

	    case '1': /* ESC $ 1 - Enable Character Editing Extent Boundary */
#ifdef DEBUG931
	       v931debugprint (dev,
			       "Enable Character Editing Extent Boundary\n");
#endif
	       vdtinfo->CharExtentEnabled = TRUE;    /* Insert/Delete Char */
	       break;

	    default: /* error */
#ifdef DEBUG931DS
	       v931DumpSequence (dev, "Unknown sequence", FirstIndex,
				 prcring->tail);
#endif
	       break;
	    }
	    break;

	 case '%': /* ESC % Attr Row Col - Repeate Attribute to Address */
	    if ((Attr = termdgetcond (dev, prcring, FirstIndex)) < 0) return;
	    if ((Row = termdgetcond (dev, prcring, FirstIndex)) < 0) return;
	    if ((Col = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    Row -= 0x20;        /* 0..24 */
	    if (Row < VDT_MIN_ROW || Row > VDT_MAX_ROW)
	       Row = VDT_MIN_ROW;

	    Col -= 0x20;        /* 0..79 */
	    if (Col < VDT_MIN_COL || Col > VDT_MAX_COL)
	       Col = VDT_MIN_COL;

#if defined(DEBUG931) || defined(DEBUG931ATTR)
	    v931debugprint (dev,
			    "Repeate Attribute to Address ( >%02X,%d,%d )\n",
			    Attr, Row, Col);
#endif
	    for (r = vdtinfo->Row; r <= Row; r++)
	    {
	       for (c = vdtinfo->Col; c <= Col; c++)
	       {
		  vdtinfo->Shadow[r][c] = Attr;
	       }
	    }
	    break;

	 case '(': /* ESC ( - Control Start */
#ifdef DEBUG931
	    v931debugprint (dev, "Control Start\n");
#endif
	    break;

	 case ')': /* ESC ) - Control End */
#ifdef DEBUG931
	    v931debugprint (dev, "Control End\n");
#endif
	    break;

	 case '4': /* ESC 4 Attr - Set Display Attribute */
	    if ((Attr = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

#if defined(DEBUG931) || defined(DEBUG931ATTR)
	    v931debugprint (dev, "Set Display Attribute ( >%02X )\n", Attr);
#endif
	    vdtinfo->Attr = Attr;
	    break;

	 case '5': /* ESC 5 - Read Cursor Position */
	    /*
	    **   RESPONSE: 01 1B 28 1B 59 Var Var 1B 29
	    **    -- SOH ESC ( ESC Y Row Col ESC )
	    */
#ifdef DEBUG931
	    v931debugprint (dev, "Read Cursor Position\n");
#endif

	    termdputring (dev, inring, 0x01);
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, '(');
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'Y');
	    termdputring (dev, inring, 0x20 + vdtinfo->Row);
	    termdputring (dev, inring, 0x20 + vdtinfo->Col);
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, ')');
	    break;

	 case '6': /* ESC 6 - Read Status */
	    /*
	    **   RESPONSE: 01 1B 28 Var ... Var 1B 29
	    **    -- SOH ESC ( ... data .. ESC )
	    */
#ifdef DEBUG931
	    v931debugprint (dev, "Read Status\n");
#endif

	    termdputring (dev, inring, 0x01); /* SOH */
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, '(');
	    termdputring (dev, inring, 0x40); /* Byte  1 - Reserved */
	    termdputring (dev, inring, 0x40); /* Byte  2 - Reserved */
	    termdputring (dev, inring, 0x40); /* Byte  3 - Reserved */
	    termdputring (dev, inring, 0x40); /* Byte  4 - Reserved */
	    Data = 0x40;
	    Data |= vdtinfo->CursorOn;
	    Data |= vdtinfo->VideoOn;
	    termdputring (dev, inring, Data); /* Byte  5 - Cur. on / Vid. On */
	    termdputring (dev, inring, 0x40); /* Byte  6 - Count Overflow Off */
	    termdputring (dev, inring, 0x45); /* Byte  7 - Aux Online & Ready */
	    termdputring (dev, inring, 0x40); /* Byte  8 - Reserved */
	    Data = 0x40;
	    Data |= vdtinfo->AlternateCharacterSet;
	    termdputring (dev, inring, Data); /* Byte  9 - Charset Off */
	    Data = 0x40;
	    Data |= vdtinfo->LockOn;
	    termdputring (dev, inring, Data); /* Byte 10 - Keyboard Unlocked */
	    termdputring (dev, inring, 0x40); /* Byte 11 - Reserved */
	    termdputring (dev, inring, 0x40); /* Byte 12 - Reserved */
	    termdputring (dev, inring, 0x40); /* Byte 13 - Reserved */
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, ')');
	    break;

	 case '7': /* ESC 7 ... */
	    if ((Char = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    switch (Char)
	    {
	    case 'J': /* ESC 7 J ESC ( Count ... data ... ESC ) - Define ABM */
	       Variant = Char;
	       break;

	    case 'L': /* ESC 7 L ESC ( ESC ) - Restore Primary Status Line */
	       Variant = Char;
	       break;

	    default: /* error */
	       Variant = 0;     /* NONE */
#ifdef DEBUG931DS
	       v931DumpSequence (dev, "Unknown sequence", FirstIndex,
				 prcring->tail);
#endif
	       break;
	    }

	    /*
	    ** ESC 7 J ...
	    */
	    if ((Char = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    if (Char != ESCAPE) /* error */
	    {
	       break;
	    }

	    /*
	    ** ESC 7 J ESC  ...
	    */
	    if ((Char = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    if (Char == '(')
	    {
	       DataLength = v931getdata (dev, prcring, databuf,
					 FirstIndex, sizeof(databuf));
	       if (DataLength < 0) return;
	    }

	    /*
	    ** ESC 7 J ESC ( Count ... data ... ESC ) - Process Buffer
	    */
	    switch (Variant)
	    {
	    case 'J': /* ESC 7 J ESC ( Count ... data ... ESC ) - Define ABM */
#ifdef DEBUG931
	       v931debugprint (dev, "Define ABM: DataLength = %d, Count = %d\n",
			       DataLength, databuf[0] - ' ');
#endif

	       if ((Count = databuf[0] - ' ') == 0)
	       {
		  vdtinfo->ABM_Count = 0; /* clear ABM */
	       }
	       else if (Count < sizeof (vdtinfo->ABM_Buffer))
	       {
		  vdtinfo->ABM_Count = DataLength - 1;
		  strcpy ((char *)vdtinfo->ABM_Buffer, &databuf[1]);
	       }
	       break;

	    case 'L': /* ESC 7 L ESC ( ESC ) - Restore Primary Status Line */
#ifdef DEBUG931
	       v931debugprint (dev, "Restore Primary Status Line\n");
#endif
	       vdtinfo->StatusActive = VDT_STATUS_PRIMARY;
	       break;

	    default: /* error */
#ifdef DEBUG931DS
	       v931DumpSequence (dev, "Unknown sequence", FirstIndex,
				 prcring->tail);
#endif
	       break;
	    }
	    break;

	 case ':': /* ESC : - Keyboard Lock */
#ifdef DEBUG931
	    v931debugprint (dev, "Keyboard Lock\n");
#endif
	    vdtinfo->LockOn = 0x01;
	    break;

	 case ';': /* ESC ; - Keyboard Unlock */
#ifdef DEBUG931
	    v931debugprint (dev, "Keyboard Unlock\n");
#endif
	    vdtinfo->LockOn = 0x00;
	    break;

	 case '<': /* ESC < - Erase Message */
#ifdef DEBUG931
	    v931debugprint (dev, "Erase Message\n");
#endif
	    if (dev->switches & SWSTATUS)
	    {
	       int c;

	       for (c = 0; c < 40; c++)
	       {
		  vdtinfo->StatusLine[VDT_STATUS_PRIMARY][c] = ' ';
	       }
	    }
	    break;

	 case '>': /* ESC > Row Col - Erase to Address */

	    if ((Row = termdgetcond (dev, prcring, FirstIndex)) < 0) return;
	    if ((Col = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    Row -= 0x20;        /* 0..24 */
	    if (Row < VDT_MIN_ROW || Row > VDT_MAX_ROW)
	       Row = VDT_MIN_ROW;

	    Col -= 0x20;        /* 0..79 */
	    if (Col < VDT_MIN_COL || Col > VDT_MAX_COL)
	       Col = VDT_MIN_COL;

#ifdef DEBUG931
	    v931debugprint (dev, "Erase to Address ( %d,%d )\n", Row, Col);
#endif

	    VDT_EraseArea (dev, vdtinfo->Row, vdtinfo->Col, Row, Col);
	    break;

	 case '?': /* ESC ? Char Row Col - Repeat To Address */

	    if ((Char = termdgetcond (dev, prcring, FirstIndex)) < 0) return;
	    if ((Row = termdgetcond (dev, prcring, FirstIndex)) < 0) return;
	    if ((Col = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    Row -= 0x20;        /* 0..24 */
	    if (Row < VDT_MIN_ROW || Row > VDT_MAX_ROW)
	       Row = VDT_MIN_ROW;

	    Col -= 0x20;        /* 0..79 */
	    if (Col < VDT_MIN_COL || Col > VDT_MAX_COL)
	       Col = VDT_MIN_COL;

#ifdef DEBUG931
	    v931debugprint (dev, "Repeat To Address ( >%02X,%d,%d )\n",
			    Char, Row, Col);
#endif
	    Char = vdtinfo->Buffer[vdtinfo->Row][vdtinfo->Col];
	    Count = (Row - vdtinfo->Row) * VDT_NUM_COLS +
	       (Col - vdtinfo->Col) - 1;
	    VDT_PutChar (dev, Char, Count);
	    break;

	 case '@': /* ESC @ Row Col - Read to Address */
	    /*
	    **   RESPONSE: 01 1B 28 Var ... Var 1B 29 
	    **    -- SOH ESC ( ... data ... ESC )
	    */

	    if ((Row = termdgetcond (dev, prcring, FirstIndex)) < 0) return;
	    if ((Col = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    Row -= 0x20;        /* 0..24 */
	    if (Row < VDT_MIN_ROW || Row > VDT_MAX_ROW)
	    {
	       Row = VDT_MIN_ROW;
	    }

	    Col -= 0x20;        /* 0..79 */
	    if (Col < VDT_MIN_COL || Col > VDT_MAX_COL)
	    {
	       Col = VDT_MIN_COL;
	    }

#ifdef DEBUG931
	    v931debugprint (dev, "Read to Address ( %d,%d )\n", Row, Col);
#endif
	    termdputring (dev, inring, 0x01);
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, '(');
	    /* TODO - FIXME - ... data ... */
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, ')');
	    break;

	 case 'A': /* ESC A - Cursor Up */
#ifdef DEBUG931
	    v931debugprint (dev, "Cursor Up\n");
#endif
	    vdtinfo->Row--;
	    VDT_AdjustCursor (dev);
	    break;

	 case 'B': /* ESC B - Cursor Down */
#ifdef DEBUG931
	    v931debugprint (dev, "Cursor Down\n");
#endif
	    vdtinfo->Row++;
	    VDT_AdjustCursor (dev);
	    break;

	 case 'C': /* ESC C - Cursor Right */
#ifdef DEBUG931
	    v931debugprint (dev, "Cursor Right\n");
#endif
	    vdtinfo->Col++;
	    VDT_AdjustCursor (dev);
	    break;

	 case 'D': /* ESC D - Cursor Left */
#ifdef DEBUG931
	    v931debugprint (dev, "Cursor Left\n");
#endif
	    vdtinfo->Col--;
	    VDT_AdjustCursor (dev);
	    break;

	 case 'F': /* ESC F ... */
	    /* 
	    ** Write Data to Device ( Printer Acknowledge )
	    ** ESC F SOH 1 ESC ( ... data ... ESC )
	    **     RESPONSE: 1B 31 -- ESC 1
	    ** Write Data to Device ( Transparent Print )
	    ** ESC F ESC ( ... data ... ESC )
	    */
	    Variant = 0;        /* ( Transparent Print ) */
	    if ((Char = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    if (Char == 0x01) /* ESC F SOH ... */
	    {
	       /*
		** Write Data to Device and Printer Acknowledge
		** RESPONSE: 1B 31 -- ESC 1
		*/
	       if ((Char = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	       if (Char == '1') /* ESC F SOH 1 ... */
	       {
		  Variant = 1;     /* ( Printer Acknowledge ) */
	       }
	    }
#ifdef DEBUG931
	    v931debugprint (dev,
			   "Write Data to Device: Variant = %d, Char = >%02X\n",
			    Variant, Char);
#endif

	    if ((Char = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    if (Char != ESCAPE)
	    {
	       break;
	    }
	    /*
	    ** ESC F SOH 1 ESC ... 
	    */
	    if ((Char = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    if (Char == '(')
	    {
	       DataLength = v931getdata (dev, prcring, databuf,
				         FirstIndex, sizeof(databuf));
	       if (DataLength < 0) return;
	    }

	    /*
	    ** ESC F SOH 1 ESC ( ... data ... ESC )
	    */
#ifdef DEBUG931
	    v931debugprint (dev, "Write Data to Device: DataLength = %d\n",
			    DataLength);
#endif
	    /* Process Buffer */
	    if (DataLength > 0)
	    {
	       /* send to attached printer */
	       int i;

	       for (i = 0; i < DataLength; i++)
	       {
		  if (vdtinfo->printfd)
		  {
		     Char = databuf[i];
		     if (Char == 0x0C) fputc (0x0A, vdtinfo->printfd);
		     if ((Char > 0x1F && Char < 0x7F) ||
			 (Char > 0x07 && Char < 0x0D))
			fputc (Char, vdtinfo->printfd);
		  }
	       }
	    }
	    if (Variant == 1)
	    {
	       /* 
	       ** ( Printer Acknowledge )
	       ** RESPONSE: 1B 31 -- ESC 1
	       */
	       termdputring (dev, inring, ESCAPE);
	       termdputring (dev, inring, '1');
	    }
	    break;

	 case 'G': /* ESC G ... */
	    if ((Char = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    switch (Char)
	    {
	    case 'A':
	       /*
	       ** ESC G A ESC ( ... data ... ESC ) - Write to Message Area
	       */
	       Variant = Char;
	       break;

	    case 'B':
	       /*
	       ** ESC G B ESC ( ... data ... ESC ) - Define Oper. Parameters
	       */
	       Variant = Char;
	       break;

	    case 'E':
	       /*
	       ** ESC G E ESC ( ... data ... ESC ) - External Loopback Test
	       **    RESPONSE:  ... data ...
	       */
	       Variant = Char;
	       break;

	    case 'I':
	       /*
	       ** ESC G I ESC ( 0 ESC ) - Reset Nonvolatile Memory
	       ** ESC G I ESC ( 1 ESC ) - Execute RAM Test
	       **     RESPONSE: 1B 47 49 1B 28 50 1B 29 -- ESC G I ESC ( P ESC )
	       ** ESC G I ESC ( 2 ESC ) - Verify All ROMs
	       **     RESPONSE: 1B 47 49 1B 28 Var ... Var 1B 29
	       **      -- ESC G I ESC ( ... data ... ESC )
	       ** ESC G I ESC ( 3 ESC ) - Barberpoll All Character Sets
	       ** ESC G I ESC ( 4 ESC ) - Display Screen Adjust Pattern
	       ** ESC G I ESC ( 5 ESC ) - Enter Keyboard Test Mode
	       ** ESC G I ESC ( 6 ESC ) - Exit Keyboard Test Mode
	       ** ESC G I ESC ( 7 ESC ) - Report Terminal ID
	       **     RESPONSE: 1B 47 49 1B 28 39 33 31 1B 29
	       **      -- ESCF G I ESC ( 9 3 1 ESC )
	       */
	       Variant = Char;
	       break;

	    case 'K':
	       /*
	       ** ESC G K ESC ( ... data ... ESC )
	       ** Write to and Select Alternate Status Line
	       */
	       Variant = Char;
	       break;

	    case 'M':
	       /*
	       ** ESC G M ESC ( ... data ... ESC ) - Define Comm. Parameters
	       */
	       Variant = Char;
	       break;

	    case 'N':
	       /*
	       ** ESC G N ESC ( ... data ... ESC ) - Define Aux Parameters
	       */
	       Variant = Char;
	       break;

	    default: /* error */
	       Variant = 0;     /* NONE */
#ifdef DEBUG931DS
	       v931DumpSequence (dev, "Unknown sequence", FirstIndex,
				 prcring->tail);
#endif
	       break;
	    }

	    if ((Char = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    if (Char != ESCAPE)
	    {
	       break;
	    }
	    /*
	    ** ESC G A ESC 
	    */
	    if ((Char = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    if (Char == '(')
	    {
	       DataLength = v931getdata (dev, prcring, databuf,
					 FirstIndex, sizeof(databuf));
	       if (DataLength < 0) return;
	    }

	    /*
	    ** ESC G A ESC ( ... data ... ESC )
	    */

	    /* process Buffer */
	    if (DataLength > 0)
	    {
	       switch (Variant)
	       {
	       case 'A':
		  /*
		  ** ESC G A ESC ( ... data ... ESC ) - Write to Message Area
		  */
#ifdef DEBUG931
		  v931debugprint (dev, "Write to Message Area\n");
#endif
		  if (dev->switches & SWSTATUS)
		  {
		     int i;

		     for (i = 0; i < DataLength; i++)
		     {
			if (i < VDT_NUM_COLS)
			   vdtinfo->StatusLine[VDT_STATUS_PRIMARY][i] =
			      databuf[i];
		     }
		  }
		  break;

	       case 'B':
		  /*
		  ** ESC G B ESC ( ... data ... ESC ) - Define Oper. Params.
		  */
#ifdef DEBUG931
		  v931debugprint (dev, "Define Operational Parameters\n");
#endif
		  break;

	       case 'E':
		  /*
		  ** ESC G E ESC ( ... data ... ESC ) - External Loopback Test
		  **     RESPONSE:  ... data ...
		  */
#ifdef DEBUG931
		  v931debugprint (dev, "External Loopback Test\n");
#endif
		  vdtinfo->CommState = VDT_COMMSTATE_LOOP; /* "Comm:Lpbk" */
		  if (DataLength > 0)
		  {
		     /* echo "... data ..." back to the Host */
		     int i;

		     for (i = 0; i < DataLength; i++)
		     {
			Char = databuf[i];
			termdputring (dev, inring, Char);
		     }
		  }
		  break;

	       case 'I': /* various */
		  Variant = databuf[0];
		  switch (Variant)
		  {
		  case '0':
		     /*
		     ** ESC G I ESC ( 0 ESC ) - Reset Nonvolatile Memory
		     */
#ifdef DEBUG931
		     v931debugprint (dev, "Reset Nonvolatile Memory\n");
#endif
		     break;

		  case '1':
		     /*
		     ** ESC G I ESC ( 1 ESC ) - Execute RAM Test
		     **    RESPONSE: 1B 47 49 1B 28 50 1B 29
		     **     -- ESC G I ESC ( P ESC )
		     */
#ifdef DEBUG931
		     v931debugprint (dev, "Execute RAM Test\n");
#endif
		     termdputring (dev, inring, ESCAPE);
		     termdputring (dev, inring, 'G');
		     termdputring (dev, inring, 'I');
		     termdputring (dev, inring, ESCAPE);
		     termdputring (dev, inring, '(');
		     termdputring (dev, inring, 'P');
		     termdputring (dev, inring, ESCAPE);
		     termdputring (dev, inring, ')');
		     break;

		  case '2':
		     /*
		     ** ESC G I ESC ( 2 ESC ) - Verify All ROMs
		     **    RESPONSE: 1B 47 49 1B 28 Var ... Var 1B 29
		     **      -- ESC G I ESC ( ... data ... ESC )
		     */
#ifdef DEBUG931
		     v931debugprint (dev, "Verify All ROMs\n");
#endif
		     termdputring (dev, inring, ESCAPE);
		     termdputring (dev, inring, 'G');
		     termdputring (dev, inring, 'I');
		     termdputring (dev, inring, ESCAPE);
		     termdputring (dev, inring, '(');
		     /* ... data ... */
		     termdputring (dev, inring, '1');  /* Version */
		     termdputring (dev, inring, ',');  /* , */
		     termdputring (dev, inring, 'A');  /* Revision */
		     termdputring (dev, inring, ',');  /* , */
		     termdputring (dev, inring, '2');  /* TI PN:2229228-0027 */
		     termdputring (dev, inring, '2');
		     termdputring (dev, inring, '2');
		     termdputring (dev, inring, '9');
		     termdputring (dev, inring, '2');
		     termdputring (dev, inring, '2');
		     termdputring (dev, inring, '8');
		     termdputring (dev, inring, '0');
		     termdputring (dev, inring, '0');
		     termdputring (dev, inring, '2');
		     termdputring (dev, inring, '7');
		     termdputring (dev, inring, ',');  /* , */
		     termdputring (dev, inring, 'P');  /* Passed */
		     termdputring (dev, inring, ',');  /* , */
		     /* ... data ... */
		     termdputring (dev, inring, '1');  /* Version */
		     termdputring (dev, inring, ',');  /* , */
		     termdputring (dev, inring, 'A');  /* Revision */
		     termdputring (dev, inring, ',');  /* , */
		     termdputring (dev, inring, '2');  /* TI PN:2229228-0028 */
		     termdputring (dev, inring, '2');
		     termdputring (dev, inring, '2');
		     termdputring (dev, inring, '9');
		     termdputring (dev, inring, '2');
		     termdputring (dev, inring, '2');
		     termdputring (dev, inring, '8');
		     termdputring (dev, inring, '0');
		     termdputring (dev, inring, '0');
		     termdputring (dev, inring, '2');
		     termdputring (dev, inring, '8');
		     termdputring (dev, inring, ',');  /* , */
		     termdputring (dev, inring, 'P');  /* Passed */
		     termdputring (dev, inring, ESCAPE);
		     termdputring (dev, inring, ')');
		     break;

		  case '3':
		     /*
		     ** ESC G I ESC ( 3 ESC ) - Barberpoll All Character Sets
		     */
#ifdef DEBUG931
		     v931debugprint (dev, "Barberpoll All Character Sets\n");
#endif
		     Char = ' ';
		     for (vdtinfo->Row = 0; vdtinfo->Row < VDT_NUM_ROWS;
			  vdtinfo->Row++)
		     {
			for (vdtinfo->Col = 0; vdtinfo->Col < VDT_NUM_COLS;
			     vdtinfo->Col++)
			{
			   vdtinfo->Buffer[vdtinfo->Row][vdtinfo->Col] = Char;
			   vdtinfo->Shadow[vdtinfo->Row][vdtinfo->Col] =
				 VDT_ATTR_DEFAULT;
			   Char++;
			   if (Char == 0x7F)
			   {
			      Char = ' ';
			   }
			}
		     }
		     vdtinfo->Row = 0;
		     vdtinfo->Col = 0;
		     break;

		  case '4':
		     /*
		     ** ESC G I ESC ( 4 ESC ) - Display Screen Adjust Pattern
		     */
#ifdef DEBUG931
		     v931debugprint (dev, "Display Screen Adjust Pattern\n");
#endif
		     Char = 'O';
		     for (vdtinfo->Row = 0; vdtinfo->Row < VDT_NUM_ROWS;
			  vdtinfo->Row++)
		     {
			for (vdtinfo->Col = 0; vdtinfo->Col < VDT_NUM_COLS;
			     vdtinfo->Col++)
			{
			   vdtinfo->Buffer[vdtinfo->Row][vdtinfo->Col] = Char;
			   vdtinfo->Shadow[vdtinfo->Row][vdtinfo->Col] =
				 VDT_ATTR_DEFAULT;
			}
		     }
		     vdtinfo->Row = 0;
		     vdtinfo->Col = 0;
		     break;

		  case '5':
		     /*
		     ** ESC G I ESC ( 5 ESC ) - Enter Keyboard Test Mode
		     */
#ifdef DEBUG931
		     v931debugprint (dev, "Enter Keyboard Test Mode\n");
#endif
		     break;

		  case '6':
		     /*
		     ** ESC G I ESC ( 6 ESC ) - Exit Keyboard Test Mode
		     */
#ifdef DEBUG931
		     v931debugprint (dev, "Exit Keyboard Test Mode\n");
#endif
		     break;

		  case '7':
		     /*
		     ** ESC G I ESC ( 7 ESC ) - Report Terminal ID
		     **    RESPONSE: 1B 47 49 1B 28 39 33 31 1B 29
		     **      -- ESC G I ESC ( 9 3 1 ESC )
		     */
#ifdef DEBUG931
		     v931debugprint (dev, "Report Terminal ID\n");
#endif
		     termdputring (dev, inring, ESCAPE);
		     termdputring (dev, inring, 'G');
		     termdputring (dev, inring, 'I');
		     termdputring (dev, inring, ESCAPE);
		     termdputring (dev, inring, '(');
		     termdputring (dev, inring, '9');
		     if (dev->emutype == EMU931)
		     {
			termdputring (dev, inring, '3');
			termdputring (dev, inring, '1');
		     }
		     else /* EMU940 */
		     {
			termdputring (dev, inring, '4');
			termdputring (dev, inring, '0');
		     }
		     termdputring (dev, inring, ESCAPE);
		     termdputring (dev, inring, ')');
		     break;

		  default: /* error */
#ifdef DEBUG931DS
		     v931DumpSequence (dev, "Unknown sequence", FirstIndex,
				       prcring->tail);
#endif
		     break;
		  }
		  break;

	       case 'K':
		  /*
		  ** ESC G K ESC ( ... data ... ESC )
		  ** Write to and Select Alternate Status Line
		  */
#ifdef DEBUG931
		  v931debugprint (dev, 
				 "Write to and Select Alternate Status Line\n");
#endif
		  if (dev->switches & SWSTATUS)
		  {
		     int i;

		     for (i = 0; i < DataLength; i++)
		     {
			Char = databuf[i];
			if (i < VDT_NUM_COLS)
			   vdtinfo->StatusLine[VDT_STATUS_ALTERNATE][i] = Char;
		     }
		     vdtinfo->StatusActive = VDT_STATUS_ALTERNATE;
		  }
		  break;

	       case 'M':
		  /*
		  ** ESC G M ESC ( ... data ... ESC )
		  ** Define Communication Parameters
		  */
#ifdef DEBUG931
		  v931debugprint (dev, "Define Communication Parameters\n");
#endif
		  break;

	       case 'N':
		  /*
		  ** ESC G N ESC ( ... data ... ESC )
		  ** Define Aux Parameters
		  */
#ifdef DEBUG931
		  v931debugprint (dev, "Define Aux Parameters\n");
#endif
		  break;

	       default: /* error */
#ifdef DEBUG931DS
		  v931DumpSequence (dev, "Unknown sequence", FirstIndex,
				    prcring->tail);
#endif
	       break;
	       }
	    }
	    break;

	 case 'H': /* ESC H - Cursor Home */
#ifdef DEBUG931
	    v931debugprint (dev, "Cursor Home\n");
#endif
	    vdtinfo->Row = VDT_MIN_ROW;
	    vdtinfo->Col = VDT_MIN_COL;
	    VDT_AdjustCursor (dev);
	    break;

	 case 'I': /* ESC I - Erase to End of Line */
#ifdef DEBUG931
	    v931debugprint (dev, "Erase to End of Line\n");
#endif
	    VDT_EraseArea (dev, vdtinfo->Row, vdtinfo->Col,
			   vdtinfo->Row, VDT_MAX_COL);
	    break;

	 case 'J': /* ESC J - Erase to End of Screen */
#ifdef DEBUG931
	    v931debugprint (dev, "Erase to End of Screen\n");
#endif
	    VDT_EraseArea (dev, vdtinfo->Row, vdtinfo->Col,
			   vdtinfo->Row, VDT_MAX_COL);
	    VDT_EraseArea (dev, vdtinfo->Row + 1, VDT_MIN_COL,
			   VDT_MAX_ROW, VDT_MAX_COL);
	    break;

	 case 'L': /* ESC L - Erase All */
#ifdef DEBUG931
	    v931debugprint (dev, "Erase All\n");
#endif
	    vdtinfo->Row = VDT_MIN_ROW;
	    vdtinfo->Col = VDT_MIN_COL;
	    vdtinfo->Fill = ' ';
	    vdtinfo->Attr = VDT_ATTR_DEFAULT;
	    vdtinfo->GraphicCharSet = FALSE;
	    vdtinfo->AlternateCharacterSet = 0x00;
	    VDT_AdjustCursor (dev);
	    VDT_EraseArea (dev, VDT_MIN_COL, VDT_MIN_COL,
			   VDT_MAX_ROW, VDT_MAX_COL);
	    break;

	 case 'M': /* ESC M - Cursor On */
#ifdef DEBUG931
	    v931debugprint (dev, "Cursor On\n");
#endif
	    vdtinfo->CursorOn = 1;
	    break;

	 case 'N': /* ESC N - Insert Line */
#ifdef DEBUG931
	    v931debugprint (dev, "Insert Line\n");
#endif
	    VDT_InsertDelete (dev, TRUE /* Insert */ , TRUE /* Line */ );
	    break;

	 case 'O': /* ESC O - Delete Line */
#ifdef DEBUG931
	    v931debugprint (dev, "Delete Line\n");
#endif
	    VDT_InsertDelete (dev, FALSE /* Delete */ , TRUE /* Line */ );
	    break;

	 case 'P': /* ESC P - Insert Character */
#ifdef DEBUG931
	    v931debugprint (dev, "Insert Character\n");
#endif
	    VDT_InsertDelete (dev, TRUE /* Insert */ , FALSE /* Char */ );
	    break;

	 case 'Q': /* ESC Q - Delete Character */
#ifdef DEBUG931
	    v931debugprint (dev, "Delete Character\n");
#endif
	    VDT_InsertDelete (dev, FALSE /* Delete */ , FALSE /* Char */ );
	    break;

	 case 'R': /* ESC R - Cursor Off */
#ifdef DEBUG931
	    v931debugprint (dev, "Cursor Off\n");
#endif
	    vdtinfo->CursorOn = 0;
	    break;

	 case 'S': /* ESC S - Cancel Output */
#ifdef DEBUG931
	    v931debugprint (dev, "Cancel Output\n");
#endif
	    break;

	 case 'V': /* ESC V ... */
	    if ((Char = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    switch (Char)
	    {
	    case '0': /* ESC V 0 Disable Cursor Blink */
#ifdef DEBUG931
	       v931debugprint (dev, "Disable Cursor Blink\n");
#endif
	       vdtinfo->BlinkOn = 0;
	       break;

	    case '1': /* ESC V 1 - Enable Cursor Blink */
#ifdef DEBUG931
	       v931debugprint (dev, "Enable Cursor Blink\n");
#endif
	       vdtinfo->BlinkOn = 1;
	       break;

	    default: /* error */
#ifdef DEBUG931DS
	       v931DumpSequence (dev, "Unknown sequence", FirstIndex,
				 prcring->tail);
#endif
	       break;
	    }
	    break;

	 case 'Y': /* ESC Y Row Col - Set Cursor Address */

	    if ((Row = termdgetcond (dev, prcring, FirstIndex)) < 0) return;
	    if ((Col = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    Row -= 0x20;        /* 0..24 */
	    if (Row < VDT_MIN_ROW || Row > VDT_MAX_ROW)
	       Row = VDT_MIN_ROW;

	    Col -= 0x20;        /* 0..79 */
	    if (Col < VDT_MIN_COL || Col > VDT_MAX_COL)
	       Col = VDT_MIN_COL;

#ifdef DEBUG931
	    v931debugprint (dev, "Set Cursor Address ( %d,%d )\n", Row, Col);
#endif
	    vdtinfo->Row = Row;
	    vdtinfo->Col = Col;
	    VDT_AdjustCursor (dev);
	    break;

	 case 'a': /* ESC a - Scroll Up */
#ifdef DEBUG931
	    v931debugprint (dev, "Scroll Up\n");
#endif
	    VDT_ScrollScreen (dev, TRUE);
	    break;

	 case 'b': /* ESC b - Scroll Down */
#ifdef DEBUG931
	    v931debugprint (dev, "Scroll Down\n");
#endif
	    VDT_ScrollScreen (dev, TRUE);
	    break;

	 case 'j': /* ESC j Attr Count - Repeat Attribute N Times */
	    if ((Attr = termdgetcond (dev, prcring, FirstIndex)) < 0) return;
	    if ((Count = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

#if defined(DEBUG931) || defined(DEBUG931ATTR)
	    v931debugprint (dev,
			    "Repeat Attribute N Times ( >%02X, >%02X:%d )\n",
			    Attr, Count, Count - 0x1F);
#endif
	    Count -= 0x1F;      /* 1..80 */
	    VDT_PutAttr (dev, Attr, Count);
	    break;

	 case 'k': /* ESC k Char Count - Repeat Character N Times */
	    if ((Char = termdgetcond (dev, prcring, FirstIndex)) < 0) return;
	    if ((Count = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

#ifdef DEBUG931
	    v931debugprint (dev,
			    "Repeat Character N Times ( >%02X, >%02X:%d )\n",
			    Char, Count, Count - 0x1F);
#endif
	    Count -= 0x1F;      /* 1..96 */
	    VDT_PutChar (dev, Char, Count);
	    break;

	 case 'w': /* ESC w ... */
	    if ((Char = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    switch (Char)
	    {
	    case '0': /* ESC w 0 Disable Personaity Module Mode */
#ifdef DEBUG931
	       v931debugprint (dev, "Disable Personaity Module Mode\n");
#endif
	       break;

	    case '1': /* ESC w 1 - Enable Personality Module Mode */
#ifdef DEBUG931
	       v931debugprint (dev, "Enable Personality Module Mode\n");
#endif
	       break;

	    default: /* error */
#ifdef DEBUG931DS
	       v931DumpSequence (dev, "Unknown sequence", FirstIndex,
				 prcring->tail);
#endif
	       break;
	    }
	    break;

	 case 'x': /* ESC x Nrows Ncols Row Col - Move Box */

	    if ((Nrows = termdgetcond (dev, prcring, FirstIndex)) < 0) return;
	    if ((Ncols = termdgetcond (dev, prcring, FirstIndex)) < 0) return;
	    if ((Row = termdgetcond (dev, prcring, FirstIndex)) < 0) return;
	    if ((Col = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    Nrows -= 0x20;      /* 0..24 */
	    if (Nrows < VDT_MIN_ROW || Nrows > VDT_MAX_ROW)
	    {
	       Nrows = VDT_MIN_ROW;
	    }

	    Ncols -= 0x20;      /* 0..79 */
	    if (Ncols < VDT_MIN_COL || Ncols > VDT_MAX_COL)
	    {
	       Ncols = VDT_MIN_COL;
	    }

	    Row -= 0x20;        /* 0..24 */
	    if (Row < VDT_MIN_ROW || Row > VDT_MAX_ROW)
	       Row = VDT_MIN_ROW;

	    Col -= 0x20;        /* 0..79 */
	    if (Col < VDT_MIN_COL || Col > VDT_MAX_COL)
	       Col = VDT_MIN_COL;

#ifdef DEBUG931
	    v931debugprint (dev, "Move Box ( %d,%d to %d,%d )\n",
			    Nrows, Ncols, Row, Col);
#endif
	    break;

	 case 'y': /* ESC y ...  */
	    if ((Char = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

	    switch (Char)
	    {
	    case '1': /* ESC y 1 - Select and Enter Alt. Char. Set #1 */
#ifdef DEBUG931
	       v931debugprint (dev, 
			       "Select and Enter Alternate Character Set #1\n");
#endif
	       vdtinfo->AlternateCharacterSet = 0x02;
	       break;

	    case '2': /* ESC y 2 - Select and Enter Alt. Char. Set #2 */
#ifdef DEBUG931
	       v931debugprint (dev, 
			       "Select and Enter Alternate Character Set #2\n");
#endif
	       vdtinfo->AlternateCharacterSet = 0x10;
	       break;

	    default: /* error */
#ifdef DEBUG931DS
	       v931DumpSequence (dev, "Unknown sequence", FirstIndex,
				 prcring->tail);
#endif
	       break;
	    }
	    break;

	 case '{': /* ESC { Attr - Attribute OR */
	    if ((Attr = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

#if defined(DEBUG931) || defined(DEBUG931ATTR)
	    v931debugprint (dev, "Attribute OR ( >%02X )", Attr);
#endif
	    vdtinfo->Attr |= Attr;
#if defined(DEBUG931) || defined(DEBUG931ATTR)
	    fprintf (stderr, " -> ( >%02X )\n", vdtinfo->Attr);
#endif
	    break;

	 case '|': /* ESC | - Video On */
#ifdef DEBUG931
	    v931debugprint (dev, "Video On\n");
#endif
	    vdtinfo->VideoOn = 0x02;
	    break;

	 case '}': /* ESC } Attr - Attribute AND */
	    if ((Attr = termdgetcond (dev, prcring, FirstIndex)) < 0) return;

#if defined(DEBUG931) || defined(DEBUG931ATTR)
	    v931debugprint (dev, "Attribute AND ( >%02X )", Attr);
#endif
	    vdtinfo->Attr &= Attr;
#if defined(DEBUG931) || defined(DEBUG931ATTR)
	    fprintf (stderr, " -> ( >%02X )\n", vdtinfo->Attr);
#endif
	    break;

	 case '~': /* ESC ~ - Video Off */
#ifdef DEBUG931
	    v931debugprint (dev, "Video Off\n");
#endif
	    vdtinfo->VideoOn = 0x00;
	    break;

	 default: /* error */
#ifdef DEBUG931DS
	    v931DumpSequence (dev, "Unknown sequence", FirstIndex,
			      prcring->tail);
#endif
	    break;
	 }
	 break;

      default:
	 /*
	 ** pass-thru
	 */
	 VDT_PutChar (dev, Char, 1);
      }
   }
} 

/***********************************************************************
* v93display - VDT 931 display.
***********************************************************************/

int
v931display (Device *dev, int fd)
{
   VDTInfo *vdtinfo;
   TermInfo *terminfo;
   uint8 *bp, *sp;
   int row, col;
   int status;

   if (!(dev->switches & SWTELNET) && !run) return (0);
#ifdef DEBUG931
   if (dev->cbreak)
      fprintf (stderr,
	       "v931display-%s: devaddr = >%04X, fd = %d, cbreak = %s\n",
	       emulations[dev->emutype],
	       dev->devaddr, fd, dev->cbreak ? "TRUE" : "FALSE");
#endif
   terminfo = &dev->info.terminfo;
   vdtinfo = &terminfo->terminal.vdtinfo;

   status = 0;
   bp = sp = vdtinfo->screen;
   if (dev->cbreak)
   {
      uint8 LastAttr;

      if (termdringempty (dev, &terminfo->outring))
	 dev->cbreak = FALSE;

      sp = VDT_DisplayClear (dev, sp);

      LastAttr = VDT_ATTR_DEFAULT;
      sp = VDT_DisplayAttr (dev, sp, LastAttr);

      for (row = 0; sp && row < VDT_NUM_ROWS; row++)
      {
	 sp = VDT_DisplayPosition (dev, sp, row + 1, 1);

	 for (col = 0; col < VDT_NUM_COLS; col++)
	 {
	    if (vdtinfo->Shadow[row][col] != LastAttr)
	    {
	       sp = VDT_DisplayScreen (dev, fd, bp, sp - bp, FALSE);
	       bp = sp;
	       LastAttr = vdtinfo->Shadow[row][col];
	       sp = VDT_DisplayAttr (dev, sp, LastAttr);
	    }
	    *sp++ = vdtinfo->Buffer[row][col];
	 }
#ifdef DEBUG931SCREEN
	 fprintf (stderr, "Buffer: row = %d\n", row);
         HEXDUMP (stderr, (char *)vdtinfo->Buffer[row], VDT_NUM_COLS, 0);
#endif
	 sp = VDT_DisplayScreen (dev, fd, bp, sp - bp, FALSE);
	 bp = sp;
      }

      if (sp && (dev->switches & SWSTATUS))
      {
	 int c;
	 char Buffer[41];

	 sprintf (Buffer, " %-5s %6s %6s Comm:%-4s %-8s ",
		  vdtinfo->CommDTR ? "Onln" : "Offln",
		  vdtinfo->CommDCD ? "DCD(1)" : "DCD(0)",
		  vdtinfo->CommDSR ? "DSR(1)" : "DSR(0)",
		  &v931_CommMessage[vdtinfo->CommState][0],
		  vdtinfo->printfd ? "Aux:Rdy" : "Aux:Off");
	 memcpy (&vdtinfo->StatusLine[VDT_STATUS_PRIMARY][40], Buffer, 40);

	 sp = VDT_DisplayPosition (dev, bp, VDT_NUM_ROWS + 1, 1);
	 sp = VDT_DisplayAttr (dev, sp, VDT_ATTR_BOLD);

	 for (c = 0; c < VDT_NUM_COLS; c++)
	 {
	    *sp++ = vdtinfo->StatusLine[vdtinfo->StatusActive][c];
	 }
      }
      sp = VDT_DisplayScreen (dev, fd, vdtinfo->screen,
			      sp - vdtinfo->screen, TRUE);
      bp = sp;

      /*
      ** Position cursor on screen
      */

      if (sp)
      {
	 sp = VDT_DisplayAttr (dev, bp, VDT_ATTR_DEFAULT);
	 sp = VDT_DisplayPosition (dev, sp, vdtinfo->Row + 1, vdtinfo->Col + 1);
	 sp = VDT_DisplayScreen (dev, fd, bp, sp - bp, TRUE);
      }
      else
      {
	 status = -1;
      }
   }

   return (status);
}
