/***********************************************************************
*
* sim990.c - Simulator for the TI 990 computer.
*
* Changes:
*   05/29/03   DGP   Original.
*   06/20/03   DGP   Added interrupt support.
*   12/09/03   DGP   Changed ROM load to >FC00.
*   12/10/03   DGP   Added 20 bit addresses for ROM and TILINE.
*   01/26/04   DGP   Added memory size option.
*   05/10/04   DGP   Added DEV_NULL support.
*   07/01/04   DGP   Added /10A cpu, model = 11.
*   04/01/05   DGP   Changed signal handling.
*   04/12/05   DGP   Changed CONOUT to use prtview in panel mode and
*                    view mode to support multi screens.
*   04/20/05   DGP   Ensure args don't exceed argc.
*   12/04/05   DGP   Added upcase switch on CONIN.
*   10/01/07   DGP   Added realtime clock arg.
*   11/03/08   DGP   Added ROM lookup to check /usr/local/share/sim990, too.
*   11/11/08   DGP   Added LCS support.
*   11/12/08   DGP   Extended /12 rom support.
*   11/14/08   DGP   Improved help facility.
*   02/18/09   DGP   Added breakpoint CRU support.
*   01/18/11   DGP   Added watch point break.
*   01/24/11   DGP   Print simloader error.
*   07/25/11   DGP   Display regs, status, etc in non-panel mode.
*   09/07/11   DGP   Added disk geometry checks.
*   02/04/12   DGP   Added fakeprefetch for /12.
*   03/07/12   DGP   Fixup /12 ROM self test issues.
*   03/23/12   DGP   Added romtowcs().
*   09/17/13   DGP   Added 9902 support.
*                    Revamp console support.
*   09/19/13   DGP   Added 9903 support.
*   10/07/13   DGP   Fixed disassembly PC.
*   10/14/13   DGP   Added CPU options.
*   02/26/14   DGP   Added interrupt queuing.
*   07/30/14   DGP   Added termdstop to flush terminals.
*                    SET mod5port properly.
*   05/01/15   DGP   Added ASR 733 emulation support.
*   04/25/16   DGP   Added waitconsole() to wait for console to flush.
*   04/27/16   DGP   Terminate if readconfig fails.
*   11/23/16   DGP   Added IDT:SYM symbolic support.
*   10/03/17   DGP   Fixed cassette and printer display.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <signal.h>
#include <errno.h>
#if defined(UNIX)
#include <unistd.h>
#endif

#include "simdef.h"
#include "screen.h"

#include "simdsk.h"

uint16 pcreg;			/* The program PC */
uint16 curpc;			/* The program PC */
uint16 opcreg;			/* The old program PC */
uint16 statreg;			/* The program status register */
uint16 wpreg;			/* The program Workspace Pointer */
uint16 lights;			/* The panel lights */
uint16 ldspc;			/* The address of the LDS inst */
uint16 lddpc;			/* The address of the LDD inst */
uint32 mpcreg;			/* Mapped PC */
uint32 curmpc;			/* Mapped PC */
uint32 breakaddr;		/* Break point address */
uint32 watchaddr;		/* Watch point address */
uint32 ldsmpc;			/* The mapped address of the LDS inst */
uint32 lddmpc;			/* The mapped address of the LDD inst */
long pancount;			/* Instructions executed for panel */
unsigned long instcount;	/* Instructions executed */
unsigned long mipcount;		/* Instructions executed for MIP calc */
unsigned long tracestart = 0;	/* Trace instruction count start */

int run = FALSE;		/* CPU running */
int idle = FALSE;		/* CPU running IDLE or JXX $ */
int runled = FALSE;		/* CPU running LED */
int idleled = FALSE;		/* CPU idle LED */
int faultled = FALSE;		/* CPU fault LED */
int enablepanel = FALSE;	/* Enable panel */
int showstatus = FALSE;		/* Show CPU status */
int breakflag = FALSE;		/* No breakpoint enabled */
int watchflag = FALSE;		/* No watch point enabled */
int traceenable = FALSE;	/* No trace enabled */
int traceregs = FALSE;		/* Include registers in trace */
int traceit = FALSE;		/* No trace enabled */
int bootfromrom = FALSE;	/* Use ROM to boot */
int mapenabled = FALSE;		/* Memory mapping enabled */
int intsenabled = TRUE;		/* Interrupts enabled */
int trclatch = FALSE;		/* Error Trace latch */
int errindex = 0;		/* Error Trace Memory index */
int trcindex = 0;		/* Error Trace Memory locked index */
int devcnt = 0;			/* Attached device count */
int maxdevcnt = 0;		/* Maximum Attached device count */
int model = 4;			/* Which model */
int deferint = 0;		/* Defer interrupts inst count */
int tracelim[2];		/* Trace memory limits */
int windowlen = 25;		/* Window length */
int viewlen;			/* View length */
int prtviewlen;			/* prtview length */
int prtviewcol;			/* prtview column */
int usertclock = FALSE;		/* Use Real Time clock */
int bkptenabled = FALSE;	/* Breakpoint facility */
int tcenable = FALSE;		/* Test 12ms clock */
int tccount = 0;
int use_LDD_map = FALSE;	/* Using the LDD map */
int use_LDS_map = FALSE;	/* Using the LDS map */
int fakeprefetch = FALSE;	/* Fake a /12 prefetch */
int diaginterrupt = FALSE;	/* /12 diag interrupt */
int forceparity = FALSE;	/* /12 force parity error */
int rom5cru = FALSE;		/* 990/5 rom cru enable */
int cpuoptions = 0;		/* CPU options */

uint16 mapcrudata = 0;		/* Memory map file CRU >1FA0 */
uint16 mapolddata = 0;		/* Old Memory map file CRU >1FA0 */
uint16 curmapindex = 0;		/* Current map index */
uint16 errcrudata = 0;		/* ERR CRU >1FC0 data */
uint16 curinst = 0;		/* Current instruction */
uint16 intrequest = 0;		/* Interrupt request bit map */
uint16 bkptcrudata = 0;		/* Breakpoint cru data */
uint16 xcard1crudata = 0;	/* Expansion Card 1 data */
uint16 xcard2crudata = 0;	/* Expansion Card 2 data */ 

uint32 maplatch = 0;		/* Memory map latch */
uint32 memlen = 56*1024;	/* Length of system memory */
long delayclock = 0;		/* Delay clock interval */

char romfile[MAXDEVPATHLEN+2];	/* ROM file name */
char view[MAXVIEW][MAXVIEWLEN+1];	/* Messages panel */
char prtview[MAXVIEW][MAXVIEWLEN+1];	/* Print messages panel */

uint8 memory[SYSMEMSIZE];	/* The CPU memory */

Device devices[MAXDEVICES];	/* The attached devices */
MapFile mapfile[MAPSIZE];	/* Memory Mapping Files */
t_uint64 wcsmemory[WCSMEMSIZE];	/* Writeable Control Store memory */
InterruptQueue intqueue[16];	/* The queue for shared ints & Exp chassis */

FILE *tracefd = NULL;		/* Trace file fd */
int traceline = 60;		/* Trace file line */
uint32 tracemem[16];		/* Error Trace Memory locked */
uint32 errtrace[16];		/* Error Trace Memory /12 only */

int trcidx = 0;			/* Inst trace index */
int trcopr = 0;			/* Inst trace operand index */
TraceInst trcbuf[TRCBUFMAX];	/* Inst trace buffer */

MUTEX_DEFINE (panel_mutex);

static int dispaddr = 0;	/* Display area address */
static int dispcnt  = 0;	/* Display count */
static int dismaddr = 0;	/* Disassembly address */
static int dismcnt = 0;		/* Disassembly count */
static uint32 displen;
static char input[80];
static char oinput[80];

char *models[] =
{
   "4", "5", "", "", "", "9", "10", "10A", "12"
};

char *emulations[] = 
{
   "", "H19", "V931", "V940", "A733"
};

typedef struct 
{
   char command;
   char *brief;
   char *helpmsg;
} HelpMsg;

static HelpMsg helpmsgs[] =
{
   { 'a', "attach",
     "attach device: a dev[/switches] devaddr intlvl {dev.file | dev.port}" },
   { 'B', "begin",
     "begin trace pc (default 0): B location" },
   { 'b', "boot",
     "boot from device: b[r][n] device" },
   { 'c', "clr b/w",
     "clear breakpoint/watchpoint: c b|w" },
   { 'D', "disasm",
     "disassemble instructions: D[r page] location [count]" },
   { 'd', "display",
     "display memory: d[r page] location [length]" },
   { 'E', "end",
     "End trace pc (default >1FFFFE): E location" },
   { 'g', "go",
     "go (run program)" },
   { 'h', "help",
     "help: h [command|*]" },
   { 'i', "insts",
     "instructions, dumps instruction buffer (last 60 instructions)" },
   { 'k', "break",
     "set breakpoint: k location" },
   { 'L', "lookup",
      "lookup symbol: L [IDT[:SYM]]" },
   { 'l', "load",
     "load program: l[n] [loadaddr] object.file" },
   { 'm', "modify",
     "modify memory: m[r page] location newvalue" },
   { 'N', "nxt dis",
     "disassemble next instructions: N [count]" },
   { 'n', "next",
     "display next memory location: n [length]" },
   { 'o', "CPU opts",
     "set CPU options: o [options]" },
   { 'p', "set pc",
     "set pc: p location" },
   { 'q', "quit",
     "quit the simulator" },
   { 'r', "reset",
     "reset the system" },
   { 's', "step",
     "single step program" },
   { 't', "trace",
     "trace program: t[r] [count] trace.file" },
   { 'W', "watch",
     "set watchpoint: W location" },
   { 'w', "set wp",
     "set workspace pointer: w location" },
   { 'z', "panel",
     "toggle panel mode: z [windowlen]" },
   { 0, NULL, NULL }
};

/***********************************************************************
* sigintr - Process interrupt signal.
***********************************************************************/

void
sigintr (int sig)
{
   int i;

   for (i = 0; i < maxdevcnt; i++)
   {
      Device *dev = &devices[i];

      if ((dev->switches & SWEMUTERM) && (dev->emutype < EMU733) &&
          (dev->infd == stdin))
	 screenposition (TRUE, VDT_NUM_ROWS+1, 1);
   }

   run = FALSE;
   showstatus = TRUE;
   signal (SIGINT, sigintr);
}

/***********************************************************************
* puthelp - Put help on screen.
***********************************************************************/

static void
puthelp (char *bp)
{
   int i, j;
   int pall = FALSE;
   int psum = TRUE;

   if (bp != NULL)
   {
      while (*bp && isspace (*bp)) bp++;
      if (*bp == '*')
         pall = TRUE;
      for (i = j = 0; helpmsgs[i].command; i++)
      {
	 if (pall || (*bp == helpmsgs[i].command))
	 {
	    snprintf (view[j], MAXVIEWLEN, "%c - %s",
		     helpmsgs[i].command, helpmsgs[i].helpmsg);
	    psum = FALSE;
	    if (!pall) break;
	    j++;
	    if (j == viewlen)
	    {
	       if (putview (TRUE, TRUE)) return;
	       j = 0;
	    }
	 }
      }
   }
   if (psum)
   {
      for (j = i = 0; helpmsgs[i].command; i++)
      {
	 char phrase[32];

	 if (i % 6 == 0)
	 {
	    if (i) j++;
	    view[j][0] = 0;
	 }
	 sprintf (phrase, "%c - %-8s ",
		  helpmsgs[i].command, helpmsgs[i].brief);
         strcat (view[j], phrase);
      }
   }
}

/***********************************************************************
* getnum - Get a number from the command.
***********************************************************************/

char *
getnum (char *bp, long *val)
{
   char *cp;
   long i = 0;

   while (*bp && isspace (*bp)) bp++;
   cp = bp;
   if (*bp == HEXSYM || *bp == '0')
   {
      if (*bp == HEXSYM) bp++;
      i = strtol (bp, &cp, 16);
   }
   else if (isdigit (*bp) || *bp == '-' || *bp == '+')
   {
      i = strtol (bp, &cp, 10);
   }
   *val = i;
   return (cp);
}

/***********************************************************************
* getexpr - Get a number from the command expression.
*  num | IDT[+/-num] | IDT:SYM[+/-num]
***********************************************************************/

char *
getexpr (char *bp, long *val)
{
   char *cp, *tp;
   long i, idtnum, symnum;
   int idtloc, symloc;
   char sign;
   char token[MAXSYMLEN+2];

   i = 0;
   while (*bp && isspace (*bp)) bp++;
   cp = bp;
   if (*bp == HEXSYM || *bp == '0')
   {
      if (*bp == HEXSYM) bp++;
      i = strtol (bp, &cp, 16);
   }
   else if (isdigit (*bp) || *bp == '-' || *bp == '+')
   {
      i = strtol (bp, &cp, 10);
   }
   else if (isalpha(*bp) || (*bp == '_') || (*bp == '$'))
   {
      idtloc = symloc = 0;
      idtnum = symnum = 0;
      tp = token;
      while (*cp && (isalnum(*cp) || (*cp == '_') || (*bp == '$')))
	 *tp++ = *cp++;
      *tp = 0;
      if ((idtloc = idtlookup (token)) >= 0)
      {
	 if (*cp == '-' || *cp == '+')
	 {
	    sign = *cp++;
	    cp = getnum (cp, &idtnum);
	    if (sign == '-') idtnum = -idtnum;
	 }
	 if (*cp == ':')
	 {
	    cp++;
	    tp = token;
	    while (*cp && (isalnum(*cp) || (*cp == '_') || (*bp == '$')))
	       *tp++ = *cp++;
	    *tp = 0;
	    if ((symloc = symlookup (idtloc, token)) >= 0)
	    {
	       if (*cp == '-' || *cp == '+')
	       {
		  sign = *cp++;
		  cp = getnum (cp, &symnum);
		  if (sign == '-') symnum = -symnum;
	       }
	    }
	    i = symloc + symnum;
	 }
	 else
	 {
	    i = idtloc + idtnum;
	 }
      }
   }
   *val = i;
   return (cp);
}

/***********************************************************************
* initpanel - Initalize panel mutex.
***********************************************************************/

int
initpanel (void)
{
   if (MUTEX_INIT (panel_mutex))
   {
      PERROR ("initpanel: Can't initialize mutex");
      return (-1);
   }
   return (0);
}
/***********************************************************************
* panel - Put up the panel.
***********************************************************************/

void
panel (void)
{
   char c1, c2;

   c1 = DISPMEM >> 8;
   c2 = DISPMEM & 0xFF;
   if (!isprint (c1)) c1 = '.';
   if (!isprint (c2)) c2 = '.';

   if (enablepanel)
   {
      MUTEX_LOCK (panel_mutex);

      screenposition (enablepanel, TOPLINE, 1);
      clearline (enablepanel);
      printf ("%s %s      /%s CPU %dK",
	      INTRO, VERSION, models[model-4], displen/1024);

      screenposition (enablepanel, REGISTERLINE, 1);
      printf (" PC  %04X  WP  %04X  ST  %04X  FP  %04X",
	       pcreg, wpreg, statreg, lights);

      screenposition (enablepanel, DISPLINE, 1);
      printf (" DISP  %06X %04X %c%c  %-5.5s %-5.5s %-5.5s",
	       dispaddr, DISPMEM, c1,c2,
	       runled ? "RUN" : " ",
	       idleled ? "IDLE" : " ",
	       faultled ? "FAULT" : " ");

      screenposition (enablepanel, BREAKLINE, 1);
      printf (" BREAK %06X  ", breakaddr);
      printf (" WATCH %06X  ", watchaddr);

#ifdef SHOW_OLDPC
      printf (" OPC %04X", opcreg);
#endif
#ifdef SHOW_MIPS
      printf ("  MIPS %2.1d.%2.2d",
	      mipcount / 1000000, (mipcount % 1000000) / 10000);
#endif
#ifdef SHOW_INTREQ
      printf (" INTREQ %04X", intrequest);
#endif

      screenposition (enablepanel, WORKSPACELINE, 1);
      printf ("Workspace:");
      screenposition (enablepanel, WORKSPACELINE+1, 1);
      printf (" R00 %04X  R01 %04X  R02 %04X  R03 %04X", R0, R1, R2, R3);
      screenposition (enablepanel, WORKSPACELINE+2, 1);
      printf (" R04 %04X  R05 %04X  R06 %04X  R07 %04X", R4, R5, R6, R7);
      screenposition (enablepanel, WORKSPACELINE+3, 1);
      printf (" R08 %04X  R09 %04X  R10 %04X  R11 %04X", R8, R9, R10, R11);
      screenposition (enablepanel, WORKSPACELINE+4, 1);
      printf (" R12 %04X  R13 %04X  R14 %04X  R15 %04X", R12, R13, R14, R15);
      screenposition (enablepanel, OUTPUTLINE+prtviewlen-1, prtviewcol);
      fflush (stdout);
      pancount = FALSE;

      MUTEX_UNLOCK (panel_mutex);
   }
   else if (showstatus)
   {
      printf ("\nPC %04X  WP %04X  ST %04X  FP %04X\n",
	       pcreg, wpreg, statreg, lights);
      printf ("BREAK %06X  WATCH %06X\n", breakaddr, watchaddr);
      printf ("Workspace:\n");
      printf (" R00 %04X  R01 %04X  R02 %04X  R03 %04X\n", R0, R1, R2, R3);
      printf (" R04 %04X  R05 %04X  R06 %04X  R07 %04X\n", R4, R5, R6, R7);
      printf (" R08 %04X  R09 %04X  R10 %04X  R11 %04X\n", R8, R9, R10, R11);
      printf (" R12 %04X  R13 %04X  R14 %04X  R15 %04X\n", R12, R13, R14, R15);
   }

}

/***********************************************************************
* romtwcs - Copy rom to WCS.
***********************************************************************/

void
romtowcs ()
{
   uint32 lp;
   int i, j;
   int wcsadd;

   if (model < 12)
      return;

#if defined(DEBUG_LCS)
   fprintf (stderr, "romtowcs: \n");
#endif

   lp = ROMSTART + TPCSSTART + (4 * 1024);
   wcsadd = 0;
   for (j = 4; j < 8; j++)
   {
      for (i = 0; i < 1024; i += 8)
      {
         t_uint64 lv;

	 lv = ((t_uint64)GETMEM0 (lp) << 48) |
	      ((t_uint64)GETMEM0 (lp+2) << 32) |
	      ((t_uint64)GETMEM0 (lp+4) << 16) |
	      (t_uint64)GETMEM0 (lp+6);
#if defined(DEBUG_LCS)
	 fprintf (stderr,"   wcsmemory[%04X] = >%016llX\n",
		  wcsadd + 0x800, lv);
#endif
	 wcsmemory[wcsadd++] = lv;
         lp += 8;
      }
   }
}

/***********************************************************************
* clearall - Clear out system.
***********************************************************************/

static int 
clearall ()
{
   FILE *rom;
   int i, j;
   int rompage;
   uint32 lp;
   uint16 val;
   char filename[MAXDEVPATHLEN+2];

   /*
   ** Clear registers
   */

   pcreg = 0;
   statreg = 0;
   wpreg = 0;
   lights = 0;

   /*
   ** Clear memory
   */

   memset (memory, 0, sizeof (memory));
   memset (wcsmemory, 0, sizeof (wcsmemory));

   /*
   ** Clear the interrupt queue
   */

   memset (intqueue, 0, sizeof (intqueue));
   
   /*
   ** Load ROM
   */

   if (romfile[0])
   {
      strcpy (filename, romfile);
      rom = fopen (filename, "rb");
   }
   else
   {
      int rommod = model;

      if (model == 11) rommod = 10;
      sprintf (filename, "ti990-%s.rom", models[rommod-4]);
      rom = fopen (filename, "rb");
#ifdef UNIX
      if (rom == NULL)
      {
         sprintf (filename, "/usr/local/share/sim990/ti990-%s.rom",
		  models[rommod-4]);
	 rom = fopen (filename, "rb");
      }
#endif
   }

   if (rom == NULL)
   {
      sprintf (view[0], "clearall: ROM open failed: %s", strerror (ERRNO));
      snprintf (view[1], MAXVIEWLEN, "filename: %s", filename);
      return (-1);
   }

   lp = ROMSTART;
   if (model == 5 || model > 9) 
      lp += TPCSSTART;

   rompage = (model == 12) ? 8 :
	     (model == 5) ? 2 :
	     1;

   for (j = 0; j < rompage; j++)
   {
      for (i = 0; i < 1024; i+=2)
      {
	 int c;

	 if ((c = fgetc (rom)) == EOF)
	 {
	    sprintf (view[0], "clearall: ROM read1 failed: %s",
		     strerror (ERRNO));
	    snprintf (view[1], MAXVIEWLEN, "filename: %s", filename);
	    return (-1);
	 }
	 val = (c & 0xFF) << 8;
	 if ((c = fgetc (rom)) == EOF)
	 {
	    sprintf (view[0], "clearall: ROM read2 failed: %s",
		     strerror (ERRNO));
	    snprintf (view[1], MAXVIEWLEN, "filename: %s", filename);
	    return (-1);
	 }
	 val = val | (c & 0xFF);
	 PUTMEM0 (lp, val);
	 lp += 2;
      }
   }
   fclose (rom);
   fakeprefetch = FALSE;
   romtowcs ();

   instcount = 0;

   /*
   ** Initialize map registers
   */

   memset (mapfile, 0, sizeof(mapfile));

   maplatch = 0;
   errcrudata &= 0xF;
   mapcrudata = 0;
   bkptcrudata = 0;
   mapenabled = FALSE;
   bkptenabled = FALSE;
   rom5cru = FALSE;
   tcenable = FALSE;
   intsenabled = TRUE;
   diaginterrupt = FALSE;
   forceparity = FALSE;

   /*
   ** Clear out error trace memory
   */

   memset (tracemem, 0, sizeof(tracemem));
   memset (errtrace, 0, sizeof(errtrace));

   errindex = 0;
   trcindex = 0;

   /*
   ** Clear out display panels.
   */

   memset (prtview, 0, sizeof(prtview));
   memset (view, 0, sizeof(view));

   /*
   ** Clear out instruction trace memory
   */

   trcidx = 0;
   memset (trcbuf, 0, sizeof(trcbuf));
   for (i = 0; i < TRCBUFMAX; i++)
   {
      trcbuf[i].pc = -1;
   }

   return (0);
}

/***********************************************************************
* printinst - Print an instruction with disassembly.
***********************************************************************/

void
printinst (char *outbuf, int index)
{
   char opr[12];

   sprintf (outbuf, "%04X %06X %04X %04X:  %04X ", 
	    trcbuf[index].pc, trcbuf[index].mpc, trcbuf[index].wp,
	    trcbuf[index].st, trcbuf[index].inst);
   if (trcbuf[index].ops[0] != -1)
   {
      sprintf (opr, "%04X ", trcbuf[index].ops[0]);
      strcat (outbuf, opr);
   }
   else
      strcat (outbuf, "     ");
   if (trcbuf[index].ops[1] != -1)
   {
      sprintf (opr, "%04X ", trcbuf[index].ops[1]);
      strcat (outbuf, opr);
   }
   else
      strcat (outbuf, "     ");
   if (trcbuf[index].ops[2] != -1)
   {
      sprintf (opr, "%04X ", trcbuf[index].ops[2]);
      strcat (outbuf, opr);
   }
   else
      strcat (outbuf, "     ");
   strcat (outbuf, "  ");
   disinst (&trcbuf[index]);
   strcat (outbuf, trcbuf[index].dis);
}

/***********************************************************************
* printsingle - Print a single instruction with disassembly.
***********************************************************************/

static void
printsingle (char *outbuf)
{
   int daddr;
   int topaddr;
   TraceInst inst;
   char opr[12];

   switch (model)
   {
   case 4:
   case 9:
      dispaddr &= 0xFFFF;
      daddr = dismaddr;
      topaddr = 0xFFFE;
      break;
   case 5:
      daddr = dismaddr;
      topaddr = 0x2003FE;
      break;
   case 12:
      daddr = dismaddr;
      topaddr = 0x201BFE;
      break;
   default:
      daddr = dismaddr;
      topaddr = 0x1FFFFE;
   }

   inst.pc = daddr;
   inst.mpc = daddr;
   inst.wp = 0;
   inst.st = 0;
   inst.inst = GETMEM0 (dismaddr);
   inst.ops[0] = GETMEM0 (dismaddr + 2);
   inst.ops[1] = GETMEM0 (dismaddr + 4);
   inst.ops[2] = GETMEM0 (dismaddr + 6);
   disinst (&inst);
   sprintf (outbuf, "%06X   %04X ", daddr, inst.inst);
   dismaddr += 2;
   if (inst.ops[0] != -1)
   {
      sprintf (opr, "%04X ", inst.ops[0]);
      strcat (outbuf, opr);
      dismaddr += 2;
   }
   else
      strcat (outbuf, "     ");
   if (inst.ops[1] != -1)
   {
      sprintf (opr, "%04X ", inst.ops[1]);
      strcat (outbuf, opr);
      dismaddr += 2;
   }
   else
      strcat (outbuf, "     ");
   if (inst.ops[2] != -1)
   {
      sprintf (opr, "%04X ", inst.ops[2]);
      strcat (outbuf, opr);
      dismaddr += 2;
   }
   else
      strcat (outbuf, "     ");
   strcat (outbuf, inst.dis);
   if (dismaddr > topaddr)
      dismaddr = 0;
}

/***********************************************************************
* dumpsingle - Dump a single line.
***********************************************************************/

static void
dumpsingle (char *outbuf)
{
   int i, j;
   int daddr;
   int topaddr;
   char opr[8];
   char chd[32];

   switch (model)
   {
   case 4:
   case 9:
      dispaddr &= 0xFFFF;
      daddr = dispaddr;
      topaddr = 0xFFFE;
      break;
   case 5:
      daddr = dispaddr;
      topaddr = 0x2003FE;
      break;
   case 12:
      daddr = dispaddr;
      topaddr = 0x201BFE;
      break;
   default:
      daddr = dispaddr;
      topaddr = 0x1FFFFE;
   }

   sprintf (outbuf, "%06X   ", daddr);
   memset (chd, ' ', sizeof (chd));
   for (j = 2, i = 0; i < 8; i++, j += 3)
   {
      uint16 wd = GETMEM0 (dispaddr);
      uint8 ch;

      sprintf (opr, "%04X ", wd);
      ch = (wd >> 8) & 0xFF;
      if (ch >= 0x20 && ch < 0x7F) chd[j] = ch;
      else chd[j] = '.';
      ch = wd & 0xFF;
      if (ch >= 0x20 && ch < 0x7F) chd[j+1] = ch;
      else chd[j+1] = '.';
      strcat (outbuf, opr);
      dispaddr += 2;
      if (dispaddr > topaddr)
	 dispaddr = 0;
   }
   chd[j] = 0;
   strcat (outbuf, chd);
}

/***********************************************************************
* romwrite - Write ROM image.
***********************************************************************/

static char *
romwrite (char *bp)
{
   FILE *rom = NULL;
   int i, j;
   int rompage;
   uint32 lp;
   uint16 val;

   while (*bp && isspace (*bp)) bp++;
   if (!*bp)
   {
      sprintf (view[0], "romwrite: No file name");
      return (bp);
   }
   if ((rom = fopen (bp, "wb")) == NULL)
   {
      sprintf (view[0], "romwrite: ROM open failed: %s", strerror (ERRNO));
      snprintf (view[1], MAXVIEWLEN, "filename: %s", bp);
      return (bp);
   }

   lp = ROMSTART;
   if (model == 5 || model > 9) 
      lp += TPCSSTART;

   rompage = (model == 12) ? 8 :
	     (model == 5) ? 2 :
	     1;

   for (j = 0; j < rompage; j++)
   {
      for (i = 0; i < 1024; i += 2)
      {
	 int c;

	 val = GETMEM0 (lp);
	 c = (val >> 8) & 0xFF;
	 if (fputc (c, rom) == EOF)
	 {
	    sprintf (view[0], "romwrite: ROM write1 failed: %s",
		     strerror (ERRNO));
	    snprintf (view[1], MAXVIEWLEN, "filename: %s", bp);
	    return (bp);
	 }
	 c = val & 0xFF;
	 if (fputc (c, rom) == EOF)
	 {
	    sprintf (view[0], "romwrite: ROM write2 failed: %s",
		     strerror (ERRNO));
	    snprintf (view[1], MAXVIEWLEN, "filename: %s", bp);
	    return (bp);
	 }
	 lp += 2;
      }
   }
   fclose (rom);
   return (bp);
}

/***********************************************************************
* chkrom - Check for ROM access mode.
***********************************************************************/

static char *
chkrom (char *bp, int *romorg, int *rompage)
{
   long val;
   int pagenum;

   pagenum = (model == 12) ? 7 :
	     (model == 5) ? 1 :
	     0;

   *romorg = 0;
   *rompage = 0;
   if (*bp == 'r')
   {
      bp++;
      if (model == 5 || model > 9) 
	 *romorg = TPCSSTART;
      bp = getnum (bp, &val);
      *rompage = val;
      if (*rompage > pagenum)
      {
	 sprintf (view[0], "ROM page %d too big for /%s cpu\n",
		  *rompage, models[model-4]);
         *rompage = -1;
      }
   }
   return (bp);
}

/***********************************************************************
* processoptions - Process CPU options
***********************************************************************/

static void
processoptions (char *bp, int interactive)
{
   while (*bp && isspace (*bp)) bp++;
   for (; *bp; bp++)
   {
      switch (*bp)
      {
      case ',':
         break;

      case 's':
	 if (model < 12)
	 {
	    sprintf (view[0], "Segment protection not supported on /%s CPU",
		     models[model-4]);
	    return;
	 }
	 if (cpuoptions & OPT_NOSEGPROT)
	    cpuoptions &= ~OPT_NOSEGPROT;
	 else
	    cpuoptions |= OPT_NOSEGPROT;
         break;

      case 't':
	 if (model == 4 || model == 9)
	 {
	    sprintf (view[0], "TILINE not supported on /%s CPU",
		     models[model-4]);
	    return;
	 }
	 if (cpuoptions & OPT_NOTILINE)
	    cpuoptions &= ~OPT_NOTILINE;
	 else
	    cpuoptions |= OPT_NOTILINE;
         break;

      default:
	 sprintf (view[0], "Invalid CPU option: %c", *bp);
      }
   }

   if (interactive)
   {
      sprintf (view[0], "Segment protection: %s",
	       (cpuoptions & OPT_NOSEGPROT) ? "OFF" : "ON");
      sprintf (view[1], "TILINE: %s",
	       (cpuoptions & OPT_NOTILINE) ? "OFF" : "ON");
   }
}

/***********************************************************************
* waitconsole - Wait for console queue to empty
***********************************************************************/

static void
waitconsole (void)
{
   TermRing *ring;

   ring = &devices[0].info.terminfo.outring;
   while (!termdringempty (&devices[0], ring)) smallsleep (10);
}

/***********************************************************************
* docommand - Do user commands.
***********************************************************************/

static int
docommand (int interactive)
{
   FILE *fd;
   char *bp;
   long val;
   int action;
   int i, j, k;
   int norun;
   int romorg, rompage;
   char fn[256];

   bp = input;
   action = NULL_ACTION;
   showstatus = TRUE;
   while (*bp && isspace (*bp)) bp++;
#ifdef DEBUGCMD
   printf ("docmommand: entered: input = '%s'\n", input);
#endif

   switch (*bp++)
   {
   case '\0':
   case '#': /* Comment */
      break;

   case 'C': /* delay Clock */
      showstatus = FALSE;
      bp = getnum (bp, &delayclock);
      action = NULL_ACTION;
      oinput[0] = '\0';
      break;

   case 'a': /* Attach device */
      showstatus = FALSE;
      while (*bp && isspace (*bp)) bp++;
      if (*bp)
	 attachdev (bp);
      if (interactive && view[0][0] == '\0')
      {
	 snprintf (view[0], MAXVIEWLEN, 
		   "DEV      SW    ADDR INT CHAS POS MUX  FILE");
	 for (i = 0, j = 1; i < devcnt; i++, j++)
	 {
	    Device *dev = &devices[i];

	    if (dev->infd)
	    {
	       char chassis[4];
	       char pos[4];
	       char mux[4];
	       char temp[MAXDEVPATHLEN+2];

	       if ((dev->type == DISK) && (dev->infd != DEV_NULL))
	          sprintf (temp, "%s:%s", dev->info.dskinfo.model, dev->file);
	       else if (dev->switches & SWEMUTERM)
	          sprintf (temp, "%s:%s", emulations[dev->emutype], dev->file);
	       else if (dev->type == VDT911)
	          sprintf (temp, "%s:%s", "V911", dev->file);
	       else
	          strcpy (temp, dev->file);

	       chassis[0] = 0;
	       pos[0] = 0;
	       if (dev->xcardsel)
	       {
	          sprintf (chassis, "%d",
		     ((dev->xcarddat >> 7) & 0xF) + dev->xcardsel);
	          sprintf (pos, "%d", (dev->xcarddat >> 2) & 0x1F);
	       }

	       mux[0] = 0;
	       if (dev->type == SERIALCI403 || dev->type == PRINTCI403)
	       {
	          sprintf (mux, "%d",
			   dev->info.terminfo.controller.ci403info.unit);
	       }

	       if (j == viewlen)
	       {
	          if (putview (interactive, TRUE)) goto DISP_DONE;
		  j = 0;
	       }
	       snprintf (view[j], MAXVIEWLEN,
			 "%-8s %04X  %04X  %2d  %3s %3s %3s  %s",
			 dev->name, dev->switches, dev->devaddr, dev->intlvl,
			 chassis, pos, mux, temp);

	       if (dev->emutype == EMU733)
	       {
		  char *fp;
		  ASRInfo *asrinfo;
		  int k;

		  asrinfo = &dev->info.terminfo.terminal.asrinfo;
		  for (k = 0; k < 2; k++)
		  {
		     j++;
		     if (j == viewlen)
		     {
			if (putview (interactive, TRUE)) goto DISP_DONE;
			j = 0;
		     }
		     sprintf (temp, "CS%d", asrinfo->baseunit + k);
		     if (k == 0)
			fp = asrinfo->cs1file;
		     else
			fp = asrinfo->cs2file;
		     snprintf (view[j], MAXVIEWLEN,
			      "%-8s %4s  %04X  %2d  %3s %3s %3s  %s",
			      temp, "", dev->devaddr, dev->intlvl, chassis, pos,
			      mux, fp);
		  }
	       }
	       else if ((dev->switches & (SWEMUTERM | SWPRINTER)) ==
			(SWEMUTERM | SWPRINTER))
	       {
	          j++;
		  if (j == viewlen)
		  {
		     if (putview (interactive, TRUE)) goto DISP_DONE;
		     j = 0;
		  }
		  sprintf (temp, "%sP", dev->name);
		  snprintf (view[j], MAXVIEWLEN,
			   "%-8s %4s  %04X  %2d  %3s %3s %3s  %s",
			   temp, "", dev->devaddr, dev->intlvl, chassis, pos,
			   mux, dev->info.terminfo.terminal.vdtinfo.printfile);
	       }
	    }
	 }
      }
   DISP_DONE:
      oinput[0] = '\0';
      break;

   case 'b': /* Boot */
      showstatus = FALSE;
      norun = FALSE;
      bootfromrom = FALSE;
      for (; *bp && !isspace (*bp); bp++)
      {
	 if (*bp == 'n') /* No start option */
	    norun = TRUE;
         else if (*bp == 'r') /* Use ROM to boot */
	    bootfromrom = TRUE;
      }
      if (!*bp)
      {
	 strcpy (view[0], "You must specify a boot device");
      }
      else if (bootdev (bp) == 0)
      {
	 if (!norun && run)
	 {
	    action = RUN_ACTION;
	    runprog ();
	    waitconsole();
#ifdef DEBUGCMD
	    printf ("docmommand: runprog returns\n");
#endif
	 }
      }
      oinput[0] = '\0';
      break;

   case 'B': /* trace limit Begin pc */
      showstatus = FALSE;
      bp = getexpr (bp, &val);
      if (view[0][0])
      {
	 oinput[0] = '\0';
	 break;
      }
      tracelim[0] = val;
      oinput[0] = '\0';
      break;

   case 'c': /* Clear breakpoint/watchpoint */
      while (*bp && isspace (*bp)) bp++;
      if (*bp == 'b')
	 breakflag = FALSE;
      else if (*bp == 'w')
	 watchflag = FALSE;
      oinput[0] = '\0';
      break;

   case 'd': /* Display memory */
      showstatus = FALSE;
      bp = chkrom (bp, &romorg, &rompage);
      bp = getexpr (bp, &val);
      if (view[0][0])
      {
	 oinput[0] = '\0';
	 break;
      }
      if ((rompage >= 0) && (val < SYSMEMSIZE))
      {
	 val &= 0x1FFFFE;
	 dispaddr = romorg + val + (rompage * 1024);
	 if (dispaddr < SYSMEMSIZE)
	 {
	    bp = getnum (bp, &val);
	    if (val > 16)
	    {
	       dispcnt = val;
	       if (dispcnt % 16) dispcnt += 16;
	    }
	    else dispcnt = 16;
	    action = DISP_ACTION;

	    for (i = j = 0; i < (dispcnt/16); i++, j++)
	    {
	       if (j == viewlen)
	       {
	          if (putview (interactive, TRUE)) break;
		  j = 0;
	       }
	       dumpsingle (view[j]);
	    }
	 }
	 else
	 {
	    dispaddr = 0;
	 }
      }
      oinput[0] = '\0';
      break;

   case 'D': /* Disassemble instruction block */
      showstatus = FALSE;
      bp = chkrom (bp, &romorg, &rompage);
      bp = getexpr (bp, &val);
      if (view[0][0])
      {
	 oinput[0] = '\0';
	 break;
      }
      if ((rompage >= 0) && (val < SYSMEMSIZE))
      {
	 val &= 0x1FFFFE;
	 dismaddr = romorg + val + (rompage * 1024);
	 if (dismaddr < SYSMEMSIZE)
	 {
	    bp = getnum (bp, &val);
	    if (val > 0) dismcnt = val;
	    else dismcnt = 1;
	    action = DISP_ACTION;
	    for (i = j = 0; i < dismcnt; i++, j++)
	    {
	       if (j == viewlen)
	       {
	          if (putview (interactive, TRUE)) break;
		  j = 0;
	       }
	       printsingle (view[j]);
	    }
	 }
	 else
	 {
	    dismaddr = 0;
	 }
      }
      oinput[0] = '\0';
      break;

   case 'E': /* trace limit End pc */
      showstatus = FALSE;
      bp = getexpr (bp, &val);
      if (view[0][0])
      {
	 oinput[0] = '\0';
	 break;
      }
      tracelim[1] = val;
      oinput[0] = '\0';
      break;

   case 'f': /* Display memory to file */
      showstatus = FALSE;
      if (*bp == '=')
      {
	 bp++;
	 for (i = 0; *bp && !isspace(*bp); i++)
	    fn[i] = *bp++;
         fn[i] = 0;
	 fd = fopen (fn, "w");
      }
      else
      {
         strcpy (view[0], "File name required\n");
	 break;
      }
      bp = getexpr (bp, &val);
      if (view[0][0])
      {
	 oinput[0] = '\0';
	 break;
      }
      if (val < SYSMEMSIZE)
      {
	 val &= 0x1FFFFE;
	 dispaddr = val;
	 if (dispaddr < SYSMEMSIZE)
	 {
	    bp = getnum (bp, &val);
	    if (val > 16)
	    {
	       dispcnt = val;
	       if (dispcnt % 16) dispcnt += 16;
	    }
	    else dispcnt = 16;
	    action = DISP_ACTION;

	    for (i = j = 0; i < (dispcnt/16); i++, j++)
	    {
	       dumpsingle (fn);
	       fprintf (fd, "%s\n", fn);
	    }
	 }
	 else
	 {
	    dispaddr = 0;
	 }
      }
      if (fd) fclose(fd);
      oinput[0] = '\0';
      break;

   case 'g': /* Go - run program */
      if (enablepanel)
         prtpanel (prtviewcol, TRUE);
      action = RUN_ACTION;
      run = TRUE;
      runprog ();
      waitconsole();
#ifdef DEBUGCMD
      printf ("docmommand: runprog returns\n");
#endif
      break;

   case 'h': /* Help */
      showstatus = FALSE;
      puthelp (bp);
      action = HELP_ACTION;
      oinput[0] = '\0';
      break;

   case 'i': /* dump Instruction trace buffer */
      showstatus = FALSE;
      sprintf (view[0],
	       " PC   MPC    WP   ST         INST                    CODE");
      for (k = 0, j = 1, i = trcidx; k < TRCBUFMAX; i++, k++)
      {
         if (i >= TRCBUFMAX) i = 0;
	 if (trcbuf[i].pc != -1)
	 {
	    printinst (view[j], i);
	    j++;
	 }
      }
      break;

   case 'I': /* set Interrupt level */
      bp = getnum (bp, &val);
      intrequest = 1 << (val & 0xF);
      oinput[0] = '\0';
      break;

   case 'k': /* set breaKpoint */
      bp = getexpr (bp, &val);
      if (view[0][0])
      {
	 showstatus = FALSE;
	 oinput[0] = '\0';
	 break;
      }
      if (val < SYSMEMSIZE)
      {
	 breakaddr = val;
	 breakflag = TRUE;
      }
      oinput[0] = '\0';
      break;

   case 'l': /* Load - use simloader */
      if (*bp == 'n') /* No start option */
      {
	 bp++;
	 norun = TRUE;
      }
      else norun = FALSE;
      bp = getnum (bp, &val);
      if (val < SYSMEMSIZE)
      {
	 action = val;
	 if (simloader (bp, action) == 0)
	 {
	    if (!norun && run)
	    {
	       action = RUN_ACTION;
	       runprog ();
	       waitconsole();
#ifdef DEBUGCMD
	       printf ("docmommand: runprog returns\n");
#endif
	    }
	    else action = LOAD_ACTION;
	 }
      }
      run = FALSE;
      oinput[0] = '\0';
      break;

   case 'L': /* Lookup symbol */
      showstatus = FALSE;
      dumpsyms (bp);
      oinput[0] = '\0';
      break;

   case 'm': /* Modify memory */
      showstatus = FALSE;
      bp = chkrom (bp, &romorg, &rompage);
      bp = getexpr (bp, &val);
      if (view[0][0])
      {
	 oinput[0] = '\0';
	 break;
      }
      if ((rompage >= 0) && (val < SYSMEMSIZE))
      {
	 dispaddr = romorg + val + (rompage * 1024);
	 if (dispaddr < SYSMEMSIZE)
	 {
	    bp = getnum (bp, &val);
	    PUTMEM0 (dispaddr, val & 0xFFFF);
	    action = DISP_ACTION;
	    if (!enablepanel)
	       printf (" (%06X) = %04X\n", dispaddr, DISPMEM);
	    oinput[0] = '\0';
	 }
	 else
	 {
	    dispaddr = 0;
	 }
      }
      break;

   case 'N': /* Disassemble next instruction block */
      showstatus = FALSE;
      bp = getnum (bp, &val);
      if (val > 0) dismcnt = val;

      for (i = j = 0; i < dismcnt; i++, j++)
      {
	 if (j == viewlen)
	 {
	    if (putview (interactive, TRUE)) break;
	    j = 0;
	 }
	 printsingle (view[j]);
      }
      break;

   case 'n': /* display Next memory location */
      showstatus = FALSE;
      bp = getnum (bp, &val);
      if (val > 16)
      {
	 dispcnt = val;
	 if (dispcnt % 16) dispcnt += 16;
      }
      else if (val > 0) dispcnt = 16;
      action = DISP_ACTION;

      for (i = j = 0; i < (dispcnt/16); i++, j++)
      {
	 if (j == viewlen)
	 {
	    if (putview (interactive, TRUE)) break;
	    j = 0;
	 }
	 dumpsingle (view[j]);
      }
      break;

   case 'o': /* set CPU Options */
      showstatus = FALSE;
      processoptions (bp, interactive);
      oinput[0] = '\0';
      break;

   case 'p': /* set PC */
      bp = getexpr (bp, &val);
      if (view[0][0])
      {
	 showstatus = FALSE;
	 oinput[0] = '\0';
	 break;
      }
      pcreg = val & 0xFFFF;
      action = PC_ACTION;
      oinput[0] = '\0';
      break;

   case 'q': /* Quit */
      action = QUIT_ACTION;
      break;

   case 'r': /* Reset system */
      showstatus = FALSE;
      clearall ();
      action = CLEAR_ACTION;
      oinput[0] = '\0';
      break;

   case 'R': /* Rom write */
      showstatus = FALSE;
      bp = romwrite (bp);
      oinput[0] = '\0';
      action = CLEAR_ACTION;
      break;

   case 's': /* Step program */
      action = STEP_ACTION;
      run = FALSE;
      stepprog ();
      printinst (view[0], trcidx-1);
      break;

   case 't': /* Trace program */
      showstatus = FALSE;
      traceenable = traceenable ? FALSE : TRUE;
      traceregs = FALSE;
      if (*bp == 'r') /* Include registers in trace */
      {
         bp++;
	 traceregs = TRUE;
      }
      if (traceenable)
      {
	 traceline = 60;
	 bp = getnum (bp, (long *)&tracestart);
	 while (*bp && isspace (*bp)) bp++;
	 if ((tracefd = fopen (bp, "w")) == NULL)
	 {
	    traceenable = FALSE;
	    sprintf (view[0], "Can't open trace file: %s",
		     strerror (ERRNO));
	    sprintf (view[1], "filename: %s", bp);
	 }
      }
      else
      {
	 fclose (tracefd);
	 tracefd = NULL;
      }
      oinput[0] = '\0';
      break;

   case 'w': /* set WP */
      bp = getexpr (bp, &val);
      if (view[0][0])
      {
	 showstatus = FALSE;
	 oinput[0] = '\0';
	 break;
      }
      wpreg = val & 0xFFFF;
      action = WP_ACTION;
      oinput[0] = '\0';
      break;

   case 'W': /* set Watchpoint */
      bp = getexpr (bp, &val);
      if (view[0][0])
      {
	 showstatus = FALSE;
	 oinput[0] = '\0';
	 break;
      }
      if (val < SYSMEMSIZE)
      {
	 watchaddr = val;
	 watchflag = TRUE;
      }
      oinput[0] = '\0';
      break;

   case 'z': /* it's Zee panel */
      bp = getnum (bp, &val);
      if (val > 0)
      {
	 windowlen = val;
	 viewlen = windowlen - OUTPUTLINE - 3;
	 prtviewlen = windowlen - OUTPUTLINE - 1;
      }
      if (!enablepanel)
      {
	 enablepanel = TRUE;
	 if (dismcnt > MAXVIEW) dismcnt = MAXVIEW;
	 clearscreen (enablepanel);
	 panel ();
      }
      else
      {
         screenposition (enablepanel, windowlen, 1);
	 enablepanel = FALSE;
      }
      oinput[0] = '\0';
      break;

   default:
      showstatus = FALSE;
      puthelp (NULL);
      oinput[0] = '\0';
   }
   fflush (stdout);
   return (action);
}

/***********************************************************************
* readconfig - Read config file.
***********************************************************************/

static int
readconfig (char *config)
{
   FILE *fd;
   int status = 0;

   if ((fd = fopen (config, "r")) == NULL)
   {
      sprintf (view[0], "readconfig: open failed: %s", strerror (ERRNO));
      snprintf (view[1], MAXVIEWLEN, "filename: %s", config);
      return (-1);
   }
   while (fgets (input, sizeof (input), fd))
   {
      char *cp;

      *(strchr (input, '\n')) = '\0';
      if ((cp = strchr (input, '\r')) != NULL)
         *cp = '\0';
      docommand (FALSE);
      if (view[0][0] != '\0')
      {
	 putview (FALSE, FALSE);
	 status = -1;
	 break;
      }
   }
   fclose (fd);
   return (status);
}

/***********************************************************************
* main - Main Procedure.
***********************************************************************/

int
main (int argc, char **argv)
{
   Device *dev;
   char *bp;
   char *pp;
   char *objfile = NULL;
   char *config = NULL;
   char *termdev;
   int action;
   int i;
   int termaddr, extaddr, ctladdr, termint;
   int termtype;
   int mod5port;

   /*
   ** Process args
   */

   termaddr = TERMINAL;
   extaddr = 0;
   ctladdr = 0;
   mod5port = FALSE;
   termint = TERMINT;
   termtype = CONEIA;
   termdev = "CE";
   cpuoptions = OPT_NOTILINE | OPT_NOSEGPROT;

   for (i = 1; i < argc; i++)
   {
      bp = argv[i];

      if (*bp == '-')
      {
         for (bp++; *bp; bp++) switch (*bp)
         {
	 case 'C': /* delay Clock interval */
	    i++;
	    if (i >= argc) goto USAGE;
	    delayclock = atoi (argv[i]);
	    break;

	 case 'c': /* Config file spec */
	    i++;
	    if (i >= argc) goto USAGE;
	    config = argv[i];
	    break;

	 case 'm': /* CPU model */
	    i++;
	    if (i >= argc) goto USAGE;
	    model = 0;
	    for (pp = argv[i]; *pp; pp++)
	    {
	       if (isdigit (*pp))
	       {
		  model = (model * 10) + (*pp - '0');
	       }
	       else
	       {
	          if (*pp == 'a' || *pp == 'A')
		  {
		     if (model == 10) model = 11;
		     else goto BAD_MODEL;
		     break;
		  }
		  else
		  {
		  BAD_MODEL:
		     fprintf (stderr, 
			      "Invalid model specification %s\n", argv[i]);
		     exit (ABORT);
		  }
	       }
	    }
	    if (!(model ==  4 || model ==  5 || model == 9 ||
		  model == 10 || model == 11 || model == 12))
	    {
	       fprintf (stderr, "Unsupported CPU model: %s\n", argv[i]);
	       goto USAGE;
	    }

	    /* Set default memory size based on model */

	    if (model >= 10 && memlen == 56*1024) /* new default for /10-/12 */
	       memlen = 256*1024;
	    if (model == 5)
	       memlen = 64*1024;

	    /* Set default CPU options */

	    cpuoptions = 0;
	    if (TRUE) /* (model != 12) */
	       cpuoptions |= OPT_NOSEGPROT;
	    if (model == 4 || model == 9)
	       cpuoptions |= OPT_NOTILINE;

	    /* 990/5 and 990/10A use 9902 for default console */

	    if (model == 5 || model == 11)
	    {
	       termaddr = TERM9902;
	       extaddr = termaddr + 0x20;
	       if (model == 11)
		  ctladdr = termaddr + 0x40;
	       else
	          mod5port = TRUE;
	       termint = INT9902;
	       termtype = CON9902;
	       termdev = "C2";
	    }

	    /* Set error cpu ID bits */

	    errcrudata = 0;
	    if (model == 11)
	       errcrudata = ERRCPU10A;
	    else if (model == 12)
	       errcrudata = ERRCPU12;
	    
	    break;

	 case 'R': /* Use Real Time clock */
            usertclock = TRUE;
            break;

	 case 'r': /* ROM file spec */
	    i++;
	    if (i >= argc) goto USAGE;
	    strcpy (romfile, argv[i]);
	    break;

	 case 's': /* Size of memory */
	    i++;
	    if (i >= argc) goto USAGE;
	    memlen = 0;
	    for (pp = argv[i]; *pp; pp++)
	    {
	       if (isdigit (*pp))
	       {
		  memlen = (memlen * 10) + (*pp - '0');
	       }
	       else
	       {
	          if (*pp == 'm' || *pp == 'M')
		  {
		     memlen *= (1024*1024);
		     break;
		  }
		  else if (*pp == 'k' || *pp == 'K')
		  {
		     memlen *= 1024;
		     break;
		  }
		  else
		  {
		     fprintf (stderr,
			      "Invalid memory specification %s\n", argv[i]);
		     exit (ABORT);
		  }
	       }
	    }
	    break;

	 case 'w': /* Set window length */
	    i++;
	    if (i >= argc) goto USAGE;
	    windowlen = atoi (argv[i]);
            /* fall through */

         case 'p': /* Panel enable */
	    enablepanel = TRUE;
            clearscreen (enablepanel);
	    break;

         default:
      USAGE:
	    fprintf (stderr, "usage:  sim990 [-options][object.file]\n");
	    fprintf (stderr, " options:\n");
	    fprintf (stderr, "   -c configfile  - Config file\n");
	    fprintf (stderr, "   -m model       - CPU model\n");
	    fprintf (stderr,
		  "            CPU = 4, 5, 9, 10, 10A, 12, default = 4\n");
	    fprintf (stderr, "   -p             - Enable panel\n");
	    fprintf (stderr, "   -r romfile     - ROM file\n");
	    fprintf (stderr, "   -s memsize     - Memory size\n");
	    fprintf (stderr, "   -w NN          - Window size for panel\n");
	    return (ABORT);
         }
      }
      else
      {
	 if (objfile) goto USAGE;
	 objfile = argv[i];
      }
   }

   /*
   ** Check memory size for CPU type
   */

   displen = memlen;
   switch (model)
   {
      case 4:
      case 9:
         if (memlen > 56*1024)
	 {
	    goto MEMERR;
	 }
	 break;

      case 5:
         if (memlen > 64*1024)
	 {
	    goto MEMERR;
	 }
	 break;

      case 11:
         if (memlen > 1024*1024)
	 {
	    goto MEMERR;
	 }
	 break;

      default: /* 10 & 12 */
	 if (memlen > 2*1024*1024)
	 {
     MEMERR:
	    fprintf (stderr, "Memory size %d too large for 990/%s CPU\n",
		     memlen, models[model-4]);
	    return (ABORT);
	 }

	 if (memlen > (TPCSSTART + TILINESTART))
	 {
	    memlen = TPCSSTART + TILINESTART;
	 }
	 break;
   }

   /*
   ** Clear system
   */

   clearall ();

   tracelim[0] = 0;
   tracelim[1] = (model == 12) ? 0x201BFE :
                 (model == 5) ? 0x2003FE :
		 0x1FFFFE;

   viewlen = windowlen - OUTPUTLINE - 3;
   prtviewlen = windowlen - OUTPUTLINE - 1;
   prtviewcol = 1;

   /*
   ** Set up initial devices
   */

   memset (devices, 0, sizeof (devices));

   dev = &devices[0];
   dev->type = termtype;
   dev->infd = stdin;
   dev->outfd = stdout;
   dev->devaddr = termaddr;
   dev->extaddr = extaddr;
   dev->ctladdr = ctladdr;
   dev->intlvl = termint;
   dev->switches = SWUPCASE;
   strcpy (dev->name, termdev);
   strcpy (dev->file, "TTY");
   dev->info.terminfo.controller.serialinfo.mod5port = mod5port;
   termdstart (dev);

   devcnt = 1;
   maxdevcnt = MAXDEVICES;

   /*
   ** On a 990/5 and 990/10a CPUs the comm ports are always present.
   ** Setup dummies, needed by ROM self test.
   */

   if (model == 5 || model == 11)
   {
      maxdevcnt -= model == 5 ? 3 : 1;
      termaddr = 0x1700;
      for (i = maxdevcnt; i < MAXDEVICES; i++)
      {
	 dev = &devices[i];
	 if (termaddr == 0x1780)
	    dev->type = CON9903;
	 else
	    dev->type = CON9902;
	 dev->infd = DEV_NULL;
	 dev->outfd = DEV_NULL;
	 dev->devaddr = termaddr;
	 dev->extaddr = termaddr + 0x20;
	 if (model == 11)
	    dev->ctladdr = termaddr + 0x40;
	 dev->intlvl = termaddr == 0x1700 ? 8 : termaddr == 0x1740 ? 14 : 6;
	 termaddr += 0x40;
      }
   }

   ttinit ();
   initpanel();
   initinterrupt();

   /*
   ** Fire up clock thread
   */

   startclk (FALSE);

   /*
   ** Print intro
   */

   if (!enablepanel)
      printf ("%s %s: /%s CPU %dK\n",
	       INTRO, VERSION, models[model-4], displen/1024);

   signal (SIGINT, sigintr);

   /*
   ** Config file specified, do the commands.
   */

   if (config)
   {
      if (readconfig (config) < 0)
         goto DIE;
   }
   showstatus = TRUE;

   /*
   ** If object file specified, load and run it.
   */

   if (objfile)
   {
      if (simloader (objfile, 0) == 0)
      {
	 if (run)
	 {
	    runprog ();
	    waitconsole();
	 }
      }
      else
      {
         putview (FALSE, FALSE);
      }
   }

   /*
   ** Command loop
   */

   input[0] = oinput[0] = '\0';
   action = NULL_ACTION;

   while (!objfile && action != QUIT_ACTION)
   {
      panel ();

      if (view[0][0] != '\0')
	 putview (TRUE, FALSE);

      if (enablepanel)
      {
	 screenposition (enablepanel, windowlen-2, 1);
	 clearline (enablepanel);
	 printf (":%s", oinput);
	 screenposition (enablepanel, windowlen-2, 2);
      }
      else
      {
	 if (oinput[0])
	    printf ("\n:(%s) ", oinput);
	 else
	    printf ("\n: ");
      }

      fflush (stdout);
      action = NULL_ACTION;
      if (fgets (input, sizeof (input), stdin) != NULL)
      {
	 if ((pp = strchr (input,'\n')) != NULL)
	    *pp = 0;

	 if (input[0] == '\0')
	    strcpy (input, oinput);
	 else
	    strcpy (oinput, input);
	 putview (TRUE, FALSE);

	 action = docommand (TRUE);
      }
      else
      {
         printf ("fget returned NULL\n");
      }
   }

   /*
   ** Close open devices
   */

   for (i = 0; i < devcnt; i++)
   {
      dev = &devices[i];

      if (dev->runtermd)
         termdstop (dev);

      if (dev->infd < DEV_VDT)
         continue;

      if (dev->switches & SWTELNET)
      {
	 CLOSE (dev->sfd);
      }
      else
      {
	 if (dev->infd != stdin)
	    fclose (dev->infd);
      }
   }

DIE:
   ttclose ();
   stopclk (TRUE);
   screenposition (enablepanel, windowlen, 1);
   return (NORMAL);
}
