/***********************************************************************
*
* sim9902.c - 9902 port processing for the TI 990 Simulator.
*
* Changes:
*   09/17/13   DGP   Original.
*   11/08/13   DGP   Added H19 support.
*   05/07/14   DGP   Added 990/10A and CI402 support.
*   06/02/15   DGP   Removed USS (z/OS) support.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>

#include "simdef.h"

extern int enablepanel;
extern int model;
extern uint16 statreg;	/* The program status register */

#ifdef DEBUG9902TRACE
extern uint16 wpreg;
extern int tracelim[2];
extern int traceregs;
extern int traceenable;
extern FILE *tracefd;
extern uint32 curmpc;

static int isrset = FALSE;
static uint8 pdtmem[256];
#endif

#ifdef DEBUG9902
static char *setbits[] =
{
   "STROBE", "", "", "", "LXDR", "LRDR", "LDIR", "LDCTRL", "TSTMD",
   "RTS", "BRK", "RIENB", "XBIENB", "TIMENB", "DSCENB", "", "", "",
   "", "", "", "", "", "", "RESET", "DTR", "ALOOP", "SRTS", "", "",
   "HDUP", "INTENB", "COMENB"
};
static char *testbits[] =
{
   "", "", "RCVERR", "RPERR", "ROVER", "RFER", "RFBD", "RSBD", "RIN",
   "RBINT", "XBINT", "", "TIMINT", "DSCINT", "RBRL", "XBRE", "XSRE",
   "TIMERR", "TIMELP", "RTS", "DSR", "CTS", "DSCH", "FLAG", "INT",
    "DCD","RING", "SPEED", "MP9214", "4MHZ", "HDUP", "INTENB",
    "COMENB"
};
#endif

/***********************************************************************
* a9902timer - Check 9902 timer. Called from the clock handler.
***********************************************************************/

void
a9902timer (Device *dev)
{
   SerialInfo *devinfo;

   devinfo = &dev->info.terminfo.controller.serialinfo;

   if (devinfo->timerenb && (devinfo->timecnt > 0))
   {
#ifdef DEBUG9902TIMER
      fprintf (stderr, "a9902timer: devaddr = >%04X, timecnt = %d\n",
	       dev->devaddr, devinfo->timecnt);
#endif
      if (--devinfo->timecnt == 0)
      {
	 devinfo->timerint = TRUE;
	 if (dev->intenabled && (dev->intdelay == 0))
	 {
	    gendevinterrupt (dev);
	 }

#if 0
         if (dev->switches & SWSWITCH)
	 {
	    if (devinfo->ringsend)
	    {
#ifdef DEBUG9902
	       fprintf (stderr,
		  "a9902timer: devaddr = >%04X, ringstep = %d, count = %d\n",
			dev->devaddr, devinfo->ringstep, ringcount);
#endif
	       ringcount = 0;
	       switch (devinfo->ringstep++)
	       {
	       case 0:
		  devinfo->ringsent = TRUE;
		  break;

	       case 1:
		  devinfo->ringsent = FALSE;
		  break;

	       case 2:
		  devinfo->ringsent = TRUE;
		  break;

	       case 3:
		  devinfo->ringsent = FALSE;
		  break;

	       case 4:
		  devinfo->ringsent = TRUE;
		  break;

	       case 5:
		  devinfo->ringsent = FALSE;
		  break;

	       case 6:
		  devinfo->dcdset = TRUE;
		  devinfo->ringsend = FALSE;
		  if (dev->switches & SWSTATUS)
		  {
		     dev->info.terminfo.terminal.vdtinfo.CommDCD = TRUE;
		  }
	       }
	    }
	 }
#endif
      }
   }
}

/***********************************************************************
* a9902seto - Set CRU bit to one.
***********************************************************************/

void
a9902seto (Device *dev, uint16 devaddr, uint8 disp)
{
   SerialInfo *devinfo;
   uint8 adjdisp;
#ifdef DEBUG9902TRACE
   int ll, ii, rr;
#endif

   devinfo = &dev->info.terminfo.controller.serialinfo;

   adjdisp = disp;
   if (dev->extaddr == devaddr)
      adjdisp += 16;
   else if (dev->ctladdr == devaddr)
      adjdisp += 32;

#ifdef DEBUG9902RAW
   fprintf (stderr, "CRU %04X SBO  %04d:\n",
	    dev->devaddr, adjdisp);
#endif
#ifdef DEBUG9902
   fprintf (stderr, "CRU %04X(%04X) SBO %d(%d) %s\n",
	    dev->devaddr, devaddr, adjdisp, disp, 
	    adjdisp > 6 ? setbits[adjdisp-7] : "");
#endif

   switch (adjdisp)
   {
   case 7:	/* strobe */
      devinfo->xmtempty = TRUE;
      dev->cbreak = TRUE;
      if (devinfo->xmtintenabled)
      {
	 gendevinterrupt (dev);
      }
      break;

   case 11:	/* LXDR */
   case 12:	/* LRDR */
   case 13:	/* LDIR */
   case 14:	/* LDCTRL */
      devinfo->loadmode = TRUE;
      break;

   case 17:	/* BRK */
      break;

   case 15:	/* TESTMD */
      dev->inchar = 0x1F;
      devinfo->testmode = TRUE;
      break;

   case 16:	/* RTS */
      devinfo->rtsset = TRUE;
      devinfo->xmtempty = TRUE;
      devinfo->loadmode = FALSE;
      devinfo->testmode = FALSE;
      devinfo->dscint = TRUE;
      devinfo->dcdset = TRUE;
      if (devinfo->mod5port)
      {
	 devinfo->dtrset = TRUE;
      }
      if (dev->intenabled && devinfo->dscenabled && devinfo->dscint)
      {
	 gendevinterrupt (dev);
      }
      break;

   case 18:	/* RIENB */
      dev->select = FALSE;
      if (devinfo->mod5port)
      {
	 dev->intenabled = TRUE;
      }
      if (dev->switches & SWSWITCH)
	 devinfo->ringsend = TRUE;
      devinfo->ringstep = 0;
      break;

   case 19:	/* XBIENB */
      devinfo->xmtintenabled = TRUE;
      if (devinfo->mod5port)
      {
	 dev->intenabled = TRUE;
      }
      if (devinfo->xmtempty && dev->type != PRINT9902)
      {
	 gendevinterrupt (dev);
      }
      break;

   case 20:	/* TIMENB */
      devinfo->timecnt = 2;
      devinfo->timerenb = TRUE;
      if (devinfo->mod5port)
	 dev->intenabled = TRUE;
      devinfo->timerint = FALSE;
      break;

   case 21:	/* DSCENB */
      devinfo->dscenabled = TRUE;
      if (devinfo->mod5port)
	 dev->intenabled = TRUE;
      devinfo->dscint = FALSE;
      break;

   case 31:	/* RESET */
#ifdef DEBUG9902TRACE
      traceenable = TRUE;
      traceregs = TRUE;
      tracefd = stderr;
#ifdef DEBUGDX10
      tracelim[0] = curmpc - 0x87E;
      tracelim[1] = tracelim[0] + 0x2FAE;
#else
      tracelim[0] = curmpc - 0x34D4;
      tracelim[1] = tracelim[0] + 0x38EC;
#endif
      fprintf (stderr, "DSR TRACE: lim0 = >%06X, lim1 = >%06X\n", 
               tracelim[0], tracelim[1]);
#ifdef DEBUGDX10
      fprintf (stderr, "PDT: wp = >%04X\n", wpreg);
      rr = wpreg;
      for (ii = 0; ii < 80; ii += 2)
      {
	 uint16 vv;
	 vv = GETMEM (rr, NOLD);
	 pdtmem[ii] = (vv >> 8) & 0xFF;
	 pdtmem[ii+1] = vv & 0xFF;
	 rr += 2;
      }
      hexdump (stderr, pdtmem, 0, 70);
      fprintf (stderr, "\nPDTEXT: R4 = >%04X\n", R4);
      ll = 60;
#else
      fprintf (stderr, "\nPDT: R4 = >%04X\n", R4);
      ll = 186;
#endif
      rr = R4;
      for (ii = 0; ii < ll; ii += 2)
      {
	 uint16 vv;
	 vv = GETMEM (rr, NOLD);
	 pdtmem[ii] = (vv >> 8) & 0xFF;
	 pdtmem[ii+1] = vv & 0xFF;
	 rr += 2;
      }
      hexdump (stderr, pdtmem, 0, ll);
#endif
      devinfo->loadmode = TRUE;
      devinfo->xmtempty = FALSE;
      devinfo->rtsset = FALSE;
      devinfo->dtrset = FALSE;
      devinfo->ringsent = FALSE;
      devinfo->ringsend = FALSE;
      devinfo->testmode = FALSE;
      devinfo->xmtintenabled = FALSE;
      devinfo->timerenb = FALSE;
      devinfo->timerint = FALSE;
      devinfo->dscenabled = FALSE;
      devinfo->dscint = FALSE;
      dev->select = FALSE;
      dev->intenabled = FALSE;
      if (dev->switches & SWSTATUS)
      {
	 dev->info.terminfo.terminal.vdtinfo.CommDTR = FALSE;
	 dev->info.terminfo.terminal.vdtinfo.CommDSR = FALSE;
	 dev->info.terminfo.terminal.vdtinfo.CommDCD = FALSE;
      }
      clrinterrupt (dev->intlvl);
      break;

   /* External control bits. 990/10A and CI402 */

   case 32:	/* DTR */
      devinfo->dscint = TRUE;
      devinfo->dtrset = TRUE;
      devinfo->dcdset = TRUE;
      if (dev->switches & SWSTATUS)
      {
	 dev->info.terminfo.terminal.vdtinfo.CommDTR = TRUE;
	 dev->info.terminfo.terminal.vdtinfo.CommDSR = TRUE;
	 dev->info.terminfo.terminal.vdtinfo.CommDCD = TRUE;
	 dev->info.terminfo.terminal.vdtinfo.CommState = VDT_COMMSTATE_RDY;
	 dev->cbreak = TRUE;
      }
      if (dev->intenabled && devinfo->dscenabled && devinfo->dscint)
      {
	 gendevinterrupt (dev);
      }
      break;

   case 33:	/* Analog loopback */
   case 34:	/* Secondary RTS */
   case 35:	/* Unused */
   case 36:	/* Unused */
   case 37:	/* Half duplex */
      break;

   case 38:	/* Interrupt enable */
      dev->intenabled = TRUE;
      break;

   case 39:	/* COMM enable */
      devinfo->dtrset = TRUE;
      devinfo->dcdset = TRUE;
      devinfo->commenabled = TRUE;
      break;

   default: ;
   }
}

/***********************************************************************
* a9902setz - Set CRU bit to zero.
***********************************************************************/

void
a9902setz (Device *dev, uint16 devaddr, uint8 disp)
{
   SerialInfo *devinfo;
   uint8 adjdisp;

   devinfo = &dev->info.terminfo.controller.serialinfo;

   adjdisp = disp;
   if (dev->extaddr == devaddr)
      adjdisp += 16;
   else if (dev->ctladdr == devaddr)
      adjdisp += 32;

#ifdef DEBUG9902RAW
   fprintf (stderr, "CRU %04X SBZ  %04d:\n",
	    dev->devaddr, adjdisp);
#endif
#ifdef DEBUG9902
   fprintf (stderr, "CRU %04X(%04X) SBZ %d(%d) %s\n",
	    dev->devaddr, devaddr, adjdisp, disp, 
	    adjdisp > 6 ? setbits[adjdisp-7] : "");
#endif

   switch (adjdisp)
   {
   case 7:	/* strobe */
      dev->cbreak = TRUE;
      devinfo->xmtempty = TRUE;
      if (devinfo->xmtintenabled)
      {
	 gendevinterrupt (dev);
      }
      break;

   case 11:	/* LXDR */
   case 12:	/* LRDR */
   case 13:	/* LDIR */
   case 14:	/* LDCTRL */
      devinfo->loadmode = FALSE;
      break;

   case 17:	/* BRK */
      break;

   case 15:	/* TESTMD */
      dev->inchar = 0;
      devinfo->testmode = FALSE;
      break;

   case 16:	/* RTS */
      devinfo->rtsset = FALSE;
      devinfo->loadmode = FALSE;
      devinfo->testmode = FALSE;
      break;

   case 18:	/* RIENB */
      dev->select = FALSE;
      break;

   case 19:	/* XBIENB */
      devinfo->xmtintenabled = FALSE;
      break;

   case 20:	/* TIMENB */
      devinfo->timecnt = 0;
      devinfo->timerenb = FALSE;
      devinfo->timerint = FALSE;
      break;

   case 21:	/* DSCENB */
      devinfo->dscenabled = FALSE;
      devinfo->dscint = FALSE;
      break;

   case 31:	/* RESET */
      devinfo->loadmode = TRUE;
      devinfo->xmtempty = FALSE;
      devinfo->rtsset = FALSE;
      devinfo->dtrset = FALSE;
      devinfo->ringsent = FALSE;
      devinfo->ringsend = FALSE;
      devinfo->testmode = FALSE;
      devinfo->xmtintenabled = FALSE;
      devinfo->timerenb = FALSE;
      devinfo->timerint = FALSE;
      devinfo->dscint = FALSE;
      dev->select = FALSE;
      dev->intenabled = FALSE;
      if (dev->switches & SWSTATUS)
      {
	 dev->info.terminfo.terminal.vdtinfo.CommDTR = FALSE;
	 dev->info.terminfo.terminal.vdtinfo.CommDSR = FALSE;
	 dev->info.terminfo.terminal.vdtinfo.CommDCD = FALSE;
      }
      clrinterrupt (dev->intlvl);
      break;

   /* External control bits. 990/10A and CI402 */

   case 32:	/* DTR */
      devinfo->dtrset = FALSE;
      devinfo->dcdset = FALSE;
      if (dev->switches & SWSTATUS)
      {
	 dev->info.terminfo.terminal.vdtinfo.CommDTR = FALSE;
	 dev->info.terminfo.terminal.vdtinfo.CommDSR = FALSE;
	 dev->info.terminfo.terminal.vdtinfo.CommDCD = FALSE;
	 dev->info.terminfo.terminal.vdtinfo.CommState = VDT_COMMSTATE_DISC;
	 dev->cbreak = TRUE;
      }
      break;

   case 33:	/* Analog loopback */
   case 34:	/* Secondary RTS */
   case 35:	/* Unused */
   case 36:	/* Unused */
   case 37:	/* Half duplex */
      break;

   case 38:	/* Interrupt enable */
      dev->intenabled = FALSE;
      break;

   case 39:	/* COMM enable */
      devinfo->commenabled = FALSE;
      break;

   default: ;
   }
}

/***********************************************************************
* a9902tb - Test CRU bit.
***********************************************************************/

void
a9902tb (Device *dev, uint16 devaddr, uint8 disp)
{
   SerialInfo *devinfo;
   uint8 adjdisp;

   devinfo = &dev->info.terminfo.controller.serialinfo;

   adjdisp = disp;
   if (dev->extaddr == devaddr)
      adjdisp += 16;
   else if (dev->ctladdr == devaddr)
      adjdisp += 32;

#ifdef DEBUG9902RAW
   fprintf (stderr, "CRU %04X TB   %04d: ",
	    dev->devaddr, adjdisp);
#endif
#ifdef DEBUG9902
   fprintf (stderr, "CRU %04X(%04X) TB %d(%d) %s",
	    dev->devaddr, devaddr, adjdisp, disp,
	    adjdisp > 6 ? testbits[adjdisp-7] : "");
#endif

   CLR_EQ;
   switch (adjdisp)
   {
   case 9:	/* RCVERR */
   case 10:	/* RPER */
   case 11:	/* ROVER */
   case 12:	/* RFER */
   case 13:	/* RFBD */
   case 14:	/* RSBD */
   case 24:	/* TIMERR */
   case 30:	/* FLAG */
      break;

   case 15:	/* RIN */
      if (dev->intenabled && dev->select)
         SET_EQ;
      break;

   case 16:	/* RBINT */
   case 21:	/* RBRL */
      if (dev->infd == DEV_NULL || devinfo->testmode)
      {
         SET_EQ;
	 break;
      }
      if (!dev->intenabled && !GET_MASK)
      {
	 int c;

	 c = serialinput (dev);
	 if (c != EOF)
	 {
	    dev->inchar = c;
	    SET_EQ;
	 }
      }
      else if (dev->select)
      {
	 SET_EQ;
      }
      break;

   case 17:	/* XBINT */
      if (devinfo->xmtintenabled && devinfo->xmtempty)
         SET_EQ;
      break;

   case 20:	/* DSCINT */
      if (devinfo->dscenabled && devinfo->dscint)
         SET_EQ;
      break;

   case 22:	/* XBRE */
      if (devinfo->xmtempty)
	 SET_EQ;
      break;

   case 23:	/* XSRE */
      SET_EQ;
      break;

   case 19:	/* TIMINT */
   case 25:	/* TIMELP */
      if (devinfo->timerenb && devinfo->timerint)
         SET_EQ;
      break;

   case 26:	/* RTS */
      if (devinfo->rtsset)
         SET_EQ;
      break;

   case 27:	/* DSR */
      if (devinfo->dtrset)
         SET_EQ;
      break;

   case 28:	/* CTS */
      if (devinfo->rtsset)
         SET_EQ;
      break;

   case 29:	/* DSCH */
      if (devinfo->dscint)
         SET_EQ;
      break;

   case 31:	/* INT */
#if defined(DEBUG9902TRACE) && defined(DEBUGDX10)
      if (!isrset)
      {
	 isrset = TRUE;
	 traceenable = TRUE;
	 traceregs = TRUE;
	 tracelim[0] = curmpc - 0xAA;
	 tracelim[1] = tracelim[0] + 0xF7A;
	 fprintf (stderr, "ISR TRACE: lim0 = >%06X, lim1 = >%06X\n", 
		  tracelim[0], tracelim[1]);
	 tracefd = stderr;
      }
#endif
      if (dev->intenabled && devinfo->timerenb && devinfo->timerint)
         SET_EQ;
      else if (dev->intenabled && devinfo->dscenabled && devinfo->dscint)
         SET_EQ;
      else if (dev->intenabled && dev->select)
         SET_EQ;
      else if (dev->intenabled && devinfo->xmtintenabled && devinfo->xmtempty)
         SET_EQ;
      break;

   /* External control bits. 990/10A and CI402 */

   case 32:	/* DCD */
      if (devinfo->dcdset)
         SET_EQ;
      break;

   case 33:	/* RING */
      if (devinfo->ringsend)
      {
	 switch (devinfo->ringstep++)
	 {
	 case 0:
	 case 2:
	 case 4:
	    SET_EQ;
	    break;

	 case 6:
	    devinfo->dcdset = TRUE;
	    devinfo->ringsend = FALSE;
	    if (dev->switches & SWSTATUS)
	    {
	       dev->info.terminfo.terminal.vdtinfo.CommDCD = TRUE;
	    }
	    break;

	 default: ;
	 }
      }
      break;

   case 34:	/* Speed Indication */
      break;

   case 35:	/* MP9214 */
      SET_EQ;
      break;

   case 36:	/* 4 MHZ */
      if (dev->devaddr == TERM9902) /* 990/10a port */
      {
	 devinfo->commenabled = TRUE;
         SET_EQ;
      }
      break;

   case 37:	/* Half duplex */
      break;

   case 38:	/* Interrupts enable */
      if (dev->intenabled)
         SET_EQ;
      break;

   case 39:	/* COMM enable */
      if (devinfo->commenabled)
         SET_EQ;
      break;

   default: ;
   }
#ifdef DEBUG9902RAW
   fprintf (stderr, "%s\n", IS_EQ ? "TRUE" : "FALSE");
#endif
#ifdef DEBUG9902
   fprintf (stderr, ": %s\n", IS_EQ ? "TRUE" : "FALSE");
#endif
}

/***********************************************************************
* a9902ldcr - Load CRU.
***********************************************************************/

void
a9902ldcr (Device *dev, uint16 devaddr, uint16 cnt, uint16 dat)
{
   SerialInfo *devinfo;

#ifdef DEBUG9902RAW
   fprintf (stderr, "CRU %04X LDCR %04d: %04X\n",
	    dev->devaddr, cnt, (cnt > 8 || cnt == 0) ? dat : (dat << 8));
#endif
#ifdef DEBUG9902
   if (cnt == 8 && (dat > 0x1F && dat < 0x7F))
      fprintf (stderr, "CRU %04X(%04X) LDCR CNT %d DAT %02X(%c)\n",
	       dev->devaddr, devaddr, cnt, dat, dat);
   else
      fprintf (stderr, "CRU %04X(%04X) LDCR CNT %d DAT %04X\n",
	       dev->devaddr, devaddr, cnt, dat);
#endif

   devinfo = &dev->info.terminfo.controller.serialinfo;
   if (devinfo->loadmode) return;
   if (devinfo->testmode) return;
   if (!devinfo->rtsset) return;
   if (dev->outfd == DEV_NULL) return;
   devinfo->xmtempty = FALSE;

   if (!devinfo->rtsset) return;

   switch (dev->type)
   {
   case PRINT9902:
   case PRINTCI402:
      printputchar (dev, dat & 0x7F);
      break;

   case CON9902:
   case SERIAL9902:
   case SERIALCI402:
      serialoutput (dev, dat);
      if (cnt > 7)
	 dev->cbreak = TRUE;
      break;

   default: ;
   }

   if (cnt > 7)
   {
      devinfo->xmtempty = TRUE;
      if (devinfo->xmtintenabled)
      {
	 gendevinterrupt (dev);
      }
   }
}

/***********************************************************************
* a9902stcr - Store CRU.
***********************************************************************/

uint16
a9902stcr (Device *dev, uint16 devaddr, uint16 cnt)
{
   SerialInfo *devinfo;
   uint16 ch;

   devinfo = &dev->info.terminfo.controller.serialinfo;

#ifdef DEBUG9902RAW
   fprintf (stderr, "CRU %04X STCR %04d: ",
	    dev->devaddr, cnt);
#endif
#ifdef DEBUG9902
   fprintf (stderr, "CRU %04X(%04X) STCR CNT %d", dev->devaddr, devaddr, cnt);
#endif

   ch = 0;
   if (devinfo->testmode)
   {
      ch = dev->inchar;
      dev->inchar = (dev->inchar << 1) | 1;
      if (ch > 0x7F) dev->inchar = 0;
   }
   else if (dev->infd != DEV_NULL)
   {
      if (devaddr == dev->devaddr)
      {
	 ch = dev->inchar;
      }
      else
      {
         ch = 0x1C40;
	 if (dev->intenabled && dev->select)
	    ch |= 0x8001;
      }
   }

#ifdef DEBUG9902RAW
   fprintf (stderr, "%04X\n", ch);
#endif
#ifdef DEBUG9902
   fprintf (stderr, " DAT = %2X(%c)\n", 
	    ch, (ch > 0x1F && ch < 0x7F) ? ch : '.');
#endif

   return (ch);
}
