/***********************************************************************
*
* sim9903.c - 9903 port processing for the TI 990 Simulator.
*
* Changes:
*   09/17/13   DGP   Original.
*   06/02/15   DGP   Removed USS (z/OS) support.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>

#include "simdef.h"

extern int enablepanel;
extern uint16 statreg;	/* The program status register */

/***********************************************************************
* a9903seto - Set CRU bit to one.
***********************************************************************/

void
a9903seto (Device *dev, uint16 devaddr, uint8 disp)
{
   SerialInfo *devinfo;

   devinfo = &dev->info.terminfo.controller.serialinfo;

#ifdef DEBUG9903
   if (dev->extaddr == devaddr)
      fprintf (stderr, "CRU %04X SBO %d\n", dev->devaddr, disp+16);
   else
      fprintf (stderr, "CRU %04X SBO %d\n", devaddr, disp);
#endif

   if (dev->extaddr == devaddr)
      disp += 16;

   switch (disp)
   {
   case 11:	/* LXDR */
   case 12:	/* LRDR */
   case 13:	/* LDIR */
   case 14:	/* LDCTRL */
   case 17:	/* BRK */
   case 20:	/* TIMENB */
   case 21:	/* DSCENB */
   case 25:	/* strobe */
      break;

   case 15:	/* TESTMD */
      dev->inchar = 0x3A;
      devinfo->testmode = TRUE;
      break;

   case 16:	/* RTS */
      devinfo->rtsset = TRUE;
      break;

   case 18:	/* RIENB */
      dev->select = FALSE;
      dev->intenabled = TRUE;
      break;

   case 19:	/* XIENB */
      devinfo->xmtintenabled = TRUE;
      geninterrupt (dev->intlvl);
      break;

   case 31:	/* RESET */
      devinfo->rtsset = FALSE;
      devinfo->testmode = FALSE;
      devinfo->xmtintenabled = FALSE;
      dev->select = FALSE;
      dev->intenabled = FALSE;
      clrinterrupt (dev->intlvl);
      break;

   default: ;
   }
}

/***********************************************************************
* a9903setz - Set CRU bit to zero.
***********************************************************************/

void
a9903setz (Device *dev, uint16 devaddr, uint8 disp)
{
   SerialInfo *devinfo;

   devinfo = &dev->info.terminfo.controller.serialinfo;

#ifdef DEBUG9903
   if (dev->extaddr == devaddr)
      fprintf (stderr, "CRU %04X SBZ %d\n", dev->devaddr, disp+16);
   else
      fprintf (stderr, "CRU %04X SBZ %d\n", devaddr, disp);
#endif

   if (dev->extaddr == devaddr)
      disp += 16;

   switch (disp)
   {
   case 11:	/* LXDR */
   case 12:	/* LRDR */
   case 13:	/* LDIR */
   case 14:	/* LDCTRL */
   case 17:	/* BRK */
   case 20:	/* TIMENB */
   case 21:	/* DSCENB */
   case 25:	/* strobe */
      break;

   case 15:	/* TESTMD */
      devinfo->testmode = FALSE;
      break;

   case 16:	/* RTS */
      devinfo->rtsset = FALSE;
      break;

   case 18:	/* RIENB */
      dev->select = FALSE;
      dev->intenabled = FALSE;
      clrinterrupt (dev->intlvl);
      break;

   case 19:	/* XIENB */
      devinfo->xmtintenabled = FALSE;
      clrinterrupt (dev->intlvl);
      break;

   case 31:	/* RESET */
      devinfo->rtsset = FALSE;
      devinfo->testmode = FALSE;
      devinfo->xmtintenabled = FALSE;
      dev->select = FALSE;
      dev->intenabled = FALSE;
      clrinterrupt (dev->intlvl);
      break;

   default: ;
   }
}

/***********************************************************************
* a9903tb - Test CRU bit.
***********************************************************************/

void
a9903tb (Device *dev, uint16 devaddr, uint8 disp)
{
   SerialInfo *devinfo;

   devinfo = &dev->info.terminfo.controller.serialinfo;

#ifdef DEBUG9903
   if (dev->extaddr == devaddr)
      fprintf (stderr, "CRU %04X TB %d", dev->devaddr, disp+16);
   else
      fprintf (stderr, "CRU %04X TB %d", devaddr, disp);
#endif

   if (dev->extaddr == devaddr)
      disp += 16;

   CLR_EQ;
   switch (disp)
   {
   case 9:	/* RCVERR */
   case 10:	/* RPER */
   case 11:	/* ROVER */
   case 12:	/* RFER */
   case 13:	/* RFBD */
   case 14:	/* RSBD */
   case 19:	/* TIMINT */
   case 20:	/* DSCINT */
   case 23:	/* XSRE */
   case 24:	/* TIMERR */
   case 25:	/* TIMELP */
   case 29:	/* DSCH */
   case 30:	/* FLAG */
      break;

   case 15:	/* RIN */
      if (dev->intenabled && dev->select)
         SET_EQ;
      break;

   case 16:	/* RBINT */
   case 21:	/* RBRL */
      if (dev->infd == DEV_NULL || devinfo->testmode)
      {
         SET_EQ;
	 break;

      }
      if (!dev->intenabled)
      {
	 int c;

	 c = serialinput (dev);
	 if (c != EOF)
	 {
	    dev->inchar = c;
	    SET_EQ;
	 }
      }
      else if (dev->select)
      {
	 SET_EQ;
      }
      break;

   case 17:	/* XBINT */
      if (devinfo->xmtintenabled)
         SET_EQ;
      break;

   case 22:	/* XBRE */
   case 26:	/* RTS */
   case 27:	/* DSR */
   case 28:	/* CTS */
      SET_EQ;
      break;

   case 31:	/* INT */
      if (dev->intenabled && dev->select)
         SET_EQ;
      break;

   default: ;
   }
#ifdef DEBUG9903
   fprintf (stderr, ": %s\n", IS_EQ ? "TRUE" : "FALSE");
#endif
}

/***********************************************************************
* a9903ldcr - Load CRU.
***********************************************************************/

void
a9903ldcr (Device *dev, uint16 devaddr, uint16 cnt, uint16 dat)
{
   SerialInfo *devinfo;

#ifdef DEBUG9903
   if (dat > 0x1F && dat < 0x7F)
      fprintf (stderr, "CRU %04X LDCR CNT %d DAT %02X(%c)\n",
	    devaddr, cnt, dat, dat);
   else
      fprintf (stderr, "CRU %04X LDCR CNT %d DAT %02X\n", devaddr, cnt, dat);
#endif

   devinfo = &dev->info.terminfo.controller.serialinfo;
   if (devinfo->testmode)
   {
      dev->inchar = dat;
      return;
   }

   if (!devinfo->rtsset) return;
   if (dev->outfd == DEV_NULL) return;

   switch (dev->type)
   {
   case PRINT9903:
      printputchar (dev, dat & 0x7F);
      break;

   case CON9903:
   case SERIAL9903:
      serialoutput (dev, dat);
      break;

   default: ;
   }

   if (devinfo->xmtintenabled)
      geninterrupt (dev->intlvl);
}

/***********************************************************************
* a9903stcr - Store CRU.
***********************************************************************/

uint16
a9903stcr (Device *dev, uint16 devaddr, uint16 cnt)
{
   SerialInfo *devinfo;
   uint16 ch;

   devinfo = &dev->info.terminfo.controller.serialinfo;

#ifdef DEBUG9903
   fprintf (stderr, "CRU %04X STCR CNT %d", devaddr, cnt);
#endif

   ch = 0;
   if (devinfo->testmode)
   {
      ch = dev->inchar;
   }
   else if (dev->infd != DEV_NULL)
   {
      if (devaddr == dev->devaddr)
      {
	 ch = dev->inchar;
      }
      else
      {
         ch = 0x1C40;
	 if (dev->intenabled && dev->select)
	    ch |= 0x8001;
      }
   }

#ifdef DEBUG9903
   fprintf (stderr, " DAT = %0X(%c)\n",
	    ch, (ch > 0x1F && ch < 0x7F) ? ch : '.');
#endif

   return (ch);
}
