/***********************************************************************
*
* sima733.c - ASR 733 Terminal support for the TI 990 Simulator.
*
* Changes:
*   05/01/15   DGP   Added ASR 733 emulation support.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>
#include <time.h>

#if defined(UNIX)
#include <unistd.h>
#include <sys/time.h>
#endif

#include "simdef.h"

extern uint16 pcreg;	/* The program PC */
extern uint16 wpreg;	/* The program Workspace Pointer */
extern int run;
extern int model;
extern int bootfromrom;

extern uint8 memory[SYSMEMSIZE];

extern char view[MAXVIEW][MAXVIEWLEN+1];

/***********************************************************************
* a733process - ASR733 process output emulation.
***********************************************************************/

void
a733process (Device *dev)
{
   TermInfo *terminfo;
   TermRing *prcring;
   TermRing *inring;
   ASRInfo *asrinfo;
   char *bp;
   int c;
   int first;
   char buff[A733BLKLEN+2];

   terminfo = &dev->info.terminfo;
   prcring = &terminfo->prcring;
   inring = &terminfo->inring;
   asrinfo = &terminfo->terminal.asrinfo;

   while (!termdringempty (dev, prcring))
   {
      first = prcring->tail;
      c = termdgetring (dev, prcring);

#ifdef DEBUGASR733
      fprintf (stderr,
	       "a733process: c = >%02X(%c), first = %d, head = %d, tail = %d\n",
	       c, isprint (c) ? c : '.', first, prcring->head, prcring->tail);
      fprintf (stderr, "   status = >%04X\n",
	       asrinfo->status);
#endif

      if (c == 0x10)
      {
	 if ((c = termdgetcond (dev, prcring, first)) < 0) return;
#ifdef DEBUGASR733
	 fprintf (stderr,
	       "   c = >%02X(%c), first = %d, head = %d, tail = %d\n",
	       c, isprint (c) ? c : '.', first, prcring->head, prcring->tail);
#endif
	 switch (c)
	 {
	 case '1': /* Rewind Cassette 1 */
	    if (realfd (asrinfo->cs1fd))
	       fseek (asrinfo->cs1fd, 0, SEEK_SET);
	    asrinfo->status |= A733BOEO1;
	    break;

	 case '2': /* Rewind Cassette 2 */
	    if (realfd (asrinfo->cs2fd))
	       fseek (asrinfo->cs2fd, 0, SEEK_SET);
	    asrinfo->status |= A733BOEO2;
	    break;

	 case '3': /* Load Cassette 1 */
	    asrinfo->status &= ~A733BOEO1;
	    break;

	 case '4': /* Load Cassette 2 */
	    asrinfo->status &= ~A733BOEO2;
	    break;

	 case '5': /* Cassette 1 in record */
	    asrinfo->status |= (A733RCRDRDY | A733RCRDC1 | A733PLAYRDY);
	    if (!realfd (asrinfo->cs1fd))
	       asrinfo->status |= A733BOEO1;
	    asrinfo->status &= ~(A733PRTRRDY | A733RCRDC2);
	    break;

	 case '6': /* Cassette 2 in record */
	    asrinfo->status |= (A733RCRDRDY | A733RCRDC2 | A733PLAYRDY);
	    if (!realfd (asrinfo->cs2fd))
	       asrinfo->status |= A733BOEO2;
	    asrinfo->status &= ~(A733PRTRRDY | A733RCRDC1);
	    break;

	 case '7': /* Block forward */
	    asrinfo->status |= A733PLAYRDY;
	    if (asrinfo->status & A733RCRDC2 && realfd (asrinfo->cs1fd))
	       bp = fgets (buff, A733BLKLEN, asrinfo->cs1fd);
	    else if (asrinfo->status & A733RCRDC1 && realfd (asrinfo->cs2fd))
	       bp = fgets (buff, A733BLKLEN, asrinfo->cs2fd);
	    else 
	       bp = NULL;

#ifdef DEBUGASR733
            fprintf (stderr, "block forward: bp = %p\n", bp);
	    if (bp)
	       HEXDUMP (stderr, buff, A733BLKLEN, 0);
#endif
	    if (bp)
	    {
	       for (; *bp; bp++)
	       {
	          if (*bp == '\n')
		     termdputring (dev, inring, '\r');
		  else
		     termdputring (dev, inring, *bp);
	       }
	    }
	    else
	    {
#ifdef DEBUGASR733
	       fprintf (stderr, "   EOF\n");
#endif
	       termdputring (dev, inring, 0x13);
	       termdputring (dev, inring, '\r');
	    }
	    break;

	 case '8': /* Block reverse */
	    break;

	 case '9': /* Printer On */
	    asrinfo->status |= A733PRTRRDY;
	    break;

	 case '0': /* Printer Off */
	    asrinfo->status &= ~A733PRTRRDY;
	    break;

	 case ':': /* ADC On */
	    asrinfo->status |= A733ADCON;
	    break;

	 case ';': /* ADC Off */
	    asrinfo->status &= ~A733ADCON;
	    break;

	 case '<': /* Request Status */
#ifdef DEBUGASR733
            fprintf (stderr, "   status = >%02X\n", asrinfo->status & 0x7F);
#endif
	    termdputring (dev, inring, asrinfo->status & 0x7F);
	    break;

	 default: ;
	 }
      }
      else if (c == 0x11) /* Playback on - Keyboard off */
      {
	 asrinfo->status |= (A733PLAYRDY | A733PLAYON);
	 while (TRUE)
	 {
	    bp = NULL;
	    if (realfd (asrinfo->cs1fd))
	       bp = fgets (buff, A733BLKLEN, asrinfo->cs1fd);
	    else if (realfd (asrinfo->cs2fd))
	       bp = fgets (buff, A733BLKLEN, asrinfo->cs2fd);
	    if (!bp) break;

#ifdef DEBUGASR733
	    fprintf (stderr, "Playback ON: bp = %p\n", bp);
	    HEXDUMP (stderr, buff, A733BLKLEN, 0);
#endif
	    for (; *bp; bp++)
	    {
	       if (*bp == '\n')
		  termdputring (dev, inring, '\r');
	       else
		  termdputring (dev, inring, *bp);
	    }
	    while (!termdringempty (dev, inring))
	    {
#ifdef LINUX
	       sched_yield ();
#else
	       smallsleep (1);
#endif
	    }
	 }
	 asrinfo->status &= ~(A733PLAYRDY | A733PLAYON);
      }
      else if (c == 0x12) /* Record on - Printer off */
      {
	 asrinfo->status |= A733RCRDON;
	 asrinfo->status &= ~A733PRTRRDY;
      }
      else if (c == 0x13) /* Playback off - Keyboard on */
      {
	 asrinfo->status &= ~A733PLAYON;
      }
      else if (c == 0x14) /* Record off - Printer on */
      {
	 asrinfo->status &= ~A733RCRDON;
	 asrinfo->status |= A733PRTRRDY;
      }
      else if (c == 0x17)
      {
      }
      else if (c > 0 && c < 0x7F)
      {
	 if (asrinfo->status & A733RCRDON)
	 {
	    if (c == '\r') continue;
	    if ((asrinfo->status & A733RCRDC1) && realfd (asrinfo->cs1fd))
	       fputc (c, asrinfo->cs1fd);
	    else if ((asrinfo->status & A733RCRDC2) && realfd (asrinfo->cs2fd))
	       fputc (c, asrinfo->cs2fd);
	 }
	 else
	    termdputring (dev, &terminfo->outring, c);
      }
   }
}

/***********************************************************************
* a733boot - ASR 733 boot
***********************************************************************/

int
a733boot (Device *dev, char *device, int unit)
{
   FILE *lfd = NULL;
   FILE *csfd;
   ASRInfo *asrinfo;
#ifdef DEBUGASR733
   char *fp;
#endif
   int lp, clp;
   int i;
   int done;
   int reclen;
   uint8 inbuf[82];

   clp = lp = LOADADDRESS;

   if (bootfromrom)
   {
      if (model == 4 || model == 9)
      {
	 wpreg = GETMEM0 (0xFFFC) & 0xFFFE;
	 pcreg = GETMEM0 (0xFFFE) & 0xFFFE;
      }
      else
      {
	 wpreg = GETMEM0 (TPCSSTART+0xFFFC) & 0xFFFE;
	 pcreg = GETMEM0 (TPCSSTART+0xFFFE) & 0xFFFE;
      }
      PUTREG (0, 0);
      run = TRUE;
      return (0);
   }

   asrinfo = &dev->info.terminfo.terminal.asrinfo;

   if (unit == 0)
   {
#ifdef DEBUGASR733
      fp = asrinfo->cs1file;
#endif
      csfd = asrinfo->cs1fd;
   }
   else
   {
#ifdef DEBUGASR733
      fp = asrinfo->cs2file;
#endif
      csfd = asrinfo->cs2fd;
   }

   if (!realfd (csfd))
   {
      sprintf (view[0], "Device %s: Is attached to NULL", device);
      return (-1);
   }

#ifdef DEBUGASR733
   lfd = fopen ("cassetteload.log", "w");
   fprintf (lfd, "cassetteboot: file = %s, loadpd = %d(%04X)\n", fp, lp, lp);
#endif

   done = FALSE;
   fseek (csfd, 0, SEEK_SET);
   for (i = 0; !done; i++)
   {
      if (fgets ((char *)inbuf, sizeof(inbuf), csfd) == NULL)
         break;
      reclen = 80;

      inbuf[reclen] = 0;
#if defined(DUMPLOADREC) && defined(DEBUGASR733)
      fprintf (lfd, "Record: %d\n", i+1);
      HEXDUMP (lfd, inbuf, reclen, 0);
#endif
      if (loadrec (lfd, inbuf, lp, &clp, FALSE) == EOF)
	 done = TRUE;
   }

#ifdef DEBUGASR733
   fclose (lfd);
#endif
   run = TRUE;

   return (0);
}
