/***********************************************************************
*
* simcard.c - Card Reader for the TI 990 Simulator.
*
* Changes:
*   09/19/13   DGP   Original. Split from simio.c
*   05/08/14   DGP   Fixed to work with DNOS.
*   04/28/15   DGP   Added boot from card reader.
*   06/02/15   DGP   Removed USS (z/OS) support.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>

#include "simdef.h"
#include "simcra.h"

extern uint16 statreg;	/* The program status register */

extern int run;

/***********************************************************************
* cardseto - Set CRU bit to one.
***********************************************************************/

void
cardseto (Device *dev, uint16 devaddr, uint8 disp)
{
#ifdef DEBUGCARD
   fprintf (stderr, "CRU %04X SBO %d\n", devaddr, disp);
#endif
   if (disp == 13) /* Enable Column interrupt */
   {
      dev->intenabled = TRUE;
      dev->select = TRUE;
      gendevinterrupt (dev);
   }
}

/***********************************************************************
* cardsetz - Set CRU bit to zero.
***********************************************************************/

void
cardsetz (Device *dev, uint16 devaddr, uint8 disp)
{
#ifdef DEBUGCARD
   fprintf (stderr, "CRU %04X SBZ %d\n", devaddr, disp);
#endif

   switch (disp)
   {
   case 9:	/* Clear presence interrupt */
   case 14:	/* Clear column interrupt */
      if (dev->select)
      {
	 dev->intenabled = TRUE;
	 gendevinterrupt (dev);
      }
      break;

   case 13:	/* Disable Column interrupt */
      dev->intenabled = FALSE;
      dev->select = FALSE;
      gendevinterrupt (dev);
      break;

   default: ;
   }
}

/***********************************************************************
* cardtb - Test CRU bit.
***********************************************************************/

void
cardtb (Device *dev, uint16 devaddr, uint8 disp)
{
#ifdef DEBUGCARD
   fprintf (stderr, "CRU %04X TB %d", devaddr, disp);
#endif

   CLR_EQ;
   switch (disp)
   {
   case 15:	/* Column Ready */
      SET_EQ;
      break;

   default: ;
   }
#ifdef DEBUGCARD
    fprintf (stderr, " %s\n", IS_EQ ? "TRUE" : "FALSE");
#endif
}

/***********************************************************************
* cardldcr - Load CRU.
***********************************************************************/

void
cardldcr (Device *dev, uint16 devaddr, uint16 cnt, uint16 dat)
{
#ifdef DEBUGCARD
   fprintf (stderr, "CRU %04X LDCR CNT %d DATA %04X\n", devaddr, cnt, dat);
#endif
   if (cnt == 0 && dat == 0x8800)
      dev->info.cardinfo.cardinit = TRUE;
}

/***********************************************************************
* cardstcr - Store CRU.
***********************************************************************/

uint16
cardstcr (Device *dev, uint16 devaddr, uint16 cnt)
{
   CARDInfo *devinfo;
   int cdat;
   uint16 dat;

   /*
   ** If record exhausted, read another
   */

   devinfo = &dev->info.cardinfo;
   if (!devinfo->cardinit)
   {
      cdat = 0;
#ifdef DEBUGCARD
      fprintf (stderr, "CRU %04X STCR CNT %d DATA %04X\n",
	       devaddr, cnt, cdat);
#endif
      return (cdat);
   }

   if (devinfo->cardndx >= 80)
   {
      char *cbuf;
      int i1, o1;
      int len;
      char temp[1024];

      /*
      ** Read record
      */

      if (!(devinfo->cardeof) && (dev->infd != DEV_NULL))
      {
	 if (fgets (temp, 1024, dev->infd) == NULL)
	 {
	    strcpy (temp, "/*\n");
	    devinfo->cardeof = TRUE;
	 }
      }
      else
      {
	 strcpy (temp, "/*\n");
      }

      if ((cbuf = strchr (temp, '\n')) != NULL)
         *cbuf = 0;

      /*
      ** Detab input record
      */

      cbuf = devinfo->carddat;
      len = strlen (temp);
      for (i1 = o1 = 0; i1 < len && o1 < 80; i1++)
      {
	 if (temp[i1] == '\t')
	 {
	    int o2;

	    for (o2 = 8 - (o1 % 8); o2; o2--)
	    {
	       cbuf[o1++] = ' ';
	    }
	 }
	 else
	 {
	    cbuf[o1++] = temp[i1];
	 }
      }
      cbuf[o1] = '\0';

      /*
      ** Pad with blanks
      */

      while (o1 < 80)
      {
	 cbuf[o1++] = ' ';
      }
      cbuf[80] = '\0';

#ifdef DEBUGCARD
      fprintf (stderr, "CARDREC:\n");
      HEXDUMP (stderr, cbuf, 80, 0);
#endif
      devinfo->cardndx = 0;
   }

   /*
   ** Get a character from the record
   */

   cdat = devinfo->carddat[devinfo->cardndx];
   devinfo->cardndx ++;

   /*
   ** Translate to card code
   */

   dat = tocard[cdat];
#ifdef DEBUGCARD
   fprintf (stderr, "CRU %04X STCR CNT %d DATA %02X(%c) - %04X\n",
	    devaddr, cnt, cdat, cdat, dat);
#endif
   return (dat);
}

/***********************************************************************
* cardboot - Boot from device.
***********************************************************************/

int
cardboot (Device *dev)
{
   FILE *lfd = NULL;
   int lp, clp;
   int i;
   int done;
   int reclen;
   uint8 inbuf[82];

   clp = lp = LOADADDRESS;

#ifdef DEBUGCARD
   lfd = fopen ("cardload.log", "w");
   fprintf (lfd, "cardboot: file = %s, loadpd = %d(%04X)\n",
	    dev->file, lp, lp);
#endif

   done = FALSE;
   for (i = 0; !done; i++)
   {
      if (fgets ((char *)inbuf, sizeof(inbuf), dev->infd) == NULL)
         break;
      reclen = 80;

      inbuf[reclen] = 0;
#if defined(DUMPLOADREC) && defined(DEBUGCARD)
      fprintf (lfd, "Record: %d\n", i+1);
      HEXDUMP (lfd, inbuf, reclen, 0);
#endif
      if (loadrec (lfd, inbuf, lp, &clp, FALSE) == EOF)
	 done = TRUE;
   }

#ifdef DEBUGCARD
   fclose (lfd);
#endif
   run = TRUE;

   return (0);
}
