/***********************************************************************
*
* simclock.c - Clock processing for the TI 990 Simulator.
*
* Changes:
*   02/27/14   DGP   Split from simio.c
*   06/02/15   DGP   Removed USS (z/OS) support.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>
#include <time.h>

#if defined (WIN32)
#define __TTYROUTINES 0
#include <conio.h>
#include <windows.h>
#include <signal.h>
#include <process.h>
#endif

#if defined(UNIX)
#include <unistd.h>
#include <sys/time.h>
#include <sys/param.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <pthread.h>
#endif

#ifndef HZ
#define HZ 100
#endif

#include "simdef.h"

extern int pancount;
extern int devcnt;
extern int usertclock;

extern unsigned long instcount;
extern unsigned long mipcount;

extern Device devices[MAXDEVICES];
extern InterruptQueue intqueue[16];

static int clockactive = FALSE;
static int clockthreadactive = FALSE;
static int clockthreadterminate = FALSE;

/***************************************************************************
* smallsleep - sleep in U seconds
* NOTE: Depending on the system the sleep time can vary widely.
***************************************************************************/

void
smallsleep (long Usec)
{
#if defined(UNIX)
#if defined(LINUX) || defined(SOLARIS)
#include <unistd.h>
   struct timespec req, rem;
#ifdef DEBUGCLOCK1
#include <time.h>
   static clock_t starttime;
   static clock_t curtime;
   static int ticks;
   static int first = TRUE;

   if (first)
   {
      first = FALSE;
      ticks = 0;
      starttime = time (NULL) + 1;
   }
#endif

   req.tv_sec = 0;
   req.tv_nsec = Usec * 1000;

   nanosleep (&req, &rem);
#ifdef DEBUGCLOCK1
   ticks++;
   curtime = time (NULL);
   if (curtime == starttime)
   {
      fprintf (stderr, "CLOCK: ticks = %d\n\r", ticks);
      starttime = curtime + 1;
      ticks = 0;
   }
#endif

#else
   struct timeval tv;

   tv.tv_sec = 0;
   tv.tv_usec = Usec;
   if (select (1, NULL, NULL, NULL, &tv) < 0)
   {
      if (ERRNO != EINTR && ERRNO != EAGAIN)
	 PERROR ("smallsleep: select failed");
   }
#endif
#endif /* UNIX */

#if defined(WIN32)
   Sleep ((unsigned long) (Usec) / 1000L);
#endif /* WIN32 */

#if defined(OS2)
   DosSleep ((unsigned long) (Usec) / 1000L);
#endif /* OS2 */

}

/***********************************************************************
* devicetimers - Count down device timers.
***********************************************************************/

static void
devicetimers (void)
{
   int i;

   for (i = 0; i < devcnt; i++)
   {
      switch (devices[i].type)
      {
      case CON9902:
      case PRINT9902:
      case SERIAL9902:
      case SERIALCI402:
         a9902timer (&devices[i]);
	 break;

      case SERIALCI401:
         ci401timer (&devices[i]);
	 break;

      case PRINTCI403:
      case SERIALCI403:
         ci403timer (&devices[i]);
	 break;

      default: ;
      }
   }
}

/***********************************************************************
* clocktask - run the system clock. The OS timers rely on a clock 
* interrupt rate of 120 times a second. The sleeptime is adjusted, based
* on load and systemic delays, to approach the desired 120.
***********************************************************************/

static void
clocktask (void *arg)
{
   long sleeptime;
   time_t start, stop;
   unsigned int prevcount;
   int panloops;
   int synclock;
   int ticks;
   int tickadj;

   synclock = FALSE;

   ticks = 0;
   panloops = 0;
   prevcount = 0;
   sleeptime = 8333;

   start = time (NULL);

   while (!clockthreadterminate)
   {
      smallsleep (sleeptime);
      ticks++;
      stop = time (NULL);

      if (clockactive)
      {
	 geninterrupt (CLOCKINT);
      }
      if (panloops++ >= 12)
      {
	 panloops = 0;
	 pancount = TRUE;
      }
      devicetimers();

      if ((stop - start) >= 1)
      {
	 mipcount = instcount - prevcount;
	 prevcount = instcount;

	 if (!synclock)
	 {
	    synclock = TRUE;
	 }
	 else
	 {
	    if (ticks < 119)
	    {
	       tickadj = 120 - ticks;
	       if (tickadj > 10)
		   sleeptime -= HZ/10;
	       else
		   sleeptime -= 1;
#ifdef DEBUGTICKS
	       printf ("ticks = %d, tickadj = %d\r\n", ticks, tickadj);
	       printf ("   sleeptime = %ld\r\n", sleeptime);
#endif
	    }
	    else if (ticks > 121)
	    {
	       tickadj = ticks - 120;
	       if (tickadj > 10)
		   sleeptime += HZ/10;
	       else
		   sleeptime += 1;
#ifdef DEBUGTICKS
	       printf ("ticks = %d, tickadj = %d\r\n", ticks, tickadj);
	       printf ("   sleeptime = %ld\r\n", sleeptime);
#endif
	    }
	 }
	 start = stop;
	 ticks = 0;
      }
   }
}

/***********************************************************************
* startclk - Start clock.
***********************************************************************/

void
startclk (int intenable)
{
   THREAD_HANDLE (handle);
#if defined(UNIX)
   struct sched_param thread_param;
   int thread_policy;
#endif

#ifdef DEBUGCKINST
   fprintf (stderr, "CKON called: enable = %s\n",
            intenable ?  "TRUE" : "FALSE");
#endif

   /*
   ** First call starts the thread
   */

   if (!clockthreadactive)
   {
      clockthreadterminate = FALSE;
      if (THREAD_CREATE (handle, clocktask, NULL))
      {
	 PERROR ("startclk: Can't start clock thread");
         exit (1);
      }

#if defined(UNIX) && !defined(CYGWIN)
      if (usertclock)
      {
	  if (pthread_getschedparam (handle, &thread_policy, &thread_param))
	  {
	     PERROR ("startclk: Can't get sched policy on clock thread");
	  }
#ifdef DEBUGCLOCK
	  fprintf (stderr, "startclk: thread_policy = %d\n", thread_policy);
	  fprintf (stderr, "startclk: sched_priority = %d\n",
		   thread_param.__sched_priority);
#endif
	  
#if defined(DARWIN)
	  thread_param.sched_priority = 20;
#else
	  thread_param.__sched_priority = 20;
#endif
	  if (pthread_setschedparam (handle, SCHED_FIFO, &thread_param))
	  {
	     PERROR ("startclk: Can't alter sched policy on clock thread");
	  }
      }
#endif
#if defined(WIN32) && !defined(MINGW)
      timeBeginPeriod (1);
#endif

      clockthreadactive = TRUE;
   }

   if (intenable)
      clockactive = TRUE;
}

/***********************************************************************
* stopclk - Stop clock.
***********************************************************************/

void
stopclk (int terminate)
{
#ifdef DEBUGCKINST
   fprintf (stderr, "CKOF/RSET called: terminate = %s\n",
            terminate ? "TRUE" : "FALSE");
#endif

   clockactive = FALSE;
   intqueue[CLOCKINT].count = 1;
   clrinterrupt (CLOCKINT);

   if (terminate)
   {
      clockthreadterminate = TRUE;
#if defined(WIN32) && !defined(MINGW)
      timeEndPeriod (1);
#endif
   }
}
