/***********************************************************************
*
* simdef.h - Simulator header for the TI 990 computer.
*
* Changes:
*   05/29/03   DGP   Original.
*   11/06/03   DGP   Added fpy* and v911* definitions.
*   11/11/03   DGP   Added System environment definitions.
*   07/14/04   DGP   Added more /12 defintions.
*   10/28/04   DGP   Added ECC Device.
*   12/06/04   DGP   Added v945 definitions.
*   03/30/05   DGP   Changed V911SCREENSIZE to 2048.
*   03/13/05   DGP   Bumped up the number of devices permitted.
*   03/28/05   DGP   Moved devices options into switches.
*   09/21/05   DGP   Added SWLOGOUT switch.
*   07/18/07   DGP   Added Hi/Lo intensity support.
*   11/07/08   DGP   Added Instruction tracing.
*   11/11/08   DGP   Added LCS support.
*   11/12/08   DGP   Extended /12 rom support.
*   09/07/11   DGP   Added disk geometry checks.
*   09/17/13   DGP   Added 9902 support.
*                    Revamp console support.
*   09/19/13   DGP   Added 9903 support.
*   10/14/13   DGP   Added CPU options.
*   02/24/14   DGP   Added device interrupt delay. Fixed .QBID bugs. 
*                    Added CRU expansion chassis support. Thanks to
*                    to Howard Wulf.
*   04/04/14   DGP   Added VDT 931 and CI401 support. Thanks to Howard Wulf.
*   04/14/14   DGP   Added CI403 support.
*   08/06/14   DGP   Fixes for WIN32 MinGW build.
*   11/21/14   DGP   Added MACROS for threads and mutex.
*   03/12/15   DGP   Added real device support for tape.
*   05/01/15   DGP   Added ASR 733 emulation support.
*   05/26/15   DGP   Change to use screen buffer/image.
*   07/21/15   DGP   Added more EIA I/F support.
*   01/28/16   DGP   Added real disk support.
*
***********************************************************************/

#include <stdio.h>
#if defined (WIN32)
#include <windows.h>
#include <process.h>
#endif
#if defined(UNIX)
#include <pthread.h>
#include <semaphore.h>
#include <termios.h>
#endif

/*
** Definitions
*/

#define VERSION "3.6.0"
#define INTRO "TI 990 Simulator"

#define NORMAL 0
#ifndef ABORT
#define ABORT  12
#endif

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif


/*
** Configuration definitions
*/

#define MEMSIZE 2*1024*1024	/* Maximum size of physical memory */
#define ROMSIZE 1024*8		/* Maximum size of ROM */
#define SYSMEMSIZE MEMSIZE+ROMSIZE /* System memory size */
#define WCSMEMSIZE 1024		/* Maximum size of WCS memory */

#define MAXDEVICES 100		/* Number of devices permitted */
#define MAXVIEW 100		/* Number of messages in view panel */
#define MAXVIEWLEN 120		/* Length of messages in view panel */
#define INTQDEPTH 20		/* Depth of the interrupt queue */
#define MAXDEVPATHLEN 256

#define TERMRINGLEN 1920	/* Serial device ring length */

#define CI4CHANS 4		/* CI403 number of channels */
#define CI4REGS 8		/* CI403 number of registers */
#define CI4RINGLEN 512		/* CI403 W3 output ring length */

#define TOPLINE       1
#define REGISTERLINE  3
#define DISPLINE      5
#define BREAKLINE     6
#define WORKSPACELINE 8
#define OUTPUTLINE    14

#define TRCBUFMAX     50

/*
** System definitions.
*/

#define TILINESTART 0xF800
#define TILINEEND   0xFC00
#define ROMSTART    0xFC00
#define TPCSSTART   0x1F0000

#define PRVCRU   0x1C00
#define CRUEND   0x1FFE

#define LOADADDRESS 0xA0

#define EOFSYM ':'
#define HEXSYM '>'

/*
** Default device addresses and interrupt levels.
*/

#define TERMINAL 0x0000		/* Serial Terminal CRU */
#define TERM9902 0x1700		/* 990/5 and 990/10A Serial Terminal CRU */

#define INTRCRU  0x17C0		/* 990/5 port interrupt control CRU */
#define ROM5CRU  0x17E0		/* 990/5 rom control CRU */

#define EXP1CRU  0x1F00		/* CRU Expansion chassis card 1 CRU */
#define EXP2CRU  0x1F20		/* CRU Expansion chassis card 2 CRU */
#define EXPMASK  0x87FF		/* CRU Expansion chassis mask */

#define BKPTCRU  0x1F80		/* Breakpoint CRU */
#define MAPPER   0x1FA0		/* Memory Mapper CRU */
#define ERRCRU   0x1FC0		/* Errors CRU */
#define PANEL    0x1FE0		/* Front Panel CRU */

#define ERRINT   2		/* Error interrupt level */
#define CLOCKINT 5		/* Clock interrupt level */
#define TERMINT  6		/* Serial Terminal interrupt level */
#define INT9902  8		/* 990/5 and 990/10A Terminal interrupt */

/*
** ERRCRU bits
*/

#define ERRCPU12   0x0001 /* 990/12 CPU ID bit */
#define ERRCPU10A  0x0002 /* 990/10A CPU ID bit */
#define ERRAOVR    0x0010 /* Arithmetic overflow */
#define ERR12MS    0x0020 /* 12MS clock */
#define ERRBKPT    0x0040 /* Breakpoint */
#define ERRSOVR    0x0080 /* Stack Over/Under flow */
#define ERRWRIT    0x0100 /* Write in non-write memory */
#define ERREXEC    0x0200 /* Exec in non-exec memory */
#define ERRMAP     0x0800 /* Memory mapping error */
#define ERRTIP     0x1000 /* TILINE memory parity error */
#define ERRILO     0x2000 /* Illegal operation */
#define ERRPIN     0x4000 /* Priv instruction */
#define ERRTIT     0x8000 /* TILINE timeout */

/*
** MAPCRU bits
*/

#define MAPENABLE 0x0008 /* Enable memory mapping */
#define MAPERROR  0x0010 /* Clear memory mapping error */

/*
** Error trace bits
*/

#define TRACE_INSTFETCH	0x00100000
#define TRACE_WSACCESS	0x00200000
#define TRACE_TLRW	0x00400000
#define TRACE_TLACCESS	0x00800000
#define TRACE_WSRW	0x01000000
#define TRACE_PRIVINST	0x02000000
#define TRACE_ILLOP	0x04000000
#define TRACE_MAPERR	0x08000000
#define TRACE_MEMDATA	0x10000000
#define TRACE_TLTIME	0x20000000
#define TRACE_EXECVIO	0x40000000
#define TRACE_WRITVIO	0x80000000

/*
** Data type definitions
*/

#define int8            signed char
#define int16           short
#define int32           int
typedef int             t_stat;                         /* status */
typedef int             t_bool;                         /* boolean */
typedef unsigned char   uint8;
typedef unsigned short  uint16;
typedef unsigned int    uint32, t_addr;                 /* address */

#if defined (WIN32)                                     /* Windows */
#define t_int64 __int64
#elif defined (__ALPHA) && defined (VMS)                /* Alpha VMS */
#define t_int64 __int64
#elif defined (__ALPHA) && defined (__unix__)           /* Alpha UNIX */
#define t_int64 long
#else                                                   /* default GCC */
#define t_int64 long long
#endif                                                  /* end OS's */
typedef unsigned t_int64        t_uint64, t_value;      /* value */
typedef t_int64                 t_svalue;               /* signed value */

/*
** Mutex macros
*/

#ifdef UNIX
#define MUTEX_DEFINE(m) pthread_mutex_t (m)
#define MUTEX_INIT(m) pthread_mutex_init(&(m), NULL)
#define MUTEX_LOCK(m) pthread_mutex_lock (&(m))
#define MUTEX_UNLOCK(m) pthread_mutex_unlock (&(m))
#endif

#ifdef WIN32
#define MUTEX_DEFINE(m) HANDLE (m)
#define MUTEX_INIT(m) (((m) = CreateMutex (NULL, FALSE, NULL)) ? 0 : 1)
#define MUTEX_LOCK(m) (WaitForSingleObject ((m), INFINITE) != WAIT_OBJECT_0)
#define MUTEX_UNLOCK(m) ReleaseMutex ((m))
#endif

/*
** Thread macros
*/

#ifdef UNIX
#define THREAD_HANDLE(h) pthread_t (h)
#define THREAD_CREATE(h,t,a) \
   pthread_create (&(h), NULL, (void *(*)(void *)) (t), (void *)(a))
#define THREAD_EXIT(s) pthread_exit((void *)&(s))
#endif

#ifdef WIN32
#define THREAD_HANDLE(h) uintptr_t (h)
#define THREAD_CREATE(h,t,a) \
   ((((h) = _beginthread ((t), 0, (void *)(a))) > 0) ? 0 : 1)
#define THREAD_EXIT(s) _endthread()
#endif

/*
** Memory
*/

#define MSB 0
#define LSB 1

#define MEMLEN		TILINESTART
#define MAPMEMLEN	1024*1024

#define MAPSIZE 3

typedef struct
{
   uint16 l1;
   uint16 b1;
   uint16 l2;
   uint16 b2;
   uint16 l3;
   uint16 b3;
} MapFile;

/*
** Mapper CRU bits
*/

#define MAP_ENABLE 0x0008
#define MAP_RESET  0x0010
#define MAP_LATCH1 0x0080
#define MAP_LATCH2 0x0040
#define MAP_LATCH3 0x0020

/*
** 990/12 CPU memory protection
*/

#define MEM_READ	0x1
#define MEM_WRITE	0x2
#define MEM_EXEC	0x3

/*
** User specifed CPU options
*/

#define OPT_NOSEGPROT	0x00001
#define OPT_NOTILINE	0x00002

/*
** Access macros
*/

#define SRC 1
#define DST 0
#define NOLD -1
#define EXEC -2

extern uint16 getinst (int);
extern uint16 getreg (uint16);
extern void putreg (uint16, uint16);
extern uint16 getmem (uint16, int);
extern uint8 getmemb (uint16, int);
extern void putmem (uint16, uint16, int);
extern void putmemb (uint16, uint8, int);

#define GETINST(x) getinst(x)
#define INCPC(v) pcreg+=(v),mpcreg+=(v)
#define DECPC(v) pcreg-=(v),mpcreg-=(v)

#define GETREG(r) getreg(r)
#define PUTREG(r,v) putreg(r,v)

#define GETMEM(m,sd) getmem(m,sd)
#define GETMEMB(m,sd) getmemb(m,sd)

#define PUTMEM(m,v,sd) putmem(m,v,sd)
#define PUTMEMB(m,v,sd) putmemb(m,v,sd)

#define R0  GETREG(0)
#define R1  GETREG(1)
#define R2  GETREG(2)
#define R3  GETREG(3)
#define R4  GETREG(4)
#define R5  GETREG(5)
#define R6  GETREG(6)
#define R7  GETREG(7)
#define R8  GETREG(8)
#define R9  GETREG(9)
#define R10 GETREG(10)
#define R11 GETREG(11)
#define R12 GETREG(12)
#define R13 GETREG(13)
#define R14 GETREG(14)
#define R15 GETREG(15)

#define GETMEM0(m) (((memory[(m)+MSB] & 0xFF) << 8)|(memory[(m)+LSB] & 0xFF))
#define GETMEMB0(m) (memory[(m)] & 0xFF)

#define PUTMEM0(m,v) \
   { memory[(m)+MSB] = (v) >> 8 & 0xFF; memory[(m)+LSB] = (v) & 0xFF; }
#define PUTMEMB0(m,v) memory[(m)] = (v) & 0xFF

#define DISPMEM GETMEM0(dispaddr)

/*
** TILINE access macros
*/

#define GETW0(d)   GETMEM0((d)->devaddr+TPCSSTART)
#define PUTW0(d,v) PUTMEM0((d)->devaddr+TPCSSTART,v)

#define GETLEN(d)   GETMEM0((d)->devaddr+TPCSSTART+8)
#define PUTLEN(d,v) PUTMEM0((d)->devaddr+TPCSSTART+8,v)

#define GETADDR(d) \
(GETMEMB0((d)->devaddr+TPCSSTART+13)<<16)|GETMEM0((d)->devaddr+TPCSSTART+10)
#define PUTADDR(d,v) \
{ PUTMEMB0((d)->devaddr+TPCSSTART+13,v>>16); \
  PUTMEM0((d)->devaddr+TPCSSTART+10,v&0xFFFF) }

#define PUTW7(d,v) PUTMEM0((d)->devaddr+TPCSSTART+14,v)
#define GETW7(d)   GETMEM0((d)->devaddr+TPCSSTART+14)

/*
** Status register set and check
*/

#define CLR_LGT    (statreg &= 0x7FFF)
#define CLR_AGT    (statreg &= 0xBFFF)
#define CLR_EQ     (statreg &= 0xDFFF)
#define CLR_CARRY  (statreg &= 0xEFFF)
#define CLR_OVER   (statreg &= 0xF7FF)
#define CLR_ODDP   (statreg &= 0xFBFF)
#define CLR_XOP    (statreg &= 0xFDFF)
#define CLR_NONPRV (statreg &= 0xFEFF)
#define CLR_MAP    (statreg &= 0xFF7F)
#define CLR_PROT   (statreg &= 0xFFBF)
#define CLR_OVERI  (statreg &= 0xFFDF)
#define CLR_WCS    (statreg &= 0xFFEF)

#define SET_LGT    (statreg |= 0x8000)
#define SET_AGT    (statreg |= 0x4000)
#define SET_EQ     (statreg |= 0x2000)
#define SET_CARRY  (statreg |= 0x1000)
#define SET_OVER   (statreg |= 0x0800)
#define SET_ODDP   (statreg |= 0x0400)
#define SET_XOP    (statreg |= 0x0200)
#define SET_NONPRV (statreg |= 0x0100)
#define SET_MAP    (statreg |= 0x0080)
#define SET_PROT   (statreg |= 0x0040)
#define SET_OVERI  (statreg |= 0x0020)
#define SET_WCS    (statreg |= 0x0010)

#define IS_LGT    (statreg & 0x8000)
#define IS_AGT    (statreg & 0x4000)
#define IS_EQ     (statreg & 0x2000)
#define IS_CARRY  (statreg & 0x1000)
#define IS_OVER   (statreg & 0x0800)
#define IS_ODDP   (statreg & 0x0400)
#define IS_XOP    (statreg & 0x0200)
#define IS_NONPRV (statreg & 0x0100)
#define IS_MAP    (statreg & 0x0080)
#define IS_PROT   (statreg & 0x0040)
#define IS_OVERI  (statreg & 0x0020)
#define IS_WCS    (statreg & 0x0010)

#define SET_MASK(m) (statreg = (statreg & 0xFFF0) | ((m) & 0x000F))
#define GET_MASK    (statreg & 0x000F)

#define GET_MAP     (IS_MAP >> 7)

/*
** Instruction types
*/

#define MAX_INST_TYPES 21
enum optypes
{
   TYPE_1=0, TYPE_2,  TYPE_3,  TYPE_4,  TYPE_5,  TYPE_6,  TYPE_7,
   TYPE_8,   TYPE_9,  TYPE_10, TYPE_11, TYPE_12, TYPE_13, TYPE_14,
   TYPE_15,  TYPE_16, TYPE_17, TYPE_18, TYPE_19, TYPE_20, TYPE_21,
   TYPE_ILL
};

/*
** Panel actions
*/

enum actions
{
   NULL_ACTION=0, QUIT_ACTION,   STEP_ACTION,   RUN_ACTION,    HELP_ACTION,
   LOAD_ACTION,   PC_ACTION,     WP_ACTION,     DISP_ACTION,   CLEAR_ACTION
};

#define IDLEINST 0x0340

/*
** Object tags
*/

#define BINIDT_TAG	01
#define IDT_TAG		'0'
#define ABSENTRY_TAG	'1'
#define RELENTRY_TAG	'2'
#define RELEXTRN_TAG	'3'
#define ABSEXTRN_TAG	'4'
#define RELGLOBAL_TAG	'5'
#define ABSGLOBAL_TAG	'6'
#define CKSUM_TAG	'7'
#define NOCKSUM_TAG	'8'
#define ABSORG_TAG	'9'
#define RELORG_TAG	'A'
#define ABSDATA_TAG	'B'
#define RELDATA_TAG	'C'
#define LOADBIAS_TAG	'D'
#define EXTNDX_TAG	'E'
#define EOR_TAG		'F'

#define RELSYMBOL_TAG	'G'
#define ABSSYMBOL_TAG	'H'
#define PGMIDT_TAG	'I'
#define CMNSYMBOL_TAG	'J'
#define COMMON_TAG	'M'
#define CMNDATA_TAG	'N'
#define CMNORG_TAG	'P'
#define CBLSEG_TAG	'Q'
#define DSEGORG_TAG	'S'
#define DSEGDATA_TAG	'T'
#define LOAD_TAG	'U'
#define RELSREF_TAG	'V'
#define CMNGLOBAL_TAG	'W'
#define CMNEXTRN_TAG	'X'
#define ABSSREF_TAG	'Y'
#define CMNSREF_TAG	'Z'

#define LRELSYMBOL_TAG	'i'
#define LABSSYMBOL_TAG	'j'
#define LRELEXTRN_TAG	'l'
#define LABSEXTRN_TAG	'm'
#define LRELGLOBAL_TAG	'n'
#define LABSGLOBAL_TAG	'o'
#define LLOAD_TAG	'p'
#define LRELSREF_TAG	'q'
#define LABSSREF_TAG	'r'

#define CHARSPERREC	66  /* Chars per object record */
#define WORDTAGLEN	5   /* Word + Object Tag length */
#define BINWORDTAGLEN	3   /* Binary Word + Object Tag length */
#define EXTRNLEN	11  /* Tag + SYMBOL + addr */
#define GLOBALLEN	11  /* Tag + SYMBOL + addr */
#define IDTTAGLEN	13  /* Tag + IDT + length */
#define SEQUENCENUM	74  /* Where to put the sequence */
#define SYMLEN		6   /* REF/DEF Symbol length */
#define IDTLEN		8   /* IDT length */
#define MAXSYMLEN 	32  /* LREF/LDEF Symbol length */

/*
** Device table entry.
*/

/* Character definitions */

#define BACKSPACE  0x08
#define BLANK      0x20
#define ESCAPE     0x1B
#define TILDE      0x7E
#define DBACKSPACE 0x7F

/* Tape definitions */

#define TAPEMAXREC	32768

#define TAPEERR		-1
#define TAPEEOF		-2
#define TAPEEOT		-4

/* Serial 931/940 VDT definition */

#define VDT_MIN_ROW    0
#define VDT_MAX_ROW    23
#define VDT_NUM_ROWS   (VDT_MAX_ROW + 1)

#define VDT_MIN_COL    0 
#define VDT_MAX_COL    79
#define VDT_NUM_COLS   (VDT_MAX_COL + 1)

#define VDT_STATUS_PRIMARY 0
#define VDT_STATUS_ALTERNATE 1

#define VDT_ATTR_BOLD      0x01
#define VDT_ATTR_REVERSE   0x02
#define VDT_ATTR_UNDERLINE 0x04
#define VDT_ATTR_HIDDEN    0x08
#define VDT_ATTR_BLINK     0x10
#define VDT_ATTR_DEFAULT   0x40

#define VDT_BUF_LEN 256

enum VDTCommState_t
{
   VDT_COMMSTATE_CONN = 0,
   VDT_COMMSTATE_DISC,
   VDT_COMMSTATE_RDY,
   VDT_COMMSTATE_BUSY,
   VDT_COMMSTATE_RING,
   VDT_COMMSTATE_LOOP
};

typedef struct
{
   FILE *printfd;
   uint8 *screen;
   int   Row;
   int   Col;
   int   LineExtentRow; /* VDT_MAX_ROW */ /* Insert/Delete Line */
   int   CharExtentRow; /* VDT_MAX_ROW */ /* Insert/Delete Char */
   int   CharExtentCol; /* VDT_MAX_COL*/ /* Insert/Delete Char */
   int   CharExtentEnabled; /* Insert/Delete Char */
   int   AlternateCharacterSet;
   int   CommDTR;
   int   CommDSR;
   int   CommDCD;
   int   CommState;
   int   StatusCol;
   int   SaveCol;
   int   SaveRow;
   uint8 Fill; /* ' ' */
   uint8 Attr; /* 0x40 */
   uint8 VideoOn; /* 0x02 */
   uint8 CursorOn; /* 0x01 */
   uint8 LockOn;
   uint8 BlinkOn;
   uint8 GraphicCharSet;
   uint8 ABM_Count;
   uint8 ABM_Buffer[32];
   uint8 Buffer[VDT_NUM_ROWS][VDT_NUM_COLS];
   uint8 Shadow[VDT_NUM_ROWS][VDT_NUM_COLS];
   uint8 StatusLine[2][VDT_NUM_COLS];
   uint8 StatusActive;
   char printfile[MAXDEVPATHLEN+2];
} VDTInfo;

/* Serial ASR733 definition */

#define A733PLAYRDY 0x0001
#define A733PLAYERR 0x0002
#define A733BOEO1   0x0004
#define A733BOEO2   0x0008
#define A733RCRDRDY 0x0010
#define A733PRTRRDY 0x0020
#define A733BIAS    0x0040

#define A733RCRDC1  0x0100
#define A733RCRDC2  0x0200
#define A733RCRDON  0x0400
#define A733PLAYON  0x0800
#define A733ADCON   0x1000

#define A733BLKLEN  86

typedef struct
{
   FILE *cs1fd;
   FILE *cs2fd;
   int  status;
   int  baseunit;
   char cs1file[MAXDEVPATHLEN+2];
   char cs2file[MAXDEVPATHLEN+2];
} ASRInfo;

/* Size of 911 onboard RAM (controls screen size) */

#define V911SCREENSIZE 2048

/* FD800 geometry */

#define FPYSECTTRK 26
#define FPYTRKCYL  77
#define FPYSECTLEN 128

/* Special file descriptors */

#define DEV_NULL ((FILE *)1)
#define DEV_LOOP ((FILE *)2)
#define DEV_VDT  ((FILE *)3)
#define DEV_TELNET ((FILE *)4)

/* Device switch masks. */

#define SWUPCASE  0x0001
#define SWPROTECT 0x0002
#define SWCREATE  0x0004
#define SWLOGOUT  0x0008
#define SWEMUTERM 0x0010
#define SWPUTRIM  0x0020
#define SWTELNET  0x0040
#define SWSTATUS  0x0080
#define SWSWITCH  0x0100
#define SWPRINTER 0x0200
#define SWRAWTERM 0x0400
#define SWNETDEV  0x0800

/* Devices supported by the system */

enum dev_t
{
   CONEIA=1,
   CON9902,
   CON9903,
   CARDREADER,
   CARDPUNCH,
   CASSETTE,
   PRINTEIA,
   PRINT9902,
   PRINT9903,
   PRINTCI402,
   PRINTCI403,
   PRINTPAR,
   FLOPPY,
   SERIALEIA,
   SERIAL9902,
   SERIAL9903,
   SERIALCI401,
   SERIALCI402,
   SERIALCI403,
   DISK,
   TAPE,
   VDT911
};

/* TILINE Disk device */

typedef struct
{
   int type;
   int cyls;
   int heads;
   int secttrk;
   int sectlen;
   int overhead;
   int tracklen;
   int curcyl;
   int curhead;
   int cursect;
   int geoverhead;
   int share;
   char model[MAXDEVPATHLEN+2];
} DiskInfo;

/* TILINE Tape device */

typedef struct
{
   long curpos;
   int online;
   uint8 record[TAPEMAXREC];
} TapeInfo;

/* Floppy disk device */

typedef struct
{
   long track;
   long sector;
   long trklocation;
   long seclocation;
   int laststat;
   int lastcmd;
   int ramload;
} FpyInfo;

/* 911 VDT device */

typedef struct
{
   int kbchar;
   int active;
   int kbint;
   int wrdsel;
   int oldsel;
   int cursorpos;
   int dispenable;
   uint8 dualenable;
   uint8 intenenable;
   uint8 intensity[V911SCREENSIZE+2];
   char screen[V911SCREENSIZE+2];
} VDT911Info;

/* Card Reader/Punch device */

typedef struct
{
   int cardinit;
   int cardeof;
   int cardndx;
   char carddat[82];
} CARDInfo;

/* Serial device (EIA, 9902, 9903, CI402) */

typedef struct
{
   int unit;
   int testmode;
   int loadmode;
   int rtsset;
   int dtrset;
   int dsrset;
   int dcdset;
   int wrqset;
   int ringsend;
   int ringsent;
   int ringstep;
   int mod5port;
   int xmtintenabled;
   int xmtempty;
   int commenabled;
   int timerenb;
   int dscenabled;
   int dscint;
   int timerint;
   int timecnt;
} SerialInfo;

/* Serial device CI401 */

typedef struct
{
    uint8 AddressRegister;
    uint8 WordIndex;
    uint8 OutputIndex;
    uint8 InputIndex;
    uint8 ClockTicks;
    uint8 StartupSequence;
    uint8 StartupStep;
    uint8 InputWord[7];
    uint8 OutputWord[6];
} CI401Info;

/* Serial device CI403 */

typedef struct
{
   int unit;
   int timerenb;
   int timecnt;
   int timerint;
   int mreset;
   int W3head;
   int W3tail;
   uint16 oldW0;
   uint16 W3ring[CI4RINGLEN];
   uint8 regs[CI4REGS];
   MUTEX_DEFINE (W3ringmutex);
} CI403Info;

/* Terminal Ring buffer definition */

typedef struct
{
   int head;
   int tail;
   char data[TERMRINGLEN];
   MUTEX_DEFINE (mutex);
} TermRing;

/* Serial terminal device */

typedef struct
{
   int started;
   int connected;
   int kbchar;
   int active;
   int kbint;
   int inescape;
   TermRing inring;		/* Data to host */
   TermRing outring;		/* Data to "terminal" */
   TermRing prcring;		/* Data from host, to process */
#if defined(WIN32)
   int termattr[2];
#endif
#if defined(UNIX)
   struct termios termattr;
#endif
   union
   {
      VDTInfo vdtinfo;
      ASRInfo asrinfo;
   } terminal;
   union
   {
      SerialInfo serialinfo;
      CI401Info ci401info;
      CI403Info ci403info;
   } controller;
} TermInfo;

/* Device table entry */

typedef struct
{
   FILE *infd;
   FILE *outfd;
   int sfd;
   int type;
   int unit;
   int select;
   int count;
   int inchar;
   int outchar;
   int cbreak;
   int intenabled;
   int viewndx;
   int switches;
   int emutype;
   int xcardsel;
   int runtermd;
   int realdevice;
   uint16 devaddr;
   uint16 extaddr;
   uint16 ctladdr;
   uint16 intlvl;
   uint16 intdelay;
   uint16 xcarddat;
   char file[MAXDEVPATHLEN+2];
   char name[80];
#ifdef WIN32
   int charqueuecnt;
   int charqueuendx;
   char charqueue[10];
#endif
   union
   {
      DiskInfo dskinfo;
      TapeInfo tapeinfo;
      FpyInfo fpyinfo;
      VDT911Info vdt911info;
      CARDInfo cardinfo;
      TermInfo terminfo;
   } info;
} Device;

/*
** Device Options 
*/

#define EMUH19 1
#define EMU931 2
#define EMU940 3
#define EMU733 4

typedef struct
{
   FILE *printfd;
   FILE *cs1fd;
   FILE *cs2fd;
   int switches;
   int emutype;
   char file1[MAXDEVPATHLEN+2];
   char file2[MAXDEVPATHLEN+2];
} DeviceOptions;

/*
** Trace Instruction 
*/

typedef struct
{
   int  pc;
   int  mpc;
   int  wp;
   int  st;
   int  inst;
   int  ops[3];
   int  regs[16];
   char dis[132];
} TraceInst;

/*
** Interrupt Queue definition
*/

typedef struct
{
   int count;
   int bottom, top;
   int xcardsel;
   uint16 xcarddat[INTQDEPTH];
} InterruptQueue;

/*
** Externals
*/

/* sim990 externals */

extern char *getnum (char *, long *);
extern void panel (void);
extern void romtowcs (void);
extern void printinst (char *, int);

/* simloader externals */

extern int loadrec (FILE *, uint8 *, int, int *, int);
extern int simloader (char *, int);

/* simops externals */

extern void stepprog (void);
extern void runprog (void);
extern uint16 getaddr (uint16, uint16, int);
extern uint8 getbyte (uint16, uint16, int, int);
extern uint16 getword (uint16, uint16, int, int);
extern void putaddr (uint16, uint16, uint16);
extern void putword (uint16, uint16, uint16, int);
extern void cmpbytezero (uint8);
extern void cmpwordzero (uint16);
extern void tracememory (uint32, uint32);
extern uint32 mapaddr (uint16, uint16, uint16, uint32, uint32);
extern int decodeinst (uint16);

/* simfops externals */

extern int proc6fltop (uint16);
extern int proc7fltop (uint16);

/* simeops externals */

extern void cmplongzero (uint32);
extern uint32 getlong (uint16, uint16, int, int);
extern int proc11op (uint16);
extern int proc12op (uint16);
extern int proc13op (uint16);
extern int proc14op (uint16);
extern int proc15op (uint16);
extern int proc16op (uint16);
extern int proc17op (uint16);
extern int proc18op (uint16);
extern int proc19op (uint16);
extern int proc20op (uint16);
extern int proc21op (uint16);

/* simio externals */

extern void setbitone (uint16);
extern void setbitzero (uint16);
extern void testbit (uint16);
extern void loadcru (uint16, uint16);
extern uint16 storecru (uint16);

extern void attachdev (char *);
extern void resetdev (void);
extern int bootdev (char *);
extern int realfd (FILE *);

extern int gettiline (uint32, int);
extern int puttiline (uint32, uint16, int);

extern void checkinterrupts (void);
extern int initinterrupt (void);
extern int geninterrupt (uint16 lvl);
extern int gendevinterrupt (Device *);
extern int chkinterrupt (uint16 lvl);
extern void clrinterrupt (uint16 lvl);

/* simterm externals */

extern void prtpanel (int, int);
extern int  putview (int, int);

extern int serialgetchar (Device *);
extern void serialputchar (Device *, uint16);
extern int serialinput (Device *);
extern void serialoutput (Device *, uint16);

extern int ttinit (void);
extern int ttraw (int, int);
extern int ttgetattr (int, void *);
extern int ttsetattr (int, void *);
extern int ttcmdstate (void);
extern int ttrunstate (void);
extern int ttclose (void);

/* simh19 externals */

extern void H19init (Device *);
extern void H19input (Device *, int);
extern void H19process (Device *);
extern int  H19display (Device *, int);

/* simclock externals */

extern void startclk (int);
extern void stopclk (int);
extern void smallsleep (long Usec);

/* simfpy externals */

extern void fpyseto (Device *, uint8);
extern void fpysetz (Device *, uint8);
extern void fpytb (Device *, uint8);
extern void fpyldcr (Device *, uint16, uint16);
extern uint16 fpystcr (Device *, uint16);
extern int fpyboot (Device *);

/* simeia externals */

extern void eiaseto (Device *, uint16, uint8);
extern void eiasetz (Device *, uint16, uint8);
extern void eiatb (Device *, uint16, uint8);
extern void eialdcr (Device *, uint16, uint16, uint16);
extern uint16 eiastcr (Device *, uint16, uint16);

/* simcard externals */

extern void cardseto (Device *, uint16, uint8);
extern void cardsetz (Device *, uint16, uint8);
extern void cardtb (Device *, uint16, uint8);
extern void cardldcr (Device *, uint16, uint16, uint16);
extern uint16 cardstcr (Device *, uint16, uint16);
extern int cardboot (Device *);

/* simpunch externals */

extern void punchseto (Device *, uint16, uint8);
extern void punchsetz (Device *, uint16, uint8);
extern void punchtb (Device *, uint16, uint8);
extern void punchldcr (Device *, uint16, uint16, uint16);
extern uint16 punchstcr (Device *, uint16, uint16);

/* sim9902 externals */

extern void a9902timer (Device *);
extern void a9902seto (Device *, uint16, uint8);
extern void a9902setz (Device *, uint16, uint8);
extern void a9902tb (Device *, uint16, uint8);
extern void a9902ldcr (Device *, uint16, uint16, uint16);
extern uint16 a9902stcr (Device *, uint16, uint16);

/* sim9903 externals */

extern void a9903seto (Device *, uint16, uint8);
extern void a9903setz (Device *, uint16, uint8);
extern void a9903tb (Device *, uint16, uint8);
extern void a9903ldcr (Device *, uint16, uint16, uint16);
extern uint16 a9903stcr (Device *, uint16, uint16);

/* sim401 externals */

extern void ci401timer (Device *);
extern void ci401seto (Device *, uint16, uint8);
extern void ci401setz (Device *, uint16, uint8);
extern void ci401tb (Device *, uint16, uint8);
extern void ci401ldcr (Device *, uint16, uint16, uint16);
extern uint16 ci401stcr (Device *, uint16, uint16);

/* sim403 externals */

extern FILE *ci403open (char *, char *, int *);
extern int ci403start (Device *);
extern void ci403timer (Device *);
extern void ci403putcmd (Device *, uint32, uint16);
extern uint16 ci403getcmd (Device *, uint32);

/* simprint externals */

extern void printputchar (Device *, uint8);

extern void printseto (Device *, uint16, uint8);
extern void printsetz (Device *, uint16, uint8);
extern void printtb (Device *, uint16, uint8);
extern void printldcr (Device *, uint16, uint16, uint16);
extern uint16 printstcr (Device *, uint16, uint16);

/* sim911 externals */

extern FILE *v911open (char *, char *, char *, int *);
extern int v911start (Device *);
extern void v911seto (Device *, uint8);
extern void v911setz (Device *, uint8);
extern void v911tb (Device *, uint8);
extern void v911ldcr (Device *, uint16, uint16);
extern uint16 v911stcr (Device *, uint16);

/* simdsk externals */

extern int dskopen (Device *);
extern int dskdocmd (Device *);
extern int dskboot (Device *);
extern void dskreset (Device *);

/* simtape externals */

extern int tapeopen (Device *);
extern int tapedocmd (Device *);
extern int tapeboot (Device *);
extern void tapereset (Device *);

/* sim931 externals */

extern void v931init (Device *);
extern void v931input (Device *, int);
extern void v931process (Device *);
extern int  v931display (Device *, int);

/* simvdt externals */

extern void VDT_ScrollScreen (Device *, int);
extern void VDT_InsertDelete (Device *, int, int);
extern void VDT_AdjustCursor (Device *);
extern void VDT_EraseArea (Device *, int, int, int, int);
extern void VDT_PutChar (Device *, uint8, uint8);
extern void VDT_PutAttr (Device *, uint8, uint8);

extern uint8 *VDT_DisplayClear (Device *, uint8 *);
extern uint8 *VDT_DisplayPosition (Device *, uint8 *, int, int);
extern uint8 *VDT_DisplayAttr (Device *, uint8 *, uint8);
extern uint8 *VDT_DisplayScreen (Device *, int, uint8 *, int, int);

/* sima733 externals */

extern void a733process (Device *);
extern int a733boot (Device *, char *, int);

/* simtermd externals */

extern FILE *termdopen (char *, char *, char *, int *);
extern int termdstart (Device *);
extern void termdstop (Device *);

extern int termdringempty (Device *, TermRing *);
extern void termdputring (Device *, TermRing *, int);
extern int termdgetring (Device *, TermRing *);
extern int termdgetcond (Device *, TermRing *, int);

extern void termdsend (Device *, uint8 *, int);
extern void termdeatchar (Device *, int, uint8 *);
extern int termdgetchar (Device *, int, uint8 *);

extern void termddoiac (int);
extern int Nwrite (int fd, uint8 *buf, int count);
extern int Nread (int fd, uint8 *buf, int count);

/* simdis externals */

extern void addidt (char *, int);
extern void addsym (char *, int, int);
extern int  idtlookup (char *);
extern int  symlookup (int, char *);
extern void dumpsyms (char *);

extern void disinst (TraceInst *);

/*
** HEXDUMP - Macro used in debugging
*/

extern void hexdump (FILE *, char *, int, int);
#define HEXDUMP(file,ptr,size,offset) hexdump(file,ptr,offset,size)

/*
** Define things specific to the system environment
*/

#if defined(VAXVMS)
#if defined(TGVTCP)
#define ERRNO		socket_errno
#define CLOSE		socket_close
#define READ		socket_read
#define WRITE		socket_write
#define PERROR		socket_perror
#endif /* TGVTCP */
#if defined(UCXTCP)
/*
** Select uses bit masks of file descriptors in longs.
** These macros manipulate such bit fields (the filesystem macros use
** chars). FD_SETSIZE may be defined by the user, but the default here
** should be >= NOFILE (param.h).
*/
#ifndef FD_SETSIZE
#define FD_SETSIZE 256
#endif
#ifndef NBBY
#define NBBY 8
#endif

#define NFDBITS (sizeof(fd_mask) * NBBY)	/* bits per mask */
#ifndef howmany
#define howmany(x, y)   (((x)+((y)-1))/(y))
#endif

#if defined(VMSV5) || defined(VMSV6)
typedef long fd_mask;

typedef struct fd_set
{
   fd_mask fds_bits[howmany (FD_SETSIZE, NFDBITS)];
}
fd_set;
#endif /* VMSV5 || VMSV6 */

#define  FD_SET(n, p)   ((p)->fds_bits[((unsigned)n)/NFDBITS] \
                        |= (1 << (((unsigned)n) % NFDBITS)))
#define  FD_CLR(n, p)   ((p)->fds_bits[((unsigned)n)/NFDBITS] \
                        &= ~(1 << (((unsigned)n) % NFDBITS)))
#define  FD_ISSET(n, p) ((p)->fds_bits[((unsigned)n)/NFDBITS] \
                        & (1 << (((unsigned)n) % NFDBITS)))
#define FD_ZERO(p) memset((char *)(p), '\0', sizeof(*(p)))

#define APPTCPBLOCKMODE	0
#endif /* UCXTCP */
#endif /* VAXVMS */

#if defined(OS2)
#define ERRNO		sock_errno()
#define CLOSE		soclose
#define PERROR		psock_errno
#endif /* OS2 */

#if defined(WIN32) 
#include <winsock.h>

#define CLOSE		closesocket
#define IOCTL(a,b,c,d)	ioctlsocket((a),(b),(unsigned long *)(c))
#define ERRNO		WSAGetLastError()
#define PERROR(m)	fprintf (stderr, "%s: errno = %d\n", (m), ERRNO);
typedef unsigned long	caddr_t;
#ifndef ETIMEDOUT
#define ETIMEDOUT	WSAETIMEDOUT
#endif
#ifndef EADDRINUSE
#define EADDRINUSE	WSAEADDRINUSE
#endif
#ifndef ECONNREFUSED
#define ECONNREFUSED	WSAECONNREFUSED
#endif
#ifndef EINPROGRESS
#define EINPROGRESS	WSAEWOULDBLOCK
#endif
#ifndef ENOTCONN
#define ENOTCONN	WSAENOTCONN
#endif
#ifndef ECONNREFUSED
#define ECONNREFUSED	WSAECONNREFUSED
#endif
#ifndef ENOBUFS
#define ENOBUFS	        WSAENOBUFS
#endif
#endif /* WIN32  */

#if defined(HPUXV9)
#define SELECT(a,b,c,d,e) select((a),(int *)(b),(int *)(c),(int *)(d),(e))
#endif /* HPUX */

#if defined(RISCOS)
extern int errno;
#endif /* RISCOS */

/*
** Define things left undefined above
*/

#ifndef ERRNO
#define ERRNO		errno
#endif
#ifndef CLOSE
#define CLOSE		close
#endif
#ifndef READ
#define READ		read
#endif
#ifndef WRITE
#define WRITE		write
#endif
#ifndef PERROR
#define PERROR		perror
#endif
#ifndef SELECT
#define SELECT		select
#endif
#ifndef IOCTL
#define IOCTL		ioctl
#endif
