/***********************************************************************
*
* simdis.c - Disassemble instructions
*
* Changes:
*   11/07/08   DGP   Original. Hacked from disops.c
*   02/03/12   DGP   Fixed EMD, EINT, DINT disassembly.
*   10/07/13   DGP   Only disassemble insts for the current model.
*   03/29/16   DGP   Allow LST and LWP in 10A mode.
*   11/21/16   DGP   Support debugging data (-g output).
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>

#include "simdef.h"

extern int model;
extern int viewlen;
extern char view[MAXVIEW][MAXVIEWLEN+1];

#define MAXIDTS 100

typedef struct symnode {
   struct symnode *next;
   char name[MAXSYMLEN+2];
   int  loc;
} sym_node;

typedef struct 
{
   char name[IDTLEN+2];
   int origin;
   sym_node *head;
   sym_node *tail;
} idt_node;

static int opidx;
static int idt_count = 0;

/*
** Opcode tables.
*/

static char *ops1op[] =
{
   "",     "",     "",     "",     "SZC",  "SZCB", "S",    "SB",
   "C",    "CB",   "A",    "AB",   "MOV",  "MOVB", "SOC",  "SOCB"
};

static char *ops2op[] =
{
   "JMP",  "JLT",  "JLE",  "JEQ",  "JHE",  "JGT",  "JNE",  "JNC",
   "JOC",  "JNO",  "JL",   "JH",   "JOP",  "SBO",  "SBZ",  "TB"
};

static char *ops3op[] =
{
   "COC",  "CZC",  "XOR",  "XOP"
};

static char *ops4op[] =
{
   "LDCR", "STCR"
};

static char *ops5op[] =
{
   "SRA",  "SRL",  "SLA",  "SRC"
};

static char *ops6op[] =
{
   "BLWP", "B",    "X",    "CLR",  "NEG",  "INV",  "INC",  "INCT",
   "DEC",  "DECT", "BL",   "SWPB", "SETO", "ABS",  "LDS",  "LDD"
};
static char *ops6fop[] = 
{
   "",     "AR",   "CIR",  "SR",   "MR",   "DR",   "LR",   "STR",
   "",     "AD",   "CID",  "SD",   "MD",   "DD",   "LD",   "STD"
};
static char *ops6eop[] = 
{
   "",     "BIND", "DIVS", "MPYS"
};

static char *ops7op[] =
{
   "",     "",     "",     "",     "IDLE", "",     "RSET", "",
   "RTWP", "",     "CKON", "",     "CKOF", "",     "LREX"
};
static char *ops7fop[] = 
{
   "CRI",  "CDI",  "NEGR", "NEGD", "CRE",  "CDE",  "CER",  "CED",
   "",     "",     "",     "",     "",     "",     "XIT",  "XIT"
};
static char *ops7eop[] =
{
   "",     "",     "",     "",     "",     "",     "",     "",
   "",     "",     "",     "",     "",     "EMD",  "EINT", "DINT"
};

static char *ops8op[] =
{
   "LI",   "",     "AI",   "",     "ANDI", "",     "ORI",  "",
   "CI",   "",     "STWP", "",     "STST", "",     "LWPI"
};

static char *ops9op[] =
{
   "MPY",  "DIV"
};

static char *ops11op[] =
{
   "RTO",  "LTO",  "CNTO", "",     "",     "BDC",  "DBC",  "SWPM",
   "XORM", "ORM",  "ANDM", "SM",   "AM"
};

static char *ops12aop[] =
{
   "",     "SNEB", "CRC",  "TS"
};
static char *ops12bop[] =
{
   "CS",   "SEQB", "MOVS", "",     "",     "",     "",     "",
   "MVSR", "MVSK", "POPS", "PSHS"
};

static char *ops13op[] =
{
   "SRAM", "SLAM"
};

static char *ops14op[] =
{
   "TMB",  "TCMB", "TSMB"
};

static char *ops16op[] =
{
   "",     "INSF", "XV",   "XF"
};

static char *ops17op[] =
{
   "SRJ",  "ARJ"
};

static char *ops18op[] =
{
   "STPC", "",     "",     "",     "LIM",  "LST",  "LWP",  "LCS"
};

static char *ops20op[] =
{
   "",     "SLSL", "SLSP"
};
static char *ops20cc[] =
{
   "EQ",   "NE",   "HE",   "L",    "GE",   "LT",   "LE",   "H",
   "LTE",  "GT"
};

/*
** Function definitions and branch table
*/

static int dis1op  (TraceInst *trcdata);
static int dis2op  (TraceInst *trcdata);
static int dis3op  (TraceInst *trcdata);
static int dis4op  (TraceInst *trcdata);
static int dis5op  (TraceInst *trcdata);
static int dis6op  (TraceInst *trcdata);
static int dis7op  (TraceInst *trcdata);
static int dis8op  (TraceInst *trcdata);
static int dis9op  (TraceInst *trcdata);
static int dis10op (TraceInst *trcdata);
static int dis11op (TraceInst *trcdata);
static int dis12op (TraceInst *trcdata);
static int dis13op (TraceInst *trcdata);
static int dis14op (TraceInst *trcdata);
static int dis15op (TraceInst *trcdata);
static int dis16op (TraceInst *trcdata);
static int dis17op (TraceInst *trcdata);
static int dis18op (TraceInst *trcdata);
static int dis19op (TraceInst *trcdata);
static int dis20op (TraceInst *trcdata);
static int dis21op (TraceInst *trcdata);

typedef struct {
   int (*proc) (TraceInst *trcdata);
} Dis_Proc;

static Dis_Proc dis_proc[MAX_INST_TYPES] = {
   {dis1op},  {dis2op},  {dis3op},  {dis4op},  {dis5op},  {dis6op},  {dis7op},
   {dis8op},  {dis9op},  {dis10op}, {dis11op}, {dis12op}, {dis13op}, {dis14op},
   {dis15op}, {dis16op}, {dis17op}, {dis18op}, {dis19op}, {dis20op}, {dis21op}
};

static idt_node idts[MAXIDTS];


/***********************************************************************
* addidt - Add IDT
***********************************************************************/

void
addidt (char *name, int val)
{
   idt_node *node;
   char *sp;

   if (idt_count + 1 >= MAXIDTS) return;

   name[IDTLEN] = 0;
   for (sp = name + IDTLEN - 1; sp > name; sp--)
   {
      if (*sp == ' ') *sp = 0;
      else break;
   }

   node = &idts[idt_count];
   memset (node, 0, sizeof (idt_node));
   strcpy (node->name, name);
   node->origin = val;
   idt_count++;
   idts[idt_count].origin = 0xFFFF;
}

/***********************************************************************
* addsym - Add Symbol
***********************************************************************/

void
addsym (char *name, int len, int val)
{
   idt_node *node;
   sym_node *sym;
   char *sp;

   if (idt_count == 0) return;

   name[len] = 0;
   for (sp = name + len - 1; sp > name; sp--)
   {
      if (*sp == ' ') *sp = 0;
      else break;
   }

   node = &idts[idt_count-1];
   for (sym = node->head; sym; sym = sym->next)
   {
      if (!strcmp (sym->name, name)) return;
   }

   if ((sym = (sym_node *) malloc (sizeof (sym_node))) == NULL) return;

   memset (sym, 0, sizeof (sym_node));
   strcpy (sym->name, name);
   sym->loc = val;

   if (node->head == NULL) node->head = sym;
   if (node->tail != NULL) node->tail->next = sym;
   node->tail = sym;
}

/***********************************************************************
* dumpsyms - Dump Symbols
***********************************************************************/

void
dumpsyms (char *bp)
{
   sym_node *sym;
   char *idt = NULL;
   char *name = NULL;
   int idtfound = FALSE;
   int namefound = FALSE;
   int i, j;

   if (idt_count == 0)
   {
      sprintf (view[0], "No symbols");
      return;
   }

   if (bp && *bp)
   {
      while (*bp && isspace (*bp)) bp++;
      if (isalpha(*bp) || (*bp == '_') || (*bp == '$'))
      {
         idt = bp;
	 while (*bp && *bp != ':') bp++;
	 if (*bp == ':')
	 {
	    *bp++ = 0;
	    name = bp;
	 }
      }
   }

   j = 0;
   for (i = 0; i < idt_count; i++)
   {
      if (idt && strcmp (idt, idts[i].name)) continue;
      idtfound = TRUE;
      sprintf (view[j++], "IDT[%d]: >%04X %s",
	       i, idts[i].origin, idts[i].name);
      if (j == viewlen)
      {
	 if (putview (TRUE, TRUE)) return;
	 j = 0;
      }
      for (sym = idts[i].head; sym; sym = sym->next)
      {
	 if (name && strcmp (name, sym->name)) continue;
	 namefound = TRUE;
	 sprintf (view[j++], "    >%04X %s",
		  sym->loc, sym->name);
	 if (j == viewlen)
	 {
	    if (putview (TRUE, TRUE)) return;
	    j = 0;
	 }
      }
   }
   if (!idtfound)
      sprintf (view[j++], "IDT %s not found", idt);
   else if (!namefound && name)
      sprintf (view[j++], "SYM %s not found", name);
}

/***********************************************************************
* idtlookup - Lookup IDT address.
***********************************************************************/

int
idtlookup (char *name)
{
   int i;

   if (idt_count == 0)
   {
      sprintf (view[0], "No symbols");
      return (-1);
   }

   for (i = 0; i < idt_count; i++)
   {
      if (!strcmp (name, idts[i].name))
      {
	 return (idts[i].origin);
      }
   }

   sprintf (view[0], "IDT %s not found", name);
   return (-1);
}

/***********************************************************************
* symlookup - Lookup symbol.
***********************************************************************/

int
symlookup (int idtorigin, char *symname)
{
   int i;
   sym_node *sym;

   if (idt_count == 0)
   {
      sprintf (view[0], "No symbols");
      return (-1);
   }

   for (i = 0; i < idt_count; i++)
   {
      if (idtorigin < idts[i+1].origin)
      {
	 for (sym = idts[i].head; sym; sym = sym->next)
	 {
	    if (!strcmp (symname, sym->name))
	       return (sym->loc);
	 }
      }
   }

   sprintf (view[0], "SYM %s not found", symname);
   return (-1);
}

/***********************************************************************
* addrlookup - Lookup symbol for address.
***********************************************************************/

static char *
addrlookup (int pc, int addr)
{
   int i;
   sym_node *sym;

   for (i = 0; i < idt_count; i++)
   {
      if (pc < idts[i+1].origin)
      {
	 for (sym = idts[i].head; sym; sym = sym->next)
	 {
	    if (sym->loc == addr)
	       return (sym->name);
	 }
      }
   }
   return (NULL);
}

/***********************************************************************
* chklabel - Check for label.
***********************************************************************/

static char *
chklabel (TraceInst *trcdata)
{
   int i;
   sym_node *sym;

   for (i = 0; i < idt_count; i++)
   {
      if (trcdata->pc < idts[i+1].origin)
      {
	 for (sym = idts[i].head; sym; sym = sym->next)
	 {
	    if (sym->loc == trcdata->pc)
	       return (sym->name);
	 }
      }
   }
   return ("");
}

/***********************************************************************
* genoperand - Process general operand
***********************************************************************/

static void
genoperand (TraceInst *trcdata, uint16 sr, uint16 st, int dst)
{
   char *sym;
   uint16 sa;
   char tbuf[40];


   switch (st)
   {
   case 0:

      sprintf (tbuf, "R%d", sr);
      break;

   case 1:

      sprintf (tbuf, "*R%d", sr);
      break;

   case 2:

      sa = trcdata->ops[opidx++];
      if ((sym = addrlookup (trcdata->pc, sa)) == NULL)
	 sprintf (tbuf, "@>%X", sa);
      else
	 sprintf (tbuf, "@%s", sym);
      if (sr)
	 sprintf (&tbuf[strlen(tbuf)], "(R%d)", sr);

      break;

   case 3:

      sprintf (tbuf, "*R%d+", sr);
      break;
   }

   if (dst) strcat (tbuf, ",");

   strcat (trcdata->dis, tbuf);

}

/***********************************************************************
* dis1op - Process type 1 operations.
***********************************************************************/

static int
dis1op (TraceInst *trcdata)
{
   uint16 inst;
   uint16 st;
   uint16 sr;
   uint16 dt;
   uint16 dr;
   uint16 op;

   inst = trcdata->inst;
   sr =  inst & 0x000F;
   st = (inst & 0x0030) >> 4;
   dr = (inst & 0x03C0) >> 6;
   dt = (inst & 0x0C00) >> 10;
   op = (inst & 0xF000) >> 12;

   switch (op)
   {
   case 4:  /* SZC  */
   case 5:  /* SZCB */
   case 6:  /* S    */
   case 7:  /* SB   */
   case 8:  /* C    */
   case 9:  /* CB   */
   case 10: /* A    */
   case 11: /* AB   */
   case 12: /* MOV  */
   case 13: /* MOVB */
   case 14: /* SOC  */
   case 15: /* SOCB */
      break;
   default:
      return (-1);
   }

   sprintf (trcdata->dis, "%-6s %-4s ", chklabel(trcdata), ops1op[op]);
   genoperand (trcdata, sr, st, TRUE);
   genoperand (trcdata, dr, dt, FALSE);
   return (0);
}

/***********************************************************************
* dis2op - Process type 2 operations.
***********************************************************************/

static int
dis2op (TraceInst *trcdata)
{
   char *sym;
   uint16 inst;
   uint16 op;
   uint16 disp;

   inst = trcdata->inst;
   disp = inst & 0x00FF;
   op = (inst & 0x0F00) >> 8;

   switch (op)
   {
   case 0:  /* JMP */
   case 1:  /* JLT */
   case 2:  /* JLE */
   case 3:  /* JEQ */
   case 4:  /* JHE */
   case 5:  /* JGT */
   case 6:  /* JNE */
   case 7:  /* JNC */
   case 8:  /* JOC */
   case 9:  /* JNO */
   case 10: /* JL  */
   case 11: /* JH  */
   case 12: /* JOP */
      break;
   case 13: /* SBO */
   case 14: /* SBZ */
   case 15: /* TB  */
      if ((sym = addrlookup (trcdata->pc, disp)) == NULL)
	 sprintf (trcdata->dis, "%-6s %-4s %d (>%02X)",
		  chklabel(trcdata), ops2op[op], disp, disp);
      else
	 sprintf (trcdata->dis, "%-6s %-4s %s (>%02X)",
		  chklabel(trcdata), ops2op[op], sym, disp);
      return (0);
   default:
      return (-1);
   }

   if (inst == 0x1000)
   {
      sprintf (trcdata->dis, "%-6s %-4s ", chklabel(trcdata), "NOP");
   }
   else
   {
      int16 wpc;
      int8 ldisp;

      ldisp = disp;
      wpc = ((trcdata->pc + 2) >> 1) & 0x7FFF;
      wpc += ldisp;
      wpc = (wpc << 1) & 0xFFFE;

      sym = addrlookup (trcdata->pc, wpc);
      if (ldisp >= 0)
      {
	 if (sym)
	    sprintf (trcdata->dis, "%-6s %-4s %s",
		     chklabel(trcdata), ops2op[op], sym);
	 else
	    sprintf (trcdata->dis, "%-6s %-4s $+%d  >%04X  >%X",
		     chklabel(trcdata), ops2op[op], ldisp*2+2,
		     (uint16)wpc, disp);
      }
      else
      {
	 if (sym)
	    sprintf (trcdata->dis, "%-6s %-4s %s",
		     chklabel(trcdata), ops2op[op], sym);
	 else
	    sprintf (trcdata->dis, "%-6s %-4s $-%d  >%04X  >%X",
		     chklabel(trcdata), ops2op[op], (1-(ldisp*2))-3,
		     (uint16)wpc, disp);
      }
   }
   return (0);
}

/***********************************************************************
* dis3op - Process type 3 operations.
***********************************************************************/

static int
dis3op (TraceInst *trcdata)
{
   uint16 inst;
   uint16 st;
   uint16 sr;
   uint16 reg;
   uint16 op;
   char chop[16];

   inst = trcdata->inst;
   sr =  inst & 0x000F;
   st = (inst & 0x0030) >> 4;
   reg = (inst & 0x03C0) >> 6;
   op = (inst & 0x1C00) >> 10;

   switch (op)
   {
   case 0: /* COC */
   case 1: /* CZC */
   case 2: /* XOR */
   case 3: /* XOP */
      break;
   default:
      return (-1);
   }

   sprintf (trcdata->dis, "%-6s %-4s ", chklabel(trcdata), ops3op[op]);
   genoperand (trcdata, sr, st, TRUE);
   if (op == 3) /* XOP */
      sprintf (chop, "%d", reg);
   else
      sprintf (chop, "R%d", reg);
   strcat (trcdata->dis, chop);
   return (0);
}

/***********************************************************************
* dis4op - Process type 4 operations.
***********************************************************************/

static int
dis4op (TraceInst *trcdata)
{
   uint16 inst;
   uint16 sr;
   uint16 st;
   uint16 cnt;
   uint16 op;
   char chop[16];

   inst = trcdata->inst;
   sr = inst & 0x000F;
   st = (inst & 0x0030) >> 4;
   cnt = (inst & 0x03C0) >> 6;
   op =  (inst & 0x0400) >> 10;

   switch (op)
   {
   case 0: /* LDCR */
   case 1: /* STCR */
      break;
   default:
      return (-1);
   }

   sprintf (trcdata->dis, "%-6s %-4s ", chklabel(trcdata), ops4op[op]);
   genoperand (trcdata, sr, st, TRUE);
   sprintf (chop, "%d", cnt);
   strcat (trcdata->dis, chop);
   return (0);
}

/***********************************************************************
* dis5op - Process type 5 operations.
***********************************************************************/

static int
dis5op (TraceInst *trcdata)
{
   uint16 inst;
   uint16 reg;
   uint16 cnt;
   uint16 op;

   inst = trcdata->inst;
   reg = inst & 0x000F;
   cnt = (inst & 0x00F0) >> 4;
   op = (inst & 0x0300) >> 8;

   switch (op)
   {
   case 0: /* SRA */
   case 1: /* SRL */
   case 2: /* SLA */
   case 3: /* SRC */
      break;
   default:
      return (-1);
   }

   sprintf (trcdata->dis, "%-6s %-4s R%d,%d", chklabel(trcdata), 
	    ops5op[op], reg, cnt);
   return (0);
}

/***********************************************************************
* dis6op - Process type 6 operations.
***********************************************************************/

static int
dis6op (TraceInst *trcdata)
{
   char *opptr;
   uint16 inst;
   uint16 sr;
   uint16 st;
   uint16 op;

   inst = trcdata->inst;
   sr = inst & 0x000F;
   st = (inst & 0x0030) >> 4;

   if ((inst & 0x0F00) >= 0x0C00) /* Floating point instructions */
   {
      if (model < 12)
	 return (-1);

      op = (inst & 0x03C0) >> 6;
      switch (op)
      {
      case 1:  /* AR   */
      case 2:  /* CIR  */
      case 3:  /* SR   */
      case 4:  /* MR   */
      case 5:  /* DR   */
      case 6:  /* LR   */
      case 7:  /* STR  */
      case 9:  /* AD   */
      case 10: /* CID  */
      case 11: /* SD   */
      case 12: /* MD   */
      case 13: /* DD   */
      case 14: /* LD   */
      case 15: /* STD  */
         break;
      default:
	 return (-1);
      }
      opptr = ops6fop[op];
   }
   else if ((inst & 0x0F00) == 0x0100) 
   {
      if (model < 11)
	 return (-1);

      op = (inst & 0x00C0) >> 6;
      switch (op)
      {
      case 1:  /* BIND */
      case 2:  /* DIVS */
      case 3:  /* MPYS */
         break;
      default:
	 return (-1);
      }
      opptr = ops6eop[op];
   }
   else
   {
      op = (inst & 0x03C0) >> 6;
      switch (op)
      {
      case 0:  /* BLWP */
      case 1:  /* B    */
      case 2:  /* X    */
      case 3:  /* CLR  */
      case 4:  /* NEG  */
      case 5:  /* INV  */
      case 6:  /* INC  */
      case 7:  /* INCT */
      case 8:  /* DEC  */
      case 9:  /* DECT */
      case 10: /* BL   */
      case 11: /* SWPB */
      case 12: /* SETO */
      case 13: /* ABS  */
         break;
      case 14: /* LDS  */
      case 15: /* LDD  */
	 if (model < 10)
	    return (-1);
	 break;
      default:
	 return (-1);
      }
      opptr = ops6op[op];
   }

   sprintf (trcdata->dis, "%-6s %-4s ", chklabel(trcdata), opptr);
   genoperand (trcdata, sr, st, FALSE);
   return (0);
}

/***********************************************************************
* dis7op - Process type 7 operations.
***********************************************************************/

static int
dis7op (TraceInst *trcdata)
{
   char *opptr;
   uint16 inst;
   uint16 op;

   inst = trcdata->inst;
   op =  (inst & 0x00F0) >> 4;
   switch (op)
   {
   case 0:
      if (model < 12)
	 return (-1);

      op = inst & 0x000F;
      switch (op)
      {
      case 0:  /* CRI  */
      case 1:  /* CDI  */
      case 2:  /* NEGR */
      case 3:  /* NEGD */
      case 4:  /* CRE  */
      case 5:  /* CDE  */
      case 6:  /* CER  */
      case 7:  /* CED  */
      case 14: /* XIT  */
      case 15: /* XIT  */
         break;
      default:
         return (-1);
      }
      opptr = ops7fop[op];
      break;
   case 2:
      if (model < 12)
	 return (-1);

      op = inst & 0x000F;
      switch (op)
      {
      case 13: /* EMD  */
      case 14: /* EINT */
      case 15: /* DINT */
         break;
      default:
         return (-1);
      }
      opptr = ops7eop[op];
      break;
   case 4:  /* IDLE */
   case 6:  /* RSET */
   case 8:  /* RTWP */
   case 10: /* CKON */
   case 12: /* CKOF */
   case 14: /* LREX */
      opptr = ops7op[op];
      break;
   default:
      return (-1);
   }

   sprintf (trcdata->dis, "%-6s %-4s ", chklabel(trcdata), opptr);
   return (0);
}

/***********************************************************************
* dis8op - Process type 8 operations.
***********************************************************************/

static int
dis8op (TraceInst *trcdata)
{
   char *sym;
   uint16 inst;
   uint16 r;
   uint16 op;
   char chop[16];

   inst = trcdata->inst;
   r = inst & 0x000F;
   op = (inst & 0x00F0) >> 4;

   if ((inst & 0xFFF0) == 0x00B0) /* BLSK */
   {
      if (model < 12)
	 return (-1);

      sprintf (trcdata->dis, "%-6s %-4s %d", chklabel(trcdata), "BLSK", r);
      return (0);
   }
   else if (inst == 0x0300) /* LIMI */
   {
      op = trcdata->ops[opidx++];
      sprintf (trcdata->dis, "%-6s %-4s >%X", chklabel(trcdata), "LIMI", op);
      return (0);
   }

   switch (op)
   {
   case 0:  /* LI   */
   case 2:  /* AI   */
   case 4:  /* ANDI */
   case 6:  /* ORI  */
   case 8:  /* CI   */
      sprintf (trcdata->dis, "%-6s %-4s R%d,",
	       chklabel(trcdata), ops8op[op], r);
      break;
   case 14: /* LWPI */
      sprintf (trcdata->dis, "%-6s %-4s ", chklabel(trcdata), ops8op[op]);
      break;
   case 10: /* STWP */
   case 12: /* STST */
      sprintf (trcdata->dis, "%-6s %-4s R%d",
	       chklabel(trcdata), ops8op[op], r);
      return (0);
   default:
      return (-1);
   }

   op = trcdata->ops[opidx++];
   if ((sym = addrlookup (trcdata->pc, op)) == NULL)
      sprintf (chop, ">%04X", op);
   else
      sprintf (chop, "%s", sym);
   strcat (trcdata->dis, chop);
   return (0);
}

/***********************************************************************
* dis9op - Process type 9 operations.
***********************************************************************/

static int
dis9op (TraceInst *trcdata)
{
   uint16 inst;
   uint16 st;
   uint16 sr;
   uint16 r;
   uint16 op;
   char chop[16];

   inst = trcdata->inst;
   sr =  inst & 0x000F;
   st = (inst & 0x0030) >> 4;
   r = (inst & 0x03C0) >> 6;
   op = (inst & 0x0400) >> 10;

   switch (op)
   {
   case 0: /* MPY */
   case 1: /* DIV */
      break;
   default:
      return (-1);
   }

   sprintf (trcdata->dis, "%-6s %-4s ", chklabel(trcdata), ops9op[op]);
   genoperand (trcdata, sr, st, TRUE);
   sprintf (chop, "R%d", r);
   strcat (trcdata->dis, chop);
   return (0);
}

/***********************************************************************
* dis10op - Process type 10 operations.
***********************************************************************/

static int
dis10op (TraceInst *trcdata)
{
   uint16 r;
   uint16 mf;

   if (model < 10)
      return (-1);

   r =  trcdata->inst & 0x000F;
   mf = (trcdata->inst & 0x0010) >> 4;

   sprintf (trcdata->dis, "%-6s %-4s R%d,%d", chklabel(trcdata), "LMF", r, mf);
   return (0);
}

/***********************************************************************
* dis11op - Process type 11 operations.
***********************************************************************/

static int
dis11op (TraceInst *trcdata)
{
   char *opptr;
   uint16 inst;
   uint16 inst1;
   uint16 dt;
   uint16 dr;
   uint16 st;
   uint16 sr;
   uint16 bc;
   uint16 op;
   char chop[16];

   if (model < 12)
      return (-1);

   inst = trcdata->inst;
   inst1 = trcdata->ops[opidx++];
   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   dr = (inst1 & 0x03C0) >> 6;
   dt = (inst1 & 0x0C00) >> 10;
   bc = (inst1 & 0xF000) >> 12;

   if (inst == 0xC08) /* NRM */
   {
      opptr = "NRM";
   }
   else
   {
      op = (inst & 0x003F) - 0x001E;
      switch (op)
      {
      case 0:  /* RTO  */
      case 1:  /* LTO  */
      case 2:  /* CNTO */
      case 5:  /* BDC  */
      case 6:  /* DBC  */
      case 7:  /* SWPM */
      case 8:  /* XORM */
      case 9:  /* ORM  */
      case 10: /* ANDM */
      case 11: /* SM   */
      case 12: /* AM   */
         break;
      default:
	 return (-1);
      }
      opptr = ops11op[op];
   }

   sprintf (trcdata->dis, "%-6s %-4s ", chklabel(trcdata), opptr);
   genoperand (trcdata, sr, st, TRUE);
   genoperand (trcdata, dr, dt, TRUE);
   sprintf (chop, "%d", bc);
   strcat (trcdata->dis, chop);
   return (0);
}

/***********************************************************************
* dis12op - Process type 12 operations.
***********************************************************************/

static int
dis12op (TraceInst *trcdata)
{
   char *opptr;
   uint16 inst;
   uint16 inst1;
   uint16 ckpt;
   uint16 dt;
   uint16 dr;
   uint16 st;
   uint16 sr;
   uint16 bc;
   uint16 op;
   char chop[16];

   if (model < 12)
      return (-1);

   inst = trcdata->inst;
   ckpt = inst & 0x000F;
   inst1 = trcdata->ops[opidx++];
   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   dr = (inst1 & 0x03C0) >> 6;
   dt = (inst1 & 0x0C00) >> 10;
   bc = (inst1 & 0xF000) >> 12;

   if ((inst & 0x0F00) == 0x0E00)
   {
      op = (inst & 0x00F0) >> 4;
      switch (op)
      {
      case 1:  /* SNEB */
      case 2:  /* CRC  */
      case 3:  /* TC   */
         break;
      default:
         return (-1);
      }
      opptr = ops12aop[op];
   }
   else
   {
      op = ((inst & 0x00F0) - 0x0040) >> 4;
      switch (op)
      {
      case 0:  /* CS   */
      case 1:  /* SEQB */
      case 2:  /* MOVS */
      case 8:  /* MVSR */
      case 9:  /* MVSK */
      case 10: /* POPS */
      case 11: /* PSHS */
         break;
      default:
         return (-1);
      }
      opptr = ops12bop[op];
   }
   sprintf (trcdata->dis, "%-6s %-4s ", chklabel(trcdata), opptr);
   genoperand (trcdata, sr, st, TRUE);
   genoperand (trcdata, dr, dt, TRUE);
   sprintf (chop, "%d,R%d", bc, ckpt);
   strcat (trcdata->dis, chop);
   return (0);
}

/***********************************************************************
* dis13op - Process type 13 operations.
***********************************************************************/

static int
dis13op (TraceInst *trcdata)
{
   uint16 inst;
   uint16 inst1;
   uint16 st;
   uint16 sr;
   uint16 bc;
   uint16 ct;
   uint16 op;
   char chop[16];

   if (model < 12)
      return (-1);

   inst = trcdata->inst;
   inst1 = trcdata->ops[opidx++];
   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   ct = (inst1 & 0x03C0) >> 6;
   bc = (inst1 & 0xF000) >> 12;

   op = (inst & 0x000F) - 0x000C;

   switch (op)
   {
   case 0:  /* SRAM */
   case 1:  /* SLAM */
      break;
   default:
      return (-1);
   }

   sprintf (trcdata->dis, "%-6s %-4s ", chklabel(trcdata), ops13op[op]);
   genoperand (trcdata, sr, st, TRUE);
   sprintf (chop, "%d,%d", ct, bc);
   strcat (trcdata->dis, chop);
   return (0);
}

/***********************************************************************
* dis14op - Process type 14 operations.
***********************************************************************/

static int
dis14op (TraceInst *trcdata)
{
   uint16 inst;
   uint16 inst1;
   uint16 st;
   uint16 sr;
   uint16 ps;
   uint16 op;
   char chop[16];

   if (model < 12)
      return (-1);

   inst = trcdata->inst;
   inst1 = trcdata->ops[opidx++];
   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   ps = (inst1 & 0xFFC0) >> 6;

   op = (inst & 0x000F) - 0x0009;

   switch (op)
   {
   case 0:  /* TMB  */
   case 1:  /* TCMB */
   case 2:  /* TSMB */
      break;
   default:
      return (-1);
   }

   sprintf (trcdata->dis, "%-6s %-4s ", chklabel(trcdata), ops14op[op]);
   genoperand (trcdata, sr, st, TRUE);
   sprintf (chop, "%d", ps);
   strcat (trcdata->dis, chop);
   return (0);
}

/***********************************************************************
* dis15op - Process type 15 operations.
***********************************************************************/

static int
dis15op (TraceInst *trcdata)
{
   uint16 inst;
   uint16 inst1;
   uint16 st;
   uint16 sr;
   uint16 wd;
   uint16 ps;
   char chop[16];

   if (model < 12)
      return (-1);

   inst = trcdata->inst;
   wd = inst & 0x000F;
   inst1 = trcdata->ops[opidx++];
   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   ps = (inst1 & 0xF000) >> 12;

   sprintf (trcdata->dis, "%-6s %-4s ", chklabel(trcdata), "IOF");
   genoperand (trcdata, sr, st, TRUE);
   sprintf (chop, "(%d,%d)", ps, wd);
   strcat (trcdata->dis, chop);
   return (0);
}

/***********************************************************************
* dis16op - Process type 16 operations.
***********************************************************************/

static int
dis16op (TraceInst *trcdata)
{
   uint16 inst;
   uint16 inst1;
   uint16 st;
   uint16 sr;
   uint16 dt;
   uint16 dr;
   uint16 wd;
   uint16 ps;
   uint16 op;
   char chop[16];

   if (model < 12)
      return (-1);

   inst = trcdata->inst;
   wd = inst & 0x000F;
   inst1 = trcdata->ops[opidx++];
   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   dr = (inst1 & 0x03C0) >> 6;
   dt = (inst1 & 0x0C00) >> 10;
   ps = (inst1 & 0xF000) >> 12;

   op = (inst & 0x000F) - 0x0009;

   switch (op)
   {
   case 1:  /* INSF */
   case 2:  /* XV   */
   case 3:  /* XF   */
      break;
   default:
      return (-1);
   }

   sprintf (trcdata->dis, "%-6s %-4s ", chklabel(trcdata), ops16op[op]);
   genoperand (trcdata, sr, st, TRUE);
   genoperand (trcdata, dr, dt, TRUE);
   sprintf (chop, "(%d,%d)", ps, wd);
   strcat (trcdata->dis, chop);
   return (0);
}

/***********************************************************************
* dis17op - Process type 17 operations.
***********************************************************************/

static int
dis17op (TraceInst *trcdata)
{
   uint16 inst;
   uint16 inst1;
   uint16 disp;
   uint16 r;
   uint16 c;
   uint16 op;
   char chop[16];

   if (model < 12)
      return (-1);

   inst = trcdata->inst;
   inst1 = trcdata->ops[opidx++];
   disp =  inst1 & 0x00FF;
   r = (inst1 & 0x0F00) >> 8;
   c = (inst1 & 0xF000) >> 12;

   op = (inst & 0x000F) - 0x000C;

   switch (op)
   {
   case 0:  /* SRJ  */
   case 1:  /* ARJ  */
      break;
   default:
      return (-1);
   }

   sprintf (trcdata->dis, "%-6s %-4s ", chklabel(trcdata), ops17op[op]);
   sprintf (chop, ">%X,%d,R%d", disp, c, r);
   strcat (trcdata->dis, chop);
   return (0);
}

/***********************************************************************
* dis18op - Process type 18 operations.
***********************************************************************/

static int
dis18op (TraceInst *trcdata)
{
   uint16 inst;
   uint16 r;
   uint16 op;

   inst = trcdata->inst;
   r =  inst & 0x000F;
   op = ((inst & 0x00F0) >> 4) - 0x0003;

   switch (op)
   {
   case 0:  /* STPC */
   case 4:  /* LIM  */
   case 7:  /* LCS  */
      if (model < 12)
	 return (-1);
      break;
   case 5:  /* LST  */
   case 6:  /* LWP  */
      if (model < 11)
	 return (-1);
      break;
   default:
      return (-1);
   }

   sprintf (trcdata->dis, "%-6s %-4s R%d", chklabel(trcdata), ops18op[op], r);
   return (0);
}

/***********************************************************************
* dis19op - Process type 19 operations.
***********************************************************************/

static int
dis19op (TraceInst *trcdata)
{
   uint16 inst1;
   uint16 st;
   uint16 sr;
   uint16 dt;
   uint16 dr;

   inst1 = trcdata->ops[opidx++];
   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   dr = (inst1 & 0x03C0) >> 6;
   dt = (inst1 & 0x0C00) >> 10;

   sprintf (trcdata->dis, "%-6s %-4s ", chklabel(trcdata), "MOVA");
   genoperand (trcdata, sr, st, TRUE);
   genoperand (trcdata, dr, dt, FALSE);
   return (0);
}

/***********************************************************************
* dis20op - Process type 20 operations.
***********************************************************************/

static int
dis20op (TraceInst *trcdata)
{
   uint16 inst;
   uint16 inst1;
   uint16 st;
   uint16 sr;
   uint16 dt;
   uint16 dr;
   uint16 cc;
   uint16 op;
   char chop[16];

   if (model < 12)
      return (-1);

   inst = trcdata->inst;
   inst1 = trcdata->ops[opidx++];
   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   dr = (inst1 & 0x03C0) >> 6;
   dt = (inst1 & 0x0C00) >> 10;
   cc = (inst1 & 0xF000) >> 12;

   op = inst & 0x000F;

   switch (op)
   {
   case 1:  /* SLSL */
   case 2:  /* SLSP */
      break;
   default:
      return (-1);
   }

   sprintf (trcdata->dis, "%-6s %-4s ", chklabel(trcdata), ops20op[op]);
   sprintf (chop, "%s,", ops20cc[cc]);
   strcat (trcdata->dis, chop);
   genoperand (trcdata, sr, st, TRUE);
   genoperand (trcdata, dr, dt, FALSE);
   return (0);
}

/***********************************************************************
* dis21op - Process type 21 operations.
***********************************************************************/

static int
dis21op (TraceInst *trcdata)
{
   uint16 inst;
   uint16 inst1;
   uint16 st;
   uint16 sr;
   uint16 sl;
   uint16 dt;
   uint16 dr;
   uint16 dl;
   char chop[16];

   if (model < 12)
      return (-1);

   inst = trcdata->inst;
   dl = inst & 0x000F;
   inst1 = trcdata->ops[opidx++];
   sr =  inst1 & 0x000F;
   st = (inst1 & 0x0030) >> 4;
   dr = (inst1 & 0x03C0) >> 6;
   dt = (inst1 & 0x0C00) >> 10;
   sl = (inst1 & 0xF000) >> 12;

   sprintf (trcdata->dis, "%-6s %-4s ", chklabel(trcdata), "EP");
   genoperand (trcdata, sr, st, TRUE);
   genoperand (trcdata, dr, dt, TRUE);
   sprintf (chop, "%d,%d", sl, dl);
   strcat (trcdata->dis, chop);
   return (0);
}

/***********************************************************************
* disinst - Disassemble an instruction.
***********************************************************************/

void 
disinst (TraceInst *trcdata)
{
   char *sym;
   int type;
   int error;

   type = decodeinst (trcdata->inst);
   trcdata->dis[0] = '\0';

   opidx = 0;
   error = -1;
   if (type != TYPE_ILL)
   {
      error = dis_proc[type].proc (trcdata);
   }

   if (error < 0)
   {
      if (trcdata->inst == 0 ||
          (sym = addrlookup (trcdata->pc, trcdata->inst)) == NULL)
	 sprintf (trcdata->dis, "%-6s %-4s >%X",
		  chklabel(trcdata), "DATA", trcdata->inst);
      else
	 sprintf (trcdata->dis, "%-6s %-4s %s",
		  chklabel(trcdata), "DATA", sym);
   }

   for (; opidx < 3; opidx++) trcdata->ops[opidx] = -1;
}
