/***********************************************************************
*
* simdsk.c - TILINE Disk IO processing for the TI 990 Simulator.
*
* Changes:
*   11/11/03   DGP   Original.
*   12/10/03   DGP   Change TPCS usage.
*   05/11/04   DGP   Added DEV_NULL support and fixed to work with boot ROM.
*   07/07/04   DGP   Added disk overhead/record support.
*   07/29/04   DGP   Added dskreset function.
*   04/28/05   DGP   Changed to use switches.
*   09/07/11   DGP   Added disk geometry checks.
*   10/29/13   DGP   Correct status on unconnected devices.
*   05/14/14   DGP   Fixed disk offline processing for DNOS.
*   07/21/14   DGP   Fixed SWPROTECT clause (removed ';').
*   07/28/15   DGP   Use data in format.
*   01/28/16   DGP   Added real disk support.
*   10/07/18   DGP   Fixed overhead.
*   08/01/20   DGP   Added FD800 support.
*   08/03/20   DGP   Added share field.
*
* The disk container file is formatted as follows:
*    4 bytes (big-endian): # cylinders
*    4 bytes (big-endian): reserved (2 bytes) | # heads
*    4 bytes (big-endian): # bytes overhead << 8 | # sectors/track
*    4 bytes (big-endian): # bytes/sector
*    N bytes: raw data for each sector - the order is:
*      cylinder 0 head 0 sector 0
*      cylinder 0 head 0 sector 1, ...
*      cylinder 0 head 0 sector N,
*      cylinder 0 head 1 sector 0, ... 
*      cylinder 0 head M sector N,
*      cylinder 1 head 0 sector 0, ...
*
***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>

#if defined (WIN32)
#define __TTYROUTINES 0
#include <conio.h>
#include <windows.h>
#include <signal.h>
#endif

#if defined(UNIX)
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <termios.h>
#include <pthread.h>
#endif

#include "simdef.h"

#define EXTERN extern
#include "simdsk.h"

extern uint16 pcreg;	/* The program PC */
extern uint16 statreg;	/* The program status register */
extern uint16 wpreg;	/* The program Workspace Pointer */

extern int run;
extern int devcnt;
extern int bootfromrom;
extern uint32 memlen;
extern unsigned long instcount;
extern char view[MAXVIEW][MAXVIEWLEN+1];

extern uint8 memory[SYSMEMSIZE];
extern Device devices[MAXDEVICES];

/*
** Command words access macros
*/

#define GETCMD(d)  (GETMEMB0((d)->devaddr+TPCSSTART+2) & 0x7)
#define GETHEAD(d) (GETMEMB0((d)->devaddr+TPCSSTART+3) & 0x3F)
#define GETSECT(d) GETMEMB0((d)->devaddr+TPCSSTART+5)
#define GETCYL(d)  (GETMEM0((d)->devaddr+TPCSSTART+6) & 0x7FF)
#define GETUNIT(d) GETMEMB0((d)->devaddr+TPCSSTART+12) & 0xF

/*
** Command word control bits
*/

#define OLBIT	0x8000		/* Off Line */
#define NRBIT	0x4000		/* Not Ready */
#define WPBIT	0x2000		/* Write Protect */
#define USBIT	0x1000		/* UnSafe */
#define ECBIT	0x0800		/* End of Cylinder */
#define SIBIT	0x0400		/* Seek Incomplete */
#define OSABIT	0x0200		/* Offset Active */
#define PCBIT	0x0100		/* Unsafe reason */

#define STBEBIT	0x20		/* Strobe Early */
#define STBLBIT	0x10		/* Strobe Late */
#define TIHBIT	0x08		/* Transfer Inhibit */

#define OSBIT	0x80		/* Head Offset */
#define OSFBIT	0x40		/* Head Offset Forward */

#define IDLEBIT	0x8000		/* Idle */
#define COMPBIT	0x4000		/* Complete */
#define ERRBIT	0x2000		/* Error */
#define INTBIT	0x1000		/* Interrupt enable */
#define LOCKBIT	0x0800		/* Lockout */
#define RETBIT	0x0400		/* Retry */
#define ECCBIT	0x0200		/* ECC corrected */
#define ABNBIT	0x0100		/* Abnormal Completion */
#define MEBIT	0x0080		/* Memory Error */
#define DEBIT	0x0040		/* Data Error */
#define TTBIT	0x0020		/* Tiline Timeout */
#define IEBIT	0x0010		/* ID Error */
#define REBIT	0x0008		/* Rate Error */
#define CTBIT	0x0004		/* Command Timer */
#define SEBIT	0x0002		/* Search Error */
#define UEBIT	0x0001		/* Unit Error */

/*
** Disk commands
*/

#define STOREREG  0		/* Store registers */
#define WRITEFMT  1		/* Write format */
#define READDATA  2		/* Read data */
#define WRITEDATA 3		/* Write data */
#define READUFMT  4		/* Read unformated */
#define WRITEUFMT 5		/* Write unformated */
#define DISKSEEK  6		/* Disk Seek */
#define RESTORE   7		/* Restore */

#define DSK_UNIT0 0x8
#define DSK_UNIT1 0x4
#define DSK_UNIT2 0x2
#define DSK_UNIT3 0x1

#if defined(DEBUGDSK) || defined(DEBUGDSKERR)
static int punit[] = { -1, 3, 2, -1, 1, -1, -1, -1, 0 };
#endif

/***********************************************************************
* dskreadint - Read an integer.
***********************************************************************/

static int
dskreadint (FILE *fd)
{
   int r;
   int i;

   r = 0;
   for (i = 0; i < 4; i++)
   {
      int c;
      if ((c = fgetc (fd)) < 0)
      {
	 sprintf (view[0], "dskreadint: read failed: %s", strerror (ERRNO));
         return (-1);
      }
      r = (r << 8) | (c & 0xFF);
   }
   return (r);
}

/***********************************************************************
* dskfindunit - Find the unit device entry.
***********************************************************************/

static Device *
dskfindunit (Device *dev, uint8 unit)
{
   Device *udev;
   int i;

   udev = NULL;
   for (i = 0; i < devcnt; i++)
   {
      if (devices[i].devaddr == dev->devaddr && devices[i].unit == unit)
      {
	 udev = &devices[i];
	 break;
      }
   }
   return (udev);
}

/***********************************************************************
* dskopen - Open disk and read geometry.
***********************************************************************/

int 
dskopen (Device *dev)
{
   int i;
   int diskdata;

#ifdef DEBUGDSK
   fprintf (stderr, "dskopen: ENTERED\n");
#endif

   if (dev->infd == DEV_NULL)
   {
      return (0);
   }

   /*
   ** Seek to beginning of the disk
   */

   if (fseek (dev->infd, 0, SEEK_SET) < 0)
   {
      sprintf (view[0], "Device %s: seek failed: %s",
	       dev->name, strerror (ERRNO));
      snprintf (view[1], MAXVIEWLEN, "filename: %s", dev->file);
      return (-1);
   }

   /*
   ** If a real device, use model info to get geometry.
   */

   if (dev->realdevice)
   {
      if (!dev->info.dskinfo.model[0])
      {
	 sprintf (view[0], "Device %s: Unspecified disk model", dev->name);
	 return (-1);
      }
      else for (i = 0; i < MAXDISKS; i++)
      {
	 if (!strcmp (dev->info.dskinfo.model, disks[i].model))
	 {
	    dev->info.dskinfo.cyls = disks[i].cyls;
	    dev->info.dskinfo.heads = disks[i].heads & 0xFFFF;
	    dev->info.dskinfo.secttrk = disks[i].sectrk;
	    dev->info.dskinfo.sectlen = disks[i].bytsec;
	    dev->info.dskinfo.overhead = disks[i].overhead;
	    dev->info.dskinfo.geoverhead = 0;
	    dev->info.dskinfo.tracklen = dev->info.dskinfo.secttrk * 
		  (dev->info.dskinfo.sectlen + dev->info.dskinfo.overhead);
	    break;
	 }
      }
      if (i == MAXDISKS)
      {
	 sprintf (view[0], "Device %s: Unknown disk model", dev->name);
	 return (-1);
      }
   }

   /*
   ** Else, Read the disk geometry
   */

   else
   {
      struct stat stbuf;

      if (fstat (fileno(dev->infd), &stbuf) < 0) return (-1);

      /* Check for SSSD FD800, based on size */
      if (stbuf.st_size == 256256) 
      {
         
	 dev->info.dskinfo.cyls = FPYTRKCYL;
	 dev->info.dskinfo.heads = 1;
	 dev->info.dskinfo.secttrk = FPYSECTTRK;
	 dev->info.dskinfo.overhead = 0;
	 dev->info.dskinfo.sectlen = FPYSECTLEN;
	 dev->info.dskinfo.geoverhead = 0;
	 dev->info.dskinfo.share = 1;
      }

      /* Othewise, read geometry overhead data */
      else
      {
	 if ((dev->info.dskinfo.cyls = dskreadint (dev->infd)) < 0) return (-1);
	 if ((dev->info.dskinfo.heads = dskreadint (dev->infd)) < 0)
	    return (-1);
	 dev->info.dskinfo.heads &= 0xFFFF;
	 if ((diskdata = dskreadint (dev->infd)) < 0) return (-1);
	 dev->info.dskinfo.secttrk = diskdata & 0xFF;
	 dev->info.dskinfo.overhead = (diskdata >> 8) & 0x3FF;
	 if ((dev->info.dskinfo.sectlen = dskreadint (dev->infd)) < 0)
	    return (-1);
	 dev->info.dskinfo.geoverhead = DSKOVERHEAD;
      }

      dev->info.dskinfo.tracklen = dev->info.dskinfo.secttrk * 
	       (dev->info.dskinfo.sectlen + dev->info.dskinfo.overhead);
	       
      /*
      ** Check geometry
      */

      for (i = 0; i < MAXDISKS; i++)
      {
	 if ((dev->info.dskinfo.cyls == disks[i].cyls) &&
	     (dev->info.dskinfo.heads == disks[i].heads) &&
	     (dev->info.dskinfo.secttrk == disks[i].sectrk) &&
	     (dev->info.dskinfo.sectlen == disks[i].bytsec) &&
	     (dev->info.dskinfo.overhead == disks[i].overhead))
	 {
	    dev->info.dskinfo.share = disks[i].share;
	    break;
	 }
      }
      if (i == MAXDISKS)
      {
	 sprintf (view[0], "Device %s: Unknown disk geometry", dev->name);
	 snprintf (view[1], MAXVIEWLEN, "filename: %s", dev->file);
	 sprintf (view[2], "   cyls      = %d", dev->info.dskinfo.cyls);
	 sprintf (view[3], "   heads     = %d", dev->info.dskinfo.heads);
	 sprintf (view[4], "   secttrk   = %d", dev->info.dskinfo.secttrk);
	 sprintf (view[5], "   sectlen   = %d", dev->info.dskinfo.sectlen);
	 sprintf (view[6], "   overhead  = %d", dev->info.dskinfo.overhead);
	 sprintf (view[7], "   tracklen  = %d", dev->info.dskinfo.tracklen);
	 return (-1);
      }
      strcpy (dev->info.dskinfo.model, disks[i].model);
   }
   dev->info.dskinfo.type = i;

#ifdef DEBUGDSK
   fprintf (stderr, "Disk geometry: realdevice = %s\n",
	    dev->realdevice ? "YES" : "NO");
   fprintf (stderr, "   type      = %s\n", dev->info.dskinfo.model);
   fprintf (stderr, "   cyls      = %d\n", dev->info.dskinfo.cyls);
   fprintf (stderr, "   heads     = %d\n", dev->info.dskinfo.heads);
   fprintf (stderr, "   secttrk   = %d\n", dev->info.dskinfo.secttrk);
   fprintf (stderr, "   sectlen   = %d\n", dev->info.dskinfo.sectlen);
   fprintf (stderr, "   overhead  = %d\n", dev->info.dskinfo.overhead);
   fprintf (stderr, "   tracklen  = %d\n", dev->info.dskinfo.tracklen);
   fprintf (stderr, "   disk size = %d bytes\n", dev->info.dskinfo.tracklen *
	    dev->info.dskinfo.heads * dev->info.dskinfo.cyls);
#endif

   /*
   ** Ensure that all disks on a controller are the same type. Except, for
   ** SCSI and FD1000/FD800 disks.
   */

   if (dev->unit != DSK_UNIT0)
   {
      Device *mdev;

      if ((mdev = dskfindunit (dev, DSK_UNIT0)) != NULL)
      {
	 int match = TRUE;

         if ((mdev->info.dskinfo.share == 0) && 
	     (dev->info.dskinfo.type != mdev->info.dskinfo.type))
	 {
	    match = FALSE;
	 }
	 else if (dev->info.dskinfo.share != mdev->info.dskinfo.share)
	 {
	    match = FALSE;
	 }
	 if (!match)
	 {
	    sprintf (view[0], "Device %s type %s: Disk geometry mismatch",
		     dev->name, disks[dev->info.dskinfo.type].model);
	    snprintf (view[1], MAXVIEWLEN, "filename: %s", dev->file);
	    sprintf (view[2], "Device %s type %s",
		     mdev->name, disks[mdev->info.dskinfo.type].model);
	    return (-1);
	 }
      }
   }

   PUTW7 (dev, IDLEBIT | COMPBIT); /* Mark idle */
   
   return (0);
}

/***********************************************************************
* dskputmem - Put data into memory.
***********************************************************************/

static void
dskputmem (uint32 ma, uint16 v)
{
   if (ma < memlen)
   {
      PUTMEM0 (ma, v);
   }
}

/***********************************************************************
* dskgetmem - Get data from memory.
***********************************************************************/

static uint16
dskgetmem (uint32 ma)
{
   if (ma >= memlen) return (0);

   return (GETMEM0 (ma));
}

/***********************************************************************
* dskreset - Reset device
***********************************************************************/

void
dskreset (Device *dev)
{
#ifdef DEBUGDSK
   fprintf (stderr, "dskreset: ENTERED\n");
#endif

   if (dev->infd != DEV_NULL)
   {
      PUTW0 (dev, ((dev->switches & SWPROTECT) ? WPBIT : 0));
      PUTW7 (dev, IDLEBIT | ERRBIT | ABNBIT); 
   }
   return;
}

/***********************************************************************
* dskseek - Seek to specified disk location.
***********************************************************************/

static int
dskseek (Device *udev, int cyl, int head, int sector)
{
   long dskloc;
   uint16 W0, W7;

   /*
   ** Ensure cylinder is OK
   */

   W0 = GETW0 (udev) & 0x00FF;
   W7 = GETW7 (udev);
   if (cyl >= udev->info.dskinfo.cyls)
   {
#if defined(DEBUGDSK) || defined(DEBUGDSKERR)
      fprintf (stderr, "dskseek: ERROR: cyl = %d, head = %d, sector = %d\n",
	       cyl, head, sector);
#endif
      W0 |= SIBIT | ((udev->switches & SWPROTECT) ? WPBIT : 0);
      W7 |= ERRBIT | IDLEBIT | COMPBIT | UEBIT;
      PUTW0 (udev, W0);
      PUTW7 (udev, W7);
      return (-1);
   }

   /*
   ** Calculate location
   */

   dskloc = (udev->info.dskinfo.tracklen * head) +
      (udev->info.dskinfo.tracklen * udev->info.dskinfo.heads * cyl) +
      ((udev->info.dskinfo.overhead + udev->info.dskinfo.sectlen) * sector) +
      udev->info.dskinfo.geoverhead;

#ifdef DEBUGDSK
   fprintf (stderr, "dskseek: real = %s. cyl = %d, head = %d, sector = %d\n",
	    udev->realdevice ? "YES" : "NO", cyl, head, sector);
   fprintf (stderr, "   dskloc = %ld\n", dskloc);
#endif

   /*
   ** Go to the specified location on disk
   */

   if (fseek (udev->infd, dskloc, SEEK_SET) < 0)
   {
      sprintf (view[0], "Device %s: seek failed: %s",
	       udev->name, strerror (ERRNO));
      snprintf (view[1], MAXVIEWLEN, "filename: %s", udev->file);
      return (-1);
   }

   return (0);
}

/***********************************************************************
* bumpsector - Bump to next sector
***********************************************************************/

static void 
bumpsector (Device *dev, int *cyl, int *head, int *sector)
{
   (*sector)++;
   if (*sector == dev->info.dskinfo.secttrk)
   {
      *sector = 0;
      (*head)++;
      if (*head == dev->info.dskinfo.heads)
      {
	 *head = 0;
	 (*cyl)++;
	 if (*cyl == dev->info.dskinfo.cyls)
	    *cyl = 0;
      }
   }
}

/***********************************************************************
* dskdocmd - Do disk command
***********************************************************************/

int 
dskdocmd (Device *dev)
{
   Device *udev;
   int cmd;
   int i;
   int len;
   int cyl, head, sector;
   int tinhibit;
   uint32 addr;
   uint16 W0, W7;
   uint16 delay;
   uint8 unit;

   /*
   ** Get common command word values
   */

   W0 = GETW0 (dev) & 0x00FF;
   W7 = GETW7 (dev);

   cmd = GETCMD (dev);
   unit = GETUNIT (dev);
   addr = ((GETADDR (dev)) + 1) & 0x1FFFFE;
   len = (GETLEN (dev) + 1) & 0xFFFE;

#ifdef DEBUGDSK
   fprintf (stderr, "%08.8ld:   pc = %04X, wp = %04X, st = %04X\n",
	    instcount, pcreg, wpreg, statreg);
   fprintf (stderr,
	    "dskdocmd: ENTRY: cmd = %d, unit = %d, addr = %06X, len = %d\n",
	    cmd, punit[unit], addr, len);
   for (i = 0; i < 16; i+=2)
      fprintf (stderr, "   ctl[%04X] %04X\n",
	       dev->devaddr+TPCSSTART+i, GETMEM0 (dev->devaddr+TPCSSTART+i));
#endif

   /*
   ** Find selected unit
   */

   if ((udev = dskfindunit (dev, unit)) == NULL)
   {
#if defined(DEBUGDSK) || defined(DEBUGDSKERR)
      fprintf (stderr, "dskdocmd: ERROR: unit = %d: Unit not selected\n",
	       punit[unit]);
#endif
      /* Drive not ready */
      W0 |= OLBIT;
      W7 |= UEBIT | ERRBIT | IDLEBIT;
      PUTW0 (dev, W0);
      PUTW7 (dev, W7);
      return (W7 & INTBIT ? dev->intlvl : 0);
   }

   if (udev->infd == DEV_NULL)
   {
#ifdef DEBUGDSK
      fprintf (stderr, "dskdocmd: unit = %d: Unit is NULL\n", punit[unit]);
#endif
      PUTW0 (udev, OLBIT);
      PUTW7 (udev, IDLEBIT | ERRBIT | UEBIT);
      return (W7 & INTBIT ? udev->intlvl : 0);
   }

   /*
   ** If command is not Store Registers or Restore, check disk parameters
   */

   cyl = head = sector = 0;
   if (cmd != STOREREG && cmd != RESTORE)
   {
      cyl = GETCYL (udev);
      if (cyl >= udev->info.dskinfo.cyls)
      {
#if defined(DEBUGDSK) || defined(DEBUGDSKERR)
	 fprintf (stderr, "dskdocmd: ERROR: unit = %d: Invalid cylinder: %d\n",
		  punit[unit], cyl);
#endif
	 W0 |= SIBIT | ((udev->switches & SWPROTECT) ? WPBIT : 0);
	 W7 |= ERRBIT | IDLEBIT | COMPBIT | UEBIT;
	 PUTW0 (udev, W0);
	 PUTW7 (udev, W7);
	 return (W7 & INTBIT ? udev->intlvl : 0);
      }
      if (cmd == DISKSEEK) /* Seeks only use cylinder */
      {
         head = 0;
	 sector = 0;
      }
      else
      {
	 head = GETHEAD (udev);
	 if (head >= udev->info.dskinfo.heads)
	 {
#if defined(DEBUGDSK) || defined(DEBUGDSKERR)
	    fprintf (stderr, "dskdocmd: ERROR: unit = %d: Invalid head: %d\n",
		     punit[unit], head);
#endif
	    W0 |= ECBIT | ((udev->switches & SWPROTECT) ? WPBIT : 0);
	    W7 |= ERRBIT | COMPBIT | IDLEBIT;
	    PUTW0 (udev, W0);
	    PUTW7 (udev, W7);
	    return (W7 & INTBIT ? udev->intlvl : 0);
	 }
	 sector = GETSECT (udev);
	 if (sector >= udev->info.dskinfo.secttrk)
	 {
#if defined(DEBUGDSK) || defined(DEBUGDSKERR)
	    fprintf (stderr, "dskdocmd: ERROR: unit = %d: Invalid sector: %d\n",
		     punit[unit], sector);
#endif
	    W0 |= TTBIT | ((udev->switches & SWPROTECT) ? WPBIT : 0);
	    W7 |= ERRBIT | COMPBIT | IDLEBIT;
	    PUTW0 (udev, W0);
	    PUTW7 (udev, W7);
	    return (W7 & INTBIT ? udev->intlvl : 0);
	 }
      }

      /*
      ** Parms OK, seek
      */

      if (dskseek (udev, cyl, head, sector) < 0)
      {
	 return (W7 & INTBIT ? udev->intlvl : 0);
      }
   }

   /*
   ** Set initial sate
   */

   udev->select = TRUE;
   udev->intenabled = FALSE;

   /*
   ** Process command
   */

   delay = 50;
   switch (cmd)
   {
   case STOREREG: /* Store registers */
#ifdef DEBUGDSK
      fprintf (stderr, "   dskcmd = STRREG: len = %d\n", len);
#endif
      if (len > 0) 
      {
         dskputmem (addr, udev->info.dskinfo.tracklen >> 1);
#ifdef DEBUGDSK
	 fprintf (stderr, "   W1 = %04X\n", udev->info.dskinfo.tracklen >> 1);
#endif
         len -= 2;
	 addr += 2;
      }
      if (len > 0) 
      {
	 
         dskputmem (addr, (udev->info.dskinfo.secttrk << 8) | 
			  (udev->info.dskinfo.overhead >> 1));
#ifdef DEBUGDSK
	 fprintf (stderr, "   W2 = %04X\n",
			(udev->info.dskinfo.secttrk << 8) | 
			(udev->info.dskinfo.overhead >> 1));
#endif
         len -= 2;
	 addr += 2;
      }
      if (len > 0) 
      {
         dskputmem (addr,
	       (udev->info.dskinfo.heads << 11) | udev->info.dskinfo.cyls);
#ifdef DEBUGDSK
	 fprintf (stderr, "   W3 = %04X\n", 
	       (udev->info.dskinfo.heads << 11) | udev->info.dskinfo.cyls);
#endif
      }
      delay = 2;
      break;

   case WRITEFMT: /* Write format */
      len = udev->info.dskinfo.tracklen;
#ifdef DEBUGDSK
      fprintf (stderr,
	    "   dskcmd = FORMAT: cyl = %d, head = %d, sector = %d, len = %d\n",
	       cyl, head, sector, len);
#endif
      if (!(udev->switches & SWPROTECT))
      {
         int truelen;
	 uint16 d;

	 d = dskgetmem (addr);

	 truelen = udev->info.dskinfo.sectlen + udev->info.dskinfo.overhead;

	 while (len)
	 {
	    int slen;

	    /*
	    ** Format one sector at a time
	    */

	    slen = len < truelen ?  len : truelen;

	    for (i = 0; i < slen; i += 2)
	    {
	       fputc ((d >> 8) & 0xFF, udev->infd);
	       fputc (d & 0xFF, udev->infd);
	    }

	    len -= slen;
	    bumpsector (udev, &cyl, &head, &sector);

	    if (len)
	       if (dskseek (udev, cyl, head, sector) < 0) 
		  return (W7 & INTBIT ? udev->intlvl : 0);
	 }
	 udev->info.dskinfo.curcyl = cyl;
	 udev->info.dskinfo.curhead = head;
	 udev->info.dskinfo.cursect = sector;
	 if (udev->realdevice)
	    fsync (fileno(udev->infd));
      }
      break;

   case READDATA: /* Read data */
#ifdef DEBUGDSK
      fprintf (stderr,
	       "   dskcmd = READ: cyl = %d, head = %d, sector = %d, len = %d\n",
	       cyl, head, sector, len);
#endif
      tinhibit = GETMEMB0 (udev->devaddr+TPCSSTART+2) & TIHBIT;
      while (len)
      {
	 int slen;

	 /*
	 ** Read one sector at a time
	 */

	 slen = (len < udev->info.dskinfo.sectlen) ?
	       len : udev->info.dskinfo.sectlen;

	 for (i = 0; i < slen; i += 2)
	 {
	    uint16 d, c;

	    d = fgetc (udev->infd);
	    c = fgetc (udev->infd);
	    if (!tinhibit)
	    {
	       d = (d << 8) | c;
	       dskputmem (addr, d);
	       addr += 2;
	    }
	 }

	 len -= slen;
	 bumpsector (udev, &cyl, &head, &sector);

	 if (len)
	    if (dskseek (udev, cyl, head, sector) < 0)
	       return (W7 & INTBIT ? udev->intlvl : 0);
      }
#ifdef DEBUGDSKDATA
      if (!tinhibit)
      {
	 HEXDUMP (stderr, &memory[(((GETADDR (dev)) + 1) & 0x1FFFFE)],
		  GETLEN (udev), (((GETADDR (dev)) + 1) & 0x1FFFFE));
   
      }
      else
      {
         fprintf (stderr, "Data inhibited\n");
      }
#endif
      udev->info.dskinfo.curcyl = cyl;
      udev->info.dskinfo.curhead = head;
      udev->info.dskinfo.cursect = sector;
      delay = 50 + len;
      break;

   case WRITEDATA: /* Write data */
#ifdef DEBUGDSK
      fprintf (stderr,
	    "   dskcmd = WRITE: cyl = %d, head = %d, sector = %d, len = %d\n",
	       cyl, head, sector, len);
#endif
      if (!(udev->switches & SWPROTECT))
      {
#ifdef DEBUGDSKDATA
	 HEXDUMP (stderr, &memory[(((GETADDR (dev)) + 1) & 0x1FFFFE)],
		  len, (((GETADDR (dev)) + 1) & 0x1FFFFE));
#endif
	 while (len)
	 {
	    int slen;

	    /*
	    ** Write one sector at a time
	    */

	    slen = (len < udev->info.dskinfo.sectlen) ?
		  len : udev->info.dskinfo.sectlen;

	    for (i = 0; i < slen; i += 2)
	    {
	       uint16 d;

	       d = dskgetmem (addr);
	       fputc ((d >> 8) & 0xFF, udev->infd);
	       fputc (d & 0xFF, udev->infd);
	       addr += 2;
	    }

	    /*
	    ** Zero fill short sectors
	    */

#ifdef DEBUGDSKDATA
	    if (i < udev->info.dskinfo.sectlen)
	       fprintf (stderr, "Pad %d bytes\n",
		     udev->info.dskinfo.sectlen - i);
#endif
	    for (; i < udev->info.dskinfo.sectlen; i++)
	       fputc ('\0', udev->infd);

	    len -= slen;
	    bumpsector (udev, &cyl, &head, &sector);

	    if (len)
	       if (dskseek (udev, cyl, head, sector) < 0)
		  return (W7 & INTBIT ? udev->intlvl : 0);
	 }

	 udev->info.dskinfo.curcyl = cyl;
	 udev->info.dskinfo.curhead = head;
	 udev->info.dskinfo.cursect = sector;
	 if (udev->realdevice)
	    fsync (fileno(udev->infd));
      }
      delay = 50 + len;
      break;

   case READUFMT: /* Unformatted read */
#ifdef DEBUGDSK
      fprintf (stderr,
	    "   dskcmd = UREAD: cyl = %d, head = %d, sector = %d, len = %d\n",
	       cyl, head, sector, len);
#endif
      if (len > 0) 
      {
         dskputmem (addr,
	       (head << 11) | cyl);
#ifdef DEBUGDSK
	 fprintf (stderr, "   W1 = %04X\n", 
		  (head << 11) | cyl);
#endif
         len -= 2;
	 addr += 2;
      }
      if (len > 0) 
      {
         dskputmem (addr, 0x100 | sector);
#ifdef DEBUGDSK
	 fprintf (stderr, "   W2 = %04X\n", 
		  0x100 | sector);
#endif
         len -= 2;
	 addr += 2;
      }
      if (len > 0) 
      {
         dskputmem (addr,
	       udev->info.dskinfo.sectlen >> 1);
#ifdef DEBUGDSK
	 fprintf (stderr, "   W3 = %04X\n", 
	       udev->info.dskinfo.sectlen >> 1);
#endif
      }
      udev->info.dskinfo.curcyl = cyl;
      udev->info.dskinfo.curhead = head;
      udev->info.dskinfo.cursect = sector;
      delay = 10;
      break;

   case WRITEUFMT: /* Unformatted write */
#ifdef DEBUGDSK
      fprintf (stderr,
	    "   dskcmd = UWRITE: cyl = %d, head = %d, sector = %d, len = %d\n",
	       cyl, head, sector, len);
#endif
      if (!(udev->switches & SWPROTECT))
      {
#ifdef DEBUGDSKDATA
	 HEXDUMP (stderr, &memory[(((GETADDR (dev)) + 1) & 0x1FFFFE)],
		  GETLEN (udev), (((GETADDR (dev)) + 1) & 0x1FFFFE));
#endif
	 for (i = 0; i < len; i += 2)
	 {
	    uint16 d;

	    d = dskgetmem (addr);
	    fputc ((d >> 8) & 0xFF, udev->infd);
	    fputc (d & 0xFF, udev->infd);
	    addr += 2;
	 }
	 udev->info.dskinfo.curcyl = cyl;
	 udev->info.dskinfo.curhead = head;
	 udev->info.dskinfo.cursect = sector;
	 if (udev->realdevice)
	    fsync (fileno(udev->infd));
      }
      break;

   case DISKSEEK: /* Seek */
#ifdef DEBUGDSK
      fprintf (stderr,
	       "   dskcmd = SEEK: cyl = %d, head = %d, sector = %d\n",
	       cyl, head, sector);
#endif
      udev->info.dskinfo.curcyl = cyl;
      udev->info.dskinfo.curhead = head;
      udev->info.dskinfo.cursect = sector;
      W0 |= (udev->unit << 4);
      delay = 200;
      break;

   case RESTORE: /* Restore */
#ifdef DEBUGDSK
      fprintf (stderr, "   dskcmd = RESTORE\n");
#endif
      udev->info.dskinfo.curcyl = 0;
      udev->info.dskinfo.curhead = 0;
      udev->info.dskinfo.cursect = 0;
      W0 |= (udev->unit << 4);
      delay = 1;
      break;

   }

   if (W7 & INTBIT)
   {
      W7 = W7 & ~INTBIT;
      udev->intenabled = TRUE;
   }
   W0 |= ((udev->switches & SWPROTECT) ? WPBIT : 0);
   W7 |= COMPBIT | IDLEBIT;
   PUTW0 (udev, W0);
   PUTW7 (udev, W7);
#ifdef DEBUGDSK
   fprintf (stderr, "   returns: intenabled = %s, delay = %d, mask = >%X\n",
	    udev->intenabled ? "TRUE" : "FALSE", delay, GET_MASK);
   for (i = 0; i < 16; i+=2)
      fprintf (stderr, "   ctl[%04X] %04X\n",
	       udev->devaddr+TPCSSTART+i, GETMEM0 (udev->devaddr+TPCSSTART+i));
#endif
   if (udev->intenabled)
      udev->intdelay = delay;
   return (0);
}

/***********************************************************************
* dskboot - Boot from device.
***********************************************************************/

int
dskboot (Device *dev)
{
   int i;
   uint16 lp;
   uint16 val;

   lp = LOADADDRESS;

   if (bootfromrom)
   {
      wpreg = GETMEM0 (TPCSSTART+0xFFFC) & 0xFFFE;
      pcreg = GETMEM0 (TPCSSTART+0xFFFE) & 0xFFFE;

      PUTREG (0, 0xFFFF);
      PUTREG (1, dev->devaddr);
      PUTREG (2, dev->unit << 8);
   }
   else
   {
      int len;
      int cyl;
      int head;
      int sect;

      if (dskseek (dev, 0, 0, 0) < 0) return (-1);

      i = dev->info.dskinfo.sectlen;
      if (fread (&memory[lp], 1, i, dev->infd) != i)
      {
	 sprintf (view[0], "Device %s: read failed: %s",
		  dev->name, strerror (ERRNO));
	 snprintf (view[1], MAXVIEWLEN, "filename: %s", dev->file);
	 return (-1);
      }

      val = GETMEM0 (lp + 0xE);
      cyl = val / dev->info.dskinfo.heads;
      head = val % dev->info.dskinfo.heads;
      sect = 0;

      pcreg = GETMEM0 (lp + 0x18);
      wpreg = 0x80;

      len = GETMEM0 (lp + 0x1A);
#ifdef DUMPBOOT
      fprintf (stderr, "dskboot: devaddr = %04X, unit = %02X\n", 
	       dev->devaddr, dev->unit);
      fprintf (stderr, "   pc = >%04X, len = %d\n", pcreg, len);
      fprintf (stderr, "cyl = 0, head = 0, sect = 0, len = %d\n", i);
      HEXDUMP (stderr, &memory[lp], i, lp);
#endif

      while (len)
      {
	 if (dskseek (dev, cyl, head, sect) < 0) return (-1);
	 i = (len < dev->info.dskinfo.sectlen) ?
	      len : dev->info.dskinfo.sectlen;
	 if (fread (&memory[lp], 1, i, dev->infd) != i)
	 {
	    sprintf (view[0], "Device %s: read failed: %s",
		     dev->name, strerror (ERRNO));
	    snprintf (view[1], MAXVIEWLEN, "filename: %s", dev->file);
	    return (-1);
	 }
#ifdef DUMPBOOT
	 fprintf (stderr, "cyl = %d, head = %d, sect = 0, len = %d\n",
		  cyl, head, i);
	 HEXDUMP (stderr, &memory[lp], i, lp);
#endif
	 lp += i;
         len -= i;

	 bumpsector (dev, &cyl, &head, &sect);
      }

      PUTREG (0, 0xABC0);
      PUTREG (1, dev->devaddr);
      PUTREG (2, dev->unit << 8);
   }

   run = TRUE;

   return (0);
}
