/***********************************************************************
*
* simeia.c - EIA port processing for the TI 990 Simulator.
*
* Changes:
*   09/17/13   DGP   Original. Split from simio.c
*   04/28/14   DGP   Changes for DNOS.
*   06/02/15   DGP   Removed USS (z/OS) support.
*   07/21/15   DGP   Added more support.
*   10/05/16   DGP   Added better LDCR control bit support.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>

#include "simdef.h"

extern int enablepanel;
extern int run;
extern uint32 mpcreg;
extern uint16 statreg;	/* The program status register */

/***********************************************************************
* eiaseto - Set CRU bit to one.
***********************************************************************/

void
eiaseto (Device *dev, uint16 devaddr, uint8 disp)
{
   SerialInfo *devinfo;

   devinfo = &dev->info.terminfo.controller.serialinfo;

#ifdef DEBUGEIA
   fprintf (stderr, "%06X: CRU %04X SBO %d\n", mpcreg, devaddr, disp);
#endif

   if (dev->infd == DEV_NULL) return;
   switch (disp)
   {
   case 9:	/* DTR */
      devinfo->dtrset = TRUE;
      break;

   case 10:	/* RTS */
      devinfo->rtsset = TRUE;
      break;

   case 11:	/* CLRWRQ */
      devinfo->wrqset = FALSE;
      if (dev->intenabled)
	 clrinterrupt (dev->intlvl);
      break;

   case 12:	/* CLRRRQ */
      dev->select = FALSE;
      if (dev->intenabled)
	 clrinterrupt (dev->intlvl);
      break;

   case 13:	/* CLRNSF */
      if (dev->intenabled)
	 clrinterrupt (dev->intlvl);
      break;

   case 14:	/* Interrupt */
      dev->intenabled = TRUE;
      break;

   case 15:	/* DIAG Mode */
      break;
   }
}

/***********************************************************************
* eiasetz - Set CRU bit to zero.
***********************************************************************/

void
eiasetz (Device *dev, uint16 devaddr, uint8 disp)
{
   SerialInfo *devinfo;

   devinfo = &dev->info.terminfo.controller.serialinfo;

#ifdef DEBUGEIA
   fprintf (stderr, "%06X: CRU %04X SBZ %d\n", mpcreg, devaddr, disp);
#endif

   if (dev->infd == DEV_NULL) return;
   switch (disp)
   {
   case 9:	/* DTR */
      devinfo->dtrset = FALSE;
      break;

   case 10:	/* RTS */
      devinfo->rtsset = FALSE;
      break;

   case 11:	/* CLRWRQ */
      devinfo->wrqset = FALSE;
      if (dev->intenabled)
	 clrinterrupt (dev->intlvl);
      break;

   case 12:	/* CLRRRQ */
      dev->select = FALSE;
      if (dev->intenabled)
	 clrinterrupt (dev->intlvl);
      break;

   case 13:	/* CLRNSF */
      if (dev->intenabled)
	 clrinterrupt (dev->intlvl);
      break;

   case 14:	/* Interrupt */
      dev->intenabled = FALSE;
      clrinterrupt (dev->intlvl);
      break;

   case 15:	/* DIAG Mode */
      break;
   }
}

/***********************************************************************
* eiatb - Test CRU bit.
***********************************************************************/

void
eiatb (Device *dev, uint16 devaddr, uint8 disp)
{
   SerialInfo *devinfo;

   devinfo = &dev->info.terminfo.controller.serialinfo;

#ifdef DEBUGEIA
   fprintf (stderr, "%06X: CRU %04X TB %d", mpcreg, devaddr, disp);
#endif

   if (dev->infd == DEV_NULL) return;
   CLR_EQ;
   switch (disp)
   {
   case 0: /* TTYI */
      if (dev->cbreak) CLR_EQ;
      else SET_EQ;
      dev->cbreak = FALSE;
      break;

   case 8:  /* XMTING */
   	break;

   case 9:  /* TIMERR */
   	break;

   case 10:  /* RCR */
   	break;

   case 11: /* WRQ */
      if (devinfo->wrqset)
         SET_EQ;
      break;

   case 12: /* RRQ */
      if (!dev->intenabled && !GET_MASK)
      {
	 int c;
	 c = serialinput (dev);
	 if (c != EOF)
	 {
	    dev->inchar = c;
	    SET_EQ;
	 }
      }
      else if (dev->select)
      {
	 SET_EQ;
      }
      break;

   case 13: /* DCD */
   case 14: /* DSR */
      SET_EQ;
      break;

   case 15: /* INT */
      if (dev->select)
         SET_EQ;
      else if (devinfo->wrqset)
         SET_EQ;
      break;

   default: ;
   }
#ifdef DEBUGEIA
   fprintf (stderr, " %s\n", IS_EQ ? "TRUE" : "FALSE");
#endif
}

/***********************************************************************
* eialdcr - Load CRU.
***********************************************************************/

void
eialdcr (Device *dev, uint16 devaddr, uint16 cnt, uint16 dat)
{
   SerialInfo *devinfo;

   devinfo = &dev->info.terminfo.controller.serialinfo;

#ifdef DEBUGEIA
   if (dat >= 0x20 && dat <= 0x7E)
      fprintf (stderr, "%06X: CRU %04X LDCR CNT %d DAT %02X(%c)\n",
	    mpcreg, devaddr, cnt, dat, dat);
   else
      fprintf (stderr, "%06X: CRU %04X LDCR CNT %d DAT %02X\n",
	       mpcreg, devaddr, cnt, dat);
#endif

   if (dev->outfd == DEV_NULL) return;

   /*
   ** If count == 0, then check control bits.
   */
   if (cnt == 0)
   {
      if (dat & 0x0200) /* DTR enable bit */
	 devinfo->dtrset = TRUE;
      else
	 devinfo->dtrset = FALSE;
      if (dat & 0x0400) /* RTS enable bit */
	 devinfo->rtsset = TRUE;
      else
	 devinfo->rtsset = FALSE;
      if (dat & 0x4000) /* Int enable bit */
	 dev->intenabled = TRUE;
      else
	 dev->intenabled = FALSE;

      devinfo->wrqset = FALSE;
      dev->select = FALSE;
   }

   if (!devinfo->rtsset) return;

   switch (dev->type)
   {
   case PRINTEIA:
      printputchar (dev, dat & 0x7F);
      break;

   case CONEIA:
   case SERIALEIA:
      serialoutput (dev, dat & 0x7F);
      break;

   default: ;
   }
   devinfo->wrqset = TRUE;
   if (dev->intenabled)
      gendevinterrupt (dev);
}

/***********************************************************************
* eiastcr - Store CRU.
***********************************************************************/

uint16
eiastcr (Device *dev, uint16 devaddr, uint16 cnt)
{
   uint16 dat;

#ifdef DEBUGEIA
   fprintf (stderr, "%06X: CRU %04X STCR CNT %d", mpcreg, devaddr, cnt);
#endif

   if (dev->infd == DEV_NULL) return (0);
   if (cnt == 0)
   {
      dat = 0x6000;
      if (dev->select)
         dat |= 0x1000;
#ifdef DEBUGEIA
      fprintf (stderr, " DAT = %4X\n", dat);
#endif
   }
   else
   {
      dat = dev->inchar;
#ifdef DEBUGEIA
      fprintf (stderr, " DAT = %2X(%c)\n", 
	       dat, (dat > 0x1F && dat < 0x7F) ? dat : '.');
#endif
   }
   return (dat);
}
