/***********************************************************************
*
* simfops.c - Simulate floating point operations for the TI 990 computer.
*
* Changes:
*   06/14/04   DGP   Original.
*   11/07/08   DGP   Added Instruction tracing and disassembler.
*   11/26/14   DGP   Improve FP Condition Code checking.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>

#include "simdef.h"
#include "ibmfloat.h"

extern uint16 pcreg;	/* The program PC */
extern uint16 statreg;	/* The program status register */
extern uint32 mpcreg;	/* Mapped PC */
extern int model;

extern int trcidx;
extern int trcopr;
extern TraceInst trcbuf[TRCBUFMAX];

/***********************************************************************
* getfloatreg - Get a float register value.
***********************************************************************/

static uint32
getfloatreg (uint16 r)
{
   return (((uint32)GETREG(r) << 16) | (uint32)GETREG(r+1));
}

/***********************************************************************
* getdoublereg - Get a double float register value.
***********************************************************************/

static t_uint64
getdoublereg (uint16 r)
{
   return (((t_uint64)GETREG(r) << 48) | ((t_uint64)GETREG(r+1) << 32) |
	   ((t_uint64)GETREG(r+2) << 16) | (t_uint64)GETREG(r+3));
}

/***********************************************************************
* putfloatreg - Put a float value into a register.
***********************************************************************/

static void
putfloatreg (uint16 r, uint32 v)
{
   PUTREG (r, (uint16)((v >> 16) & 0xFFFF));
   PUTREG (r+1, (uint16)(v & 0xFFFF));
}

/***********************************************************************
* putdoublereg - Put a double float value into a register.
***********************************************************************/

static void
putdoublereg (uint16 r, t_uint64 v)
{
   PUTREG (r, (uint16)((v >> 48) & 0xFFFF));
   PUTREG (r+1, (uint16)((v >> 32) & 0xFFFF));
   PUTREG (r+2, (uint16)((v >> 16) & 0xFFFF));
   PUTREG (r+3, (uint16)(v & 0xFFFF));
}

/***********************************************************************
* getfloat - Get a float operand. If memory access is past the end of
* defined memory, return -1;
***********************************************************************/

static uint32
getfloat (uint16 sr, uint16 st, int inc, int srcdst)
{
   uint16 sa;
   uint32 sval = -1;

   switch (st)
   {
   case 0:

      sval = getfloatreg (sr);
      break;

   case 1:

      sa = GETREG (sr);
      sval = ((uint32)GETMEM (sa & 0xFFFE, srcdst) << 16) | 
	      (uint32)GETMEM ((sa+2) & 0xFFFE, srcdst);
      break;

   case 2:

      sa = GETINST(TRUE);
      if (inc)
      {
	 trcbuf[trcidx].ops[trcopr++] = sa;
	 INCPC(2);
      }
      if (sr != 0) sa = (int16)sa + (int16)GETREG (sr);
      sval = ((uint32)GETMEM (sa & 0xFFFE, srcdst) << 16) | 
	      (uint32)GETMEM ((sa+2) & 0xFFFE, srcdst);
      break;

   case 3:

      sa = GETREG (sr);
      sval = ((uint32)GETMEM (sa & 0xFFFE, srcdst) << 16) | 
	      (uint32)GETMEM ((sa+2) & 0xFFFE, srcdst);
      if (inc) PUTREG (sr, GETREG (sr) + 4);
      break;
   }
   return (sval);
}

/***********************************************************************
* getdouble - Get a double operand. If memory access is past the end of
* defined memory, return -1;
***********************************************************************/

static t_uint64
getdouble (uint16 sr, uint16 st, int inc, int srcdst)
{
   uint16 sa;
   t_uint64 sval = -1;

   switch (st)
   {
   case 0:

      sval = getdoublereg(sr);
      break;

   case 1:

      sa = GETREG (sr);
      sval = ((t_uint64)GETMEM (sa & 0xFFFE, srcdst) << 48) | 
	     ((t_uint64)GETMEM ((sa+2) & 0xFFFE, srcdst) << 32) |
	     ((t_uint64)GETMEM ((sa+4) & 0xFFFE, srcdst) << 16) |
	      (t_uint64)GETMEM ((sa+6) & 0xFFFE, srcdst);
      break;

   case 2:

      sa = GETINST(TRUE);
      if (inc)
      {
	 trcbuf[trcidx].ops[trcopr++] = sa;
	 INCPC(2);
      }
      if (sr != 0) sa = (int16)sa + (int16)GETREG (sr);
      sval = ((t_uint64)GETMEM (sa & 0xFFFE, srcdst) << 48) | 
	     ((t_uint64)GETMEM ((sa+2) & 0xFFFE, srcdst) << 32) |
	     ((t_uint64)GETMEM ((sa+4) & 0xFFFE, srcdst) << 16) |
	      (t_uint64)GETMEM ((sa+6) & 0xFFFE, srcdst);
      break;

   case 3:

      sa = GETREG (sr);
      sval = ((t_uint64)GETMEM (sa & 0xFFFE, srcdst) << 48) | 
	     ((t_uint64)GETMEM ((sa+2) & 0xFFFE, srcdst) << 32) |
	     ((t_uint64)GETMEM ((sa+4) & 0xFFFE, srcdst) << 16) |
	      (t_uint64)GETMEM ((sa+6) & 0xFFFE, srcdst);
      if (inc) PUTREG (sr, GETREG (sr) + 8);
      break;
   }
   return (sval);
}
/***********************************************************************
* putfloat - Put a float operand.  If memory access is beyond allowed 
* memory (in ROM space) it is ignored.
***********************************************************************/

static void
putfloat (uint16 dr, uint16 dt, uint32 dval, int srcdst)
{
   uint16 da;

   switch (dt)
   {
   case 0:

      PUTREG (dr, (uint16)((dval >> 16) & 0xFFFF));
      PUTREG (dr+1, (uint16)(dval & 0xFFFF));
      break;

   case 1:

      da = GETREG (dr);
      PUTMEM (da, (uint16)((dval >> 16) & 0xFFFF), srcdst);
      PUTMEM (da+2, (uint16)(dval & 0xFFFF), srcdst);
      break;

   case 2:

      da = GETINST(TRUE);
      trcbuf[trcidx].ops[trcopr++] = da;
      INCPC(2);
      if (dr != 0) da = (int16)da + (int16)GETREG (dr);
      PUTMEM (da, (uint16)((dval >> 16) & 0xFFFF), srcdst);
      PUTMEM (da+2, (uint16)(dval & 0xFFFF), srcdst);
      break;

   case 3:

      da = GETREG (dr);
      PUTREG (dr, GETREG (dr) + 4);
      PUTMEM (da, (uint16)((dval >> 16) & 0xFFFF), srcdst);
      PUTMEM (da+2, (uint16)(dval & 0xFFFF), srcdst);
      break;
   }
}

/***********************************************************************
* putdouble - Put a double operand.  If memory access is beyond allowed 
* memory (in ROM space) it is ignored.
***********************************************************************/

static void
putdouble (uint16 dr, uint16 dt, t_uint64 dval, int srcdst)
{
   uint16 da;

   switch (dt)
   {
   case 0:

      PUTREG (dr, (uint16)((dval >> 48) & 0xFFFF));
      PUTREG (dr+1, (uint16)((dval >> 32) & 0xFFFF));
      PUTREG (dr+2, (uint16)((dval >> 16) & 0xFFFF));
      PUTREG (dr+3, (uint16)(dval & 0xFFFF));
      break;

   case 1:

      da = GETREG (dr);
      PUTMEM (da, (uint16)((dval >> 48) & 0xFFFF), srcdst);
      PUTMEM (da+2, (uint16)((dval >> 32) & 0xFFFF), srcdst);
      PUTMEM (da+4, (uint16)((dval >> 16) & 0xFFFF), srcdst);
      PUTMEM (da+6, (uint16)(dval & 0xFFFF), srcdst);
      break;

   case 2:

      da = GETINST(TRUE);
      trcbuf[trcidx].ops[trcopr++] = da;
      INCPC(2);
      if (dr != 0) da = (int16)da + (int16)GETREG (dr);
      PUTMEM (da, (uint16)((dval >> 48) & 0xFFFF), srcdst);
      PUTMEM (da+2, (uint16)((dval >> 32) & 0xFFFF), srcdst);
      PUTMEM (da+4, (uint16)((dval >> 16) & 0xFFFF), srcdst);
      PUTMEM (da+6, (uint16)(dval & 0xFFFF), srcdst);
      break;

   case 3:

      da = GETREG (dr);
      PUTREG (dr, GETREG (dr) + 8);
      PUTMEM (da, (uint16)((dval >> 48) & 0xFFFF), srcdst);
      PUTMEM (da+2, (uint16)((dval >> 32) & 0xFFFF), srcdst);
      PUTMEM (da+4, (uint16)((dval >> 16) & 0xFFFF), srcdst);
      PUTMEM (da+6, (uint16)(dval & 0xFFFF), srcdst);
      break;
   }
}

/***********************************************************************
* cmpfloatzero - Compare float value to zero.
***********************************************************************/

static void
cmpfloatzero (uint32 val)
{
   uint32 zero = 0;
   int cc;

   cc = ibm_cmps (&val, &zero);
   if (val > zero) SET_LGT;
   else CLR_LGT;
   if (cc == CC_GT) SET_AGT;
   else CLR_AGT;
   if (cc == CC_EQ) SET_EQ;
   else CLR_EQ;
}

/***********************************************************************
* cmpdoublezero - Compare double value to zero.
***********************************************************************/

static void
cmpdoublezero (t_uint64 val)
{
   t_uint64 zero = ZERO;
   int cc;

   cc = ibm_cmpl (&val, &zero);
   if (val > zero) SET_LGT;
   else CLR_LGT;
   if (cc == CC_GT) SET_AGT;
   else CLR_AGT;
   if (cc == CC_EQ) SET_EQ;
   else CLR_EQ;
}

/***********************************************************************
* checkfloatcc - Compare double value to zero.
***********************************************************************/

static void
checkfloatcc (int cc)
{
   if (cc == PGM_EXPONENT_UNDERFLOW_EXCEPTION) 
   {
      CLR_CARRY;
      SET_OVER;
   }
   else if (cc == PGM_EXPONENT_OVERFLOW_EXCEPTION)
   {
      SET_CARRY;
      SET_OVER;
   }
}

/***********************************************************************
* proc6fltop - Process type 6 floating point operations.
***********************************************************************/

int
proc6fltop (uint16 inst)
{
   int cc;
   uint16 sr;
   uint16 st;
   uint16 val;
   uint32 fop;
   uint32 acc;
   t_uint64 dop;
   t_uint64 dacc;

   if (model < 12) return (-1);

   sr = inst & 0x000F;
   st = (inst & 0x0030) >> 4;

   switch ((inst & 0x3C0) >> 6)
   {
   case 1: /* AR */

#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = AR sr = %d, st = %d\n",
	       sr, st);
#endif
      fop = getfloat (sr, st, TRUE, SRC);
      acc = getfloatreg (0);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   acc = >%08X\n", acc);
      fprintf (stderr,
	       "   fop = >%08X\n", fop);
#endif
      cc = ibm_adds (&acc, &fop);
      putfloatreg (0, acc);
      cmpfloatzero (acc);
      checkfloatcc (cc);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   cc = %d\n   acc = >%08X\n", cc, acc);
#endif
      break;

   case 2: /* CIR */

#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = CIR sr = %d, st = %d\n",
	       sr, st);
#endif
      val = getword (sr, st, TRUE, SRC);
      ibm_flts (&acc, (int16)val);
      putfloatreg (0, acc);
      cmpfloatzero (acc);
      break;

   case 3: /* SR */

#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = SR sr = %d, st = %d\n",
	       sr, st);
#endif
      fop = getfloat (sr, st, TRUE, SRC);
      acc = getfloatreg (0);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   acc = >%08X\n", acc);
      fprintf (stderr,
	       "   fop = >%08X\n", fop);
#endif
      cc = ibm_subs (&acc, &fop);
      putfloatreg (0, acc);
      cmpfloatzero (acc);
      checkfloatcc (cc);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   cc = %d\n   acc = >%08X\n", cc, acc);
#endif
      break;

   case 4: /* MR */

#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = MR sr = %d, st = %d\n",
	       sr, st);
#endif
      fop = getfloat (sr, st, TRUE, SRC);
      acc = getfloatreg (0);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   acc = >%08X\n", acc);
      fprintf (stderr,
	       "   fop = >%08X\n", fop);
#endif
      cc = ibm_mpys (&acc, &fop);
      putfloatreg (0, acc);
      cmpfloatzero (acc);
      checkfloatcc (cc);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   cc = %d\n   acc = >%08X\n", cc, acc);
#endif
      break;

   case 5: /* DR */

#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = DR sr = %d, st = %d\n",
	       sr, st);
#endif
      fop = getfloat (sr, st, TRUE, SRC);
      acc = getfloatreg (0);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   acc = >%08X\n", acc);
      fprintf (stderr,
	       "   fop = >%08X\n", fop);
#endif
      cc = ibm_divs (&acc, &fop);
      putfloatreg (0, acc);
      cmpfloatzero (acc);
      checkfloatcc (cc);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   cc = %d\n   acc = >%08X\n", cc, acc);
#endif
      break;

   case 6: /* LR */

#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = LR sr = %d, st = %d\n",
	       sr, st);
#endif
      acc = getfloat (sr, st, TRUE, SRC);
      putfloatreg (0, acc);
      cmpfloatzero (acc);
      break;

   case 7: /* STR */

#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = STR sr = %d, st = %d\n",
	       sr, st);
#endif
      acc = getfloatreg (0);
      putfloat (sr, st, acc, SRC);
      cmpfloatzero (acc);
      break;

   case 9: /* AD */

#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = AD sr = %d, st = %d\n",
	       sr, st);
#endif
      dop = getdouble (sr, st, TRUE, SRC);
      dacc = getdoublereg (0);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   dacc = >%016llX\n", dacc);
      fprintf (stderr,
	       "   dop = >%016llX\n", dop);
#endif
      cc = ibm_addl (&dacc, &dop);
      putdoublereg (0, dacc);
      cmpdoublezero (dacc);
      checkfloatcc (cc);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   cc = %d\n   dacc = >%016llX\n", cc, dacc);
#endif
      break;

   case 10: /* CID */

#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = CID sr = %d, st = %d\n",
	       sr, st);
#endif
      val = getword (sr, st, TRUE, SRC);
      ibm_fltl (&dacc, (int16)val);
      putdoublereg (0, dacc);
      cmpdoublezero (dacc);
      break;

   case 11: /* SD */

#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = SD sr = %d, st = %d\n",
	       sr, st);
#endif
      dop = getdouble (sr, st, TRUE, SRC);
      dacc = getdoublereg (0);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   dacc = >%016llX\n", dacc);
      fprintf (stderr,
	       "   dop = >%016llX\n", dop);
#endif
      cc = ibm_subl (&dacc, &dop);
      putdoublereg (0, dacc);
      cmpdoublezero (dacc);
      checkfloatcc (cc);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   cc = %d\n   dacc = >%016llX\n", cc, dacc);
#endif
      break;

   case 12: /* MD */

#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = MD sr = %d, st = %d\n",
	       sr, st);
#endif
      dop = getdouble (sr, st, TRUE, SRC);
      dacc = getdoublereg (0);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   dacc = >%016llX\n", dacc);
      fprintf (stderr,
	       "   dop = >%016llX\n", dop);
#endif
      cc = ibm_mpyl (&dacc, &dop);
      putdoublereg (0, dacc);
      cmpdoublezero (dacc);
      checkfloatcc (cc);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   cc = %d\n   dacc = >%016llX\n", cc, dacc);
#endif
      break;

   case 13: /* DD */

#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = DD sr = %d, st = %d\n",
	       sr, st);
#endif
      dop = getdouble (sr, st, TRUE, SRC);
      dacc = getdoublereg (0);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   dacc = >%016llX\n", dacc);
      fprintf (stderr,
	       "   dop = >%016llX\n", dop);
#endif
      cc = ibm_divl (&dacc, &dop);
      putdoublereg (0, dacc);
      cmpdoublezero (dacc);
      checkfloatcc (cc);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   cc = %d\n   dacc = >%016llX\n", cc, dacc);
#endif
      break;

   case 14: /* LD */

#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = LD sr = %d, st = %d\n",
	       sr, st);
#endif
      dacc = getdouble (sr, st, TRUE, SRC);
      putdoublereg (0, dacc);
      cmpdoublezero (dacc);
      break;

   case 15: /* STD */

#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = STD sr = %d, st = %d\n",
	       sr, st);
#endif
      dacc = getdoublereg (0);
      putdouble (sr, st, dacc, SRC);
      cmpdoublezero (dacc);
      break;

   default:
      return (-1);
   }
   return (0);
}

/***********************************************************************
* proc7fltop - Process type 7 floating point operations.
***********************************************************************/

int
proc7fltop (uint16 inst)
{
   t_uint64 dval;
   uint32 val;
   uint32 cval;

   if (model < 12) return (-1);

   switch (inst & 0x000F)
   {
   case 0: /* CRI */

      val = getfloatreg (0);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = CRI\n   bv = >%08X\n", val);
#endif
      CLR_CARRY;
      CLR_OVER;
      cval = ibm_fixs (&val, 0);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   av = >%08X\n", cval);
#endif
      if ((int32)cval > 32767 || (int32)cval < -32768)
      {
         SET_OVER;
	 SET_CARRY;
      }
      cval &= 0xFFFF;
      PUTREG(0, cval);
      cmpwordzero (cval);
      break;

   case 1: /* CDI */

      dval = getdoublereg (0);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = CDI\n   bv = >%016llX\n", dval);
#endif
      CLR_CARRY;
      CLR_OVER;
      cval = ibm_fixl (&dval, 0);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   av = >%08X\n", cval);
#endif
      if ((int32)cval > 32767 || (int32)cval < -32768)
      {
         SET_OVER;
	 SET_CARRY;
      }
      cval &= 0xFFFF;
      PUTREG(0, cval);
      cmpwordzero (cval);
      break;

   case 2: /* NEGR */

      val = getfloatreg (0);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = NEGR\n   bv = >%08X\n", val);
#endif
      if (val)
	 ibm_negs (&val);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   av = >%08X\n", val);
#endif
      putfloatreg (0, val);
      cmpfloatzero (val);
      break;

   case 3: /* NEGD */

      dval = getdoublereg(0);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = NEGD\n   bv = >%016llX\n", dval);
#endif
      if (dval)
	 ibm_negl (&dval);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   av = >%016llX\n", dval);
#endif
      putdoublereg (0, dval);
      cmpdoublezero (dval);
      break;

   case 4: /* CRE */

      val = getfloatreg (0);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = CRE\n   bv = >%016llX\n", dval);
#endif
      CLR_CARRY;
      CLR_OVER;
      cval = ibm_fixs (&val, 1);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   av = >%08X\n", cval);
#endif
      putfloatreg (0, cval);
      cmplongzero (cval);
      break;

   case 5: /* CDE */

      dval = getdoublereg (0);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = CDE\n   bv = >%016llX\n", dval);
#endif
      CLR_CARRY;
      CLR_OVER;
      cval = ibm_fixl (&dval, 1);
#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "   av = >%08X\n", cval);
#endif
      putfloatreg (0, cval);
      cmplongzero (cval);
      break;

   case 6: /* CER */

#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = CER\n");
#endif
      cval = getfloatreg(0);
      ibm_flts (&val, cval);
      putfloatreg (0, val);
      cmpfloatzero (val);
      break;

   case 7: /* CED */

#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = CED\n");
#endif
      cval = getfloatreg(0);
      ibm_fltl (&dval, cval);
      putdoublereg (0, dval);
      cmpdoublezero (dval);
      break;

   case 14: /* XIT */
   case 15: /* XIT */

#ifdef DEBUGFINSTR
      fprintf (stderr,
	       "INST = XIT\n");
#endif
      break;

   default:
      return (-1);
   }
   return (0);
}

