/***********************************************************************
*
* simfpy.c - FD800 floppy IO processing for the TI 990 Simulator.
*
* Changes:
*   05/29/03   DGP   Original.
*   06/20/03   DGP   Added interrupt support.
*   07/01/03   DGP   Fixed floppy unit on STCR of status.
*   07/08/03   DGP   Changed FLOPPY to return status if unit not present.
*   11/06/03   DGP   Make a seperate module.
*   01/26/04   DGP   Added memory size option.
*   12/12/08   DGP   Added some Maint. command support.
*   12/16/08   DGP   Correct unit select.
*   09/18/13   DGP   Fixed Clear Status Port to return zero.
*   10/29/13   DGP   Fixed ROM use to "force" floppy on boot.
*
* The disk is formatted as follows:
*    N bytes: raw data for each sector - the order is:
*      track 0 sector 0
*      track 0 sector 1, ...
*      track 0 sector N,
*      track 1 sector 0, ...
*      track M sector N, ...
*
***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>

#if defined (WIN32)
#define __TTYROUTINES 0
#include <conio.h>
#include <windows.h>
#include <signal.h>
#endif

#if defined(UNIX)
#include <sys/time.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <pthread.h>
#endif

#include "simdef.h"

extern uint16 pcreg;	/* The program PC */
extern uint16 statreg;	/* The program status register */
extern uint16 wpreg;	/* The program Workspace Pointer */

extern int run;
extern int devcnt;
extern int model;
extern int bootfromrom;
extern uint32 memlen;
extern char view[MAXVIEW][MAXVIEWLEN+1];

extern uint8 memory[SYSMEMSIZE];
extern Device devices[MAXDEVICES];

#ifdef DEBUGFLOPPY
static char *cmdname[] = {
   "Select", "Seek", "Restore", "Sect Len", "Read", "Read Id", "Read Unf",
   "Write", "Write Del", "Format", "Ld Int Mask", "Stop", "Step Head", 
   "", "IPL", "Clr St Port"
};
#endif

/***********************************************************************
* fpyseto - Set CRU bit to one.
***********************************************************************/

void
fpyseto (Device *dev, uint8 disp)
{
#ifdef DEBUGFLOPPY
   fprintf (stderr, "INTMASK %X CRU %04X SBO %d\n", GET_MASK, R12, disp);
#endif
   return;
}

/***********************************************************************
* fpysetz - Set CRU bit to zero.
***********************************************************************/

void
fpysetz (Device *dev, uint8 disp)
{
#ifdef DEBUGFLOPPY
   fprintf (stderr, "INTMASK %X CRU %04X SBZ %d\n", GET_MASK, R12, disp);
#endif
   return;
}

/***********************************************************************
* fpytb - Test CRU bit.
***********************************************************************/

void
fpytb (Device *dev, uint8 disp)
{
#ifdef DEBUGFLOPPY
   fprintf (stderr, "INTMASK %X CRU %04X TB %d\n", GET_MASK, R12, disp);
#endif

   switch (disp)
   {
   case 0: /* OP complete (OPCOMP) */
      if (dev->info.fpyinfo.laststat & 0x4) /* Drive not ready */
         CLR_EQ;
      else
         SET_EQ;
      break;

   case 1: /* Xfer ready (CXFER) */
   case 8:
   case 9:
   case 11:
   case 15: /* Interrupt */
      SET_EQ;
      break;

   case 10: /* Controller busy (CBUSY) */
   default:
      CLR_EQ;
   }
   return;
}

/***********************************************************************
* fpyldcr - Load CRU.
***********************************************************************/

void
fpyldcr (Device *dev, uint16 cnt, uint16 dat)
{
   int i;
   int genint;
   uint16 devaddr;
   uint16 delay;
#ifdef DEBUGFLOPPY
   uint8 ch;
   uint8 ch1;
#endif

   devaddr = R12;

#ifdef DEBUGFLOPPY
   fprintf (stderr, "INTMASK %X LDCR %04X", GET_MASK, devaddr);
#endif

   genint = dev->intenabled;
   if (devaddr == dev->devaddr) /* Data transfer */
   {
      if (dev->info.fpyinfo.ramload)
      {
#ifdef DEBUGFLOPPY
	 fprintf (stderr, " TRK %d SEC %d LOC %d(%d)", 
		  dev->info.fpyinfo.track, dev->info.fpyinfo.sector, 
		  dev->info.fpyinfo.trklocation, dev->info.fpyinfo.seclocation);
	 fprintf (stderr, " RAM value = %d\n", dat);
#endif
         dev->info.fpyinfo.ramload = FALSE;
	 return;
      }
      for (i = 0; i < devcnt; i++) /* find selected unit */
	 if (devices[i].select && devices[i].devaddr == devaddr)
	 {
#ifdef DEBUGFLOPPY
            fprintf (stderr, " UNIT %d", devices[i].unit);
#endif
	    break;
	 }
      if (i == devcnt)
      {
	 dev->info.fpyinfo.laststat = 0x4; /* Drive not ready */
	 return;
      }
      dev = &devices[i];
#ifdef DEBUGFLOPPY
      ch = (dat >> 8) & 0xFF;
      if (!isprint (ch)) ch = '.';
      ch1 = dat & 0xFF;
      if (!isprint (ch1)) ch1 = '.';
      fprintf (stderr, " DATA %04X %c%c\n", dat, ch, ch1);
#endif
      fputc ((dat >> 8) & 0xFF, dev->infd);
      fputc (dat & 0xFF, dev->infd);
      dev->count -= 2;
      if (dev->count && genint)
	 gendevinterrupt (dev);
   }

   else /* Command */
   {
      int diskloc;
      uint16 devcmd;
      uint16 cmd;
      uint16 unit;
      uint16 sec;
      uint16 trk;

      cmd = dat;
      devcmd = (cmd & 0xF000) >> 12;
      dev->info.fpyinfo.lastcmd = devcmd;
#ifdef DEBUGFLOPPY
      fprintf (stderr, " CMD %04X(%s)", cmd, cmdname[devcmd]);
#endif

      /* If Clear Status port, return */
      if (devcmd == 15)
      {
#ifdef DEBUGFLOPPY
	 fprintf (stderr, "\n");
#endif
	 for (i = 0; i < devcnt; i++) /* find selected unit */
	    if (devices[i].extaddr == devaddr)
	    {
	       dev->cbreak = FALSE;
	       dev->select = FALSE;
	       dev->info.fpyinfo.laststat = 0;
	    }
	 return;
      }

      /* Process Maint. commands here */
      else if (devcmd == 13)
      {
         switch ((cmd & 0x0F00) >> 8)
	 {
	 case 0: /* Reset */
#ifdef DEBUGFLOPPY
	    fprintf (stderr, " Reset\n");
#endif
	    dev->cbreak = FALSE;
	    dev->select = FALSE;
	    dev->info.fpyinfo.ramload = FALSE;
	    break;
	 case 5: /* RAM LOAD */
#ifdef DEBUGFLOPPY
	    fprintf (stderr, " RAM LOAD, offset = %x\n", cmd & 0xFF);
#endif
	    dev->info.fpyinfo.ramload = TRUE;
	    break;
	 default: 
#ifdef DEBUGFLOPPY
	    fprintf (stderr, " Maint command: %x\n", (cmd & 0xF00) >> 8);
#endif
	    break;
	 }
	 return;
      }

      unit = (cmd & 0x0C00) >> 10;
#ifdef DEBUGFLOPPY
      fprintf (stderr, " UNIT %d", unit);
#endif

      for (i = 0; i < devcnt; i++) /* New command clear prior unit selection */
	 if (devices[i].extaddr == devaddr)
	 {
	    devices[i].select = FALSE;
	    devices[i].cbreak = FALSE;
	 }
      for (i = 0; i < devcnt; i++) /* find selected unit */
	 if (devices[i].extaddr == devaddr && devices[i].unit == unit)
	    break;
      if (i == devcnt)
      {
#ifdef DEBUGFLOPPY
	 fprintf (stderr, " not selected\n");
#endif
	 dev->info.fpyinfo.laststat = 0x4 | (unit << 13); /* Drive not ready */
	 if (genint)
	    gendevinterrupt (dev);
	 return;
      }
      dev->info.fpyinfo.laststat = 0;
      dev = &devices[i];

      delay = 100;
      switch (devcmd)
      {
      case 0: /* Select */
	 dev->select = TRUE;
	 delay = 1;
	 break;

      case 14: /* IPL */
         dev->count = FPYSECTLEN;
	 /* Fall through */

      case 1: /* Seek */
	 trk = cmd & 0x007F;
	 diskloc = (FPYSECTLEN * FPYSECTTRK) * trk;
	 if (fseek (dev->infd, diskloc, SEEK_SET) < 0)
	 {
	    sprintf (view[0], "Device %s: seek failed: %s",
		     dev->name, strerror (ERRNO));
	    snprintf (view[1], MAXVIEWLEN, "filename: %s", dev->file);
	    run = FALSE;
	    return;
	 }
#ifdef DEBUGFLOPPY
	 fprintf (stderr, " TRACK %d LOC %ld", trk, diskloc);
#endif
	 dev->info.fpyinfo.track = trk;
	 dev->info.fpyinfo.trklocation = diskloc;
	 dev->select = TRUE;
	 break;

      case 2: /* Restore */
	 dev->select = TRUE;
	 dev->info.fpyinfo.track = 0;
	 dev->info.fpyinfo.sector = 1;
	 dev->info.fpyinfo.trklocation = 0;
	 dev->info.fpyinfo.seclocation = 0;
	 delay = 1;
	 break;

      case 3: /* Sector Length */
	 dev->select = TRUE;
	 delay = 1;
	 break;

      case 4: /* Read */
      case 6: /* Read Unformatted */
      case 7: /* Write */
      case 8: /* Write Delete */
	 sec = cmd & 0x001F;
	 diskloc = dev->info.fpyinfo.trklocation + (FPYSECTLEN * (sec-1));
#ifdef DEBUGFLOPPY
	 fprintf (stderr, " SECTOR %d LOC %ld", sec, diskloc);
#endif
	 if (fseek (dev->infd, diskloc, SEEK_SET) < 0)
	 {
	    sprintf (view[0], "Device %s: seek failed: %s",
		     dev->name, strerror (ERRNO));
	    snprintf (view[1], MAXVIEWLEN, "filename: %s", dev->file);
	    run = FALSE;
	    return;
	 }
	 dev->info.fpyinfo.sector = sec;
	 dev->info.fpyinfo.seclocation = diskloc;
         dev->count = FPYSECTLEN;
	 dev->select = TRUE;
	 break;

      case 5: /* Read ID */
	 break;

      case 9: /* Format Track */
	 trk = cmd & 0x007F;
	 diskloc = (FPYSECTLEN * FPYSECTTRK) * trk;
#ifdef DEBUGFLOPPY
	 fprintf (stderr, " TRACK %d LOC %ld", trk, diskloc);
#endif
	 if (fseek (dev->infd, diskloc, SEEK_SET) < 0)
	 {
	    sprintf (view[0], "Device %s: seek failed: %s",
		     dev->name, strerror (ERRNO));
	    snprintf (view[1], MAXVIEWLEN, "filename: %s", dev->file);
	    run = FALSE;
	    return;
	 }
	 for (i = 0; i < FPYSECTTRK*FPYSECTLEN; i++) fputc ('\0', dev->infd);
	 dev->info.fpyinfo.track = trk + 1;
	 dev->info.fpyinfo.trklocation = diskloc;
	 dev->select = TRUE;
	 break;

      case 10: /* Load Int Mask */
	 dev->select = TRUE;
	 dev->intenabled = TRUE;
	 genint = TRUE;
	 delay = 1;
	 break;

      case 11: /* Stop */
	 dev->select = TRUE;
	 dev->cbreak = TRUE;
	 dev->count = 0;
	 delay = 1;
	 break;

      case 12: /* Step Head */
	 delay = 10;
	 break;

      default: ; 
      }
#ifdef DEBUGFLOPPY
      fprintf (stderr, "\n");
#endif
      if (genint)
	 dev->intdelay = delay;
   }
   return;
}

/***********************************************************************
* fpystcr - Store CRU.
***********************************************************************/

uint16
fpystcr (Device *dev, uint16 cnt)
{
   int i;
   int genint;
   uint16 devaddr;
   uint16 dat;
#ifdef DEBUGFLOPPY
   uint8 ch;
   uint8 ch1;
#endif

   dat = 0;
   devaddr = R12;

#ifdef DEBUGFLOPPY
   fprintf (stderr, "INTMASK %X STCR %04X", GET_MASK, devaddr);
#endif

   genint = dev->intenabled;
   if (devaddr == dev->devaddr) /* Data transfer */
   {
      uint16 d;
      uint16 c;

      for (i = 0; i < devcnt; i++) /* find selected unit */
	 if (devices[i].select && devices[i].devaddr == devaddr)
	 {
#ifdef DEBUGFLOPPY
            fprintf (stderr, " UNIT %d", devices[i].unit);
#endif
	    break;
	 }
      if (i == devcnt)
      {
	 dev->info.fpyinfo.laststat = 0x4; /* Drive not ready */
	 return (dat);
      }
      dev = &devices[i];
      d = fgetc (dev->infd);
      c = fgetc (dev->infd);
      d = (d << 8) | c;
      dat = d;
#ifdef DEBUGFLOPPY
      ch = (dat >> 8) & 0xFF;
      if (!isprint (ch)) ch = '.';
      ch1 = dat & 0xFF;
      if (!isprint (ch1)) ch1 = '.';
      fprintf (stderr, " DATA %04X %c%c\n", dat, ch, ch1);
#endif
      dev->count -= 2;
      if (dev->count && genint)
	 gendevinterrupt (dev);
   }

   else /* Command */
   {
      uint16 unit = 0;

      for (i = 0; i < devcnt; i++)
	 if (devices[i].select && devices[i].extaddr == devaddr)
	 {
	    dev = &devices[i];
	    unit = dev->unit;
#ifdef DEBUGFLOPPY
	    fprintf (stderr, " UNIT %d", unit);
#endif

	    if (dev->cbreak)
	    {
	       dev->cbreak = FALSE;
	       dev->select = FALSE;
	    }
	    break;
	 }

      if (dev->info.fpyinfo.lastcmd == 15)
         dat = 0;
      else if (dev->info.fpyinfo.laststat)
	 dat = dev->info.fpyinfo.laststat | 0x8001;
      else
	 dat = 0x8003 | (unit << 13);

#ifdef DEBUGFLOPPY
      fprintf (stderr, " DATA %04X\n", dat);
#endif
   }

   return (dat);
}

/***********************************************************************
* fpyboot - Boot from device.
***********************************************************************/

int
fpyboot (Device *dev)
{
   uint16 lp;

   lp = LOADADDRESS;

   if (bootfromrom)
   {
      /*
      ** Depending on model "patch" ROM to force floppy boot.
      */

      if ((model == 5) && (GETMEM0 (TPCSSTART+0xFC12) == 0x0700))
      {
	 PUTMEM0 (TPCSSTART+0xFC12, 0x0200);
	 PUTMEM0 (TPCSSTART+0xFC14, dev->devaddr);
	 PUTMEM0 (TPCSSTART+0xFC16, 0xC040);

	 PUTMEM0 (TPCSSTART+0xFFF4, 0x39C6);
      }
      else if (((model == 10) || (model == 11)) &&
	       (GETMEM0 (TPCSSTART+0xFC12) == 0x0700))
      {
	 PUTMEM0 (TPCSSTART+0xFC12, 0x0200);
	 PUTMEM0 (TPCSSTART+0xFC14, dev->devaddr);
	 PUTMEM0 (TPCSSTART+0xFC16, 0xC040);

	 PUTMEM0 (TPCSSTART+0xFFF4, 0xAD70);
      }
      else if ((model == 12) && (GETMEM0 (TPCSSTART+0xFC5C) == 0x0700))
      {
	 PUTMEM0 (TPCSSTART+0xFC5C, 0x0200);
	 PUTMEM0 (TPCSSTART+0xFC5E, dev->devaddr);
	 PUTMEM0 (TPCSSTART+0xFC60, 0xC040);

	 PUTMEM0 (TPCSSTART+0xFEB4, 0xCE43);
	 PUTMEM0 (TPCSSTART+0xFEB6, 0x0459);
      }

      if (model == 4 || model == 9)
      {
	 wpreg = GETMEM0 (0xFFFC) & 0xFFFE;
	 pcreg = GETMEM0 (0xFFFE) & 0xFFFE;
      }
      else
      {
	 wpreg = GETMEM0 (TPCSSTART+0xFFFC) & 0xFFFE;
	 pcreg = GETMEM0 (TPCSSTART+0xFFFE) & 0xFFFE;
      }
      PUTREG (2, LOADADDRESS);
   }
   else 
   {
      int i;
      uint16 val;
      int sect;

      if (fseek (dev->infd, 0, SEEK_SET) < 0)
      {
	 sprintf (view[0], "Device %s: seek failed: %s",
		  dev->name, strerror (ERRNO));
	 snprintf (view[1], MAXVIEWLEN, "filename: %s", dev->file);
	 return (-1);
      }
      sect = 0;

      do {

	 for (i = 0; i < FPYSECTLEN; i+=2)
	 {
	    int c;

	    if ((c = fgetc (dev->infd)) == EOF)
	    {
	       sprintf (view[0], "Device %s: read failed: %s",
			dev->name, strerror (ERRNO));
	       snprintf (view[1], MAXVIEWLEN, "filename: %s", dev->file);
	       return (-1);
	    }
	    val = (c & 0xFF) << 8;
	    if ((c = fgetc (dev->infd)) == EOF)
	    {
	       sprintf (view[0], "Device %s: read failed: %s",
			dev->name, strerror (ERRNO));
	       snprintf (view[1], MAXVIEWLEN, "filename: %s", dev->file);
	       return (-1);
	    }
	    val = val | (c & 0xFF);
	    PUTMEM0 (lp, val);
	    lp += 2;
	    if (lp >= memlen)
	    {
	       sprintf (view[0], "Device %s: Image larger than memory",
			dev->name);
	       snprintf (view[1], MAXVIEWLEN, "filename: %s", dev->file);
	       return (-1);
	    }
	 }

	 sect++;
	 lp -= 2;

      } while (sect < (FPYSECTTRK * FPYTRKCYL) && (val != 0xFFFF));
#ifdef DUMPBOOT
      HEXDUMP (stderr, &memory[LOADADDRESS], lp - LOADADDRESS, LOADADDRESS);
#endif

      wpreg = 0x80;
      PUTMEM0 (LOADADDRESS, (dev->unit << 10));
      pcreg = LOADADDRESS + 2;

      PUTREG (2, 0xFFFF);
   }

   run = TRUE;

   return (0);
}
