/***********************************************************************
*
* simh19.c - H19 Terminal support for the TI 990 Simulator.
*
* Changes:
*   04/22/14   DGP   Split from simterm.c
*   05/26/15   DGP   Change to use screen buffer/image.
*   06/02/15   DGP   Removed USS (z/OS) support.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>

#include "simdef.h"
#include "screen.h"

extern int run;
 
/***********************************************************************
* H19init - H19 Initialization.
***********************************************************************/

void
H19init (Device *dev)
{
   VDTInfo *vdtinfo;
   int r;
   int c;

   vdtinfo = &dev->info.terminfo.terminal.vdtinfo;

#ifdef DEBUGH19
      fprintf (stderr, "H19init\n");
#endif
   if ((vdtinfo->screen =
          (uint8 *) malloc (VDT_NUM_ROWS * VDT_NUM_COLS * 5)) == NULL)
   {
      fprintf (stderr, "H19init: can't allocate screen buffer\n");
      return;
   }

   vdtinfo->Fill = ' ';
   vdtinfo->Attr = VDT_ATTR_DEFAULT;

   for (r = 0; r < VDT_NUM_ROWS; r++)
   {
      for (c = 0; c < VDT_NUM_COLS; c++)
      {
	 vdtinfo->Buffer[r][c] = vdtinfo->Fill;
	 vdtinfo->Shadow[r][c] = vdtinfo->Attr;
      }
   }
   vdtinfo->Row = VDT_MIN_ROW;
   vdtinfo->Col = VDT_MIN_COL;

   for (c = 0; c < VDT_NUM_COLS; c++)
   {
      vdtinfo->StatusLine[0][c] = ' ';
   }
   vdtinfo->VideoOn = 0x02;
   vdtinfo->CursorOn = 0x01;
   vdtinfo->AlternateCharacterSet = 0;
   vdtinfo->LockOn = 0x00;
   vdtinfo->GraphicCharSet = FALSE;
   vdtinfo->SaveRow = 0;
   vdtinfo->SaveCol = 0;
}

/***********************************************************************
* H19input - H19 input emulation.
***********************************************************************/

void
H19input (Device *dev, int fd)
{
   TermRing *inring;
   uint8 buf[2];

   inring = &dev->info.terminfo.inring;

#ifdef DEBUGH19
   fprintf (stderr, "h19input: entered: devaddr = >%04X\n", dev->devaddr);
#endif

   termdgetchar (dev, fd, buf);

#ifdef DEBUGH19
   fprintf (stderr, "   c1 = %02X(%c)\n", buf[0], buf[0]);
#endif

   switch (buf[0])
   {
   case 'O':
      termdgetchar (dev, fd, buf);
#ifdef DEBUGH19
      fprintf (stderr, "   c2 = %02X(%c)\n", buf[0], buf[0]);
#endif
      switch (buf[0])
      {
      case 'F': /* END - End insert char */
	 termdputring (dev, inring, ESCAPE);
	 termdputring (dev, inring, 'O');
	 break;
      case 'P': /* F1 */
	 termdputring (dev, inring, ESCAPE);
	 termdputring (dev, inring, 'S');
	 break;
      case 'Q': /* F2 */
	 termdputring (dev, inring, ESCAPE);
	 termdputring (dev, inring, 'T');
	 break;
      case 'R': /* F3 */
	 termdputring (dev, inring, ESCAPE);
	 termdputring (dev, inring, 'U');
	 break;
      case 'S': /* F4 */
	 termdputring (dev, inring, ESCAPE);
	 termdputring (dev, inring, 'V');
	 break;
      default: ;
      }
      break;

   case '[':
      termdgetchar (dev, fd, buf);
#ifdef DEBUGH19
      fprintf (stderr, "   c2 = %02X(%c)\n", buf[0], buf[0]);
#endif
      switch (buf[0])
      {
      case 'A': /* Up arrow */
	 termdputring (dev, inring, ESCAPE);
	 termdputring (dev, inring, 'A');
	 break;
      case 'B': /* Down arrow */
	 termdputring (dev, inring, ESCAPE);
	 termdputring (dev, inring, 'B');
	 break;
      case 'C': /* Right arrow */
	 termdputring (dev, inring, ESCAPE);
	 termdputring (dev, inring, 'C');
	 break;
      case 'D': /* Left arrow */
	 termdputring (dev, inring, ESCAPE);
	 termdputring (dev, inring, 'D');
	 break;
      case '1':
	 termdgetchar (dev, fd, buf);
#ifdef DEBUGH19
	 fprintf (stderr, "   c3 = %02X(%c)\n", buf[0], buf[0]);
#endif
	 switch (buf[0])
	 {
	 case '5': /* F5 */
	    termdeatchar (dev, fd, buf);
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'W');
	    break;
	 case '7': /* F6 - BLUE */
	    termdeatchar (dev, fd, buf);
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'P');
	    break;
	 case '8': /* F7 - RED */
	    termdeatchar (dev, fd, buf);
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'Q');
	    break;
	 case '9': /* F8 - GRAY */
	    termdeatchar (dev, fd, buf);
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'R');
	    break;
	 case '~': /* HOME */
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'H');
	    break;
	 default: 
	    termdeatchar (dev, fd, buf);
	 }
	 break;
      case '2':
	 termdgetchar (dev, fd, buf);
#ifdef DEBUGH19
	 fprintf (stderr, "   c3 = %02X(%c)\n", buf[0], buf[0]);
#endif
	 switch (buf[0])
	 {
	 case '0': /* F9 - Insert Line */
	    termdeatchar (dev, fd, buf);
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'L');
	    break;
	 case '1': /* F10 - Delete Line */
	    termdeatchar (dev, fd, buf);
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'M');
	    break;
	 case '3': /* F11 */
	    termdeatchar (dev, fd, buf);
	    break;
	 case '4': /* F12 */
	    termdeatchar (dev, fd, buf);
	    break;
	 case '~': /* INSERT - Enter insert char */
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, '@');
	    break;
	 default: 
	    termdeatchar (dev, fd, buf);
	 }
	 break;
      case '3':
	 termdgetchar (dev, fd, buf);
#ifdef DEBUGH19
	 fprintf (stderr, "   c3 = %02X(%c)\n", buf[0], buf[0]);
#endif
	 switch (buf[0])
	 {
	 case '~': /* DELETE - Delete char */
	    termdeatchar (dev, fd, buf);
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'N');
	    break;
	 default: 
	    termdeatchar (dev, fd, buf);
	 }
	 break;
      case '5': /* PgUp */
	 termdeatchar (dev, fd, buf);
	 break;

      case '6': /* PgDn */
	 termdeatchar (dev, fd, buf);
	 break;

      default: 
	 termdeatchar (dev, fd, buf);
      }
      break;

   default: ;
   }

}

/***********************************************************************
* H19process - H19 process output emulation.
***********************************************************************/

void
H19process (Device *dev)
{
   VDTInfo *vdtinfo;
   TermInfo *terminfo;
   TermRing *prcring;
   TermRing *inring;
   int c;
   int num;
   int row, col;
   int first;

   terminfo = &dev->info.terminfo;
   prcring = &terminfo->prcring;
   inring = &terminfo->inring;
   vdtinfo = &terminfo->terminal.vdtinfo;

   while (!termdringempty (dev, prcring))
   {
      first = prcring->tail;
      c = termdgetring (dev, prcring);

#ifdef DEBUGH19
      fprintf (stderr,
	       "H19process: c = >%02X(%c), first = %d, head = %d, tail = %d\n",
	       c, isprint (c) ? c : '.', first, prcring->head, prcring->tail);
#endif

      switch (c)
      {
      case 0x00:
	 break;

      case 0x07: /* BEL */
	 break;

      case 0x08: /* BS */
	 vdtinfo->Col--;
	 VDT_AdjustCursor (dev);
	 break;

      case 0x0A: /* LF */
	 vdtinfo->Row++;
	 VDT_AdjustCursor (dev);
	 break;

      case 0x0D: /* CR */
	 vdtinfo->Col = VDT_MIN_COL;
	 VDT_AdjustCursor (dev);
	 break;

      case ESCAPE: /* ESCAPE */
	 if ((c = termdgetcond (dev, prcring, first)) < 0) return;
	 switch (c)
	 {
	 case '@': /* ESC @ - Insert character mode on */
#ifdef DEBUGH19
	    fputs ("[Insert char mode on]\n", stderr);
#endif
	    break;

	 case 'A': /* ESC A - Cursor up */
#ifdef DEBUGH19
	    fputs ("[Cursor up]\n", stderr);
#endif
	    vdtinfo->Row--;
	    VDT_AdjustCursor (dev);
	    break;

	 case 'B': /* ESC B - Cursor down */
#ifdef DEBUGH19
	    fputs ("[Cursor down]\n", stderr);
#endif
	    vdtinfo->Row++;
	    VDT_AdjustCursor (dev);
	    break;

	 case 'C': /* ESC C - Cursor forward */
#ifdef DEBUGH19
	    fputs ("[Cursor forward]\n", stderr);
#endif
	    vdtinfo->Col++;
	    VDT_AdjustCursor (dev);
	    break;

	 case 'D': /* ESC D - Cursor backward */
#ifdef DEBUGH19
	    fputs ("[Cursor backward]\n", stderr);
#endif
	    vdtinfo->Col--;
	    VDT_AdjustCursor (dev);
	    break;

	 case 'E': /* ESC E - Clear screen */
#ifdef DEBUGH19
	    fputs ("[Clear Screen]\n", stderr);
#endif
	    vdtinfo->Row = VDT_MIN_ROW;
	    vdtinfo->Col = VDT_MIN_COL;
	    vdtinfo->Fill = ' ';
	    vdtinfo->Attr = VDT_ATTR_DEFAULT;
	    vdtinfo->GraphicCharSet = FALSE;
	    vdtinfo->AlternateCharacterSet = 0x00;
	    VDT_AdjustCursor (dev);
	    VDT_EraseArea (dev, VDT_MIN_COL, VDT_MIN_COL,
			   VDT_MAX_ROW, VDT_MAX_COL);
	    break;

	 case 'F': /* ESC F - Enter graphics mode */
#ifdef DEBUGH19
	    fputs ("[Enter graphics mode]\n", stderr);
#endif
	    vdtinfo->GraphicCharSet = TRUE;
	    break;

	 case 'G': /* ESC G - Exit graphics mode */
#ifdef DEBUGH19
	    fputs ("[Exit graphics mode]\n", stderr);
#endif
	    vdtinfo->GraphicCharSet = FALSE;
	    break;

	 case 'H': /* ESC H - Home */
#ifdef DEBUGH19
	    fputs ("[Cursor Home]\n", stderr);
#endif
	    vdtinfo->Row = VDT_MIN_ROW;
	    vdtinfo->Col = VDT_MIN_COL;
	    VDT_AdjustCursor (dev);
	    break;

	 case 'J': /* ESC J - Erase to EOP */
#ifdef DEBUGH19
	    fputs ("[Erase EOP]\n", stderr);
#endif
	    VDT_EraseArea (dev, vdtinfo->Row, vdtinfo->Col,
			   vdtinfo->Row, VDT_MAX_COL);
	    VDT_EraseArea (dev, vdtinfo->Row + 1, VDT_MIN_COL,
			   VDT_MAX_ROW, VDT_MAX_COL);
	    break;

	 case 'K': /* ESC K - Erase to EOL */
#ifdef DEBUGH19
	    fputs ("[Erase EOL]\n", stderr);
#endif
	    if (vdtinfo->StatusActive)
	    {
	       for (col = vdtinfo->StatusCol; col < VDT_MAX_COL; col++)
		  vdtinfo->StatusLine[0][col] = ' ';
	    }
	    else
	    {
	       VDT_EraseArea (dev, vdtinfo->Row, vdtinfo->Col,
			      vdtinfo->Row, VDT_MAX_COL);
	    }
	    break;

	 case 'L': /* ESC L - Insert line */
#ifdef DEBUGH19
	    fputs ("[Insert line]\n", stderr);
#endif
	    vdtinfo->LineExtentRow = VDT_MAX_ROW;
	    VDT_InsertDelete (dev, TRUE, TRUE);
	    break;

	 case 'M': /* ESC M - Delete line */
#ifdef DEBUGH19
	    fputs ("[Delete line]\n", stderr);
#endif
	    vdtinfo->LineExtentRow = VDT_MAX_ROW;
	    VDT_InsertDelete (dev, FALSE, TRUE);
	    break;

	 case 'N': /* ESC N - Delete character */
#ifdef DEBUGH19
	    fputs ("[Delete character]\n", stderr);
#endif
	    VDT_InsertDelete (dev, FALSE, FALSE);
	    break;

	 case 'O': /* ESC O - Insert char mode off */
#ifdef DEBUGH19
	    fputs ("[Insert char mode off]\n", stderr);
#endif
	    break;

	 case 'Y': /* ESC Y Row Col - Set Cursor position */
	    if ((row = termdgetcond (dev, prcring, first)) < 0) return;
	    if ((col = termdgetcond (dev, prcring, first)) < 0) return;
#ifdef DEBUGH19
	    fprintf (stderr, "[Set Cursor position]{%c,%c}", row, col);
#endif
	    row -= 31;
	    col -= 31;
#ifdef DEBUGH19
	    fprintf (stderr, "->{%d,%d}\n", row, col);
#endif
	    if (vdtinfo->StatusActive && row == VDT_NUM_ROWS)
	    {
	       vdtinfo->StatusCol = col - 1;
	    }
	    else
	    {
	       if (row < VDT_MIN_ROW || row > VDT_MAX_ROW)
		  row = VDT_MIN_ROW;
	       if (col < VDT_MIN_COL || col > VDT_MAX_COL)
		  col = VDT_MIN_COL;

	       vdtinfo->Row = row;
	       vdtinfo->Col = col;
	       VDT_AdjustCursor (dev);
	    }
	    break;

	 case 'b':  /* ESC b - Erase from beginning of screen */
#ifdef DEBUGH19
	    fputs ("[Erase begin screen]\n", stderr);
#endif
	    VDT_EraseArea (dev, VDT_MIN_ROW, VDT_MAX_COL,
			   vdtinfo->Row - 1, VDT_MAX_COL);
	    VDT_EraseArea (dev, vdtinfo->Row, VDT_MIN_COL,
			   vdtinfo->Row, vdtinfo->Col);
	    break;

	 case 'j':  /* ESC j - Save cursor and modes */
#ifdef DEBUGH19
	    fputs ("[Save Cursor]\n", stderr);
#endif
	    vdtinfo->SaveRow = vdtinfo->Row;
	    vdtinfo->SaveCol = vdtinfo->Col;
	    break;

	 case 'k':  /* ESC k - Restore cursor and modes */
#ifdef DEBUGH19
	    fputs ("[Restore Cursor]\n", stderr);
#endif
	    vdtinfo->Row = vdtinfo->SaveRow;
	    vdtinfo->Col = vdtinfo->SaveCol;
	    vdtinfo->StatusActive = FALSE;
	    VDT_AdjustCursor (dev);
	    break;

	 case 'l': /* ESC l - Erase line */
#ifdef DEBUGH19
	    fputs ("[Erase line]\n", stderr);
#endif
	    VDT_EraseArea (dev, vdtinfo->Row, VDT_MIN_COL,
			   vdtinfo->Row, VDT_MAX_COL);
	    break;

	 case 'n': /* ESC n - Cursor report */
#ifdef DEBUGH19
	    fprintf (stderr, "[Cursor report]{%d,%d}->{%c,%c}\n",
		     vdtinfo->Row, vdtinfo->Col,
		     vdtinfo->Row + 31, vdtinfo->Col + 31);
#endif
	    termdputring (dev, inring, ESCAPE);
	    termdputring (dev, inring, 'Y');
	    termdputring (dev, inring, vdtinfo->Row + 31);
	    termdputring (dev, inring, vdtinfo->Col + 31);
	    break;

	 case 'o': /* ESC o - Erase from beginning of line */
#ifdef DEBUGH19
	    fputs ("[Erase begin line]\n", stderr);
#endif
	    VDT_EraseArea (dev, vdtinfo->Row, VDT_MIN_COL,
			   vdtinfo->Row, vdtinfo->Col);
	    break;

	 case 'p': /* ESC p - Enter reverse video */
#ifdef DEBUGH19
	    fputs ("[Enter reverse video]\n", stderr);
#endif
	    vdtinfo->Attr = VDT_ATTR_REVERSE;
	    break;

	 case 'q': /* ESC q - Exit reverse video */
#ifdef DEBUGH19
	    fputs ("[Exit reverse video]\n", stderr);
#endif
	    vdtinfo->Attr = VDT_ATTR_DEFAULT;
	    break;

	 case 'r': /* ESC r Rate - Set bit rate */
	    if ((num = termdgetcond (dev, prcring, first)) < 0) return;
#ifdef DEBUGH19
	    fprintf (stderr, "[Set Bit Rate]{%d}\n", num);
#endif
	    break;

	 case 't': /* ESC t - Shifted keypad mode */
#ifdef DEBUGH19
	    fputs ("[Shifted keypad mode]\n", stderr);
#endif
	    break;

	 case 'x': /* ESC x Mode - Set mode bit */
	    if ((num = termdgetcond (dev, prcring, first)) < 0) return;
#ifdef DEBUGH19
	    fprintf (stderr, "[Set mode bit]{%c}\n", num);
#endif
	    if (num == '1') /* Enable line 25 */
	    {
	       vdtinfo->StatusActive = TRUE;
	       vdtinfo->StatusCol = 0;
	    }
	    break;

	 case 'y': /* ESC y Mode - Reset mode bit */
	    if ((num = termdgetcond (dev, prcring, first)) < 0) return;
#ifdef DEBUGH19
	    fprintf (stderr, "[Reset mode bit]{%c}\n", num);
#endif
	    if (num == '1') /* Disable line 25 */
	    {
	       vdtinfo->StatusActive = FALSE;
	    }
	    break;

	 case 'z': /* ESC z - Reset configuration */
#ifdef DEBUGH19
	    fputs ("[Reset configuration]\n", stderr);
#endif
	    vdtinfo->Row = VDT_MIN_ROW;
	    vdtinfo->Col = VDT_MIN_COL;
	    vdtinfo->StatusActive = FALSE;
	    vdtinfo->GraphicCharSet = FALSE;
	    vdtinfo->StatusCol = 0;
	    vdtinfo->SaveRow = 0;
	    vdtinfo->SaveCol = 0;
	    break;

	 default: ;
#ifdef DEBUGH19
	    fprintf (stderr, "[Unsupported %c(%02X)]\n", c, c);
#endif
	 }
	 break;

      default:
	 if (vdtinfo->StatusActive)
	 {
	    if (vdtinfo->StatusCol < VDT_MAX_COL)
	       vdtinfo->StatusLine[0][vdtinfo->StatusCol++] = c;
	 }
	 else
	 {
	    VDT_PutChar (dev, c, 1);
	 }
      }
   }
}

/***********************************************************************
* H19display - H19 display.
***********************************************************************/

int
H19display (Device *dev, int fd)
{
   VDTInfo *vdtinfo;
   TermInfo *terminfo;
   uint8 *bp, *sp;
   int row, col;
   int status;

   if (!(dev->switches & SWTELNET) && !run) return (0);
#ifdef DEBUGH19
   if (dev->cbreak)
      fprintf (stderr,
	       "H19display: devaddr = >%04X, fd = %d, cbreak = %s\n",
	       dev->devaddr, fd, dev->cbreak ? "TRUE" : "FALSE");
#endif
   terminfo = &dev->info.terminfo;
   vdtinfo = &terminfo->terminal.vdtinfo;

   bp = sp = vdtinfo->screen;
   status = 0;
   if (dev->cbreak)
   {
      uint8 LastAttr;

      if (termdringempty (dev, &terminfo->outring))
	 dev->cbreak = FALSE;

      sp = VDT_DisplayClear (dev, sp);

      LastAttr = VDT_ATTR_DEFAULT;
      sp = VDT_DisplayAttr (dev, sp, LastAttr);

      for (row = 0; sp && row < VDT_NUM_ROWS; row++)
      {
	 sp = VDT_DisplayPosition (dev, sp, row + 1, 1);

	 for (col = 0; col < VDT_NUM_COLS; col++)
	 {
	    if (vdtinfo->Shadow[row][col] != LastAttr)
	    {
	       sp = VDT_DisplayScreen (dev, fd, bp, sp - bp, FALSE);
	       bp = sp;
	       LastAttr = vdtinfo->Shadow[row][col];
	       sp = VDT_DisplayAttr (dev, sp, LastAttr);
	    }
	    *sp++ = vdtinfo->Buffer[row][col];
	 }
#ifdef DEBUGH19SCREEN
	 fprintf (stderr, "\nBuffer: row = %d\n", row);
         HEXDUMP (stderr, (char *)vdtinfo->Buffer[row], VDT_NUM_COLS, 0);
#endif
	 sp = VDT_DisplayScreen (dev, fd, bp, sp - bp, FALSE);
	 bp = sp;
      }

      sp = VDT_DisplayPosition (dev, bp, VDT_NUM_ROWS + 1, 1);
      sp = VDT_DisplayAttr (dev, sp, VDT_ATTR_DEFAULT);

      for (col = 0; col < VDT_NUM_COLS; col++)
      {
	 *sp++ = vdtinfo->StatusLine[0][col];
      }
#ifdef DEBUGH19SCREEN
      fprintf (stderr, "\nBuffer: row = %d\n", VDT_NUM_ROWS);
      HEXDUMP (stderr, (char *)vdtinfo->StatusLine[0], VDT_NUM_COLS, 0);
#endif
      sp = VDT_DisplayScreen (dev, fd, vdtinfo->screen,
			      sp - vdtinfo->screen, TRUE);
      bp = sp;

      /*
      ** Position cursor on screen
      */

      if (sp)
      {
	 sp = VDT_DisplayPosition (dev, bp, vdtinfo->Row + 1, vdtinfo->Col + 1);
	 sp = VDT_DisplayScreen (dev, fd, bp, sp - bp, TRUE);
      }
      else
      {
	 status = -1;
      }
   }

   return (status);
}
