/***********************************************************************
*
* simio.c - IO processing for the TI 990 Simulator.
*
* Changes:
*   05/29/03   DGP   Original.
*   06/20/03   DGP   Added interrupt support.
*   07/01/03   DGP   Fixed floppy unit on STCR of status.
*   07/08/03   DGP   Changed FLOPPY to return status if unit not present.
*   07/09/03   DGP   Forced PC and WP to be on even boundry for interrupts.
*   07/15/03   DGP   Added Card Reader support.
*   07/23/03   DGP   Added ASCII/EBCDIC support for IBM OS/390.
*   08/07/03   DGP   Fixed PANEL support such that selftest and load works.
*   08/12/03   DGP   Added gen/clr/chk interrupt routine and /10 priv checks.
*   11/06/03   DGP   Put Floppy stuff in simfpy.c
*   11/07/03   DGP   Added VDT911 support.
*   11/12/03   DGP   Added TILINE support.
*   12/02/03   DGP   Improved card reader support.
*   05/10/04   DGP   Added DEV_NULL support.
*   07/29/04   DGP   Added devreset function.
*   12/03/04   DGP   Added CRU address adjustment.
*   04/01/05   DGP   Changed IDLE wait time.
*   04/12/05   DGP   Changed CONOUT to use prtview in panel mode.
*   04/13/05   DGP   Added device switches.
*   04/27/05   DGP   Modulo 4 device names for TILINE devices.
*   09/21/05   DGP   Added SWLOGOUT switch.
*   12/04/05   DGP   Added upcase switch on CONIN.
*   08/17/06   DGP   Added mutex to intrequest set/clear.
*   09/28/07   DGP   Dynamically adjust sleeptime for the interval timer.
*   10/01/07   DGP   Use Real Time clock.
*   10/23/07   DGP   Added WIN32 mutex.
*   10/25/07   DGP   Try to improve WIN32 sleep ganularity.
*   11/13/08   DGP   Improve CRU bit addressing.
*   02/18/09   DGP   Added breakpoint CRU support.
*   03/01/09   DGP   Added Diagnostic interrupt and test clock.
*   10/26/11   DGP   Improve card reader logic.
*   02/04/12   DGP   Added fakeprefetch for /12.
*   03/07/12   DGP   Fixup /12 ROM self test issues.
*   07/13/12   DGP   OS X DARWIN changes.
*   09/17/13   DGP   Added 9902 support.
*                    Revamp console support.
*   09/18/13   DGP   Split Card Reader support to new module.
*   09/19/13   DGP   Added 9903 support.
*   10/29/13   DGP   Correct status on unconnected tiline devices.
*   11/08/13   DGP   Added H19 support.
*   12/11/13   DGP   Added Card Punch support.
*   02/24/14   DGP   Added device interrupt delay. Fixed .QBID bugs. 
*                    Added CRU expansion chassis support. Thanks to
*                    to Howard Wulf.
*   02/26/14   DGP   Added interrupt queuing.
*   04/04/14   DGP   Added general telnet support for serial devices.
*   05/14/14   DGP   Fixed disk offline processing for DNOS.
*   07/30/14   DGP   Added termdstop to flush terminals.
*   07/31/14   DGP   Added detachit to disconnect terminals.
*   03/12/15   DGP   Added real device support for tape.
*   04/28/15   DGP   Added boot from card reader.
*   04/30/15   DGP   Improve error detection and messages.
*   05/01/15   DGP   Added ASR 733 emulation support.
*   06/02/15   DGP   Removed USS (z/OS) support.
*   01/28/16   DGP   Added real disk support.
*   05/01/16   DGP   Modified open mode with SWPROTECT.
*   05/18/16   DGP   Fixed CI403 unit setting. Fixed PRINTCI403 usage.
*   02/25/17   DGP   Added RAW terminal switch.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>
#include <time.h>

#if defined (WIN32)
#define __TTYROUTINES 0
#include <conio.h>
#include <windows.h>
#include <signal.h>
#include <process.h>
#endif

#if defined(UNIX)
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <pthread.h>
#endif

#include "simdef.h"
#define EXTERN
#include "simdsk.h"

extern uint16 pcreg;	/* The program PC */
extern uint16 curpc;	/* The program PC */
extern uint16 bkptpc;	/* The breakpoint PC */
extern uint32 mpcreg;	/* Mapped PC */
extern uint16 statreg;	/* The program status register */
extern uint16 wpreg;	/* The program Workspace Pointer */
extern uint16 lights;	/* The panel lights */
extern uint32 curmpc;	/* Mapped PC */

extern int run;
extern int idle;
extern int runled;
extern int faultled;
extern int devcnt;
extern int maxdevcnt;
extern int model;
extern int mapenabled;
extern int intsenabled;
extern int bkptenabled;
extern int tcenable;
extern int tccount;
extern int fakeprefetch;
extern int diaginterrupt;
extern int forceparity;
extern int rom5cru;
extern unsigned long instcount;
extern char view[MAXVIEW][MAXVIEWLEN+1];

extern uint16 bkptcrudata;
extern uint16 errcrudata;
extern uint16 mapcrudata;
extern uint16 mapolddata;
extern uint16 curmapindex;
extern uint16 intrequest;
extern uint32 maplatch;
extern uint16 xcard1crudata;
extern uint16 xcard2crudata;
extern uint8 memory[SYSMEMSIZE];

extern MapFile mapfile[MAPSIZE];
extern Device devices[MAXDEVICES];
extern InterruptQueue intqueue[16];

extern uint32 tracemem[16];
extern uint32 errtrace[16];
extern int errindex;
extern int trcindex;
extern int trclatch;

extern char *models[];

static MUTEX_DEFINE(genint_mutex);

static char *devusage[] =
{
   "Devices are:",
   " ce  = EIA console    c2  = 9902 console   c3  = 9903 console",
   " ",
   " pe  = EIA printer    p2  = 9902 printer   p3  = 9903 printer",
   " p4n = CI403 printer  pp  = Parallel printer",
   " ",
   " a1n = CI401 async    a2n = CI402 async    a3n = CI403 async",
   " ",
   " sen = EIA serial     s2n = 9902 serial    s3n = 9903 serial",
   " ",
   " cr  = Card reader    cp  = Card punch     csn = ASR 733 Cassette",
   " ",
   " fn  = Floppy Disk    vn  = 911 VDT",
   " dn  = TILINE DISK    mn  = TILINE TAPE",
   " ",
   ""
};

static int devmap[] =
{
   0, 1, 1, 1, 4, 3, 2, 2, 2, 2, 2, 2, 5, 1, 1, 1, 1, 1, 1, 6, 7, 1
};

static char *devtype[] =
{
   "",
   "TERMINAL",
   "PRINTER",
   "CARD PUNCH",
   "CARD READER",
   "FLOPPY",
   "DISK",
   "TAPE"
};

/***********************************************************************
* initinterrupt - Initalize interrupt.
***********************************************************************/

int
initinterrupt (void)
{
   if (MUTEX_INIT (genint_mutex))
   {
      PERROR ("initinterrupt: Can't create interrupt mutex");
      return (-1);
   }
   return (0);
}

/***********************************************************************
* geninterrupt - Generate interrupt.
***********************************************************************/

int
geninterrupt (uint16 lvl)
{
#ifdef DEBUGERRINT
   if (lvl == ERRINT)
      fprintf (stderr,
	       "%04X: ERRINT requested: level = >%X, errcrudata = >%04X\n",
	       curpc, GET_MASK, errcrudata);
#endif

   /*
   ** Error trace on /12 ERRINT
   */

   if (model == 12 && lvl == ERRINT && GET_MASK > 1)
   {
      int i, j;
      
      tracememory (0, mpcreg-2);
      trclatch = TRUE;
      trcindex = 0;
      j = errindex;
#ifdef DEBUGMAPCRU
      fprintf (stderr, "%04X: ERRINT: capture trace: errindex = %d\n",
	       curpc, errindex);
#endif
      for (i = 0; i < 16; i++)
      {
	 j --;
	 if (j < 0) j = 15;
	 tracemem[i] = errtrace[j];
#if defined(DEBUGMAPCRU) || defined(DEBUGERRINT)
	 fprintf (stderr, "   trc[%2d] = err[%2d] = %08X\n", i, j, tracemem[i]);
#endif
      }
   }

   MUTEX_LOCK (genint_mutex);

   intqueue[lvl].count++;
   intrequest = intrequest | (1 << lvl);

   MUTEX_UNLOCK (genint_mutex);

   return (0);
}

/***********************************************************************
* gendevinterrupt - Generate device interrupt.
***********************************************************************/

int
gendevinterrupt (Device *dev)
{
   InterruptQueue *queue;
   uint16 lvl;

   lvl = dev->intlvl;
   queue = &intqueue[lvl];

   MUTEX_LOCK (genint_mutex);

   /*
   ** Check if CRU expansion chassis interrupt
   */

   if (dev->xcardsel)
   {
#ifdef DEBUGEXPCHASSIS
      fprintf (stderr, "gendevinterrupt: xcarddat = >%04X, lvl = %d\n",
	       dev->xcarddat, lvl);
#endif
      if (queue->count)
      {
         queue->xcarddat[queue->top++] = dev->xcarddat;
	 if (queue->top == INTQDEPTH)
	    queue->top = 0;
      }
      else
      {
	 if (dev->xcardsel == 1)
	    xcard1crudata = dev->xcarddat;
	 else
	    xcard2crudata = dev->xcarddat;
      }
   }

   queue->count++;
   intrequest = intrequest | (1 << lvl);

   MUTEX_UNLOCK (genint_mutex);

   return (0);
}

/***********************************************************************
* clrinterrupt - Clear interrupt
***********************************************************************/

void
clrinterrupt (uint16 lvl)
{
   InterruptQueue *queue;

   queue = &intqueue[lvl];

   MUTEX_LOCK (genint_mutex);

   queue->count--;
   if (queue->count < 0)
      queue->count = 0;

   if (queue->count && (queue->bottom != queue->top))
   {
      uint16 lclcard = queue->xcarddat[queue->bottom++];

      if (queue->xcardsel == 1)
         xcard1crudata = lclcard;
      else
         xcard2crudata = lclcard;

      if (queue->bottom == INTQDEPTH)
         queue->bottom = 0;
   }

   if (queue->count == 0)
   {
      intrequest = intrequest & ~(1 << lvl);
   }

   MUTEX_UNLOCK (genint_mutex);
}

/***********************************************************************
* chkinterrupt - Check interrupt
***********************************************************************/

int
chkinterrupt (uint16 lvl)
{
   uint16 i;

   if (lvl > 0 && !intsenabled) return (0);

   MUTEX_LOCK (genint_mutex);

   i = intrequest & (1 << lvl);

   MUTEX_UNLOCK (genint_mutex);

   return (i);
}

/***********************************************************************
* checkinterrupts - If any pending and not masked, take it.
***********************************************************************/

void 
checkinterrupts (void)
{
   Device *dev;
   int curlvl;
   static int scancount = 0;
   int i;
   int keyhit;
   uint16 curinst;
#ifdef DEBUGINTZERO
   uint16 oldstat;
#endif

#ifdef DEBUGCKINST1
   fprintf (stderr, "checkinterrupts called: idle = %s, lvl = %d\n",
	    idle ? "TRUE" : "FALSE", GET_MASK);
#endif

   /*
   ** If mask == 0, return.
   */

   if (!GET_MASK) return;

   do
   {
      if (diaginterrupt)
      {
	 uint16 mask = bkptcrudata;
	 int lvlcnt = 0;

	 for (i = 0; i < 16; i++)
	 {
	    if ((mask & 0x0001) == 0) lvlcnt++;
	    else break;
	    mask >>= 1;
	 }
	 if (GET_MASK >= lvlcnt)
	 {
#ifdef DEBUGBKPT
	    fprintf (stderr, "%04X: Diagnostic interrupt: lvl = %d\n",
		     curpc, lvlcnt);
#endif
	    geninterrupt (lvlcnt);
	 }
      }

      /*
      ** Count down device delays for interrupts
      */

      for (i = 0; i < devcnt; i++)
      {
         if (devices[i].intdelay > 0)
	 {
	    if (--devices[i].intdelay == 0)
	       gendevinterrupt (&devices[i]);
	 }
      }

      scancount++;
      curinst = GETINST (FALSE);

      /*
      ** Check the keyboards for interrupts.
      */

      keyhit = FALSE;
      if (idle || scancount > 10000)
      {
	 int c;

         /*
         ** Check for console input, console always device 0.
         */

	 dev = &devices[0];
	 c = EOF;
	 if (dev->infd != DEV_NULL)
	 {
	    if (!termdringempty (dev, &dev->info.terminfo.inring))
	    {
	       dev->info.terminfo.kbint = FALSE;
	       dev->info.terminfo.kbchar = FALSE;
	       c = termdgetring (dev, &dev->info.terminfo.inring);
	    }
	    if (c != EOF)
	    {
	       dev->select = TRUE;
	       keyhit = TRUE;
	       dev->inchar = c;
	       if (dev->intenabled)
		  gendevinterrupt (dev);
	    }
	    else if (run == FALSE)
	       return;
	 }

         /*
         ** Check for VDT or SERIAL input
         */

	 for (i = 1; i < devcnt; i++)
	 {
	    dev = &devices[i];
	    switch (dev->type)
	    {
	    case SERIALEIA:
	    case SERIAL9902:
	    case SERIAL9903:
	    case SERIALCI402:
	       c = EOF;
	       if (dev->infd != DEV_NULL)
	       {
		  if (!termdringempty (dev, &dev->info.terminfo.inring))
		  {
		     dev->info.terminfo.kbint = FALSE;
		     dev->info.terminfo.kbchar = FALSE;
		     c = termdgetring (dev, &dev->info.terminfo.inring);
		  }
		  if (c != EOF)
		  {
		     dev->select = TRUE;
		     keyhit = TRUE;
		     dev->inchar = c;
		     if (dev->intenabled)
			gendevinterrupt (dev);
		  }
	       }
	       break;

	    case VDT911:
	       if (dev->info.vdt911info.kbint)
	       {
		  dev->info.vdt911info.kbint = FALSE;
		  keyhit = TRUE;
#ifdef DEBUG911TASK
		  fprintf (stderr, "Interrupt 911 keyboard\n");
#endif
		  if (dev->intenabled)
		     gendevinterrupt (dev);
	       }
	       break;

	    default: ;
	    }
	 }
	 scancount = 0;
      }

      /*
      ** If we have interrupts, process them.
      */

      if (intrequest)
      {
	 curlvl = GET_MASK;
	 for (i = 0; i <= curlvl; i++)
	 {
	    if (chkinterrupt (i))
	    {
	       uint16 opc, npc;
	       uint16 owp, nwp;
	       uint16 ost;

	       if (curinst == IDLEINST) pcreg += 2;
	       ost = statreg;
	       opc = pcreg;
	       owp = wpreg;
	       clrinterrupt (i);
#ifdef DEBUGINTZERO
	       oldstat = statreg;
#endif
	       SET_MASK (i-1);
#ifdef DEBUGINTZERO
	       if (((oldstat & 15) == 1) && ((statreg & 15) == 0))
		  fprintf (stderr,
			   "%04X: Interrupt: oldstat = %04X, statreg = %04X\n",
			   pcreg-2, oldstat, statreg);
#endif
	       CLR_NONPRV;
	       CLR_MAP;
	       CLR_XOP;
	       CLR_OVERI;
	       CLR_WCS;
	       nwp = GETMEM0 (i * 4) & 0xFFFE;
	       npc = GETMEM0 (i * 4 + 2) & 0xFFFE;
	       pcreg = npc;
	       wpreg = nwp;
#ifdef DEBUGERRINT
	       if (GET_MASK == 0) {
		  fprintf (stderr, 
			  "%04X: ERRINT interrupt: ZERO mask = >%X\n",
			  curpc, GET_MASK);
	       }
	       if (i == ERRINT)
	       {
		  fprintf (stderr, 
			  "%04X: ERRINT interrupt taken: mask = >%X\n",
			  curpc, GET_MASK);
	       }
	       if (pcreg == 0)
	       {
		  fprintf (stderr,
		       "%04X: INVALID interrupt trap: wpreg = %d, pcreg = %d\n",
			curpc, wpreg, pcreg);
		  fprintf (stderr, "   curlvl = %X, i = %d\n", curlvl, i);
		  fprintf (stderr, "   owp = %04X, opc = %04X, ost = %04X\n",
			   owp, opc, ost);
		  run = FALSE;
		  return;
	       }
#endif
	       PUTREG (13, owp);
	       PUTREG (14, opc);
	       PUTREG (15, ost);
	       idle = FALSE;
	       return;
	    }
	 }
      }

      /*
      ** If nothing to do snooze a bit
      */

      if (idle && !scancount && !keyhit)
      {
#ifdef LINUX
         sched_yield ();
#else
	 smallsleep (1);
#endif
      }
   } while (idle);
}

/***********************************************************************
* realfd - Check if real fd.
***********************************************************************/

int
realfd (FILE *fd)
{
   if (fd && ((long)fd > (long)DEV_TELNET))
      return TRUE;
   return FALSE;
}

/***********************************************************************
* finddev - Find attached device.
***********************************************************************/

static int
finddev (uint16 devaddr)
{
   int i;

   for (i = 0; i < devcnt; i++)
   {
      if (devices[i].devaddr == devaddr && devices[i].infd != NULL) break;
      if (devices[i].extaddr == devaddr && devices[i].infd != NULL) break;
      if (devices[i].ctladdr == devaddr && devices[i].infd != NULL) break;
   }

   if (i == devcnt)
   {
      /*
      ** On a 990/5 and 990/10a CPUs the comm ports are always present.
      ** Dummy if not configured. Needed by ROM self test.
      */
      if (model == 5 || model == 11)
      {
	 if (devaddr == 0x1700 || devaddr == 0x1720)
	    return (maxdevcnt);
	 if (model == 5 && (devaddr == 0x1740 || devaddr == 0x1760))
	    return (maxdevcnt + 1);
	 if (model == 5 && (devaddr == 0x1780 || devaddr == 0x17A0))
	    return (maxdevcnt + 2);
      }
#ifdef DEBUGFINDDEV
      fprintf (stderr, "Device for devaddr %04X is not attached\n", devaddr);
#endif
      return (-1);
   }
   return (i);
}

/***********************************************************************
* findtype - Find attached device by type.
***********************************************************************/

static int
findtype (int type, char *dt)
{
   int i;

   for (i = 0; i < devcnt; i++)
   {
      if (devices[i].type == type && devices[i].infd != NULL &&
	  !strcmp (devices[i].name, dt) ) return (i);
   }

   return (-1);
}

/***********************************************************************
* cassopen - Open an ASR 733 cassette
***********************************************************************/

static FILE *
cassopen (char *filename, char *fdev, char *mode, int unit)
{
   FILE *fd;

   if (!strcmp (filename, "NULL") || !strcmp (filename, "null"))
   {
      return (DEV_NULL);
   }

   if ((fd = fopen (filename, mode)) == NULL)
   {
      sprintf (view[0], "Device %s: open failed: %s", fdev, strerror (ERRNO));
      snprintf (view[1], MAXVIEWLEN, "filename: %s", filename);
      return (NULL);
   }
   fseek (fd, 0, SEEK_SET);
   return (fd);
}

/***********************************************************************
* parseterminal - Parse terminal option.
***********************************************************************/

static char *
parseterminal (char *fdev, char *bp, DeviceOptions *options, int *error)
{
   char *tp;
   int gotargs;
   int i;
   char termname[32];
   char termargs[MAXDEVPATHLEN+2];

   gotargs = FALSE;
   if (*bp == '=')
   {
      bp++;
      tp = termname; 
      for (i = 0; *bp && !isspace (*bp) && i < sizeof (termname); i++)
      {
	 if (*bp == ',') break;
	 if (*bp == ':')
	 {
	    gotargs = TRUE;
	    break;
	 }
	 if (islower (*bp))
	    *tp++ = toupper (*bp++);
	 else
	    *tp++ = *bp++;
      }
      *tp = 0;

      if (gotargs)
      {
	 bp++;
	 tp = termargs;
	 for (i = 0; *bp && !isspace (*bp) && i < MAXDEVPATHLEN; i++)
	 {
	    if (*bp == ',') break;
	    *tp++ = *bp++;
	 }
	 *tp = 0;
      }

      if (!strcmp (termname, "H19"))
      {
	 options->switches |= SWEMUTERM;
         options->emutype = EMUH19;
      }
      else if (!strcmp (termname, "V931"))
      {
	 options->switches |= SWEMUTERM;
         options->emutype = EMU931;
      }
      else if (!strcmp (termname, "V940"))
      {
	 options->switches |= SWEMUTERM;
         options->emutype = EMU940;
      }
      else if (!strcmp (termname, "A733"))
      {
	 options->switches |= SWEMUTERM;
         options->emutype = EMU733;
	 options->cs1fd = DEV_NULL;
	 options->cs2fd = DEV_NULL;
	 strcpy (options->file1, "NULL");
	 strcpy (options->file2, "NULL");
      }
      else
      {
	 sprintf (view[0], "Device %s terminal %s: Unsupported emulation",
		  fdev, termname);
         *error = TRUE;
      }

      if (!*error && gotargs)
      {
	 switch (options->emutype)
	 {
	 case EMU940:
	 case EMU931:
	    options->switches |= SWPRINTER;
	    strcpy (options->file1, termargs);
	    if ((options->printfd = fopen (options->file1, "w")) == NULL)
	    {
	       options->switches &= ~SWPRINTER;
	       sprintf (view[0],
			"Device %s terminal %s: printer open failed: %s",
			fdev, termname, strerror (ERRNO));
	       snprintf (view[1], MAXVIEWLEN, "filename: %s", options->file1);
	       *error = TRUE;
	    }
	    break;

	 default:
	    sprintf (view[0],
		     "Device %s terminal %s: Doesn't support attached printer",
		     fdev, termname);
	    *error = TRUE;
	    break;

	 }
      }
   }

   if (*bp == ' ') bp--;
   return (bp);
}

/***********************************************************************
* parsediskmodel - Parse disk model option.
***********************************************************************/

static char *
parsediskmodel (char *fdev, char *bp, DeviceOptions *options, int *error)
{
   char *mp;
   int i;
   char modelname[32];

   if (*bp == '=')
   {
      bp++;
      mp = modelname; 
      for (i = 0; *bp && !isspace (*bp) && i < sizeof (modelname); i++)
      {
	 if (*bp == ',') break;
	 if (islower (*bp))
	    *mp++ = toupper (*bp++);
	 else
	    *mp++ = *bp++;
      }
      *mp = 0;

      for (i = 0; i < MAXDISKS; i++)
      {
	 if (!strcmp (modelname, disks[i].model)) break;
      }
      if (i == MAXDISKS)
      {
	 sprintf (view[0], "Device %s model %s: Unsupported model",
		  fdev, modelname);
         *error = TRUE;
      }
      else
      {
	 strcpy (options->file1, modelname);
      }
   }
   if (*bp == ' ') bp--;
   return (bp);
}

/***********************************************************************
* parseoptions - Parse option switches for a device.
***********************************************************************/

static char *
parseoptions (char *fdev, char *bp, int type, DeviceOptions *options,
	      int *error)
{
   bp++;
   switch (type)
   {
   case VDT911:
      for ( ;*bp && !isspace (*bp); bp++)
      {
	 if (*bp == ',') continue;
	 if (*bp == 'u') options->switches |= SWUPCASE;
	 else if (*bp == 'l') options->switches |= SWLOGOUT;
	 else goto PARSE_USAGE;
      }
      break;

   case CONEIA:
   case CON9902:
   case CON9903:
   case SERIALEIA:
   case SERIAL9902:
   case SERIAL9903:
   case SERIALCI401:
   case SERIALCI402:
   case SERIALCI403:
      for ( ;*bp && !isspace (*bp); bp++)
      {
	 if (*bp == ',') continue;
	 if (*bp == 'u') options->switches |= SWUPCASE;
	 else if (*bp == 'c') options->switches |= SWCREATE;
	 else if (*bp == 'm') options->switches |= SWSWITCH;
	 else if (*bp == 'l') options->switches |= SWLOGOUT;
	 else if (*bp == 'r') options->switches |= SWRAWTERM;
	 else if (*bp == 's') options->switches |= SWSTATUS;
	 else if (*bp == 't') options->switches |= SWTELNET;
	 else if (*bp == 'n') options->switches |= SWNETDEV;
	 else if (*bp == 'e') bp = parseterminal (fdev, ++bp, options, error);
	 else goto PARSE_USAGE;
      }
      break;

   case DISK:
      for ( ;*bp && !isspace (*bp); bp++)
      {
	 if (*bp == ',') continue;
	 if (*bp == 'r') options->switches |= SWPROTECT;
	 else if (*bp == 'm') bp = parsediskmodel (fdev, ++bp, options, error);
	 else goto PARSE_USAGE;
      }
      break;

   case TAPE:
      for ( ;*bp && !isspace (*bp); bp++)
      {
	 if (*bp == ',') continue;
	 if (*bp == 'c') options->switches |= SWCREATE;
	 else if (*bp == 'r') options->switches |= SWPROTECT;
	 else goto PARSE_USAGE;
      }
      break;

   case CARDPUNCH:
      for ( ;*bp && !isspace (*bp); bp++)
      {
	 if (*bp == ',') continue;
	 if (*bp == 't') options->switches |= SWPUTRIM;
	 else goto PARSE_USAGE;
      }
      break;

   case CASSETTE:
      for ( ;*bp && !isspace (*bp); bp++)
      {
	 if (*bp == ',') continue;
	 if (*bp == 'c') options->switches |= SWCREATE;
	 else goto PARSE_USAGE;
      }
      break;

   default:
   PARSE_USAGE:
      sprintf (view[0], "Device %s: Illegal option: %c", fdev, *bp);
      *error = TRUE;
      while (*bp && !isspace (*bp)) bp++;
   }
   return (bp);
}

/***********************************************************************
* checkaddress - Check the base unit device entry address.
***********************************************************************/

static int
checkaddress (char *fdev, int devaddr, int type, int devnum)
{
   int i;
   char devname[32];

   i = devnum - (devnum % 4);
   if (type == PRINTCI403)
      sprintf (devname, "P4:%d", i);
   else if (type == SERIALCI403)
      sprintf (devname, "A3:%d", i);
   else if (type == DISK || type == TAPE)
      sprintf (devname, "%c%d", fdev[0], i);
   else 
      return (0);

   for (i = 0; i < devcnt; i++)
   {
      if (!strcmp (devices[i].name, devname))
      {
	 Device *mdev = &devices[i];

	 if (mdev->devaddr != devaddr)
	 {
	    sprintf (view[0],
		     "Device %s address >%04X: Address not on controller",
		     fdev, devaddr);
	    sprintf (view[1], "Device %s address >%04X",
		     mdev->name, mdev->devaddr);
	    return (-1);
	 }
	 break;
      }
   }
   return (0);
}

/***********************************************************************
* checkinterrupt - Check the base unit device entry interrupt.
***********************************************************************/

static int
checkinterrupt (char *fdev, int interrupt, int type, int devnum)
{
   int i;
   char devname[32];

   i = devnum - (devnum % 4);
   if (type == PRINTCI403)
      sprintf (devname, "P4:%d", i);
   else if (type == SERIALCI403)
      sprintf (devname, "A3:%d", i);
   else if (type == DISK || type == TAPE)
      sprintf (devname, "%c%d", fdev[0], i);
   else 
      return (0);

   for (i = 0; i < devcnt; i++)
   {
      if (!strcmp (devices[i].name, devname))
      {
	 Device *mdev = &devices[i];

	 if (mdev->intlvl != interrupt)
	 {
	    sprintf (view[0],
		     "Device %s interrupt %d: Interrupt mismatch",
		     fdev, interrupt);
	    sprintf (view[1], "Device %s interrupt %d",
		     mdev->name, mdev->intlvl);
	    return (-1);
	 }
	 break;
      }
   }
   return (0);
}

/***********************************************************************
* attachit - Attach a file to a device (do the dirty work).
***********************************************************************/

static void
attachit (char *bp, int type, int devnum, char *mode, char *fdev)
{
   Device *dev;
   FILE *infd;
   FILE *onfd;
   char *ep;
   long val;
   int devndx;
   int sfd;
   int devaddr;
   int interrupt;
   int expcheck;
   int expdevice;
   int newdevice;
   int realdevice;
   uint16 chassisnum;
   uint16 chassispos;
   char lclmode[32];
   DeviceOptions options;
   Device olddev;

   /*
   ** Initialize locals.
   */

   strcpy (lclmode, mode);
   memset (&options, 0, sizeof (options));
   chassisnum = 0;
   chassispos = 0;

   /*
   ** Check for device options
   */

   if (*bp == '/')
   {
      int err = FALSE;
      bp = parseoptions (fdev, bp, type, &options, &err);
      if (err) return;
   }

   /*
   ** Get device address
   */

   bp = getnum (bp, &val);
   devaddr = val & 0xFFFF;

   /*
   ** Sanity check device address
   */

   expcheck = TRUE;
   expdevice = FALSE;
   switch (type)
   {
   case TAPE:
   case DISK:
   case SERIALCI403:
   case PRINTCI403:
      if (model == 4 || model == 9)
      {
	 sprintf (view[0], "Device %s: TILINE devices not supported on 990/%s",
		  fdev, models[model-4]);
	 return;
      }
      expcheck = FALSE;
      if (devaddr < TILINESTART || devaddr > TILINEEND)
      {
	 sprintf (view[0], "Device %s: Illegal TILINE device address >%04X",
		  fdev, devaddr);
	 return;
      }
      break;

   case SERIALCI402:
      if (devaddr < 0x800 || devaddr > 0x1B00)
         goto BADCRUADDR;
      expcheck = FALSE;
      break;

   case CON9902:
   case PRINT9902:
   case SERIAL9902:
   case CON9903:
   case PRINT9903:
   case SERIAL9903:
      expcheck = FALSE;
      /* fall through */

   default:
      if (devaddr > CRUEND)
      {
      BADCRUADDR:
	 sprintf (view[0], "Device %s: Illegal CRU device address >%04X",
		  fdev, devaddr);
	 return;
      }
   }
   if (checkaddress (fdev, devaddr, type, devnum) < 0) return;

   /*
   ** Sanity check device interrupt
   */

   bp = getnum (bp, &val);
   interrupt = val & 0xFFFF;
   if (interrupt > 0)
   {
      int badint = FALSE;

      if (model == 4 && (interrupt < 2 || interrupt > 7))
      {
	 badint = TRUE;
      }
      else if (interrupt < 2 || interrupt > 15)
      {
	 badint = TRUE;
      }
      if (badint)
      {
	 sprintf (view[0], "Device %s: Illegal interrupt %d for 990/%s",
		  fdev, interrupt, models[model-4]);
	 return;
      }
      if (checkinterrupt (fdev, interrupt, type, devnum) < 0) return;
   }

   /*
   ** Sanity check if expansion device
   */

   if (expcheck && devaddr > 0x03FF && devaddr < 0x1F00)
   {
      chassisnum = (devaddr & 0x1C00) / 0x0400;  /* 1..8  */
      chassispos = (devaddr & 0x03FF) / 0x0020;  /* 0..31 */
      
#ifdef DEBUGEXPCHASSIS
      fprintf (stderr, 
	 "Exp: Device %s: devaddr = >%04X, chassisnum = %d, chassispos = %d\n",
	    fdev, devaddr, chassisnum, chassispos);
#endif

      if (chassisnum > 8 || chassispos > 31)
      {
	 sprintf (view[0], "Device %s: Illegal CRU Exp devaddr = >%04X, "
		  "chassisnum = %d, chassispos = %d",
		  fdev, devaddr, chassisnum, chassispos);
	 return;
      }
      expdevice = TRUE;
   }

   /* Skip to file name and truncate if trailing spaces */
   while (*bp && isspace (*bp)) bp++;
   for (ep = bp; *ep; ep++)
   {
      if (isspace (*ep))
      {
         *ep = '\0';
	 break;
      }
   }

   /*
   ** Process Cassette device here.
   */

   if (type == CASSETTE)
   {
      if ((devndx = finddev (devaddr)) < 0)
      {
         sprintf (view[0], "Device %s: no device found on address >%04X",
		  fdev, devaddr);
	 return;
      }
      dev = &devices[devndx];
      if (dev->intlvl != interrupt)
      {
         sprintf (view[0], "Device %s: interrupt mismatch for device %s",
		  fdev, dev->name);
	 return;
      }

      switch (dev->type)
      {
      case CONEIA:
      case CON9902:
      case CON9903:
      case SERIALEIA:
      case SERIAL9902:
      case SERIAL9903:
         if ((dev->switches & SWEMUTERM) && (dev->emutype == EMU733))
	 {
	    ASRInfo *asrinfo = &dev->info.terminfo.terminal.asrinfo;
	    int unit = atoi (&fdev[2]);
	    int base = ((unit % 2) == 0) ? unit : unit - 1;

	    if (options.switches & SWCREATE)
	       lclmode[0] = 'w';
	    asrinfo->baseunit = base;
	    if ((unit % 2) == 0)
	    {
	       if (realfd (asrinfo->cs1fd))
	          fclose (asrinfo->cs1fd);
	       asrinfo->cs1fd = cassopen (bp, fdev, lclmode, unit);
	       if (asrinfo->cs1fd == NULL)
	          return;
	       strcpy (asrinfo->cs1file, bp);
	    }
	    else
	    {
	       if (realfd (asrinfo->cs2fd))
	          fclose (asrinfo->cs2fd);
	       asrinfo->cs2fd = cassopen (bp, fdev, lclmode, unit);
	       if (asrinfo->cs2fd == NULL)
	          return;
	       strcpy (asrinfo->cs2file, bp);
	    }
	 }
	 else
	 {
	    sprintf (view[0], "Device %s: is not configured for ASR 733",
		     dev->name);
	 }
         break;

      default:
         sprintf (view[0], "Device %s: does not support Cassette", dev->name);
      }
      return;
   }

   /*
   ** Do the open
   */

   realdevice = FALSE;
   switch (type)
   {
   case VDT911:
      onfd = infd = v911open (fdev, bp, lclmode, &sfd);
      break;

   case CONEIA:
   case CON9902:
   case CON9903:
      devices[0].type = type;		/* Console aways device 0 */
      strcpy (devices[0].name, fdev);
      /* Fall through */

   case SERIALEIA:
   case SERIAL9902:
   case SERIAL9903:
   case SERIALCI401:
   case SERIALCI402:
   case SERIALCI403:
      if (options.switches & SWTELNET)
      {
         onfd = infd = termdopen (fdev, bp, lclmode, &sfd);
	 break;
      }
      /* Fall through */

   default:
      if (!strcmp (bp, "NULL") || !strcmp (bp, "null"))
      {
	 strcpy (bp, "NULL");
	 onfd = infd = DEV_NULL;
      }
      else if (!strcmp (bp, "LOOP") || !strcmp (bp, "loop"))
      {
	 strcpy (bp, "LOOP");
	 onfd = infd = DEV_LOOP;
      }
      else if (!strcmp (bp, "TTY") || !strcmp (bp, "tty"))
      {
	 strcpy (bp, "TTY");
	 infd = stdin;
	 onfd = stdout;
      }
      else
      {
#ifdef UNIX
	 if (!strncmp (bp, "/dev/", 5))
	    realdevice = TRUE;
#endif
	 if (options.switches & SWCREATE)
	    lclmode[0] = 'w';
	 else if (options.switches & SWPROTECT)
	    lclmode[1] = 0;
	 if ((onfd = infd = fopen (bp, lclmode)) == NULL)
	 {
	    sprintf (view[0], "Device %s: open failed: %s",
		     fdev, strerror (ERRNO));
	    snprintf (view[1], MAXVIEWLEN, "filename: %s", bp);
	 }
      }
   }

   if (infd == NULL) return;

   /*
   ** Locate a device table slot
   */

   newdevice = FALSE;
   if ((devndx = findtype (type, fdev)) >= 0)
   {
      memcpy (&olddev, &devices[devndx], sizeof (Device));
      if (!(devices[devndx].switches & SWTELNET))
      {
	 if (devices[devndx].infd != stdin && devices[devndx].infd != DEV_NULL)
	    fclose (devices[devndx].infd);
	 if (devices[devndx].infd != devices[devndx].outfd)
	 {
	    if (devices[devndx].outfd != stdout &&
	        devices[devndx].outfd != DEV_NULL)
	       fclose (devices[devndx].outfd);
	 }
      }
   }
   else
   {
      newdevice = TRUE;
      if (devcnt == maxdevcnt)
      {
	 sprintf (view[0], "Too many devices: maxdevcnt = %d", maxdevcnt);
	 return;
      }
      else
      {
	 devndx = devcnt;
	 memset (&olddev, 0, sizeof (Device));
      }
   }

   /*
   ** Fill it in
   */

   dev = &devices[devndx];
   memset (dev, 0, sizeof (Device));

   dev->infd = infd;
   dev->outfd = onfd;
   dev->devaddr = devaddr;
   dev->type = type;
   dev->select = 0;
   dev->count = 0;
   dev->unit = 0;
   dev->emutype = 0;
   dev->viewndx = 0;
   dev->intenabled = FALSE;
   dev->runtermd = FALSE;
   dev->realdevice = realdevice;
   dev->intlvl = interrupt;
   dev->extaddr = 0xFFFF;
   dev->ctladdr = 0xFFFF;

   /*
   ** Configure Expansion Chassis devices
   ** Page 3-15 of "990 CRU/TILINE Expansion" (March 1984)
   ** Page 3-169 of "Model 990/10 Computer System Hardware Reference Manual"
   **   (15 November 1980)
   */
    
   if (expdevice)
   {
      if (chassisnum < 5)
      {
	 /* Expansion chassis #1 thru #4 are on CRU Expansion CARD1 at >1F00 */
	 /* intlvl is CARD1 */
	 /* chassisnum is 1..4 */
	 /* chassispos is 0..31 */
	 dev->xcardsel = 1;
      }
      else
      {
	 /* Expansion chassis #5 thru #8 are on CRU Expansion CARD2 at >1F20 */
	 /* intlvl is CARD2 */
	 /* chassisnum is 5..8 */
	 /* chassispos is 0..31 */
	 dev->xcardsel = 5;
      }
      dev->xcarddat = 0x8000 |
		     ((chassisnum - dev->xcardsel) << 7) | (chassispos << 2);
      intqueue[interrupt].xcardsel = dev->xcardsel;
#ifdef DEBUGEXPCHASSIS
      fprintf (stderr, "   xcardsel = %d, xcarddat = >%04X\n",
	       dev->xcardsel, dev->xcarddat);
#endif
   }

   dev->switches = options.switches;
   if (dev->switches & SWEMUTERM)
   {
      dev->emutype = options.emutype;
      if (dev->emutype == EMU733)
      {
	 ASRInfo *asrinfo = &dev->info.terminfo.terminal.asrinfo;

	 if (newdevice && (devndx > 0))
	 {
	    int base = 0;
	    for (devndx--; devndx >= 0; devndx--)
	    {
	       if ((devices[devndx].switches & SWEMUTERM) && 
		  (devices[devndx].emutype == EMU733))
	       {
		  base =
		     devices[devndx].info.terminfo.terminal.asrinfo.baseunit;
		  base += 2;
		  asrinfo->baseunit = base;
		  break;
	       }
	    }
	 }
         asrinfo->cs1fd = options.cs1fd;
         asrinfo->cs2fd = options.cs2fd;
         asrinfo->status = A733BIAS;
	 strcpy (asrinfo->cs1file, options.file1);
	 strcpy (asrinfo->cs2file, options.file2);
      }
      else if (dev->switches & SWPRINTER)
      {
	 VDTInfo *vdtinfo = &dev->info.terminfo.terminal.vdtinfo;
         vdtinfo->printfd = options.printfd;
	 strcpy (vdtinfo->printfile, options.file1);
      }
   }

   strcpy (dev->file, bp);
   strcpy (dev->name, fdev);

   switch (type)
   {
   case CONEIA:
   case SERIALEIA:
   case SERIALCI401:
      if (dev->switches & SWTELNET)
      {
	 dev->unit = atoi (bp); /* TCP port into unit */
	 dev->sfd = sfd;
      }
      if (termdstart (dev) < 0) return;
      break;

   case CON9902:
   case CON9903:
   case SERIAL9902:
   case SERIAL9903:
      if (dev->switches & SWTELNET)
      {
	 dev->unit = atoi (bp); /* TCP port into unit */
	 dev->sfd = sfd;
      }
      if (termdstart (dev) < 0) return;
      /* fall through */

   case PRINT9902:
   case PRINT9903:
      dev->extaddr = devaddr + 0x20;
      /* On a 990/10A the port runs like a CI402. */
      if (model == 11 && devaddr == TERM9902)
	 dev->ctladdr = devaddr + 0x40;
      else
         dev->info.terminfo.controller.serialinfo.mod5port = TRUE;
      break;

   case SERIALCI402:
      dev->extaddr = devaddr + 0x20;
      dev->ctladdr = devaddr + 0x40;
      if (dev->switches & SWTELNET)
      {
	 dev->unit = atoi (bp); /* TCP port into unit */
	 dev->sfd = sfd;
      }
      if (termdstart (dev) < 0) return;
      break;

   case SERIALCI403:
      if (dev->switches & SWTELNET)
      {
	 dev->unit = atoi (bp); /* TCP port */
	 dev->sfd = sfd;
      }
      if (termdstart (dev) < 0) return;
      /* fall through */

   case PRINTCI403:
      dev->info.terminfo.controller.ci403info.unit = devnum % 4;
      if (ci403start (dev) < 0) return;
      break;

   case DISK:
      dev->unit = (1 << (3 -(atoi (&fdev[1]) % 4)));
      strcpy (dev->info.dskinfo.model, options.file1);
      if (dskopen (dev) < 0) return;
#ifdef DEBUGDSK
      fprintf (stderr, "   unit = %02X\n", dev->unit);
#endif
      break;

   case TAPE:
      dev->unit = (1 << (3 -(atoi (&fdev[1]) % 4)));
      if (tapeopen (dev) < 0) return;
#ifdef DEBUGTAPE
      fprintf (stderr, "   unit = %02X\n", dev->unit);
#endif
      break;

   case FLOPPY:
      dev->extaddr = devaddr + 0x20;
      dev->unit = atoi (&fdev[1]) % 4;
#ifdef DEBUGFLOPPY
      fprintf (stderr, "Attach floppy, unit = %d, cru = %04x to %s\n",
	       dev->unit, dev->devaddr, bp);
#endif
      break;

   case VDT911:
      dev->switches |= SWTELNET;
      dev->extaddr = devaddr + 0x10;
      dev->unit = atoi (bp); /* TCP port into unit */
      dev->sfd = sfd;
      if (v911start (dev) < 0) return;
      break;

   case CARDPUNCH:
      dev->info.cardinfo.cardeof = FALSE;
      dev->info.cardinfo.cardndx = 0;
      break;

   case CARDREADER:
      if (!newdevice)
	 dev->info.cardinfo.cardinit = olddev.info.cardinfo.cardinit;
      dev->info.cardinfo.cardeof = FALSE;
      dev->info.cardinfo.cardndx = 100;
      break;

   default: ;
   }

   if (newdevice)
      devcnt++;
}

/***********************************************************************
* detachit - Detach a terminal device.
***********************************************************************/

static void
detachit (int type, char *fdev)
{
   Device *dev;
   int devndx;

   if ((devndx = findtype (type, fdev)) >= 0)
   {
      dev = &devices[devndx];
      if (dev->runtermd)
	 termdstop (dev);
      if (!(dev->switches & SWTELNET))
      {
	 if (dev->infd != stdin && dev->infd != DEV_NULL)
	    fclose (dev->infd);
	 if (dev->infd != dev->outfd)
	 {
	    if (dev->outfd != stdout && dev->outfd != DEV_NULL)
	       fclose (dev->outfd);
	 }
      }
      dev->infd = DEV_NULL;
      dev->outfd = DEV_NULL;
   }
}

/***********************************************************************
* attachdev - Attach a file to a device.
***********************************************************************/

void
attachdev (char *bp)
{
   char *cp;
   long val;
   int i;
   int  devnum;
   char fdev[10];

   while (*bp && isspace (*bp)) bp++;
   cp = bp;

   devnum = 0;
   switch (*bp++)
   {
      case 'a': /* Asynchronous Comm */
	 switch (*bp++)
	 {
	 case '1': /* CI401 */
	    if (!isdigit (*bp)) goto BADUNIT;
	    bp = getnum (bp, &val);
	    devnum = val & 0xFFFF;
	    sprintf (fdev, "A1:%d", devnum);
	    detachit (SERIALCI401, fdev);
	    attachit (bp, SERIALCI401, devnum, "r+", fdev);
	    break;

	 case '2': /* CI402 */
	    if (!isdigit (*bp)) goto BADUNIT;
	    bp = getnum (bp, &val);
	    devnum = val & 0xFFFF;
	    sprintf (fdev, "A2:%d", devnum);
	    detachit (SERIALCI402, fdev);
	    attachit (bp, SERIALCI402, devnum, "r+", fdev);
	    break;

	 case '3': /* CI403 */
	    if (!isdigit (*bp)) goto BADUNIT;
	    bp = getnum (bp, &val);
	    devnum = val & 0xFFFF;
	    sprintf (fdev, "A3:%d", devnum);
	    detachit (SERIALCI403, fdev);
	    attachit (bp, SERIALCI403, devnum, "r+", fdev);
	    break;

	 default: 
	    goto BADDEV;
	 }
	 break;

      case 'c':
	 switch (*bp++)
	 {
	 case 'e': /* EIA Console */
	    detachit (devices[0].type, devices[0].name);
	    attachit (bp, CONEIA, devnum, "r", "CE");
	    break;

	 case '2': /* 9902 Console */
	    if (!(model == 5 || model == 11))
	       goto BADDEVCPU;
	    detachit (devices[0].type, devices[0].name);
	    attachit (bp, CON9902, devnum, "r", "C2");
	    break;

	 case '3': /* 9903 Console */
	    if (model != 5)
	       goto BADDEVCPU;
	    detachit (devices[0].type, devices[0].name);
	    attachit (bp, CON9903, devnum, "r", "C3");
	    break;

	 case 'p': /* Card punch */
	    attachit (bp, CARDPUNCH, devnum, "w", "CP");
	    break;

	 case 'r': /* Card reader */
	    attachit (bp, CARDREADER, devnum, "r", "CR");
	    break;

	 case 's': /* ASR733 Cassette */
	    if (!isdigit (*bp)) goto BADUNIT;
	    bp = getnum (bp, &val);
	    devnum = val & 0xFFFF;
	    sprintf (fdev, "CS%d", devnum);
	    attachit (bp, CASSETTE, devnum, "r+b", fdev);
	    break;

	 default: 
	    goto BADDEV;
	 }
	 break;

      case 'd': /* Disk */
	 if (!isdigit (*bp)) goto BADUNIT;
	 bp = getnum (bp, &val);
	 devnum = val & 0xFFFF;
	 sprintf (fdev, "D%d", devnum);
	 attachit (bp, DISK, devnum, "r+b", fdev);
	 break;

      case 'f': /* Floppy disk */
	 if (!isdigit (*bp)) goto BADUNIT;
	 bp = getnum (bp, &val);
	 devnum = val & 0xFFFF;
	 sprintf (fdev, "F%d", devnum);
	 attachit (bp, FLOPPY, devnum, "r+b", fdev);
	 break;

      case 'm': /* MagTape */
	 if (!isdigit (*bp)) goto BADUNIT;
	 bp = getnum (bp, &val);
	 devnum = val & 0xFFFF;
	 sprintf (fdev, "M%d", devnum);
	 attachit (bp, TAPE, devnum, "r+b", fdev);
	 break;

      case 'p': /* Printer */
	 switch (*bp++)
	 {
	 case ' ':
	 case 'r':
	 case 'e': /* EIA Printer */
	    attachit (bp, PRINTEIA, devnum, "w", "PE");
	    break;

	 case '2': /* 9902 Printer */
	    if (!(model == 5 || model == 11))
	       goto BADDEVCPU;
	    attachit (bp, PRINT9902, devnum, "w", "P2");
	    break;

	 case '3': /* 9903 Printer */
	    if (model != 5)
	       goto BADDEVCPU;
	    attachit (bp, PRINT9903, devnum, "w", "P3");
	    break;

	 case '4': /* CI403 Printer */
	    if (!isdigit (*bp)) goto BADUNIT;
	    bp = getnum (bp, &val);
	    devnum = val & 0xFFFF;
	    sprintf (fdev, "P4:%d", devnum);
	    attachit (bp, PRINTCI403, devnum, "w", fdev);
	    break;

	 case 'p': /* Parallel Printer */
	    attachit (bp, PRINTPAR, devnum, "w", "PP");
	    break;

	 default: 
	    goto BADDEV;
	 }
	 break;

      case 's': /* Serial */
	 switch (*bp++)
	 {
	 case 'e': /* EIA */
	    if (!isdigit (*bp)) goto BADUNIT;
	    bp = getnum (bp, &val);
	    devnum = val & 0xFFFF;
	    sprintf (fdev, "SE:%d", devnum);
	    detachit (SERIALEIA, fdev);
	    attachit (bp, SERIALEIA, devnum, "r+", fdev);
	    break;

	 case '2': /* 9902 */
	    if (!(model == 5 || model == 11))
	       goto BADDEVCPU;
	    if (!isdigit (*bp)) goto BADUNIT;
	    bp = getnum (bp, &val);
	    devnum = val & 0xFFFF;
	    sprintf (fdev, "S2:%d", devnum);
	    detachit (SERIAL9902, fdev);
	    attachit (bp, SERIAL9902, devnum, "r+", fdev);
	    break;

	 case '3': /* 9903 */
	    if (model != 5)
	       goto BADDEVCPU;
	    if (!isdigit (*bp)) goto BADUNIT;
	    bp = getnum (bp, &val);
	    devnum = val & 0xFFFF;
	    sprintf (fdev, "S3:%d", devnum);
	    detachit (SERIAL9903, fdev);
	    attachit (bp, SERIAL9903, devnum, "r+", fdev);
	    break;

	 default: 
	    goto BADDEV;
	 }
	 break;

      case 'v': /* 911 vdt */
	 if (!isdigit (*bp)) goto BADUNIT;
	 bp = getnum (bp, &val);
	 devnum = val & 0xFFFF;
	 sprintf (fdev, "V%d", devnum);
	 attachit (bp, VDT911, devnum, "r+", fdev);
	 break;

      default:
BADDEV:
	 for (bp = cp; *bp && !isspace (*bp); bp++) ;
	 *bp = 0;
	 sprintf (view[0], "Device %s: Undefined", cp);
	 for (i = 0; devusage[i][0]; i++)
	    strcpy (view[i+1], devusage[i]);
	 break;
   }
   return;

BADUNIT:
   for (bp = cp; *bp && !isspace (*bp); bp++) ;
   *bp = 0;
   sprintf (view[0], "Device %s: Invalid or missing unit", cp);
   return;

BADDEVCPU:
   for (bp = cp; *bp && !isspace (*bp); bp++) ;
   *bp = 0;
   sprintf (view[0], "Device %s: Unsupported on 990/%s",
	    cp, models[model-4]);
   return;
}

/***********************************************************************
* resetdev - Reset devices for RSET.
***********************************************************************/

void
resetdev (void)
{
   Device *dev;
   int i;

   intrequest = 0; /* Reset all interrupts */
   for (i = 0; i < 16; i++)
   {
      intqueue[i].count = 0;
      intqueue[i].top = 0;
      intqueue[i].bottom = 0;
   }

   for (i = 0; i < devcnt; i++)
   {
      dev = &devices[i];
      switch (dev->type)
      {
      case TAPE:
	 tapereset (dev);
	 break;
      case DISK:
	 dskreset (dev);
	 break;
      default: ;
      }
   }
}

/***********************************************************************
* checkcru - Check cru address and adjust.
***********************************************************************/

static int
checkcru (uint16 *pdevaddr, int8 *pdisp)
{
   uint16 devaddr = *pdevaddr;
   int8 disp = *pdisp;

#if defined(DEBUGCRU)
   fprintf (stderr, "checkcru: input CRU %04X %d\n", devaddr, disp);
#endif

   /*
   ** Adjust CRU address
   */

   if (disp > 15)
   {
      while (disp > 15)
      {
	 disp -= 16;
	 devaddr += 0x20;
      }
#ifdef DEBUGADJCRU
      fprintf (stderr, "  adjusted %04X(%04X) %d(%d)\n",
               devaddr, *pdevaddr, disp, *pdisp);
#endif
   }
   else if (disp < 0)
   {
      while (disp < 0)
      {
	 disp += 16;
	 devaddr -= 0x20;
      }
#ifdef DEBUGADJCRU
      fprintf (stderr, "  adjusted %04X(%04X) %d(%d)\n",
               devaddr, *pdevaddr, disp, *pdisp);
#endif
   }

   /*
   ** Adjust to base address of device
   */

   if (devaddr & 0x1F)
   {
      disp += ((devaddr & 0x1F) >> 1);
      devaddr &= 0xFFE0;
#if defined(DEBUGCRU)
      fprintf (stderr, "  adjusted %04X(%04X) %d(%d)\n",
               devaddr, *pdevaddr, disp, *pdisp);
#endif
   }

   devaddr &= CRUEND;

   /*
   ** Check for privleged CRU address
   */

   if (model >= 10 && IS_NONPRV)
   {
      if (devaddr + disp >= PRVCRU)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%d: Priv CRU: cru = %04X, disp = %02x\n",
		  instcount, devaddr, disp);
#endif
	 errcrudata |= ERRPIN;
	 if (GET_MASK >= ERRINT)
	    geninterrupt (ERRINT);
	 return (-1);
      }
   }

   *pdevaddr = devaddr;
   *pdisp = disp;
   return (0);
}

/***********************************************************************
* setbitone - Set CRU bit to one.
***********************************************************************/

void
setbitone (uint16 inst)
{
   Device *dev;
   int i;
   uint16 devaddr;
   int8 disp;
#if defined(LOGUNKNOWNDEVICE) || defined(DEBUG9902)
   uint16 origdevaddr;
   uint8 origdisp;
#endif

   disp = inst & 0x00FF;
   devaddr = R12;

#if defined(LOGUNKNOWNDEVICE) || defined(DEBUG9902)
   origdevaddr = devaddr;
   origdisp = disp;
#endif
#ifdef DEBUGCRU
   fprintf (stderr, "CRU %04X SBO %d\n", devaddr, disp);
#endif

   /*
   ** Check CRU address
   */

   if (checkcru (&devaddr, &disp) < 0)
   {
      return;
   }

   /*
   ** 990/5 ROM control
   */

   if (devaddr == ROM5CRU && model == 5)
   {
#ifdef DEBUGROM5
      fprintf (stderr, "CRU %04X SBO %d\n", devaddr, disp);
#endif
      if (disp == 12)
         rom5cru = TRUE;
      return;
   }

   /*
   ** Front panel is always attached
   */

   if (devaddr == PANEL)
   {
#ifdef DEBUGPANEL
      fprintf (stderr, "CRU %04X SBO %d\n", devaddr, disp);
#endif
      switch (disp)
      {
      case 10:
	 runled = TRUE;
	 faultled = FALSE;
	 rom5cru = FALSE;
	 break;
      case 11:
	 faultled = TRUE;
	 break;
      case 12:
	 errcrudata &= ~ ERRTIP; /* 990/4 compatible */
	 break;
      default: 
#if defined(PANELNONE)
	 fprintf (stderr, "CRU %04X SBO %d\n", devaddr, disp);
#endif
         ;
      }
      return;
   }
   
   /*
   ** Memory mapper
   */

   if (devaddr == MAPPER && model >= 10)
   {
#ifdef DEBUGMAPCRU
      fprintf (stderr, "%d: CRU %04X SBO %d\n", instcount, devaddr, disp);
#endif
      if (model == 12)
      {
	 switch (disp)
	 {
	 case 0: /* Index through trace memory */
	    trclatch = FALSE;
	    trcindex++;
	    if (trcindex == 32) trcindex = 0;
	    break;
	 case 1: /* Test clock */
	    tcenable = TRUE;
	    tccount = 30;
	    break;
	 case 3: /* Mapping enable */
	    mapenabled = TRUE;
	    break;
	 case 4: /* Error clear */
	    errcrudata &= ~ ERRMAP;
	    break;
	 case 8: /* Diagnostic interrupt */
#ifdef DEBUGBKPT
	    fprintf (stderr, "%04X: CRU %04X SBO %d\n", curpc, devaddr, disp);
#endif
	    diaginterrupt = TRUE;
	    break;
	 case 9: /* Force Parity error */
	    forceparity = TRUE;
	    break;
	 default:
#if defined(PANELNONE)
	    fprintf (stderr, "CRU %04X SBO %d\n", devaddr, disp);
#endif
            ;
	 }
      }
      else
      {
	 switch (disp)
	 {
	 case 3:
	    mapenabled = TRUE;
	    break;
	 case 4:
	    errcrudata &= ~ ERRMAP;
	 case 5:
	 case 6:
	 case 7:
	    maplatch = 0;
	    break;
	 default: ;
	 }
      }
      mapcrudata |= 1 << disp;
#ifdef DEBUGMAPCRU
      fprintf (stderr,"   mapcrudata = >%04X\n", mapcrudata);
#endif
      return;
   }

   /*
   ** Error control
   */

   if (devaddr == ERRCRU && (model == 5 || model >= 10))
   {
#ifdef DEBUGERRCRU
      fprintf (stderr, "%04X: CRU %04X SBO %d MASK %4.4X VAL %4.4X\n",
	       curpc, devaddr, disp, 1 << disp, errcrudata);
#endif
      if (disp < 16)
      {
	 if (disp > 3)
	    errcrudata |= (1 << disp);
	 if ((GET_MASK >= ERRINT) && (errcrudata & 0xFFF0) && (disp > 12))
	    geninterrupt (ERRINT);
      }
      return;
   }
   
   /*
   ** Breakpoint facility
   */

   if (devaddr == BKPTCRU && model == 12)
   {
#ifdef DEBUGBKPT
      fprintf (stderr, "%04X: CRU %04X SBO %d\n", curpc, devaddr, disp);
#endif
      if (disp < 16)
      {
	 if (disp == 0)
	    bkptenabled = TRUE;
	 else 
	    bkptcrudata |= (1 << disp);
      }
#ifdef DEBUGBKPT
      fprintf (stderr, "   bkptenabled = %s, bkptcrudata = %04X, mask = %d\n",
               bkptenabled ? "TRUE" : "FALSE", bkptcrudata,
	       (mapcrudata >> 6) & 3);
#endif
      return;
   }

   /*
   ** 990/5 port interrupt control.
   */

   if (devaddr == INTRCRU && model == 5)
   {
#if defined(DEBUG9902) || defined(DEBUG9903) || defined(DEBUGROM5)
      fprintf (stderr, "CRU %04X SBO %d\n", devaddr, disp);
#endif
      return;
   }

   /*
   ** Check for CRU Expansion chassis.
   */

   if (devaddr == EXP1CRU)
   {
#ifdef DEBUGEXPCHASSIS
       fprintf (stderr, "CRU %04X SBO %d\n", devaddr, disp);
#endif
       if (disp < 16)
       {
           xcard1crudata |= 1 << disp;
           xcard1crudata &= EXPMASK;
       }       
       return;
   }
   if (devaddr == EXP2CRU)
   {
#ifdef DEBUGEXPCHASSIS
       fprintf (stderr, "CRU %04X SBO %d\n", devaddr, disp);
#endif
       if (disp < 16)
       {
           xcard2crudata |= 1 << disp;
           xcard2crudata &= EXPMASK;
       }
       return;
   }

   /*
   ** Peripheral device
   */

   if ((i = finddev (devaddr)) < 0)
   {
#ifdef LOGUNKNOWNDEVICE
      fprintf (stderr, "**CRU %04X(%04X) SBO %d(%d)\n",
	       devaddr, origdevaddr, disp, origdisp);
#endif
      return;
   }
   dev = &devices[i];

   switch (dev->type)
   {
   case CONEIA:
   case SERIALEIA:
   case PRINTEIA:
      eiaseto (dev, devaddr, disp);
      break;

   case CON9902:
   case SERIAL9902:
   case PRINT9902:
   case SERIALCI402:
#ifdef DEBUG9902A
      fprintf (stderr, "++CRU %04X(%04X) SBO %d(%d)\n",
	       devaddr, origdevaddr, disp, origdisp);
#endif
      a9902seto (dev, devaddr, disp);
      break;

   case CON9903:
   case SERIAL9903:
   case PRINT9903:
      a9903seto (dev, devaddr, disp);
      break;

   case SERIALCI401:
      ci401seto (dev, devaddr, disp);
      break;

   case PRINTPAR:
      printseto (dev, devaddr, disp);
      break;

   case VDT911:
      v911seto (dev, disp);
      break;

   case FLOPPY:
      fpyseto (dev, disp);
      break;

   case CARDPUNCH:
      punchseto (dev, devaddr, disp);
      break;

   case CARDREADER:
      cardseto (dev, devaddr, disp);
      break;

   default: ;
   }
   return;
}

/***********************************************************************
* setbitzero - Set CRU bit to zero.
***********************************************************************/

void
setbitzero (uint16 inst)
{
   Device *dev;
   int i;
   uint16 devaddr;
   int8 disp;
#if defined(LOGUNKNOWNDEVICE) || defined(DEBUG9902)
   uint16 origdevaddr;
   uint8 origdisp;
#endif

   disp = inst & 0x00FF;
   devaddr = R12;

#if defined(LOGUNKNOWNDEVICE) || defined(DEBUG9902)
   origdevaddr = devaddr;
   origdisp = disp;
#endif
#ifdef DEBUGCRU
   fprintf (stderr, "CRU %04X SBZ %d\n", devaddr, disp);
#endif

   /*
   ** Check CRU address
   */

   if (checkcru (&devaddr, &disp) < 0)
   {
      return;
   }

   /*
   ** 990/5 ROM control
   */

   if (devaddr == ROM5CRU && model == 5)
   {
#ifdef DEBUGROM5
      fprintf (stderr, "CRU %04X SBZ %d\n", devaddr, disp);
#endif
      if (disp == 12)
         rom5cru = FALSE;
      return;
   }

   /*
   ** Front panel is always attached
   */

   if (devaddr == PANEL)
   {
#ifdef DEBUGPANEL
      fprintf (stderr, "CRU %04X SBZ %d\n", devaddr, disp);
#endif
      switch (disp)
      {
      case 11:
	 faultled = FALSE;
	 break;
      case 12:
	 errcrudata &= ~ ERRTIP; /* 990/4 compatible */
	 break;
      default: 
#ifdef DEBUGPANELNONE
	 fprintf (stderr, "CRU %04X SBZ %d\n", devaddr, disp);
#endif
	 ;
      }
      return;
   }
   
   /*
   ** Memory mapper
   */

   if (devaddr == MAPPER && model >= 10)
   {
#ifdef DEBUGMAPCRU
      fprintf (stderr, "%d: CRU %04X SBZ %d\n", instcount, devaddr, disp);
#endif
      if (model == 12)
      {
	 switch (disp)
	 {
	 case 0: /* Index through trace memory */
	    trclatch = FALSE;
	    trcindex++;
	    if (trcindex == 32) trcindex = 0;
	    break;
	 case 1: /* Test clock */
	    tcenable = FALSE;
	    tccount = 0;
	    break;
	 case 3: /* Mapping enable */
	    mapenabled = FALSE;
	    maplatch = 0;
	    break;
	 case 4: /* Error clear */
	    errcrudata &= ~ ERRMAP;
	    maplatch = MAPPER | (mapcrudata & 0x7); /* The diag wants this?? */
	    break;
	 case 8: /* Diagnostic interrupt */
#ifdef DEBUGBKPT
	    fprintf (stderr, "%04X: CRU %04X SBZ %d\n", curpc, devaddr, disp);
#endif
	    diaginterrupt = FALSE;
	    break;
	 case 9: /* Force Parity error */
	    forceparity = FALSE;
	    break;
	 default: ;
	 }
      }
      else
      {
	 switch (disp)
	 {
	 case 3:
	    mapenabled = FALSE;
	    maplatch = 0;
	    break;
	 case 4:
	    errcrudata &= ~ ERRMAP;
	    maplatch = MAPPER | (mapcrudata & 0x7); /* The diag wants this?? */
	    break;
	 case 5:
	 case 6:
	 case 7:
	    maplatch = 0;
	    break;
	 default: ;
	 }
      }
      mapcrudata &= ~(1 << disp);
#ifdef DEBUGMAPCRU
      fprintf (stderr,"   mapcrudata = >%04X\n", mapcrudata);
#endif
      return;
   }
   
   /*
   ** Error control
   */

   if (devaddr == ERRCRU && (model == 5 || model >= 10))
   {
#ifdef DEBUGERRCRU
      fprintf (stderr, "%04X: CRU %04X SBZ %d MASK %4.4X VAL %4.4X\n",
	       curpc, devaddr, disp, 1 << disp, errcrudata);
#endif
      if (disp < 16)
      {
	 if (disp > 3)
	    errcrudata &= ~(1 << disp);
	 if ((GET_MASK >= ERRINT) && (errcrudata & 0xFFF0) && (disp > 12))
	    geninterrupt (ERRINT);
      }
      return;
   }
   
   /*
   ** Breakpoint facility
   */

   if (devaddr == BKPTCRU && model == 12)
   {
#ifdef DEBUGBKPT
      fprintf (stderr, "%04X: CRU %04X SBZ %d\n", curpc, devaddr, disp);
#endif
      if (disp < 16)
      {
	 if (disp == 0)
	    bkptenabled = FALSE;
	 else 
	    bkptcrudata &= ~(1 << disp);
      }
#ifdef DEBUGBKPT
      fprintf (stderr, "   bkptenabled = %s, bkptcrudata = %04X, mask = %d\n",
               bkptenabled ? "TRUE" : "FALSE", bkptcrudata,
	       (mapcrudata >> 6) & 3);
#endif
      return;
   }

   /*
   ** 990/5 port interrupt control.
   */

   if (devaddr == INTRCRU && model == 5)
   {
#if defined(DEBUG9902) || defined(DEBUG9903) || defined(DEBUGROM5)
      fprintf (stderr, "CRU %04X SBZ %d\n", devaddr, disp);
#endif
      return;
   }

   /*
   ** Check for CRU Expansion chassis.
   */

   if (devaddr == EXP1CRU)
   {
#ifdef DEBUGEXPCHASSIS
       fprintf (stderr, "CRU %04X SBZ %d\n", devaddr, disp);
#endif
       if (disp < 16)
       {
           xcard1crudata &= ~(1 << disp);
           xcard1crudata &= EXPMASK;
       }       
       return;
   }
   if (devaddr == EXP2CRU)
   {
#ifdef DEBUGEXPCHASSIS
       fprintf (stderr, "CRU %04X SBZ %d\n", devaddr, disp);
#endif
       if (disp < 16)
       {
           xcard2crudata &= ~(1 << disp);
           xcard2crudata &= EXPMASK;
       }
       return;
   }

   /*
   ** Peripheral device
   */

   if ((i = finddev (devaddr)) < 0)
   {
#ifdef LOGUNKNOWNDEVICE
      fprintf (stderr, "**CRU %04X(%04X) SBZ %d(%d)\n",
	       devaddr, origdevaddr, disp, origdisp);
#endif
      return;
   }
   dev = &devices[i];

   switch (dev->type)
   {
   case CONEIA:
   case SERIALEIA:
   case PRINTEIA:
      eiasetz (dev, devaddr, disp);
      break;

   case CON9902:
   case SERIAL9902:
   case PRINT9902:
   case SERIALCI402:
#ifdef DEBUG9902A
      fprintf (stderr, "++CRU %04X(%04X) SBZ %d(%d)\n",
	       devaddr, origdevaddr, disp, origdisp);
#endif
      a9902setz (dev, devaddr, disp);
      break;

   case CON9903:
   case SERIAL9903:
   case PRINT9903:
      a9903setz (dev, devaddr, disp);
      break;

   case SERIALCI401:
      ci401setz (dev, devaddr, disp);
      break;

   case PRINTPAR:
      printsetz (dev, devaddr, disp);
      break;

   case VDT911:
      v911setz (dev, disp);
      break;

   case FLOPPY:
      fpysetz (dev, disp);
      break;

   case CARDPUNCH:
      punchsetz (dev, devaddr, disp);
      break;

   case CARDREADER:
      cardsetz (dev, devaddr, disp);
      break;

   default: ;
   }

   return;
}

/***********************************************************************
* testbit - Test CRU bit.
***********************************************************************/

void
testbit (uint16 inst)
{
   Device *dev;
   int i;
   uint16 devaddr;
   int8 disp;
#if defined(LOGUNKNOWNDEVICE) || defined(DEBUG9902) || defined(DEBUGCI401)
   uint16 origdevaddr;
   uint8 origdisp;
#endif

   disp = inst & 0x00FF;
   devaddr = R12;

#if defined(LOGUNKNOWNDEVICE) || defined(DEBUG9902) || defined(DEBUGCI401)
   origdevaddr = devaddr;
   origdisp = disp;
#endif
#ifdef DEBUGCRU
   fprintf (stderr, "CRU %04X TB %d\n", devaddr, disp);
#endif

   /*
   ** Check CRU address
   */

   if (checkcru (&devaddr, &disp) < 0)
   {
      return;
   }

   /*
   ** 990/5 ROM control
   */

   if (devaddr == ROM5CRU && model == 5)
   {
      CLR_EQ;
      if (disp == 12 && rom5cru)
         SET_EQ;
#ifdef DEBUGROM5
      fprintf (stderr, "CRU %04X TB %d VAL %s\n",
               devaddr, disp, IS_EQ ? "ON" : "OFF");
#endif
      return;
   }

   /*
   ** Front panel is always attached
   */

   if (devaddr == PANEL)
   {
#ifdef DEBUGPANEL
      fprintf (stderr, "CRU %04X TB %d\n", devaddr, disp);
#endif
      CLR_EQ;
      if (disp == 11) SET_EQ;
      else if (disp == 14) SET_EQ;
#if defined(UNDEFBITS)
      else
	 fprintf (stderr, "CRU %04X TB %d\n", devaddr, disp);
#endif
      return;
   }
   
   /*
   ** Memory mapper
   */

   if (devaddr == MAPPER && model >= 10)
   {
#ifdef DEBUGMAPCRU
      fprintf (stderr, "%d: CRU %04X TB %d\n", instcount, devaddr, disp);
      fprintf (stderr,"   mapcrudata = >%04X\n", mapcrudata);
#endif
      if ((mapcrudata & (1 << disp)))
      {
	 if (disp == 3) mapenabled = TRUE;
#if defined(UNDEFBITS)
	 else
	    fprintf (stderr, "CRU %04X TB %d\n", devaddr, disp);
#endif
	 SET_EQ;
      }
      else
      {
	 if (disp == 3) mapenabled = FALSE;
#if defined(UNDEFBITS)
	 else
	    fprintf (stderr, "CRU %04X TB %d\n", devaddr, disp);
#endif
	 CLR_EQ;
      }
      return;
   }
   
   /*
   ** Error control
   */

   if (devaddr == ERRCRU && (model == 5 || model >= 10))
   {
#ifdef DEBUGERRCRU
      fprintf (stderr, "%04X: CRU %04X TB  %d MASK %4.4X VAL %4.4X",
	       curpc, devaddr, disp, 1 << disp, errcrudata);
#endif
      if (disp < 16 && (errcrudata & (1 << disp)))
      {
	 SET_EQ;
#ifdef DEBUGERRCRU
	 fprintf (stderr, " EQUAL\n");
#endif
      }
      else
      {
	 CLR_EQ;
#ifdef DEBUGERRCRU
	 fprintf (stderr, " NOT EQUAL\n");
#endif
      }
      return;
   }
   
   /*
   ** Breakpoint facility
   */

   if (devaddr == BKPTCRU && model == 12)
   {
#if defined(UNDEFBITS)
      fprintf (stderr, "CRU %04X TB %d\n", devaddr, disp);
#endif
#ifdef DEBUGBKPT
      fprintf (stderr, "%04X: CRU %04X TB  %d MASK %4.4X VAL %4.4X",
	       curpc, devaddr, disp, 1 << disp, bkptcrudata);
#endif
      if (disp < 16 && (bkptcrudata & (1 << disp)))
      {
	 SET_EQ;
#ifdef DEBUGBKPT
	 fprintf (stderr, " EQUAL\n");
#endif
      }
      else
      {
	 CLR_EQ;
#ifdef DEBUGBKPT
	 fprintf (stderr, " NOT EQUAL\n");
#endif
      }
      return;
   }

   /*
   ** 990/5 port interrupt control.
   */

   if (devaddr == INTRCRU && model == 5)
   {
#if defined(DEBUG9902) || defined(DEBUG9903) || defined(DEBUGROM5)
      fprintf (stderr, "CRU %04X TB %d\n", devaddr, disp);
#endif
      CLR_EQ;
      return;
   }

   /*
   ** Check for CRU Expansion chassis.
   */

   if (devaddr == EXP1CRU)
   {
      xcard1crudata &= EXPMASK;
#ifdef DEBUGEXPCHASSIS
      fprintf (stderr, "CRU %04X TB %d data %04X (%s)\n",
	       devaddr, disp, xcard1crudata,
	       (xcard1crudata & (1 << disp)) ? "ON" : "OFF");
#endif
      if (disp < 16 && (xcard1crudata & (1 << disp)))
      {
	 SET_EQ;
      }
      else
      {
	 CLR_EQ;
      }
      return;
   }
   if (devaddr == EXP2CRU)
   {
      xcard2crudata &= EXPMASK;
#ifdef DEBUGEXPCHASSIS
      fprintf (stderr, "CRU %04X TB %d data %04X (%s)\n",
	       devaddr, disp, xcard2crudata,
	       (xcard2crudata & (1 << disp)) ? "ON" : "OFF");
#endif
      if (disp < 16 && (xcard2crudata & (1 << disp)))
      {
	 SET_EQ;
      }
      else
      {
         CLR_EQ;
      }
      return;
   }

   /*
   ** Peripheral device
   */

   if ((i = finddev (devaddr)) < 0)
   {
#ifdef LOGUNKNOWNDEVICE
      fprintf (stderr, "**CRU %04X(%04X) TB %d(%d)\n",
	       devaddr, origdevaddr, disp, origdisp);
#endif
#if defined(UNDEFBITS)
      fprintf (stderr, "CRU %04X TB %d\n", devaddr, disp);
#endif
      CLR_EQ;
      return;
   }
   dev = &devices[i];

   switch (dev->type)
   {
   case CONEIA:
   case SERIALEIA:
   case PRINTEIA:
      eiatb (dev, devaddr, disp);
      break;

   case CON9902:
   case SERIAL9902:
   case PRINT9902:
   case SERIALCI402:
#ifdef DEBUG9902A
      fprintf (stderr, "++CRU %04X(%04X) TB %d(%d)\n",
	       devaddr, origdevaddr, disp, origdisp);
#endif
      a9902tb (dev, devaddr, disp);
      break;

   case CON9903:
   case SERIAL9903:
   case PRINT9903:
      a9903tb (dev, devaddr, disp);
      break;

   case SERIALCI401:
      ci401tb (dev, devaddr, disp);
      break;

   case PRINTPAR:
      printtb (dev, devaddr, disp);
      break;

   case VDT911:
      v911tb (dev, disp);
      break;

   case FLOPPY:
      fpytb (dev, disp);
      break;

   case CARDPUNCH:
      punchtb (dev, devaddr, disp);
      break;

   case CARDREADER:
      cardtb (dev, devaddr, disp);
      break;

   default: ;
   }
   return;
}

/***********************************************************************
* loadcru - Load CRU.
***********************************************************************/

void
loadcru (uint16 inst, uint16 dat)
{
   Device *dev;
   int i, j;
   uint16 cnt;
   uint16 devaddr;
   uint16 mask;
   uint16 sdat;
   uint8 ch;
#if defined(LOGUNKNOWNDEVICE) || defined(DEBUG9902)
   uint16 origdevaddr;
#endif

   cnt = (inst & 0x03C0) >> 6;

   devaddr = R12;

#if defined(LOGUNKNOWNDEVICE) || defined(DEBUG9902)
   origdevaddr = devaddr;
#endif
#ifdef DEBUGCRU
   fprintf (stderr, "CRU %04X LDCR CNT %d DAT %04X\n", devaddr, cnt, dat);
#endif

   sdat = dat;
   mask = 0;
   j = cnt ? cnt : 16;
   for (i = 0; i < j; i++)
   {
      mask |= (1 << i);
   }
#ifdef DEBUGCRU
   fprintf (stderr, "   mask = >%04X\n", mask);
#endif

   if (devaddr & 0x1F)
   {
#if defined(DEBUGCRU)
      fprintf (stderr,
	       "loadcru: CRU address biased: devaddr = >%4X, cnt = %d\n",
	       devaddr, cnt);
#endif
      mask <<= ((devaddr & 0x1F) >> 1);
      sdat <<= ((devaddr & 0x1F) >> 1);
      devaddr &= 0xFFE0;
#ifdef DEBUGCRU
      fprintf (stderr,
	       "   adjusted: devaddr = >%4X, sdat = >%4X, mask = >%4X\n",
	       devaddr, sdat, mask);
#endif
   }

   /*
   ** Check for privleged CRU address
   */

#ifdef DEBUGPRIVCRU
   if (devaddr >= PRVCRU)
      fprintf (stderr, "CRU %04X LDCR CNT %d DAT %04X\n", devaddr, cnt, dat);
#endif

   if (model >= 10 && IS_NONPRV)
   {
      if (devaddr >= PRVCRU)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%d: Priv CRU: cru = %04X\n",
		  instcount, devaddr);
#endif
	 errcrudata |= ERRPIN;
	 if (GET_MASK >= ERRINT)
	    geninterrupt (ERRINT);
	 return;
      }
   }

   /*
   ** Front panel is always attached
   */

   if (devaddr == PANEL)
   {
#ifdef DEBUGPANEL
      fprintf (stderr, "CRU %04X LDCR CNT %d DAT %04X\n", devaddr, cnt, dat);
#endif
      ch = (uint8)dat;
      lights = (lights << 8) | ch;
      return;
   }

   /*
   ** Memory mapper
   */

   if (devaddr == MAPPER && model >= 10)
   {
#ifdef DEBUGMAPCRU
      fprintf (stderr, "%d: CRU %04X LDCR CNT %d DAT %04X\n",
	       instcount, devaddr, cnt, sdat);
#endif
      /* For a /12 and a ROM bank switch, fake an inst prefetch */
      if ((model == 12) && (mask == 0x1C00))
      {
	 mapolddata = mapcrudata;
         fakeprefetch = TRUE;
      }
      mapcrudata = (mapcrudata & ~mask) | (sdat & mask);
      mapenabled = (mapcrudata & MAPENABLE) ? TRUE : FALSE;
#ifdef DEBUGMAPCRU
      fprintf (stderr,"   mapcrudata = >%04X\n", mapcrudata);
#endif
      return;
   }

   /*
   ** Error control
   */

   if (devaddr == ERRCRU && (model == 5 || model >= 10))
   {
#ifdef DEBUGERRCRU
      fprintf (stderr, "%04X: CRU %04X LDCR CNT %d DAT %04X\n",
	       curpc, devaddr, cnt, dat);
#endif
      errcrudata = (errcrudata & 0xF) | ((dat & mask) & 0xFFF0);
      return;
   }

   /*
   ** Breakpoint facility
   */

   if (devaddr == BKPTCRU && model == 12)
   {
#ifdef DEBUGBKPT
      fprintf (stderr, "%04X: CRU %04X LDCR CNT %d DAT %04X\n",
	       curpc, devaddr, cnt, dat);
#endif
      bkptcrudata = dat & 0xFFFE;
      bkptenabled = dat & 1;
#ifdef DEBUGBKPT
      fprintf (stderr, "   bkptenabled = %s, bkptcrudata = %04X, mask = %d\n",
               bkptenabled ? "TRUE" : "FALSE", bkptcrudata,
	       (mapcrudata >> 6) & 3);
#endif
      return;
   }

   /*
   ** 990/5 port interrupt control.
   */

   if (devaddr == INTRCRU && model == 5)
   {
#if defined(DEBUG9902) || defined(DEBUG9903) || defined(DEBUGROM5)
      fprintf (stderr, "CRU %04X LDCR CNT %d DAT %04X\n", devaddr, cnt, dat);
#endif
      return;
   }
   
   /*
   ** Check for CRU Expansion chassis.
   */

   if (devaddr == EXP1CRU)
   {
#ifdef DEBUGEXPCHASSIS
      fprintf (stderr, "CRU %04X LDCR CNT %d DAT %04X\n", devaddr, cnt, dat);
#endif
      xcard1crudata = dat;
      xcard1crudata &= EXPMASK;
      return;
   }
   if (devaddr == EXP2CRU)
   {
#ifdef DEBUGEXPCHASSIS
      fprintf (stderr, "CRU %04X LDCR CNT %d DAT %04X\n", devaddr, cnt, dat);
#endif
      xcard2crudata = dat;
      xcard2crudata &= EXPMASK;
      return;
   }

   /*
   ** Peripheral device
   */

   if ((i = finddev (devaddr)) < 0)
   {
#ifdef LOGUNKNOWNDEVICE
      fprintf (stderr, "**CRU %04X(%04X) LDCR CNT %d DAT %04X\n",
	       devaddr, origdevaddr, cnt, dat);
#endif
      return;
   }
   dev = &devices[i];

   switch (dev->type)
   {
   case CONEIA:
   case SERIALEIA:
   case PRINTEIA:
      eialdcr (dev, devaddr, cnt, dat);
      break;

   case CON9902:
   case SERIAL9902:
   case PRINT9902:
   case SERIALCI402:
      a9902ldcr (dev, devaddr, cnt, dat);
      break;

   case CON9903:
   case SERIAL9903:
   case PRINT9903:
      a9903ldcr (dev, devaddr, cnt, dat);
      break;

   case SERIALCI401:
      ci401ldcr (dev, devaddr, cnt, dat);
      break;

   case PRINTPAR:
      printldcr (dev, devaddr, cnt, dat);
      break;

   case VDT911:
      v911ldcr (dev, cnt, dat);
      break;

   case FLOPPY:
      fpyldcr (dev, cnt, dat);
      break;

   case CARDPUNCH:
      punchldcr (dev, devaddr, cnt, dat);
      break;

   case CARDREADER:
      cardldcr (dev, devaddr, cnt, dat);
      break;

   default: ;
   }

   return;
}

/***********************************************************************
* storecru - Store CRU.
***********************************************************************/

uint16
storecru (uint16 inst)
{
   Device *dev;
   int i;
   uint16 cnt;
   uint16 devaddr;
   uint16 dat;
#if defined(LOGUNKNOWNDEVICE) || defined(DEBUG9902)
   uint16 origdevaddr;
#endif

   cnt = (inst & 0x03C0) >> 6;

   dat = 0;
   devaddr = R12;

#if defined(LOGUNKNOWNDEVICE) || defined(DEBUG9902)
   origdevaddr = devaddr;
#endif
#ifdef DEBUGCRU
   fprintf (stderr, "CRU %04X STCR CNT %d ", devaddr, cnt);
#endif

#if defined(CHECKCRU)
   if (devaddr & 0x1F)
   {
      fprintf (stderr,
	       "storecru: CRU address biased: devaddr = >%4X, cnt = %d\n",
	       devaddr, cnt);
   }
#endif

   /*
   ** Check for privleged CRU address
   */

#ifdef DEBUGPRIVCRU
   if (devaddr >= PRVCRU)
      fprintf (stderr, "CRU %04X STCR CNT %d ", devaddr, cnt);
#endif

   if (model >= 10 && IS_NONPRV)
   {
      if (devaddr >= PRVCRU)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%d: Priv CRU: cru = %04X\n",
		  instcount, devaddr);
#endif
	 errcrudata |= ERRPIN;
	 if (GET_MASK >= ERRINT)
	    geninterrupt (ERRINT);
	 return (0);
      }
   }

   /*
   ** Front panel is always attached
   */

   if (devaddr == PANEL)
   {
#ifdef DEBUGPANEL
      fprintf (stderr, "CRU %04X STCR CNT %d DAT %04X\n", devaddr, cnt, 0);
#endif
#ifdef DEBUGCRU
      fprintf (stderr, "DAT %d\n", dat);
#endif
#ifdef DEBUGPRIVCRU
      fprintf (stderr, "DAT %04X\n", dat);
#endif
      return (dat);
   }
   
   /*
   ** Memory mapper
   */

   if (devaddr == MAPPER && model >= 10)
   {
      uint16 retval = 0xFFFF;

      if (model < 12)
      {
#ifdef DEBUGMAPCRU
	 fprintf (stderr, "%d: CRU %04X STCR CNT %d DAT %04X\n",
		  instcount, devaddr, cnt, mapcrudata);
#endif
#ifdef DEBUGCRU
	 fprintf (stderr, "DAT %04X\n", mapcrudata);
#endif
	 switch (mapcrudata & 0x7)
	 {
	    case 0:
	       retval = mapfile[curmapindex].b1;
	       break;
	    case 1:
	       retval = mapfile[curmapindex].b2;
	       break;
	    case 2:
	       retval = mapfile[curmapindex].b3;
	       break;
	    case 3:
	       retval = mapfile[curmapindex].l1;
	       break;
	    case 4:
	       retval = mapfile[curmapindex].l2;
	       break;
	    case 5:
	       retval = mapfile[curmapindex].l3;
	       break;
	    case 6:
	       retval = maplatch & 0xFFFE;
	       break;
	    case 7:
	       retval = (maplatch >> 16) & 0x001F;
	 }
#ifdef DEBUGMAPCRU
	 fprintf (stderr, "   RETVAL >%04X\n", retval);
#endif
      }
      else
      {
	 int i = trcindex >> 1;

	 if (trcindex & 1)
	    retval = (tracemem[i] >> 16) & 0xFFFF;
	 else
	    retval = tracemem[i] & 0xFFFF;
#ifdef DEBUGMAPCRU
	 fprintf (stderr, "%d: CRU %04X STCR CNT %d DAT %04X\n",
		  instcount, devaddr, cnt, retval);
	 fprintf (stderr, "   trcindex = %d(%d), tracemem = >%08X\n",
		  trcindex, i, tracemem[i]);
#endif
      }
#ifdef DEBUGPRIVCRU
      fprintf (stderr, "DAT %04X\n", retval);
#endif
      return (retval);
   }
   
   /*
   ** Error control
   */

   if (devaddr == ERRCRU && (model == 5 || model >= 10))
   {
#ifdef DEBUGERRCRU
      fprintf (stderr, "%04X: CRU %04X STCR CNT %d DAT %04X\n",
	       curpc, devaddr, cnt, errcrudata);
#endif
#ifdef DEBUGCRU
      fprintf (stderr, "DAT %04X\n", errcrudata);
#endif
#ifdef DEBUGPRIVCRU
      fprintf (stderr, "DAT %04X\n", errcrudata);
#endif
      return (errcrudata);
   }

   /*
   ** Breakpoint facility
   */

   if (devaddr == BKPTCRU && model == 12)
   {
      dat = bkptcrudata;
#ifdef DEBUGBKPT
      fprintf (stderr, "%04X: CRU %04X STCR CNT %d DAT %04X\n",
	       curpc, devaddr, cnt, dat);
#endif
#ifdef DEBUGPRIVCRU
      fprintf (stderr, "DAT %04X\n", dat);
#endif
      return (dat);
   }

   /*
   ** 990/5 port interrupt control.
   */

   if (devaddr == INTRCRU && model == 5)
   {
#if defined(DEBUG9902) || defined(DEBUG9903) || defined(DEBUGROM5)
      fprintf (stderr, "CRU %04X STCR CNT %d DAT %04X\n", devaddr, cnt, 0);
#endif
      return (0);
   }
   
   /*
   ** Check for CRU Expansion chassis.
   */

   if (devaddr == EXP1CRU)
   {
      xcard1crudata &= EXPMASK;
#ifdef DEBUGEXPCHASSIS
      fprintf (stderr, "CRU %04X STCR CNT %d DAT %04X\n",
		devaddr, cnt, xcard1crudata);
#endif
      dat = xcard1crudata;
      xcard1crudata &= 0x7FFF;
      return (dat);
   }
   if (devaddr == EXP2CRU)
   {
      xcard2crudata &= EXPMASK;
#ifdef DEBUGEXPCHASSIS
      fprintf (stderr, "CRU %04X STCR CNT %d DAT %04X\n",
		devaddr, cnt, xcard1crudata);
#endif
      dat = xcard2crudata;
      xcard2crudata &= 0x7FFF;
      return (dat);
   }

   /*
   ** Peripheral device
   */

   if ((i = finddev (devaddr)) < 0)
   {
#ifdef LOGUNKNOWNDEVICE
      fprintf (stderr, "**CRU %04X(%04X) STCR CNT %d DAT FFFF\n",
	       devaddr, origdevaddr, cnt);
#endif
      return (-1);
   }
   dev = &devices[i];

   switch (dev->type)
   {
   case CONEIA:
   case SERIALEIA:
   case PRINTEIA:
      dat = eiastcr (dev, devaddr, cnt);
      break;

   case CON9902:
   case SERIAL9902:
   case PRINT9902:
   case SERIALCI402:
      dat = a9902stcr (dev, devaddr, cnt);
      break;

   case CON9903:
   case SERIAL9903:
   case PRINT9903:
      dat = a9903stcr (dev, devaddr, cnt);
      break;

   case SERIALCI401:
      dat = ci401stcr (dev, devaddr, cnt);
      break;

   case PRINTPAR:
      dat = printstcr (dev, devaddr, cnt);
      break;

   case VDT911:
      dat = v911stcr (dev, cnt);
      break;

   case FLOPPY:
      dat = fpystcr (dev, cnt);
      break;

   case CARDPUNCH:
      dat = punchstcr (dev, devaddr, cnt);
      break;

   case CARDREADER:
      dat = cardstcr (dev, devaddr, cnt);
      break;

   default: ;
   }
#ifdef DEBUGCRU
   fprintf (stderr, "DAT %04X\n", dat);
#endif
#ifdef DEBUGPRIVCRU
   if (devaddr >= PRVCRU)
      fprintf (stderr, "DAT %04X\n", dat);
#endif

   return (dat);
}

/***********************************************************************
* gettiline - Get TILINE memory.
***********************************************************************/

int
gettiline (uint32 addr, int bflg)
{
   Device *dev;
   int i;
   uint16 devaddr;

   devaddr = addr & 0xFFF0;

#ifdef ECCADDR
   if (devaddr == ECCADDR)
   {
      if (bflg)
      {
	 i = GETMEMB0 (addr);
      }
      else
      {
	 i = GETMEM0 (addr);
      }
#ifdef DEBUGECC
      fprintf (stderr, "gettiline: addr = %06X, bflg = %d\n", addr, bflg);
      fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n",
	       devaddr, addr & 0x000F);
      fprintf (stderr, "   val = %04X\n", i);
#endif
      return (i);
   }
#endif

#ifdef DEBUGTILINE
   fprintf (stderr, "gettiline: addr = %06X, bflg = %d\n", addr, bflg);
   fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n",
	    devaddr, addr & 0x000F);
#endif

   if ((i = finddev (devaddr)) < 0)
   {
#if defined(DEBUGTILINE)
      fprintf (stderr, "%d: No TILINE dev-get: devaddr = %04X\n",
	       instcount, devaddr);
#endif
      errcrudata |= ERRTIT;
      if (GET_MASK >= ERRINT)
	 geninterrupt (ERRINT);
      return (0);
   }

   dev = &devices[i];

   if (dev->type == SERIALCI403 || dev->type == PRINTCI403)
   {
      i = ci403getcmd (dev, addr);
#ifdef DEBUGCI403T
      fprintf (stderr, "gettiline: addr = %06X, bflg = %d\n", addr, bflg);
      fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n",
               devaddr, addr & 0x000F);
      fprintf (stderr, "   val = %04X\n", i);
#endif
      return (i & 0xFFFF);
   }

   if (bflg)
   {
      i = GETMEMB0 (addr);
   }
   else
   {
      i = GETMEM0 (addr);
   }

#ifdef DEBUGTAPE
   if (dev->type == TAPE)
   {
      fprintf (stderr, "gettiline: addr = %06X, bflg = %d\n", addr, bflg);
      fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n",
               devaddr, addr & 0x000F);
      fprintf (stderr, "   val = %04X\n", i);
   }
#endif

#ifdef DEBUGDSK
   if (dev->type == DISK)
   {
      fprintf (stderr, "%04X %06X ", curpc, curmpc);
      fprintf (stderr, "gettiline: addr = %06X, bflg = %d\n", addr, bflg);
      fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n",
               devaddr, addr & 0x000F);
      fprintf (stderr, "   val = %04X\n", i);
   }
#endif

#ifdef DEBUGTILINE1
   fprintf (stderr, "   val = %04X\n", i);
#endif

   return (i & 0xFFFF);
}

/***********************************************************************
* puttiline - Put TILINE memory.
***********************************************************************/

int
puttiline (uint32 addr, uint16 v, int bflg)
{
   Device *dev;
   int i;
   int intlevel;
   int start;
   int attn;
   int bit0;
   int dcrndx;
   uint16 devaddr;

   devaddr = addr & 0xFFF0;
   dcrndx = addr & 0x000F;

#ifdef ECCADDR
   if (devaddr == ECCADDR)
   {
      if (bflg)
      {
	 PUTMEMB0 (addr, v);
      }
      else
      {
	 PUTMEM0 (addr, v);
      }
#ifdef DEBUGECC
      fprintf (stderr, "puttiline: addr = %06X, v = %04X bflg = %d\n",
	       addr, v, bflg);
      fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n", devaddr, dcrndx);
#endif
      return (0);
   }
#endif

#ifdef DEBUGTILINE
   fprintf (stderr, "puttiline: addr = %06X, v = %04X bflg = %d\n",
	 addr, v, bflg);
   fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n", devaddr, dcrndx);
#endif

   if ((i = finddev (devaddr)) < 0)
   {
#if defined(DEBUGTILINE)
      fprintf (stderr, "%d: No TILINE dev-put: devaddr = %04X\n",
	       instcount, devaddr);
#endif
      errcrudata |= ERRTIT;
      if (GET_MASK >= ERRINT)
	 geninterrupt (ERRINT);
      return (0);
   }

   dev = &devices[i];

   if (dev->type == SERIALCI403 || dev->type == PRINTCI403)
   {
      ci403putcmd (dev, addr, v);
#ifdef DEBUGCI403T
      fprintf (stderr, "puttiline: addr = %06X, v = %04X bflg = %d\n",
	       addr, v, bflg);
      fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n", devaddr, dcrndx);
#endif
      return (0);
   }

   attn = FALSE;
   start = FALSE;
   bit0 = FALSE;
   intlevel = 0;

   if (bflg)
   {
      if (dcrndx == 1)
      {
	 if (v & 0xF)
	    attn = TRUE;
      }
      else if (dcrndx == 14)
      {
	 start = TRUE;
	 bit0 = v & 0x80 ? TRUE : FALSE;
      }
      PUTMEMB0 (addr, v);
   }
   else
   {
      if (dcrndx == 0)
      {
	 if (v & 0xF)
	    attn = TRUE;
      }
      else if (dcrndx == 14)
      {
	 start = TRUE;
	 bit0 = v & 0x8000 ? TRUE : FALSE;
      }
      PUTMEM0 (addr, v);
   }

#ifdef DEBUGTAPE
   if (dev->type == TAPE)
   {
      fprintf (stderr, "puttiline: addr = %06X, v = %04X bflg = %d\n",
	    addr, v, bflg);
      fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n", devaddr, dcrndx);
   }
#endif

   if (dev->type == DISK)
   {
#ifdef DEBUGDSK
      fprintf (stderr, "%04X %06X ", curpc, curmpc);
      fprintf (stderr, "puttiline: addr = %06X, v = %04X bflg = %d\n",
	 addr, v, bflg);
      fprintf (stderr, "   devaddr = %04X, dcrndx = %d\n", devaddr, dcrndx);
      fprintf (stderr, "   start = %s, bit0 = %s, attn = %s\n", 
	       start ? "TRUE" : "FALSE",
	       bit0 ? "TRUE" : "FALSE",
	       attn ? "TRUE" : "FALSE");
#endif
      /*
      ** DNOS expects the status word to be zero for any put into the TILINE
      ** space.
      */
      if (((addr & 0xF) > 1) && !start)
	 PUTMEM0 ((addr & 0x1FFFF0), 0);
   }

   if (start && !bit0)
   {
      switch (dev->type)
      {
      case TAPE:
	 intlevel = tapedocmd (dev);
	 break;

      case DISK:
	 intlevel = dskdocmd (dev);
	 break;

      default: ;
      }

      if (intlevel > 0)
	 gendevinterrupt (dev);
   }
   else if (attn)
   {
      gendevinterrupt (dev);
   }

   return (0);
}

/***********************************************************************
* bootdev - Boot from device.
***********************************************************************/

int
bootdev (char *device)
{
   Device *dev;
   char *bp;
   int rc;
   int i;

   run = FALSE;
   rc = 0;

   if (model == 12)
      mapcrudata = (mapcrudata & 0xE3FF); /* select ROM page 0 */

   while (*device && isspace (*device)) device++;
   for (bp = device; *bp; bp++)
   {
      if (islower (*bp))
	 *bp = toupper (*bp);
   }

   if (!strncmp (device, "CS", 2))
   {
      int unit = atoi (&device[2]);
      int base = ((unit % 2) == 0) ? unit : unit - 1;

      for (i = 0; i < devcnt; i++)
      {
	 dev = &devices[i];
         if ((dev->switches & SWEMUTERM) && (dev->emutype == EMU733) &&
	     (dev->info.terminfo.terminal.asrinfo.baseunit == base))
	 {
	    return (a733boot (dev, device, unit % 2));
	 }
      }
   }

   else for (i = 0; i < devcnt; i++)
   {
      if (!strcmp (devices[i].name, device))
	 break;
   }

   if (i == devcnt)
   {
      sprintf (view[0], "Device %s: Is not attached", device);
      return (-1);
   }

   dev = &devices[i];

   if (dev->infd == DEV_NULL)
   {
      sprintf (view[0], "Device %s: Is attached to NULL", device);
      return (-1);
   }

   SET_MASK (0);

   switch (dev->type)
   {
   case CARDREADER: /* Boot from Card Reader */
      rc = cardboot (dev);
      break;

   case DISK: /* Boot from TILINE disk */
      rc = dskboot (dev);
      break;

   case FLOPPY: /* Boot from FLOPPY */
      rc = fpyboot (dev);
      break;

   case TAPE: /* Boot from TILINE tape */
      rc = tapeboot (dev);
      break;

   default:
      sprintf (view[0], "Device %s: Boot is not supported on %s",
	       device, devtype[devmap[dev->type]]);
      rc = -1;
   }

   return (rc);
}
