/***********************************************************************
*
* simloader.c - Binary loader for the TI 990 simulator.
*
* Changes:
*   05/29/03   DGP   Original.
*   07/02/03   DGP   Added compressed binary support.
*   12/30/03   DGP   Moved loader code to loadrec function.
*   04/12/04   DGP   Changed to allow non-80 byte records.
*   04/14/04   DGP   Ignore extended object tags; Add checksum support.
*   08/03/11   DGP   Generate error if program is too big.
*   02/12/14   DGP   Fixed compressed (binary) load.
*   03/06/14   DGP   Added no checksum tag.
*   07/02/16   DGP   Ignore PGMIDT_TAG 'I'.
*   11/21/16   DGP   Support debugging data (-g output).
*
***********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <signal.h>
#include <ctype.h>
#include <errno.h>

#include "simdef.h"

extern int run;
extern uint32 memlen;
extern uint16 pcreg;
extern uint8 memory[SYSMEMSIZE];
extern char view[MAXVIEW][MAXVIEWLEN+1];

static int pgmlen;
static int idtloaded = FALSE;
static uint16 idtorigin;
static char idtname[IDTLEN+2];

int
loadrec (FILE *lfd, uint8 *inbuf, int loadaddr, int *curraddr, int binarymode)
{
   static int wordtaglen = WORDTAGLEN;
   uint8 *op = (uint8 *)inbuf;
   int i, j;
   int16 cksum;

   if (*op == EOFSYM)
   {
      wordtaglen = WORDTAGLEN;
      return (EOF);
   }

   cksum = 0;
   for (i = 0; i < 80; i++)
   {
      int tmp;
      uint16 wdata;
      uint8 otag;
      uint8 item[40];

      otag = *op++;
      if (binarymode)
      {
	 wdata = (*op << 8) | *(op+1);
      }
      else
      {
	 cksum += otag;
	 memcpy (item, op, 4);
	 item[4] = '\0';
	 if (otag != CKSUM_TAG)
	    for (j = 0; j < 4; j++) cksum += item[j];
	 sscanf ((char *)item, "%4X", &tmp);
	 wdata = tmp & 0xFFFF;
      }
      wdata &= 0xFFFF;
#ifdef DEBUGLOADER
      if (lfd != NULL)
      {
	 fprintf (lfd, "loadaddr = %04X, curraddr = %04X, otag = %c\n",
		  loadaddr, *curraddr,
		  otag == 0x01 ? '0' : otag);
	 fprintf (lfd, "data = %04X\n", wdata & 0xFFFF);
      }
#endif

      switch (otag)
      {
      case BINIDT_TAG: /* Binary IDT */
	 wordtaglen = BINWORDTAGLEN;
	 wdata = (*op << 8) | *(op+1);
#ifdef DEBUGLOADER
	 if (lfd != NULL)
	    fprintf (lfd, "Binary mode: wdata = %04x\n", wdata);
#endif
      case IDT_TAG:
	 pgmlen = wdata;
	 wdata += loadaddr;
	 if (wdata >= memlen)
	 {
	    sprintf (view[0], "simloader: Program too large for memory");
	    return (-21);
	 }
	 op += wordtaglen-1;
	 strncpy (idtname, (char *)op, IDTLEN);
	 idtorigin = loadaddr;
	 if (!binarymode)
	    for (j = 0; j < IDTLEN; j++) cksum += *op++;
	 else
	    op += IDTLEN;
	 break;

      case PGMIDT_TAG:
	 wdata += loadaddr;
	 op += wordtaglen-1;
	 strncpy ((char *)item, (char *)op, IDTLEN);
	 addidt ((char *)item, wdata);
	 idtloaded = TRUE;
	 if (!binarymode)
	    for (j = 0; j < IDTLEN; j++) cksum += *op++;
	 else
	    op += IDTLEN;
	 break;

      case RELORG_TAG:
	 wdata += loadaddr;
      case ABSORG_TAG:
	 *curraddr = wdata;
	 op += wordtaglen-1;
	 break;

      case RELDATA_TAG:
	 wdata += loadaddr;
      case ABSDATA_TAG:
	 PUTMEM0 (*curraddr, wdata);
	 *curraddr += 2;
	 op += wordtaglen-1;
	 break;

      case RELENTRY_TAG:
	 wdata += loadaddr;
      case ABSENTRY_TAG:
	 pcreg = wdata;
	 run = TRUE;
	 op += wordtaglen-1;
	 break;

      case CKSUM_TAG:
#ifdef DEBUGLOADER
         if (lfd != NULL)
	    fprintf (lfd, "cksum = %04X\n", cksum);
#endif
         cksum += (int16)wdata;
#ifdef DEBUGLOADER
         if (lfd != NULL)
	    fprintf (lfd, "cksum+wdata = %04X\n", cksum);
#endif
	 if (cksum != 0)
	 {
	    sprintf (view[0], "simloader: Checksum error");
	    return (-20);
	 }
      case NOCKSUM_TAG:
	 op += wordtaglen-1;
         break;

      case RELEXTRN_TAG:
      case RELGLOBAL_TAG:
      case RELSYMBOL_TAG:
      case RELSREF_TAG:
	 wdata += loadaddr;
      case ABSEXTRN_TAG:
      case ABSGLOBAL_TAG:
      case ABSSYMBOL_TAG:
      case LOAD_TAG:
      case ABSSREF_TAG:
	 op += wordtaglen-1;
	 if (!idtloaded)
	 {
	    addidt (idtname, idtorigin);
	    idtloaded = TRUE;
	 }
	 strncpy ((char *)item, (char *)op, SYMLEN);
	 addsym ((char *)item, SYMLEN, wdata);
	 if (!binarymode)
	    for (j = 0; j < SYMLEN; j++) cksum += *op++;
	 break;

      case LRELEXTRN_TAG:
      case LRELGLOBAL_TAG:
      case LRELSYMBOL_TAG:
      case LRELSREF_TAG:
	 wdata += loadaddr;
      case LABSEXTRN_TAG:
      case LABSGLOBAL_TAG:
      case LABSSYMBOL_TAG:
      case LLOAD_TAG:
      case LABSSREF_TAG:
	 op += wordtaglen-1;
	 if (!idtloaded)
	 {
	    addidt (idtname, idtorigin);
	    idtloaded = TRUE;
	 }
	 strncpy ((char *)item, (char *)op, MAXSYMLEN);
	 addsym ((char *)item, MAXSYMLEN, wdata);
	 if (!binarymode)
	    for (j = 0; j < MAXSYMLEN; j++) cksum += *op++;
	 break;

      case EXTNDX_TAG:
	 op += wordtaglen-1;
	 PUTMEM0 (*curraddr, wdata);
	 *curraddr += 2;
	 if (!binarymode)
	    for (j = 0; j < 4; j++) cksum += *op++;
         break;

      case EOR_TAG:
         i = 81;
	 break;

      default: 
	 sprintf (view[0], "simloader: Illegal object tag: %c(%02X), %9.9s",
	 	  otag, otag, &inbuf[71]);
	 return (-22);
      }
   }
   return (0);
}

int
simloader (char *file, int loadpt)
{
   FILE *fd;
   FILE *lfd = NULL;
   int status;
   int done;
   int binarymode = FALSE;
   int loadaddr = LOADADDRESS;
   int curraddr = LOADADDRESS;
   uint8 inbuf[82];

   while (isspace (*file)) file++;
#ifdef DEBUGLOADER
   lfd = fopen ("fileload.log", "w");
   fprintf (lfd, "simloader: file = '%s', loadpd = %d(%04X)\n",
	    file, loadpt, loadpt);
#endif

   if (loadpt > 0)
   {
      loadaddr = loadpt;
      curraddr = loadpt;
   }

   if ((fd = fopen (file, "rb")) == NULL)
   {
      sprintf (view[0], "simloader: open failed: %s",
	       strerror (ERRNO));
      snprintf (view[1], MAXVIEWLEN, "filename: %s", file);
      run = FALSE;
#ifdef DEBUGLOADER
      fprintf (lfd, "error: %s\n", view[0]);
      fprintf (lfd, "       %s\n", view[1]);
      fclose (lfd);
#endif
      return (-1);
   }

   status = fgetc (fd);
   if (status == BINIDT_TAG)
      binarymode = TRUE;
   ungetc (status,fd);

   done = FALSE;
   while (!done)
   {
      if (binarymode)
      {
         if ((fread (inbuf, 1, 81, fd) != 81) || feof (fd))
	 {
	    done = TRUE;
	    break;
	 }
      }
      else
      {
	 if (fgets ((char *)inbuf, 82, fd) == NULL)
	 {
	    done = TRUE;
	    break;
	 }
      }

#ifdef DUMPLOADREC
      fprintf (lfd, "Record:\n");
      HEXDUMP (lfd, inbuf, binarymode ? 81 : strlen ((char *)inbuf), 0);
#endif
      if ((status = loadrec (lfd, inbuf, loadaddr, &curraddr, binarymode)) != 0)
      {
	 if (status == EOF) break;
	 snprintf (view[1], MAXVIEWLEN, "filename: %s", file);

	 run = FALSE;
#ifdef DEBUGLOADER
	 fprintf (lfd, "error: %s\n", view[0]);
	 fprintf (lfd, "       %s\n", view[1]);
	 fclose (lfd);
#endif
	 fclose (fd);
	 return (-1);
      }
   }

#ifdef DUMPPROG
   HEXDUMP (stderr, &memory[loadaddr], pgmlen, loadaddr);
   run = FALSE;
#endif

#ifdef DEBUGLOADER
   fclose (lfd);
   dumpsyms();
#endif
   fclose (fd);

   return (0);
}
