/***********************************************************************
*
* simops.c - Simulate instruction operations for the TI 990 computer.
*
* Changes:
*   05/29/03   DGP   Original.
*   06/22/03   DGP   Fixed XOP REG function.
*   07/08/03   DGP   Fixed status on Subtract and indexing with negative
*                    values.
*   07/09/03   DGP   Forced PC and WP to be on even boundry.
*   07/13/03   DGP   Forced word and address access to even boundry.
*   08/07/03   DGP   Fixed carry on Add and Add Byte.
*                    Fixed MPY and DIV (had signed problems).
*                    Fixed B Rn (branch to register).
*   08/12/03   DGP   Added Map file instructions and priv checks.
*   11/26/03   DGP   Fixup OVER & CARRY status checking.
*   12/10/03   DGP   Streamline TILINE & ROM access with 20 bits.
*   01/26/04   DGP   Added memory size option.
*   04/20/04   DGP   Changed SLA to detect sign change during shift.
*   05/06/04   DGP   Fix LREX to run in 990/10 mode.
*   06/14/04   DGP   Added floating point instructions.
*   06/16/04   DGP   Added extended /12 instructions.
*   07/13/04   DGP   Added /12 memory protection checks
*   03/30/05   DGP   Fixed XOP for Mapping.
*   01/09/07   DGP   Added EMD code.
*   11/07/08   DGP   Added Instruction tracing and disassembler.
*   11/11/08   DGP   Added LCS support.
*   11/12/08   DGP   Extended /12 rom support.
*   12/02/08   DGP   Fix RSET to not clear mapping if it's enabled.
*   02/18/09   DGP   Added breakpoint CRU support.
*   01/18/11   DGP   Added watch point break.
*   02/04/12   DGP   Added fakeprefetch for /12.
*   03/07/12   DGP   Fixup /12 ROM self test issues.
*   10/14/13   DGP   Added CPU options.
*   09/02/14   DGP   Fixed LDS/X sequence bug.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>

#include "simdef.h"

extern uint16 pcreg;	/* The program PC */
extern uint16 curpc;	/* The program PC */
extern uint16 opcreg;	/* The old program PC */
extern uint32 mpcreg;	/* Mapped PC */
extern uint32 curmpc;	/* Mapped PC */
extern uint16 statreg;	/* The program status register */
extern uint16 wpreg;	/* The program Workspace Pointer */
extern uint16 lights;	/* The panel lights */
extern uint16 ldspc;	/* The address of the LDS inst */
extern uint16 lddpc;	/* The address of the LDD inst */
extern uint32 breakaddr;/* The break point address */
extern uint32 watchaddr;/* The watch point address */
extern uint32 ldsmpc;	/* The mapped address of the LDS inst */
extern uint32 lddmpc;	/* The mapped address of the LDD inst */
extern long pancount;	/* Instructions executed for panel */
extern unsigned long instcount;
extern unsigned long tracestart;

extern int run;
extern int idle;
extern int runled;
extern int idleled;
extern int model;
extern int breakflag;
extern int watchflag;
extern int enablepanel;
extern int mapenabled;
extern int intsenabled;
extern int traceenable;
extern int traceregs;
extern int tracelim[2];
extern int deferint;
extern int traceit;
extern int windowlen;
extern int bkptenabled;
extern int forceparity;
extern int rom5cru;
extern int cpuoptions;
extern int maxdevcnt;
extern int showstatus;
extern uint16 bkptcrudata;
extern uint16 mapcrudata;
extern uint16 mapolddata;
extern uint16 curmapindex;
extern uint16 errcrudata;
extern uint16 curinst;
extern uint32 memlen;
extern uint32 maplatch;
extern long delayclock;
extern char view[MAXVIEW][MAXVIEWLEN+1];
extern FILE *tracefd;
extern uint32 tracemem[16];
extern uint32 errtrace[16];
extern int errindex;
extern int trcindex;
extern int traceline;
extern int tcenable;
extern int tccount;
extern int fakeprefetch;

extern Device devices[MAXDEVICES];
extern uint8 memory[SYSMEMSIZE];
extern MapFile mapfile[MAPSIZE];

static void runinst (uint16, int);
extern int use_LDD_map;
extern int use_LDS_map;
static int use_map_cnt = 0;
static int tiline = FALSE;
static int inhibitwrite = FALSE;

static int proc1op (uint16 inst);
static int proc2op (uint16 inst);
static int proc3op (uint16 inst);
static int proc4op (uint16 inst);
static int proc5op (uint16 inst);
static int proc6op (uint16 inst);
static int proc7op (uint16 inst);
static int proc8op (uint16 inst);
static int proc9op (uint16 inst);
static int proc10op (uint16 inst);

extern int trcidx;
extern int trcopr;
extern TraceInst trcbuf[TRCBUFMAX];

typedef struct {
   int (*proc) (uint16 inst);
} Inst_Proc;

Inst_Proc inst_proc[MAX_INST_TYPES] = {
   {proc1op},  {proc2op},  {proc3op},  {proc4op},  {proc5op},  {proc6op},
   {proc7op},  {proc8op},  {proc9op},  {proc10op}, {proc11op}, {proc12op},
   {proc13op}, {proc14op}, {proc15op}, {proc16op}, {proc17op}, {proc18op},
   {proc19op}, {proc20op}, {proc21op}
};

static char outbuf[128];
static int tracema = FALSE;

/***********************************************************************
* decodeinst - Decode instructions.
***********************************************************************/

int
decodeinst (uint16 inst)
{
   if (inst & 0xF000)
   {
      if (inst & 0xC000) return TYPE_1;
      if ((inst & 0xF800) == 0x3000) return TYPE_4;
      if ((inst & 0xF000) == 0x3000) return TYPE_9;
      if ((inst & 0xF000) == 0x2000) return TYPE_3;
      return TYPE_2;
   }
   else if (inst & 0x0F00)
   {
      if ((inst & 0x0F00) == 0x0F00) return TYPE_6;
      else if ((inst & 0x0F00) == 0xE00)
      {
	 if ((inst & 0x0FF0) == 0x0E00) return TYPE_15;
	 if ((inst & 0x0FC0) == 0x0E00) return TYPE_12;
	 return TYPE_6;
      }
      else if ((inst & 0x0F00) == 0x0D00) return TYPE_6;
      else if ((inst & 0x0F00) == 0x0C00)
      {
	 if ((inst & 0x0FFF) == 0x0C0F) return TYPE_7;
	 if ((inst & 0x0FFF) == 0x0C0E) return TYPE_7;
	 if ((inst & 0x0FFF) == 0x0C08) return TYPE_11;
	 if ((inst & 0x0FFE) == 0x0C0C) return TYPE_17;
	 if ((inst & 0x0FF8) == 0x0C08) return TYPE_14;
	 if ((inst & 0x0FF0) == 0x0C00) return TYPE_7;
	 if ((inst & 0x0FC0) == 0x0C00) return TYPE_16;
	 return TYPE_6;
      }
      else if ((inst & 0x0C00) == 0x0800) return TYPE_5;
      else if ((inst & 0x0C00) == 0x0400) return TYPE_6;
      else if ((inst & 0x0F00) == 0x0300)
      {
	 if (inst == 0x0300) return TYPE_8;
	 if ((inst & 0x0FF0) == 0x03F0) return TYPE_21;
	 if ((inst & 0x0FE0) == 0x0320) return TYPE_10;
	 return TYPE_7;
      }
      else if ((inst & 0x0F00) == 0x0200) return TYPE_8;
      else if ((inst & 0x0F00) == 0x0100) return TYPE_6;
   }
   else if (inst & 0x00F0)
   {
      if ((inst & 0x00F0) == 0x00B0) return TYPE_8;
      else if ((inst & 0x00F0) == 0x0070) return TYPE_18;
      else if ((inst & 0x0040) == 0x0040) return TYPE_12;
      else if ((inst & 0x00F0) == 0x0020)
      {
	 if ((inst & 0x00FF) == 0x0021) return TYPE_20;
	 if ((inst & 0x00FF) == 0x0022) return TYPE_20;
	 if ((inst & 0x00FF) == 0x002B) return TYPE_19;
	 if ((inst & 0x00FC) == 0x002C) return TYPE_7;
	 if ((inst & 0x00FC) == 0x0028) return TYPE_11;
	 return TYPE_11;
      }
      else if ((inst & 0x00F0) == 0x0010)
      {
	 if ((inst & 0x00FE) == 0x001E) return TYPE_11;
	 return TYPE_13;
      }
      return TYPE_18;
   }
   return (TYPE_ILL);
}


/***********************************************************************
* tracememory - Trace memory data
***********************************************************************/

void
tracememory (uint32 code, uint32 ma)
{
   if (model < 12) return;

   errtrace[errindex] = code | ((ma & 0x1FFFFE) >> 1);
   errindex++;
   if (errindex == 16) errindex = 0;
}

/***********************************************************************
* mapaddr - Map address to a 22 bit physical address.
***********************************************************************/

uint32
mapaddr (uint16 pa, uint16 mf, uint16 prot, uint32 accesstype, uint32 tlaccess)
{
   uint32 ma;
   uint16 latchmask;
   uint16 limit;
   uint16 base;
#if defined(DEBUGMAPPER) || defined(DEBUGSEGFAULT)
   int limreg;
#endif

   /*
   ** Check if hardware breakpoint has been encountered
   */

   if (bkptenabled && (model == 12) && ((pa & 0xFFFE) == bkptcrudata) &&
      (((mapcrudata >> 5) & 1) == mf))
   {
      int dobkpt = FALSE;
      int mask = (mapcrudata >> 6) & 3;

      if (mask == 3)
         dobkpt = TRUE;
      else if (prot == MEM_WRITE && mask == 2)
         dobkpt = TRUE;
      else if (prot == MEM_EXEC && mask == 1)
         dobkpt = TRUE;
      else if (prot == MEM_READ && mask == 0)
         dobkpt = TRUE;

      if (dobkpt)
      {
#ifdef DEBUGBKPT
         fprintf (stderr,
	"%04X: Breakpoint: pc = %04X(%06X), pa = %04X, mask = %d, prot = %d\n",
	          curpc, pcreg, mpcreg, pa, mask, prot);
#endif
	 bkptenabled = FALSE;
	 bkptcrudata = 0;
	 errcrudata |= ERRBKPT;
	 if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
      }
   }

   /*
   ** If not in mapping mode use directly
   */

   if (!mapenabled)
   {
      ma = pa;
      if (pa >= TILINESTART)
      {
	 ma += TPCSSTART;
	 if (pa < TILINEEND)
	 {
	    if (!(cpuoptions & OPT_NOTILINE))
	    {
	       tiline = TRUE;
	       accesstype |= tlaccess;
	    }
	    else
	    {
	       ma = pa;
	    }
	 }
	 else if (model == 12)
	 {
	    if (fakeprefetch)
	    {
	       ma += ((mapolddata >> 10) & 0x7) * 1024;
#ifdef DEBUGROM12
	       fprintf (stderr,
	       "mapaddr: pa = >%04X, mapOLDdata = >%04X, c = %x, ma = >%06X\n",
	             pa, mapolddata, (mapolddata >> 10) & 0x7, ma);
#endif
	    }
	    else
	    {
	       ma += ((mapcrudata >> 10) & 0x7) * 1024;
#ifdef DEBUGROM12
	       fprintf (stderr,
	       "mapaddr: pa = >%04X, mapcrudata = >%04X, c = %x, ma = >%06X\n",
	             pa, mapcrudata, (mapcrudata >> 10) & 0x7, ma);
#endif
	    }
	 }
      }
   }

   /*
   ** Mapping mode, perform any required mapping
   */

   else
   {
      if (mf >= MAPSIZE)
      {
	 inhibitwrite = TRUE;
	 errcrudata |= ERRMAP;
	 if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
#if defined(DEBUGERRINT) || defined(DEBUGMAPPER)
	 fprintf (stderr, "%04X: Map %d error pa = >%04X\n", curpc, mf, pa);
#endif
	 return (0xFFFFFFFF);
      }
      
      /*
      ** Check TILINE access
      */

      if ((mf == 0) && (pa >= TILINESTART))
      {
	 ma = pa + TPCSSTART;
	 if (pa < TILINEEND)
	 {
	    if (!(cpuoptions & OPT_NOTILINE))
	    {
	       tiline = TRUE;
	       accesstype |= tlaccess;
	    }
	    else
	    {
	       ma = pa;
	    }
	 }
	 else if (model == 12)
	 {
	    if (fakeprefetch)
	    {
	       ma += ((mapolddata >> 10) & 0x7) * 1024;
#ifdef DEBUGROM12
	       fprintf (stderr,
	       "mapaddr: pa = >%04X, mapOLDdata = >%04X, c = %x, ma = >%06X\n",
	             pa, mapolddata, (mapolddata >> 10) & 0x7, ma);
#endif
	    }
	    else
	    {
	       ma += ((mapcrudata >> 10) & 0x7) * 1024;
#ifdef DEBUGROM12
	       fprintf (stderr,
	       "mapaddr: pa = >%04X, mapcrudata = >%04X, c = %x, ma = >%06X\n",
	             pa, mapcrudata, (mapcrudata >> 10) & 0x7, ma);
#endif
	    }
	 }
      }

      /*
      ** Not TILINE map address
      */

      else
      {
	 if ((pa & 0xFFE0) <= (~mapfile[mf].l1 & 0xFFE0))
	 {
#if defined(DEBUGMAPPER) || defined(DEBUGSEGFAULT)
	    limreg = 1;
#endif
	    limit = mapfile[mf].l1;
	    base = mapfile[mf].b1;
	    latchmask = MAP_LATCH1;
	 }

	 else if ((pa & 0xFFE0) <= (~mapfile[mf].l2 & 0xFFE0))
	 {
#if defined(DEBUGMAPPER) || defined(DEBUGSEGFAULT)
	    limreg = 2;
#endif
	    limit = mapfile[mf].l2;
	    base = mapfile[mf].b2;
	    latchmask = MAP_LATCH2;
	 }

	 else if ((pa & 0xFFE0) <= (~mapfile[mf].l3 & 0xFFE0))
	 {
#if defined(DEBUGMAPPER) || defined(DEBUGSEGFAULT)
	    limreg = 3;
#endif
	    limit = mapfile[mf].l3;
	    base = mapfile[mf].b3;
	    latchmask = MAP_LATCH3;
	 }

	 else
	 {
#if defined(DEBUGERRINT) || defined(DEBUGMAPPER)
	    int wr = mapaddr (wpreg, GET_MAP, MEM_READ, 0, 0);
	    int pr = mapaddr (curpc, GET_MAP, MEM_READ, 0, 0);
	    fprintf (stderr,
		     "%04X: Map %d error >L3 pa = >%04X pa&>FFE0 = >%04X\n",
		     curpc, mf, pa, pa & 0xFFE0);
	    fprintf (stderr,
	       "   pcreg = >%04X(%06X), wpreg = >%04X(%06X), statreg = >%04X\n",
		     curpc, pr, wpreg, wr, statreg);
	    fprintf (stderr, "Code:\n");
	    HEXDUMP (stderr, &memory[pr-16], 32, pr-16);
	    fprintf (stderr, "WS:\n");
	    HEXDUMP (stderr, &memory[wr], 32, wr);
	    fprintf (stderr,
	      "   mapfile[%d].l1 = >%04X ~l1 = >%04X, mapfile[%d].b1 = >%04X\n",
		  mf, mapfile[mf].l1, (~mapfile[mf].l1 & 0xFFE0),
		  mf, mapfile[mf].b1);
	    fprintf (stderr,
	      "   mapfile[%d].l2 = >%04X ~l2 = >%04X, mapfile[%d].b2 = >%04X\n",
		  mf, mapfile[mf].l2, (~mapfile[mf].l2 & 0xFFE0),
		  mf, mapfile[mf].b2);
	    fprintf (stderr,
	      "   mapfile[%d].l3 = >%04X ~l3 = >%04X, mapfile[%d].b3 = >%04X\n",
		  mf, mapfile[mf].l3, (~mapfile[mf].l3 & 0xFFE0),
		  mf, mapfile[mf].b3);
#endif
	    inhibitwrite = TRUE;
	    if (!maplatch)
	    {
	       maplatch = pa;
#ifdef DEBUGMAPPER
	       fprintf (stderr, "   maplatch = >%04X\n", maplatch);
#endif
	    }
	    tracememory (accesstype | TRACE_MAPERR, pa);
	    errcrudata |= ERRMAP;
	    if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
	    return (0xFFFFFFFF);
	 }

	 ma = base & 0xFFFF;
	 ma = (((ma << 5) + pa) & 0x1FFFFF);
#ifdef TRACEMAPPER
	 if (traceit && tracema)
	 {
	    fprintf (tracefd, "mf = %d: pa = %04X -> ma = %06X\n", mf, pa, ma);
	 }
#endif
#ifdef DEBUGMAPPER
	 if (traceit)
	 {
	    fprintf (tracefd,
      "%04X: Map %d <L%d pa = >%04X, base = >%04X, ~limit = >%04X, ma = %06X\n",
		     curpc, mf, limreg, pa, base, ~limit & 0xFFE0, ma);
	 }
	 else
	 {
	    fprintf (stderr,
      "%04X: Map %d <L%d pa = >%04X, base = >%04X, ~limit = >%04X, ma = %06X\n",
		     curpc, mf, limreg, pa, base, ~limit & 0xFFE0, ma);
	 }
#endif
	 if (!maplatch && ((mf == 2) || (mapcrudata & latchmask)))
	 {
	    maplatch = ma;
#ifdef DEBUGMAPPER
	       fprintf (stderr,
		    "%04X:   latchmask = >%04X, maplatch = >%06X, pa = >%04X\n",
			curpc, latchmask, maplatch, pa);
#endif
	 }

	 /*
	 ** Apply segment protection on 990/12.
	 */

	 if (model == 12 && !(cpuoptions & OPT_NOSEGPROT) && IS_PROT)
	 {
	    int segfault = FALSE;

	    switch (limit & 0x3)
	    {
	       case 0: /* Read, Write, Execute */
		  break;
	       case 1: /* Read, Execute */
		  if (prot == MEM_WRITE)
		     segfault = TRUE;
		  break;
	       case 2: /* Read, Write */
		  if (prot == MEM_EXEC)
		     segfault = TRUE;
		  break;
	       case 3: /* Read */
		  if (prot == MEM_EXEC || prot == MEM_WRITE)
		     segfault = TRUE;
		  break;
	       default: ;
	    }
	    if (segfault)
	    {
#if defined(DEBUGERRINT) || defined(DEBUGSEGFAULT)
	       fprintf (stderr,
	      "%04X: %06X: ProtViolation Map %d L%d rprot = >%X lprot = >%X\n",
			curpc, curmpc, mf, limreg, prot, limit & 0x3);
	       fprintf (stderr, "   pa = >%X, ma = >%X\n", pa, ma);
#endif
	       inhibitwrite = TRUE;
	       if (!maplatch)
	       {
		  maplatch = ma;
#ifdef DEBUGSEGFAULT
	       fprintf (stderr, "   maplatch = >%06X\n", ma);
#endif
	       }
	       errcrudata |= ((prot == MEM_EXEC) ? ERREXEC : ERRWRIT);
	       tracememory (accesstype | ((prot == MEM_EXEC) ?
			    TRACE_EXECVIO : TRACE_WRITVIO), ma);
	       if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
	       return (0xFFFFFFFF);
	    }
	 }

	 if (ma >= memlen)
	 {
	    /*
	    ** A true memory exception
	    */

#if defined(DEBUGERRINT) || defined(DEBUGMAPPER)
	    fprintf (stderr,
		     "%04X: >memlen Map %d <L%d pa = >%04X, base = >%04X,"
		     "~limit = >%04X, ma = %06X\n",
		     curpc, mf, limreg, pa, base, ~limit & 0xFFE0, ma);
	    fprintf (stderr, "   stat = >%04X\n", statreg);
#endif
	    inhibitwrite = TRUE;
	    if (!maplatch)
	    {
	       maplatch = ma;
#ifdef DEBUGMAPPER
	       fprintf (stderr, "   maplatch = >%06X\n", ma);
#endif
	    }
	    tracememory (accesstype | TRACE_TLTIME, ma);
	    errcrudata |= ERRTIT;
	    if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
	    return (0xFFFFFFFF);
	 }
      }

   }

   tracememory (accesstype, ma);
   if (watchflag && (ma == watchaddr))
   {
      if (prot == MEM_WRITE) // && !(accesstype & TRACE_WSACCESS)) 
      {
	 run = FALSE;
	 runled = FALSE;
	 showstatus = TRUE;
      }
   }
   return (ma);
}

/***********************************************************************
* getinst - Get an Instruction.
***********************************************************************/

uint16
getinst (int bkpt)
{
   uint32 ma;

   if (model < 10)
   {
      ma = pcreg;
      if (model == 5 && ma >= TILINESTART)
      {
	 ma += TPCSSTART;
	 if (pcreg < TILINEEND)
	 {
	    if (rom5cru)
	    {
	       ma = (pcreg - TILINESTART) + 0x200000;
#ifdef DEBUGROM51
	       fprintf (stderr, "getinst: pcreg = >%04X, ma = >%06X\n",
			pcreg, ma);
#endif
	    }
	    else if (cpuoptions & OPT_NOTILINE)
	    {
	       ma = pcreg;
	    }
	 }
      }
   }
   else
   {
      int oldbkptenabled = bkptenabled;

      if (use_LDD_map || use_LDS_map) curmapindex = 2;
      else curmapindex = GET_MAP;
      if (!bkpt) bkptenabled = FALSE;
      ma = mapaddr (pcreg, GET_MAP, MEM_EXEC, TRACE_INSTFETCH, 0);
      bkptenabled = oldbkptenabled;
      if (bkpt && fakeprefetch)
         fakeprefetch = FALSE;
      if (ma == 0xFFFFFFFF) return (0xFFFF);
   }
   mpcreg = ma;

   return (GETMEM0 (ma));
}

/***********************************************************************
* getreg - Get a register value.
***********************************************************************/

uint16
getreg (uint16 r)
{
   uint16 pa;
   uint32 ma;

   pa = wpreg + (r * 2);
   if (model < 10)
   {
      ma = pa;
   }
   else
   {
      ma = mapaddr (pa, GET_MAP, MEM_READ, TRACE_WSACCESS, 0);
      if (ma == 0xFFFFFFFF) return (0xFFFF);
   }

   return (GETMEM0 (ma));
}

/***********************************************************************
* putreg - Put a value into a register.
***********************************************************************/

void
putreg (uint16 r, uint16 v)
{
   uint16 pa;
   uint32 ma;

   pa = wpreg + (r * 2);
   if (model < 10)
   {
      ma = pa;
   }
   else
   {
      ma = mapaddr (pa, GET_MAP, MEM_READ, TRACE_WSACCESS | TRACE_WSRW, 0);
   }

   if (!inhibitwrite)
   {
      PUTMEM0 (ma, v);
   }
}

/***********************************************************************
* getmem - Get a word from memory.
***********************************************************************/

uint16
getmem (uint16 pa, int srcdst)
{
   int tr;
   int mr;
   uint32 ma;
   uint16 mf;

   tiline = FALSE;
   if (forceparity && (GET_MASK >= ERRINT))
   {
      errcrudata |= ERRTIP;
      geninterrupt (ERRINT);
   }

   pa &= 0xFFFE;
   if (model < 10)
   {
      ma = pa;
      if (model == 5 && pa >= TILINESTART)
      {
	 ma += TPCSSTART;
	 if (pa < TILINEEND)
	 {
	    if (rom5cru)
	    {
	       ma = (pa - TILINESTART) + 0x200000;
#ifdef DEBUGROM51
	       fprintf (stderr, "getmem: pa = >%04X, ma = >%06X\n",
			pa, ma);
#endif
	    }
	    else if (!(cpuoptions & OPT_NOTILINE))
	    {
	       tiline = TRUE;
	    }
	    else
	    {
	       ma = pa;
	    }
	 }
      }
   }
   else
   {
      mr = 0;
      if (srcdst >= 0)
      {
	 if (srcdst) mr = use_LDS_map;
	 else mr = use_LDD_map;
      }
      if (mr) mf = 2;
      else mf = GET_MAP;
      if (srcdst == EXEC)
      {
	 if (use_LDS_map) mf = 2;
	 ma = mapaddr (pa, mf, MEM_EXEC, TRACE_INSTFETCH, 0);
      }
      else
      {
	 ma = mapaddr (pa, mf, MEM_READ, TRACE_TLACCESS, 0);
      }
      if (ma == 0xFFFFFFFF) return (0xFFFF);
   }

   if (tiline)
   {
      tr = gettiline (ma, FALSE);
      return (tr & 0xFFFF);
   }

   return (GETMEM0 (ma));
}

/***********************************************************************
* getmemb - Get a byte from memory.
***********************************************************************/

uint8
getmemb (uint16 pa, int srcdst)
{
   int tr;
   int mr;
   uint32 ma;
   uint16 mf;

   tiline = FALSE;
   if (forceparity && (GET_MASK >= ERRINT))
   {
      errcrudata |= ERRTIP;
      geninterrupt (ERRINT);
   }

   if (model < 10)
   {
      ma = pa;
      if (model == 5 && pa >= TILINESTART)
      {
	 ma += TPCSSTART;
	 if (pa < TILINEEND)
	 {
	    if (rom5cru)
	    {
	       ma = (pa - TILINESTART) + 0x200000;
#ifdef DEBUGROM51
	       fprintf (stderr, "getmemb: pa = >%04X, ma = >%06X\n",
			pa, ma);
#endif
	    }
	    else if (!(cpuoptions & OPT_NOTILINE))
	    {
	       tiline = TRUE;
	    }
	    else
	    {
	       ma = pa;
	    }
	 }
      }
   }
   else
   {
      mr = 0;
      if (srcdst >= 0)
      {
	 if (srcdst) mr = use_LDS_map;
	 else mr = use_LDD_map;
      }
      if (mr) mf = 2;
      else mf = GET_MAP;
      ma = mapaddr (pa, mf, MEM_READ, TRACE_TLACCESS, 0);
      if (ma == 0xFFFFFFFF) return (0xFF);
   }

   if (tiline)
   {
      tr = gettiline (ma, TRUE);
      return (tr & 0xFF);
   }

   return (GETMEMB0 (ma));
}

/***********************************************************************
* putmem - Put a word in memory.
***********************************************************************/

void
putmem (uint16 pa, uint16 v, int srcdst)
{
   int mr;
   uint32 ma;
   uint16 mf;

   tiline = FALSE;
   pa &= 0xFFFE;
   if (model < 10)
   {
      ma = pa;
      if (model == 5 && pa >= TILINESTART)
      {
	 ma += TPCSSTART;
	 if (pa < TILINEEND)
	 {
	    if (rom5cru)
	    {
	       ma = (pa - TILINESTART) + 0x200000;
#ifdef DEBUGROM51
	       fprintf (stderr, "putmem: pa = >%04X, ma = >%06X\n",
			pa, ma);
#endif
	    }
	    else if (!(cpuoptions & OPT_NOTILINE))
	    {
	       tiline = TRUE;
	    }
	    else
	    {
	       ma = pa;
	    }
	 }
      }

      if (tiline)
      {
	 puttiline (ma, v, FALSE);
      }
      else if (pa < memlen)
      {
	 PUTMEM0 (ma, v);
      }
   }
   else
   {
      mr = 0;
      if (srcdst >= 0)
      {
	 if (srcdst) mr = use_LDS_map;
	 else mr = use_LDD_map;
      }
      if (mr) mf = 2;
      else mf = GET_MAP;

      ma = mapaddr (pa & 0xFFFE, mf, MEM_WRITE, TRACE_TLACCESS, TRACE_TLRW);
      if (ma == 0xFFFFFFFF) return;
      if (ma >= (TPCSSTART + ROMSTART)) return;

      if (tiline)
      {
	 puttiline (ma, v, FALSE);
      }
      else if (!inhibitwrite)
      {
	 PUTMEM0 (ma, v);
      }
   }
}

/***********************************************************************
* putmemb - Put a byte in memory.
***********************************************************************/

void
putmemb (uint16 pa, uint8 v, int srcdst)
{
   int mr;
   uint32 ma;
   uint16 mf;
   uint16 uv = v;

   tiline = FALSE;
   if (model < 10)
   {
      ma = pa;
      if (model == 5 && pa >= TILINESTART)
      {
	 ma += TPCSSTART;
	 if (pa < TILINEEND)
	 {
	    if (rom5cru)
	    {
	       ma = (pa - TILINESTART) + 0x200000;
#ifdef DEBUGROM51
	       fprintf (stderr, "putmemb: pa = >%04X, ma = >%06X\n",
			pa, ma);
#endif
	    }
	    else if (!(cpuoptions & OPT_NOTILINE))
	    {
	       tiline = TRUE;
	    }
	    else
	    {
	       ma = pa;
	    }
	 }
      }

      if (tiline)
      {
	 puttiline (ma, uv, TRUE);
      }
      else if (pa < memlen)
      {
	 PUTMEMB0 (ma, v);
      }
   }
   else
   {
      mr = 0;
      if (srcdst >= 0)
      {
	 if (srcdst) mr = use_LDS_map;
	 else mr = use_LDD_map;
      }
      if (mr) mf = 2;
      else mf = GET_MAP;

      ma = mapaddr (pa, mf, MEM_WRITE, TRACE_TLACCESS, TRACE_TLRW);
      if (ma == 0xFFFFFFFF) return;
      if (ma >= (TPCSSTART + ROMSTART)) return;

      if (tiline)
      {
	 puttiline (ma, uv, TRUE);
      }
      else if (!inhibitwrite)
      {
	 PUTMEMB0 (ma, v);
      }
   }
}

/***********************************************************************
* getaddr - Get an effective address.
***********************************************************************/

uint16
getaddr (uint16 sr, uint16 st, int inc)
{
   uint16 sa = 0;

   switch (st)
   {
   case 0:

      sa = wpreg + (sr * 2);
      break;

   case 1:

      sa = GETREG (sr);
      break;

   case 2:

      sa = GETINST(TRUE);
      trcbuf[trcidx].ops[trcopr++] = sa;
      INCPC(2);
      if (sr != 0) sa = (int16)sa + (int16)GETREG (sr);
      break;

   case 3:

      sa = GETREG (sr);
      if (inc) PUTREG (sr, sa + 2);
      break;
   }
   return (sa);
}

/***********************************************************************
* getbyte - Get a byte operand. If memory access is past the end of
* defined memory, return -1;
***********************************************************************/

uint8
getbyte (uint16 sr, uint16 st, int inc, int srcdst)
{
   uint16 sa;
   uint8 sval = 0xFF;

   switch (st)
   {
   case 0:

      sval = (GETREG (sr) & 0xFF00) >> 8;
      break;

   case 1:

      sa = GETREG (sr);
      tracema = TRUE;
      sval = GETMEMB (sa, srcdst);
      tracema = FALSE;
      break;

   case 2:

      sa = GETINST(TRUE);
      if (inc)
      {
	 trcbuf[trcidx].ops[trcopr++] = sa;
	 INCPC(2);
      }
      if (sr != 0) sa = (int16)sa + (int16)GETREG (sr);
      if (inc) tracema = TRUE;
      sval = GETMEMB (sa, srcdst);
      tracema = FALSE;
      break;

   case 3:

      sa = GETREG (sr);
      if (inc) tracema = TRUE;
      sval = GETMEMB (sa, srcdst);
      tracema = FALSE;
      if (inc) PUTREG (sr, sa + 1);
      break;
   }
   return (sval);
}

/***********************************************************************
* getword - Get a word operand. If memory access is past the end of
* defined memory, return -1;
***********************************************************************/

uint16
getword (uint16 sr, uint16 st, int inc, int srcdst)
{
   uint16 sa;
   uint16 sval = 0xFFFF;

   switch (st)
   {
   case 0:

      sval = GETREG (sr);
      break;

   case 1:

      sa = GETREG (sr);
      tracema = TRUE;
      sval = GETMEM (sa & 0xFFFE, srcdst);
      tracema = FALSE;
      break;

   case 2:

      sa = GETINST(TRUE);
      if (inc)
      {
	 trcbuf[trcidx].ops[trcopr++] = sa;
	 INCPC(2);
      }
      if (sr != 0) sa = (int16)sa + (int16)GETREG (sr);
      if (inc) tracema = TRUE;
      sval = GETMEM (sa & 0xFFFE, srcdst);
      tracema = FALSE;
      break;

   case 3:

      sa = GETREG (sr);
      if (inc) tracema = TRUE;
      sval = GETMEM (sa & 0xFFFE, srcdst);
      tracema = FALSE;
      if (inc) PUTREG (sr, sa + 2);
      break;
   }
   return (sval);
}

/***********************************************************************
* putaddr - Bump address by count
***********************************************************************/

void
putaddr (uint16 sr, uint16 st, uint16 inc)
{
   if (st == 3)
   {
      PUTREG (sr, GETREG (sr) + inc);
   }
}

/***********************************************************************
* putbyte - Put a byte operand.  If memory access is beyond allowed 
* memory (in ROM space) it is ignored.
***********************************************************************/

static void
putbyte (uint16 dr, uint16 dt, uint8 dval, int srcdst)
{
   uint16 da;

   switch (dt)
   {
   case 0:

      da = (GETREG (dr) & 0x00FF) | (dval << 8);
      PUTREG (dr, da);
      break;

   case 1:

      da = GETREG (dr);
      tracema = TRUE;
      PUTMEMB (da, dval, srcdst);
      tracema = FALSE;
      break;

   case 2:

      da = GETINST(TRUE);
      trcbuf[trcidx].ops[trcopr++] = da;
      INCPC(2);
      if (dr != 0) da = (int16)da + (int16)GETREG (dr);
      tracema = TRUE;
      PUTMEMB (da, dval, srcdst);
      tracema = FALSE;
      break;

   case 3:

      da = GETREG (dr);
      PUTREG (dr, da + 1);
      tracema = TRUE;
      PUTMEMB (da, dval, srcdst);
      tracema = FALSE;
      break;
   }
}

/***********************************************************************
* putword - Put a word operand.  If memory access is beyond allowed 
* memory (in ROM space) it is ignored.
***********************************************************************/

void
putword (uint16 dr, uint16 dt, uint16 dval, int srcdst)
{
   uint16 da;

   switch (dt)
   {
   case 0:

      PUTREG (dr, dval);
      break;

   case 1:

      da = GETREG (dr);
      tracema = TRUE;
      PUTMEM (da, dval, srcdst);
      tracema = FALSE;
      break;

   case 2:

      da = GETINST(TRUE);
      trcbuf[trcidx].ops[trcopr++] = da;
      INCPC(2);
      if (dr != 0) da = (int16)da + (int16)GETREG (dr);
      tracema = TRUE;
      PUTMEM (da, dval, srcdst);
      tracema = FALSE;
      break;

   case 3:

      da = GETREG (dr);
      PUTREG (dr, da + 2);
      tracema = TRUE;
      PUTMEM (da, dval, srcdst);
      tracema = FALSE;
      break;
   }
}

/***********************************************************************
* chkaddword - Check addition for carry and overflow
***********************************************************************/

static void
chkaddword (int32 acc, int16 sv1, int16 sv2)
{
   CLR_CARRY;
   CLR_OVER;

   if ((acc & 0xFFFF) < (sv1 & 0xFFFF)) SET_CARRY;

   if (((sv1 & 0x8000) == (sv2 & 0x8000)) &&
       ((sv2 & 0x8000) != (acc & 0x8000)))
   {
      SET_OVER;
      if (IS_OVERI && (GET_MASK >= ERRINT))
      {
	 errcrudata |= ERRAOVR;
	 geninterrupt (ERRINT);
      }
   }
}

/***********************************************************************
* chkaddbyte - Check addition for carry and overflow
***********************************************************************/

static void
chkaddbyte (int32 acc, int8 bsv1, int8 bsv2)
{
   CLR_CARRY;
   CLR_OVER;

   if ((acc & 0xFF) < (bsv1 & 0xFF)) SET_CARRY;

   if (((bsv1 & 0x80) == (bsv2 & 0x80)) &&
       ((bsv2 & 0x80) != (acc & 0x80)))
   {
      SET_OVER;
      if (IS_OVERI && (GET_MASK >= ERRINT))
      {
	 errcrudata |= ERRAOVR;
	 geninterrupt (ERRINT);
      }
   }
}

/***********************************************************************
* chksubword - Check subtraction for carry and overflow
***********************************************************************/

static void
chksubword (int32 acc, int16 sv1, int16 sv2)
{
   CLR_CARRY;
   CLR_OVER;

   if ((uint16)sv2 >= (uint16)sv1) SET_CARRY;

   if (((sv1 & 0x8000) != (sv2 & 0x8000)) &&
       ((sv2 & 0x8000) != (acc & 0x8000)))
   {
      SET_OVER;
      if (IS_OVERI && (GET_MASK >= ERRINT))
      {
	 errcrudata |= ERRAOVR;
	 geninterrupt (ERRINT);
      }
   }
}

/***********************************************************************
* chksubbyte - Check subtraction for carry and overflow
***********************************************************************/

static void
chksubbyte (int32 acc, int8 bsv1, int8 bsv2)
{
   CLR_CARRY;
   CLR_OVER;

   if ((uint8)bsv2 >= (uint8)bsv1) SET_CARRY;

   if (((bsv1 & 0x80) != (bsv2 & 0x80)) &&
       ((bsv2 & 0x80) != (acc & 0x80)))
   {
      SET_OVER;
      if (IS_OVERI && (GET_MASK >= ERRINT))
      {
	 errcrudata |= ERRAOVR;
	 geninterrupt (ERRINT);
      }
   }
}

/***********************************************************************
* chkparity - Check parity of a byte.
***********************************************************************/

static void
chkparity (uint8 bval)
{
   int c;
   int i;

   for (c = 0, i = 0; i < 8; i++)
   {
      c = c + (bval & 0x01);
      bval = bval >> 1;
   }
   if (c & 0x01) SET_ODDP;
   else CLR_ODDP;
}

/***********************************************************************
* cmpbytezero - Compare byte value to zero.
***********************************************************************/

void
cmpbytezero (uint8 bval)
{
   if (bval > 0) SET_LGT;
   else CLR_LGT;
   if ((int8)bval > 0) SET_AGT;
   else CLR_AGT;
   if (bval == 0) SET_EQ;
   else CLR_EQ;
}

/***********************************************************************
* cmpwordzero - Compare value to zero.
***********************************************************************/

void
cmpwordzero (uint16 val)
{
   if (val > 0) SET_LGT;
   else CLR_LGT;
   if ((int16)val > 0) SET_AGT;
   else CLR_AGT;
   if (val == 0) SET_EQ;
   else CLR_EQ;
}

/***********************************************************************
* proc1op - Process type 1 operations.
***********************************************************************/

static int
proc1op (uint16 inst)
{
   int32 acc;
   uint16 st;
   uint16 sr;
   uint16 val;
   uint16 cval;
   uint16 dt;
   uint16 dr;
   uint16 op;
   int16 sv1, sv2;
   uint16 uv1;
   uint8 bval;
   uint8 bcval;
   int8 bsv1, bsv2;
   uint8 buv1;

   sr =  inst & 0x000F;
   st = (inst & 0x0030) >> 4;
   dr = (inst & 0x03C0) >> 6;
   dt = (inst & 0x0C00) >> 10;
   op = (inst & 0xF000) >> 12;

   switch (op)
   {
   case 4: /* SZC */

      cval = getword (sr, st, TRUE, SRC);
      uv1 = getword (dr, dt, FALSE, DST);
      val = uv1 & (~cval);
      putword (dr, dt, val, DST);
      cmpwordzero (val);
      break;

   case 5: /* SZCB */

      bcval = getbyte (sr, st, TRUE, SRC);
      buv1 = getbyte (dr, dt, FALSE, DST);
      bval = buv1 & (~bcval);
      putbyte (dr, dt, bval, DST);
      cmpbytezero (bval);
      chkparity (bval);
      break;

   case 6: /* S */

      sv1 = (int16)getword (sr, st, TRUE, SRC);
      acc = sv1;
      sv2 = (int16)getword (dr, dt, FALSE, DST);
      acc = sv2 - acc;
      uv1 = acc & 0xFFFF;
      acc = (int16)uv1;
      chksubword (acc, sv1, sv2);
      val = acc & 0xFFFF;
      putword (dr, dt, val, DST);
      cmpwordzero (val);
      break;

   case 7: /* SB */

      bsv1 = (int8)getbyte (sr, st, TRUE, SRC);
      acc = bsv1;
      bsv2 = (int8)getbyte (dr, dt, FALSE, DST);
      acc = bsv2 - acc;
      buv1 = acc & 0xFF;
      acc = (int8)buv1;
      chksubbyte (acc, bsv1, bsv2);
      bval = acc &0xFF;
      putbyte (dr, dt, bval, DST);
      cmpbytezero (bval);
      chkparity (bval);
      break;

   case 8: /* C */

      val = getword (sr, st, TRUE, SRC);
      cval = getword (dr, dt, TRUE, DST);
      if (val > cval) SET_LGT;
      else CLR_LGT;
      if ((int16)val > (int16)cval) SET_AGT;
      else CLR_AGT;
      if (val == cval) SET_EQ;
      else CLR_EQ;
      break;

   case 9: /* CB */

      bval = getbyte (sr, st, TRUE, SRC);
      bcval = getbyte (dr, dt, TRUE, DST);
      if (bval > bcval) SET_LGT;
      else CLR_LGT;
      if ((int8)bval > (int8)bcval) SET_AGT;
      else CLR_AGT;
      if (bval == bcval) SET_EQ;
      else CLR_EQ;
      chkparity (bval);
      break;

   case 10: /* A */

      sv1 = (int16)getword (sr, st, TRUE, SRC);
      acc = sv1;
      sv2 = (int16)getword (dr, dt, FALSE, DST);
      acc = acc + sv2;
      uv1 = acc & 0xFFFF;
      acc = (int16)uv1;
      chkaddword (acc, sv1, sv2);
      val = acc & 0xFFFF;
      putword (dr, dt, val, DST);
      cmpwordzero (val);
      break;

   case 11: /* AB */

      bsv1 = (int8)getbyte (sr, st, TRUE, SRC);
      acc = bsv1;
      bsv2 = (int8)getbyte (dr, dt, FALSE, DST);
      acc = acc + bsv2;
      buv1 = acc & 0xFF;
      acc = (int8)buv1;
      chkaddbyte (acc, bsv1, bsv2);
      bval = acc & 0xFF;
      putbyte (dr, dt, bval, DST);
      cmpbytezero (bval);
      chkparity (bval);
      break;

   case 12: /* MOV */

      val = getword (sr, st, TRUE, SRC);
      putword (dr, dt, val, DST);
      cmpwordzero (val);
      break;

   case 13: /* MOVB */

      bval = getbyte (sr, st, TRUE, SRC);
      putbyte (dr, dt, bval, DST);
      cmpbytezero (bval);
      chkparity (bval);
      break;

   case 14: /* SOC */

      cval = getword (sr, st, TRUE, SRC);
      uv1 = getword (dr, dt, FALSE, DST);
      val = cval | uv1;
      putword (dr, dt, val, DST);
      cmpwordzero (val);
      break;

   case 15: /* SOCB */

      bcval = getbyte (sr, st, TRUE, SRC);
      buv1 = getbyte (dr, dt, FALSE, DST);
      bval = bcval | buv1;
      putbyte (dr, dt, bval, DST);
      cmpbytezero (bval);
      chkparity (bval);
      break;

   default:
      return (-1);
   }
   return (0);
}

/***********************************************************************
* proc2op - Process type 2 operations.
***********************************************************************/

static int
proc2op (uint16 inst)
{
   int32 mwpc;
   int16 wpc;
   int8 disp;

   disp = inst & 0x00FF;
   wpc = (pcreg >> 1) & 0x7FFF;
   mwpc = (mpcreg >> 1) & 0xFFFFF;

   switch ((inst & 0x0F00) >> 8)
   {
   case 0: /* JMP */

      break;

   case 1: /* JLT */

      if (!IS_AGT && !IS_EQ) break;
      return (0);

   case 2: /* JLE */

      if (!IS_LGT || IS_EQ) break;
      return (0);

   case 3: /* JEQ */

      if (IS_EQ) break;
      return (0);

   case 4: /* JHE */

      if (IS_LGT || IS_EQ) break;
      return (0);

   case 5: /* JGT */

      if (IS_AGT) break;
      return (0);

   case 6: /* JNE */

      if (!IS_EQ) break;
      return (0);

   case 7: /* JNC */

      if (!IS_CARRY) break;
      return (0);

   case 8: /* JOC */

      if (IS_CARRY) break;
      return (0);

   case 9: /* JNO */

      if (!IS_OVER) break;
      return (0);

   case 10: /* JL */

      if (!IS_LGT && !IS_EQ) break;
      return (0);

   case 11: /* JH */

      if (IS_LGT && !IS_EQ) break;
      return (0);

   case 12: /* JOP */

      if (IS_ODDP) break;
      return (0);

   case 13: /* SBO */

      setbitone (inst);
      return (0);

   case 14: /* SBZ */

      setbitzero (inst);
      return (0);

   case 15: /* TB */

      testbit (inst);
      return (0);

   default:
      return (-1);
   }

   /*
   ** Fall through, take the jump
   */

   if ((uint8)disp == 0xFF)
   {
      idle = TRUE;
      if (GET_MASK == 0)
	 run = FALSE;
   }
   else
      opcreg = pcreg;

   wpc += disp;
   mwpc += disp;
   pcreg = (wpc << 1) & 0xFFFE;
   mpcreg = (mwpc << 1) & 0x1FFFFE;
   return (0);
}

/***********************************************************************
* proc3op - Process type 3 operations.
***********************************************************************/

static int
proc3op (uint16 inst)
{
   uint16 st;
   uint16 sr;
   uint16 reg;
   uint16 val;
   uint16 cval;
   uint16 op;

   sr =  inst & 0x000F;
   st = (inst & 0x0030) >> 4;
   reg = (inst & 0x03C0) >> 6;
   op = (inst & 0x1C00) >> 10;

   switch (op)
   {
   case 0: /* COC */

      val = getword (sr, st, TRUE, SRC);
      cval = val & GETREG (reg);
      if (cval == val) SET_EQ;
      else CLR_EQ;
      break;

   case 1: /* CZC */

      val = getword (sr, st, TRUE, SRC);
      cval = val & (~GETREG (reg));
      if (cval == val) SET_EQ;
      else CLR_EQ;
      break;

   case 2: /* XOR */

      val = getword (sr, st, TRUE, SRC);
      val ^= GETREG (reg);
      PUTREG (reg, val);
      cmpwordzero (val);
      break;

   case 3: /* XOP */

      val = wpreg;
      cval = getaddr (sr, st, TRUE);
      op = statreg;
#ifdef DEBUGSVC
      if (traceit)
      {
	 fprintf (tracefd, "SVC:\n");
	 HEXDUMP (tracefd, (char *)&memory[cval], 32, cval);
	 if (memory[cval] == 0 && memory[cval+2] == 0x91)
	 {
	    uint16 pval = GETMEM0(cval+22);
	    fprintf (tracefd, "Path:\n");
	    HEXDUMP (tracefd, (char *)&memory[pval], 32, pval);
	 }
      }
#endif
      SET_XOP;
      CLR_NONPRV; /* Privileged */
      CLR_MAP; /* Use Map 0 */
#if defined(DEBUG_LCSX)
      if ((model == 12) && IS_WCS)
         fprintf (stderr, "%04X: XOP -> WCS: reg = %d\n", curpc, reg);
#endif
      if (FALSE) /*((model == 12) && IS_WCS) Until we have the WCS sim */
      {
	 /* Call microcode simulator */
	 CLR_WCS;
	 CLR_XOP;
      }
      else
      {
	 wpreg = GETMEM0 (reg * 4 + 0x0040) & 0xFFFE;
	 PUTREG (11, cval);
	 cval = pcreg;
	 mpcreg = pcreg = GETMEM0 (reg * 4 + 0x0042) & 0xFFFE;
	 PUTREG (13, val);
	 PUTREG (14, cval);
	 PUTREG (15, op);
	 deferint = 1;
      }
      break;

   default:
      return (-1);
   }
   return (0);
}

/***********************************************************************
* proc4op - Process type 4 operations.
***********************************************************************/

static int
proc4op (uint16 inst)
{
   uint16 sr;
   uint16 st;
   uint16 cnt;
   uint16 dat;
   uint8 bdat;

   sr = inst & 0x000F;
   st = (inst & 0x0030) >> 4;
   cnt = (inst & 0x03C0) >> 6;

   switch ((inst & 0x0400) >> 10)
   {
   case 0: /* LDCR */

      if (cnt > 1 && cnt < 9)
      {
         bdat = getbyte (sr, st, TRUE, SRC);
	 cmpbytezero (bdat);
	 chkparity (bdat);
	 dat = bdat;
      }
      else
      {
         dat = getword (sr, st, TRUE, SRC);
	 cmpwordzero (dat);
      }
      loadcru (inst, dat);
      break;

   case 1: /* STCR */

      dat = storecru (inst);
      if (cnt > 1 && cnt < 9)
      {
         bdat = (uint8)dat;
	 putbyte (sr, st, bdat, SRC);
	 cmpbytezero (bdat);
	 chkparity (bdat);
      }
      else
      {
	 putword (sr, st, dat, SRC);
	 cmpwordzero (dat);
      }
      break;

   default:
      return (-1);
   }
   return (0);
}

/***********************************************************************
* proc5op - Process type 5 operations.
***********************************************************************/

static int
proc5op (uint16 inst)
{
   uint32 acc;
   uint16 reg;
   uint16 cnt;
   uint16 val;
   uint16 sbit;

   reg = inst & 0x000F;
   cnt = (inst & 0x00F0) >> 4;
   if (cnt == 0)
   {
      cnt = R0 & 0x000F;
      if (cnt == 0) cnt = 16;
   }
   val = GETREG (reg);

   switch ((inst & 0x0300) >> 8)
   {
   case 0: /* SRA */

      acc = val;
      acc = acc << 16;
      acc = (int32)acc >> cnt;
      if (acc & 0x00008000) SET_CARRY;
      else CLR_CARRY;
      val = (acc >> 16) & 0xFFFF;
      PUTREG (reg, val);
      cmpwordzero (val);
      break;

   case 1: /* SRL */

      acc = val;
      acc = acc << 16;
      acc = acc >> cnt;
      if (acc & 0x00008000) SET_CARRY;
      else CLR_CARRY;
      val = (acc >> 16) & 0xFFFF;
      PUTREG (reg, val);
      cmpwordzero (val);
      break;

   case 2: /* SLA */

      CLR_OVER;
      acc = val;
      sbit = acc & 0x8000;
      while (cnt)
      {
	 acc = acc << 1;
	 if (sbit != (acc & 0x8000)) SET_OVER;
	 sbit = acc & 0x8000;
         cnt--;
      }
      if (acc & 0x10000) SET_CARRY;
      else CLR_CARRY;
      val = acc & 0xFFFF;
      PUTREG (reg, val);
      cmpwordzero (val);
      break;

   case 3: /* SRC */

      while (cnt)
      {
	 uint16 ob;

         ob = val & 0x0001;
	 val = val >> 1;
	 if (ob)
	 {
	    val = val | 0x8000;
	    SET_CARRY;
	 }
	 else CLR_CARRY;
	 cnt--;
      }
      PUTREG (reg, val);
      cmpwordzero (val);
      break;

   default:
      return (-1);
   }
   return (0);
}

/***********************************************************************
* proc6op - Process type 6 operations.
***********************************************************************/

static int
proc6op (uint16 inst)
{
   int32 acc;
   int32 lval;
   int32 rval;
   int32 rem;
   int16 v;
   uint16 sr;
   uint16 st;
   uint16 val;
   uint16 sval;
   uint16 xinst;

   sr = inst & 0x000F;
   st = (inst & 0x0030) >> 4;

   if ((inst & 0x0F00) >= 0x0C00) /* Floating point instructions */
   {
      return (proc6fltop (inst));
   }
   else if ((inst & 0x0F00) == 0x0100) 
   {
      switch ((inst & 0x00C0) >> 6)
      {
      case 1: /* BIND */

	 if (model < 11) return (-1);
	 val = getword (sr, st, TRUE, SRC);
	 pcreg = val & 0xFFFE;
         break;

      case 2: /* DIVS */

	 if (model < 11) return (-1);
	 rval = (int32)R0 << 16 | (int32)R1;
	 v = (int16)getword (sr, st, TRUE, SRC);
	 lval = v;
#if defined(DEBUG_DIVS)
	 fprintf (stderr,
	       "INST = DIVS sr = %d, st = %d\n",
	       sr, st);
	 fprintf (stderr,"   oldstat = >%04X\n", statreg);
	 fprintf (stderr,"   rval = >%08X, lval = >%08X\n", rval, lval);
#endif
	 SET_OVER;
	 if (lval != 0)
	 {
	    acc = rval / lval;
	    rem = rval % lval;
	    if ((labs (rem) < labs (lval)) &&
	        ((lval * acc + rem) == rval) &&
		(acc > -32769) && (acc < 32768) &&
		(rem > -32769) && (rem < 32768))
	    {
	       CLR_OVER;
	       val = acc & 0xFFFF;
	       PUTREG (0, val);
#if defined(DEBUG_DIVS)
	       fprintf (stderr, "   %ld = %ld / %ld\n", acc, rval, lval);
	       fprintf (stderr, "   %08X = %08X / %08X\n", acc, rval, lval);
#endif
	       sval = rem & 0xFFFF;
	       PUTREG (1, sval);
	       cmpwordzero (val);
#if defined(DEBUG_DIVS)
	       fprintf (stderr, "   %ld = %ld %% %ld\n", rem, rval, lval);
	       fprintf (stderr, "   %08X = %08X %% %08X\n", rem, rval, lval);
	       fprintf (stderr, "   newstat = >%04X\n", statreg);
#endif
	    }
	 }
         break;

      case 3: /* MPYS */

	 if (model < 11) return (-1);
	 v = (int16)getword (sr, st, TRUE, SRC);
	 lval = v;
	 v = (int16)R0;
	 rval = v;
#if defined(DEBUG_MPYS)
	 fprintf (stderr,
	       "INST = MPYS sr = %d, st = %d\n",
	       sr, st);
	 fprintf (stderr,"   oldstat = >%04X\n", statreg);
	 fprintf (stderr,"   rval = %ld, lval = %d\n", rval, lval);
#endif
	 acc = rval * lval;
	 PUTREG (0, (acc >> 16) & 0xFFFF);
	 PUTREG (1, acc & 0xFFFF);
	 cmplongzero (acc);
#if defined(DEBUG_MPYS)
	 fprintf (stderr, "   %ld = %ld * %ld\n", acc, rval, lval);
	 fprintf (stderr, "   %08X = %08X * %08X\n", acc, rval, lval);
	 fprintf (stderr, "   newstat = >%04X\n", statreg);
#endif
         break;

      default:
         return (-1);
      }
   }
   else switch ((inst & 0x03C0) >> 6)
   {
   case 0: /* BLWP */

      val = wpreg;
      if (st == 0)
      {
	 sval = GETREG (sr+1);
         wpreg = GETREG (sr) & 0xFFFE;
      }
      else
      {
	 sval = getaddr (sr, st, TRUE);
	 wpreg = GETMEM (sval, NOLD) & 0xFFFE;
	 sval = GETMEM (sval+2, NOLD);
      }
      PUTREG (13, val);
      PUTREG (14, pcreg);
      PUTREG (15, statreg);
      opcreg = pcreg;
      mpcreg = pcreg = sval & 0xFFFE;
      deferint = 1;
      break;

   case 1: /* B */

      val = getaddr (sr, st, TRUE);
      opcreg = pcreg;
      mpcreg = pcreg = val & 0xFFFE;
      break;

   case 2: /* X */

      val = getaddr (sr, st, TRUE);
      xinst = GETMEM (val, EXEC);
      runinst (xinst, TRUE);
      break;

   case 3: /* CLR */

      putword (sr, st, 0, SRC);
      break;

   case 4: /* NEG */

      sval = val = getword (sr, st, FALSE, SRC);
      val = -(int16)val;
      if (val == 0) SET_CARRY;
      else CLR_CARRY;
      if (sval == 0x8000) SET_OVER;
      else CLR_OVER;
      goto SAVE_CHECK_ZERO;

   case 5: /* INV */

      val = ~getword (sr, st, FALSE, SRC);
      goto SAVE_CHECK_ZERO;

   case 6: /* INC */

      val = getword (sr, st, FALSE, SRC);
      acc = (int16)val;
      sval = 1;
      goto CHECK_INC_STATUS;

   case 7: /* INCT */

      val = getword (sr, st, FALSE, SRC);
      acc = (int16)val;
      sval = 2;

CHECK_INC_STATUS:
      acc += sval;
      chkaddword (acc, (int16)val, (int16)sval);
      val = acc & 0xFFFF;
      goto SAVE_CHECK_ZERO;


   case 8: /* DEC */

      val = getword (sr, st, FALSE, SRC);
      acc = (int16)val;
      sval = 1;
      goto CHECK_DEC_STATUS;

   case 9: /* DECT */

      val = getword (sr, st, FALSE, SRC);
      acc = (int16)val;
      sval = 2;

CHECK_DEC_STATUS:
      acc -= sval;
      chksubword (acc, (int16)sval, (int16)val);
      val = acc & 0xFFFF;

SAVE_CHECK_ZERO:
      putword (sr, st, val, SRC);
      cmpwordzero (val);
      break;

   case 10: /* BL */

      val = getaddr (sr, st, TRUE);
      PUTREG (11, pcreg);
      opcreg = pcreg;
      mpcreg = pcreg = val & 0xFFFE;
      break;

   case 11: /* SWPB */

      val = getword (sr, st, FALSE, SRC);
      sval = val >> 8;
      sval |= (val & 0xFF) << 8;
      putword (sr, st, sval, SRC);
      break;

   case 12: /* SETO */

      putword (sr, st, 0xFFFF, SRC);
      break;

   case 13: /* ABS */

      CLR_OVER;
      CLR_CARRY;
      sval = val = getword (sr, st, FALSE, SRC);
      cmpwordzero (val);
      if ((int16)val < 0)
      {
	 if (val == 0x8000) SET_OVER;
	 val = -(int16)val;
      }
      putword (sr, st, val, SRC);
      deferint = 1;
      break;

   case 14: /* LDS */

#ifdef DEBUGLDSLDD
      tracefd = stderr;
      traceenable = TRUE;
      traceit = TRUE;
      tracestart = 0;
#endif
      if (model < 10) return (-1);
      if (IS_NONPRV)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%04X: ERRINT requested: nonPRIV LDS\n", curpc);
#endif
	 tracememory (TRACE_PRIVINST, curmpc);
	 errcrudata |= ERRPIN;
	 if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
         return (0);
      }
      ldspc = curpc;
      ldsmpc = curmpc;
      sval = val = getaddr (sr, st, TRUE);
      xinst = GETMEM (val, SRC);
      xinst &= (model == 12 ? 0xFFE3 : 0xFFE0);
      mapfile[2].l1 = xinst;

      val += 2;
      xinst = GETMEM (val, SRC);
      mapfile[2].b1 = xinst;

      val += 2;
      xinst = GETMEM (val, SRC);
      xinst &= (model == 12 ? 0xFFE3 : 0xFFE0);
      mapfile[2].l2 = xinst;

      val += 2;
      xinst = GETMEM (val, SRC);
      mapfile[2].b2 = xinst;

      val += 2;
      xinst = GETMEM (val, SRC);
      xinst &= (model == 12 ? 0xFFE3 : 0xFFE0);
      mapfile[2].l3 = xinst;

      val += 2;
      xinst = GETMEM (val, SRC);
      mapfile[2].b3 = xinst;

      use_LDS_map = TRUE;
      use_map_cnt = 2;
#if defined(DEBUGMAPPER) || defined(DEBUGLDSLDD)
      fprintf (stderr, "\n%04X: LDS val = >%04X\n", curpc, sval);
      fprintf (stderr, "mapfile[%d].l1 = >%04X, mapfile[%d].b1 = >%04X\n",
	       2, mapfile[2].l1,
	       2, mapfile[2].b1);
      fprintf (stderr, "mapfile[%d].l2 = >%04X, mapfile[%d].b2 = >%04X\n",
	       2, mapfile[2].l2,
	       2, mapfile[2].b2);
      fprintf (stderr, "mapfile[%d].l3 = >%04X, mapfile[%d].b3 = >%04X\n",
	       2, mapfile[2].l3,
	       2, mapfile[2].b3);
#endif
      deferint = 1;
      break;

   case 15: /* LDD */

#ifdef DEBUGLDSLDD
      tracefd = stderr;
      traceenable = TRUE;
      traceit = TRUE;
      tracestart = 0;
#endif
      if (model < 10) return (-1);
      if (IS_NONPRV)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%04X: ERRINT requested: nonPRIV LDD\n", curpc);
#endif
	 tracememory (TRACE_PRIVINST, curmpc);
	 errcrudata |= ERRPIN;
	 if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
         return (0);
      }
      lddpc = curpc;
      lddmpc = curmpc;
      sval = val = getaddr (sr, st, TRUE);
      xinst = GETMEM (val, SRC);
      xinst &= (model == 12 ? 0xFFE3 : 0xFFE0);
      mapfile[2].l1 = xinst;

      val += 2;
      xinst = GETMEM (val, SRC);
      mapfile[2].b1 = xinst;
      val += 2;

      xinst = GETMEM (val, SRC);
      xinst &= (model == 12 ? 0xFFE3 : 0xFFE0);
      mapfile[2].l2 = xinst;

      val += 2;
      xinst = GETMEM (val, SRC);
      mapfile[2].b2 = xinst;

      val += 2;
      xinst = GETMEM (val, SRC);
      xinst &= (model == 12 ? 0xFFE3 : 0xFFE0);
      mapfile[2].l3 = xinst;

      val += 2;
      xinst = GETMEM (val, SRC);
      mapfile[2].b3 = xinst;

      use_LDD_map = TRUE;
      use_map_cnt = 2;
#if defined(DEBUGMAPPER) || defined(DEBUGLDSLDD)
      fprintf (stderr, "\n%04X: LDD val = >%04X\n", curpc, sval);
      fprintf (stderr, "mapfile[%d].l1 = >%04X, mapfile[%d].b1 = >%04X\n",
	       2, mapfile[2].l1,
	       2, mapfile[2].b1);
      fprintf (stderr, "mapfile[%d].l2 = >%04X, mapfile[%d].b2 = >%04X\n",
	       2, mapfile[2].l2,
	       2, mapfile[2].b2);
      fprintf (stderr, "mapfile[%d].l3 = >%04X, mapfile[%d].b3 = >%04X\n",
	       2, mapfile[2].l3,
	       2, mapfile[2].b3);
#endif
      deferint = 1;
      break;

   default:
      return (-1);
   }
   return (0);
}

/***********************************************************************
* proc7op - Process type 7 operations.
***********************************************************************/

static int
proc7op (uint16 inst)
{
#ifdef DEBUGINTZERO
   uint16 oldstat;
#endif
   uint16 val;

   switch ((inst & 0x00F0) >> 4)
   {
   case 0: /* Floating point instructions */

      return (proc7fltop (inst));

   case 2:
      if (model < 12) return (-1);
      switch (inst & 0x000F)
      {
      case 13: /* EMD */
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  "INST = EMD\n");
#endif
	 if (IS_NONPRV)
	 {
#ifdef DEBUGERRINT
	    fprintf (stderr, "%04X: ERRINT requested: nonPRIV EMD\n", curpc);
#endif
	    tracememory (TRACE_PRIVINST, curmpc);
	    errcrudata |= ERRPIN;
	    if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
	    return (0);
	 }
	 /*
	 ** Fake it
	 */
	 statreg = 0;
	 mapenabled = FALSE;
	 mapfile[0].l1 = 0;
	 mapfile[0].b1 = 0;
	 mapfile[0].l2 = 0;
	 mapfile[0].b2 = 0;
	 mapfile[0].l3 = 0;
	 mapfile[0].b3 = 0;
         break;

      case 14: /* EINT */
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  "INST = EINT\n");
#endif
	 if (IS_NONPRV)
	 {
#ifdef DEBUGERRINT
	    fprintf (stderr, "%04X: ERRINT requested: nonPRIV EINT\n", curpc);
#endif
	    tracememory (TRACE_PRIVINST, curmpc);
	    errcrudata |= ERRPIN;
	    if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
	    return (0);
	 }
	 intsenabled = TRUE;
	 deferint = 1;
         break;

      case 15: /* DINT */
#ifdef DEBUGEINSTR
	 fprintf (stderr,
		  "INST = DINT\n");
#endif
	 if (IS_NONPRV)
	 {
#ifdef DEBUGERRINT
	    fprintf (stderr, "%04X: ERRINT requested: nonPRIV DINT\n", curpc);
#endif
	    tracememory (TRACE_PRIVINST, curmpc);
	    errcrudata |= ERRPIN;
	    if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
	    return (0);
	 }
	 intsenabled = FALSE;
         break;

      default:
         return (-1);
      }
      break;

   case 4: /* IDLE */

      if (model >= 10 && IS_NONPRV)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%04X: ERRINT requested: nonPRIV IDLE\n", curpc);
#endif
	 tracememory (TRACE_PRIVINST, curmpc);
	 errcrudata |= ERRPIN;
	 if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
         return (0);
      }
      idle = TRUE;
      idleled = TRUE;
      if (GET_MASK == 0)
      {
	 run = FALSE;
	 INCPC(2);
      }
      instcount--;
      DECPC(2);
      break;

   case 6: /* RSET */

      if (model >= 10 && IS_NONPRV)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%04X: ERRINT requested: nonPRIV RSET\n", curpc);
#endif
	 tracememory (TRACE_PRIVINST, curmpc);
	 errcrudata |= ERRPIN;
	 if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
         return (0);
      }
      SET_MASK (0);
      stopclk (FALSE);
      errcrudata &= 0xF;
      if (!IS_MAP)
      {
	 mapcrudata = 0;
	 maplatch = 0;
	 mapenabled = FALSE;
      }
      intsenabled = TRUE;
      bkptenabled = FALSE;
      fakeprefetch = FALSE;
      romtowcs ();
      resetdev ();
      PUTMEM0 (0x80, 0xFFFF);
      PUTMEM0 (0x82, 0xF800);
      PUTMEM0 (0x84, 0x0800);
      break;

   case 8: /* RTWP */

      val = R13 & 0xFFFE;
#ifdef DEBUGINTZERO
      oldstat = statreg;
#endif
      mpcreg = pcreg = R14 & 0xFFFE;
      if (IS_NONPRV)
      {
	 uint16 st;
         st = statreg & 0x01DF;
	 st |= R15 & 0xFE20;
         statreg = st;
      }
      else
      {
	 statreg = R15;
      }
#ifdef DEBUGINTZERO
      if (((oldstat & 15) == 1) && ((statreg & 15) == 0))
	 fprintf (stderr, "%04X: RTWP: oldstat = %04X, statreg = %04X\n",
		  curpc, oldstat, statreg);
#endif
      wpreg = val;
      break;

   case 10: /* CKON */

      if (model >= 10 && IS_NONPRV)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%04X: ERRINT requested: nonPRIV CKON\n", curpc);
#endif
	 tracememory (TRACE_PRIVINST, curmpc);
	 errcrudata |= ERRPIN;
	 if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
         return (0);
      }
      startclk (TRUE);
      break;

   case 12: /* CKOF */

      if (model >= 10 && IS_NONPRV)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%04X: ERRINT requested: nonPRIV CKOF\n", curpc);
#endif
	 tracememory (TRACE_PRIVINST, curmpc);
	 errcrudata |= ERRPIN;
	 if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
         return (0);
      }
      stopclk (FALSE);
      break;

   case 14: /* LREX */

      if (model >= 10 && IS_NONPRV)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%04X: ERRINT requested: nonPRIV LREX\n", curpc);
#endif
	 tracememory (TRACE_PRIVINST, curmpc);
	 errcrudata |= ERRPIN;
	 if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
         return (0);
      }
      val = wpreg;
      CLR_MAP;
      if (model == 4 || model == 9)
	 wpreg = GETMEM0 (0xFFFC) & 0xFFFE;
      else
	 wpreg = GETMEM0 (TPCSSTART+0xFFFC) & 0xFFFE;
      PUTREG (13, val);
      PUTREG (14, pcreg);
      PUTREG (15, statreg);
      if (model == 4 || model == 9)
	 mpcreg = pcreg = GETMEM0 (0xFFFE) & 0xFFFE;
      else
	 mpcreg = pcreg = GETMEM0 (TPCSSTART+0xFFFE) & 0xFFFE;
      SET_MASK (0);
      break;

   default:
      return (-1);
   }
   return (0);
}

/***********************************************************************
* proc8op - Process type 8 operations.
***********************************************************************/

static int
proc8op (uint16 inst)
{
   int32 acc;
   uint16 r;
   uint16 val;
   uint16 cval;
   uint16 tval;
   int32 sv1, sv2;

   r = inst & 0x000F;

   if ((inst & 0xFFF0) == 0x00B0) /* BLSK */
   {
      if (model < 12) return (-1);
      val = GETINST(TRUE);
      trcbuf[trcidx].ops[trcopr++] = val;
#ifdef DEBUGEINSTR
      fprintf (stderr,
	       "INST = BLSK r = %d addr = >%04X\n", r, val);
#endif
      cval = GETREG(r) - 2;
      PUTMEMB (cval, ((pcreg + 2) >> 8) & 0xFF, SRC);
      PUTMEMB (cval+1, (pcreg + 2) & 0xFF, SRC);
      PUTREG (r, cval);
      mpcreg = pcreg = (val & 0xFFFE);
   }
   else if (inst == 0x0300) /* LIMI */
   {
      if (model >= 10 && IS_NONPRV)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%04X: ERRINT requested: nonPRIV LIMI\n", curpc);
#endif
	 tracememory (TRACE_PRIVINST, curmpc);
	 errcrudata |= ERRPIN;
	 if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
         return (0);
      }
      val = GETINST(TRUE);
      INCPC(2);
      trcbuf[trcidx].ops[trcopr++] = val;
      SET_MASK (val);
   }

   else switch ((inst & 0x00F0) >> 4)
   {
   case 0: /* LI */

      val = GETINST(TRUE);
      INCPC(2);
      trcbuf[trcidx].ops[trcopr++] = val;
      PUTREG (r, val);
      cmpwordzero (val);
      break;

   case 2: /* AI */

      sv1 = GETINST(TRUE);
      INCPC(2);
      trcbuf[trcidx].ops[trcopr++] = sv1;
      cval = sv1;
      acc = sv1;
      tval = GETREG (r);
      sv2 = tval;
      acc = acc + sv2;
      val = acc & 0xFFFF;
      acc = (int16)val;
      chkaddword (acc, sv1, sv2);
      val = acc & 0xFFFF;
      PUTREG (r, val);
      cmpwordzero (val);
      break;

   case 4: /* ANDI */

      cval = GETINST(TRUE);
      INCPC(2);
      trcbuf[trcidx].ops[trcopr++] = cval;
      tval = GETREG (r);
      val = tval & cval;
      PUTREG (r, val);
      cmpwordzero (val);
      break;

   case 6: /* ORI */

      cval = GETINST(TRUE);
      INCPC(2);
      trcbuf[trcidx].ops[trcopr++] = cval;
      tval = GETREG (r);
      val = tval | cval;
      PUTREG (r, val);
      cmpwordzero (val);
      break;

   case 8: /* CI */

      cval = GETINST(TRUE);
      INCPC(2);
      trcbuf[trcidx].ops[trcopr++] = cval;
      val = GETREG (r);
      if (val > cval) SET_LGT;
      else CLR_LGT;
      if ((int16)val > (int16)cval) SET_AGT;
      else CLR_AGT;
      if (val == cval) SET_EQ;
      else CLR_EQ;
      break;

   case 10: /* STWP */

      PUTREG (r, wpreg);
      break;

   case 12: /* STST */

      PUTREG (r, statreg);
      break;

   case 14: /* LWPI */

      val = wpreg;
      wpreg = GETINST(TRUE) & 0xFFFE;
      INCPC(2);
      trcbuf[trcidx].ops[trcopr++] = wpreg;
      break;

   default:
      return (-1);
   }
   return (0);
}

/***********************************************************************
* proc9op - Process type 9 operations.
***********************************************************************/

static int
proc9op (uint16 inst)
{
   uint32 acc;
   uint16 val;
   uint16 st;
   uint16 sr;
   uint16 r;
   uint16 uval;
   uint16 op;

   sr =  inst & 0x000F;
   st = (inst & 0x0030) >> 4;
   r = (inst & 0x03C0) >> 6;
   op = (inst & 0x0400) >> 10;

   switch (op)
   {
   case 0: /* MPY */

      val = getword (sr, st, TRUE, SRC);
      acc = GETREG (r);
      acc = acc * (uint32)val;
      uval = (acc >> 16) & 0xFFFF;
      PUTREG (r, uval);
      uval = acc & 0xFFFF;
      PUTREG (r+1, uval);
      break;

   case 1: /* DIV */

      val = getword (sr, st, TRUE, SRC);
      if (val <= GETREG (r))
      {
         SET_OVER;
      }
      else
      {
	 CLR_OVER;
	 acc = GETREG (r);
	 acc = (acc << 16) | GETREG (r+1);
	 PUTREG (r, acc / val);
	 PUTREG (r+1, acc % val);
      }
      break;

   default:
      return (-1);
   }
   return (0);
}

/***********************************************************************
* proc10op - Process type 10 operations.
***********************************************************************/

static int
proc10op (uint16 inst)
{
#ifdef DEBUGLMF
   FILE *oldtracefd;
   int oldtrace;
#endif
   uint32 ma;
   uint16 r;
   uint16 mf;
   uint16 addr;
   uint16 xinst;

   r =  inst & 0x000F;
   mf = (inst & 0x0010) >> 4;
   if (model < 10) return (-1);
   if (IS_NONPRV)
   {
#ifdef DEBUGERRINT
	 fprintf (stderr, "%04X: ERRINT requested: nonPRIV LMF\n", curpc);
#endif
      tracememory (TRACE_PRIVINST, curmpc);
      errcrudata |= ERRPIN;
      if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
      return (0);
   }

#ifdef DEBUGLMF
   oldtracefd = tracefd;
   oldtrace = traceit;
   tracefd = stderr;
   traceit = TRUE;
#endif

   /* Get address using current map */
   addr = GETREG (r);
   ma = mapaddr (addr & 0xFFFE, GET_MAP, MEM_READ, TRACE_TLACCESS, 0);

   xinst = GETMEM0 (ma);
   xinst &= (model == 12 ? 0xFFE3 : 0xFFE0);
   mapfile[mf].l1 = xinst;

   ma += 2;
   xinst = GETMEM0 (ma);
   mapfile[mf].b1 = xinst;

   ma += 2;
   xinst = GETMEM0 (ma);
   xinst &= (model == 12 ? 0xFFE3 : 0xFFE0);
   mapfile[mf].l2 = xinst;

   ma += 2;
   xinst = GETMEM0 (ma);
   mapfile[mf].b2 = xinst;

   ma += 2;
   xinst = GETMEM0 (ma);
   xinst &= (model == 12 ? 0xFFE3 : 0xFFE0);
   mapfile[mf].l3 = xinst;

   ma += 2;
   xinst = GETMEM0 (ma);
   mapfile[mf].b3 = xinst;

#ifdef DEBUGLMF
   tracefd = oldtracefd;
   traceit = oldtrace;
#endif

#ifdef DEBUGMAPPER
   fprintf (stderr, "%04X: LMF %d,%d, addr = >%04X\n",
	    curpc, r, mf, GETREG (r));
   fprintf (stderr, "mapfile[%d].l1 = >%04X, mapfile[%d].b1 = >%04X\n",
	    mf, mapfile[mf].l1,
	    mf, mapfile[mf].b1);
   fprintf (stderr, "mapfile[%d].l2 = >%04X, mapfile[%d].b2 = >%04X\n",
	    mf, mapfile[mf].l2,
	    mf, mapfile[mf].b2);
   fprintf (stderr, "mapfile[%d].l3 = >%04X, mapfile[%d].b3 = >%04X\n",
	    mf, mapfile[mf].l3,
	    mf, mapfile[mf].b3);
#endif
   return (0);
}

/***********************************************************************
* printtrace - Print an instruction trace entry.
***********************************************************************/

static void
printtrace (void)
{
   if (traceit)
   {
      if (traceregs)
      {
	 fprintf (tracefd, "R00 %04X  R01 %04X  R02 %04X  R03 %04X\n",
		   trcbuf[trcidx].regs[0], trcbuf[trcidx].regs[1],
		   trcbuf[trcidx].regs[2], trcbuf[trcidx].regs[3]);
	 fprintf (tracefd, "R04 %04X  R05 %04X  R06 %04X  R07 %04X\n",
		   trcbuf[trcidx].regs[4], trcbuf[trcidx].regs[5],
		   trcbuf[trcidx].regs[6], trcbuf[trcidx].regs[7]);
	 fprintf (tracefd, "R08 %04X  R09 %04X  R10 %04X  R11 %04X\n",
		   trcbuf[trcidx].regs[8], trcbuf[trcidx].regs[9],
		   trcbuf[trcidx].regs[10], trcbuf[trcidx].regs[11]);
	 fprintf (tracefd, "R12 %04X  R13 %04X  R14 %04X  R15 %04X\n",
		   trcbuf[trcidx].regs[12], trcbuf[trcidx].regs[13],
		   trcbuf[trcidx].regs[14], trcbuf[trcidx].regs[15]);
      }
      else
      {
	 if (traceline >= 60)
	 {
	    traceline = 0;
	    fprintf (tracefd,
		  " PC   MPC    WP   ST         INST                 CODE\n");
	 }
      }

      printinst (outbuf, trcidx);
      fprintf (tracefd, "%s\n", outbuf);
      traceline++;
      if (traceregs) fprintf (tracefd, "\n");
   }
}

/***********************************************************************
* runinst - Run an instruction.
***********************************************************************/

static void 
runinst (uint16 inst, int Xinst)
{
   int error;
   int type;
   int i;
   int oldbkptenabled;
   uint32 lclmpc;
   uint16 lclpc;

   if (traceenable && curmpc >= tracelim[0] && curmpc <= tracelim[1] &&
      instcount >= tracestart)
      traceit = TRUE;
   else
      traceit = FALSE;

   lclpc = curpc;
   lclmpc = curmpc;
   if (Xinst)
   {
      lclpc = trcbuf[trcidx].pc;
      lclmpc = trcbuf[trcidx].mpc;
      printtrace ();
      trcidx++;
   }

   if (inst != IDLEINST) idleled = FALSE;
   curinst = inst;
   type = decodeinst (inst);

#if defined(DEBUGINSTR)
   if (enablepanel)
   {
      screenposition (windowlen,1);
      clearline (enablepanel);
   }
   printf ("INST = %04X TYPE = %d", inst, type);
   if (!enablepanel) printf ("\n");
#endif

   deferint = 0;
   error = -1;

   oldbkptenabled = bkptenabled;
   bkptenabled = FALSE;
   if (trcidx >= TRCBUFMAX) trcidx = 0;
   trcbuf[trcidx].pc = lclpc;
   trcbuf[trcidx].mpc = lclmpc;
   trcbuf[trcidx].wp = wpreg;
   trcbuf[trcidx].st = statreg;
   trcbuf[trcidx].inst = inst;
   for (i = 0; i < 16; i++)
   {
      trcbuf[trcidx].regs[i] = getreg (i);
   }
   trcbuf[trcidx].ops[0] = trcbuf[trcidx].ops[1] = trcbuf[trcidx].ops[2] = -1;
   trcopr = 0;
   bkptenabled = oldbkptenabled;

   if (type != TYPE_ILL)
      error = inst_proc[type].proc (inst);

   if (!Xinst)
   {
      printtrace ();
      trcidx++;
   }

   if (error < 0)
   {
      if (model >= 10)
      {
#ifdef DEBUGERRINT
	 fprintf (stderr,
		  "%04X: Illegal instruction: inst = >%04X, pc = >%04X\n",
		  curpc, inst, curpc);
         
         runled = FALSE;
#endif
	 tracememory (TRACE_ILLOP, curmpc);
	 errcrudata |= ERRILO;
	 if (model == 12) errcrudata |= ERRCPU12;
	 if (model == 11) errcrudata |= ERRCPU10A;
	 if (GET_MASK >= ERRINT) geninterrupt (ERRINT);
      }
#ifdef DEBUGERRINT
      else
	 sprintf (view[0], "Illegal instruction: inst = >%04X, pc = >%04X",
		  inst, curpc);
#endif
   }

   instcount++;

   if (use_map_cnt)
   {
#ifdef DEBUGLDSLDD
      fprintf (stderr, "use_map_cnt = %d\n", use_map_cnt);
#endif
#ifdef DEBUGMAPPER1
      tracefd = stderr;
      traceenable = TRUE;
      tracestart = 0;
#endif
      use_map_cnt--;
      if (use_map_cnt == 0)
      {
	 use_LDS_map = FALSE;
	 use_LDD_map = FALSE;
#ifdef DEBUGLDSLDD
	 traceenable = FALSE;
#endif
#ifdef DEBUGMAPPER1
	 traceenable = FALSE;
#endif
      }
   }
   inhibitwrite = FALSE;
}

/***********************************************************************
* stepprog - Step through a program.
***********************************************************************/

void 
stepprog (void)
{
   uint16 inst;

   inst = GETINST(TRUE);
   curpc = pcreg;
   curmpc = mpcreg;
   pcreg = (pcreg + 2) & 0xFFFE;
   mpcreg = (mpcreg + 2) & 0x1FFFFE;
   runinst (inst, FALSE);
   tracememory (TRACE_INSTFETCH, pcreg);
}

/***********************************************************************
* runprog - Run a program.
***********************************************************************/

void
runprog (void)
{
   int i;

   ttrunstate ();

   for (i = 0; i < maxdevcnt; i++)
   {
      Device *dev = &devices[i];

      if ((dev->switches & SWEMUTERM) && (dev->emutype < EMU733) &&
          (dev->infd == stdin))
	 dev->cbreak = TRUE;
   }

   while (run)
   {
      idle = FALSE;
      runled = TRUE;
      stepprog ();

      if (tcenable)
      {
         if (--tccount == 0)
	 {
	    tcenable = FALSE;
	    errcrudata |= ERR12MS;
	    geninterrupt (ERRINT);
	 }
      }

      if (breakflag && curmpc == breakaddr)
      {
         run = FALSE;
         runled = FALSE;
	 showstatus = TRUE;
      }
      else
      {
	 if (enablepanel && pancount) panel ();
	 if (!deferint)
	    checkinterrupts ();
      }
      if (run && delayclock)
      {
         long i, j;
	 j = delayclock;
	 for (i = 0; i < j; i++) ;
      }
   }

   ttcmdstate ();
}
