/***********************************************************************
*
* simprint.c - Fast (parallel) printer for the TI 990 Simulator.
*
* Changes:
*   09/20/13   DGP   Original.
*   06/02/15   DGP   Removed USS (z/OS) support.
*   05/19/16   DGP   Added real device support.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>

#include "simdef.h"

extern uint16 statreg;	/* The program status register */

/***********************************************************************
* printputchar - Put character on printer.
***********************************************************************/

void
printputchar (Device *dev, uint8 ch)
{
#ifdef DEBUGPRINT
   fprintf (stderr, "printputchar: ch = %c(%02X)\n",
	    isprint(ch) ? ch : '.', ch);
#endif

   if (dev->realdevice)
   {
      fputc (ch, dev->outfd);
      return;
   }

   if (ch == 0x0C) fputc (0x0A, dev->outfd);
   if ((ch > 0x1F && ch < 0x7F) || (ch > 0x07 && ch < 0x0D))
      fputc (ch, dev->outfd);
}

/***********************************************************************
* printseto - Set CRU bit to one.
***********************************************************************/

void
printseto (Device *dev, uint16 devaddr, uint8 disp)
{
#ifdef DEBUGPRINT
   fprintf (stderr, "CRU %04X SBO %d\n", devaddr, disp);
#endif
   switch (disp)
   {
   case 8:	/* Data Strobe */
      break;
   case 9:	/* Paper instructions. */
      break;
   case 11:	/* interface verification */
      break;
   case 14:
      dev->intenabled = TRUE;
      dev->select = TRUE;
      break;
   case 15:	/* Clear interrupt */
      break;
   default: ;
   }
}

/***********************************************************************
* printsetz - Set CRU bit to zero.
***********************************************************************/

void
printsetz (Device *dev, uint16 devaddr, uint8 disp)
{
#ifdef DEBUGPRINT
   fprintf (stderr, "CRU %04X SBZ %d\n", devaddr, disp);
#endif
   switch (disp)
   {
   case 8:	/* Data Strobe */
      break;
   case 9:	/* Paper instructions. */
      break;
   case 11:	/* interface verification */
      break;
   case 14:
      dev->intenabled = FALSE;
      dev->select = FALSE;
      break;
   case 15:	/* Clear interrupt */
      break;
   default: ;
   }
}

/***********************************************************************
* printtb - Test CRU bit.
***********************************************************************/

void
printtb (Device *dev, uint16 devaddr, uint8 disp)
{
#ifdef DEBUGPRINT
   fprintf (stderr, "CRU %04X TB %d\n", devaddr, disp);
#endif

   CLR_EQ;
   switch (disp)
   {
      case 7:	/* interface verification */
	 break;
      case 12:	/* Online */
	 if (dev->outfd != NULL)
	    SET_EQ;
	 break;
      case 13:	/* Data demand */
	 break;
      case 14:	/* Ready */
	 break;
      case 15:	/* Interrupt */
	 break;
      default: ;
   }
}

/***********************************************************************
* printldcr - Load CRU.
***********************************************************************/

void
printldcr (Device *dev, uint16 devaddr, uint16 cnt, uint16 dat)
{
#ifdef DEBUGPRINT
   fprintf (stderr, "CRU %04X LDCR CNT %d DATA %04X\n", devaddr, cnt, dat);
#endif

   printputchar (dev, (~dat) & 0x7F);
   if (dev->intenabled)
      gendevinterrupt (dev);
}

/***********************************************************************
* printstcr - Store CRU.
***********************************************************************/

uint16
printstcr (Device *dev, uint16 devaddr, uint16 cnt)
{
#ifdef DEBUGPRINT
   fprintf (stderr, "CRU %04X STCR CNT %d \n", devaddr, cnt);
#endif
   return 0;
}
