/***********************************************************************
*
* simpunch.c - Card Punch for the TI 990 Simulator.
*
* Changes:
*   12/11/13   DGP   Original.
*   06/02/15   DGP   Removed USS (z/OS) support.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>

#include "simdef.h"
#include "simcra.h"

extern int run;
extern uint16 statreg;	/* The program status register */

/***********************************************************************
* punchcard - Write the card buffer.
***********************************************************************/

void
punchcard (Device *dev, CARDInfo *devinfo)
{
   char *cbuf;
   int i;

   if (devinfo->cardndx == 0) return;

   cbuf = &devinfo->carddat[devinfo->cardndx];

   /*
   ** Trim trailing blanks
   */

   if (dev->switches & SWPUTRIM)
   {
      cbuf--;
      for (i = devinfo->cardndx; i && (*cbuf == ' '); i--)
      {
	 *cbuf-- = 0;
      }
      cbuf++;
   }

   /*
   ** Not trimming blanks, pad to 80 if necessary.
   */

   else if (devinfo->cardndx < 80)
   {
      for (i = devinfo->cardndx; i < 80; i++)
         *cbuf++ = ' ';
   }

   *cbuf++ = '\n';
   *cbuf = 0;

   /*
   ** Write record
   */

   if (dev->outfd)
   {
      if (fputs (devinfo->carddat, dev->outfd) < 0)
      {
	 devinfo->cardeof = TRUE;
      }
      fflush (dev->outfd);
   }

#ifdef DEBUGPUNCH
   fprintf (stderr, "CARDREC:\n");
   HEXDUMP (stderr, devinfo->carddat, 80, 0);
#endif
   devinfo->cardndx = 0;
}

/***********************************************************************
* punchseto - Set CRU bit to one.
***********************************************************************/

void
punchseto (Device *dev, uint16 devaddr, uint8 disp)
{
   CARDInfo *devinfo;

   devinfo = &dev->info.cardinfo;
#ifdef DEBUGPUNCH
   fprintf (stderr, "CRU %04X SBO %d\n", devaddr, disp);
#endif

   switch (disp)
   {
   case 12:	/* Start the card */
      punchcard (dev, devinfo);
      dev->select = TRUE;
      break;

   case 13:	/* Enable Column interrupt */
      dev->intenabled = TRUE;
      dev->select = TRUE;
      gendevinterrupt (dev);
      break;

   default: ;
   }
}

/***********************************************************************
* punchsetz - Set CRU bit to zero.
***********************************************************************/

void
punchsetz (Device *dev, uint16 devaddr, uint8 disp)
{
   CARDInfo *devinfo;

   devinfo = &dev->info.cardinfo;
#ifdef DEBUGPUNCH
   fprintf (stderr, "CRU %04X SBZ %d\n", devaddr, disp);
#endif

   switch (disp)
   {
   case 12:	/* Flush card */
      punchcard (dev, devinfo);
      dev->select = FALSE;
      break;

   case 13:	/* Disable Column interrupt */
      dev->intenabled = FALSE;
      dev->select = FALSE;
      clrinterrupt (dev->intlvl);
      break;

   case 14:	/* Clear column interrupt */
      if (dev->select)
      {
	 dev->intenabled = TRUE;
      }
      break;

   default: ;
   }
}

/***********************************************************************
* punchtb - Test CRU bit.
***********************************************************************/

void
punchtb (Device *dev, uint16 devaddr, uint8 disp)
{
#ifdef DEBUGPUNCH
   fprintf (stderr, "CRU %04X TB %d\n", devaddr, disp);
#endif

   switch (disp)
   {
   case 15:	/* Column Ready */
      SET_EQ;
      break;

   default: 
      CLR_EQ;
   }
}

/***********************************************************************
* punchstcr - Store CRU.
***********************************************************************/

uint16
punchstcr (Device *dev, uint16 devaddr, uint16 cnt)
{
#ifdef DEBUGPUNCH
   fprintf (stderr, "CRU %04X STCR CNT %d\n", devaddr, cnt);
#endif
   return 0;
}

/***********************************************************************
* punchldcr - Load CRU.
***********************************************************************/

void
punchldcr (Device *dev, uint16 devaddr, uint16 cnt, uint16 dat)
{
   CARDInfo *devinfo;
   int cdat;
   int i;

   /*
   ** Translate from card code
   */

   devinfo = &dev->info.cardinfo;

   cdat = 0x20;
   if (dat != 0x0000)
   {
      for (i = 33; i < 127; i++)
      {
         if (tocard[i] == dat)
	 {
	    cdat = i;
	    break;
	 }
      }
   }
#ifdef DEBUGPUNCH
   fprintf (stderr, "CRU %04X LDCR CNT %d DATA %02X(%c) - %04X\n",
	    devaddr, cnt, cdat, cdat, dat);
#endif

   /*
   ** Put a character in the record
   */

   devinfo->carddat[devinfo->cardndx] = cdat;
   devinfo->cardndx ++;

   /*
   ** If record full, write it out
   */

   if (devinfo->cardndx >= 80)
      punchcard (dev, devinfo);

   if (dev->intenabled)
      gendevinterrupt (dev);
}
