/***********************************************************************
*
* simtape.c - TILINE Tape IO processing for the TI 990 Simulator.
*
* Changes:
*   11/18/03   DGP   Original.
*   12/10/03   DGP   Change TPCS usage.
*   12/30/03   DGP   Added non-rom boot code.
*   05/10/04   DGP   Added DEV_NULL support.
*   07/29/04   DGP   Added tapereset function.
*   12/02/04   DGP   Correct tape read.
*   12/05/04   DGP   Correct backspace.
*   04/28/05   DGP   Changed to use switches.
*   10/29/13   DGP   Correct status on unconnected devices.
*   02/12/14   DGP   Fixed compressed (binary) load.
*   03/12/15   DGP   Added real device support for tape.
*   05/02/16   DGP   Don't set density, run with current setting.
*   05/04/16   DGP   Added tapecheck() function.
*   07/05/16   DGP   Fixed unit offline with interrupts enabled.
*   02/21/19   DGP   MacOS Darwin changes
*
* Tape record format:
*   4 bytes (little-endian): block length
*   N bytes: raw data for the record
*   4 bytes (little-endian): repeat block length
*
* Tape mark (EOF):
*   4 bytes: zero data
*
***********************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>

#if defined (WIN32)
#define __TTYROUTINES 0
#include <conio.h>
#include <windows.h>
#include <signal.h>
#endif

#if defined(UNIX)
#include <sys/time.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <pthread.h>
#if !defined(DARWIN)
#define HAVE_SYS_MTIO_H 1
#include <sys/mtio.h>
#endif
#endif

#include "simdef.h"

extern uint16 pcreg;	/* The program PC */
extern uint16 statreg;	/* The program status register */
extern uint16 wpreg;	/* The program Workspace Pointer */

extern int run;
extern int devcnt;
extern int bootfromrom;
extern uint32 memlen;
extern char view[MAXVIEW][MAXVIEWLEN+1];

extern uint8 memory[SYSMEMSIZE];
extern Device devices[MAXDEVICES];

/*
** Command words access macros
*/

#define GETCMD(d)   GETMEMB0((d)->devaddr+TPCSSTART+12) & 0xF
#define GETUNIT(d)  (GETMEMB0((d)->devaddr+TPCSSTART+12) >> 4) & 0xF

/*
** Command word control bits
*/

#define OLBIT	0x8000		/* Off Line */
#define BOTBIT	0x4000		/* Beginning Of Tape */
#define EORBIT	0x2000		/* End Of Record */
#define EOFBIT	0x1000		/* End Of File */
#define EOTBIT	0x0800		/* End Of Tape */
#define WPBIT	0x0400		/* Write Protect */
#define RWBIT	0x0200		/* Tape Rewinding */
#define TOBIT	0x0100		/* Command Time Out */

#define IDLEBIT	0x8000		/* Idle */
#define COMPBIT	0x4000		/* Complete */
#define ERRBIT	0x2000		/* Error */
#define INTBIT	0x1000		/* Interrupt Enable */
#define sp0BIT	0x0800		/* spare */
#define sp1BIT	0x0400		/* spare */
#define PEFBIT	0x0200		/* PE Format */
#define ABNBIT	0x0100		/* Abnormal Completion */
#define IPEBIT	0x0080		/* Interface Parity error */
#define ECEBIT	0x0040		/* Error Correction enabled */
#define HWEBIT	0x0020		/* Hard Error */
#define TMRBIT	0x0010		/* Tiline memory read error */
#define TTEBIT	0x0008		/* Tiline Timing error */
#define TTOBIT	0x0004		/* Tiline Timout error */
#define FMEBIT	0x0002		/* Format error */
#define TPEBIT	0x0001		/* Tape error */

#if defined(UNIX) && defined(HAVE_SYS_MTIO_H)
static struct mtop mt_weof   = { MTWEOF, 1 };
static struct mtop mt_rew    = { MTREW, 1 };
static struct mtop mt_bsr    = { MTBSR, 1 };
static struct mtop mt_fsr    = { MTFSR, 1 };
static struct mtop mt_setblk = { MTSETBLK, 0 };  /* blockize = 0 (variable) */
/*static struct mtop mt_setden = { MTSETDENSITY, 0x02 };*/ /* density = 1600 */
#endif

/***********************************************************************
* tapereadint - Read an integer.
***********************************************************************/

static int
tapereadint (FILE *fd)
{
   int r;

   r = fgetc (fd);
   r = r | (fgetc (fd) << 8);
   r = r | (fgetc (fd) << 16);
   r = r | (fgetc (fd) << 24);
   if (feof (fd)) return (TAPEEOT);
   return (r);
}

/***********************************************************************
* tapewriteint - Write an integer.
***********************************************************************/

static void
tapewriteint (FILE *fd, int v)
{
   v &= 0x7FFF;
   fputc (v & 0xFF, fd);
   fputc ((v >> 8) & 0xFF, fd);
   fputc ((v >> 16) & 0xFF, fd);
   fputc ((v >> 24) & 0xFF, fd);
}

/***********************************************************************
* tapeputmem - Put data into memory.
***********************************************************************/

static void
tapeputmem (uint32 ma, uint16 v)
{
   if (ma < memlen)
   {
      PUTMEM0 (ma, v);
   }
}

/***********************************************************************
* tapegetmem - Get data from memory.
***********************************************************************/

static uint16
tapegetmem (uint32 ma)
{
   if (ma >= memlen) return (0);

   return (GETMEM0 (ma));
}

/***********************************************************************
* tapesetchar - Set tape characteristics.
***********************************************************************/

static void 
tapesetchar (Device *udev)
{
#if defined(UNIX) && defined(HAVE_SYS_MTIO_H)
   if (udev->realdevice)
   {
      ioctl (udev->sfd, MTIOCTOP, &mt_setblk);
      /*ioctl (udev->sfd, MTIOCTOP, &mt_setden);*/
   }
#endif
}

/***********************************************************************
* taperewind - Rewind tape.
***********************************************************************/

static int 
taperewind (Device *udev)
{
   int rc = 0;

#ifdef DEBUGTAPE
   fprintf (stderr, "taperewind: \n");
#endif

   udev->info.tapeinfo.curpos = 0;
#if defined(UNIX) && defined(HAVE_SYS_MTIO_H)
   if (udev->realdevice)
      rc = ioctl (udev->sfd, MTIOCTOP, &mt_rew);
   else
#endif
      rc = fseek (udev->infd, 0, SEEK_SET);

   if (rc < 0)
   {
      sprintf (view[0], "Device %s: REWIND failed: %s",
	       udev->name, strerror (ERRNO));
      snprintf (view[1], MAXVIEWLEN, "filename: %s", udev->file);
   }

   return (rc);
}

/***********************************************************************
* tapeweof - Write EOF on tape.
***********************************************************************/

static int 
tapeweof (Device *udev)
{
   int rc = 0;

#ifdef DEBUGTAPE
   fprintf (stderr, "tapeweof: \n");
#endif

#if defined(UNIX) && defined(HAVE_SYS_MTIO_H)
   if (udev->realdevice)
      rc = ioctl (udev->sfd, MTIOCTOP, &mt_weof);
   else
#endif
      tapewriteint (udev->infd, 0);

   if (rc < 0)
   {
      sprintf (view[0], "Device %s: WRITEEOF failed: %s",
	       udev->name, strerror (ERRNO));
      snprintf (view[1], MAXVIEWLEN, "filename: %s", udev->file);
   }

   return (rc);
}

/***********************************************************************
* tapebsr - Backspace record on tape.
***********************************************************************/

static int 
tapebsr (Device *udev, int count)
{
   int rc = 0;

#ifdef DEBUGTAPEBSR
   fprintf (stderr, "tapebsr: count = %d, curpos = %ld\n",
	    count, udev->info.tapeinfo.curpos);
#endif

   if (udev->info.tapeinfo.curpos <= 0)
   {
      udev->info.tapeinfo.curpos = 0;
      count = 0;
   }
   else while (count && !rc)
   {
#if defined(UNIX) && defined(HAVE_SYS_MTIO_H)
      if (udev->realdevice)
      {
	 rc = ioctl (udev->sfd, MTIOCTOP, &mt_bsr);
      }
      else
#endif
      {
	 int reclen;

#ifdef DEBUGTAPEBSR
	 fprintf (stderr, "   cp0 = %ld\n", udev->info.tapeinfo.curpos);
#endif
	 udev->info.tapeinfo.curpos -= 4;
#ifdef DEBUGTAPEBSR
	 fprintf (stderr, "   cp1 = %ld\n", udev->info.tapeinfo.curpos);
#endif

	 if (fseek (udev->infd, udev->info.tapeinfo.curpos, SEEK_SET) < 0)
	 {
	    rc = TAPEERR;
#ifdef DEBUGTAPEBSR
	    perror ("fseek1 failed");
#endif
	    break;
	 }
	 reclen = tapereadint (udev->infd);
#ifdef DEBUGTAPEBSR
	 fprintf (stderr, "   reclen = %d\n", reclen);
#endif
	 if (reclen > 0)
	    udev->info.tapeinfo.curpos -= reclen + 4;
#ifdef DEBUGTAPEBSR
	 fprintf (stderr, "   cp2 = %ld\n", udev->info.tapeinfo.curpos);
#endif
	 if (fseek (udev->infd, udev->info.tapeinfo.curpos, SEEK_SET) < 0)
	 {
#ifdef DEBUGTAPEBSR
	    perror ("fseek2 failed");
#endif
	    rc = TAPEERR;
	    break;
	 }
      }
      count --;
   }

   if (rc < 0)
   {
      sprintf (view[0], "Device %s: BCKSKIP failed: %s",
	       udev->name, strerror (ERRNO));
      snprintf (view[1], MAXVIEWLEN, "filename: %s", udev->file);
   }

   PUTLEN (udev, count); 
   return (rc);
}

/***********************************************************************
* tapefsr - Forwardspace record on tape.
***********************************************************************/

static int 
tapefsr (Device *udev, int count)
{
   int rc = 0;

#ifdef DEBUGTAPE
   fprintf (stderr, "tapefsr: count = %d\n", count);
#endif

   while (count && !rc)
   {
#if defined(UNIX) && defined(HAVE_SYS_MTIO_H)
      if (udev->realdevice)
      {
	 rc = ioctl (udev->sfd, MTIOCTOP, &mt_fsr);
	 if (rc < 0 && ERRNO == ENOSPC)
	    rc = TAPEEOT;
      }
      else
#endif
      {
	 int i;
	 int reclen;

	 reclen = tapereadint (udev->infd);
	 if (reclen > 0)
	 {
	    for (i = 0; i < reclen; i++)
	    {
	       fgetc (udev->infd);
	    }
	    i = tapereadint (udev->infd);
	    if (i != reclen)
	    {
	       rc = TAPEERR;
	    }
	 }
	 else if (reclen == 0)
	 {
	    rc = TAPEEOF;
	 }
	 else if (reclen == -2)
	 {
	    rc = TAPEEOT;
	 }
      }
      count--;
   }

   if (rc < 0)
   {
      sprintf (view[0], "Device %s: FWDSKIP failed: %s",
	       udev->name, strerror (ERRNO));
      snprintf (view[1], MAXVIEWLEN, "filename: %s", udev->file);
   }

   PUTLEN (udev, count); 
   return (rc);
}

/***********************************************************************
* taperead - Read a record from tape.
***********************************************************************/

static int 
taperead (Device *udev, uint32 addr, int buflen)
{
   int rc = 0;
   int reclen;
   int i, j;


#ifdef UNIX
   if (udev->realdevice)
   {
     reclen = read (udev->sfd, udev->info.tapeinfo.record, TAPEMAXREC);
   }
   else
#endif
   {
      reclen = tapereadint (udev->infd);
      if (reclen > 0)
      {
	 fread (udev->info.tapeinfo.record, 1, reclen, udev->infd);
	 i = tapereadint (udev->infd);
	 if (i != reclen)
	 {
#ifdef DEBUGTAPE
	    fprintf (stderr, "taperead: error: reclen = %d\n", reclen);
	    fprintf (stderr, "   traillen = %d\n", i);
#endif
	    rc = TAPEERR;
	 }
      }
   }

#ifdef DEBUGTAPE
   fprintf (stderr, "taperead: buflen = %d, reclen = %d\n", buflen, reclen);
   if (!udev->realdevice)
      fprintf (stderr, "   cp = %ld\n", udev->info.tapeinfo.curpos);
#endif

   if (!rc && reclen > 0)
   {
      for (j = i = 0; i < reclen;)
      {
	 uint16 d, c;

	 d = udev->info.tapeinfo.record[i++];
	 if (i < reclen)
	 {
	    c = udev->info.tapeinfo.record[i++];
	 }
	 else
	    c = 0xFF;
	 d = (d << 8) | c;
	 if (j < buflen)
	 {
	    tapeputmem (addr, d);
	    j += 2;
	    addr += 2;
	 }
      }
      if (i < buflen)
      {
	 buflen -= i;
      }
      else
      {
	 buflen = 0;
      }
      rc = buflen;

#ifdef DEBUGTAPEDATA
      HEXDUMP (stderr, &memory[(((GETADDR (udev)) + 1) & 0x1FFFFE)],
	       j, (((GETADDR (udev)) + 1) & 0x1FFFFE));
#endif
      PUTLEN (udev, buflen); 
      PUTADDR (udev, addr);
   }
   else if (reclen == 0)
   {
      rc = TAPEEOF;
   }
   else if (reclen == -2)
   {
      rc = TAPEEOT;
   }

   return (rc);
}

/***********************************************************************
* tapewrite - Write a record to tape.
***********************************************************************/

static int 
tapewrite (Device *udev, uint32 addr, int buflen)
{
   int rc = 0;
   int i = 0;

#ifdef DEBUGTAPE
   fprintf (stderr, "tapewrite: buflen = %d\n", buflen);
#endif
   if (!(udev->switches & SWPROTECT))
   {
#ifdef DEBUGTAPEDATA
      HEXDUMP (stderr, &memory[(((GETADDR (udev)) + 1) & 0x1FFFFE)],
	       GETLEN(udev), (((GETADDR (udev)) + 1) & 0x1FFFFE));
#endif
      for (i = 0; i < buflen; )
      {
	 uint16 d;

	 d = tapegetmem (addr);
	 udev->info.tapeinfo.record[i++] =  (d >> 8) & 0xFF;
	 if (i < buflen)
	 {
	    udev->info.tapeinfo.record[i++] = d & 0xFF;
	 }
	 addr += 2;
      }

#ifdef UNIX
      if (udev->realdevice)
      {
         i = write (udev->sfd, udev->info.tapeinfo.record, buflen);
      }
      else
#endif
      {
	 tapewriteint (udev->infd, buflen);
	 i = fwrite (udev->info.tapeinfo.record, 1, buflen, udev->infd);
	 tapewriteint (udev->infd, buflen);
      }
      if (i != buflen)
	 rc = TAPEERR;
   }
   PUTLEN (udev, buflen-i);
   PUTADDR (udev, addr);
   return (rc);
}

/***********************************************************************
* tapecheck - Check tape for valid format.
***********************************************************************/

static int 
tapecheck (Device *dev)
{
   int rc;
   int rlen;

   if ((rc = taperewind (dev)) == 0)
   {
      rlen = tapereadint (dev->infd);
      if ((rc = fseek (dev->infd, rlen + 4, SEEK_SET)) == 0)
      {
	 if (rlen != tapereadint(dev->infd))
	    rc = -1;
      }
   }
   if (rc < 0)
   {
      sprintf (view[0], "Device %s: Invalid tape format", dev->name);
      snprintf (view[1], MAXVIEWLEN, "filename: %s", dev->file);
   }
   return (rc);
}

/***********************************************************************
* tapeopen - Open tape.
***********************************************************************/

int 
tapeopen (Device *dev)
{
   int rc;

#ifdef DEBUGTAPE
   fprintf (stderr, "tapeopen: ENTERED\n");
   fprintf (stderr, "   file = %s\n", dev->file);
#endif

   if (dev->infd == DEV_NULL)
   {
      PUTW0 (dev, OLBIT);
      PUTW7 (dev, IDLEBIT | ERRBIT | ABNBIT); /* Mark idle */
      return (0);
   }

   if (!dev->realdevice)
   {
      /*
      ** Check to see if a valid tape file.
      */
      if (tapecheck (dev) < 0)
	 return (-1);
   }
   else
      dev->sfd = fileno (dev->infd);

   /*
   ** Rewind the tape.
   */

   rc = taperewind (dev);
   tapesetchar (dev);

   PUTW0 (dev, BOTBIT | ((dev->switches & SWPROTECT) ? WPBIT : 0));
   PUTW7 (dev, IDLEBIT); /* Mark idle */

   return (rc);
}

/***********************************************************************
* tapereset - Reset device
***********************************************************************/

void
tapereset (Device *dev)
{
#ifdef DEBUGTAPE
   fprintf (stderr, "tapereset: ENTERED\n");
#endif

   if (dev->infd == DEV_NULL)
   {
      PUTW0 (dev, OLBIT);
      PUTW7 (dev, IDLEBIT | ERRBIT | ABNBIT ); 
   }
   else
   {
      PUTW0 (dev, BOTBIT | ((dev->switches & SWPROTECT) ? WPBIT : 0));
      PUTW7 (dev, IDLEBIT); 
   }

   return;
}

/***********************************************************************
* tapedocmd - Do tape command
***********************************************************************/

int 
tapedocmd (Device *dev)
{
   Device *udev;
   int cmd;
   int i;
   int len;
   uint32 addr;
   uint16 W0, W7;
   uint16 delay;
   uint8 unit;

   /*
   ** Get common command word values
   */

   W0 = GETW0 (dev);
   W7 = GETW7 (dev);

   unit = GETUNIT (dev);
   addr = ((GETADDR (dev)) + 1) & 0x1FFFFE;
   len = GETLEN(dev);

#ifdef DEBUGTAPE
   cmd = GETCMD (dev);
   fprintf (stderr,
	    "tapedocmd: ENTRY: cmd = %d, unit = %02X, addr = %06X, len = %d\n",
	    cmd, unit, addr, len);
   fprintf (stderr, "   pc = %04X, wp = %04X, st = %04X\n",
	    pcreg, wpreg, statreg);
   for (i = 0; i < 16; i+=2)
      fprintf (stderr, "   ctl[%04X] %04X\n",
	       dev->devaddr+TPCSSTART+i, GETMEM0 (dev->devaddr+TPCSSTART+i));
#endif

   /*
   ** Find selected unit
   */

   for (i = 0; i < devcnt; i++) /* find selected unit */
      if (devices[i].devaddr == dev->devaddr && devices[i].unit == unit)
	 break;
   if (i == devcnt)
   {
#ifdef DEBUGTAPE
      fprintf (stderr, "tapedocmd: unit = %d: Unit not selected\n", unit);
#endif
      /* Drive not ready */
      if (W7 & INTBIT)
	 dev->intdelay = 10;
      W0 |= OLBIT | ((dev->switches & SWPROTECT) ? WPBIT : 0);
      W7 |= ABNBIT | ERRBIT | IDLEBIT;
      PUTW0 (dev, W0);
      PUTW7 (dev, W7);
      return (0);
   }
   udev = &devices[i];

   if (udev->infd == DEV_NULL)
   {
#ifdef DEBUGTAPE
      fprintf (stderr, "tapedocmd: unit = %d: Unit is NULL\n", unit);
#endif
      if (W7 & INTBIT)
	 udev->intdelay = 10;
      W7 |= ABNBIT | ERRBIT | IDLEBIT;
      PUTW0 (udev, OLBIT );
      PUTW7 (udev, W7);
      return (0);
   }

   /*
   ** Set initial sate
   */

   udev->select = TRUE;
   udev->intenabled = FALSE;
   if (!udev->realdevice)
      udev->info.tapeinfo.curpos = ftell (udev->infd);
   cmd = GETCMD (udev);

   /*
   ** Process command
   */

   delay = 50;
   switch (cmd)
   {
   case  0: /* NOP */
   case 12: /* NOP */
   case 13: /* NOP */
   case 14: /* NOP */
      delay = 1;
      break;

   case 1: /* Buffer Sync */
#ifdef DEBUGTAPE
      fprintf (stderr, "tapedocmd: BUFFSYNC: \n");
#endif
      delay = 10;
      break;

   case 2: /* Write EOF */
      if (!(dev->switches & SWPROTECT))
      {
	 tapeweof (udev);
	 W0 |= EOTBIT | OLBIT;
      }
      break;

   case 3: /* Record skip backward */
      if (tapebsr (udev, len) < 0)
         return (0);
      if (udev->info.tapeinfo.curpos == 0)
      {
	 W0 |= (udev->unit << 4) | BOTBIT;
	 W7 |= COMPBIT | TPEBIT;
      }
      break;

   case 4: /* Binary read data */
      if ((i = taperead (udev, addr, len)) < 0)
      {
	 W7 |= ERRBIT;
	 if (i == TAPEEOF)
	 {
	    W7 |= TPEBIT;
	    W0 |= EOFBIT;
	 }
	 else if (i == TAPEEOT)
	 {
	    W7 |= TPEBIT;
	    W0 |= EOTBIT;
	 }
      }
      else if (i > 0)
      {
         W0 |= EORBIT;
	 W7 |= TPEBIT;
      }
      delay = 100 + len;
      break;

   case 5: /* Record skip forward */
      if ((i = tapefsr (udev, len) < 0))
      {
	 W7 |= ERRBIT;
	 if (i == TAPEEOF)
	 {
	    W7 |= TPEBIT;
	    W0 |= EOFBIT;
	 }
	 else if (i == TAPEEOT)
	 {
	    W7 |= TPEBIT;
	    W0 |= EOTBIT;
	 }
      }
      break;

   case 6: /* Binary write data */
      tapewrite (udev, addr, len);
      delay = 100 + len;
      break;

   case 7: /* Erase */
#ifdef DEBUGTAPE
      fprintf (stderr, "tapedocmd: ERASE: \n");
#endif
      break;

   case 8: /* Read transport status */
   case 9: /* Read transport status */
#ifdef DEBUGTAPE
      fprintf (stderr, "tapedocmd: READST: \n");
#endif
      if (udev->info.tapeinfo.curpos == 0)
	 W0 |= BOTBIT;
      break;

   case 10: /* Rewind */
      if (taperewind (udev) < 0)
         return (0);
      W0 |= (udev->unit << 4) | BOTBIT;
      W7 |= COMPBIT | TPEBIT;
      break;

   case 11: /* Rewind and offline */
#ifdef DEBUGTAPE
      fprintf (stderr, "tapedocmd: REWOFFLINE: \n");
#endif
      if (taperewind (udev) < 0)
         return (0);
      W0 |= (udev->unit << 4) | BOTBIT | OLBIT;
      W7 |= COMPBIT | TPEBIT;
      delay = 200;
      break;
   }

   if (W7 & INTBIT)
      udev->intenabled = TRUE;
   W7 |= COMPBIT | IDLEBIT;
   W0 |= ((dev->switches & SWPROTECT) ? WPBIT : 0);
   PUTW0 (udev, W0);
   PUTW7 (udev, W7);
#ifdef DEBUGTAPE
   fprintf (stderr, "returns: intenabled = %s, delay = %d\n",
	    udev->intenabled ? "TRUE" : "FALSE", delay);
   for (i = 0; i < 16; i+=2)
      fprintf (stderr, "   ctl[%04X] %04X\n",
	       udev->devaddr+TPCSSTART+i, GETMEM0 (udev->devaddr+TPCSSTART+i));
#endif

   if (udev->intenabled)
      udev->intdelay = delay;
   return (0);
}


/***********************************************************************
* tapeboot - Boot from device.
***********************************************************************/

int
tapeboot (Device *dev)
{
   if (bootfromrom)
   {
      wpreg = GETMEM0 (TPCSSTART+0xFFFC) & 0xFFFE;
      pcreg = GETMEM0 (TPCSSTART+0xFFFE) & 0xFFFE;

      PUTREG (0, 0xFFFF);
      PUTREG (1, dev->devaddr);
      PUTREG (2, dev->unit << 8);
   }
   else 
   {
      FILE *lfd = NULL;
      int lp, clp;
      int i;
      int done;
      int binarymode;
      int reclen;
      uint8 inbuf[82];

      clp = lp = LOADADDRESS;

#ifdef DEBUGLOADER
      lfd = fopen ("tapeload.log", "w");
      fprintf (lfd, "tapeboot: file = %s, loadpd = %d(%04X)\n",
	       dev->file, lp, lp);
#endif

      if (taperewind (dev) < 0)
	 return (-1);

      done = FALSE;
      binarymode = FALSE;
      for (i = 0; !done; i++)
      {
#ifdef UNIX
	 if (dev->realdevice)
	    reclen = read (dev->sfd, inbuf, sizeof (inbuf));
	 else
#endif
	 {
	    if ((reclen = tapereadint (dev->infd)) == 80)
	    {
	       if (fread ((char *)inbuf, 1, reclen, dev->infd) != reclen)
	       {
		  reclen = 0;
	       }
	       else if (tapereadint (dev->infd) != reclen)
	       {
		  reclen = 0;
	       }
	    }
	 }
	 if (reclen != 80)
	    break;

	 inbuf[reclen] = 0;
#if defined(DUMPLOADREC) && defined(DEBUGLOADER)
	 fprintf (lfd, "Record: %d\n", i+1);
	 HEXDUMP (lfd, inbuf, reclen, 0);
#endif
	 if (inbuf[0] == BINIDT_TAG)
	    binarymode = TRUE;
	 if (loadrec (lfd, inbuf, lp, &clp, binarymode) == EOF)
	    done = TRUE;
      }

#ifdef DEBUGLOADER
      fclose (lfd);
#endif
      if (reclen != 80)
      {
	 sprintf (view[0], "Device %s: read failed: %s",
		  dev->name, strerror (ERRNO));
	 snprintf (view[1], MAXVIEWLEN, "filename: %s", dev->file);
         run = FALSE;
	 return (-1);
      }
   }

   run = TRUE;

   return (0);
}
