/***********************************************************************
*
* simterm.c - Terminal support for the TI 990 Simulator.
*
* Changes:
*   02/27/14   DGP   Split from simio.c
*   04/04/14   DGP   Added general telnet support for serial devices.
*   05/01/15   DGP   Added ASR 733 emulation support.
*   06/02/15   DGP   Removed USS (z/OS) support.
*   06/09/15   DGP   Added WIN32 function key processing for console.
*   01/05/17   DGP   Added real device support.
*   05/18/20   DGP   Remove Ctrl-C change in serialgetchar.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>
#include <time.h>

#if defined (WIN32)
#define __TTYROUTINES 0
#include <conio.h>
#include <windows.h>
#include <signal.h>
#include <process.h>
#endif

#if defined(UNIX)
#include <unistd.h>
#include <sys/time.h>
#include <sys/param.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <pthread.h>
#endif

#include "simdef.h"
#include "screen.h"

extern int run;
extern int prtviewlen;
extern int prtviewcol;
extern int enablepanel;
extern int windowlen;
extern int viewlen;
extern int maxdevcnt;
extern int showstatus;

extern Device devices[MAXDEVICES];
extern char prtview[MAXVIEW][MAXVIEWLEN+1];
extern char view[MAXVIEW][MAXVIEWLEN+1];
extern MUTEX_DEFINE(panel_mutex);

static int sim_int_char = 005;

/***********************************************************************
* Terminal routines borrowed from Bub Supnik's simh
***********************************************************************/

/*
** Windows routines
*/

#if defined(WIN32)
static volatile int sim_win_ctlc = 0;

static void
win_handler (int sig)
{
   sim_win_ctlc = 1;
   return;
}

int
ttraw (int fd, int clocal)
{
   return 0;
}

int
ttgetattr (int fd, void *attr)
{
   return 0;
}

int
ttsetattr (int fd, void *attr)
{
   return 0;
}

int
ttinit (void)
{
   return 0;
}

int
ttrunstate (void)
{
   if (signal (SIGINT, win_handler) == SIG_ERR) return -1;
   SetThreadPriority (GetCurrentThread (), THREAD_PRIORITY_BELOW_NORMAL);
   return 0;
}

int
ttcmdstate (void)
{
   SetThreadPriority (GetCurrentThread (), THREAD_PRIORITY_NORMAL);
   return 0;
}

int
ttclose (void)
{
   return 0;
}

static int
os_getfunc (Device *dev, int c)
{
   int c1 = _getch();

   dev->charqueue[0] = 0;
   switch (c)
   {
   case 0x00:
      switch (c1)
      {
      case 0x3B: /* F1 */
         strcpy (dev->charqueue, "OP");
	 break;
      case 0x3C: /* F2 */
         strcpy (dev->charqueue, "OQ");
	 break;
      case 0x3D: /* F3 */
         strcpy (dev->charqueue, "OR");
	 break;
      case 0x3E: /* F4 */
         strcpy (dev->charqueue, "OS");
	 break;
      case 0x3F: /* F5 */
         strcpy (dev->charqueue, "[15~");
	 break;
      case 0x40: /* F6 */
         strcpy (dev->charqueue, "[17~");
	 break;
      case 0x41: /* F7 */
         strcpy (dev->charqueue, "[18~");
	 break;
      case 0x42: /* F8 */
         strcpy (dev->charqueue, "[19~");
	 break;
      case 0x43: /* F9 */
         strcpy (dev->charqueue, "[20~");
	 break;
      case 0x44: /* F10 */
         strcpy (dev->charqueue, "[21~");
	 break;
      default:
         return (c);
      }
      dev->charqueuecnt = strlen (dev->charqueue);
      dev->charqueuendx = 0;
      c = ESCAPE;
      break;
   case 0xE0:
      switch (c1)
      {
      case 0x47: /* HOME */
         strcpy (dev->charqueue, "[1~");
	 break;
      case 0x49: /* Page Up */
         strcpy (dev->charqueue, "[5~");
	 break;
      case 0x48: /* Up Arrow*/
         strcpy (dev->charqueue, "[A");
	 break;
      case 0x4B: /* Left Arrow*/
         strcpy (dev->charqueue, "[D");
	 break;
      case 0x4D: /* Right Arrow*/
         strcpy (dev->charqueue, "[C");
	 break;
      case 0x4F: /* END */
         strcpy (dev->charqueue, "OF");
	 break;
      case 0x50: /* Down Arrow*/
         strcpy (dev->charqueue, "[B");
	 break;
      case 0x51: /* Page Down */
         strcpy (dev->charqueue, "[6~");
	 break;
      case 0x52: /* INSERT */
         strcpy (dev->charqueue, "[2~");
	 break;
      case 0x53: /* DELETE */
         strcpy (dev->charqueue, "[3~");
	 break;
      case 0x85: /* F11 */
         strcpy (dev->charqueue, "[23~");
	 break;
      case 0x86: /* F12 */
         strcpy (dev->charqueue, "[24~");
	 break;
      default:
         return (c);
      }
      dev->charqueuecnt = strlen (dev->charqueue);
      dev->charqueuendx = 0;
      c = ESCAPE;
      break;
   default: ;
   }
   return (c);
}

static int
os_getchar (Device *dev)
{
   int c;

   if (dev->charqueuecnt)
   {
      c = dev->charqueue[dev->charqueuendx++];
      dev->charqueuecnt--;
      return (c);
   }
   if (sim_win_ctlc)
   {
      sim_win_ctlc = 0;
      signal (SIGINT, win_handler);
      return 003;
   }
   if (!kbhit ()) return EOF;
   c = _getch ();

   if (c == 0x00 || c == 0xE0)
      c = os_getfunc (dev, c);
   if ((c & 0177) == '\b') c = 0177;
   if ((c & 0177) == sim_int_char)
   {
      int i;
      for (i = 0; i < maxdevcnt; i++)
      {
	 Device *dev = &devices[i];

	 if ((dev->switches & SWEMUTERM) && (dev->emutype < EMU733) &&
	     (dev->infd == stdin))
	    screenposition (TRUE, VDT_NUM_ROWS + 1, 1);
      }
      run = FALSE;
      showstatus = TRUE;
      return EOF;
   }
   return c;
}

static int
os_putchar (uint8 c)
{
   c &= 0x7F;
   if (c != 0x7F && c != 0)
      _putch (c);
   return 0;
}
#endif

/*
** Unix routines
*/

#if defined(UNIX)
static struct termios cmdtty, runtty;

int
ttraw (int fd, int clocal)
{
   struct termios rawtty;

   if (!isatty (fd))
      return (-1);

   tcgetattr (fd, &rawtty);

   rawtty.c_iflag = (IGNBRK | IGNPAR);
   rawtty.c_lflag = 0;
   rawtty.c_oflag = 0;
   rawtty.c_cflag = (CRTSCTS | HUPCL | CREAD);
   rawtty.c_cflag |= (B9600 | CS8);
   if (clocal) 
      rawtty.c_cflag |= CLOCAL;
   else
      rawtty.c_cflag &= ~CLOCAL;
   rawtty.c_cc[VMIN] = 1;
   rawtty.c_cc[VTIME] = 0;
   rawtty.c_cc[VINTR] = 0;
   rawtty.c_cc[VQUIT] = 0;
   rawtty.c_cc[VERASE] = 0;
   rawtty.c_cc[VKILL] = 0;
   rawtty.c_cc[VEOF] = 0;
   rawtty.c_cc[VEOL] = 0;
   rawtty.c_cc[VSTART] = 0;
   rawtty.c_cc[VSUSP] = 0;
   rawtty.c_cc[VSTOP] = 0;
#if defined (VREPRINT)
   rawtty.c_cc[VREPRINT] = 0;
#endif
#if defined (VDISCARD)
   rawtty.c_cc[VDISCARD] = 0;
#endif
#if defined (VWERASE)
   rawtty.c_cc[VWERASE] = 0;
#endif
#if defined (VLNEXT)
   rawtty.c_cc[VLNEXT] = 0;
#endif
#if defined (VDSUSP)
   rawtty.c_cc[VDSUSP] = 0;
#endif
#if defined (VSTATUS)
   rawtty.c_cc[VSTATUS] = 0;
#endif
   if (tcsetattr (fd, TCSANOW, &rawtty) < 0)
      return (-1);

   return 0;
}

int
ttgetattr (int fd, void *attr)
{
   if (!isatty (fd))
      return (-1);

   return (tcgetattr (fd, (struct termios *)attr));
}

int
ttsetattr (int fd, void *attr)
{
   if (!isatty (fd))
      return (-1);

   return (tcsetattr (fd, TCSANOW, (struct termios *)attr));
}


int
ttinit (void)
{
   if (!isatty (fileno (stdin))) return 0;		/* skip if !tty */
   if (tcgetattr (0, &cmdtty) < 0) return -1;		/* get old flags */
   runtty = cmdtty;
   runtty.c_lflag = runtty.c_lflag & ~(ECHO | ICANON);	/* no echo or edit */
   runtty.c_oflag = runtty.c_oflag & ~OPOST;		/* no output edit */
   runtty.c_iflag = runtty.c_iflag & ~ICRNL;		/* no cr conversion */
   runtty.c_cc[VINTR] = sim_int_char;			/* interrupt */
   runtty.c_cc[VQUIT] = 0;				/* no quit */
   runtty.c_cc[VERASE] = 0;
   runtty.c_cc[VKILL] = 0;
   runtty.c_cc[VEOF] = 0;
   runtty.c_cc[VEOL] = 0;
   runtty.c_cc[VSTART] = 0;				/* no host sync */
   runtty.c_cc[VSUSP] = 0;
   runtty.c_cc[VSTOP] = 0;
#if defined (VREPRINT)
   runtty.c_cc[VREPRINT] = 0;				/* no specials */
#endif
#if defined (VDISCARD)
   runtty.c_cc[VDISCARD] = 0;
#endif
#if defined (VWERASE)
   runtty.c_cc[VWERASE] = 0;
#endif
#if defined (VLNEXT)
   runtty.c_cc[VLNEXT] = 0;
#endif
   runtty.c_cc[VMIN] = 0;				/* no waiting */
   runtty.c_cc[VTIME] = 0;
#if defined (VDSUSP)
   runtty.c_cc[VDSUSP] = 0;
#endif
#if defined (VSTATUS)
   runtty.c_cc[VSTATUS] = 0;
#endif
   return 0;
}

int
ttrunstate (void)
{
   if (!isatty (fileno (stdin))) return 0;		/* skip if !tty */
   runtty.c_cc[VINTR] = sim_int_char;			/* in case changed */
   if (tcsetattr (0, TCSAFLUSH, &runtty) < 0) return -1;
   return 0;
}

int
ttcmdstate (void)
{
   if (!isatty (fileno (stdin))) return 0;		/* skip if !tty */
   if (tcsetattr (0, TCSAFLUSH, &cmdtty) < 0) return -1;
   return 0;
}

int
ttclose (void)
{
   return ttcmdstate ();
}

static int
os_getchar (Device *dev)
{
   int status;
   unsigned char buf[1];

   status = read (fileno(dev->infd), buf, 1);

   if (status != 1) return EOF;
   if (buf[0] == sim_int_char)
   {
      int i;
      for (i = 0; i < maxdevcnt; i++)
      {
	 Device *dev = &devices[i];

	 if ((dev->switches & SWEMUTERM) && (dev->emutype < EMU733) &&
	     (dev->infd == stdin))
	    screenposition (TRUE, VDT_NUM_ROWS + 1, 1);
      }
      run = FALSE;
      showstatus = TRUE;
      return (EOF);
   }
   return (buf[0]);
}

static int
os_putchar (uint8 out)
{
   char c;

   c = out & 0x7F;
   if (c != 0x7F && c != 0)
      write (1, &c, 1);
   return 0;
}
#endif

/***********************************************************************
* prtpanel - Prints panel messages on screen.
***********************************************************************/

void
prtpanel (int index, int refresh)
{
   int i, j;

   MUTEX_LOCK (panel_mutex);

   prtviewcol = index + 1;
   if (refresh || index == 0)
   {
      for (j = 0; j < prtviewlen; j++)
      {
	 screenposition (enablepanel, OUTPUTLINE+j,1);
	 clearline (enablepanel);
	 if (!refresh)
	    strcpy (prtview[j], prtview[j+1]);
	 printf ("%s", prtview[j]);
      }
#ifdef DEBUGCONPANEL
      fprintf (stderr, "prtpanel: \n");
      fprintf (stderr, "%s", prtview[j-2]);
      HEXDUMP (stderr, prtview[j-2], 80, 0);
#endif
   }

   i = prtviewlen - 1;
   screenposition (enablepanel, OUTPUTLINE+i,1);
   clearline (enablepanel);
   printf ("%s", prtview[i]);
   screenposition (enablepanel, OUTPUTLINE+i,prtviewcol);

   MUTEX_UNLOCK (panel_mutex);
}

/***********************************************************************
* putpanel - Puts chars in the "panel"
***********************************************************************/

static void
putpanel (Device *dev, uint8 ch)
{
   ch &= 0x7F;
   if (ch > 0)
   {
      if (ch == 0x08)
      {
	 dev->viewndx--;
      }
      else if (ch > 0x1F)
      {
	 prtview[prtviewlen - 1][dev->viewndx++] = ch; 
      }
      if (ch != 0x0D)
      {
	 prtview[prtviewlen - 1][dev->viewndx] = '\0';
	 if (dev->viewndx > MAXVIEWLEN || ch == 0x0A)
	 {
	    dev->viewndx = 0;
	 }
	 prtpanel (dev->viewndx, FALSE);
      }
   }
}

/***********************************************************************
* putview - Puts messages on screen with continuation prompts.
***********************************************************************/

int
putview (int interactive, int wait)
{
   int i, j;
   int ch;

   if (interactive && enablepanel)
      for (j = 0; j < viewlen; j++)
      {
	 screenposition (enablepanel, OUTPUTLINE+j, 1);
	 clearline (enablepanel);
      }

   for (j = i = 0; i < MAXVIEW; i++)
   {
      if (interactive && enablepanel)
      {
	 j++;
	 if (j >= viewlen)
	 {
	    screenposition (enablepanel, windowlen-2, 1);
	    clearline (enablepanel);
	    printf ("press enter to continue:");
	    fflush (stdout);

	    ch = fgetc (stdin);
	    if (ch == 'q')
	    {
	       while ((ch = fgetc (stdin)) != '\n') ;
	       return (0);
	    }

	    if (interactive && enablepanel)
	       for (j = 0; j < viewlen; j++)
	       {
		  screenposition (enablepanel, OUTPUTLINE+j, 1);
		  clearline (enablepanel);
	       }
	    j = 0;
	 }

	 screenposition (enablepanel, OUTPUTLINE+j, 1);
	 clearline (enablepanel);
      }
      if (view[i][0])
      {
	 printf ("%s", view[i]);
	 if (!enablepanel) printf ("\n");
      }
      else return (0);
      view[i][0] = '\0';
   }
   fflush (stdout);

   if (wait && interactive && enablepanel)
   {
      screenposition (enablepanel, windowlen-2, 1);
      clearline (enablepanel);
      printf ("press enter to continue:");
      fflush (stdout);

      ch = fgetc (stdin);
      if (ch == 'Q' || ch == 'q') return (1);
   }
   return (0);
}

/***********************************************************************
* serialgetchar - Gets a char from a non-Telnet serial device.
***********************************************************************/

int
serialgetchar (Device *dev)
{
   int c;

   if (dev->infd == stdin)
   {
      c = os_getchar (dev);
   }
   else if (dev->realdevice)
   {
      unsigned char bf[2];

      read (fileno(dev->infd), bf, 1);
      c = bf[0] & 0xFF;
#ifdef DEBUGREALTTY
      fprintf (stderr, "serialgetchar: c = >%02X(%c)\n", 
	       c, isprint (c) ? c : '.');
#endif
   }
   else
   {
      c = fgetc (dev->infd);
   }

   return (c);
}

/***********************************************************************
* serialputchar - Puts a char on non-Telnet serial device.
***********************************************************************/

void
serialputchar (Device *dev, uint16 dat)
{
   uint8 ch;

   ch = (uint8)dat;

#ifdef DEBUGSERIALPUT
   fprintf (stderr, "serialputchar: c = >%02X(%c)\n", dat,
	    isprint (dat) ? dat : '.');
#endif

   if (enablepanel && dev->outfd == stdout)
   {
      putpanel (dev, ch);
   }
   else
   {
      if (dev->outfd == stdout)
      {
	 os_putchar (ch);
      }
      else if (dev->realdevice)
      {
         unsigned char bf[2];
#ifdef DEBUGREALTTY
	 fprintf (stderr, "serialputchar: c = >%02X(%c)\n", 
		  ch, isprint (ch) ? ch : '.');
#endif
	 bf[0] = ch;
         write (fileno(dev->outfd), bf, 1);
      }
      else
      {
	 fputc (ch, dev->outfd);
      }
   }
   fflush (dev->outfd);
}

/***********************************************************************
* serialinput - Input character processing for a Serial terminal.
***********************************************************************/

int
serialinput (Device *dev)
{
   TermInfo *terminfo;
   int dat;

   terminfo = &dev->info.terminfo;

   if ((dat = termdgetring (dev, &terminfo->inring)) == EOF)
      smallsleep (10);

#ifdef DEBUGSERIALIN
   fprintf (stderr, "serialinput: c = >%02X(%c)\n", dat,
	    isprint (dat) ? dat : '.');
#endif

   return (dat);
}

/***********************************************************************
* serialoutput - Ouput character processing for a Serial terminal.
***********************************************************************/

void
serialoutput (Device *dev, uint16 dat)
{
   TermInfo *terminfo;

#ifdef DEBUGSERIALOUT
   fprintf (stderr, "serialoutput: c = >%02X(%c)\n", dat,
	    isprint (dat) ? dat : '.');
#endif

   terminfo = &dev->info.terminfo;
   if (dev->outfd == DEV_LOOP)
   {
      termdputring (dev, &terminfo->inring, dat);
      return;
   }

   if (!dev->realdevice && dat == 0) return;

   if (dev->emutype)
      termdputring (dev, &terminfo->prcring, dat);
   else
      termdputring (dev, &terminfo->outring, dat);
}
