/***********************************************************************
*
* simtermd.c - Terminal processing daemon for the TI 990 Simulator.
*
* Changes:
*   04/02/14   DGP   Original.
*   04/04/14   DGP   Added keyboard ring.
*   07/30/14   DGP   Added termdstop to flush terminals.
*   07/31/14   DGP   Added logic to terminate the termdtask.
*   01/29/15   DGP   Removed resetting of the Comm settings.
*   06/01/15   DGP   Fixed to not start terminal task if device is NULL.
*   06/02/15   DGP   Removed USS (z/OS) support.
*   06/12/15   DGP   Changed BACKSPACE and DBACKSPACE to generate left arrow.
*   02/18/16   DGP   Added SO_LINGER to clear socket on disconnect.
*   05/19/16   DGP   Eat input linefeed (PuTTY).
*   01/05/17   DGP   Added real device support.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>
#include <signal.h>

#if defined (WIN32)
#define __TTYROUTINES 0
#include <conio.h>
#include <windows.h>
#include <signal.h>
#include <process.h>
#include <winsock.h>
#endif

#if defined(UNIX)
#include <unistd.h>
#include <sys/time.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <arpa/telnet.h>
#include <pthread.h>
#endif

#ifndef IAC
#define IAC 255
#define DONT 254
#define WILL 251
#define TELOPT_ECHO 1
#define TELOPT_SGA 3
#endif

#include "simdef.h"

#define DEPTH 1

extern int run;
extern uint16 statreg;	/* The program status register */
extern char view[MAXVIEW][MAXVIEWLEN+1];
extern char *emulations[];

static uint8 will[] = {IAC,WILL,'%','c',0};
static uint8 clear[] = {ESCAPE,'[','2','J',ESCAPE,'[','1',';','1','H'};

/***********************************************************************
* Nwrite - Network write.
***********************************************************************/

int
Nwrite (int fd, uint8 *buf, int count)
{
   while (count > 0)
   {
      int wcnt;
      if ((wcnt = send (fd, buf, count, 0)) < 0)
      {
         return (-1);
      }
      buf += wcnt;
      count -= wcnt;
   }
   return (0);
}

/***********************************************************************
* Nread - Network read.
***********************************************************************/

int
Nread (int fd, uint8 *buf, int count)
{
   int cnt;

   cnt = recv (fd, buf, count, 0);
   return (cnt);
}

/***********************************************************************
* termdringinit - Initialize a ring.
***********************************************************************/

static void
termdringinit (Device *dev, TermRing *ring)
{
   ring->head = 0;
   ring->tail = 0;

   if (MUTEX_INIT (ring->mutex))
   {
      sprintf (view[0], "Device %s: ring mutex init failed", dev->name);
#ifdef DEBUGTERMDTASK
      fprintf (stderr, "%s\n", view[0]);
#endif
   }
}

/***********************************************************************
* termdringempty - Is ring empty.
***********************************************************************/

int
termdringempty (Device *dev, TermRing *ring)
{
   int status;

   status = FALSE;

   MUTEX_LOCK (ring->mutex);

   if (ring->head == ring->tail)
      status = TRUE;

#ifdef DEBUGTERMDRINGE
   fprintf (stderr, "termdringempty: ring = %s, empty = %s\n",
	    ring == &dev->info.terminfo.inring ? "IN " :
	    ring == &dev->info.terminfo.outring ? "OUT" : "PRC",
	    status ? "YES" : "NO");
#endif

   MUTEX_UNLOCK (ring->mutex);
   return (status);
}

/***********************************************************************
* termdputring - Put char in the ring buffer.
***********************************************************************/

void
termdputring (Device *dev, TermRing *ring, int ch)
{

   ch &= 0xFF;
   MUTEX_LOCK (ring->mutex);

#ifdef DEBUGTERMDRING
   fprintf (stderr,
	 "termdputring: ring = %s, head = %d, tail = %d, data = >%02X(%c)\n",
	    ring == &dev->info.terminfo.inring ? "IN " :
	    ring == &dev->info.terminfo.outring ? "OUT" : "PRC",
            ring->head, ring->tail, ch, isprint(ch) ? ch : '.');
#endif

   ring->data[ring->head++] = ch;
   if (ring->head == TERMRINGLEN)
      ring->head = 0;

   MUTEX_UNLOCK (ring->mutex);
}

/***********************************************************************
* termdgetring - Get a char from the ring buffer.
***********************************************************************/

int
termdgetring (Device *dev, TermRing *ring)
{
   int ch;

   ch = EOF;

   MUTEX_LOCK (ring->mutex);

   if (ring->head != ring->tail)
   {
#ifdef DEBUGTERMDRING
      fprintf (stderr, "termdgetring: ring = %s, head = %d, tail = %d",
	       ring == &dev->info.terminfo.inring ? "IN " :
	       ring == &dev->info.terminfo.outring ? "OUT" : "PRC",
	       ring->head, ring->tail);
#endif
      ch = ring->data[ring->tail++];
      ch &= 0xFF;
      if (ring->tail == TERMRINGLEN)
	 ring->tail = 0;
#ifdef DEBUGTERMDRING
      fprintf (stderr, ", data = >%02X(%c)\n", ch, isprint(ch) ? ch : '.');
#endif
   }

   MUTEX_UNLOCK (ring->mutex);

   return (ch);
}

/***********************************************************************
* termdsend - Send on connection.
***********************************************************************/

void
termdsend (Device *dev, uint8 *msg, int msglen)
{
   TermInfo *terminfo;
   TermRing *ring;
   int i;

   terminfo = &dev->info.terminfo;
   ring = &terminfo->outring;

#ifdef DEBUGTERMDTASK
   fprintf (stderr, "termdsend: msglen = %d, connected = %s\n",
	    msglen, terminfo->connected ? "YES" : "NO");
   HEXDUMP (stderr, msg, msglen, 0);
#endif

   MUTEX_LOCK (ring->mutex);

   for (i = 0; i < msglen; i++)
   {
      ring->data[ring->head++] = msg[i];
      if (ring->head == TERMRINGLEN)
	 ring->head = 0;
   }

   MUTEX_UNLOCK (ring->mutex);
}

/***********************************************************************
* termdgetchar - Get next character from the device.
***********************************************************************/

int
termdgetchar (Device *dev, int fd, uint8 *buf)
{
   int c;

   if (dev->switches & SWTELNET)
      return (Nread (fd, buf, 1));

   if ((c = serialgetchar (dev)) == EOF)
      return (0);

   *buf = c;
   return (1);
}

/***********************************************************************
* termdeatchar - Eat characters to tilde.
***********************************************************************/

void
termdeatchar (Device *dev, int fd, uint8 *buf)
{
   *buf = 0;
   while (*buf != '~')
   {
      if (termdgetchar (dev, fd, buf) != 1) return;
#ifdef DEBUG931
      fprintf (stderr, "   cx = %02X(%c)\n", buf[0], buf[0]);
#endif
   }
}

/***********************************************************************
* termdgetcond - Conditionaly get a character from the ring.
***********************************************************************/

int
termdgetcond (Device *dev, TermRing *ring, int first)
{
   if (termdringempty (dev, ring))
   {
      ring->tail = first; /* rewind */
      return (-1);
   }
   return (termdgetring (dev, ring));
}

/***********************************************************************
* termddoiac - IAC processor. We just eat them.
***********************************************************************/

void
termddoiac (int fd)
{
   uint8 buf[4];

   if (Nread (fd, &buf[1], 1) == 1)
   {
      if (Nread (fd, &buf[2], 1) == 1)
      {
#ifdef DEBUGTERMDIAC
	 fprintf (stderr, "termddoiac: c1 = %d, c2 = %d\n", buf[1], buf[2]);
#endif
	 if (buf[1] == WILL)
	 {
	    buf[0] = IAC;
	    buf[1] = DONT;
	    Nwrite (fd, buf, 3);
	 }
      }
   }
}

/***********************************************************************
* termdtask - Telnet connection task
***********************************************************************/

static void
termdtask (void *vdev)
{
   Device *dev;
   TermInfo *terminfo;
   int sfd;
   int ifd, ofd;
   int maxfd;
   int i;
   int fromlen;
   int didwork;
   int exit_val = 0;
   struct timeval tv;
   struct sockaddr_in peer;
   struct sockaddr_in frominet;
   struct linger linger;
   fd_set readfds;
   uint8 msg[82];
   uint8 buf[2];


   dev = (Device *)vdev;
   terminfo = &dev->info.terminfo;

   sfd = dev->sfd;

   /* Do terminal specific initialization */
   if (dev->emutype == EMU931 || dev->emutype == EMU940)
      v931init (dev);
   else if (dev->emutype == EMUH19)
      H19init (dev);

SERVER_LOOP:

#ifdef DEBUGTERMDTASK
   fprintf (stderr, 
	    "termdtask: starting: dev = %s, devaddr = >%04X, sfd = %d\n",
	    dev->name, dev->devaddr, sfd);
#endif

   if (dev->switches & SWTELNET)
   {
      sprintf (dev->file, "%d:LISTEN", dev->unit);

      /*
      ** Await user connection.
      */

      maxfd = 0;
      while (maxfd == 0)
      {
	 FD_ZERO (&readfds);
	 FD_SET (sfd, &readfds);

	 tv.tv_sec = 0;
	 tv.tv_usec = 1000;
	 if ((maxfd = select (sfd+1, &readfds, NULL, NULL, &tv)) < 0)
	 {
	    if (ERRNO != EINTR && ERRNO != EAGAIN)
	    {
	       fprintf (stderr,
		  "termdtask: select failed: dev = %s, file = %s, fd = %d: %s",
			dev->name, dev->file, ifd, strerror (ERRNO));
	       goto SERVER_LOOP;
	    }
	 }

	 if (!dev->runtermd)
	 {
	    goto TERMINATE;
	 }
      }

      /*
      ** Accept user connection
      */

      fromlen = sizeof(struct sockaddr_in);
      if ((ifd = accept (sfd, (struct sockaddr *)&frominet,
                        (void *)&fromlen)) < 0)
      {
	 if (ERRNO == EAGAIN || ERRNO == EINTR)
	 {
	    goto SERVER_LOOP;
	 }
#if defined(WIN32)
	 if (ifd == INVALID_SOCKET) return;
#endif
	 PERROR ("termdtask: accept");
	 return;
      }
      ofd = ifd;
#ifdef DEBUGTERMDTASK
      fprintf (stderr,
	       "termdtask: accepting: dev = %s, devaddr = >%04X, ifd = %d\n",
	       dev->name, dev->devaddr, ifd);
#endif

      linger.l_onoff = 1;
      linger.l_linger = 0;
      if (setsockopt (ifd, SOL_SOCKET, SO_LINGER, &linger, sizeof(linger)) < 0)
      {
	 PERROR ("termdtask: setsockopt");
	 CLOSE (ifd);
	 goto SERVER_LOOP;
      }

      i = sizeof(peer);

      if (getpeername (ifd, (struct sockaddr *)&peer, (void *)&i) < 0)
      {
	 PERROR ("termdtask: getpeername");
	 CLOSE (ifd);
	 goto SERVER_LOOP;
      }
      sprintf (dev->file, "%d:%s", dev->unit, inet_ntoa (peer.sin_addr));

#ifdef DEBUGTERMDTASK
      fprintf (stderr, "termdtask: dev = %s, devaddr = >%04X, from = %s\n",
	       dev->name, dev->devaddr, dev->file);
#endif
      
      /*
      ** Tell the telnet client we're doing the work
      */

      sprintf ((char *)msg, (char *)will, TELOPT_SGA);
      Nwrite (ofd, msg, 3);
      sprintf ((char *)msg, (char *)will, TELOPT_ECHO);
      Nwrite (ofd, msg, 3);

      /*
      ** Display the introduction banner
      */

      Nwrite (ofd, clear, sizeof (clear));
      if (dev->switches & SWEMUTERM)
	 sprintf ((char *)msg, "sim990 %s TERMD %s(%s) on %s\r\n",
		  VERSION, dev->name, emulations[dev->emutype], dev->file);
      else
	 sprintf ((char *)msg, "sim990 %s TERMD %s on %s\r\n",
		  VERSION, dev->name, dev->file);
      Nwrite (ofd, msg, strlen((char *)msg));
   }
   else
   {
      ifd = fileno (dev->infd);
      ofd = fileno (dev->outfd);
   }
#ifdef DEBUGTERMDTASKA
   fprintf (stderr, "   ifd = %d, ofd = %d\n", ifd, ofd);
#endif

   /*
   ** Process connection
   */

   dev->cbreak = TRUE;
   terminfo->connected = TRUE;
   terminfo->inescape = 0;
   while (TRUE)
   {
      /*
      ** Go display the screen, if updated.
      */

      didwork = FALSE;
      if (dev->emutype == EMU931 || dev->emutype == EMU940)
      {
         if (v931display (dev, ofd) < 0)
	    break;
      }
      else if (dev->emutype == EMUH19)
      {
         if (H19display (dev, ofd) < 0)
	    break;
      }

      /*
      ** If the CPU is not running and device is stdin, DO NOT read.
      */

      if (!run && (dev->infd == stdin))
      {
	 smallsleep (100);
      }
      else
      {
	 /*
	 ** Await terminal input or timeout
	 */

#ifdef WIN32
         if (dev->infd == stdin)
	 {
	    smallsleep (1000);
	    maxfd = 0;
	    if (kbhit()) 
	       maxfd = 1;
	 }
	 else
	 {
#endif
	 FD_ZERO (&readfds);
	 FD_SET (ifd, &readfds);

	 tv.tv_sec = 0;
	 tv.tv_usec = 10000;
	 if ((maxfd = select (ifd+1, &readfds, NULL, NULL, &tv)) < 0)
	 {
	    if (ERRNO != EINTR && ERRNO != EAGAIN)
	    {
	       fprintf (stderr,
		  "termdtask: select failed: dev = %s, file = %s, fd = %d: %s",
			dev->name, dev->file, ifd, strerror (ERRNO));
	       break;
	    }
	 }
#ifdef WIN32
         }
#endif
	 /*
	 ** If input data from the terminal, process it
	 */

	 if (maxfd > 0)
	 {
#ifdef DEBUGTERMDTASK
	    fprintf (stderr,
		  "Select: dev = %s, devaddr = >%04X, ifd = %d, maxfd = %d\n",
		     dev->name, dev->devaddr, ifd, maxfd);
#endif
	    didwork = TRUE;
	    if (termdgetchar (dev, ifd, buf) != 1) break;

#ifdef DEBUGTERMDTASK
	    fprintf (stderr, "   inchar = %02X(%c)\n",
		     buf[0] & 0xFF, isprint(buf[0]) ? buf[0] : '.');
#endif
	    if (dev->realdevice)
	    {
	       goto PUT_CHAR;
	    }
	    else if ((dev->switches & SWTELNET) && (buf[0] == IAC))
	    {
	       termddoiac (ifd);
	    }
	    else if (buf[0] == ESCAPE && (dev->switches & SWEMUTERM))
	    {
	       terminfo->kbchar = TRUE;
	       if (dev->emutype == EMU733)
		  goto PUT_CHAR;
	       else if (dev->emutype == EMUH19)
		  H19input (dev, ifd);
	       else if (dev->emutype == EMU931 || dev->emutype == EMU940)
		  v931input (dev, ifd);
	       if (dev->intenabled)
		  terminfo->kbint = TRUE;
	    }
	    else if (buf[0] == 0x0A) ; /* Eat linefeed (Thanks PuTTY) */
	    else if (buf[0] > 0 && buf[0] <= 0x7F)
	    {
	       if (buf[0] == ESCAPE) /* Allow for raw terminal pass-thru */
		  terminfo->inescape = 2;
	       if (terminfo->inescape)
		  terminfo->inescape--;
	       else if ((dev->switches & SWUPCASE) && islower(buf[0]))
		  buf[0] = toupper(buf[0]);
	       if ((buf[0] == BACKSPACE || buf[0] == DBACKSPACE) &&
		   (dev->switches & SWEMUTERM) && (dev->emutype < EMU733))
	       {
		  termdputring (dev, &terminfo->inring, ESCAPE);
		  buf[0] = 'D';
	       }
	    PUT_CHAR:
	       termdputring (dev, &terminfo->inring, buf[0]);
	       terminfo->kbchar = TRUE;
	       if (dev->intenabled)
		  terminfo->kbint = TRUE;
	    }
	 }
      }

      /*
      ** Check if we have something to process from the host.
      */

      if (!termdringempty (dev, &terminfo->prcring))
      {
#ifdef DEBUGTERMDTASK
	 fprintf (stderr, "PRCRING input: dev = %s, devaddr = >%04X\n",
		  dev->name, dev->devaddr);
#endif
	 didwork = TRUE;
         if (dev->emutype == EMU931 || dev->emutype == EMU940)
	 {
	    v931process (dev);
	 }
         else if (dev->emutype == EMUH19)
	 {
	    H19process (dev);
	 }
         else if (dev->emutype == EMU733)
	 {
	    a733process (dev);
	 }
         else /* Not emulating, copy to output ring */
	 {
	    while ((i = termdgetring (dev, &terminfo->prcring)) != EOF)
	    {
#ifdef DEBUGTERMDTASK
	       fprintf (stderr, "   prcring data = >%02X(%c)\n", 
	                i, isprint(i) ? i : '.');
#endif
	       termdputring (dev, &terminfo->outring, i);
	    }
	 }
      }

      /*
      ** Check if we have something to send to the terminal.
      */

      if (!termdringempty (dev, &terminfo->outring))
      {
#ifdef DEBUGTERMDTASK
	 fprintf (stderr, "OUTRING input: dev = %s, devaddr = >%04X\n",
		  dev->name, dev->devaddr);
#endif
	 didwork = TRUE;
	 while ((i = termdgetring (dev, &terminfo->outring)) != EOF)
	 {
#ifdef DEBUGTERMDTASK
	    fprintf (stderr, "   outring data = >%02X(%c)\n", 
		     i, isprint(i) ? i : '.');
#endif
	    if (dev->switches & SWTELNET)
	    {
	       if (i > 0)
	       {
		  msg[0] = i;
		  Nwrite (ofd, msg, 1);
	       }
	    }
	    else
	    {
	       serialputchar (dev, i);
	    }
	 }
      }

      /*
      ** Check if we are being asked to quit.
      */

      else if (!dev->runtermd)
      {
	 if (dev->switches & SWTELNET)
	    CLOSE (ifd);
         goto TERMINATE;
      }

      if (!didwork)
         smallsleep (100);
   }

   /*
   ** If LOGOUT and TELNET specified, Send Logout sequence
   */

   if (dev->switches & SWTELNET)
   {
      if (dev->switches & SWLOGOUT)
      {
	 terminfo->kbchar = TRUE;
	 termdputring (dev, &terminfo->inring, 'Q');
	 if (dev->intenabled)
	    terminfo->kbint = TRUE;
	 smallsleep (100);
	 terminfo->kbchar = TRUE;
	 termdputring (dev, &terminfo->inring, '\r');
	 if (dev->intenabled)
	    terminfo->kbint = TRUE;
      }

      CLOSE (ifd);
   }

   terminfo->connected = FALSE;
   terminfo->active = FALSE;
   dev->cbreak = TRUE;
   goto SERVER_LOOP;

TERMINATE:

#ifdef DEBUGTERMDTASK
   fprintf (stderr, "termdtask: exitting: dev = %s, devaddr = >%04X\n",
	    dev->name, dev->devaddr);
#endif

   THREAD_EXIT (exit_val);
   return;
}

/***********************************************************************
* termdstart - Start terminal processing task
***********************************************************************/

int
termdstart (Device *dev)
{
   TermInfo *terminfo;
   THREAD_HANDLE(thandle);

   if (dev->infd == DEV_NULL || dev->infd == DEV_LOOP)
      return (0);

   terminfo = &dev->info.terminfo;

   terminfo->connected = FALSE;
   terminfo->kbchar = FALSE;
   terminfo->kbint = FALSE;

#ifdef DEBUGTERMDTASK
   fprintf (stderr, "termdstart: starting: dev = %s, devaddr = >%04X\n",
	    dev->name, dev->devaddr);
#endif

   termdringinit (dev, &terminfo->inring);
   termdringinit (dev, &terminfo->outring);
   termdringinit (dev, &terminfo->prcring);

   /*
   ** If a real device, just pass through (raw)
   */
   if (dev->realdevice && dev->switches & SWRAWTERM)
   {
      if (ttgetattr (fileno(dev->outfd), &terminfo->termattr) < 0)
      {
	 sprintf (view[0], "Device %s: get attributes failed: %s",
		  dev->name, strerror (ERRNO));
         return (-1);
      }
      if (ttraw (fileno(dev->outfd), TRUE) < 0)
      {
	 sprintf (view[0], "Device %s: setting raw failed: %s",
		  dev->name, strerror (ERRNO));
         return (-1);
      }
      smallsleep (200);
      if (ttraw (fileno(dev->outfd), FALSE) < 0)
      {
	 sprintf (view[0], "Device %s: setting raw failed: %s",
		  dev->name, strerror (ERRNO));
         return (-1);
      }
   }

   dev->runtermd = TRUE;
   if (!terminfo->started)
   {
      if (THREAD_CREATE (thandle, termdtask, dev))
      {
	 sprintf (view[0], "Device %s: thread create failed: %s",
		  dev->name, strerror (ERRNO));
#ifdef DEBUGTERMDTASK
	 fprintf (stderr, "%s\n", view[0]);
#endif
         return (-1);
      }
   }

   terminfo->started = TRUE;
   smallsleep (10000);

   return (0);
}

/***********************************************************************
* termdstop - Stop terminal processing task
***********************************************************************/

void
termdstop (Device *dev)
{
   TermInfo *terminfo;
   int i;

   terminfo = &dev->info.terminfo;

#ifdef DEBUGTERMDTASK
   fprintf (stderr, "termdstop: stopping: dev = %s, devaddr = >%04X\n",
	    dev->name, dev->devaddr);
#endif

   dev->runtermd = FALSE;
   for (i = 0; (i < 5000) && !termdringempty (dev, &terminfo->outring); i++)
   {
#ifdef LINUX
      sched_yield ();
#else
      smallsleep (1);
#endif
   }

   smallsleep (10000);
   if (dev->switches & SWEMUTERM)
   {
      if (dev->emutype == EMU931 || dev->emutype == EMU940)
      {
	 VDTInfo *vdtinfo = &terminfo->terminal.vdtinfo;

	 if (vdtinfo->printfd)
	 {
	    fclose (vdtinfo->printfd);
	    vdtinfo->printfd = NULL;
	 }
      }
   }

   if (dev->realdevice && (dev->switches & SWRAWTERM))
   {
      if (ttsetattr (fileno(dev->outfd), &terminfo->termattr) < 0)
      {
	 sprintf (view[0], "Device %s: set attributes failed: %s",
		  dev->name, strerror (ERRNO));
      }
   }

   if (dev->switches & SWTELNET)
   {
      CLOSE (dev->sfd);
   }
   terminfo->started = FALSE;
}

/***********************************************************************
* termdopen - Open a connection for the Telnet device.
***********************************************************************/

FILE *
termdopen (char *fdev, char *bp, char *mode, int *pfd)
{
   int sfd;
   short port;
   struct sockaddr_in sinme;
#if defined(WIN32)
   WSADATA nt_data;
#endif /* WIN32 */

#ifdef DEBUGTERMDTASK
   fprintf (stderr, "termdopen: bp = %s\n", bp);
#endif

   memset ((char *)&sinme, '\0', sizeof(sinme));

#if defined(WIN32)
   if (WSAStartup (MAKEWORD (1, 1), &nt_data) != 0)
   {
      sprintf (view[0], "Device %s port %s: WSAStartup failed: %s",
	       fdev, bp, strerror (ERRNO));
#ifdef DEBUGTERMDTASK
      fprintf (stderr, "%s\n", view[0]);
#endif
      return (NULL);
   }
#endif /* WIN32 */

   port = atoi(bp);
   sinme.sin_port = htons (port);
   sinme.sin_family = AF_INET;

   if ((sfd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
   {
      sprintf (view[0], "Device %s port %d: socket failed: %s",
	       fdev, port, strerror (ERRNO));
#ifdef DEBUGTERMDTASK
      fprintf (stderr, "%s\n", view[0]);
#endif
      return (NULL);
   }

   if (bind (sfd, (struct sockaddr *)&sinme, sizeof(sinme)) < 0)
   {
      sprintf (view[0], "Device %s port %d: bind failed: %s",
	       fdev, port, strerror (ERRNO));
#ifdef DEBUGTERMDTASK
      fprintf (stderr, "%s\n", view[0]);
#endif
      return (NULL);
   }

   if (listen (sfd, DEPTH) < 0)
   {
      sprintf (view[0], "Device %s port %d: listen failed: %s",
	       fdev, port, strerror (ERRNO));
#ifdef DEBUGTERMDTASK
      fprintf (stderr, "%s\n", view[0]);
#endif
      return (NULL);
   }

   *pfd = sfd;
   return (DEV_TELNET);
}
