/***********************************************************************
*
* simvdt.c - Serial VDT support routines for the TI 990 Simulator.
*
* Changes:
*   06/03/15   DGP   Moved from sim931.c
*
***********************************************************************/

#include <unistd.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>
#include <time.h>

#include "simdef.h"
#include "screen.h"

/*
**
** VDT Display Attribute bits:
**
** Bit 7:                       OFF Always
** Bit 6: ON Always
** Bit 5:                       OFF Always
** Bit 4: ON == Blinking      , OFF == Normal --> ESC [ 5 m
** Bit 3: ON == Hidden        , OFF == Normal --> N/A
** Bit 2: ON == Underline     , OFF == Normal --> ESC [ 4 m
** Bit 1: ON == Reverse Image , OFF == Normal --> ESC [ 7 m
** Bit 0: ON == High Intensity, OFF == Normal --> ESC [ 1 m
*/


static uint8 clear[] = {
   ESCAPE, '[', '2', 'J', ESCAPE, '[', '1', ';', '1', 'H'
};

static uint8 VDT_GraphicChars[33] = {
   /* 20   21   22   23   04   05   06   27  */
      ' ', '|', '|', '|', '|', '|', '|', '|',
   /* 28   09   0A   0B   0C   0D   0E   2F  */
      '|', '|', '+', '+', '+', '+', '/', '-',
   /* 30   11   12   13   14   15   16   37  */
      '-', '-', '-', '-', '-', '-', '-', '+',
   /* 38   19   1A   1B   1C   1D   1E   3F  */
      '+', '_', '+', '+', '+', '+', '\\', '|',
   /* 40  */
      '_'
};

#if defined(DEBUGVDT)
extern char *emulations[];

/***********************************************************************
* VDTdebugprint - VDT DEBUG printing.
***********************************************************************/

static void
VDTdebugprint (Device *dev, const char *format, ...)
{
   va_list ap;

   fprintf (stderr, "VDTDEBUG-%s: devaddr = >%04X: ", 
	    emulations[dev->emutype], dev->devaddr);
   va_start (ap, format);
   vfprintf (stderr, format, ap);
   va_end (ap);
}
#endif

/***********************************************************************
* VDT_ScrollScreen - Scroll the screen.
***********************************************************************/

void
VDT_ScrollScreen (Device *dev, int IsUp)
{
   VDTInfo *vdtinfo;
   
   vdtinfo = &dev->info.terminfo.terminal.vdtinfo;
   if (IsUp)
   {
      /* Scroll up - normal case */
      /* SCROLL -- FIXME -- TODO */
      int i;

      for (i = VDT_MIN_ROW; i < VDT_MAX_ROW; i++)
      {
         memcpy (&vdtinfo->Buffer[i][0], &vdtinfo->Buffer[i + 1][0],
                 VDT_NUM_COLS);
         memcpy (&vdtinfo->Shadow[i][0], &vdtinfo->Shadow[i + 1][0],
                 VDT_NUM_COLS);
      }
      memset (&vdtinfo->Buffer[i][0], vdtinfo->Fill, VDT_NUM_COLS);
      memset (&vdtinfo->Shadow[i][0], vdtinfo->Attr, VDT_NUM_COLS);
   }
   else
   {
      /* Scroll down - backward case */
      /* SCROLL -- FIXME -- TODO */
      int i;

      for (i = VDT_MAX_ROW; i > VDT_MIN_ROW; i--)
      {
         memcpy (&vdtinfo->Buffer[i][0], &vdtinfo->Buffer[i - 1][0],
                 VDT_NUM_COLS);
         memcpy (&vdtinfo->Shadow[i][0], &vdtinfo->Shadow[i - 1][0],
                 VDT_NUM_COLS);
      }
      memset (&vdtinfo->Buffer[i][0], vdtinfo->Fill, VDT_NUM_COLS);
      memset (&vdtinfo->Shadow[i][0], vdtinfo->Attr, VDT_NUM_COLS);
   }
   dev->cbreak = TRUE;
}

/***********************************************************************
* VDT_InsertDelete - Insert/delete entities from the screen.
***********************************************************************/

void
VDT_InsertDelete (Device *dev, int IsInsert, int IsLine)
{
   VDTInfo *vdtinfo;
   int i;
   int j;
   int r;
   int c;

   vdtinfo = &dev->info.terminfo.terminal.vdtinfo;
   if (IsLine)
   {
      /* Line Editing */
      i = vdtinfo->LineExtentRow;

      if (IsInsert)
      {
         /* Insert Line */
         for (r = i; r > vdtinfo->Row; r--)
         {
            memcpy (&vdtinfo->Buffer[r][0], &vdtinfo->Buffer[r - 1][0],
                    VDT_NUM_COLS);
            memcpy (&vdtinfo->Shadow[r][0], &vdtinfo->Shadow[r - 1][0],
                    VDT_NUM_COLS);
         }
         memset (&vdtinfo->Buffer[r][0], vdtinfo->Fill, VDT_NUM_COLS);
         memset (&vdtinfo->Shadow[r][0], vdtinfo->Attr, VDT_NUM_COLS);
      }
      else
      {
         /* Delete Line */
         for (r = vdtinfo->Row; r < i; r++)
         {
            memcpy (&vdtinfo->Buffer[r][0], &vdtinfo->Buffer[r + 1][0],
                    VDT_NUM_COLS);
            memcpy (&vdtinfo->Shadow[r][0], &vdtinfo->Shadow[r + 1][0],
                    VDT_NUM_COLS);
         }
         memset (&vdtinfo->Buffer[r][0], vdtinfo->Fill, VDT_NUM_COLS);
         memset (&vdtinfo->Shadow[r][0], vdtinfo->Attr, VDT_NUM_COLS);
      }
   }
   else
   {
      /* Char Editing */
      if (vdtinfo->CharExtentEnabled)
      {
         i = vdtinfo->CharExtentCol;
         j = vdtinfo->CharExtentRow;
      }
      else
      {
         i = VDT_MAX_COL;
         j = VDT_MAX_ROW;
      }

      if (IsInsert)
      {
         /* Insert Char */
         for (r = j; r > vdtinfo->Row; r--); /* not implemented */

         for (c = i; c > vdtinfo->Col; c--)
         {
            vdtinfo->Buffer[vdtinfo->Row][c] =
               vdtinfo->Buffer[vdtinfo->Row][c - 1];
            vdtinfo->Shadow[vdtinfo->Row][c] =
               vdtinfo->Shadow[vdtinfo->Row][c - 1];
         }
         vdtinfo->Buffer[vdtinfo->Row][c] = vdtinfo->Fill;
         vdtinfo->Shadow[vdtinfo->Row][c] = vdtinfo->Attr;
      }
      else
      {
         /* Delete Char */
         for (r = vdtinfo->Row; r < j; r++); /* not implemented */

         for (c = vdtinfo->Col; c < i; c++)
         {
            vdtinfo->Buffer[vdtinfo->Row][c] =
               vdtinfo->Buffer[vdtinfo->Row][c + 1];
            vdtinfo->Shadow[vdtinfo->Row][c] =
               vdtinfo->Shadow[vdtinfo->Row][c + 1];
         }
         vdtinfo->Buffer[vdtinfo->Row][c] = vdtinfo->Fill;
         vdtinfo->Shadow[vdtinfo->Row][c] = vdtinfo->Attr;
      }
   }
}

/***********************************************************************
* VDT_AdjustCursor - Adjust the screen cursor position.
***********************************************************************/

void
VDT_AdjustCursor (Device *dev)
{
   VDTInfo *vdtinfo;
   
   vdtinfo = &dev->info.terminfo.terminal.vdtinfo;
   if (vdtinfo->Col < VDT_MIN_COL)
   {
      /* WRAP */
      vdtinfo->Col = VDT_MAX_COL;
      vdtinfo->Row--;
   }
   if (vdtinfo->Col > VDT_MAX_COL)
   {
      /* WRAP */
      vdtinfo->Col = VDT_MIN_COL;
      vdtinfo->Row++;
   }
   if (vdtinfo->Row < VDT_MIN_ROW)
   {
      vdtinfo->Row = VDT_MIN_ROW;
   }

   if (vdtinfo->Row > VDT_MAX_ROW)
   {
      VDT_ScrollScreen (dev, TRUE);
      vdtinfo->Row = VDT_MAX_ROW;
   }
   dev->cbreak = TRUE;
}

/***********************************************************************
* VDT_EraseArea - Erase an area from the screen.
***********************************************************************/

void
VDT_EraseArea (Device *dev, int Top, int Left, int Bottom, int Right)
{
   VDTInfo *vdtinfo;
   int Row, Col;

   vdtinfo = &dev->info.terminfo.terminal.vdtinfo;
   for (Row = Top; Row <= Bottom; Row++)
   {
      for (Col = Left; Col <= Right; Col++)
      {
         vdtinfo->Buffer[Row][Col] = vdtinfo->Fill;
         vdtinfo->Shadow[Row][Col] = vdtinfo->Attr;
      }
   }
   dev->cbreak = TRUE;
}

/***********************************************************************
* VDT_PutChar - Put a character on the screen.
***********************************************************************/

void
VDT_PutChar (Device *dev, uint8 Char, uint8 Count)
{
   VDTInfo *vdtinfo;
   int i;

   vdtinfo = &dev->info.terminfo.terminal.vdtinfo;
   Char &= 0x7F;   /* MSBit is often set following ESC sequence */

   if (vdtinfo->GraphicCharSet) /* Graphic */
   {
      if (Char > 0x1F && Char < 0x41)
      {
         Char = VDT_GraphicChars[Char - ' '];
#ifdef DEBUGVDT
	 VDTdebugprint (dev, "Graphic ( %c )\n", Char);
#endif
      }
      else /* error */
      {
#ifdef DEBUGVDT
	 VDTdebugprint (dev, "Unknown Graphic ( >%02X )\n", Char);
#endif
         Char = vdtinfo->Fill;
      }
   }
   else /* Normal */
   {
      if (Char > 0x1F && Char < 0x7F)
      {
#ifdef DEBUGVDT
	 VDTdebugprint (dev, "Row: %d Col: %d: Normal ( %c )\n",
			vdtinfo->Row, vdtinfo->Col, Char);
#endif
      }
      else /* error */
      {
#ifdef DEBUGVDT
	 VDTdebugprint (dev, "Unknown Normal ( >%02X )\n", Char);
#endif
         Char = vdtinfo->Fill;
      }
   }
   for (i = 0; i < Count; i++)
   {
      vdtinfo->Buffer[vdtinfo->Row][vdtinfo->Col] = Char;
      vdtinfo->Shadow[vdtinfo->Row][vdtinfo->Col] = vdtinfo->Attr;
      vdtinfo->Col++;
      if (vdtinfo->Col > VDT_MAX_COL)
      {
         vdtinfo->Col = 0;
	 vdtinfo->Row++;
	 if (vdtinfo->Row > VDT_MAX_ROW)
	    vdtinfo->Row = 0;
      }
      VDT_AdjustCursor (dev);
   }
   dev->cbreak = TRUE;
}

/***********************************************************************
* VDT_PutAttr - Put attributes for the screen.
***********************************************************************/

void
VDT_PutAttr (Device *dev, uint8 Attr, uint8 Count)
{
   VDTInfo *vdtinfo;
   int i;
   int c, r;

   vdtinfo = &dev->info.terminfo.terminal.vdtinfo;
   c = vdtinfo->Col;
   r = vdtinfo->Row;
   for (i = 0; i < Count; i++)
   {
      vdtinfo->Shadow[r][c] = Attr;
      c++;
      if (c > VDT_MAX_COL)
      {
         c = 0;
	 r++;
	 if (r > VDT_MAX_ROW)
	    r = 0;
      }
   }
}

/***********************************************************************
* VDT_DisplayClear - Clear the VDT display.
***********************************************************************/

uint8 *
VDT_DisplayClear (Device *dev, uint8 *sp)
{

#ifdef DEBUGVDTDISPLAY
   fprintf (stderr, "VDT_DisplayClear: sp = %p\n", sp);
#endif

#ifdef WIN32
   if (!(dev->switches & SWTELNET) && (dev->infd == stdin))
   {
      clearscreen (TRUE);
   }
   else
   {
#endif
      memcpy (sp, clear, 10);
      sp += 10;
#ifdef WIN32
   }
#endif
   return (sp);
}

/***********************************************************************
* VDT_DisplayPosition - Position the cursor on VDT display.
***********************************************************************/

uint8 *
VDT_DisplayPosition (Device *dev, uint8 *sp, int row, int col)
{
   uint8 msg[82];

#ifdef DEBUGVDTDISPLAY
   fprintf (stderr, "VDT_DisplayPosition: sp = %p, row = %d, col = %d\n",
	    sp, row, col);
#endif

#ifdef WIN32
   if (!(dev->switches & SWTELNET) && (dev->infd == stdin))
   {
      screenposition (TRUE, row, col);
   }
   else
   {
#endif
      sprintf ((char *) msg, "%c[%d;%dH", ESCAPE, row, col);
      memcpy (sp, msg, strlen ((char *) msg));
      sp += strlen ((char *) msg);
#ifdef WIN32
   }
#endif
   return (sp);
}

/***********************************************************************
* VDT_DisplayAttr - Display the VDT Attribute.
***********************************************************************/

uint8 *
VDT_DisplayAttr (Device *dev, uint8 *sp, uint8 Attr)
{
   uint8 setattr[] = { ESCAPE, '[', '0', 'm' };

#ifdef DEBUGVDTDISPLAY
   fprintf (stderr, "VDT_DisplayAttr: sp = %p, Attr = %02x\n", sp, Attr);
#endif

#ifdef WIN32
   if (!(dev->switches & SWTELNET) && (dev->infd == stdin))
   {
      screenattribute (TRUE, Attr);
   }
   else
   {
#endif
      if (Attr & VDT_ATTR_REVERSE) /* Reverse Image */
      {
	 setattr[2] = '7';
      }
      else if (Attr & VDT_ATTR_BLINK) /* Blinking */
      {
	 setattr[2] = '5';
      }
      else if (Attr & VDT_ATTR_UNDERLINE) /* Underline */
      {
	 setattr[2] = '4';
      }
      else if (Attr & VDT_ATTR_BOLD) /* High Intensity */
      {
	 setattr[2] = '1';
      }
      else /* Normal */
      {
	 setattr[2] = '0';
      }
      memcpy (sp, setattr, 4);
      sp += 4;
#ifdef WIN32
   }
#endif
   return (sp);
}

/***********************************************************************
* VDT_DisplayScreen - Put screen on the VDT display.
***********************************************************************/

uint8 *
VDT_DisplayScreen (Device *dev, int fd, uint8 *sp, int len, int atend)
{
   VDTInfo *vdtinfo;
   int status = 0;

#ifdef DEBUGVDTDISPLAY
   fprintf (stderr, "VDT_DisplayScreen: sp = %p, len = %d, atend = %d\n",
	    sp, len, atend);
   if (atend)
      HEXDUMP (stderr, (char *)sp, len, 0);
#endif

   vdtinfo = &dev->info.terminfo.terminal.vdtinfo;

#ifdef WIN32
   if (!(dev->switches & SWTELNET) && (dev->infd == stdin))
   {
      if (write (fd, sp, len) < 0) return (NULL);
      sp = vdtinfo->screen;
   }
   else
   {
#endif
      if (atend)
      {
	 if (dev->switches & SWTELNET)
	    status = Nwrite (fd, sp, len);
	 else
	    status = write (fd, sp, len);
         if (status < 0) return (NULL);
	 sp = vdtinfo->screen;
      }
      else
      {
         sp += len;
      }
#ifdef WIN32
   }
#endif
   return (sp);
}

