#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/svc.h>
#include <dirent.h>

#if defined(DEBUGDIRIO) || defined(DEBUGSTAT)
#include "hexdump.h"
#endif

#ifndef MAXPATHNAMESIZE
#define MAXPATHNAMESIZE 255
#endif

#if !defined(dx10) && !defined(dnos)
#define d_flags d_ino
#endif

static DIR *
txopendir (const char *path)
{
   DIR *dp;
   FileTable *ft;
   SVCIODDisk *dsksvc;
   int fd;
   SVCIOBlk svc;
   int stats[16];
   char filepath[MAXPATHNAMESIZE+2];

#ifdef DEBUGDIRIO
   printf ("txopendir: path = %s\n", path);
#endif

   bzero (&svc, sizeof(SVCIOBlk));

   svc.opcode = SVCSETPRV;
   _issue_svc ((SVCBlk *)&svc);

   /*
   ** Copy pathname and upcase
   */

   _setpath (filepath, (char *)path);

   /*
   ** Assign luno to the file
   */

   _errno = 0;
   for (fd = __MINLUNO; fd < __MAXLUNO; fd++)
   {
      ft = &_file_table[fd];
      if (!ft->inuse) break;
   }
   if (fd == __MAXLUNO)
   {
      _errno = EMFILE;
      return (NULL);
   }

   svc.opcode = SVCIO;
   svc.luno = fd;
   svc.subopcode = ASSIGNFILE;
   svc.utilityflags = 0x1080;
   svc.pathname = filepath;

   _issue_svc ((SVCBlk *)&svc);
#ifdef DEBUGDIRIO
   HEXDUMP (stdout, (char *)&svc, sizeof(SVCIOBlk), 0);
#endif
   if (svc.status)
   {
#ifdef DEBUGDIRIO
   printf ("   assign status = %02X\n", svc.status);
#endif
      _maperr (svc.status);
      return (NULL);
   }
   
   /*
   ** Open the file
   */

   dsksvc = (SVCIODDisk *)&svc;
   dsksvc->subopcode = OPEN;
   dsksvc->sectorrecord = 1;
   dsksvc->sectoraddress = 6;

   _issue_svc ((SVCBlk *)dsksvc);
#ifdef DEBUGDIRIO
   HEXDUMP (stdout, (char *)dsksvc, sizeof(SVCIOBlk), 0);
#endif
   if (dsksvc->status)
   {
#ifdef DEBUGDIRIO
   printf ("   open status = %02X\n", dsksvc->status);
#endif
      _maperr (dsksvc->status);
      goto RELEASE;
   }

   /*
   ** Read the device characteristics
   */

   dsksvc->subopcode = DEVICECHAR;
   dsksvc->userflags = 1;
   dsksvc->buffer = &stats;
   dsksvc->logicalrecordlength = 22;
   dsksvc->charactercount = 22;

   _issue_svc ((SVCBlk *)dsksvc);
#ifdef DEBUGDIRIO
   HEXDUMP (stdout, (char *)dsksvc, sizeof(SVCIOBlk), 0);
#endif
   if (dsksvc->status)
   {
#ifdef DEBUGDIRIO
   printf ("   readchar status = %02X\n", dsksvc->status);
#endif
      _errno = EIO;
      goto CLOSE_ALL;
   }
#ifdef DEBUGDIRIO
   printf ("   filechar: \n");
   HEXDUMP (stdout, (char *)&stats, 28, 0);
#endif

   if ((dp = malloc (sizeof (DIR))) == NULL)
   {
      _errno = ENOMEM;
      goto CLOSE_ALL;
   }
   ft->inuse = 1;
   ft->type = (int)svc.buffer;
   ft->recnum = 0;

   dp->dd_fd = fd;
   dp->dd_count = TXDIRPSEC * TXSECAU;
   dp->dd_pos = 0;
   dp->dd_sec = (off_t)-1;

   return (dp);

CLOSE_ALL:
   svc.subopcode = CLOSE;
   _issue_svc ((SVCBlk *)&svc);
RELEASE:
   svc.subopcode = RELEASEFILE;
   _issue_svc ((SVCBlk *)&svc);
   return (NULL);
}

static struct dirent *
txreaddir (DIR *dp)
{
   FileTable *ft;
   struct dirent *ep;
   char *fp, *tp;
   int fd;
   int i;
   int done;
   int rec;
   int sec;
   SVCIODDisk dsksvc;

   _errno = 0;

   if (dp == NULL)
   {
      _errno = EBADF;
      return (NULL);
   }

   bzero (&dsksvc, sizeof (SVCIODDisk));
   dsksvc.opcode = SVCSETPRV;
   _issue_svc ((SVCBlk *)&dsksvc);

   ep = NULL;
   fd = dp->dd_fd;
   ft = &_file_table[fd];

#ifdef DEBUGDIRIO
   printf ("txreaddir: dp = %p, fd = %d\n", dp, fd);
#endif

   /*
   ** Read from directory, empty entries have ".." in the name
   */

   done = FALSE;
   while (!done)
   {
      sec = dp->dd_pos / TXDIRPSEC;
      rec = dp->dd_pos % TXDIRPSEC;

      if (dp->dd_sec != sec)
      {
   #ifdef DEBUGDIRIO
	 printf ("   pos = %u, sec = %d, rec = %d\n", dp->dd_pos, sec, rec);
   #endif
	 if (dp->dd_pos >= dp->dd_count) return (NULL);

	 dp->dd_sec = sec;
	 dsksvc.opcode = SVCIO;
	 dsksvc.subopcode = READBINARY;
	 dsksvc.luno = fd;
	 dsksvc.userflags = 1;
	 dsksvc.buffer = (char *) &dp->dd_buf.dir.txdir;
	 dsksvc.logicalrecordlength = sizeof(dp->dd_buf.dir.txdir);
	 dsksvc.charactercount = sizeof(dp->dd_buf.dir.txdir);
	 dsksvc.trackaddress = TXDIRAU;
	 dsksvc.sectorrecord = 1;
	 dsksvc.sectoraddress = sec;

	 _issue_svc ((SVCBlk *)&dsksvc);
	 if (dsksvc.status)
	 {
	    _errno = EIO;
	    return (NULL);
	 }
      }
      dp->dd_pos++;

#ifdef DEBUGDIRIO
      HEXDUMP (stdout, (char *)&dp->dd_buf.dir.txdir[rec], sizeof(txdir_t), 0);
#endif

      if (strncmp (dp->dd_buf.dir.txdir[rec].filename, "..", 2))
         done = TRUE;
   }

   ep = (struct dirent *) &dp->dd_dir;
   ep->d_ino = dp->dd_pos;
   ep->d_flags = (dp->dd_buf.dir.txdir[rec].fileprot == 'U') ? 1 : 0;
   fp = dp->dd_buf.dir.txdir[rec].filename;
   tp = ep->d_name;
   for (i = 0; i < TXFILELEN && (*fp != ' '); i++) *tp++ = *fp++;
   *tp++ = '.';
   fp = dp->dd_buf.dir.txdir[rec].fileext;
   for (i = 0; i < TXEXTLEN && (*fp != ' '); i++) *tp++ = *fp++;
   *tp = '\0';

   return (ep);
}


static int
txclosedir (DIR *dp)
{
   FileTable *ft;
   int status;
   int fd;
   SVCIOBlk svc;

   /*
   ** Ensure that DIR ptr is good.
   */

   if (dp == NULL)
   {
      _errno = EBADF;
      return (-1);
   }

   bzero (&svc, sizeof(SVCIOBlk));
   svc.opcode = SVCSETPRV;
   _issue_svc ((SVCBlk *)&svc);

   fd = dp->dd_fd;
   free (dp);
   ft = &_file_table[fd];
   _errno = 0;
   status = 0;

   /*
   ** Close file
   */

   svc.opcode = SVCIO;
   svc.subopcode = CLOSE;
   svc.luno = fd;
   _issue_svc ((SVCBlk *)&svc);
   if (svc.status)
   {
      _errno = EIO;
      status = -1;
   }

   /*
   ** Release file luno
   */

   svc.subopcode = RELEASEFILE;
   _issue_svc ((SVCBlk *)&svc);

   ft->inuse = 0;

   return (status);
}

static int
txstat (const char *path, struct stat *buf)
{
   DIR *dp;
   struct dirent *ep;
   char *fn;
   char *cp;
   int found;
   char temp[MAXPATHNAMESIZE+2];
   char file[MAXPATHNAMESIZE+2];

   if (buf == NULL)
   {
      _errno = EINVAL;
      return (-1);
   }

   bzero (buf, sizeof(struct stat));
   strcpy (temp, path);
   if ((fn = strrchr (temp, ':')) == NULL)
   {
      buf->st_mode = S_IFBLK;
      return (0);
   }

   *fn++ = 0;
   for (cp = fn; *cp; cp++)
      if (*cp >= 'a' && *cp <= 'z') *cp -= 0x20;
   strcpy (file, fn);
   fn = file;

#ifdef DEBUGSTAT
   printf ("stat: path = %s, temp = %s, fn = %s\n", path, temp, fn);
#endif
   if ((dp = txopendir (temp)) == NULL)
   {
      return (-1);
   }
   found = FALSE;
   while ((ep = txreaddir(dp)) != NULL)
   {
#ifdef DEBUGSTAT
      printf ("   ep->d_name = %s\n", ep->d_name);
#endif
      if (!strcmp (fn, ep->d_name))
      {
	 found = TRUE;
	 buf->st_mode = S_IRUSR | S_IRGRP | S_IROTH;
	 buf->st_mode |= S_IFREG;
	 if (ep->d_flags)
	    buf->st_mode |= S_IWUSR | S_IWGRP | S_IWOTH;
         break;
      }
   }
   txclosedir (dp);
   if (!found)
   {
      _errno = ENOENT;
      return (-1);
   }
   return (0);
}

int
main(int argc, char **argv)
{
   DIR *dp;
   struct dirent *ep;
   int i;
   
   printf ("%s:\n", argv[0]);

   for (i = 1; i < argc; i++)
   {
      printf ("arg[%d] = %s\n", i, argv[i]);
   }

   if (argc != 2)
   {
      printf ("usage: %s dirname\n", argv[0]);
      return 1;
   }

   if ((dp = txopendir (argv[1])) == NULL)
   {
      printf ("txopendir failed: error = %s\n", strerror(errno));
      return 1;
   }

   printf ("txreaddir:\n");
   while ((ep = txreaddir (dp)) != NULL)
   {
      char fn[MAXPATHNAMESIZE+2];
      struct stat stbuf;
      printf ("   name = %s, ino = %d\n", ep->d_name, ep->d_ino);

#if 1
      sprintf (fn, "%s:%s", argv[1], ep->d_name);
      if (txstat (fn, &stbuf) < 0)
	 printf ("stat failed: error = %s\n", strerror(errno));
      else
         printf ("   mode = 0%06o, len = %ld\n", stbuf.st_mode, stbuf.st_size);
#endif
   }

   if (errno != 0)
      printf ("txreaddir failed: error = %s\n", strerror(errno));

   txclosedir (dp);

   return 0;
}
