/************************************************************************
* lseek - Seek to a location in file. 
*         Only supported on relative record (random) files. 
************************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <stddef.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/svc.h>

static int
rewdev (int fd)
{
   SVCIOBlk svc;

   memset (&svc, 0, sizeof(SVCIOBlk));

   svc.opcode = SVCIO;
   svc.subopcode = REWIND;
   svc.luno = fd;

   _issue_svc ((SVCBlk *)&svc);
   if (svc.status)
   {
      _maperr(svc.status);
#ifdef DEBUGIO
      printf ("   rewdev: status = %d, errno = %d\n", svc.status, _errno);
#endif
      return (-1);
   }
   return (0);
}

off_t
lseek (int fd, off_t offset, int whence)
{
   FileTable *ft;

   _errno = 0;

   if (fd >= __MAXLUNO)
   {
      _errno = EBADF;
      return (-1);
   }

   ft = &_file_table[fd];
   if (!(ft->modes & FTINUSE))
   {
      _errno = EBADF;
      return (-1);
   }

#ifdef DEBUGIO
   if (ft->type & 0xFF == 0xFF)
   {
      printf ("lseek-file: fd = %d, type = %d, attrflags = %X, lrl = %d\n",
	     fd, ft->type, ft->attrflags, ft->logicalrecordlength);
      printf ("   offset = %d, whence = %d\n", offset, whence);
      printf ("   %X & %X = %X == %X\n",
	 ft->attrflags, ATTRTYPEMASK, ft->attrflags & ATTRTYPEMASK,
	 ATTRRELRECORD);
   }
   else
   {
      printf ("lseek-dev: fd = %d, type = %d\n", fd, ft->type);
   }
#endif

   if (ft->type & 0xFF != 0xFF)
   {
      if (offset == 0 && whence == SEEK_SET)
         return (rewdev (fd));
      _errno = EINVAL;
      return (-1);
   }

   if ((ft->attrflags & ATTRTYPEMASK) != ATTRRELRECORD)
   {
      _errno = EPERM;
      return (-1);
   }

   if (whence == SEEK_SET)
      ft->recnum = offset / ft->logicalrecordlength;
   else if (whence == SEEK_CUR)
      ft->recnum += offset / ft->logicalrecordlength;
   else if (whence == SEEK_END)
      ft->recnum = ft->numberofrecords + (offset / ft->logicalrecordlength);
   else
   {
      _errno = EINVAL;
      return (-1);
   }

#ifdef DEBUGIO
   printf ("   new recnum = %d\n", ft->recnum);
#endif

   return (ft->recnum * ft->logicalrecordlength);
}
