/************************************************************************
*
* dis990 - Disassembles objects from asm990
*
* Changes:
*   05/05/05   DGP   Original. Hacked from lnk990.c
*   06/27/06   DGP   Added dump file input.
*   01/16/07   DGP   Added DSEG support.
*   08/09/11   DGP   Added CPU model support.
*	
************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <time.h>

#include "utildef.h"

int pcreg; /* the pc */
int dsegpc;
int absolute = FALSE;
int dumpfile = FALSE;
int symbolcount = 0;
int pgmlen = 0;
int dseglen = 0;
int entry = -1;
int modcount = 0;
int model;
int maxcsegs = 0;
int cseglen[MAXCSEGS];

char inbuf[MAXLINE];
char idtbuf[IDTSIZE+2];
char dsegname[IDTSIZE+2];
char csegname[MAXCSEGS][IDTSIZE+2];

SymNode *symbols[MAXSYMBOLS];
Module modules[MAXMODULES];

uint8 memory[MEMSIZE];
Memory memctl[MEMSIZE];

uint8 dsegmem[MEMSIZE];
Memory dsegctl[MEMSIZE];

uint8 csegmem[MAXCSEGS][MEMSIZE];
Memory csegctl[MAXCSEGS][MEMSIZE];


/***********************************************************************
* printsymbols - Print the symbol table.
***********************************************************************/

static void
printsymbols (void)
{
   int i;

   for (i = 0; i < symbolcount; i++)
   {
      if (symbols[i]->flags & GLOBAL)
      {
	 printf ("\tDEF\t%s\n", symbols[i]->symbol);
      }
      else if (symbols[i]->flags & EXTERNAL)
      {
	 printf ("\tREF\t%s\n", symbols[i]->symbol);
      }
   }
}

/***********************************************************************
* printabsgbl - Print absolute global symbols.
***********************************************************************/

static void
printabsgbl (void)
{
   SymNode *s;
   int gotone;
   int i;

   gotone = FALSE;
   for (i = 0; i < symbolcount; i++)
   {
      s = symbols[i];
      if ((s->flags & GLOBAL) && !(s->flags & RELOCATABLE))
      {
	 printf ("%s\tEQU\t>%04X\n", s->symbol, s->value);
	 gotone = TRUE;
      }
   }
   if (gotone)
      printf ("*\n");
}

/***********************************************************************
* Main procedure
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *infd = NULL;
   char *bp;
   char *infile;
   int i;
   int status = 0;
  
#ifdef DEBUG
   fprintf (stderr, "dis990: Entered:\n");
   fprintf (stderr, "args:\n");
   for (i = 1; i < argc; i++)
      fprintf (stderr, "   arg[%2d] = %s\n", i, argv[i]);
#endif

   /*
   ** Clear symboltable
   */

   for (i = 0; i < MAXSYMBOLS; i++)
      symbols[i] = NULL;

   /*
   ** Scan off the the args.
   */

   idtbuf[0] = '\0';
   infile = NULL;
   model = 4;

   for (i = 1; i < argc; i++)
   {
      bp = argv[i];

      if (*bp == '-')
      {
	 char *pp;

         bp++;
	 switch (*bp)
	 {
	 case 'd':
	    dumpfile = TRUE;
	    break;

	 case 'm': /* CPU model */
	    i++;
	    if (i >= argc) goto USAGE;
	    model = 0;
	    for (pp = argv[i]; *pp; pp++)
	    {
	       if (isdigit(*pp))
	       {
		  model = (model * 10) + (*pp - '0');
	       }
	       else
	       {
	          if (*pp == 'a' || *pp == 'A')
		  {
		     if (model == 10) model = 11;
		     else goto BAD_MODEL;
		     break;
		  }
		  else
		  {
		  BAD_MODEL:
		     fprintf (stderr, 
			      "Invalid model specification %s\n", argv[i]);
		     exit (ABORT);
		  }
	       }
	    }
	    if (!(model ==  4 || model ==  5 || model == 9 ||
		  model == 10 || model == 11 || model == 12))
	    {
	       fprintf (stderr, "Unsupported CPU model: %s\n", argv[i]);
	       goto USAGE;
	    }
	    break;

	 default:
	 USAGE:
	    fprintf (stderr, "dis990 - version %s\n", VERSION);
	    fprintf (stderr, "usage: dis990 [-options] infile.bin...\n");
	    fprintf (stderr, "    -d        - Process dump file\n");
	    fprintf (stderr, "    -m model  - CPU model\n");
	    return (ABORT);
	 }
      }

      else
      {
	 if (infile != NULL) goto USAGE;
         infile = argv[i];
      }

   }

   if (!infile) goto USAGE;

   /*
   ** Open the input.
   */

   if ((infd = fopen (infile, "r")) == NULL)
   {
      sprintf (inbuf, "dis990: Can't open input file %s", infile);
      perror (inbuf);
      exit (ABORT);
   }

   /*
   ** Clear memory.
   */

   memset (&memory, '\0', sizeof(memory));
   memset (&dsegmem, '\0', sizeof(dsegmem));
   for (i = 0; i < MEMSIZE; i++)
   {
      memset (&memctl[i], '\0', sizeof (Memory));
      memset (&dsegctl[i], '\0', sizeof (Memory));
   }

   /*
   ** Load the object.
   */

   pcreg = 0;
   status = disloader (infd, pcreg, infile);

   /*
   ** Output module preamble.
   */

   printf ("*\n");
   printf ("* Module %s \n", idtbuf);
   if (modules[0].time[0])
      printf ("* Created on %s %s by %s\n",
	    modules[0].time, modules[0].date, modules[0].creator);
   else
      printf ("* Created by %s\n",
	    modules[0].creator);
   printf ("*\n");

   if (idtbuf[0])
   {
      printf ("\tIDT\t'%s'\n", idtbuf);
      printf ("\tOPTION\tBUNLST,DUNLST,TUNLST\n");
      printf ("*\n");
   }

   for (i = 0; i < 16; i++)
      printf ("R%d\tEQU\t%d\n", i, i);
   printf ("*\n");
   printf ("\tDXOP\tSVC,15\n");
   printf ("*\n");

   printsymbols();
   printf ("*\n");
   printabsgbl();

   /*
   ** Process it.
   */

   pcreg = 0;
   disprog (0, 0);

   /*
   ** Check if CSEG data
   */

   for (i = 0; i < maxcsegs; i++)
   {
      pcreg = 0;
      disprog (CSEG, i);
   }

   /*
   ** Check if DSEG data
   */

   if (dseglen)
   {
      pcreg = 0;
      disprog (DSEG, 0);
   }

   inbuf[0] = 0;
   if (entry >= 0)
   {
      SymNode *s;
      if ((s = vallookup (entry, FALSE, 0, 0)) != NULL)
	 sprintf (inbuf, "\t%s", s->symbol);
      else
	 sprintf (inbuf, "\t>%04X", entry);
   }
   printf ("\tEND%s\n", inbuf);

   /*
   ** Close the input
   */

   fclose (infd);

   return (status == 0 ? NORMAL : ABORT);
}
