/***********************************************************************
*
* objcmp.c - Object compare (Loads into memory).
*
* Changes:
*   05/29/03   DGP   Original.
*   07/02/03   DGP   Added compressed binary support.
*   12/30/03   DGP   Moved loader code to loadrec function.
*   04/12/04   DGP   Changed to allow non-80 byte records.
*   08/03/11   DGP   Generate error if program is too big.
*
***********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <signal.h>
#include <ctype.h>
#include <errno.h>

#include "utildef.h"
#include "hexdump.h"
#include "support.h"

#define MAXMEMLEN 64*1024
#define LOADADDRESS 0xA0

static int pgmlen;
static int firstaddr = -1;

/***********************************************************************
* binloader - Load object into specified memory area.
***********************************************************************/

static int
binloader (FILE *lfd, unsigned char *memory, int loadpt)
{
   int done;
   int status;
   int binarymode;
   int wordtaglen;
   int reccount;
   int loadaddr = LOADADDRESS;
   int curraddr = LOADADDRESS;
   unsigned char inbuf[82];

   if (loadpt >= 0)
   {
      loadaddr = loadpt;
      curraddr = loadpt;
   }

   binarymode = FALSE;
   status = fgetc (lfd);
   if (status == BINIDT_TAG)
   {
      binarymode = TRUE;
   }
   ungetc (status, lfd);

   done = FALSE;
   reccount = 0;
   wordtaglen = WORDTAGLEN;

   while (!done)
   {
      unsigned char *op;
      int i;

      if (binarymode)
      {
	 fread (inbuf, 1, 81, lfd);
	 if (feof(lfd))
	    return (-1);
      }
      else
      {
	 if (fgets ((char *)inbuf, 82, lfd) == NULL)
	    return (-1);
      }

#ifdef DEBUG
      printf ("record: %d\n", ++reccount);
      hexdump (stdout, (char *)inbuf, 0, 80);
#endif
      op = (unsigned char *)inbuf;

      if (*op == EOFSYM)
      {
	 return (0);
      }

      for (i = 0; i < 80; i++)
      {
	 int tmp;
	 unsigned int wdata;
	 unsigned char otag;
	 unsigned char item[16];

	 otag = *op++;
	 if ((otag == BINIDT_TAG) || binarymode)
	 {
	    wdata = (*op << 8) | *(op+1);
	 }
	 else
	 {
	    memcpy (item, op, 4);
	    item[4] = '\0';
	    sscanf ((char *)item, "%4X", &tmp);
	    wdata = tmp & 0xFFFF;
	 }
	 wdata &= 0xFFFF;

#ifdef DEBUG
	 printf ("tag = %c(%02X), wdata = %04X\n", otag, otag, wdata);
#endif
	 switch (otag)
	 {
	 case BINIDT_TAG: /* Binary IDT */
	    binarymode = TRUE;
	    wordtaglen = BINWORDTAGLEN;
	    wdata = (*op << 8) | *(op+1);
	 case IDT_TAG:
	    pgmlen = wdata;
	    wdata += loadaddr;
	    if (wdata >= MAXMEMLEN)
	    {
	       fprintf (stderr, "binloader: Program too large for memory");
	       return (-21);
	    }
	    op += wordtaglen-1;
	    printf ("IDT = %-8.8s, length = %04X", op, pgmlen);
	    op += IDTSIZE;
	    break;

	 case RELORG_TAG:
	    wdata += loadaddr;
	 case ABSORG_TAG:
	    curraddr = wdata;
	    op += wordtaglen-1;
	    break;

	 case RELDATA_TAG:
	    wdata += loadaddr;
	 case ABSDATA_TAG:
	    if (firstaddr < 0) firstaddr = curraddr;
	    PUTMEM (curraddr, wdata);
	    curraddr += 2;
	    op += wordtaglen-1;
	    break;

	 case RELENTRY_TAG:
	    wdata += loadaddr;
	 case ABSENTRY_TAG:
	    op += wordtaglen-1;
	    break;

	 case PGMIDT_TAG:
	    op += IDTSIZE;
	 case NOCKSUM_TAG:
	 case CKSUM_TAG:
	    op += wordtaglen-1;
	    break;

	 case LRELGLOBAL_TAG:
	 case LABSGLOBAL_TAG:
	 case LRELEXTRN_TAG:
	 case LABSEXTRN_TAG:
	 case LRELSYMBOL_TAG:
	 case LABSSYMBOL_TAG:
	    op += (MAXSYMLEN - MAXSHORTSYMLEN);

	 case RELGLOBAL_TAG:
	 case ABSGLOBAL_TAG:
	 case RELEXTRN_TAG:
	 case ABSEXTRN_TAG:
	 case RELSREF_TAG:
	 case LOAD_TAG:
	 case RELSYMBOL_TAG:
	 case ABSSYMBOL_TAG:
	    op += MAXSHORTSYMLEN;

	 case DSEGORG_TAG:
	 case DSEGDATA_TAG:
	    op += wordtaglen-1;
	    break;

	 case CMNGLOBAL_TAG:
	 case CMNEXTRN_TAG:
	 case COMMON_TAG:
	    op += MAXSHORTSYMLEN;
	 case CMNORG_TAG:
	    op += wordtaglen-1;
	    op += wordtaglen-1;
	    break;

	 case CMNDATA_TAG:
	    op += wordtaglen-1;
	    if (firstaddr < 0) firstaddr = curraddr;
	    PUTMEM (curraddr, wdata);
	    curraddr += 2;
	    op += wordtaglen-1;
	    break;

	 case EXTNDX_TAG:
	    op += wordtaglen-1;
	    if (firstaddr < 0) firstaddr = curraddr;
	    PUTMEM (curraddr, wdata);
	    curraddr += 2;
	    op += wordtaglen-1;
	    break;

	 case EOR_TAG:
	    i = 81;
	    break;

	 default: 
	    fprintf (stderr, "binloader: Illegal object tag: %c(%02X)\n",
		     otag, otag);
	    return (-22);
	 }
      }
   }

   if (!pgmlen)
      pgmlen = curraddr - loadaddr;

   return (0);
}

/***********************************************************************
* Main procedure
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *fd1, *fd2;
   int done;
   int loadpt;
   unsigned char memory1[MAXMEMLEN];
   unsigned char memory2[MAXMEMLEN];

   if (argc < 3)
   {
      fprintf (stderr, "usage: objcmp file1.obj file2.obj [loadadd]\n");
      exit (1);
   }
   loadpt = 0;
   if (argc == 4)
   {
      loadpt = getnum (argv[3]);
   }

   if ((fd1 = fopen (argv[1], "rb")) == NULL)
   {
      fprintf (stderr, "Can't open file 1: %s: %s\n",
	       argv[1], strerror (errno));
      exit (1);
   }
   if ((fd2 = fopen (argv[2], "rb")) == NULL)
   {
      fprintf (stderr, "Can't open file 2: %s: %s\n",
	       argv[1], strerror (errno));
      fclose (fd1);
      exit (1);
   }

   done = FALSE;
   while (!done)
   {
      memset (memory1, 0, MAXMEMLEN);
      memset (memory2, 0, MAXMEMLEN);

      if (binloader (fd1, memory1, loadpt) < 0)
         break;
      fputs (", ", stdout);
      if (binloader (fd2, memory2, loadpt) < 0)
         break;

      if (firstaddr < 0) firstaddr = loadpt;
      fputc ('\n', stdout);
      if (memcmp (memory1, memory2, MAXMEMLEN))
      {
	 int i;

         fprintf (stdout, "COMPARE Error\n");
	 for (i = 0; i < pgmlen; i+=2)
	 {
	    unsigned char *memory;
	    uint16 d1, d2;

	    memory = memory1;
	    d1 = GETMEM (firstaddr + i);
	    memory = memory2;
	    d2 = GETMEM (firstaddr + i);
	    if (d1 != d2)
	    {
	       fprintf (stdout, "  %04X: %04X %04X\n", firstaddr + i, d1, d2);
	    }
	 }
#ifdef DUMPMEM
	 fprintf (stderr, "Memory 1:\n");
	 hexdump (stderr, (char *)&memory1[firstaddr], firstaddr, pgmlen);
	 fprintf (stderr, "Memory 2:\n");
	 hexdump (stderr, (char *)&memory2[firstaddr], firstaddr, pgmlen);
#endif
	 break;
      }
   }
   fclose (fd1);
   fclose (fd2);
   return (0);
}
