/***********************************************************************
*
* SCANDISK - Scans a TI990 tiline disk image file for a pattern.
*
* Changes:
*      11/30/07   DGP   Original.
*      10/07/18   DGP   Fixed overhead.
*
***********************************************************************/
 
#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>

#define NORMAL		0
#define ABORT		16

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#include "simdsk.h"
#include "hexdump.h"
#include "support.h"

/***********************************************************************
* checkstring - Check for string pattern in this sector.
***********************************************************************/

static void
checkstring (char *buf, int cyl, int head, int sect, char *str, int sectlen)
{
   int i;
   int len = strlen (str);
   int dumpit;

#ifdef DEBUGDATA
   printf ("checkstring: cyl = %d, head = %d, sect = %d, string = '%s'\n",
	   cyl, head, sect, str);
#endif
   dumpit = FALSE;
   for (i = 0; i < sectlen; i++)
   {
      if (!strncmp (&buf[i], str, len))
      {
	 if (!dumpit)
	 {
	    printf ("\n");
	    dumpit = TRUE;
	 }

	 /*
	 ** Found the pattern, log the location.
	 */

	 printf ("cyl %d head %d sect %d loc >%02X\n",
		 cyl, head, sect, i);
      }
      else if ((i + len) > sectlen)
      {
         len--;
	 if (len == 2)
	    break;
      }
   }

   /*
   ** If we found the string, dump the sector.
   */

   if (dumpit)
   {
      HEXDUMP (stdout, buf, sectlen, 0);
   }
}

/***********************************************************************
* checksect - Check for pattern in this sector.
***********************************************************************/

static void
checksect (char *buf, int cyl, int head, int sect, int pattern, int sectlen)
{
   int i;
   int dumpit;

#ifdef DEBUGDATA
   printf ("checksect: cyl = %d, head = %d, sect = %d, pattern = >%04X\n",
	   cyl, head, sect, pattern);
#endif
   dumpit = FALSE;
   for (i = 0; i < sectlen; i += 2)
   {
      int test;

      test = ((buf[i] & 0xFF) << 8) | (buf[i+1] & 0xFF);

#ifdef DEBUGDATA
      printf ("   i = %3d(>%04X), test = >%04X\n", i, i, test);
#endif
      if (test == pattern)
      {
	 if (!dumpit)
	 {
	    printf ("\n");
	    dumpit = TRUE;
	 }

	 /*
	 ** Found the pattern, log the location.
	 */

	 printf ("cyl %d head %d sect %d loc >%02X\n",
		 cyl, head, sect, i);
      }
   }   

   /*
   ** If we found the pattern, dump the sector.
   */

   if (dumpit)
   {
      HEXDUMP (stdout, buf, sectlen, 0);
   }
}

/***********************************************************************
* main
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *fd;
   char *fname;
   char *str;
   char *model;
   int dskloc;
   int realdev;
   int i;
   int j;
   int ic, ih, is;
   int cyls, heads, sectrk, overhead, sectlen;
   int tracklen;
   int head, cyl;
   int pattern;
   int scanstring;
   int start, end;
   int startcyl, endcyl;
   int starthead, endhead;
   int startsect, endsect;
   int dumpdisk;
   char buf[512];
   
   fname = NULL;
   model = NULL;
   pattern = -1;
   cyl = -1;
   head = -1;
   start = -1;
   end = -1;
   dumpdisk = 0;
   scanstring = 0;
   realdev = 0;

   j = 0;
   for (i = 1; i < argc; i++)
   {
      if (argv[i][0] == '-')
      {
         if (argv[i][1] == 'm')
	 {
	    i++;
	    model = argv[i];
	 }
	 else
	    goto USAGE;
      }
      else switch (j++)
      {
      case 0:
         fname = argv[i];
	 break;
      case 1:
         pattern = getnum(argv[i]);
	 if (pattern == -1)
	 {
	    dumpdisk = 1;
	    pattern = 0;
	 }
	 else if (pattern == -2)
	 {
	    scanstring = 1;
	    pattern = 0;
	    i++;
	    str = argv[i];
	 }
	 pattern &= 0xFFFF;
	 break;
      case 2:
         cyl = getnum(argv[i]);
	 break;
      case 3:
         head = getnum(argv[i]);
	 break;
      case 4:
         start = getnum(argv[i]);
	 end = start;
	 break;
      case 5:
         end = getnum(argv[i]);
	 break;
      default:
      USAGE:
	 fprintf (stderr,
   "usage: scandisk [-m model] disk.file pattern [cylinder [head [startsec [endsec]]]]\n");
	 exit (ABORT);
      }
   }
   if (fname == NULL) goto USAGE;
   if (!strncmp (fname, "/dev/", 5))
   {
      if (model == NULL)
      {
	 fprintf (stderr, "You must specify a model for a real device.\n");
	 goto USAGE;
      }
      realdev = 1;
   }

   if (pattern < 0) goto USAGE;
   if (!dumpdisk && !scanstring)
      printf ("scandisk: pattern = >%04X\n", pattern);
   if (scanstring)
      printf ("scandisk: pattern = '%s'\n", str);

   /*
   ** Open the disk image file
   */

   if ((fd = fopen (fname, "rb")) == NULL)
   {
      perror ("Can't open disk file");
      exit (ABORT);
   }

   /*
   ** Read the disk geometry
   */

   if (!realdev)
   {
      if ((cyls = dskreadint (fd)) < 0)
      {
	 perror ("Can't read disk geometry");
	 exit (ABORT);
      }
      if ((heads = dskreadint (fd)) < 0) 
      {
	 perror ("Can't read disk geometry");
	 exit (ABORT);
      }
      heads &= 0xFFFF;
      if ((dskloc = dskreadint (fd)) < 0)
      {
	 perror ("Can't read disk geometry");
	 exit (ABORT);
      }
      sectrk = dskloc & 0xFF;
      overhead = (dskloc >> 8) & 0x3FF;
      if ((sectlen = dskreadint (fd)) < 0)
      {
	 perror ("Can't read disk geometry");
	 exit (ABORT);
      }

      tracklen = sectrk * (sectlen + overhead);

      for (i = 0; i < MAXDISKS; i++)
      {
	 if (disks[i].cyls == cyls &&
	     disks[i].heads == heads &&
	     disks[i].sectrk == sectrk &&
	     disks[i].bytsec == sectlen &&
	     disks[i].overhead == overhead)
	    break;
      }
      if (i == MAXDISKS)
      {
	 printf ("*Unknown Disk model*\n");
      }
   }
   else
   {
      for (i = 0; i < MAXDISKS; i++)
      {
	 if (!strcmp (disks[i].model, model))
	    break;
      }
      if (i == MAXDISKS)
      {
	 fprintf (stderr, "Unknown Disk model*\n");
	 exit (ABORT);
      }
      cyls = disks[i].cyls;
      heads = disks[i].heads;
      sectrk  = disks[i].sectrk;
      sectlen = disks[i].bytsec;
      overhead = disks[i].overhead;
      tracklen = sectrk * (sectlen + overhead);
   }
	       
   printf ("%s disk geometry:\n", disks[i].model);
   printf ("   cyls      = %d\n", cyls);
   printf ("   heads     = %d\n", heads);
   printf ("   sectrk    = %d\n", sectrk);
   printf ("   sectlen   = %d\n", sectlen);
   printf ("   overhead  = %d\n", overhead);
   printf ("   tracklen  = %d\n", tracklen);
   printf ("   disk size = %d bytes\n", tracklen * heads * cyls);

   if (cyl > 0 && cyl >= cyls)
   {
      printf ("Invalid disk cylinder specified, abort\n");
      exit (ABORT);
   }
   if (head > 0 && head >= heads)
   {
      printf ("Invalid disk head specified, abort\n");
      exit (ABORT);
   }
   if (start > 0 && start >= sectrk)
   {
      printf ("Invalid disk start sector specified, abort\n");
      exit (ABORT);
   }
   if (end > 0 && end >= sectrk)
   {
      printf ("Invalid disk end sector specified, abort\n");
      exit (ABORT);
   }

   /*
   ** Scan whole disk is default.
   */

   startcyl = 0;
   endcyl = cyls;
   starthead = 0;
   endhead = heads;
   startsect = 0;
   endsect = sectrk;

   if (cyl >= 0)
   {
      startcyl = cyl;
      endcyl = cyl + 1;
      if (head >= 0)
      {
         starthead = head;
	 endhead = head + 1;
	 if (start >= 0)
	 {
	    startsect = start;
	    endsect = start + 1;
	    if (end >= 0)
	    {
	       endsect = end;
	    }
	 }
      }
   }

   for (ic = startcyl; ic < endcyl; ic++)
   {
      for (ih = starthead; ih < endhead; ih++)
      {
	 for (is = startsect; is < endsect; is++)
	 {
	    /*
	    ** Calculate location
	    */

	    dskloc = (tracklen * ih) + (tracklen * heads * ic) +
	       ((overhead + sectlen) * is) + (realdev ? 0 : DSKOVERHEAD);

#ifdef DEBUG
	    printf ("   cyl = %d, head = %d, sector = %d",
		     ic, ih, is);
	    printf (", dskloc = %d\n", dskloc);
#endif

	    /*
	    ** Go to the specified location on disk
	    */

	    if (fseek (fd, dskloc, SEEK_SET) < 0)
	    {
	       perror ("seek failed");
	       exit (ABORT);
	    }

	    /*
	    ** Read sector
	    */

	    for (i = 0; i < sectlen; i++)
	       buf[i] = fgetc (fd);

	    if (dumpdisk)
	    {
	       printf ("cyl %d head %d sect %d:\n", ic, ih, is);
	       HEXDUMP (stdout, buf, sectlen, 0);
	    }
	    else if (scanstring)
	    {
	       checkstring (buf, ic, ih, is, str, sectlen);
	    }
	    else
	    {
	       checksect (buf, ic, ih, is, pattern, sectlen);
	    }
	 }
      }
   }

   return (NORMAL);
}
