#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/dir.h>
#include <dirent.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>

#define TBLOCK	512
#define NBLOCK	20
#define NAMSIZ	100

#define SUID	04000
#define SGID	02000
#define ROWN	0400
#define WOWN	0200
#define XOWN	0100
#define RGRP	040
#define WGRP	020
#define XGRP	010
#define ROTH	04
#define WOTH	02
#define XOTH	01
#define STXT	01000

static int m1[] = { 1, ROWN, 'r', '-' };
static int m2[] = { 1, WOWN, 'w', '-' };
static int m3[] = { 2, SUID, 's', XOWN, 'x', '-' };
static int m4[] = { 1, RGRP, 'r', '-' };
static int m5[] = { 1, WGRP, 'w', '-' };
static int m6[] = { 2, SGID, 's', XGRP, 'x', '-' };
static int m7[] = { 1, ROTH, 'r', '-' };
static int m8[] = { 1, WOTH, 'w', '-' };
static int m9[] = { 2, STXT, 't', XOTH, 'x', '-' };

static int *m[] = { m1, m2, m3, m4, m5, m6, m7, m8, m9 };

static union hblock
{
   char dummy[TBLOCK];
   struct header
   {
      char name[NAMSIZ];
      char mode[8];
      char uid[8];
      char gid[8];
      char size[12];
      char mtime[12];
      char chksum[8];
      char linkflag;
      char linkname[NAMSIZ];
   } dbuf;
} dblock, tbuf[NBLOCK];

static int buflen;
static char bufrec[TBLOCK];

static struct stat stbuf;
static long st_size;

static int cflag, xflag, vflag, tflag;
static int chksum;

static int recno = 0;
static int first = 0;
static int nblock = NBLOCK;
static int mt;

static char *usefile;
static char magtape[] = "MT01";
static char dxwd[256];

#if defined(DUMPTAPE)
#include "hexdump.h"
#endif

int
atoio (char *p)
{
   int ret = 0;

   for (; *p >= '0' && *p < '8'; p++)
   {
      ret = (ret << 3) | (*p - '0');
   }
   return ret;
}

void
atolo (char *p, long *v)
{
   long ret = 0;

   for (; *p >= '0' && *p < '8'; p++)
   {
      ret = (ret << 3) | (*p - '0');
   }
   *v = ret;
   return;
}

void
copy (char *to, char *from)
{
   int i;

   i = TBLOCK;
   do
   {
      *to++ = *from++;
   }
   while (--i);
}

int
readtape (char *buffer)
{
   int i, j;

   if (recno >= nblock || first == 0)
   {
      if (first == 0 && nblock == 0)
         j = NBLOCK;
      else
         j = nblock;
      if ((i = read (mt, tbuf, TBLOCK * j)) < 0)
      {
         fprintf (stderr, "tar: tape read error\n");
         exit (3);
      }
      if (first == 0)
      {
         if ((i % TBLOCK) != 0)
         {
            fprintf (stderr, "tar: tape blocksize error\n");
            exit (3);
         }
         i /= TBLOCK;
         if (i != nblock)
         {
            fprintf (stderr, "tar: blocksize = %d\n", i);
            nblock = i;
         }
      }
      recno = 0;
   }
   first = 1;
   copy (buffer, (char *)&tbuf[recno++]);
#ifdef DUMPTAPE
   fprintf (stderr, "BLK: recno = %d\n", recno);
   hexdump (stderr, buffer, 0, TBLOCK);
#endif
   return (TBLOCK);
}

int
writetape (char *buffer)
{
   first = 1;
   if (nblock == 0)
      nblock = 1;

#ifdef DUMPTAPE
   fprintf (stderr, "BLK: recno = %d\n", recno);
   hexdump (stderr, buffer, 0, TBLOCK);
#endif
   if (recno >= nblock)
   {
      if (write (mt, tbuf, TBLOCK*nblock) < 0)
      {
	 fprintf (stderr, "tar: tape write error\n");
	 exit(3);
      }
      recno = 0;
   }
   copy ((char *)&tbuf[recno++], buffer);
   if (recno >= nblock)
   {
      if (write (mt, tbuf, TBLOCK*nblock) < 0)
      {
	 fprintf (stderr, "tar: tape write error\n");
	 exit(3);
      }
      recno = 0;
   }
   return (TBLOCK);
}

int
writebuf (char *buf, int len)
{
   char *bp;
   int wblk;
   int i;

   wblk = 0;
   bp = &bufrec[buflen];
   for (i = 0; i < len; i++)
   {
      if (buflen == TBLOCK)
      {
         writetape (bufrec);
	 wblk++;
	 buflen = 0;
	 bp = bufrec;
      }
      *bp++ = *buf++;
      buflen++;
   }
   return (wblk);
}

int
flushbuf (void)
{
   int wblk;
   int i;

   wblk = 0;
   if (buflen > 0)
   {
      for (i = buflen; i < TBLOCK; i++)
      {
         bufrec[i] = '\0';
      }
      writetape (bufrec);
      wblk = 1;
      buflen = 0;
   }
   return (wblk);
}

int
endtape ()
{
   if (dblock.dbuf.name[0] == '\0')
      return (1);
   return (0);
}

void
passtape ()
{
   long blocks;
   char buf[TBLOCK];

   if (dblock.dbuf.linkflag == '1')
      return;
   blocks = st_size;
   blocks += TBLOCK - 1;
   blocks /= TBLOCK;

   while (blocks-- > 0)
      readtape (buf);
}

void
tomodes(struct stat *sp)
{
   char *cp;

   for (cp = dblock.dummy; cp < &dblock.dummy[TBLOCK]; cp++)
      *cp = '\0';
   sprintf (dblock.dbuf.mode, "%6o ", sp->st_mode & 07777);
   sprintf (dblock.dbuf.uid, "%6o ", 200);
   sprintf (dblock.dbuf.gid, "%6o ", 200);
   sprintf (dblock.dbuf.size, "%11lo ", sp->st_size);
   sprintf (dblock.dbuf.mtime, "%11lo ", sp->st_atime);
}

checksum ()
{
   int i;
   char *cp, *ep;

   ep = dblock.dbuf.chksum + sizeof (dblock.dbuf.chksum);
   for (cp = dblock.dbuf.chksum; cp < ep; cp++)
      *cp = ' ';
   i = 0;
   ep = &dblock.dummy[TBLOCK];
   for (cp = dblock.dummy; cp < ep; cp++)
      i += *cp;
   return (i);
}

void
getdir ()
{
   struct stat *sp;
   int i, sum;

   readtape ((char *) &dblock);
   if (dblock.dbuf.name[0] == '\0')
      return;
   sp = &stbuf;
   i = atoio (dblock.dbuf.mode);
   sp->st_mode = i;
   i = atoio (dblock.dbuf.uid);
   sp->st_uid = i;
   i = atoio (dblock.dbuf.gid);
   sp->st_gid = i;
   atolo (dblock.dbuf.size, &st_size);
   atolo (dblock.dbuf.mtime, &sp->st_mtime);
   chksum = atoio (dblock.dbuf.chksum);
   sum = checksum ();
   if (chksum != sum)
   {
      fprintf (stderr, "tar: directory checksum error: >%04X != >%04X\n",
               chksum, sum);
      exit (2);
   }
}

char
response ()
{
   char c;

   c = getchar ();
   if (c != '\n')
      while (getchar () != '\n');
   else
      c = 'n';
   return (c);
}

void
selmode (int *pairp, struct stat *st)
{
   int n, *ap;

   ap = pairp;
   n = *ap++;
   while (--n >= 0 && (st->st_mode & *ap++) == 0)
      ap++;
   printf ("%c", *ap);
}

void
pmode (struct stat *st)
{
   int **mp;

   for (mp = &m[0]; mp < &m[9];)
      selmode (*mp++, st);
}

void
longt (struct stat *st)
{
   char *cp;

   pmode (st);
   printf ("%3d/%3d ", st->st_uid, st->st_gid);
   printf ("%7ld", st_size);
   cp = ctime (&st->st_mtime);
   cp[16] = 0;
   cp[24] = 0;
   printf (" %s %s ", cp + 4, cp + 20);
}

#ifdef EXTRACT
void
cvtname (char *uname, char *dxdir, char *dxname)
{
   char *cp;
   char *ep;
   int len;
   char lclname[NAMSIZ+2];

   strcpy (lclname, uname);
   strcpy (dxname, dxdir);
   cp = lclname;
   if ((ep = strrchr (cp, '/')) != NULL)
   {
      cp = ep + 1;
   }
   if ((ep = strrchr (cp, '.')) != NULL)
   {
      *ep++ = 0;
   }
   len = 0;
   for (ep = cp; *ep; ep++)
   {
      if (*ep == '_') *ep = '$';
      if (*ep == '.') *ep = '$';
      if (len >= 8)
      {
         *ep = '\0';
      }
      len++;
   }
   strcat (dxname, cp);
}

checkdir (char *uname, char *dxname)
{
   char *ep;
   char *dp, *up;
   int len;
   char name[NAMSIZ+2];

   /* Copy UNIX name */
   strcpy (name, uname);
   up = name;

   /* If it starts with '/' start at ".", volume root */
   if (*up == '/')
   {
      strcpy (dxname, ".");
      up++;
   }

   /* Otherwise, start at WD synonym (env) */
   else 
   {
      strcpy (dxname, dxwd);
      strcat (dxname, ".");
   }
   dp = dxname + strlen (dxname);

   /* Isolate extension, we will create a dir for it */
   ep = NULL;
   if (uname[strlen(uname)-1] != '/')
   {
      if ((ep = strrchr (up, '/')) != NULL)
      {
	 *ep++ = '/';
	 *ep++ = '\0';
	 ep = strrchr (ep, '.');
      }
   }

   len = 0;
   for (; *up; up++)
   {
      if (*up == '/')
      {
	 len = 0;
         *dp = '\0';
#ifdef DEBUGX
	 printf ("checkdir: dxname-1 = '%s'\n", dxname);
#endif
         if (access (dxname, W_OK) < 0)
         {
#ifdef DEBUGX
	    printf ("checkdir: access-1 errno = %d\n", errno);
#endif
	    if (mkdir (dxname, 0777) < 0)
	    {
	       fprintf (stderr, "tar: %s -> %s - cannot mkdir: %s\n",
			dblock.dbuf.name, dxname, strerror (errno));
	       exit (0);
	    }
         }
#ifdef DEBUGX
         else
	 {
	    printf ("checkdir: access-1 OK\n");
	 }
#endif
         *dp++ = '.';
      }
      else if (len < 8)
      {
	 if (*up == '.')
	    *dp++ = '$';
	 else 
	    *dp++ = *up;
	 len++;
      }
   }
   *dp = '\0';

   if (ep)
   {
      ep++;
      len = 0;
      while (*ep)
      {
	 if (len < 8)
	 {
	    *dp++ = *ep++;
	    len++;
	 }
      }
      *dp = '\0';
#ifdef DEBUGX
      printf ("checkdir: dxname-2 = '%s'\n", dxname);
#endif
      if (access (dxname, W_OK) < 0)
      {
#ifdef DEBUGX
	 printf ("checkdir: access-2 errno = %d\n", errno);
#endif
	 if (mkdir (dxname, 0) < 0)
	 {
	    fprintf (stderr, "tar: %s -> %s - cannot mkdir: %s\n",
		     dblock.dbuf.name, dxname, strerror (errno));
	    exit (0);
	 }
      }
#ifdef DEBUGX
      else
      {
	 printf ("checkdir: access-2 OK\n");
      }
#endif
      *dp++ = '.';
      *dp = '\0';
   }
}
#endif

int
prefix (char *s1, char *s2)
{
   while (*s1)
      if (*s1++ != *s2++)
         return (0);
   if (*s2)
      return (*s2 == '/');
   return (1);
}

void
flushtape(void)
{
   write(mt, tbuf, TBLOCK*nblock);
}

void
putempty(void)
{
   char *cp;
   char buf[TBLOCK];

   for (cp = buf; cp < &buf[TBLOCK]; )
      *cp++ = '\0';
   writetape (buf);
}

void
putfile (char *plongname, char *pshortname)
{
   long blocks;
   char *cp, *cp2;
   DIR *dp;
   struct dirent *ep;
   int fd;
   int i, j;
   char ch;
   char buf[TBLOCK];
   char shortname[256];
   char longname[256];
    
   strcpy (shortname, pshortname);
   strcpy (longname, plongname);

   if (stat (longname, &stbuf) < 0)
   {
      fprintf (stderr, "tar: %s: cannot stat file\n", longname);
      return;
   }

   if ((stbuf.st_mode & S_IFMT) == S_IFDIR)
   {
      if ((dp = opendir (longname)) == NULL)
      {
	 fprintf (stderr, "tar: %s: cannot stat file\n", longname);
	 return;
      }
      for (i = 0, cp = buf; *cp++ = longname[i++];);
      *--cp = '.';
      cp++;
      i = 0;
      while ((ep = readdir (dp)) != NULL)
      {
	 cp2 = cp;
	 for (j=0; j < DIRSIZ; j++)
	    *cp2++ = ep->d_name[j];
	 *cp2 = '\0';
	 putfile (buf, cp);
      }
      closedir (dp);
      return;
   }

   if ((stbuf.st_mode & S_IFMT) != S_IFREG)
   {
      fprintf (stderr, "tar: %s is not a regular file. Not dumped\n", longname);
      return;
   }

   if (stbuf.st_size == 0)
   {
      fprintf (stderr, "tar: %s zero length file. Not dumped\n", longname);
      return;
   }

   tomodes (&stbuf);

   cp2 = longname;
   cp = dblock.dbuf.name;
   for (i = 0; i < NAMSIZ; i++)
   {
      ch = *cp2++;
      if (!ch) break;
      if (ch == '.') ch = '/';
      else if (ch == '$') ch = '_';
      else if (ch >= 'A' && ch <= 'Z') ch += 0x20;
      *cp++ = ch;
   }

   if (i >= NAMSIZ)
   {
      fprintf (stderr, "tar: %s: file name too long\n", longname);
      return;
   }

   fd = open (longname, O_RDONLY);

   blocks = (stbuf.st_size + (TBLOCK-1)) / TBLOCK;
   if (vflag)
   {
      fprintf (stderr, "a %s ", longname);
      fprintf (stderr, "%ld blocks\n", blocks);
   }
   sprintf (dblock.dbuf.chksum, "%6o", checksum());
   writetape ((char *)&dblock);

   buflen = 0;
   while ((i = read (fd, buf, TBLOCK)) > 0 && blocks > 0)
   {
      buf[i++] = '\n';
      blocks =  blocks - writebuf (buf, i);
   }
   blocks =  blocks - flushbuf (); 
   close (fd);
   //if (blocks != 0 || i != 0)
   //   fprintf (stderr, "tar: %s: file changed size\n", longname);
   while (blocks-- >  0)
      putempty();
}

void
docreat (char *argv[])
{
   char *cp, *cp2;

   while (*argv)
   {
      cp2 = *argv;
      for (cp = *argv; *cp; cp++)
	 if (*cp == '.')
	    cp2 = cp;
      if (cp2 != *argv) {
	 *cp2 = '\0';
	 printf("dir = %s\n", *argv);
	 *cp2 = '.';
	 cp2++;
      }
      putfile(*argv++, cp2);
   }

   putempty();
   putempty();
   flushtape();
}

#ifdef EXTRACT
void
doxtract (char *argv[])
{
   long blocks, bytes;
   char **cp;
   char *ep;
   char *rp;
   int ofile;
   int  blklen, reclen;
   char buf[TBLOCK];
   char dxdir[256];
   char dxname[256];
   char record[256];

   for (;;)
   {
      getdir ();
      if (endtape ())
         break;

      if (*argv == 0)
         goto gotit;

      for (cp = argv; *cp; cp++)
         if (prefix (*cp, dblock.dbuf.name))
            goto gotit;
      passtape ();
      continue;

    gotit:
#ifdef DEBUGX 
      printf ("extract: gotit: name = '%s'\n", dblock.dbuf.name);
#endif
      checkdir (dblock.dbuf.name, dxdir);
      if ((ep = strrchr (dblock.dbuf.name, '/')) != NULL)
      {
         if (*(ep + 1) == 0)
         {
            if (vflag)
               fprintf (stderr,
                        "x %s, %ld bytes, %ld tape blocks\n",
                        dblock.dbuf.name, 0L, 0L);
            continue;
         }
      }

      if (dblock.dbuf.linkflag == '1')
      {
         fprintf (stderr, "tar: links are not supported: %s\n",
		  dblock.dbuf.name);
         continue;
      }

      cvtname (dblock.dbuf.name, dxdir, dxname);
#ifdef DEBUGX
      printf ("extract: dxname = '%s'\n", dxname);
#endif
      if ((ofile = creat (dxname, stbuf.st_mode & 07777)) < 0)
      {
         fprintf (stderr, "tar: %s -> %s - cannot create: %s\n",
		  dblock.dbuf.name, dxname, strerror (errno));
         passtape ();
         continue;
      }

      blocks = ((bytes = st_size) + TBLOCK - 1) / TBLOCK;
      if (vflag)
         fprintf (stderr, "x %s -> %s, %ld bytes, %ld tape blocks\n",
                  dblock.dbuf.name, dxname, bytes, blocks);

      rp = record;
      reclen = 0;
      blklen = TBLOCK;
      while (blocks-- > 0)
      {
	 char c;

	 ep = buf;
         readtape (buf);

	 while (bytes)
	 {
	    c = *ep++;
	    bytes--;
	    blklen--;
	    if (c == '\n')
	    {
	       *rp = 0;
	       if (write (ofile, record, reclen) < 0)
	       {
		  fprintf (stderr, "tar: %s: HELP - extract write error: %s\n",
			   dxname, strerror(errno));
		  exit (2);
	       }
	       rp = record;
	       reclen = 0;
	    }
	    else
	    {
	       *rp++ = c;
	       reclen++;
	    }
	    if (blklen == 0)
	    {
	       blklen = TBLOCK;
	       break;
	    }
	 }
      }
      close (ofile);
   }
}
#endif

void
dotable ()
{
   for (;;)
   {
      getdir ();
      if (endtape ())
         break;
      if (vflag)
         longt (&stbuf);
      printf ("%s", dblock.dbuf.name);
      if (dblock.dbuf.linkflag == '1')
         printf (" linked to %s", dblock.dbuf.linkname);
      printf ("\n");
      passtape ();
   }
}

void
usage ()
{
   fprintf (stderr,
#ifdef EXTRACT
            "usage: tar -{ctx}[vfb] [tapefile] [blocksize] file1 file2...\n");
#else
            "usage: tar -{ct}[vfb] [tapefile] [blocksize] file1 file2...\n");
#endif
   exit (1);
}

int
main (int argc, char **argv)
{
   char *cp;

   if (argc < 2)
      usage ();

   usefile = magtape;
   argv[argc] = 0;
   argv++;
   cflag = 0;
   tflag = 0;
   vflag = 0;
   xflag = 0;
   for (cp = *argv++; *cp; cp++)
      switch (*cp)
      {
      case 'f':
         usefile = *argv++;
         nblock = NBLOCK;
         break;
      case 'v':
         vflag++;
         break;
      case 'c':
         cflag++;
         break;
#ifdef EXTRACT
      case 'x':
         xflag++;
         break;
#endif
      case 't':
         tflag++;
         break;
      case '-':
         break;
      case 'b':
         nblock = atoi (*argv++);
         if (nblock > NBLOCK || nblock <= 0)
         {
            fprintf (stderr, "tar: invalid blocksize. (Max %d)\n", NBLOCK);
            exit (1);
         }
         break;
      default:
         fprintf (stderr, "tar: %c: unsupported option\n", *cp);
         usage ();
      }

   strcpy (dxwd, getenv ("wd"));

   if (cflag)
   {
      if ((mt = open (usefile, O_CREAT | O_WRONLY)) < 0)
      {
	 fprintf (stderr, "tar: cannot open %s: %s\n",
		  usefile, strerror (errno));
	 exit (1);
      }

      docreat (argv);
   }
#ifdef EXTRACT
   else if (xflag)
   {
      if ((mt = open (usefile, O_RDONLY)) < 0)
      {
	 fprintf (stderr, "tar: cannot open %s: %s\n",
		  usefile, strerror (errno));
	 exit (1);
      }

      doxtract (argv);
   }
#endif
   else if (tflag)
   {
      if ((mt = open (usefile, O_RDONLY)) < 0)
      {
	 fprintf (stderr, "tar: cannot open %s: %s\n",
		  usefile, strerror (errno));
	 exit (1);
      }

      dotable ();
   }
   else
      usage ();

   lseek (mt, 0, SEEK_SET);
   close (mt);
   return 0;
}
