/***********************************************************************
*
* TLOAD - Load TI-990 disk image from tape.
*
* Changes:
*   06/02/16   DGP   Original
*
***********************************************************************/

#include <stdio.h>
#include <string.h>

#include "tapeio.h"
#include "diskio.h"
#ifdef DEBUG
#include "hexdump.h"
#endif

#define BLOCKSIZE 512
#define SECSBLOCK 40

static char buffer[BLOCKSIZE*SECSBLOCK];

extern void lights(int);

static int
getnum (char *buf)
{
   unsigned char *bp = (unsigned char *)buf;
   return (bp[0] << 8 | bp[1]);
}

int
main (int argc, char **argv)
{
   long outsecs;
   int done;
   int i, j, k;
   int cyls, heads, sects, sectlen;
   int cyl, head, sect, len;
   int zerodisk, maxcyl;
   int verbose;
   int bootload;
   int debug;
   int ch;

   zerodisk = 0;
   verbose = 0;
   bootload = 0;
   debug = 0;
   maxcyl = -1;
   for (i = 1; i < argc; i++)
   {
      char *ap = argv[i];
      if (*ap == '-')
      {
         ap++;
         if (*ap == 'b')
	    bootload = 1;
         else if (*ap == 'd')
	    verbose = debug = 1;
         else if (*ap == 'm')
	 {
	    if (++i >= argc) goto USAGE;
	    maxcyl = atoi(argv[i]);
	 }
         else if (*ap == 'v')
	    verbose = 1;
	 else if (*ap == 'z')
	    zerodisk = 1;
         else
	 {
	 USAGE:
	    printf ("usage: tload [-b][-m nn][-v][-z]\n");
	    return (1);
	 }
      }
      else
         goto USAGE;
   }

   if (diskinit (0) < 0)
   {
      printf ("tload: can't initialize disk\n");
      return (1);
   }
   
   if (diskgeometry (0, &cyls, &heads, &sects, &sectlen) < 0)
   {
      printf ("tload: can't get disk geometry\n");
      return (1);
   }

   if (verbose)
   {
      printf ("tload: Disk Geometry:\n");
      printf ("   cyls = %d, heads = %d, sects = %d, sectlen = %d\n",
	      cyls, heads, sects, sectlen);
   }

   if (bootload)
   {
      if ((k = taperead (0, buffer, sizeof(buffer))) != 0)
      {
	 printf ("tload: can't skip EOF: stat = %d\n", k);
	 return (1);
      }
   }
   else
   {
      if (taperewind (0) < 0)
      {
	 printf ("tload: can't rewind tape\n");
	 return (1);
      }
   }

   if (taperead (0, buffer, BLOCKSIZE) < 0)
   {
      printf ("tload: can't read geometry from tape\n");
      return (1);
   }

   cyl = getnum (&buffer[0]);
   head = getnum (&buffer[2]);
   sect = getnum (&buffer[4]);
   len = getnum (&buffer[6]);
   if (verbose)
   {
      printf ("tload: Disk Geometry on tape:\n");
      printf ("   cyls = %d, heads = %d, sects = %d, sectlen = %d\n",
	      cyl, head, sect, len);
   }

   if (cyl != cyls)
   {
      printf ("tload: geometry mismatch: cyls = %d != %d\n",
	       cyls, cyl);
      return (1);
   }
   if (head != heads)
   {
      printf ("tload: geometry mismatch: heads = %d != %d\n",
	       heads, head);
      return (1);
   }
   if (sect != sects)
   {
      printf ("tload: geometry mismatch: sects = %d != %d\n",
	       sects, sect);
      return (1);
   }
   if (len != sectlen)
   {
      printf ("tload: geometry mismatch: sectlen = %d != %d\n",
	       sectlen, len);
      return (1);
   }

   printf ("tload: press enter to start: ");
   fflush (stdout);
   ch = getchar();

   if (zerodisk)
   {
      if (maxcyl < 0) maxcyl = cyls;
      if (verbose)
	 printf ("tload: zero disk, cyls = %d\n", maxcyl);

      memset (buffer, 0, sectlen);

      for (i = 0; i < maxcyl; i++)
      {
	 for (k = 0; k < heads; k++)
	 {
	    lights (i << 5 | k);
	    for (j = 0; j < sects; j++)
	    {
	       if (diskwrite (0, i, k, j, buffer, sectlen) < 0)
	       {
		  printf ("tload: can't zero disk\n");
		  return (1);
	       }
	    }
	 }
      }
   }

   if (verbose)
      printf ("tload: copy tape to disk\n");

   done = 0;
   outsecs = 0;
   while (!done)
   {
      char *bp;

      if ((len = taperead (0, buffer, sizeof(buffer))) < 0)
      {
	 printf ("tload: can't read data from tape\n");
	 return (1);
      }
      if (len == 0)
      {
	 done = 1;
	 break;
      }

      bp = buffer;
      for (; len > 0; len -= BLOCKSIZE)
      {
	 cyl = getnum (bp);
	 head = getnum (bp+2);
	 sect = getnum (bp+4);
	 lights (cyl << 5 | head);
	 if (debug)
	 {
	    printf ("   write: cyl = %d, head = %d, sect = %d\n",
		    cyl, head, sect);
#ifdef DEBUG
	    hexdump (stdout, bp+8, 0, sectlen);
#endif
	 }
	 if (diskwrite (0, cyl, head, sect, bp+8, sectlen) < 0)
	 {
	    printf ("tload: can't write data to disk\n");
	    return (1);
	 }
	 bp += BLOCKSIZE;
	 outsecs++;
      }
   }

   taperewind (0);
   if (verbose)
      printf ("tload: DONE: %ld sectors written\n", outsecs);
   return (0);
}
