/***********************************************************************
*
* chkdisk - Checks the disk image from the TI computer. Use in
*           conjunction with the 990 program ddisk.asm
*
* Changes:
*      02/17/12   DGP   Original
*      10/07/18   DGP   Fixed overhead.
*
***********************************************************************/
 
#include <stdio.h>
#include <stdlib.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <termios.h>

#define NORMAL	0
#define ABORT	16

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#include "simdsk.h"
#include "support.h"

struct termios cmdtty, runtty;
int fd;

/*
** Terminal routines borrowed from Bub Supnik's simh
*/

int
ttinit (FILE *ifd)
{
   fd = fileno(ifd);
   if (tcgetattr (fd, &cmdtty) < 0) return -1;		/* get old flags */
   runtty = cmdtty;
   runtty.c_lflag = runtty.c_lflag & ~(ECHO | ICANON);	/* no echo or edit */
   runtty.c_oflag = runtty.c_oflag & ~OPOST;		/* no output edit */
   runtty.c_iflag = runtty.c_iflag & ~ICRNL;		/* no cr conversion */
   runtty.c_cc[VINTR] = 0;				/* interrupt */
   runtty.c_cc[VQUIT] = 0;				/* no quit */
   runtty.c_cc[VERASE] = 0;
   runtty.c_cc[VKILL] = 0;
   runtty.c_cc[VEOF] = 0;
   runtty.c_cc[VEOL] = 0;
   runtty.c_cc[VSTART] = 0;				/* no host sync */
   runtty.c_cc[VSUSP] = 0;
   runtty.c_cc[VSTOP] = 0;
#if defined (VREPRINT)
   runtty.c_cc[VREPRINT] = 0;				/* no specials */
#endif
#if defined (VDISCARD)
   runtty.c_cc[VDISCARD] = 0;
#endif
#if defined (VWERASE)
   runtty.c_cc[VWERASE] = 0;
#endif
#if defined (VLNEXT)
   runtty.c_cc[VLNEXT] = 0;
#endif
   runtty.c_cc[VMIN] = 0;				/* no waiting */
   runtty.c_cc[VTIME] = 0;
#if defined (VDSUSP)
   runtty.c_cc[VDSUSP] = 0;
#endif
#if defined (VSTATUS)
   runtty.c_cc[VSTATUS] = 0;
#endif
   return 0;
}

int
ttrunstate (void)
{
   runtty.c_cc[VINTR] = 0;
   if (tcsetattr (fd, TCSAFLUSH, &runtty) < 0) return -1;
   return 0;
}

int
ttcmdstate (void)
{
   if (tcsetattr (fd, TCSAFLUSH, &cmdtty) < 0) return -1;
   return 0;
}

int
ttclose (void)
{
   return ttcmdstate ();
}

int
poll_kbd (void)
{
   int status;
   unsigned char buf[1];

   status = read (fd, buf, 1);
   if (status != 1) return -1;
   else return (buf[0]);
}

/***********************************************************************
* Main program.
***********************************************************************/

int
main(int argc, char **argv)
{
   int  *memory;
   FILE *iterm;
   FILE *oterm;
   FILE *ofd;
   int cyls, heads, sectrk, overhead, sectlen;
   int track = 0;
   int diskndx;
   int disksize;
   int i, k, j;
   int c;
   int tracksize;
   int status;

   /*
   ** Check args
   */

   if (argc != 3)
   {
      fprintf (stderr, "usage: chkdisk model disk.file\n");
      exit (ABORT);
   }

   /*
   ** Open the disk image file
   */

   if ((ofd = fopen (argv[2], "rb")) == NULL)
   {
      perror ("Can't open file");
      exit (ABORT);
   }

   /*
   ** Read the disk geometry
   */

   if ((cyls = dskreadint (ofd)) < 0)
   {
      perror ("Can't read disk geometry");
      fclose (ofd);
      exit (ABORT);
   }
   if ((heads = dskreadint (ofd)) < 0) 
   {
      perror ("Can't read disk geometry");
      fclose (ofd);
      exit (ABORT);
   }
   heads &= 0xFFFF;
   if ((j = dskreadint (ofd)) < 0)
   {
      perror ("Can't read disk geometry");
      fclose (ofd);
      exit (ABORT);
   }
   sectrk = j & 0xFF;
   overhead = (j >> 8) & 0x3FF;
   if ((sectlen = dskreadint (ofd)) < 0)
   {
      perror ("Can't read disk geometry");
      fclose (ofd);
      exit (ABORT);
   }

   /*
   ** Check disk model
   */

   for (diskndx = 0; diskndx < MAXDISKS; diskndx++)
   {
      if (!strcmp (disks[diskndx].model, argv[1])) break;
   }

   if (diskndx == MAXDISKS)
   {
      fprintf (stderr, "chkdisk: Unknown disk model: %s\n", argv[1]);
      fclose (ofd);
      exit (ABORT);
   }

   i = FALSE;
   if (disks[diskndx].cyls == cyls &&
       disks[diskndx].heads == heads &&
       disks[diskndx].sectrk == sectrk &&
       disks[diskndx].bytsec == sectlen &&
       disks[diskndx].overhead == overhead)
      i = TRUE;

   if (!i)
   {
      fprintf (stderr, "chkdisk: Geometry mismatch: %s\n", argv[1]);
      fclose (ofd);
      exit (ABORT);
   }

   /*
   ** Calculate the size of the disk and track for display
   */

   tracksize = (disks[diskndx].overhead + disks[diskndx].bytsec) *
	       disks[diskndx].sectrk;
   disksize = tracksize * disks[diskndx].heads * disks[diskndx].cyls;

   printf ("%s: disksize = %d, tracksize = %d\n\n",
	    argv[1], disksize, tracksize);

   /*
   ** Calculate the size of the disk and track less overhead for transfer
   */

   tracksize = disks[diskndx].bytsec * disks[diskndx].sectrk;
   disksize = tracksize * disks[diskndx].heads * disks[diskndx].cyls;

   /*
   ** Allocate the needed memory
   */

   if ((memory = (int *)malloc (tracksize * sizeof(int))) == NULL)
   {
      fprintf (stderr, "chkdisk: Can't malloc memory\n");
      exit (ABORT);
   }

   /*
   ** Set up the port
   */

   if ((iterm = fopen ("/dev/ttyS0", "rb")) == NULL)
   {
      perror ("Can't open terminal");
      fclose (ofd);
      exit (ABORT);
   }
   if ((oterm = fopen ("/dev/ttyS0", "w")) == NULL)
   {
      perror ("Can't open terminal");
      fclose (ofd);
      exit (ABORT);
   }

   ttinit (iterm);

   /*
   ** Tell the ddisk program to go
   */

   fputc ('\n', oterm);

   ttrunstate();

   /*
   ** Read the disk one track at a time into memory
   */

   status = NORMAL;
   for (k = tracksize, j = i = 0; i < disksize; i++)
   {
      while ((c = poll_kbd()) == -1) ;
      memory[j++] = c;
      k--;
      if (k == 0)
      {
	 printf ("track %d\n", track++);

	 /*
	 ** Check the disk track
	 */

	 for (j = 0; j < tracksize; j++)
	 {
	    if ((c = fgetc (ofd)) < 0)
	    {
	       perror ("Read error");
	       status = ABORT;
	       goto DIE;
	    }
	    if (memory[j] != c)
	    {
	       fprintf (stderr, "chkdisk: Compare error: j = %d\n", j);
	       status = ABORT;
	       goto DIE;
	    }
	 }

	 /*
	 ** If overhead is specified, skip that amount
	 */

	 if (disks[diskndx].overhead)
	 {
	    for (j = 0; j < disks[diskndx].overhead; j++)
	       if ((c = fgetc (ofd)) < 0)
	       {
		  perror ("Read error");
		  status = ABORT;
		  goto DIE;
	       }
	 }

	 k = tracksize;
	 j = 0;

	 /*
	 ** Send go ahead for next track
	 */

	 fputc ('\n', oterm);
      }
   }
   fputc ('\n', oterm); /* Send go ahead to finish */

   /*
   ** Clean up and quit
   */

DIE:
   ttcmdstate();
   ttclose();

   fclose (iterm);
   fclose (oterm);
   fclose (ofd);
   return (status);
}
