/************************************************************************
*
* disloader - Loads objects from asm990 for disassembly.
*
* Changes:
*   05/05/05   DGP   Original. Hacked from lnkloader.c
*   12/26/05   DGP   Correct COMMON field offset.
*   06/27/06   DGP   Added SPI dump load format.
*   01/16/07   DGP   Added DSEG support.
*   12/03/07   DGP   Added COMMON EXTERN/GLOBAL support and rework.
*   02/09/11   DGP   Added CSEG support.
*   08/17/11   DGP   Allow multiple CSEGs.
*   08/27/13   DGP   Fixed BACK ORG bug.
*   09/15/14   DGP   Fixed Jump target with correct mask.
*
************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>
#include <errno.h>

#include "utildef.h"

extern FILE *lstfd;
extern int pgmlen;
extern int dseglen;
extern int entry;
extern int modcount;
extern int symbolcount;
extern int dumpfile;
extern int maxcsegs;
extern int cseglen[MAXCSEGS];

extern char idtbuf[IDTSIZE+2];
extern char dsegname[IDTSIZE+2];
extern char csegname[MAXCSEGS][IDTSIZE+2];

extern uint8 memory[MEMSIZE];
extern Memory memctl[MEMSIZE];
extern uint8 dsegmem[MEMSIZE];
extern Memory dsegctl[MEMSIZE];
extern uint8 csegmem[MAXCSEGS][MEMSIZE];
extern Memory csegctl[MAXCSEGS][MEMSIZE];
extern Module modules[MAXMODULES];
extern SymNode *symbols[MAXSYMBOLS];

static int order;
static int csegndx;

/************************************************************************
* getnumfield - Gets the next numeric field.
************************************************************************/

static uint16
getnumfield (char *op, int binarymode)
{
   uint16 wdata;
   char item[80];

   if (binarymode)
   {
      wdata = ((*op & 0xFF) << 8) | (*(op+1) & 0xFF);
   }
   else
   {
      int wd;

      strncpy (item, op, 4);
      item[4] = '\0';
      sscanf (item, "%4X", &wd);
      wdata = wd & 0xFFFF;
   }
   return (wdata);
}

/************************************************************************
* getctl - Gets the Memory control address.
************************************************************************/

static Memory *
getctl (int mask, int address)
{
   if (mask & DSEG)
   {
      return (&dsegctl[address]);
   }
   else if (mask & CSEG)
   {
      return (&csegctl[csegndx][address]);
   }
   return (&memctl[address]);
}

/************************************************************************
* getmem - Gets the contents of Memory.
************************************************************************/

static int
getmem (int mask, int address)
{
   if (mask & DSEG)
   {
      return (GETDSEGMEM (address));
   }
   else if (mask & CSEG)
   {
      return (GETCSEGMEM (csegndx, address));
   }
   return (GETMEM (address));
}

/************************************************************************
* putmem - Puts the value into Memory.
************************************************************************/

static void
putmem (int mask, int address, int val)
{
   if (mask & DSEG)
   {
      PUTDSEGMEM (address, val);
   }
   else if (mask & CSEG)
   {
      PUTCSEGMEM (csegndx, address, val);
   }
   else
   {
      PUTMEM (address, val);
   }
}

/************************************************************************
* dumploader - Load dump file output (ie. SPI).
************************************************************************/

static int
dumploader (FILE *fd, char *file)
{
   char *bp, *fp;
   int done, first;
   int i;
   int insame = FALSE;
   uint16 data = 0;
   uint16 curraddr;
   uint16 oldaddr = 0xFFFF;
   char inbuf[82];

#ifdef DEBUGLOADER
   fprintf (stderr, "dumploader: file = %s\n",
	    file);
#endif

   curraddr = 0;
   done = FALSE;
   first = TRUE;

   /*
   ** Use the file name to hack the IDT and module info
   */

   strcpy (inbuf, file);
   if ((bp = strrchr (inbuf, '.')) != NULL) *bp = '\0';
   fp = inbuf;
   if ((bp = strrchr (inbuf, '/')) != NULL) fp = bp+1;
   bp = idtbuf;
   for (i = 0; *fp && i < IDTSIZE; i++)
   {
      *bp++ = islower(*fp) ? toupper(*fp) : *fp;
      fp++;
   }
   *bp = '\0';

   strcpy (modules[modcount].creator, "SPI");

   while (!done)
   {
      if (fgets (inbuf, sizeof(inbuf), fd) == NULL)
      {
	 done = TRUE;
	 if (first)
	    return (EOF);
	 break;
      }
#ifdef DEBUGLOADER
      fprintf (stderr, "%s", inbuf);
#endif

      first = FALSE;
      if (strlen(inbuf) >= 4)
      {
	 bp = inbuf;
	 if (!strncmp (bp, "SAME", 4))
	 {
	    insame = TRUE;
	 }
	 else
	 {
	    curraddr = getnumfield (bp, FALSE);
#ifdef DEBUGLOADER
	    fprintf (stderr,
		     "   curraddr = %04X, oldaddr = %04X, insame = %d\n",
		     curraddr, oldaddr, insame);
#endif
	    if (insame || (oldaddr < curraddr))
	    {
	       insame = FALSE;
	       while (oldaddr < curraddr)
	       {
		  PUTMEM (oldaddr, data);
		  memctl[oldaddr].tag = ABSDATA_TAG;
		  memctl[oldaddr].relocatable = FALSE;
		  oldaddr += 2;
	       }
	    }
	    bp += 7;
	    for (i = 0; i < 8; i++)
	    {
	       if (isspace(*bp))
		  data = 0;
	       else
		  data = getnumfield (bp, FALSE);
	       bp += 5;
#ifdef DEBUGLOADER
	       fprintf (stderr, "   memory[%04X] = %04X\n",
			curraddr, data);
#endif
	       PUTMEM (curraddr, data);
	       memctl[curraddr].tag = ABSDATA_TAG;
	       memctl[curraddr].relocatable = FALSE;
	       curraddr += 2;
	    }
	    oldaddr = curraddr;
	 }
      }
   }
   pgmlen = curraddr;
   return (0);
}

/************************************************************************
* processdef - Process DEF.
************************************************************************/

static void
processdef (char *item, char *module, int relo, int flags, uint16 wdata)
{
   SymNode *s;
   char *bp;

   bp = item;
   for (; *bp; bp++) if (isspace(*bp)) *bp = '\0';
#ifdef DEBUGLOADER
   fprintf (stderr, ", %c%s = %s",
	 relo ? 'R' : 'A',
	 (flags & LONGSYM) ? "LDEF" :
	 (flags & CSEG) ? "CMNDEF" : 
	 (flags & DSEG) ? "DSEGDEF" : "DEF",
	 item);
#endif
   if ((s = symlookup (item, module, FALSE)) == NULL)
   {
      s = symlookup (item, module, TRUE);
      s->flags = relo ? RELOCATABLE : 0;
      s->flags |= GLOBAL | flags;
      s->value = wdata;
      s->cmnndx = csegndx;
#ifdef DEBUGLOADER1
      fprintf (stderr, "\n      flags = >%04X, value = >%04X",
	    s->flags, s->value);
#endif
   }
}

/************************************************************************
* processref - Process REF.
************************************************************************/

static void
processref (char *item, char *module, int relo, int flags,
	    char otag, uint16 wdata)
{
   SymNode *s;
   char *bp;

   bp = item;
   for (; *bp; bp++) if (isspace(*bp)) *bp = '\0';
#ifdef DEBUGLOADER
   fprintf (stderr, ", %c%s = %s",
	 relo ? 'R' : 'A',
	 (flags & LONGSYM) ? "LREF" :
	 (flags & CSEG) ? "CMNREF" :
	 (flags & DSEG) ? "DSEGREF" : "REF",
	 item);
#endif

   if ((s = symlookup (item, module, FALSE)) == NULL)
   {
      s = symlookup (item, module, TRUE);
      s->flags = relo ? RELOCATABLE : 0;
      s->flags |= EXTERNAL | UNDEF | flags;
      s->value = wdata;
      s->order = order++;
      s->cmnndx = csegndx;
#ifdef DEBUGLOADER
      fprintf (stderr, ", order = %d, cmnndx = %d", s->order, s->cmnndx);
#endif
   }
}

/************************************************************************
* processrefchain - Process REF chain.
************************************************************************/

static void
processrefchain (int mask)
{
   int i;
   int segmask;

   segmask = mask & (CSEG|DSEG);
#ifdef DEBUGLOADER
   fprintf (stderr, "processrefchain: mode = %s, mask = >%04X(%04X)\n",
	    (mask & DSEG) ? "DSEG" : (mask & CSEG) ? "CSEG" : "P/R", mask, segmask);
#endif

   for (i = 0; i < symbolcount; i++)
   {
      SymNode *s;

      s = symbols[i];
      if (((s->flags & mask) == mask) &&
	  (mask ? TRUE : !(s->flags & mask)))
      {
	 Memory *ctl;
	 int refaddr;

	 refaddr = s->value;
	 ctl = getctl (mask, refaddr);
	 if (!segmask && (ctl->tag < ABSDATA_TAG || ctl->tag > RELDATA_TAG))
	    continue;
	 else if (segmask & DSEG && ctl->tag != DSEGDATA_TAG)
	    continue;
	 else if (segmask & CSEG && ctl->tag != CMNDATA_TAG)
	    continue;

         if ((s->flags & SREF) && (refaddr == 0))
	    continue;
#ifdef DEBUGLOADER
         fprintf (stderr, "   sym = %-6.6s, flags = %04X, refaddr = >%04X\n",
		  s->symbol, s->flags, refaddr);
#endif
	 if (mask && (refaddr == 0))
	 {
	    ctl->external = TRUE;
	    ctl->relocatable = TRUE;
	    ctl->tag = RELDATA_TAG;
	 }
	 while (refaddr)
	 {
	    int k;

	    ctl = getctl (mask, refaddr);
	    k = getmem (mask, refaddr);
	    if (mask || (k == 0))
	    {
	       ctl->tag = RELDATA_TAG;
	    }
	    ctl->external = TRUE;
	    ctl->relocatable = TRUE;
	    putmem (mask, refaddr, s->value);
	    refaddr = k;
#ifdef DEBUGLOADER
	    fprintf (stderr,
		  "                 refaddr = >%04X, tag = %c, val = >%04X\n",
		     refaddr, ctl->tag, s->value);
#endif
	 }
      }
   }
}

/************************************************************************
* processlabels - Process and add labels for recognizable targets.
************************************************************************/

static int
processlabels (int mask, int length, int labndx, char *module)
{
   int pc;
   int segflag;
   int segmask;

#ifdef DEBUGLOADER
   fprintf (stderr, "processlabels: mode =  %s, mask = >%04X, length = >%04X",
	    (mask & DSEG) ? "DSEG" : (mask & CSEG) ? "CSEG" : "P/R", mask,
	    length);
   fprintf (stderr, ", labndx = >%04X, module = %s\n",
            labndx, module);
#endif

   segmask = mask & (CSEG|DSEG);
   for (pc = 0; pc < length; )
   {
      SymNode *s;
      Memory *ctl;
      uint16 wdata;
      char otag;

      ctl = getctl (segmask, pc);
      otag = ctl->tag;
#ifdef DEBUGLOADER
      fprintf (stderr, "   pc = %04X, otag = '%c'\n", pc, otag);
#endif
      if (otag == RELDATA_TAG || otag == DSEGDATA_TAG || otag == CMNDATA_TAG)
      {
	 if (!ctl->external && !ctl->global)
	 {
	    wdata = getmem (segmask, pc);
	    segflag = 0;
	    if (otag == DSEGDATA_TAG)
	       segflag = dseglen ? DSEG : CSEG;
	    else if (otag == CMNDATA_TAG)
	       segflag = CSEG;

	    if ((s = vallookup (wdata, FALSE, segflag, csegndx)) == NULL)
	    {
#ifdef DEBUGLOADER
		  fprintf (stderr, "      rlabel = %3.3s%03d, pc = >%04X",
			   module, labndx+1, pc);
#endif
		  if ((s = symadd (module, ++labndx)) != NULL)
		  {
		     s->value = getmem (segmask, pc);
#ifdef DEBUGLOADER
		     fprintf (stderr, ", value = >%04X", s->value);
#endif
		     if (otag == DSEGDATA_TAG)
		        s->flags |= segflag;
		     if (otag == CMNDATA_TAG)
		     {
		        s->flags |= segflag;
			s->cmnndx = csegndx;
		     }
		  }
#ifdef DEBUGLOADER
		  fprintf (stderr, "\n");
#endif
	    }
	 }
         pc += 2;
      }
      else if (otag == ABSDATA_TAG)
      {
	 if ((s = vallookup (pc, FALSE, segmask, csegndx)) != NULL)
	 {
	    if ((s->flags & segmask) == segmask)
	    {
#ifdef DEBUGLOADER
	       fprintf (stderr, "   target = %-6.6s, pc = >%04X\n",
			s->symbol, pc);
#endif
	    }
	 }
	 wdata = getmem (segmask, pc);
	 if (!segmask && ((wdata & 0xF000) == 0x1000))
	 {
	    if ((wdata & 0x0F00) < 0x0D00)
	    {
	       int16 wpc;
	       int8 disp;

	       disp = wdata & 0x00FF;
	       if ((wdata != 0x1000) && ((uint8)disp != 0xFF))
	       {
		  wpc = ((pc+2) >> 1) & 0x7FFF;

		  wpc += disp;
		  wpc = (wpc << 1) & 0xFFFE;
		  if ((wpc < length) &&
		      ((s = vallookup (wpc & 0xFFFF, FALSE,
		                       segmask, csegndx)) == NULL))
		  {
#ifdef DEBUGLOADER
		     fprintf (stderr, "      jlabel = %3.3s%03d, pc = >%04X",
			      module, labndx+1, pc);
#endif
		     if ((s = symadd (module, ++labndx)) != NULL)
		     {
#ifdef DEBUGLOADER
			fprintf (stderr, ", value = >%04X", wpc & 0xFFFF);
#endif
			s->value = wpc & 0xFFFF;
			s->flags |= segmask;
		     }
#ifdef DEBUGLOADER
		     fprintf (stderr, "\n");
#endif
		  }
	       }
	    }
	 }
         pc += 2;
      }
      else if (otag == RELORG_TAG)
      {
         pc = getmem (segmask, pc); 
      }
      else
      {
	 pc += 2;
      }
   }
   return (labndx);
}

/************************************************************************
* disloader - Object loader for dis990.
************************************************************************/

int
disloader (FILE *fd, int loadpt, char *file)
{
   Memory *ctl;
   int i;
   int done;
   int status = 0;
   int binarymode = FALSE;
   int reclen;
   int curraddr;
   int origin;
   int relo;
   int refaddr;
   int segmask;
   int seglen;
   int dsegpc;
   int dsegorg;
   int csegpc;
   int csegorg;
   int wordlen = WORDTAGLEN - 1;
   uint16 wdata;
   uint16 odata;
   char inbuf[82];
   char module[MAXSYMLEN+2];

#ifdef DEBUGLOADER
   fprintf (stderr, "disloader: loadpt = %04X, file = %s\n",
	    loadpt, file);
#endif

   if (dumpfile)
   {
      return (dumploader (fd, file));
   }

   segmask = 0;
   order = 0;
   pgmlen = 0;
   dsegorg = dseglen = dsegpc = 0;
   csegorg = csegpc = 0;
   origin = curraddr = loadpt;

   i = fgetc (fd);
   if (i == BINIDT_TAG)
   {
      binarymode = TRUE;
   }
   ungetc (i, fd);

   done = FALSE;
   while (!done)
   {
      char *op = inbuf;
      int i;

      if (binarymode)
      {
	 reclen = fread (inbuf, 1, 81, fd);
	 if (feof(fd))
	 {
	    status = EOF;
	    done = TRUE;
	    break;
	 }
      }
      else
      {
	 if (fgets (inbuf, 82, fd) == NULL)
	 {
	    status = EOF;
	    done = TRUE;
	    break;
	 }
	 reclen = strlen (inbuf);
      }

#ifdef DEBUGLOADER
      fprintf (stderr, "New record, reclen = %d\n", reclen);
#endif
      if (*op == EOFSYM)
      {
	 if (segmask == CSEG)
	    cseglen[csegndx] = csegpc;
	 if (reclen > 60)
	 {
	    strncpy (modules[modcount].date, &inbuf[TIMEOFFSET], 8);
	    modules[modcount].date[9] = '\0';
	    strncpy (modules[modcount].time, &inbuf[DATEOFFSET], 8);
	    modules[modcount].time[9] = '\0';
	    strncpy (modules[modcount].creator, &inbuf[CREATOROFFSET], 8);
	    modules[modcount].creator[9] = '\0';
	 }
	 modcount++;
	 loadpt = curraddr;
	 done = TRUE;
	 continue;
      }

      for (i = 0; i < reclen; i++)
      {
	 char *bp;
	 char otag;
	 char item[80];

	 relo = FALSE;
	 otag = *op++;
	 if (otag == EOR_TAG)
	    wdata = 0;
	 else
	    wdata = getnumfield (op, binarymode);

#ifdef DEBUGLOADER
	 fprintf (stderr, "   otag = %c, loadpt = >%04X, curraddr = >%04X",
		  isprint(otag) ? otag : '0', loadpt, curraddr);
	 fprintf (stderr, ", indseg = %s, dsegpc = >%04X, data = >%04X\n",
		  segmask & DSEG ? "TRUE" : "FALSE", dsegpc, wdata & 0xFFFF);
	 fprintf (stderr, "      incseg = %s, csegpc = >%04X, data = >%04X",
		  segmask & CSEG ? "TRUE" : "FALSE", csegpc, wdata & 0xFFFF);
#endif

	 refaddr = 0;

         switch (otag)
	 {
	 case BINIDT_TAG: /* Binary IDT */
	    binarymode = TRUE;
#ifdef DEBUGLOADER
	    fprintf (stderr, ", Binary mode");
#endif
	    wordlen = BINWORDTAGLEN - 1;
	 case IDT_TAG:
	    op += wordlen;
	    strncpy (item, op, IDTSIZE);
	    item[IDTSIZE] = '\0';
#ifdef DEBUGLOADER
	    fprintf (stderr, ", IDT = %s", item);
#endif
	    bp = item;
	    for (; *bp; bp++) if (isspace(*bp)) *bp = '\0';
	    strcpy (module, item);
	    strcpy (modules[modcount].objfile, file);
	    strcpy (modules[modcount].name, item);
	    modules[modcount].length = wdata;
	    modules[modcount].origin = loadpt;
	    modules[modcount].creator[0] = '\0';
	    modules[modcount].date[0] = '\0';
	    modules[modcount].time[0] = '\0';
	    if (idtbuf[0] == '\0')
	    {
	       strcpy (idtbuf, item);
	    }
	    op += IDTSIZE;
	    break;

         case RELORG_TAG:
	    wdata += loadpt;
         case ABSORG_TAG:
	    if (segmask == CSEG)
	    {
	       cseglen[csegndx] = csegpc;
	    }
	    segmask = 0;
	    if (wdata >= curraddr)
	    {
#ifdef DEBUGLOADER
	       fprintf (stderr, ", ORGTAG = >%04X", wdata);
#endif
	       PUTMEM (curraddr, wdata);
	       memctl[curraddr].tag = otag;
	    }
#ifdef DEBUGLOADER
	    else
	       fprintf (stderr, ", BACK ORGTAG = >%04X", wdata);
#endif
	    curraddr = wdata & 0xFFFF;
	    op += wordlen;
	    break;

	 case RELDATA_TAG:
	    wdata += loadpt;
	    relo = TRUE;
	 case ABSDATA_TAG:
	    op += wordlen;
	    if (segmask & DSEG)
	    {
#ifdef DEBUGLOADER
	       fprintf (stderr, ", DSEGDATA = >%04X", wdata);
#endif
	       ctl = getctl (segmask, dsegpc);
	       putmem (segmask, dsegpc, wdata);
	       dsegpc += 2;
	    }
	    else if (segmask & CSEG)
	    {
#ifdef DEBUGLOADER
	       fprintf (stderr, ", CSEGDATA = >%04X", wdata);
#endif
	       ctl = getctl (segmask, csegpc);
	       putmem (segmask, csegpc, wdata);
	       csegpc += 2;
	    }
	    else
	    {
	       ctl = getctl (0, curraddr);
	       putmem (0, curraddr, wdata);
	       curraddr += 2;
	    }
	    ctl->tag = otag;
	    ctl->relocatable = relo;
	    break;

	 case RELENTRY_TAG:
	    op += wordlen;
	    entry = (wdata + loadpt) & 0xFFFF;
	    break;

	 case ABSENTRY_TAG:
	    op += wordlen;
	    entry = wdata & 0xFFFF;
	    break;

	 case RELEXTRN_TAG:
	    wdata += loadpt;
	    relo = TRUE;
	 case ABSEXTRN_TAG:
	    refaddr = wdata;
	    op += wordlen;
	    strncpy (item, op, MAXSHORTSYMLEN);
	    item[MAXSHORTSYMLEN] = '\0';
	    op += MAXSHORTSYMLEN;
	    processref (item, module, relo, 0, otag, wdata);
	    break;

	 case RELSREF_TAG:
	    wdata += loadpt;
	    relo = TRUE;
	    refaddr = wdata;
	    op += wordlen;
	    strncpy (item, op, MAXSHORTSYMLEN);
	    item[MAXSHORTSYMLEN] = '\0';
	    op += MAXSHORTSYMLEN;
	    processref (item, module, relo, SREF, otag, wdata);
	    break;

	 case RELGLOBAL_TAG:
	    wdata += loadpt;
	    relo = TRUE;
	 case ABSGLOBAL_TAG:
	    op += wordlen;
	    strncpy (item, op, MAXSHORTSYMLEN);
	    item[MAXSHORTSYMLEN] = '\0';
	    op += MAXSHORTSYMLEN;
	    processdef (item, module, relo, 0, wdata);
	    break;

	 case LRELEXTRN_TAG:
	    wdata += loadpt;
	    relo = TRUE;
	 case LABSEXTRN_TAG:
	    refaddr = wdata;
	    op += wordlen;
	    strncpy (item, op, MAXSYMLEN);
	    item[MAXSYMLEN] = '\0';
	    op += MAXSYMLEN;
	    processref (item, module, relo, LONGSYM, ABSEXTRN_TAG, wdata);
	    break;

	 case LRELGLOBAL_TAG:
	    wdata += loadpt;
	    relo = TRUE;
	 case LABSGLOBAL_TAG:
	    op += wordlen;
	    strncpy (item, op, MAXSYMLEN);
	    item[MAXSYMLEN] = '\0';
	    op += MAXSYMLEN;
	    processdef (item, module, relo, LONGSYM, wdata);
	    break;

	 case COMMON_TAG:
	    op += wordlen;
	    strncpy (item, op, MAXSHORTSYMLEN);
	    item[MAXSHORTSYMLEN] = '\0';
	    seglen = wdata;
#ifdef DEBUGLOADER
	    fprintf (stderr, ", CMNNAME = %s", item);
#endif
	    op += MAXSHORTSYMLEN;
	    wdata = getnumfield (op, binarymode);
	    op += wordlen;
#ifdef DEBUGLOADER
	    fprintf (stderr, ", FLD = >%04X", wdata);
#endif
	    if (wdata)
	    {
	       strcpy (csegname[maxcsegs], item);
	       csegndx = maxcsegs;
	       for (csegpc = 0; csegpc < seglen; csegpc += 2)
	       {
		  PUTCSEGMEM (csegndx, csegpc, 0);
		  csegctl[csegndx][csegpc].tag = ABSDATA_TAG;
	       }
	       maxcsegs++;
	       if (maxcsegs >= MAXCSEGS)
	       {
	          fprintf (stderr, "dis990: MAX CSEGs exceeded.\n");
		  exit (ABORT);
	       }
	    }
	    else
	    {
	       strcpy (dsegname, item);
	       for (dsegpc = 0; dsegpc < seglen; dsegpc += 2)
	       {
		  PUTDSEGMEM (dsegpc, 0);
		  dsegctl[dsegpc].tag = ABSDATA_TAG;
	       }
	    }
	    break;

	 case CMNORG_TAG:
	    op += wordlen;
#ifdef DEBUGLOADER
	    fprintf (stderr, ", CSEGORG = >%04X", wdata);
#endif
	    odata = getnumfield (op, binarymode) - 1;
	    op += wordlen;
#ifdef DEBUGLOADER
	    fprintf (stderr, ", CMN = %d", odata+1);
#endif
	    if (segmask == CSEG)
	    {
	       if (odata != csegndx)
	       {
		  cseglen[csegndx] = csegpc;
		  csegpc = 0;
	       }
	    }
	    else
	    {
	       csegpc = 0;
	    }
	    csegndx = odata;
	    PUTCSEGMEM (csegndx, csegpc, csegorg+wdata);
	    csegctl[csegndx][csegpc].tag = RELORG_TAG;
	    csegpc = csegorg + wdata;
	    segmask = CSEG;
	    break;

	 case CMNEXTRN_TAG:
	    refaddr = wdata;
	    op += wordlen;
	    strncpy (item, op, MAXSHORTSYMLEN);
	    item[MAXSYMLEN] = '\0';
	    op += MAXSHORTSYMLEN;
	    wdata = getnumfield (op, binarymode) - 1;
	    op += wordlen;
#ifdef DEBUGLOADER
	    fprintf (stderr, ", CMN = %d", wdata+1);
#endif
	    csegndx = wdata;
	    processref (item, module, TRUE, CSEG, otag, wdata+csegorg);
	    break;

	 case CMNGLOBAL_TAG:
	    op += wordlen;
	    strncpy (item, op, MAXSHORTSYMLEN);
	    item[MAXSYMLEN] = '\0';
	    op += MAXSHORTSYMLEN;
	    wdata = getnumfield (op, binarymode);
	    op += wordlen;
#ifdef DEBUGLOADER
	    fprintf (stderr, ", CMN = %d", wdata+1);
#endif
	    csegndx = wdata;
	    processdef (item, module, TRUE, CSEG, wdata+csegorg);
	    break;

         case CMNDATA_TAG:
            op += wordlen;
            odata = getnumfield (op, binarymode);
            op += wordlen;
#ifdef DEBUGLOADER
	    fprintf (stderr, ", CSEGREF = %04X, NDX = %04X", wdata, odata);
#endif
	    if (segmask == CSEG)
	    {
	       PUTCSEGMEM (odata-1, csegpc, wdata);
	       csegctl[odata-1][csegpc].tag = otag;
	       csegpc += 2;
	    }
	    else
	    {
	       PUTMEM (curraddr, wdata);
	       memctl[curraddr].tag = otag;
	       curraddr += 2;
	    }
            break;

	 case EXTNDX_TAG:
	    op += wordlen;
	    PUTMEM (curraddr, wdata);
	    memctl[curraddr].tag = otag;
	    wdata = getnumfield (op, binarymode);
	    op += 2;
	    memctl[curraddr].value = wdata;
	    curraddr += 2;
#ifdef DEBUGLOADER
	    fprintf (stderr, ", FLD = >%04X", wdata);
#endif
	    break;
	    
	 case DSEGORG_TAG:
	    if (segmask == CSEG)
	    {
	       cseglen[csegndx] = csegpc;
	    }
	    op += wordlen;
	    PUTDSEGMEM (dsegpc, dsegorg+wdata);
	    dsegctl[dsegpc].tag = RELORG_TAG;
	    dsegpc = dsegorg + wdata;
	    segmask = DSEG;
#ifdef DEBUGLOADER
	    fprintf (stderr, ", DSEGORG = >%04X", wdata);
#endif
	    break;

	 case DSEGDATA_TAG:
	    op += wordlen;
	    wdata += dsegorg;
	    if (segmask & DSEG)
	    {
#ifdef DEBUGLOADER
	       fprintf (stderr, ", DSEGDATA = >%04X", wdata);
#endif
	       ctl = getctl (segmask, dsegpc);
	       putmem (segmask, dsegpc, wdata);
	       dsegpc += 2;
	    }
	    else if (segmask & CSEG)
	    {
#ifdef DEBUGLOADER
	       fprintf (stderr, ", CSEGDATA = >%04X", wdata);
#endif
	       ctl = getctl (segmask, csegpc);
	       putmem (segmask, csegpc, wdata);
	       csegpc += 2;
	    }
	    else
	    {
	       ctl = getctl (0, curraddr);
	       putmem (0, curraddr, wdata);
	       curraddr += 2;
	    }
	    ctl->tag = otag;
	    ctl->relocatable = TRUE;
	    break;

	 case LOAD_TAG:
	    relo = TRUE;
	    op += wordlen;
	    strncpy (item, (char *)op, MAXSHORTSYMLEN);
	    item[MAXSYMLEN] = '\0';
	    op += MAXSHORTSYMLEN;
#ifdef DEBUGLOADER
            fprintf (stderr, ", LOAD = %s, NDX = >%04X", item, wdata);
#endif
	    processref (item, module, relo, LOAD, otag, wdata);
	    break;

	 case EOR_TAG:
	    i = 81;
	    break;

	 case PGMIDT_TAG:
	    op += IDTSIZE;
	 case NOCKSUM_TAG:
	 case CKSUM_TAG:
	    op += wordlen;
	    break; 

	 default:
	    fprintf (stderr, "dis990: UNSUPPORTED TAG, tag = %c, %s\n",
	    	     otag, &inbuf[72]);
	    op += wordlen;
	 }
#ifdef DEBUGLOADER
	 fprintf (stderr, "\n");
#endif
         if (curraddr > pgmlen)
	    pgmlen = curraddr;
      }
   }

   if (status == 0)
   {
      dseglen = dsegpc;

      /*
      ** Go through external chain and set value for symbol.
      */

      processrefchain (RELOCATABLE | EXTERNAL);
      processrefchain (RELOCATABLE | EXTERNAL | DSEG);
      for (csegndx = 0; csegndx < maxcsegs; csegndx++)
	 processrefchain (RELOCATABLE | EXTERNAL | CSEG);

      /*
      ** Go through the code and generate relocatable symbols.
      */

      strcpy (module, idtbuf);
      refaddr = processlabels (GLOBAL, curraddr, 0, module);
      refaddr = processlabels (GLOBAL | DSEG, dsegpc, refaddr, module);
      for (csegndx = 0; csegndx < maxcsegs; csegndx++)
	 refaddr = processlabels (GLOBAL | CSEG, cseglen[csegndx],
				  refaddr, csegname[csegndx]);
      if (entry >= 0)
      {
	 SymNode *s;

	 if ((s = vallookup (entry, FALSE, 0, 0)) == NULL)
	 {
#ifdef DEBUGLOADER
	    fprintf (stderr, "ELABEL = %3.3s%03d, pc = >%04X",
		     module, refaddr+1, curraddr);
#endif
	    if ((s = symadd (module, ++refaddr)) != NULL)
	    {
#ifdef DEBUGLOADER
	       fprintf (stderr, ", value = >%04X", entry);
#endif
	       s->value = entry;
	    }
#ifdef DEBUGLOADER
	    fprintf (stderr, "\n");
#endif
         }
      }

#ifdef DEBUGLOADER
      fprintf (stderr, "\nSymbols:\n");
      for (i = 0; i < symbolcount; i++)
      {
	 SymNode *s;

	 s = symbols[i];
	 fprintf (stderr,
		  " symbol[%d]: '%-8.8s' '%-8.8s' flags = >%04X, value = >%04X",
		  i, s->symbol, s->module, s->flags, s->value);
	 fprintf (stderr,
		  ", order = %d, cmnndx = %d\n",
		  s->order, s->cmnndx);
      }
#endif
   }

   return (status);

}
