/***********************************************************************
*
* dissupt.c - Support routines for the TI 990 disassembler.
*
* Changes:
*   05/05/05   DGP   Original. Hacked from lnksupt.c
*   12/03/07   DGP   Changed to use flags.
*   08/17/11   DGP   Allow multiple CSEGs.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <time.h>

#include "utildef.h"

extern int pcreg;
extern int symbolcount;
extern int absolute;
extern int pgmlength;

extern char idtbuf[IDTSIZE+2];

extern SymNode *symbols[MAXSYMBOLS];
extern uint8 memory[MEMSIZE];
extern Memory memctl[MEMSIZE];


/***********************************************************************
* extlookup - Lookup an external in the symbol table.
***********************************************************************/

SymNode *
extlookup (int val, int useorder)
{
   SymNode *ret = NULL;
   SymNode *s;
   int i;
   int cond;

   for (i = 0; i < symbolcount; i++)
   {
      s = symbols[i];
      if (useorder)
	 cond = (s->flags & EXTERNAL) && (s->order == val);
      else
	 cond = (s->flags & EXTERNAL) && (s->value == val);

      if (cond)
      {
         ret = s;
	 break;
      }
   }

   return (ret);
}

/***********************************************************************
* vallookup - Lookup a value in the symbol table.
***********************************************************************/

SymNode *
vallookup (int val, int allowext, int mask, int segndx)
{
   SymNode *ret = NULL;
   SymNode *s;
   int cond;
   int i;

   for (i = 0; i < symbolcount; i++)
   {
      s = symbols[i];
      cond = (mask ? ((s->flags & mask) == mask) : !(s->flags & (CSEG|DSEG))) &&
	     (allowext ? TRUE : !(s->flags & EXTERNAL)) &&
	     (s->flags & RELOCATABLE) && (s->value == val) && 
	     (s->cmnndx == segndx);

      if (cond)
      {
         ret = s;
	 break;
      }
   }

   return (ret);
}

/***********************************************************************
* symlookup - Lookup a symbol in the symbol table.
***********************************************************************/

SymNode *
symlookup (char *sym, char *module, int add)
{
   SymNode *ret = NULL;
   int done = FALSE;
   int mid;
   int last = 0;
   int lo;
   int up;
   int r;

#ifdef DEBUGSYM
   fprintf (stderr, "symlookup: Entered: sym = %s, module = %s, add = %s\n",
   	    sym, module, add ? "TRUE" : "FALSE");
#endif

   /*
   ** Empty symbol table.
   */

   if (symbolcount == 0)
   {
      if (!add) return (NULL);

#ifdef DEBUGSYM
      fprintf (stderr, "add symbol at top\n");
#endif
      if ((symbols[symbolcount] = (SymNode *)malloc (sizeof (SymNode))) == NULL)
      {
         fprintf (stderr, "dis990: Unable to allocate memory\n");
	 exit (ABORT);
      }
      strcpy (symbols[0]->symbol, sym);
      strcpy (symbols[0]->module, module);
      symbols[0]->value = pcreg;
      symbols[0]->flags = 0;
      symbols[0]->order = 0;
      if (!absolute)
	 symbols[0]->flags |= RELOCATABLE;
      symbolcount++;
      return (symbols[0]);
   }

   /*
   ** Locate using binary search
   */

   lo = 0;
   up = symbolcount;
   last = -1;
   
   while (!done)
   {
      mid = (up - lo) / 2 + lo;
#ifdef DEBUGSYM
      fprintf (stderr, " mid = %d, last = %d\n", mid, last);
#endif
      if (last == mid) break;
      r = strcmp (symbols[mid]->symbol, sym);
      if (r == 0)
      {
	 if (add) return (NULL); /* must be a duplicate */
         return (symbols[mid]);
      }
      else if (r < 0)
      {
         lo = mid;
      }
      else 
      {
         up = mid;
      }
      last = mid;
   }

   /*
   ** Not found, check to add
   */

   if (add)
   {
      SymNode *new;

#ifdef DEBUGSYM
      fprintf (stderr, "add new symbol\n");
#endif
      if (symbolcount+1 > MAXSYMBOLS)
      {
         fprintf (stderr, "dis990: Symbol table exceeded\n");
	 exit (ABORT);
      }

      if ((new = (SymNode *)malloc (sizeof (SymNode))) == NULL)
      {
         fprintf (stderr, "dis990: Unable to allocate memory\n");
	 exit (ABORT);
      }

      strcpy (new->symbol, sym);
      strcpy (new->module, module);
      new->value = pcreg;
      new->flags = 0;
      new->order = 0;
      if (!absolute)
	 new->flags |= RELOCATABLE;

      /*
      ** Insert pointer in sort order.
      */

      for (lo = 0; lo < symbolcount; lo++)
      {
         if (strcmp (symbols[lo]->symbol, sym) > 0)
	 {
	    for (up = symbolcount + 1; up > lo; up--)
	    {
	       symbols[up] = symbols[up-1];
	    }
	    symbols[lo] = new;
	    symbolcount++;
	    return (symbols[lo]);
	 }
      }
      symbols[symbolcount] = new;
      ret = symbols[symbolcount];
      symbolcount++;
   }
   return (ret);
}

/***********************************************************************
* symadd - Add a local symbol.
***********************************************************************/

SymNode *
symadd (char *name, int val)
{
   SymNode *ret = NULL;
   char lclname[MAXSYMLEN+2];

   strcpy (lclname, name);
   sprintf (&lclname[3], "%03d", val);
   ret = symlookup (lclname, idtbuf, TRUE);
#ifdef DEBUGSYM
   if (ret)
      fprintf (stderr, "symadd: name = %s(%s), val = %03d, flags = >%04X\n",
	       name, lclname, val, ret->flags);
#endif
   return (ret);
}

