/***********************************************************************
*
* fpydir - Directory listing for the TI 990 FD800 disks.
*
* Changes:
*   06/21/03   DGP   Original.
*   07/16/03   DGP   Sorted the dir entries.
*   12/15/08   DGP   Added UCSD Pascal support.
*	
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>

#include "utiltypes.h"
#include "fd800defs.h"
#include "ucsdtypes.h"
#include "tx990types.h"

#define FALSE 0
#define TRUE  1

static int longlist = FALSE;
static int pascallist = FALSE;
static int extentlist = FALSE;

typedef struct
{
   char filename[PASNAMLEN+2];
   int	blocks;
   int  start;
   int  last;
   int  type;
   int  date;
} pasdirent_t;

typedef struct
{
   char filename[TXFILELEN+TXEXTLEN+2];
   char fileprot;
   int  fcbaddr;
} stddirent_t;

#ifdef DEBUG
#include "hexdump.h"
#endif

/***********************************************************************
* getint - Convert integer to local endian.
***********************************************************************/

static uint16
getint (uint16 *val)
{
   uint8 *cp;
   uint16 cval;

   cp = (uint8 *)val;
   cval = (*cp << 8) | *(cp + 1);

#ifdef DEBUG
   printf ("getint: val = %d(%x), cval = %d(%x)\n", *val, *val, cval, cval);
#endif
   return (cval);
}

/***********************************************************************
* cvtblk - Convert Pascal block address into Track/sector
***********************************************************************/

static int
cvtblk (int blk, int *ntrk, int *nsec)
{
   int trk;
   int sec;

   trk = (blk * PASSECBLK) / FPYSECTRK;
   sec = ((blk * PASSECBLK) % FPYSECTRK) + 1;

   *ntrk = trk;
   *nsec = sec;
#ifdef DEBUG
   printf ("cvtblk: blk = %d, trk = %d, sec = %d\n", blk, trk, sec);
#endif
   return (0);
}

/***********************************************************************
* cvtau - Convert AU address into Track/sector
***********************************************************************/

static int
cvtau (int au, int aus, int *ntrk, int *nsec)
{
   int trk;
   int sec;

   if (au < FPYMAXAUS && aus < FPYSECAU)
   {
      trk = (au * FPYSECAU + aus) / FPYSECTRK;
      sec = (au * FPYSECAU + aus) % FPYSECTRK;
      sec = fpyskew[sec] + 1;
      *ntrk = trk;
      *nsec = sec;
#ifdef DEBUG
      printf ("trk = %d, sec = %d\n", trk, sec);
#endif
      return (0);
   }
   return (-1);
}

/***********************************************************************
* readdisk - Read the disk
***********************************************************************/

static int
readdisk (FILE *diskfd, int au, int aus, uint8 *buffer)
{
   int diskloc;
   int trk, sec;

#ifdef DEBUG
   printf ("readdisk: au = %d, aus = %d\n", au, aus);
#endif
   cvtau (au, aus, &trk, &sec);
   diskloc = ((FPYSECLEN * FPYSECTRK) * trk) + (FPYSECLEN * (sec-1));
   if (fseek (diskfd, diskloc, SEEK_SET) < 0)
   {
      perror("Disk seek error");
      return (-1);
   }

   if (FPYSECLEN != fread (buffer, 1, FPYSECLEN, diskfd))
   {
      perror ("Disk read error");
      return (-1);
   }
#ifdef DEBUG
   printf ("Buffer:\n");
   HEXDUMP (stdout, buffer, FPYSECLEN, 0);
#endif

   return (FPYSECLEN);

}

/***********************************************************************
* pasdirlist - Pascal floppy directory list
***********************************************************************/

static int
pasdirlist (FILE *diskfd)
{
   pasdirent_t *dirent[PASMAXDIR];
   pasdir_t *dp, *vp;
   char *bp;
   int i, j, k, l;
   int diskloc;
   int trk, sec;
   int len;
   int dirlen;
   int dircnt = 0;
   unsigned char pasdirblk[PASDIRLEN];
   unsigned char volname[PASVOLLEN+2];

   cvtblk (PASDIRBLK, &trk, &sec);
   diskloc = ((FPYSECLEN * FPYSECTRK) * trk) + (FPYSECLEN * (sec-1));
   if (fseek (diskfd, diskloc, SEEK_SET) < 0)
   {
      perror("Disk seek error");
      return (-1);
   }

   if (PASDIRLEN != fread (pasdirblk, 1, PASDIRLEN, diskfd))
   {
      perror ("Disk read error");
      return (-1);
   }
#ifdef DEBUG
   printf ("pasdirblk:\n");
   HEXDUMP (stdout, pasdirblk, PASDIRLEN, 0);
#endif

   vp = (pasdir_t *)pasdirblk;
   dirlen = getint(&(vp->filekind.volent.dnumfiles));
   if ((dirlen > PASMAXDIR) ||
       (getint(&(vp->dfkind)) != 0) || 
       (vp->filekind.volent.dvid[0] > PASVOLLEN))
   {
      fprintf (stderr, "Invalid disk format\n");
      return (-1);;
   }

#ifdef DEBUG
   printf ("dirent: pasdir_t %d\n", sizeof(pasdir_t));
   HEXDUMP (stdout, (char *)vp, sizeof(pasdir_t), 0);
   printf ("   dnumfiles = %d, deovblk = %d\n", 
	 getint(&(vp->filekind.volent.dnumfiles)),
	 getint(&(vp->filekind.volent.deovblk)));
#endif
   bp = vp->filekind.volent.dvid;
   len = *bp++;
   for (i = 0; i < len; i++)
   {
      volname[i] = *bp++;
   }
   volname[len] = '\0';
   printf ("%s:\n", volname);

   dp = vp + 1;
   for (j = 0; j < dirlen; j++)
   {
      /*
      ** Prcocess entry if length != 0
      */
#ifdef DEBUG
      printf ("dirent: pasdir_t %d\n", sizeof(pasdir_t));
      HEXDUMP (stdout, (char *)dp, sizeof(pasdir_t), 0);
#endif

      if (dp->filekind.filent.dtid[0])
      {
	 pasdirent_t *dir;
	 int start, end;

	 if ((dir = (pasdirent_t *)malloc (sizeof (pasdirent_t))) == NULL)
	 {
	    fprintf (stderr, "Unable to allocate memeory\n");
	    exit (1);
	 }

	 bp = dp->filekind.filent.dtid;
	 len = *bp++;
	 for (i = 0; i < len; i++)
	 {
	    dir->filename[i] = *bp++;
	 }
	 dir->filename[len] = '\0';

	 start = getint(&(dp->dfirstblk));
	 end = getint(&(dp->dlastblk));
	 dir->start = start;
	 dir->blocks = end - start;
	 dir->type = getint(&(dp->dfkind));
	 dir->last = getint(&(dp->filekind.filent.dlastbyte));
	 dir->date = getint(&(dp->filekind.filent.daccess));

	 if (!longlist)
	 {
	    /*
	    ** If not longlist, Insert pointer in alpha sort order.
	    */

	    if (dircnt == 0)
	    {
	       dirent[dircnt] = dir;
	       dircnt++;
	    }
	    else
	    { 
	       for (k = 0; k < dircnt; k++)
	       {
		  if (strcmp (dirent[k]->filename, dir->filename) > 0)
		  {
		     for (l = dircnt + 1; l > k; l--)
		     {
			dirent[l] = dirent[l-1];
		     }
		     dirent[k] = dir;
		     dircnt++;
		     break;
		  }
	       }
	       if (k == dircnt)
	       {
		  dirent[dircnt] = dir;
		  dircnt++;
	       }
	    }
	 }
	 else
	 {
	    dirent[dircnt] = dir;
	    dircnt++;
	 }
      }
      dp++;
   }

   len = 6;
   for (j = 0; j < dircnt; j++)
   {
      pasdirent_t *dir;

      dir = dirent[j];
      printf ("%-16.16s %4d  %2d-%s-%02d",
	      dir->filename, dir->blocks,
	      (dir->date & 0x01F0) >> 4,
	      pasmonths[dir->date & 0x000F],
	      (dir->date & 0xF700) >> 9);
      if (longlist)
      {
         printf (" %5d %5d  %s",
		 dir->start, dir->last,
		 pasfiletypes[dir->type]);
      }
      printf ("\n");
      len += dir->blocks;
      free (dir);
      dirent[j] = NULL;
   }
   printf ("%d files, %d blocks used\n", dircnt, len);

   return (0);
}

/***********************************************************************
* stddirlist - Standard floppy directory list
***********************************************************************/

static void
stddirlist (FILE *diskfd)
{
   stddirent_t *dirent[FPYSECAU*TXDIRPSEC];
   int dircnt = 0;
   int i, j, k, l;
   disklabel_t disklabel;
   txdir_t dirblock[TXDIRPSEC];
   txfcb_t fcbblock;

   /*
   ** Read the disk label.
   */

   readdisk (diskfd, TXLABELAU, 0, (uint8 *)&disklabel);
   for (i = 0; i < 32; i++)
      if (disklabel.label[i] < ' ' || disklabel.label[i] > 'z')
      {
         fprintf (stderr, "Invalid disk format\n");
	 return;
      }

   if (longlist)
      printf ("DISK ID: %-32.32s     VOLNAME: %-4.4s\n\n",
	       disklabel.label, disklabel.volname);

   /*
   ** Read the directory, the size of the directory is 1 AU.
   */

   for (i = 0; i < FPYSECAU; i++)
   {
      /*
      ** Read a directory sector.
      */

      readdisk (diskfd, TXDIRAU, i, (uint8 *)&dirblock);
      for (j = 0; j < TXDIRPSEC; j++)
      {
	 /*
	 ** A ".." in the filename is an empty slot.
	 */

	 if (strncmp (dirblock[j].filename, "..", 2))
	 {
	    stddirent_t *dir;
	    char file[TXFILELEN+1];
	    char ext[TXEXTLEN+1];

	    /*
	    ** Get filename component
	    */

	    strncpy (file, dirblock[j].filename, TXFILELEN);
	    file[TXFILELEN] = '\0';
	    for (l = 0; l < TXFILELEN; l++) if (file[l] == ' ') file[l] = '\0';

	    /*
	    ** Get extension component
	    */

	    strncpy (ext, dirblock[j].fileext, TXEXTLEN);
	    ext[TXEXTLEN] = '\0';
	    for (l = 0; l < TXEXTLEN; l++) if (ext[l] == ' ') ext[l] = '\0';

	    if ((dir = (stddirent_t *)malloc (sizeof (stddirent_t))) == NULL)
	    {
	       fprintf (stderr, "Unable to allocate memeory\n");
	       exit (1);
	    }

	    sprintf (dir->filename, "%s.%s", file, ext);

	    /*
	    ** Get the address of the FCB
	    */

	    dir->fcbaddr = getint (&(dirblock[j].filefcb));

	    /*
	    ** Get the file protection
	    */

	    dir->fileprot = dirblock[j].fileprot;

	    /*
	    ** Insert pointer in alpha sort order.
	    */

	    if (dircnt == 0)
	    {
	       dirent[dircnt] = dir;
	       dircnt++;
	    }
	    else
	    { 
	       for (k = 0; k < dircnt; k++)
	       {
		  if (strcmp (dirent[k]->filename, dir->filename) > 0)
		  {
		     for (l = dircnt + 1; l > k; l--)
		     {
			dirent[l] = dirent[l-1];
		     }
		     dirent[k] = dir;
		     dircnt++;
		     break;
		  }
	       }
	       if (k == dircnt)
	       {
		  dirent[dircnt] = dir;
		  dircnt++;
	       }
	    }
         }
      }
   }

   /*
   ** Print the entries
   */

   k = 0;
   for (j = 0; j < dircnt; j++)
   {
      stddirent_t *dir;

      dir = dirent[j];

      printf ("%-11.11s  ", dir->filename);
      if (longlist)
      {
	 int fcbext;
	 int totalaus;
	 int start, numaus;

	 /*
	 ** Print the file protection
	 */

	 printf ("  %c  ", dir->fileprot);

#ifdef DEBUG
	 printf (" read fcb: fcbaddr = %d\n", dir->fcbaddr);
#endif
	 /*
	 ** Read the FCB
	 */

	 readdisk (diskfd, dir->fcbaddr, 0, (uint8 *)&fcbblock);

	 /*
	 ** Go through extents to get total size of file.
	 */

	 fcbext = 0;
	 totalaus = 0;
	 while (TRUE)
	 { 
	    start = getint (&(fcbblock.extents[fcbext].startau));
	    if (start == 0xFFFF) break;
	    numaus = getint (&(fcbblock.extents[fcbext].numaus));
	    fcbext++;
	    totalaus += numaus;
	 }
	 printf (" %3d  ", totalaus);

	 /*
	 ** If extentlist, print extents, AU:SIZE pairs.
	 */

	 if (extentlist)
	 {
	    fcbext = 0;
	    while (TRUE)
	    { 
	       start = getint (&(fcbblock.extents[fcbext].startau));
	       if (start == 0xFFFF) break;
	       numaus = getint (&(fcbblock.extents[fcbext].numaus));
	       fcbext++;
	       printf (" %d:%d", start, numaus);
	    }
	 }
	 k = 7;
      }
      else
	 k++;
      if (k >= 6)
      {
	 k = 0;
	 printf ("\n");
      }
      free (dirent[j]);
      dirent[j] = NULL;
   }
   printf ("\n");
}

/***********************************************************************
* Main procedure
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *diskfd;
   char *disk = NULL;
   char *bp;
   int i;
   char temp[256];

   /*
   ** Process args
   */

   for (i = 1; i < argc; i++)
   {
      bp = argv[i];

      if (*bp == '-')
      {
	 for (bp++; *bp; bp++) switch (*bp)
	 {
	 case 'e':
	    longlist = TRUE;
	    extentlist = TRUE;
	    break;

	 case 'l':
	    longlist = TRUE;
	    break;

	 case 'p':
	    pascallist = TRUE;
	    break;

	 default:
      USAGE:
	    printf ("usage: fpydir [-p][-l][-e] floppy.file\n");
	    exit (1);
	 }
      }
      else
      {
         if (disk != NULL) goto USAGE;
	 disk = argv[i];
      }
   }
   if (disk == NULL) goto USAGE;
   
   /*
   ** Open the disk image.
   */

   if ((diskfd = fopen (disk, "rb")) == NULL)
   {
      sprintf (temp, "fpydir: Can't open disk image: %s", disk);
      perror (temp);
      return (1);
   }

   if (pascallist)
   {
      pasdirlist(diskfd);
   }

   else
   {
      stddirlist (diskfd);
   }

   fclose (diskfd);

   return (0);

}
