/***********************************************************************
*
* fpyfile - Dump file from TI-900 FD800 disk to stdout.
*
* Changes:
*   06/24/03   DGP   Original.
*   12/17/08   DGP   Added UCSD Pascal support.
*	
***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>
#include <stdlib.h>

#include "utiltypes.h"
#include "fd800defs.h"
#include "ucsdtypes.h"
#include "tx990types.h"

#define FALSE 0
#define TRUE  1

static char *ibp;
static int textmode = FALSE;
static int pascalmode = FALSE;
static int loadermode = FALSE;
static int loadaddr;
static int reclen;
static int cnt = 0;
static char buf[1024];

static unsigned char datablock[FPYSECLEN];

#include "hexdump.h"
#include "support.h"

/***********************************************************************
* getint - Convert integer to local endian.
***********************************************************************/

static uint16
getint (uint16 *val)
{
   uint8 *cp;
   uint16 cval;

   cp = (uint8 *)val;
   cval = (*cp << 8) | *(cp + 1);

#ifdef DEBUG
   fprintf (stderr, "getint: val = %d(%x), cval = %d(%x)\n",
	    *val, *val, cval, cval);
#endif
   return (cval);
}

/***********************************************************************
* cvtblk - Convert Pascal block address into Track/sector
***********************************************************************/

static int
cvtblk (int blk, int *ntrk, int *nsec)
{
   int trk;
   int sec;

   trk = (blk * PASSECBLK) / FPYSECTRK;
   sec = ((blk * PASSECBLK) % FPYSECTRK) + 1;

   *ntrk = trk;
   *nsec = sec;
#ifdef DEBUG
   fprintf (stderr, "cvtblk: blk = %d, trk = %d, sec = %d\n",
	    blk, trk, sec);
#endif
   return (0);
}

/***********************************************************************
* cvtau - Convert AU address into Track/sector
***********************************************************************/

static int
cvtau (int au, int aus, int *ntrk, int *nsec)
{
   int trk;
   int sec;

   if (au < FPYMAXAUS && aus < FPYSECAU)
   {
      trk = (au * FPYSECAU + aus) / FPYSECTRK;
      sec = (au * FPYSECAU + aus) % FPYSECTRK;
      sec = fpyskew[sec] + 1;
      *ntrk = trk;
      *nsec = sec;
#ifdef DEBUG
      fprintf (stderr, "   trk = %d, sec = %d\n", trk, sec);
#endif
      return (0);
   }
   return (-1);
}

/***********************************************************************
* readdisk - Read the disk
***********************************************************************/

static int
readdisk (FILE *diskfd, int au, int aus, unsigned char *buffer)
{
   int diskloc;
   int trk, sec;

#ifdef DEBUG
   fprintf (stderr, "readdisk: au = %d, aus = %d\n", au, aus);
#endif
   cvtau (au, aus, &trk, &sec);
   diskloc = ((FPYSECLEN * FPYSECTRK) * trk) + (FPYSECLEN * (sec-1));
   if (fseek (diskfd, diskloc, SEEK_SET) < 0)
   {
      perror("Disk seek error");
      return (-1);
   }

   if (FPYSECLEN != fread (buffer, 1, FPYSECLEN, diskfd))
   {
      perror ("Disk read error");
      return (-1);
   }
#ifdef DEBUG
   fprintf (stderr, "Buffer:\n");
   HEXDUMP (stderr, buffer, FPYSECLEN, 0);
#endif

   return (FPYSECLEN);

}

/***********************************************************************
* pasgetdisk - Get Pascal disk data and process into records.
***********************************************************************/

static int
pasgetdisk (FILE *diskfd, int type, int start, int end, int rem)
{
   int i;
   int sec, trk;
   int diskloc;
   int blklen;
   int reclen = 0;

#ifdef DEBUG
   fprintf (stderr, "pasgetdisk: type = %d, start = %d, end = %d, rem = %d\n",
	    type, start, end, rem);
#endif
   if (loadermode)
   {
      if (type == CODEFILE)
	 printf ("A%04X", 0);
      if (type == DATAFILE)
	 printf ("9%04X", loadaddr & 0xFFFF);
      reclen = 5;
   }

   if (type == TEXTFILE) start += 2;
   while (start < end)
   {
      uint8 *cp;

      cvtblk (start, &trk, &sec);
      diskloc = ((FPYSECLEN * FPYSECTRK) * trk) + (FPYSECLEN * (sec-1));
#ifdef DEBUG
      fprintf (stderr, "   trk = %d, sec = %d, diskloc = %d\n",
	       trk, sec, diskloc);
#endif
      if (fseek (diskfd, diskloc, SEEK_SET) < 0)
      {
	 perror("Disk seek error");
	 return (-1);
      }
      if (PASBLKLEN != fread (buf, 1, PASBLKLEN, diskfd))
      {
	 perror ("Disk read error");
	 return (-1);
      }
#ifdef DEBUG
      fprintf (stderr, "Buffer:\n");
      HEXDUMP (stderr, buf, PASBLKLEN, 0);
#endif
      if (start == (end - 1))
         blklen = rem;
      else
         blklen = PASBLKLEN;

      cp = (uint8 *)buf;
      switch (type)
      {
      case DATAFILE:
      case CODEFILE:
	 if (loadermode)
	 {
	    for (i = 0; i < blklen; i += 2)
	    {
	       uint16 word;

	       word = getint ((uint16 *)cp);
	       cp += 2;
	       if (reclen > 65)
	       {
		  printf ("80000F\n");
		  reclen = 0;
	       }
	       printf ("B%04X", word);
	       reclen += 5;
	    }
	 }
	 else
	 {
	    printf ("block: %d\n", start);
	    HEXDUMP (stdout, buf, PASBLKLEN, 0);
	 }
	 break;

      case TEXTFILE:
	 for (i = 0; i < blklen; i++)
	 {
	    if (*cp == PASDLE)
	    {
	       int j, l;
	       cp++;
	       i++;
	       l = *cp++ - 32;
	       for (j = 0; j < l; j++)
		  putchar (' ');
	    }
	    else if (*cp == PASEOL)
	    {
	       cp++;
	       putchar ('\n');
	    }
	    else if (*cp == 0)
	    {
	       cp++;
	       break;
	    }
	    else
	    {
	       putchar (*cp++);
	    }
	 }
	 break;

      default: 
         fprintf (stderr, "File type %s is unsupported\n",
		 pasfiletypes[type]);
	 return (-1);
      }
      start++;
   }

   if (loadermode && ((type == DATAFILE) || (type == CODEFILE)))
   {
      if (reclen)
	 printf ("80000F\n");
      printf (":\n");
   }

   return (0);
}

/***********************************************************************
* stdgetdisk - Get Standard disk data and process into records.
***********************************************************************/

static int
stdgetdisk (FILE *diskfd, int start, int numaus, int offset)
{
   int incompress;
   int binescape;
   int i, j, k;

#ifdef DEBUG
   fprintf (stderr, "stdgetdisk: start = %d, numaus = %d, offset = %d\n",
   	    start, numaus, offset);
#endif
   incompress = FALSE;
   binescape = FALSE;
   for (i = 0; i < numaus; i++)
   {
      for (j = offset; j < FPYSECAU; j++)
      {
	 unsigned char *cp;

	 offset = 0;
         if (readdisk (diskfd, start+i, j, datablock) < 0)
	    return (-1);

	 cp = datablock + (textmode ? 2 : 0);
	 for (k = textmode ? 2 : 0; k < FPYSECLEN; k++)
	 {
	    unsigned char ch;

	    ch = *cp++;
#ifdef DEBUG
	    fprintf (stderr, "ch = %02X(%c), cnt = %d\n", ch, ch, cnt);
#endif
	    if (binescape)
	    {
	       binescape = FALSE;
	       if (ch == 0) ch = 0xFA;
	       *ibp++ = ch;
	       cnt++;
	    }
	    else if (incompress)
	    {
	       incompress = FALSE;
	       for (; ch; ch--) { *ibp++ = ' '; cnt++; }
	    }
	    else switch (ch)
	    {
	    case ENDOFRECORD: /* 0xFF - End of record */
	       if (textmode)
	       {
	          for (ibp--; cnt && (*ibp == ' '); ibp--) cnt--;
		  ibp++;
		  cnt++;
	       }
	       else cnt++;
	       *ibp++ = '\n';
	       *ibp = 0;
	       fwrite (buf, 1, cnt, stdout);
#ifdef DEBUG
	       fprintf (stderr, "%s", buf);
#endif
	       ibp = buf;
	       cnt = 0;
	       break;

	    case COMPBLANKS: /* 0xFE - Compressed blank */
	       incompress = TRUE;
	       break;

	    case 0xFC: /* ??? - Can't remember */
	       break;

	    case ENDOFFILE: /* 0xFB - End of file */
	       if (cnt)
	       {
		  *ibp++ = '\n';
		  *ibp = 0;
		  fwrite (buf, 1, cnt, stdout);
	       }
	       return (EOF);

	    case BINARYESCAPE: /* 0xFA - Binary escape */
	       binescape = TRUE;
	       break;

	    default:
	       if (cnt < reclen)
	       {
	       if (textmode)
	       {
		  if (isprint(ch) || ch == 0x0C)
		  {
		     *ibp++ = ch;
		     cnt++;
		  }
	       }
	       else
	       {
		  *ibp++ = ch;
		  cnt++;
	       }
	       }
	    }
	 }
      }
   }
   return (0);
}

/***********************************************************************
* passcandir - Scan Pascal directory for filename
***********************************************************************/

static int
passcandir (FILE *diskfd, char *filename)
{
   pasdir_t *dp;
   uint8 *bp;
   int diskloc;
   int dirlen;
   int trk, sec;
   int i, j, l;
   unsigned char pasdirblk[PASDIRLEN];

   cvtblk (PASDIRBLK, &trk, &sec);
   diskloc = ((FPYSECLEN * FPYSECTRK) * trk) + (FPYSECLEN * (sec-1));
   if (fseek (diskfd, diskloc, SEEK_SET) < 0)
   {
      perror("Disk seek error");
      return (-1);
   }

   if (PASDIRLEN != fread (pasdirblk, 1, PASDIRLEN, diskfd))
   {
      perror ("Disk read error");
      return (-1);
   }
#ifdef DEBUG
   fprintf (stderr, "passcandir:\n");
   HEXDUMP (stderr, pasdirblk, PASDIRLEN, 0);
#endif

   dp = (pasdir_t *)pasdirblk;
   dirlen = getint (&(dp->filekind.volent.dnumfiles));
   if ((dirlen > PASMAXDIR) ||
       (getint(&(dp->dfkind)) != 0) || 
       (dp->filekind.volent.dvid[0] > PASVOLLEN))
   {
         fprintf (stderr, "Invalid disk format\n");
	 return (-1);
   }

   dp++;

   for (j = 0; j < dirlen; j++)
   {
      if (dp->filekind.filent.dtid[0])
      {
	 char pasfile[PASNAMLEN+2];

	 bp = (uint8 *)dp->filekind.filent.dtid;
#ifdef DEBUG
	 fprintf (stderr, "dirent:\n");
	 HEXDUMP (stderr, (uint8 *)dp, sizeof (pasdir_t), 0);
#endif
	 l = *bp++;
	 for (i = 0; i < l; i++)
	 {
	    pasfile[i] = *bp++;
	 }
	 pasfile[l] = '\0';
	 if (!strcmp (filename, pasfile))
	 {
	    int type, start, end, rem;

	    start = getint (&(dp->dfirstblk));
	    end = getint (&(dp->dlastblk));
	    type = getint (&(dp->dfkind));
	    rem = getint (&(dp->filekind.filent.dlastbyte));

	    return (pasgetdisk (diskfd, type, start, end, rem));
	 }
      }
      dp++;
   }
   return (1);
}

/***********************************************************************
* stdscandir - Scan standard directory for filename
***********************************************************************/

static int
stdscandir (FILE *diskfd, char *filename)
{
   int i, j, k, l;
   int done;
   char temp[256];
   txfcb_t fcbblock;
   disklabel_t disklabel;
   txdir_t dirblock[TXDIRPSEC];

   /*
   ** Read the disk label.
   */

   readdisk (diskfd, TXLABELAU, 0, (uint8 *)&disklabel);
   for (i = 0; i < 32; i++)
      if (disklabel.label[i] < ' ' || disklabel.label[i] > 'z')
      {
         fprintf (stderr, "Invalid disk format\n");
	 return (-1);
      }

   ibp = buf;

   /*
   ** Scan directory for filename
   */

   for (i = 0; i < FPYSECAU; i++)
   {
      readdisk (diskfd, TXDIRAU, i, (uint8 *)&dirblock);
      for (j = 0; j < TXDIRPSEC; j++)
      {
	 if (strncmp (dirblock[j].filename, "..", 2))
	 {
	    char file[TXFILELEN+1];
	    char ext[TXEXTLEN+1];

	    strncpy (file, dirblock[j].filename, TXFILELEN);
	    file[TXFILELEN] = '\0';
	    for (l = 0; l < TXFILELEN; l++) if (file[l] == ' ') file[l] = '\0';
	    strncpy (ext, dirblock[j].fileext, TXEXTLEN);
	    ext[TXEXTLEN] = '\0';
	    for (l = 0; l < TXEXTLEN; l++) if (ext[l] == ' ') ext[l] = '\0';
	    sprintf (temp, "%s.%s", file, ext);

	    /*
	    ** If this is the file, then process it
	    */

	    if (!strcmp (temp, filename))
	    {
	       int fcbaddr;
	       int start, numaus;

	       /*
	       ** Get file control block
	       */

	       fcbaddr = getint (&(dirblock[j].filefcb));
#ifdef DEBUG
               fprintf (stderr, " read fcb: fcbaddr = %d\n", fcbaddr);
#endif
	       readdisk (diskfd, fcbaddr, 0, (uint8 *)&fcbblock);
	       fcbaddr = 0;
	       k = 1;

	       /*
	       ** Read the sectors and process
	       */

	       done = FALSE;
	       while (!done)
	       { 
		  start = getint (&(fcbblock.extents[fcbaddr].startau));
#ifdef DEBUG
		  fprintf (stderr, "   start = %d\n", start);
#endif
		  if (start == 0xFFFF) break; /* Last extent */
		  numaus = getint (&(fcbblock.extents[fcbaddr].numaus));
		  fcbaddr++;
#ifdef DEBUG
		  fprintf (stderr, "   numaus = %d\n", numaus);
#endif
	          if (stdgetdisk (diskfd, start, numaus, k) == EOF) done = TRUE;
		  k = 0;
	       }
	       return (0);
	    }
	 }
      }
   }
   return (1);
}

/***********************************************************************
* Main procedure
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *diskfd;
   char *diskname = NULL;
   char *filename = NULL;
   char *bp;
   int retval;
   int i, j;
   char temp[256];

   /*
   ** Process args
   */

   reclen = 128;
   for (i = 1; i < argc; i++)
   {
      bp = argv[i];

      if (*bp == '-')
      {
         for (bp++; *bp; bp++) switch (*bp)
         {
	 case 'l':
	    loadermode = TRUE;
	    i++;
	    loadaddr = getnum (argv[i]);
	    break;

	 case 'p':
	    pascalmode = TRUE;
	    break;

	 case 'r':
	    i++;
	    reclen = getnum (argv[i]);
	    break;

	 case 't':
	    textmode = TRUE;
	    break;

         default:
      USAGE:
	    printf ("usage: fpyfile [-p[-l NN]][-t] floppy.file filename\n");
	    return (1);
         }
      }
      else
      {
         if (diskname == NULL) 
	    diskname = argv[i];
         else if (filename == NULL)
	 {
	    filename = argv[i];
	    for (j = 0; j < strlen(filename); j++)
	       if (islower(filename[j])) filename[j] = toupper(filename[j]);
	 }
         else goto USAGE;
      }
   }
   if (diskname == NULL || filename == NULL) goto USAGE;
   
   if ((diskfd = fopen (diskname, "rb")) == NULL)
   {
      sprintf (temp, "fpyfile: Can't open disk image: %s", diskname);
      perror (temp);
      return (1);
   }


   if (pascalmode)
   {
      retval = passcandir (diskfd, filename);
   }
   else
   {
      retval = stdscandir (diskfd, filename);
   }

   if (retval > 0)
      printf ("File %s not found on disk %s\n", filename, diskname);

   fclose (diskfd);
   return (retval);
}
