/***********************************************************************
*
* HD - Hexdump a disk/floppy/tape image.
*
* Changes:
*   03/31/05   DGP   Original
*   01/03/11   DGP   Added tape mode dump.
*   10/07/18   DGP   Fixed overhead.
*
***********************************************************************/
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/param.h>

#define MAXMEMSIZE 1024
#define MAXTAPERECLEN 32768

#include "hexdump.h"
#include "support.h"
#include "simdsk.h"

/***********************************************************************
* main
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *fd;
   char *infile;
   char *bp;
   char *mem;
   off_t seekoff;
   long seeklen;
   int i;
   int size;
   int ch;
   int usegeo;
   int reclen;
   int reccnt;
   int filecnt;
   int totcnt;
   int alloclen;
   int blkmode;
   int tapemode;

   /*
   ** Scan off args 
   */

   infile = NULL;
   reclen = MAXMEMSIZE;
   alloclen = MAXMEMSIZE;
   seeklen = 0;
   seekoff = 0;
   blkmode = 0;
   tapemode = 0;
   usegeo = 0;
   totcnt = -1;

   for (i = 1; i < argc; i++)
   {
      bp = argv[i];

      if (*bp == '-')
      {
         for (bp++; *bp; bp++) switch (*bp)
         {
	 case 'b':
            reclen = DEV_BSIZE;
	    blkmode = 1;
	    break;

	 case 'c':
	    i++;
	    totcnt = atoi (argv[i]);
	    break;

	 case 'g':
	    usegeo = 1;
	    blkmode = 1;
	    break;

	 case 'r':
	    i++;
	    reclen = atoi (argv[i]);
	    break;

	 case 's':
	    i++;
	    seeklen = atol (argv[i]);
	    break;

	 case 't':
            reclen = MAXTAPERECLEN;
	    tapemode = 1;
	    break;

         default:
      USAGE:
	    printf ("usage: hd [-options] in.file \n");
            printf (" options:\n");
            printf ("    -b           - Block mode\n");
            printf ("    -c NN        - Record/Block dump count\n");
            printf ("    -r NN        - Record/Block length\n");
            printf ("    -g           - Use disk geometry data\n");
            printf ("    -s NN        - Seek location\n");
            printf ("    -t           - Tape dump mode\n");
	    return (1);
         }
      }
      else if (!infile)
      {
        infile = argv[i];
      }
      else
      {
         goto USAGE;
      }

   }
   if (!infile) goto USAGE;

   if ((fd = fopen (infile, "rb")) == NULL)
   {
      perror ("hd: Can't open input file");
      exit (1);
   }

   /*
   ** Read the disk geometry
   */

   if (usegeo)
   {
      int tracklen, cyls, heads, dskloc, sectrk, overhead, sectlen;

      if ((cyls = dskreadint (fd)) < 0)
      {
	 perror ("Can't read disk geometry");
	 exit (1);
      }
      if ((heads = dskreadint (fd)) < 0) 
      {
	 perror ("Can't read disk geometry");
	 exit (1);
      }
      heads &= 0xFFFF;
      if ((dskloc = dskreadint (fd)) < 0)
      {
	 perror ("Can't read disk geometry");
	 exit (1);
      }
      sectrk = dskloc & 0xFF;
      overhead = (dskloc >> 8) & 0x3FF;
      if ((sectlen = dskreadint (fd)) < 0)
      {
	 perror ("Can't read disk geometry");
	 exit (1);
      }

      tracklen = sectrk * (sectlen + overhead);

      for (i = 0; i < MAXDISKS; i++)
      {
	 if (disks[i].cyls == cyls &&
	     disks[i].heads == heads &&
	     disks[i].sectrk == sectrk &&
	     disks[i].bytsec == sectlen &&
	     disks[i].overhead == overhead)
	    break;
      }
      if (i == MAXDISKS)
      {
	 printf ("*Unknown Disk model*\n");
      }
		  
      printf ("%s disk geometry:\n", disks[i].model);
      printf ("   cyls      = %d\n", cyls);
      printf ("   heads     = %d\n", heads);
      printf ("   sectrk    = %d\n", sectrk);
      printf ("   sectlen   = %d\n", sectlen);
      printf ("   overhead  = %d\n", overhead);
      printf ("   tracklen  = %d\n", tracklen);
      printf ("   disk size = %d bytes\n", tracklen * heads * cyls);

      reclen = sectlen;
   }

   if (reclen > alloclen) alloclen = reclen;
   if ((mem = malloc (alloclen)) == NULL)
   {
      fprintf (stderr, "hd: Can't allocate memory\n");
      exit (1);
   }

   size = 0;
   reccnt = 0;
   filecnt = 0;

   printf ("Dump of %s: %s %s mode\n", infile,
	    tapemode ? "tape" : "disk",
	    blkmode ? "block" : "record");

   if (tapemode)
   {
      while (1)
      {
	 size = tapereadint (fd);
	 if (size == 0)
	 {
	    printf ("File %d Record %d: EOF\n", filecnt+1, ++reccnt);
	    reccnt = 0;
	    filecnt++;
	    continue;
	 }
	 if (size < 0) break;
	 fread (mem, size, 1, fd);
	 if (++reccnt >= seeklen)
	 {
	    printf ("File %d Record %d: Length %d\n",
		    filecnt+1, reccnt, size);
	    HEXDUMP (stdout, mem, size, 0);
	 }
	 tapereadint (fd);
	 if (totcnt >= 0 && reccnt >= totcnt)
	 {
	    break;
	 }
      }
   }
   else
   {
      if (blkmode)
      {
	 seekoff = (off_t)seeklen * (off_t)reclen;
	 if (usegeo) seekoff += DSKOVERHEAD;

	 if (fseek (fd, seekoff, SEEK_SET) < 0)
	 {
	    perror ("hd: Can't seek in file");
	    fclose (fd);
	    free (mem);
	    exit (1);
	 }
      }

      while ((ch = fgetc (fd)) != EOF)
      {
	 mem[size++] = ch % 0xFF;
	 if (size == reclen)
	 {
	    printf ("%s %ld:\n", blkmode ? "Block" : "Record",
	            reccnt + seeklen);
	    HEXDUMP (stdout, mem, size, 0);
	    size = 0;
	    reccnt++;
	    if (totcnt >= 0 && reccnt >= totcnt)
	    {
	       break;
	    }
	 }
      }

      if (size)
      {
	 printf ("%s %ld:\n", blkmode ? "Block" : "Record", reccnt + seeklen);
	 HEXDUMP (stdout, mem, size, 0);
      }
   }

   fclose (fd);
   free (mem);
   
   return (0);
}
